/*****************************************************************************
 * Copyright: 2010-2011 Christian Fetzer <fetzer.ch@googlemail.com>          *
 * Copyright: 2010-2011 Michael Zanetti <mzanetti@kde.org>                   *
 *                                                                           *
 * This program is free software: you can redistribute it and/or modify      *
 * it under the terms of the GNU General Public License as published by      *
 * the Free Software Foundation, either version 3 of the License, or         *
 * (at your option) any later version.                                       *
 *                                                                           *
 * This program is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
 * GNU General Public License for more details.                              *
 *                                                                           *
 * You should have received a copy of the GNU General Public License         *
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.     *
 *                                                                           *
 ****************************************************************************/

#include "geomap.h"

#include <cmath>

#include <QtCore/QDebug>

#include <QtLocation/QGeoCoordinate>
#include <QDebug>
#include <QGeoMapObject>
#include <QGeoBoundingBox>

#if defined(Q_OS_SYMBIAN) || defined(Q_OS_WINCE_WM) || defined(Q_WS_MAEMO_5) || defined(Q_WS_MAEMO_6)
static const bool  s_enableKineticPanning = true;
static const qreal s_kineticPanningHalflife = 200.0; // time until kinetic panning speed slows down to 50%, in msec
static const qreal s_panSpeedNormal = 0.3; // keyboard panning speed without modifiers, in pixels/msec
static const qreal s_panSpeedFast = 1.0; // keyboard panning speed with shift, in pixels/msec
static const qreal s_kineticPanSpeedThreshold = 0.02; // minimum panning speed, in pixels/msec
static const int   s_kineticPanningResolution = 75; // temporal resolution. Smaller values take more CPU but improve visual quality
static const int   s_holdTimeThreshold = 200; // maximum time between last mouse move and mouse release for kinetic panning to kick in
#else
static const bool  s_enableKineticPanning = true;
static const qreal s_kineticPanningHalflife = 300.0; // time until kinetic panning speed slows down to 50%, in msec
static const qreal s_panSpeedNormal = 0.3; // keyboard panning speed without modifiers, in pixels/msec
static const qreal s_panSpeedFast = 1.0; // keyboard panning speed with shift, in pixels/msec
static const qreal s_kineticPanSpeedThreshold = 0.005; // minimum panning speed, in pixels/msec
static const int   s_kineticPanningResolution = 30; // temporal resolution. Smaller values take more CPU but improve visual quality
static const int   s_holdTimeThreshold = 100; // maximum time between last mouse move and mouse release for kinetic panning to kick in
#endif

GeoMap::GeoMap(QGeoMappingManager *manager) :
    QGraphicsGeoMap(manager),
    m_panActive(false),
    m_markerPressed(false),
    m_kineticTimer(new QTimer)
{
    for (int i = 0; i < 5; ++i) mouseHistory.append(MouseHistoryEntry());

    connect(m_kineticTimer, SIGNAL(timeout()), this, SLOT(kineticTimerEvent()));
    m_kineticTimer->setInterval(s_kineticPanningResolution);
}

GeoMap::~GeoMap()
{
}

double GeoMap::centerLatitude() const
{
    return center().latitude();
}

double GeoMap::centerLongitude() const
{
    return center().longitude();
}

void GeoMap::setCenterLatitude(double lat)
{
    QGeoCoordinate c = center();
    c.setLatitude(lat);
    setCenter(c);
}

void GeoMap::setCenterLongitude(double lon)
{
    QGeoCoordinate c = center();
    c.setLongitude(lon);
    setCenter(c);
}

void GeoMap::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    setFocus();
    if (event->button() == Qt::LeftButton) {
        m_panActive = true;

        // When pressing, stop the timer and stop all current kinetic panning
        m_kineticTimer->stop();
        m_kineticPanSpeed = QPointF();
        m_lastMoveTime = QTime::currentTime();

        // Object clicked?
        m_markerPressed = false;

        // There seems to be a bug in mapObjectsAtScreenPosition which doesn't
        // return any objects on certain zoom levels even if there are one...
        // Let's calclulate the stuff ourselves

        // http://bugreports.qt.nokia.com/browse/QTMOBILITY-841

        // QList<QGeoMapObject*> objects = mapObjectsAtScreenPosition(event->pos());
        QList<QGeoMapObject*> objects;
        foreach(QGeoMapObject *obj, mapObjectsInViewport()) {
            qDebug() << "object in screen" << coordinateToScreenPosition(obj->boundingBox().topLeft()) << coordinateToScreenPosition(obj->boundingBox().bottomRight());
            QPointF topLeft = coordinateToScreenPosition(obj->boundingBox().topLeft());
            QPointF bottomRight = coordinateToScreenPosition(obj->boundingBox().bottomRight());
            if(topLeft.x() <= event->pos().x() &&
                    topLeft.y() <= event->pos().y() &&
                    bottomRight.x() >= event->pos().x() &&
                    bottomRight.y() >= event->pos().y()) {
                qDebug() << "appending...";
                objects.append(obj);
            }
        }

        if (objects.size() > 0) {
            m_pressed = objects;
            m_markerPressed = true;
        }
    }
    event->accept();
}

void GeoMap::mouseReleaseEvent(QGraphicsSceneMouseEvent *event)
{
    if (event->button() == Qt::LeftButton && m_panActive) {
        m_panActive = false;

        if (m_markerPressed) {
            qDebug() << "marker pressed" << m_pressed << m_pressed.count();

            // There seems to be a bug in mapObjectsAtScreenPosition which doesn't
            // return any objects on certain zoom levels even if there are one...
            // Let's calclulate the stuff ourselves

            // http://bugreports.qt.nokia.com/browse/QTMOBILITY-841

            //QList<QGeoMapObject*> objects = mapObjectsAtScreenPosition(event->pos());
            QList<QGeoMapObject*> objects;
            foreach(QGeoMapObject *obj, mapObjectsInViewport()) {
                qDebug() << "object in screen" << coordinateToScreenPosition(obj->boundingBox().topLeft()) << coordinateToScreenPosition(obj->boundingBox().bottomRight());
                QPointF topLeft = coordinateToScreenPosition(obj->boundingBox().topLeft());
                QPointF bottomRight = coordinateToScreenPosition(obj->boundingBox().bottomRight());
                if(topLeft.x() <= event->pos().x() &&
                        topLeft.y() <= event->pos().y() &&
                        bottomRight.x() >= event->pos().x() &&
                        bottomRight.y() >= event->pos().y()) {
                    qDebug() << "appending...";
                    objects.append(obj);
                }
            }

            foreach (QGeoMapObject* pressed, m_pressed) {
                if (objects.contains(pressed)) {
                    emit clicked(pressed);
                }
            }
            m_markerPressed = false;
        }

        if (!s_enableKineticPanning || m_lastMoveTime.msecsTo(QTime::currentTime()) > s_holdTimeThreshold) {
            return;
        }

        m_kineticPanSpeed = QPointF();
        int entries_considered = 0;

        QTime currentTime = QTime::currentTime();
        foreach (MouseHistoryEntry entry, mouseHistory) {
            // first=speed, second=time
            int deltaTime = entry.second.msecsTo(currentTime);
            if (deltaTime < s_holdTimeThreshold) {
                m_kineticPanSpeed += entry.first;
                entries_considered++;
            }
        }
        if (entries_considered > 0) m_kineticPanSpeed /= entries_considered;
        m_lastMoveTime = currentTime;

        // When releasing the mouse button/finger while moving, start the kinetic panning timer
        m_kineticTimer->start();
        m_panDecellerate = true;
    }

    event->accept();
}

void GeoMap::mouseMoveEvent(QGraphicsSceneMouseEvent *event)
{
    //m_markerPressed = false;

    if (m_panActive) {
        // Calculate time delta
        QTime currentTime = QTime::currentTime();
        int deltaTime = m_lastMoveTime.msecsTo(currentTime);
        m_lastMoveTime = currentTime;

        // Calculate position delta
        QPointF delta = event->lastPos() - event->pos();

        // Calculate and set speed
        if (deltaTime > 0) {
            m_kineticPanSpeed = delta / deltaTime;

            mouseHistory.push_back(MouseHistoryEntry(m_kineticPanSpeed, currentTime));
            mouseHistory.pop_front();
        }

        // Pan map
        panFloatWrapper(delta);

        emit panned();
    }

    event->accept();
}

void GeoMap::kineticTimerEvent()
{
    QTime currentTime = QTime::currentTime();
    int deltaTime = m_lastMoveTime.msecsTo(currentTime);
    m_lastMoveTime = currentTime;

    if (m_panDecellerate)
        m_kineticPanSpeed *= pow(qreal(0.5), qreal(deltaTime / s_kineticPanningHalflife));

    QPointF scaledSpeed = m_kineticPanSpeed * deltaTime;

    if (m_kineticPanSpeed.manhattanLength() < s_kineticPanSpeedThreshold) {
        // Kinetic panning is almost halted -> stop it.
        m_kineticTimer->stop();
        return;
    }
    panFloatWrapper(scaledSpeed);
}

// Wraps the pan(int, int) method to achieve floating point accuracy, which is needed to scroll smoothly.
void GeoMap::panFloatWrapper(const QPointF& delta)
{
    // Add to previously stored panning distance
    m_remainingPan += delta;

    // Convert to integers
    QPoint move = m_remainingPan.toPoint();

    // Commit mouse movement
    pan(move.x(), move.y());

    // Store committed mouse movement
    m_remainingPan -= move;
}

// Evaluates the panDir field and sets kineticPanSpeed accordingly. Used in MapWidget::keyPressEvent and MapWidget::keyReleaseEvent
void GeoMap::applyPan(const Qt::KeyboardModifiers& modifiers)
{
    Q_ASSERT(m_panDir.manhattanLength() <= 2);

    if (m_panDir.manhattanLength() == 0) {
        // If no more direction keys are held down, decellerate
        m_panDecellerate = true;
    } else {
        // Otherwise, set new direction
        qreal panspeed = (modifiers & Qt::ShiftModifier) ? s_panSpeedFast : s_panSpeedNormal;

        if (m_panDir.manhattanLength() == 2) {
            // If 2 keys are held down, adjust speed to achieve the same speed in all 8 possible directions
            panspeed *= sqrt(0.5);
        }

        // Finally set the current panning speed
        m_kineticPanSpeed = QPointF(m_panDir) * panspeed;
    }
}

void GeoMap::keyPressEvent(QKeyEvent *event)
{
    switch (event->key()) {
        case Qt::Key_Minus:
#ifdef Q_OS_SYMBIAN
        case Qt::Key_VolumeDown:
#endif
#ifdef Q_WS_MAEMO_5
        case Qt::Key_F8:
#endif
            if (zoomLevel() > minimumZoomLevel()) {
                setZoomLevel(zoomLevel() - 1);
            }
            break;

        case Qt::Key_Plus:
#ifdef Q_OS_SYMBIAN
        case Qt::Key_VolumeUp:
#endif
#ifdef Q_WS_MAEMO_5
        case Qt::Key_F7:
#endif
            if (zoomLevel() < maximumZoomLevel()) {
                setZoomLevel(zoomLevel() + 1);
            }
            break;

        case Qt::Key_T:
            if (mapType() == QGraphicsGeoMap::StreetMap)
                setMapType(QGraphicsGeoMap::SatelliteMapDay);
            else if (mapType() == QGraphicsGeoMap::SatelliteMapDay)
                setMapType(QGraphicsGeoMap::StreetMap);
            break;

        case Qt::Key_Shift:
            // If there's no current movement, we don't need to handle shift.
            if (m_panDir.manhattanLength() == 0) break;
        case Qt::Key_Left:
        case Qt::Key_Right:
        case Qt::Key_Up:
        case Qt::Key_Down:
            if (!event->isAutoRepeat()) {
                switch (event->key()) {
                    case Qt::Key_Left:
                        m_panDir.setX(-1);
                        break;

                    case Qt::Key_Right:
                        m_panDir.setX(1);
                        break;

                    case Qt::Key_Up:
                        m_panDir.setY(-1);
                        break;

                    case Qt::Key_Down:
                        m_panDir.setY(1);
                        break;
                }

                m_lastMoveTime = QTime::currentTime();
                m_kineticTimer->start();
                m_panDecellerate = false;

                applyPan(event->modifiers());
            }
            break;
    }

    event->accept();
}

void GeoMap::keyReleaseEvent(QKeyEvent* event)
{
    event->accept();

    // Qt seems to have auto-repeated release events too...
    if (event->isAutoRepeat()) return;

    switch (event->key()) {
        case Qt::Key_Left:
        case Qt::Key_Right:
            m_panDir.setX(0);
            break;

        case Qt::Key_Up:
        case Qt::Key_Down:
            m_panDir.setY(0);
            break;

        case Qt::Key_Shift:
            if (m_panDir.manhattanLength() == 0) return;
            break;

        default:
            return;
    }

    applyPan(event->modifiers());
}

void GeoMap::wheelEvent(QGraphicsSceneWheelEvent *event)
{
    qreal panx = event->pos().x() - size().width() / 2.0;
    qreal pany = event->pos().y() - size().height() / 2.0;
    pan(panx, pany);
    if (event->delta() > 0) {   // zoom in
        if (zoomLevel() < maximumZoomLevel()) {
            setZoomLevel(zoomLevel() + 1);
        }
    } else {                    // zoom out
        if (zoomLevel() > minimumZoomLevel()) {
            setZoomLevel(zoomLevel() - 1);
        }
    }
    pan(-panx, -pany);
    event->accept();
}
