/* GemRB - Infinity Engine Emulator
 * Copyright (C) 2003 The GemRB Project
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 *
 */

#include "GUI/ScrollBar.h"

#include "win32def.h"

#include "Interface.h"
#include "Variables.h"
#include "Video.h"
#include "GUI/EventMgr.h"
#include "GUI/Window.h"

ScrollBar::ScrollBar(void)
{
	Pos = 0;
	Value = 10;// ???: not sure why we magically set this to 10.
	State = 0;
	ResetEventHandler( ScrollBarOnChange );
	ta = NULL;
	for(int i=0;i<SB_RES_COUNT;i++)  {
		Frames[i]=NULL;
	}
}

ScrollBar::~ScrollBar(void)
{
	Video *video=core->GetVideoDriver();
	for(int i=0;i<SB_RES_COUNT;i++)  {
		if(Frames[i]) {
			video->FreeSprite(Frames[i]);
		}
	}
}

/** Sets a new position, relays the change to an associated textarea and calls
	any existing GUI OnChange callback */
void ScrollBar::SetPos(ieDword NewPos)
{
	if (Value && NewPos >= Value) NewPos = Value - 1;
	
	if (( State & SLIDER_GRAB ) == 0){
		// set the slider to the exact y for NewPos. in SetPosForY(y) it is set to any arbitrary position that may lie between 2 values.
		// if the slider is grabbed dont set position! otherwise you will get a flicker as it bounces between exact positioning and arbitrary
		SliderYPos = ( unsigned short )
		( Frames[IE_GUI_SCROLLBAR_UP_UNPRESSED]->Height +
		 ( NewPos * ( ( Height - Frames[IE_GUI_SCROLLBAR_SLIDER]->Height - Frames[IE_GUI_SCROLLBAR_UP_UNPRESSED]->Height - Frames[IE_GUI_SCROLLBAR_DOWN_UNPRESSED]->Height) /
					 ( double ) ( Value <= 1 ? 1 : Value - 1 ) ) ) );
	}
	if (Pos && ( Pos == NewPos )) {
		return;
	}
	
	Changed = true;
	Pos = (ieWord) NewPos;
	if (ta) {
		TextArea* t = ( TextArea* ) ta;
		t->SetRow( Pos );
	}
	if (VarName[0] != 0) {
		core->GetDictionary()->SetAt( VarName, Pos );
	}
	RunEventHandler( ScrollBarOnChange );
	core->RedrawAll();
}

/** Sets the Pos for a given y coordinate (control coordinates) */
/** Provides per-pixel scrolling, however, no corresponding controls (i.e. TextArea) support per-pixel scrolling at this time. */
void ScrollBar::SetPosForY(unsigned short y)
{
	if (Value > 1) {// if the value is 1 we are simultaniously at both the top and bottom so there is nothing to do
		unsigned short YMin = Frames[IE_GUI_SCROLLBAR_UP_UNPRESSED]->Height;
		unsigned short YMax = Height - YMin - Frames[IE_GUI_SCROLLBAR_DOWN_UNPRESSED]->Height;
		if (y < YMin) y = YMin;
		else if (y > YMax) y = YMax;
		
		double step = YMax / ( double ) ( Value <= 1 ? 1 : Value - 1 );
		if (step) {
			unsigned short NewPos = y / step;
			// SetPos will call core->RedrawAll()
			if (Pos != NewPos) {
				// will move the slider to an exact position for NewPos
				SetPos( NewPos );
			} else {
				// move the slider to y while keeping Pos the same
				SliderYPos = y;
				core->RedrawAll();
			}
		}
	}
}

/** Refreshes the ScrollBar according to a guiscript variable */
void ScrollBar::RedrawScrollBar(const char* Variable, int Sum)
{
	if (strnicmp( VarName, Variable, MAX_VARIABLE_LENGTH )) {
		return;
	}
	SetPos( Sum );
}

/** SDL < 1.3 Mousewheel support */
void ScrollBar::ScrollUp()
{
	if( Pos ) SetPos( Pos - 1 );
}

/** SDL < 1.3 Mousewheel support */
void ScrollBar::ScrollDown()
{
	SetPos( Pos + 1 );
}

/** Draws the ScrollBar control */
void ScrollBar::Draw(unsigned short x, unsigned short y)
{
	if (!Changed && !(Owner->Flags&WF_FLOAT) ) {
		return;
	}
	Changed = false;
	if (XPos == 65535) {
		return;
	}
	int upMy = Frames[IE_GUI_SCROLLBAR_UP_UNPRESSED]->Height;
	int doMy = Frames[IE_GUI_SCROLLBAR_DOWN_UNPRESSED]->Height;
	unsigned int domy = (Height - doMy);
	
	unsigned short slx = ( unsigned short ) ((Width - Frames[IE_GUI_SCROLLBAR_SLIDER]->Width) / 2 );
	//draw the up button
	if (( State & UP_PRESS ) != 0) {
		core->GetVideoDriver()->BlitSprite( Frames[IE_GUI_SCROLLBAR_UP_PRESSED],
										   x + XPos, y + YPos, true );
	} else {
		core->GetVideoDriver()->BlitSprite( Frames[IE_GUI_SCROLLBAR_UP_UNPRESSED],
										   x + XPos, y + YPos, true );
	}
	//draw the trough
	int maxy = y + YPos + Height -
	Frames[IE_GUI_SCROLLBAR_DOWN_UNPRESSED]->Height;
	int stepy = Frames[IE_GUI_SCROLLBAR_TROUGH]->Height;
	Region rgn( x + XPos, y + YPos + upMy, Width, domy - upMy);
	for (int dy = y + YPos + upMy; dy < maxy; dy += stepy) {
		core->GetVideoDriver()->BlitSprite( Frames[IE_GUI_SCROLLBAR_TROUGH],
										   x + XPos + ( ( Width / 2 ) -
													   Frames[IE_GUI_SCROLLBAR_TROUGH]->Width / 2 ),
										   dy, true, &rgn );
	}
	//draw the down button
	if (( State & DOWN_PRESS ) != 0) {
		core->GetVideoDriver()->BlitSprite( Frames[IE_GUI_SCROLLBAR_DOWN_PRESSED],
										   x + XPos, maxy, true );
	} else {
		core->GetVideoDriver()->BlitSprite( Frames[IE_GUI_SCROLLBAR_DOWN_UNPRESSED],
										   x + XPos, maxy, true );
	}
	//draw the slider
	core->GetVideoDriver()->BlitSprite( Frames[IE_GUI_SCROLLBAR_SLIDER],
									   x + XPos + slx + Frames[IE_GUI_SCROLLBAR_SLIDER]->XPos,
									   y + YPos + Frames[IE_GUI_SCROLLBAR_SLIDER]->YPos + SliderYPos,
									   true );
}

/** Sets a ScrollBar GUI resource */
void ScrollBar::SetImage(unsigned char type, Sprite2D* img)
{
	if (type >= SB_RES_COUNT) {
		return;
	}
	if (Frames[type]) {
		core->GetVideoDriver()->FreeSprite(Frames[type]);
	}
	Frames[type] = img;
	Changed = true;
}

/** Mouse Button Down */
void ScrollBar::OnMouseDown(unsigned short /*x*/, unsigned short y,
							unsigned short Button, unsigned short /*Mod*/)
{
	//removing the double click flag, use a more sophisticated method
	//if it is needed later
	Button&=GEM_MB_NORMAL;
	if (Button==GEM_MB_SCRLUP) {
		ScrollUp();
		return;
	}
	if (Button==GEM_MB_SCRLDOWN) {
		ScrollDown();
		return;
	}
	
	if (y <= Frames[IE_GUI_SCROLLBAR_UP_UNPRESSED]->Height ) {
		State |= UP_PRESS;
		ScrollUp();
		return;
	}
	if (y >= Height - Frames[IE_GUI_SCROLLBAR_DOWN_UNPRESSED]->Height) {
		State |= DOWN_PRESS;
		ScrollDown();
		return;
	}
	if (y >= SliderYPos && y < SliderYPos + Frames[IE_GUI_SCROLLBAR_SLIDER]->Height) {
		State |= SLIDER_GRAB;
		return;
	}
	SetPosForY(y - (Frames[IE_GUI_SCROLLBAR_SLIDER]->Height / 2));
}

/** Mouse Button Up */
void ScrollBar::OnMouseUp(unsigned short /*x*/, unsigned short /*y*/,
						  unsigned short /*Button*/, unsigned short /*Mod*/)
{
	Changed = true;
	State = 0;
}

/** Mousewheel scroll */
void ScrollBar::OnMouseWheelScroll(short /*x*/, short y){
	if ( State == 0 ){//dont allow mousewheel to do anything if the slider is being interacted with already.
		unsigned short fauxY = SliderYPos;
		if ((short)fauxY + y <= 0) fauxY = 0;
		else fauxY += y;
		SetPosForY(fauxY);
	}
}

/** Mouse Over Event */
void ScrollBar::OnMouseOver(unsigned short /*x*/, unsigned short y)
{
	if (( State & SLIDER_GRAB ) != 0) {
		SetPosForY(y - (Frames[IE_GUI_SCROLLBAR_SLIDER]->Height / 2));
	}
}

/** Sets the Maximum Value of the ScrollBar */
void ScrollBar::SetMax(unsigned short Max)
{
	Value = Max;
	if (Max == 0) {
		SetPos( 0 );
	} else if (Pos >= Max) {
		SetPos( Max - 1 );
	}
}

/** Sets the ScrollBarOnChange event (guiscript callback) */
bool ScrollBar::SetEvent(int eventType, EventHandler handler)
{
	Changed = true;

	switch (eventType) {
	case IE_GUI_SCROLLBAR_ON_CHANGE:
		ScrollBarOnChange = handler;
		break;
	default:
		return false;
	}

	return true;
}
