/*
 * Copyright 2010 Felipe Crochik <foss@crochik.com>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "localsearchitemdelegate.h"

#include <QDebug>
#include <QPainter>
#include <QTextItem>

#include "business.h"

LocalSearchItemDelegate::LocalSearchItemDelegate(QListView *pParent) :
    QStyledItemDelegate(pParent)
{
    m_leftMargin = 0;
    m_rightMargin = 0;
    m_vertMargin = 0;
}

void LocalSearchItemDelegate::paint( QPainter * painter, const QStyleOptionViewItem & option, const QModelIndex & index ) const {
    QStyleOptionViewItemV4 options = option;
    initStyleOption(&options, index);

    painter->save();

    QColor highlightColor = options.palette.color(QPalette::Highlight);

    Business *pBusiness = static_cast<Business *>(index.internalPointer());
    Q_ASSERT(pBusiness);

    /* Call this to get the focus rect and selection background. */
    QStyleOptionViewItemV4 bkgOptions = options;
    bkgOptions.text = "";
    bkgOptions.rect.setLeft(bkgOptions.rect.left()+m_leftMargin);
    bkgOptions.rect.setRight(bkgOptions.rect.right() - m_rightMargin);
    bkgOptions.widget->style()->drawControl(QStyle::CE_ItemViewItem, &bkgOptions, painter);

    // calculate title height
    QFontMetrics metrics(m_font);
    QRect r = metrics.boundingRect(bkgOptions.rect, Qt::TextWordWrap, options.text); // pBusiness->description()
    int hTitle = r.height();
    int hAddress = option.fontMetrics.height();

    // title
    r = bkgOptions.rect;
    // r.setLeft(r.left()+m_leftMargin);
    // r.setRight(options.rect.right() - m_rightMargin);
    r.setBottom(r.top()+hTitle+(r.height()-hTitle-hAddress)/2);
    painter->setFont(m_font);
    painter->drawText(r, Qt::TextWordWrap | Qt::AlignBottom, options.text); // pBusiness->description()

    // address
    painter->setPen(highlightColor);
    r.setTop(r.bottom());
    r.setBottom(bkgOptions.rect.bottom());
    painter->setFont(options.font);

    QString address = QString("%1, %2, %3").arg(
        pBusiness->m_address.m_address,
        pBusiness->m_address.m_city,
        pBusiness->m_address.m_region
    );
    address = options.fontMetrics.elidedText(address, Qt::ElideRight, r.width());
    painter->drawText(r, Qt::TextSingleLine | Qt::AlignTop, address);

    painter->setPen(Qt::lightGray);

    // step index
    r = options.rect;
    r.setWidth(m_leftMargin);
    painter->setFont(options.font);
    QString letter('A' + index.row());
    painter->drawText(r, Qt::AlignVCenter | Qt::AlignHCenter, letter);

    /*
    r = options.rect;
    r.setLeft(options.rect.right()-m_rightMargin);
    r.setTop(r.top()+m_vertMargin);
    r.setBottom(options.rect.top()+option.rect.height()/2);

    // distance
    painter->setPen(QColor(255,128,0));
    painter->drawText(r, Qt::AlignTop | Qt::AlignHCenter, pBusiness->m_distanceStr);

    // time
    painter->setPen(Qt::lightGray);
    painter->drawText(r, Qt::AlignBottom | Qt::AlignHCenter, pBusiness->m_durationStr);

    if ( index.row()>0 ) {
        painter->setPen(QColor(255,128,0));
        painter->drawLine(options.rect.left(), options.rect.top(),options.rect.right(), options.rect.top());
    }
    */

    painter->restore();
}

QSize LocalSearchItemDelegate::sizeHint( const QStyleOptionViewItem & option, const QModelIndex & index ) const {
    QListView *pList = dynamic_cast<QListView *>(parent());
    if ( !pList ) return QStyledItemDelegate::sizeHint(option, index);

    int w = pList->viewport()->width();
    w -= m_leftMargin;
    w -= m_rightMargin;

    QRect r(0,0,w,10);
    QFontMetrics metrics(m_font);
    r = metrics.boundingRect(r, Qt::TextWordWrap, index.data().toString());

    int h = r.height();
    h += option.fontMetrics.height();
    h += m_vertMargin*2;

    return QSize(pList->viewport()->width(), h);
}
