#include "exerciseclassic.h"

#include <QVBoxLayout>
#include <QFont>
#include <QPushButton>
#include <QSpinBox>
#include <QFormLayout>
#include <QCheckBox>
#include <QFontMetrics>
#include <QScrollBar>
#include <QFile>
#include <QTextStream>
#include <QGroupBox>
#include <QRadioButton>

ExerciseClassic::ExerciseClassic(QWidget *parent) :
        Exercise(parent)
{
    m_currentPos=0;
    m_interval = 900;

    QVBoxLayout *vbLayout = new QVBoxLayout;

    m_plainTextViewer = new PlainTextViewer;
    m_plainTextViewer->setSizePolicy(QSizePolicy::Preferred,QSizePolicy::MinimumExpanding);

    m_lblExerciseInfo.setSizePolicy(QSizePolicy::Preferred,QSizePolicy::MinimumExpanding);
    m_lblExerciseInfo.setGeometry(geometry());
    m_lblExerciseInfo.setAlignment(Qt::AlignJustify|Qt::AlignTop);
    m_lblExerciseInfo.setWordWrap(true);

    vbLayout->addWidget(m_plainTextViewer);
    vbLayout->addWidget(&m_lblExerciseInfo);

    setLayout(vbLayout);

    showExerciseInfo();

    generateSettings();

    connect(&m_timer,SIGNAL(timeout()),this,SLOT(tick()));

    m_pxmGreenPoint.load(":/img/greenpoint.png");
    m_lblGreenPoint.setParent(this);
    m_lblGreenPoint.setPixmap(m_pxmGreenPoint);
    m_lblGreenPoint.setScaledContents(true);
    m_lblGreenPoint.setGeometry(0,0,10,10);
    m_lblGreenPoint.hide();
    m_lblGreenPoint.raise();

    m_lblShutter.setAutoFillBackground(true);
    m_lblShutter.setParent(this);
    m_lblShutter.setGeometry(m_plainTextViewer->geometry().left(),m_plainTextViewer->geometry().top(),width(),0);
    m_lblShutter.hide();
    m_lblShutter.raise();

    m_lblPause.setParent(this);
    m_lblPause.hide();
    m_lblPause.setStyleSheet("background: rgba(133,133,135,50%); border: 0px;border-radius: 5px;");
    m_lblPause.setPixmap(QIcon::fromTheme("media-playback-pause", QIcon(":/img/paused.png")).pixmap(100));
    m_lblPause.setAlignment(Qt::AlignCenter);
    m_lblPause.raise();
    m_lblPause.resize(100,100);
}

void ExerciseClassic::showExerciseInfo(){
    m_plainTextViewer->hide();

    QString info = "<b>"+tr("Classic mode will help you to increase your speed of reading by forcing scrolling.")+"</b><br>"+
                   tr("Available modes:")+"<br>"+
                   tr("1. Green point will move through the text and you should follow it. This will help you to decrease the number of eye movements.")+"<br>"+
                   tr("2. Reading by storm. Text will be scrolled automatically.");//+"<br>"+
                   //tr("Tap to screen or press \"Start\" button to start (if text is loaded).");

    m_lblExerciseInfo.setText(info);

    m_lblExerciseInfo.show();
}

QWidget* ExerciseClassic::settingsWidget(){
    return &m_settingsWidget;
}

bool ExerciseClassic::advancedMode(){
    return true;
}

void ExerciseClassic::setText(QString& str){
    m_plainTextViewer->setPlainText(str);
    emit maxProgressChanged(m_plainTextViewer->verticalScrollBar()->maximum());
}

void ExerciseClassic::start(){
    if (m_currentPos>=m_textLineVisibleNumber)
        m_currentPos = 0;
    m_lblPause.hide();
    m_lblExerciseInfo.hide();
    m_plainTextViewer->show();
    if (m_settings.type==ExerciseClassic::Points){
        m_lblGreenPoint.show();
        m_lblShutter.hide();
    } else {
        m_lblGreenPoint.hide();
        m_lblShutter.show();
    }
    tick();
    m_timer.start(m_interval);
}

void ExerciseClassic::stop(){
    m_currentPos=0;
    m_lblPause.hide();
    showExerciseInfo();
    m_timer.stop();
}

void ExerciseClassic::pause(){
    m_lblPause.show();
    m_timer.stop();
}

void ExerciseClassic::tick(){
    QFontMetrics fm(m_plainTextViewer->font());
    static int stepsCount=0;
    static int linesHandled=0;
    static int ticksDone=0;
    m_textLineVisibleNumber = m_plainTextViewer->height()/fm.lineSpacing();

    if (m_settings.type==ExerciseClassic::Points){
        int step=m_plainTextViewer->width()/4;
        stepsCount++;
        m_lblGreenPoint.setGeometry(step*stepsCount,fm.height()*linesHandled+m_plainTextViewer->geometry().top(),fm.height(),fm.height());
        if (stepsCount==3){
            stepsCount=0;
            linesHandled++;
        }
    } else {
        if (m_plainTextViewer->verticalScrollBar()->value()>0)
            m_lblShutter.setGeometry(m_plainTextViewer->geometry().left(),m_plainTextViewer->geometry().top(),
                                     m_plainTextViewer->width(),fm.height()*linesHandled);
        else
            m_lblShutter.setGeometry(m_plainTextViewer->geometry().left(),m_plainTextViewer->geometry().top(),
                                     m_plainTextViewer->width(),fm.height()*linesHandled-fm.leading());
        linesHandled++;
    }

    if (linesHandled>m_textLineVisibleNumber){
        m_plainTextViewer->verticalScrollBar()->setValue(m_plainTextViewer->verticalScrollBar()->value()+linesHandled);
        linesHandled = 0;
    }

    ticksDone++;
    if (m_settings.accelerate&&(m_timer.interval()>0)&&ticksDone>=(30000/m_timer.interval())){
        ticksDone = 0;
        faster();
    }
}

void ExerciseClassic::zoomIn(){
    QFont font(m_plainTextViewer->font());
    int i=font.pointSize();
    if (i<24) {
        font.setPointSize(++i);

        m_plainTextViewer->setFont(font);
        emit minZoomExceed(false);
        if (i>=24)
            emit maxZoomExceed(true);
    } else
        emit maxZoomExceed(true);
}

void ExerciseClassic::zoomOut(){
    QFont font(m_plainTextViewer->font());
    int i=font.pointSize();
    if (i>8){
        font.setPointSize(--i);

        m_plainTextViewer->setFont(font);
        emit maxZoomExceed(false);
        if (i<=8)
            emit minZoomExceed(true);
    } else
        emit minZoomExceed(true);
}

bool ExerciseClassic::isEmptyText(){
    return m_plainTextViewer->toPlainText().isEmpty();
}

int ExerciseClassic::maxProgress(){
    if (m_plainTextViewer->toPlainText().isEmpty())
        return 0;
    else
        return m_plainTextViewer->verticalScrollBar()->maximum();
}

void ExerciseClassic::mousePressEvent(QMouseEvent *){
    emit clicked();
}

void ExerciseClassic::faster(){
    int n=60000/m_settings.maxSpeed;
    if (m_interval>(n+40)){
        m_interval-=40;
        emit speedChanged(60000/m_interval);
        emit minSpeedExceed(false);
        if (m_interval<=n)
            emit maxSpeedExceed(true);
    } else {
        m_interval=n;
        emit speedChanged(60000/m_interval);
        emit maxSpeedExceed(true);
    }
    m_timer.setInterval(m_interval);
}

void ExerciseClassic::slower(){
    int n=60000/m_settings.minSpeed;
    if (m_interval<(n-40)){
        m_interval+=40;
        emit speedChanged(60000/m_interval);
        emit maxSpeedExceed(false);
        if (m_interval>=n)
            emit minSpeedExceed(true);
    } else {
        m_interval=n;
        emit speedChanged(60000/m_interval);
        emit minSpeedExceed(true);
    }
    m_timer.setInterval(m_interval);
}

void ExerciseClassic::generateSettings(){
    m_settingsWidget.setAutoFillBackground(true);

    QFormLayout *formLayout = new QFormLayout;
    formLayout->setRowWrapPolicy(QFormLayout::WrapLongRows);
    m_settingsWidget.setLayout(formLayout);

    QLabel *lbl = new QLabel("<b>"+tr("Settings of Classic mode")+"</b>");
    formLayout->addRow(lbl);

    QSpinBox *sb = new QSpinBox;
    sb->setMinimum(20);
    sb->setMaximum(300);
    sb->setValue(m_settings.minSpeed);
    connect(sb,SIGNAL(valueChanged(int)),this,SLOT(setMinSpeed(int)));

    formLayout->addRow(tr("Minimum speed (lines/minute):"),sb);

    sb = new QSpinBox;
    sb->setMinimum(20);
    sb->setMaximum(300);
    sb->setValue(m_settings.maxSpeed);
    connect(sb,SIGNAL(valueChanged(int)),this,SLOT(setMaxSpeed(int)));

    formLayout->addRow(tr("Maximum speed (lines/minute):"),sb);

    QCheckBox *cb = new QCheckBox;
    cb->setChecked(m_settings.accelerate);
    connect(cb,SIGNAL(toggled(bool)),this,SLOT(setAccelerate(bool)));

    formLayout->addRow(tr("Accelerate"),cb);

//    QGroupBox *gb = new QGroupBox;
//    gb->setSizePolicy(QSizePolicy::Preferred,QSizePolicy::Maximum);
//    gb->setTitle(tr("Classic exercise mode"));
    QHBoxLayout *hbLayout = new QHBoxLayout;
    QRadioButton *rb = new QRadioButton;
    rb->setText(tr("Points"));
    rb->setSizePolicy(QSizePolicy::Preferred,QSizePolicy::Preferred);
    rb->setChecked(m_settings.type==ExerciseClassic::Points);
    connect(rb,SIGNAL(clicked()),this,SLOT(setTypePoints()));
    hbLayout->addWidget(rb);

    rb = new QRadioButton;
    rb->setText(tr("Shutter"));
    rb->setSizePolicy(QSizePolicy::Preferred,QSizePolicy::Preferred);
    rb->setChecked(m_settings.type==ExerciseClassic::Shutter);
    connect(rb,SIGNAL(clicked()),this,SLOT(setTypeShutter()));
    hbLayout->addWidget(rb);

//    rp = new QRadioButton;
//    rp->setText(tr("Page"));
//    rp->setChecked(m_settings.type==ExerciseClassic::Page);
//    connect(rp,SIGNAL(clicked()),this,SLOT(setTypePage()));
//    vbLayout->addWidget(rp);

    formLayout->addRow(hbLayout);

    QPushButton *btn = new QPushButton(tr("OK"));
    btn->setIcon(QIcon::fromTheme("window-close", QIcon(":/img/ok.png")));
    connect(btn,SIGNAL(clicked()),&m_settingsWidget,SLOT(hide()));

    formLayout->addRow(btn);
}

void ExerciseClassic::resizeEvent(QResizeEvent *event){
    QWidget::resizeEvent(event);
    m_lblPause.move((width()-m_lblPause.width())/2,(height()-m_lblPause.height())/2);
    if (parentWidget()!=NULL){
        QSize s=parentWidget()->size();
        m_settingsWidget.setGeometry(0,0,s.width(),s.height());
    }
    emit maxProgressChanged(m_plainTextViewer->verticalScrollBar()->maximum());
}

bool ExerciseClassic::isActive(){
    return m_timer.isActive();
}

void ExerciseClassic::setMinSpeed(int n){
    m_settings.minSpeed = n;
    if (m_interval<60000/n) {
        m_interval=60000/n;
        m_timer.setInterval(m_interval);
    }
}

void ExerciseClassic::setMaxSpeed(int n){
    m_settings.maxSpeed = n;
    if (m_interval>60000/n) {
        m_interval=60000/n;
        m_timer.setInterval(m_interval);
    }
}

void ExerciseClassic::setAccelerate(bool b){
    m_settings.accelerate = b;
}

int ExerciseClassic::speed(){
    return 60000/m_interval;
}

void ExerciseClassic::setOffset(int n){
    m_currentPos = n;
    if (m_currentPos!=m_plainTextViewer->verticalScrollBar()->maximum()){
        m_plainTextViewer->verticalScrollBar()->setValue(m_currentPos);
        tick();
    } else
        pause();
}

void ExerciseClassic::setTypePoints(){
    m_settings.type = ExerciseClassic::Points;
}

void ExerciseClassic::setTypeShutter(){
    m_settings.type = ExerciseClassic::Shutter;
}

void ExerciseClassic::setTypePage(){
    m_settings.type = ExerciseClassic::Page;
}


int ExerciseClassic::mode(){
    return Exercise::Classic;
}
