/*
 * This file is part of fuelpad, the fuel diary
 *
 * Copyright (c) 2007-2011 Julius Luukko <julle.luukko@quicknet.inet.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "uilocationview.h"
#include "uisettings.h"

/*******************************************************************
 *
 * Private definitions
 *
 *******************************************************************/

/*******************************************************************
 *
 * Private variables
 *
 *******************************************************************/

gint locationdispcol=0xffff;

static
gchar *column_headers[] = {gettext_noop("Latitude"),
			   gettext_noop("Longitude"),
			   gettext_noop("Place"),
			   gettext_noop("Alias"),
			   "icon", // for icon
			   "fancy", // for the fancy combined column
};

/*******************************************************************
 *
 * Prototypes of private functions
 *
 *******************************************************************/

static gboolean
visible_func (GtkTreeModel *model,
              GtkTreeIter  *iter,
              gpointer      data);
static
void cell_data_func (GtkTreeViewColumn *col,
		     GtkCellRenderer   *renderer,
		     GtkTreeModel      *model,
		     GtkTreeIter       *iter,
		     gpointer           user_data);
static
void new_cell_data_func (GtkTreeViewColumn *col,
			 GtkCellRenderer   *renderer,
			 GtkTreeModel      *model,
			 GtkTreeIter       *iter,
			 gpointer           user_data);
static
GtkTreeModel *create_and_fill_locationmodel (void);

/*******************************************************************
 *
 * Private functions
 *
 *******************************************************************/

static gboolean
visible_func (GtkTreeModel *model,
              GtkTreeIter  *iter,
              gpointer      data)
{
  gchar *str;
  char *date;
  gdouble colval;
  gboolean visible = FALSE;
  gboolean compres;
  GList *filt;
  GList *lp;
/*   recordfilter_t *p; */
  gboolean first;
  gboolean and;

/*   filt=*(GList **)data; */

  gtk_tree_model_get (model, iter, LOCATION_COL_ALIAS, &str, -1);

  if (str) {
    visible = TRUE;
  }

  return visible;
}

static
void cell_data_func (GtkTreeViewColumn *col,
		     GtkCellRenderer   *renderer,
		     GtkTreeModel      *model,
		     GtkTreeIter       *iter,
		     gpointer           user_data)
{
  double  value;
  gchar   buf[20];

  gtk_tree_model_get(model, iter, ((data_func_t *)user_data)->col, &value, -1);

  switch (((data_func_t *)user_data)->col) {
  default:
    g_snprintf(buf, sizeof(buf), ((data_func_t *)user_data)->format, value);
  }

  g_object_set(renderer, "text", buf, NULL);
}

static
void new_cell_data_func (GtkTreeViewColumn *col,
			 GtkCellRenderer   *renderer,
			 GtkTreeModel      *model,
			 GtkTreeIter       *iter,
			 gpointer           user_data)
{
  gchar *markup;

  char* place;
  char* alias;
  double latitude;
  double longitude;
  sqlite_int64 id;

  gtk_tree_model_get(model, iter, 
		     LOCATION_COL_LATITUDE, &latitude, LOCATION_COL_LONGITUDE, &longitude,
		     LOCATION_COL_PLACE, &place, LOCATION_COL_ALIAS, &alias,
		     -1);

  markup=g_markup_printf_escaped("<span foreground=\"%s\">%s</span>\n<span foreground=\"%s\" size=\"xx-small\">%s (lat %f, lon %f)</span>",
				 get_default_text_color(),
				 alias,
				 get_secondary_text_color(),
				 place,
				 latitude,
				 longitude);
  PDEBUG("%s\n",markup);
  g_object_set(renderer, "markup", markup, NULL);
  g_free(markup);
  g_free(place);
  g_free(alias);
}

static
GtkTreeModel *create_and_fill_locationmodel (void)
{
  GtkListStore  *store;
  GtkTreeIter    iter;
  GtkTreeModel        *filter;
  GtkTreeSortable     *sortable;
  const unsigned char* place;
  const unsigned char* alias;
  double latitude;
  double longitude;
  sqlite_int64 id;

  if (db_open()) {
    store = gtk_list_store_new (LOCATION_NUM_COLS, G_TYPE_DOUBLE, G_TYPE_DOUBLE, 
				G_TYPE_STRING, G_TYPE_STRING,
				GDK_TYPE_PIXBUF, G_TYPE_STRING, 
				G_TYPE_INT64, G_TYPE_BOOLEAN);

    if (DB_OK == db_init_location_query()) {
      while(DB_ROW == db_step_location_query_data(&latitude, 
						  &longitude, 
						  &place, 
						  &alias, 
						  &id)) {

	gtk_list_store_append (store, &iter);
	gtk_list_store_set (store, &iter,
			    LOCATION_COL_LATITUDE, latitude,
			    LOCATION_COL_LONGITUDE, longitude,
			    LOCATION_COL_PLACE, place,
			    LOCATION_COL_ALIAS, alias,
			    LOCATION_COL_ID, id,
			    LOCATION_COL_VISIBLE, TRUE,
			    -1);

      }
    }
    else {
      PDEBUG("db_init_location_query returned != DB_OK\n");
    }

    db_reset_location_query();

    filter = gtk_tree_model_filter_new(GTK_TREE_MODEL(store), NULL);
/*     gtk_tree_model_filter_set_visible_column(GTK_TREE_MODEL_FILTER(filter), */
/* 					     COL_VISIBLE); */

    gtk_tree_model_filter_set_visible_func(GTK_TREE_MODEL_FILTER(filter),
					   visible_func,
                                           NULL,
					   NULL);
    sortable = GTK_TREE_SORTABLE(gtk_tree_model_sort_new_with_model(filter));

/*     gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(sortable), COL_DAY,  */
/* 				    sort_iter_compare_func, */
/* 				    GINT_TO_POINTER(COL_DAY), NULL); */

/*     gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(sortable), COL_TRIP, */
/* 				    sort_iter_compare_func, */
/* 				    GINT_TO_POINTER(COL_TRIP), NULL); */

/*     gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(sortable), COL_CONSUM, */
/* 				    sort_iter_compare_func, */
/* 				    GINT_TO_POINTER(COL_CONSUM), NULL); */

/*     gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(sortable), COL_PRICEPERTRIP, */
/* 				    sort_iter_compare_func, */
/* 				    GINT_TO_POINTER(COL_PRICEPERTRIP), NULL); */

/*     gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(sortable), COL_PRICEPERLITRE, */
/* 				    sort_iter_compare_func, */
/* 				    GINT_TO_POINTER(COL_PRICEPERLITRE), NULL); */

/*     /\* set initial sort order *\/ */
    gtk_tree_sortable_set_sort_column_id(sortable, LOCATION_COL_PLACE, GTK_SORT_DESCENDING);

/*     g_signal_connect(sortable, "sort-column-changed", */
/* 		     G_CALLBACK(sort_column_changed), &ui); */

    return GTK_TREE_MODEL (sortable);
  }
  else
    return NULL;
}

/*******************************************************************
 *
 * Public functions
 *
 *******************************************************************/

void update_locationsview_row_data(GtkListStore  *store,
				   GtkTreeIter   *iter,
				   char *place,
				   char *alias,
				   double latitude,
				   double longitude,
				   sqlite_int64 id)
{
  PDEBUG("Updating locationsview\n");

  if (latitude != -1) {
    gtk_list_store_set(store, iter, LOCATION_COL_LATITUDE, latitude, -1);
  }
  if (longitude != -1) {
    gtk_list_store_set(store, iter, LOCATION_COL_LONGITUDE, longitude, -1);
  }
  if (place != NULL) {
    gtk_list_store_set(store, iter, LOCATION_COL_PLACE, place, -1);
  }
  if (alias != NULL) {
    gtk_list_store_set(store, iter, LOCATION_COL_ALIAS, alias, -1);
  }
  gtk_list_store_set(store, iter, LOCATION_COL_ID, id, -1);
}

GtkWidget *create_locationview_and_model (void)
{
  GtkWidget *view;
/*   static data_func_t dfdata1; */
  static data_func_t dfdata2;
  static data_func_t dfdata3;
  static data_func_t dfdata4;
  static data_func_t dfdata5;
  static data_func_t dfdata6;
  static data_func_t dfdata7;
  static data_func_t dfdata8;
  static data_func_t dfdata9;
  static data_func_t dfdata10;
  static data_func_t dfdata11;
  static data_func_t dfdata12;

  GtkTreeViewColumn   *col;
  GtkCellRenderer     *renderer;
  GtkTreeModel        *model;
  GtkTreeSelection    *select;

  GtkWidget           *label;
  PangoAttrList       *attrs;
  PangoAttribute      *attr;

  /* Scale texts smaller: create pango attributes */
  attrs = pango_attr_list_new();
  attr = pango_attr_scale_new(fontscalefactors[fontsize]);
  attr->start_index = 0;
  attr->end_index = -1;
  pango_attr_list_insert(attrs, attr);

  model = create_and_fill_locationmodel ();
#if MAEMO_VERSION_MAJOR == 5
  view = hildon_gtk_tree_view_new_with_model (HILDON_UI_MODE_EDIT, 
					      model);
#else
  view = gtk_tree_view_new_with_model (model);
#endif


  /* --- Column #1 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[LOCATION_COL_LATITUDE]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", LOCATION_COL_LATITUDE);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  dfdata2.col = LOCATION_COL_LATITUDE;
  strncpy(dfdata2.format,"%.5f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata2, NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), LOCATION_COL_LATITUDE);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),LOCATION_COL_LATITUDE), locationdispcol & (1<<LOCATION_COL_LATITUDE) && (uistyle==UICLASSIC));

  /* --- Column #2 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[LOCATION_COL_LONGITUDE]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", LOCATION_COL_LONGITUDE);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  dfdata3.col = LOCATION_COL_LONGITUDE;
  strncpy(dfdata3.format,"%.5f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata3, NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), LOCATION_COL_LONGITUDE);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view), LOCATION_COL_LONGITUDE), locationdispcol & (1<<LOCATION_COL_LONGITUDE) && (uistyle==UICLASSIC));

  /* --- Column #5 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[LOCATION_COL_PLACE]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", LOCATION_COL_PLACE);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), LOCATION_COL_PLACE);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view), LOCATION_COL_PLACE), locationdispcol & (1<<LOCATION_COL_PLACE) && (uistyle==UICLASSIC));

  /* --- Column #8 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[LOCATION_COL_ALIAS]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", LOCATION_COL_ALIAS);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), LOCATION_COL_ALIAS);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view), LOCATION_COL_ALIAS), locationdispcol & (1<<LOCATION_COL_ALIAS) && (uistyle==UICLASSIC));


  /* Icon column */
  /* --- Column #0 --- */
  col = gtk_tree_view_column_new();

//  gtk_tree_view_column_set_title(col, "Icon");

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_pixbuf_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "pixbuf", LOCATION_COL_ICON);
/*   gtk_tree_view_column_set_cell_data_func(col, renderer, pixbuf_cell_data_func, view, NULL); */

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),LOCATION_COL_ICON), uistyle==UIFANCY);
  /* --- Column #0.5 --- */

  col = gtk_tree_view_column_new();

/*   label=gtk_label_new(gettext(column_headers[COL_COMBINED])); */
/*   gtk_tree_view_column_set_widget(col, GTK_WIDGET(label)); */
/*   gtk_label_set_attributes(GTK_LABEL(label), attrs); */
/*   gtk_widget_show(label); */

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "markup", LOCATION_COL_COMBINED);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_cell_data_func(col, renderer, new_cell_data_func, NULL, NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), LOCATION_COL_COMBINED);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),LOCATION_COL_COMBINED), uistyle==UIFANCY);

  /* Invisible column to distinguish the results */
  /* --- Column #last --- */

  col = gtk_tree_view_column_new();

  gtk_tree_view_column_set_title(col, "Id");

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", LOCATION_COL_ID);

/*   renderer = gtk_cell_renderer_text_new (); */
/*   gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view), */
/*                                                -1,       */
/*                                                "Id",   */
/*                                                renderer, */
/*                                                "text", LOCATION_COL_ID, */
/*                                                NULL); */
  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),LOCATION_COL_ID), FALSE);

  /* Invisible visibility column */
  /* --- Column #last+1 --- */

  col = gtk_tree_view_column_new();

  gtk_tree_view_column_set_title(col, "Visible");

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", LOCATION_COL_VISIBLE);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),LOCATION_COL_VISIBLE), FALSE);

#if 1
  if (uistyle==UIFANCY) {
    gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(view), FALSE);
  }
  else {
    gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(view), TRUE);
  }
#else
  gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(view), TRUE);
#endif

  /* The tree view has acquired its own reference to the
   *  model, so we can drop ours. That way the model will
   *  be freed automatically when the tree view is destroyed */

  g_object_unref (model);

  select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
  gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);
/*   g_signal_connect(view, "row-activated", */
/* 		   G_CALLBACK (callback_recordactivated), */
/* 		   &ui); */

  return view;
}
