/*
 * This file is part of fuelpad, the fuel diary
 *
 * Copyright (c) 2007-2010 Julius Luukko <julle.luukko@quicknet.inet.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <math.h>

#include <location/location-gps-device.h>
#include <location/location-gpsd-control.h>
#include <location/location-distance-utils.h>
#include <libgnomevfs/gnome-vfs.h>

#include <glib.h>

#include "uidrivinglog.h"
#include "uilogview.h"
#include "uiconnection.h"
#include "uisettings.h"
#include "uilocations.h"
#include "gpsutil.h"
#include "geocode.h"
#include "mapif.h"

/*******************************************************************
 *
 * Public variables
 *
 *******************************************************************/

char dbdatetimeformat[]="%Y-%m-%dT%H:%M.%S";

/*******************************************************************
 *
 * Private definitions
 *
 *******************************************************************/

#define TIMEOUT_CHECK_FIX 10

typedef struct {
  time_t starttime;
  double lat_s;
  double lon_s;
  char *startplace;
  time_t endtime;
  double lat_e;
  double lon_e;
  char *endplace;
  float distance;
  time_t effectivetime;
  time_t pausestart;
  time_t pausetime;
  char *explanation;
} trip_t;

/*******************************************************************
 *
 * Private enumerations
 *
 *******************************************************************/
typedef enum {
  LOG_START,
  LOG_STOP
} log_point_t;

/*******************************************************************
 *
 * Private variables
 *
 *******************************************************************/

gboolean paused=FALSE;
trip_t currenttrip;
char *unknown = gettext_noop("unknown");
guint tagtimoutfunc;

/*******************************************************************
 *
 * Prototypes of private functions
 *
 *******************************************************************/
static char *reverse_geocode(double lat, double lon);
static void get_coordinates(log_point_t point, double *lat, double *lon);
static gboolean reverse_geocode_idle(gpointer data);
static gboolean get_from_locations_cache(double lat, double lon, char **address);
static gboolean reverse_geocode_from_cache(double lat, double lon, log_point_t point);

static void log_on_changed(LocationGPSDevice *device, AppUIData *pui);
static gboolean check_fix_timeout(AppUIData *pui);
static void ui_init_location(AppUIData *pui);
static void ui_stop_location();
static void destroy_logwin( GtkWidget *widget, gpointer data );
static void start_to_cache( GtkAction * action, AppUIData *pui );
static void start_from_cache( GtkAction * action, AppUIData *pui );
static void start_from_network( GtkAction * action, AppUIData *pui );
static void show_start_on_map( GtkAction * action, AppUIData *pui );
static void end_to_cache( GtkAction * action, AppUIData *pui );
static void end_from_cache( GtkAction * action, AppUIData *pui );
static void end_from_network( GtkAction * action, AppUIData *pui );
static void show_end_on_map( GtkAction * action, AppUIData *pui );
static gboolean update_distance_ui(AppUIData *pui);
static void update_start_button_icon(AppUIData *pui);
static void add_trip_response (GtkDialog *dialog, gint arg1, AppUIData *pui);
static void edit_trip_response (GtkDialog *dialog, gint arg1, AppUIData *pui);
static void create_trip_editwin(AppUIData *pui, GtkWidget *dialog, gint add);
static void callback_edittrip( GtkAction * action, AppUIData *pui );
static void callback_deletetrip( GtkAction * action, AppUIData *pui );
static void callback_startgps( GtkAction * action, AppUIData *pui );
static void callback_stopgps( GtkAction * action, AppUIData *pui );
#if PLAINGTK == 1
static void create_secondary_toolbar(GtkWidget * main_window, GtkWidget *vbox, AppUIData *pui);
#else
static void create_secondary_toolbar(HildonWindow * main_window, AppUIData *pui);
#endif
#if PLAINGTK == 1
static void create_logwin_menu(GtkWidget *window, GtkWidget *root, AppUIData *pui);
#else
static void create_logwin_menu(HildonWindow *window, AppUIData *pui);
#endif

static time_t get_effective_time(void);
static void init_trip(void);
static void trip_save_start_point(time_t time, double lat, double lon);
static void trip_save_end_point(time_t time, double lat, double lon);
static void trip_debug_show_data(void);
static gboolean toggle_log_paused(AppUIData *pui);
static void driving_log_clear( GtkAction * action, AppUIData *pui );
static void driving_log_pause( GtkAction * action, AppUIData *pui );
static void driving_log_start( GtkAction * action, AppUIData *pui );
static void driving_log_stop( GtkAction * action, AppUIData *pui );

/*******************************************************************
 *
 * Private functions
 *
 *******************************************************************/

/*******************************************************************
 * Interface to the location framework
 *******************************************************************/



/*******************************************************************
 * Reverse geocode
 *******************************************************************/

static
char *reverse_geocode(double lat, double lon)
{
  gboolean success;
  char *address;
  char *psave;
  size_t len;

  if (netconnected) {
    success = nominatim_reverse_geocode(&address, lat, lon);
  }
  else {
    success = FALSE;
  }

  if (!success) {
    len = strlen(unknown);
    address = (char *)malloc((len+1)*sizeof(char));
    strncpy(address, unknown, len+1);
  }

  len = strlen(address);
  psave = (char *)malloc((len+1)*sizeof(char));
  strncpy(psave, address, len+1);

  g_print(" Address %s\n",address);
  trip_debug_show_data();
  free(address);

  return psave;
}

static
void get_coordinates(log_point_t point, double *lat, double *lon)
{
  switch (point) {
  case LOG_START:
    *lat = currenttrip.lat_s;
    *lon = currenttrip.lon_s;
    break;
  case LOG_STOP:
    *lat = currenttrip.lat_e;
    *lon = currenttrip.lon_e;
    break;
  default:
    *lat = 0.0;
    *lon = 0.0;
    break;
  }
}

static
gboolean reverse_geocode_idle(gpointer data)
{
  log_point_t point;
  double lat;
  double lon;
  char *address;

  point = (log_point_t)data;
  get_coordinates(point, &lat, &lon);
  address = reverse_geocode(lat, lon);

  switch (point) {
  case LOG_START:
    currenttrip.startplace = address;
    break;
  case LOG_STOP:
    currenttrip.endplace = address;
    break;
  default:
    break;
  }

  return FALSE;
}

static
gboolean get_from_locations_cache(double lat, double lon, char **address)
{
  sqlite_int64 id;
  const unsigned char *place;
  const unsigned char *alias;
  unsigned char *psave;
  size_t len;
  gboolean ret=FALSE;

  id = db_find_nearest_location(lat, lon, locationaliasradius, &place, &alias);
  if (id>0) {
    if (strlen(alias) > 0) {
      psave = alias;
    }
    else {
      psave = place;
    }
    ret = TRUE;
  }
  else {
    len = strlen(unknown);
    psave = unknown;
    ret = FALSE;
  }
  len = strlen(psave);
  *address = (char *)malloc((len+1)*sizeof(char));
  strncpy(*address, psave, len+1);

  return ret;
}


static
gboolean reverse_geocode_from_cache(double lat, double lon, log_point_t point)
{
  sqlite_int64 id;
  const unsigned char *place;
  const unsigned char *alias;
  char *address;
  size_t len;
  gboolean ret=FALSE;

  ret = get_from_locations_cache(lat, lon, &address);
  if (ret) {
    switch (point) {
    case LOG_START:
      currenttrip.startplace = address;
      break;
    case LOG_STOP:
      currenttrip.endplace = address;
      break;
    default:
      break;
    }
    g_print(" Address %s\n",address);
  }

  return ret;
}

/*******************************************************************
 * UI related functions
 *******************************************************************/

static
void log_on_changed(LocationGPSDevice *device, AppUIData *pui)
{
  g_idle_add(update_distance_ui,pui);
}

static 
gboolean check_fix_timeout(AppUIData *pui)
{
  gboolean notfix;

  notfix = !device->fix;
  if (!notfix) {
    notfix = device->fix->mode < LOCATION_GPS_DEVICE_MODE_2D;
  }

  if (notfix) {
    g_idle_add(update_distance_ui,pui);
  }

  return TRUE;
}

static
void ui_init_location(AppUIData *pui)
{
  gboolean addcb;

  addcb=(device==NULL);
  gps_init_location(pui);
  if (addcb) {
    g_signal_connect_after(device,  "changed", G_CALLBACK(log_on_changed), pui); 
  }
  if (tagtimoutfunc == 0) {
    tagtimoutfunc = g_timeout_add_seconds(TIMEOUT_CHECK_FIX, check_fix_timeout, pui);
  }
}

static
void ui_stop_location()
{
  gps_stop_location();
  g_source_remove(tagtimoutfunc);
  tagtimoutfunc = 0;
}


static
void destroy_logwin( GtkWidget *widget,
		      gpointer   data )
{
  AppUIData *a;

  a=(AppUIData *)data;

  gtk_widget_destroy(GTK_WIDGET(a->app->drivinglogwindow));
  a->app->drivinglogwindow=NULL;

  update_driving_log_button(a);
}

static
void start_to_cache( GtkAction * action, AppUIData *pui )
{
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreeModel *model;
  char* place;
  double lat;
  double lon;

  ui_show_banner(_("Saving start place to cache"));
  if (db_open()) {
    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->logview));
    if (gtk_tree_selection_get_selected (selection, &model, &iter)) {

      gtk_tree_model_get(model, &iter, 
			 LOG_COL_STARTLAT, &lat, LOG_COL_STARTLON, &lon,
			 LOG_COL_STARTPLACE, &place,
			 -1);

      db_add_locationalias(lat, lon, place, place);
      g_free(place);
    }
  }
}

static
void start_from_cache( GtkAction * action, AppUIData *pui )
{
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreeModel *model;
  double lat;
  double lon;
  char *address;
  gboolean ret;

  ui_show_banner(_("Getting start place from cache"));

  if (db_open()) {
    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->logview));
    if (gtk_tree_selection_get_selected (selection, &model, &iter)) {

      gtk_tree_model_get(model, &iter, 
			 LOG_COL_STARTLAT, &lat, LOG_COL_STARTLON, &lon,
			 -1);

      ret = get_from_locations_cache(lat, lon, &address);
      if (ret) {
	g_print("Found address %s from cache", address);
	gtk_entry_set_text(GTK_ENTRY(pui->logstartplaceentry), address);
      }
      else {
	g_print("Did not find address from cache");
      }
      free(address);
    }
  }
}

static
void start_from_network( GtkAction * action, AppUIData *pui )
{
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreeModel *model;
  double lat;
  double lon;
  char *address;
  gboolean ret;

  ui_show_banner(_("Getting start place from network"));

  if (db_open()) {
    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->logview));
    if (gtk_tree_selection_get_selected (selection, &model, &iter)) {

      gtk_tree_model_get(model, &iter, 
			 LOG_COL_STARTLAT, &lat, LOG_COL_STARTLON, &lon,
			 -1);

      (void)ui_connection_ensure();
      address = reverse_geocode(lat, lon);

      g_print("Found address %s from network", address);
      gtk_entry_set_text(GTK_ENTRY(pui->logstartplaceentry), address);
      g_print("Did not find address from network");

      free(address);
    }
  }
}


static
void show_start_on_map( GtkAction * action, AppUIData *pui )
{
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreeModel *model;
  double lat;
  double lon;

  if (db_open()) {
    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->logview));
    if (gtk_tree_selection_get_selected (selection, &model, &iter)) {

      gtk_tree_model_get(model, &iter, 
			 LOG_COL_STARTLAT, &lat, LOG_COL_STARTLON, &lon,
			 -1);

      map_show_location(lat, lon);
    }
  }
}

static
void end_to_cache( GtkAction * action, AppUIData *pui )
{
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreeModel *model;
  char* place;
  double lat;
  double lon;

  ui_show_banner(_("Saving end place to cache"));
  if (db_open()) {
    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->logview));
    if (gtk_tree_selection_get_selected (selection, &model, &iter)) {

      gtk_tree_model_get(model, &iter, 
			 LOG_COL_ENDLAT, &lat, LOG_COL_ENDLON, &lon,
			 LOG_COL_ENDPLACE, &place,
			 -1);

      db_add_locationalias(lat, lon, place, place);
      g_free(place);
    }
  }
}

static
void end_from_cache( GtkAction * action, AppUIData *pui )
{
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreeModel *model;
  double lat;
  double lon;
  char *address;
  gboolean ret;

  ui_show_banner(_("Getting end place from cache"));

  if (db_open()) {
    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->logview));
    if (gtk_tree_selection_get_selected (selection, &model, &iter)) {

      gtk_tree_model_get(model, &iter, 
			 LOG_COL_ENDLAT, &lat, LOG_COL_ENDLON, &lon,
			 -1);

      ret = get_from_locations_cache(lat, lon, &address);
      if (ret) {
	g_print("Found address %s from cache", address);
	gtk_entry_set_text(GTK_ENTRY(pui->logendplaceentry), address);
      }
      else {
	g_print("Did not find address from cache");
      }
      free(address);
    }
  }
}

static
void end_from_network( GtkAction * action, AppUIData *pui )
{
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreeModel *model;
  double lat;
  double lon;
  char *address;
  gboolean ret;

  ui_show_banner(_("Getting stop place from network"));

  if (db_open()) {
    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->logview));
    if (gtk_tree_selection_get_selected (selection, &model, &iter)) {

      gtk_tree_model_get(model, &iter, 
			 LOG_COL_ENDLAT, &lat, LOG_COL_ENDLON, &lon,
			 -1);

      (void)ui_connection_ensure();
      address = reverse_geocode(lat, lon);

      g_print("Found address %s from network", address);
      gtk_entry_set_text(GTK_ENTRY(pui->logendplaceentry), address);
      g_print("Did not find address from network");

      free(address);
    }
  }
}

static
void show_end_on_map( GtkAction * action, AppUIData *pui )
{
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreeModel *model;
  double lat;
  double lon;

  if (db_open()) {
    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->logview));
    if (gtk_tree_selection_get_selected (selection, &model, &iter)) {

      gtk_tree_model_get(model, &iter, 
			 LOG_COL_ENDLAT, &lat, LOG_COL_ENDLON, &lon,
			 -1);

      map_show_location(lat, lon);
    }
  }
}

static
gboolean update_distance_ui(AppUIData *pui)
{
  GString *str;
  char *timestr;

  if (hildon_window_get_is_topmost(HILDON_WINDOW(pui->app->drivinglogwindow))) {
    str=g_string_new("Status: ");
    if (device == NULL) {
      g_string_append_printf(str, _("GPS not started."));
    }
    else {
      g_string_append_printf(str, "speed");
      if (device->fix->fields & LOCATION_GPS_DEVICE_SPEED_SET) {
	g_string_append_printf(str, " %.0f ", SIlength2user(device->fix->speed));
      }
      else {
	g_string_append_printf(str, " -- ");
      }
      if (curunit.lengthunit==SI)
	g_string_append_printf(str, _("km/h"));
      else
	g_string_append_printf(str, _("mph"));

      convunixtime(&timestr, "%T", get_effective_time());
      g_string_append_printf(str, " time %s ", timestr);
      free(timestr);

      g_string_append_printf(str, " distance %.1f ", SIlength2user(distance/1000.0));
      if (curunit.lengthunit==SI)
	g_string_append_printf(str, _("km"));
      else
	g_string_append_printf(str, _("miles"));
      g_string_append_printf(str, " Fix: %d", device->fix->mode);
    }

    gtk_label_set_text(GTK_LABEL(pui->stb_logdistancelabel), str->str);
    g_string_free(str, TRUE);
  }

  return FALSE;
} 

static
void update_start_button_icon(AppUIData *pui)
{
  GtkWidget *recimage = NULL;
  GtkWidget *playimage = NULL;
  GtkWidget *pauseimage = NULL;
  GtkWidget *image;

  if (driving_log_logging()) {
    if (recimage == NULL) {
      /* location_applet_track_stop_ani */
      recimage = gtk_image_new_from_icon_name ("camera_video_recording", GTK_ICON_SIZE_BUTTON);
    }
    image = recimage;
    hildon_button_set_text (HILDON_BUTTON (pui->logstartbutton), _("Pause"),
						    NULL);
  }
  else {
    if (driving_log_paused()) {
      if (pauseimage == NULL) {
	/* location_applet_track_start */
	pauseimage = gtk_image_new_from_icon_name ("camera_video_pause", GTK_ICON_SIZE_BUTTON);
      }
      image = pauseimage;
      hildon_button_set_text (HILDON_BUTTON (pui->logstartbutton), _("Continue"),
			      NULL);
    }
    else {
      if (playimage == NULL) {
	/* location_applet_track_start */
	playimage = gtk_image_new_from_icon_name ("location_applet_track_start", GTK_ICON_SIZE_BUTTON);
      }
      image = playimage;
      hildon_button_set_text (HILDON_BUTTON (pui->logstartbutton), _("Start"),
			      NULL);
    }
  }
  hildon_button_set_image (HILDON_BUTTON (pui->logstartbutton), image);
}


/**
 * \fn void add_trip_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
 * \brief Response callback for add trip dialog
 *
 * This callback adds the data entered in the add trip dialog to the
 * database.
 *
 *
 */
static
void add_trip_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  char *starttime;
  char *endtime;
  char *timestr;
  time_t effectivetime;
  sqlite_int64 recordid;
  GtkListStore  *store;
  GtkTreeIter    storeiter;
  GString *explanationstr;

  explanationstr = get_entry(pui->logexlanationentry);

  effectivetime=currenttrip.endtime-currenttrip.starttime-currenttrip.pausetime;
  unixtimefmt(currenttrip.starttime, &starttime, dbdatetimeformat);
  unixtimefmt(currenttrip.endtime, &endtime, dbdatetimeformat);
  convunixtime(&timestr, "%T", (time_t)(effectivetime));
  recordid = db_add_drivinglog(currenttrip.distance, starttime,
			       currenttrip.lat_s, currenttrip.lon_s,
			       currenttrip.startplace, endtime,
			       currenttrip.lat_e, currenttrip.lon_e,
			       currenttrip.endplace, 
			       effectivetime,
			       explanationstr->str);

  /* update view */
  get_store_and_iter(NULL, pui->logview, NULL, &store, NULL);
  gtk_list_store_append (store, &storeiter);
  update_logview_row_data(store, &storeiter, 
			  currenttrip.distance, starttime,
			  currenttrip.lat_s, currenttrip.lon_s,
			  currenttrip.startplace, endtime,
			  currenttrip.lat_e, currenttrip.lon_e,
			  currenttrip.endplace, timestr,
			  explanationstr->str,recordid);

  free(starttime);
  free(endtime);
  free(timestr);
  g_string_free(explanationstr, TRUE);
  gtk_widget_destroy(GTK_WIDGET(dialog));

}

/**
 * \fn void edit_trip_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
 * \brief Response callback for edit trip dialog
 *
 * This callback alters the data edited in the edit trip dialog to
 * the database.
 *
 */
static
void edit_trip_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkListStore  *store;
  GtkListStore  *fullstore;
  GtkTreeIter    iter;
  GtkTreeIter    storeiter;
  GString *explanation;
  GString *startplace;
  GString *endplace;

  gint64 id;

  if (!db_open()) {
       gtk_widget_destroy(GTK_WIDGET(dialog));
  }
  else {
    switch (arg1) {
    case GTK_RESPONSE_ACCEPT:
      selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->logview));
      if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
	gtk_tree_model_get (model, &iter, LOG_COL_ID, &id, -1);
      

	explanation = get_entry(pui->logexlanationentry);
	startplace = get_entry(pui->logstartplaceentry);
	endplace = get_entry(pui->logendplaceentry);
	if (db_open()) {
	  if (db_update_trip(id, startplace->str, 
			     endplace->str, 
			     explanation->str) != id) {
	    PDEBUG("Problem saving result");
	  }
	  else {
	    get_store_and_iter(model, pui->logview, &iter, &store, &storeiter);
	    update_logview_row_data(store, &storeiter,
				    -1, NULL,
				    -1, -1,
				    startplace->str, NULL,
				    -1, -1,
				    endplace->str,
				    NULL,
				    explanation->str,id);
	  }
	}
	g_string_free(explanation, TRUE);
	g_string_free(startplace, TRUE);
	g_string_free(endplace, TRUE);
      }
      else {
	PDEBUG("Something wrong with finding the selection\n");
      }
      gtk_widget_destroy(GTK_WIDGET(dialog));
      break;
    case GTK_RESPONSE_REJECT:
      gtk_widget_destroy(GTK_WIDGET(dialog));
      break;
    }
  }
}

/**
 * \fn void create_trip_editwin(AppUIData *pui, GtkWidget *dialog, gint add)
 * \brief Creates the contents of a trip adding or editing dialog
 * \param *pui Pointer to the UI data structure
 * \param *dialog Pointer to the already created dialog
 * \param add Set this to 1 if an add dialog is needed
 *
 * This function is used for creating an edit window both for adding a
 * new trip and editing an existing one.
 *
 */
static
void create_trip_editwin(AppUIData *pui, GtkWidget *dialog, gint add)
{
  enum
  {
    TRIP_EDIT_CAR=0,
    TRIP_EDIT_DRIVER,
    TRIP_EDIT_EXPLANATON,
    TRIP_EDIT_ADDTOLOCATIONS,
    TRIP_EDIT_GETFROMLOCATIONS,
    TRIP_EDIT_GETFROMNETWORK,
    TRIP_EDIT_SHOWONMAP
  };

  enum
  {
    TRIP_HDR_CARDRIVER=0,
    TRIP_HDR_EXPLANATION,
    TRIP_HDR_STARTPLACE,
    TRIP_HDR_ENDPLACE,
    TRIP_HDR_DISTANCE
  };

  GtkWidget *scrollwin;
#if MAEMO_VERSION_MAJOR == 5
  gint       row;
#else
  GtkWidget *notebook;
#endif
  GtkWidget *svbox;
  GtkWidget *frame;
  GtkWidget *uvbox;
  GtkWidget *table;
  GtkWidget *label;
  GtkWidget *button;
#if PLAINGTK == 1
  char date[ENTRYDATEMAX];
  char *aika;
#endif
  GtkListStore *store;
  GtkEntryCompletion *completion;
  gchar *labels[] = {gettext_noop("Car"),
		     gettext_noop("Driver"),
		     gettext_noop("Enter explanation"),
		     gettext_noop("Add to locations"),
		     gettext_noop("Get from locations"),
		     gettext_noop("Get from network"),
		     gettext_noop("Show on map")
  };
  gchar *headers[] = {gettext_noop("Driver and car"),
		      gettext_noop("Explanation"),
		      gettext_noop("Start place"),
		      gettext_noop("End place"),
		      gettext_noop("Distance")
  };

#if MAEMO_VERSION_MAJOR == 5
  scrollwin = hildon_pannable_area_new();
  gtk_widget_set_size_request(scrollwin, -1, DIALOG_MIN_HEIGHTMAX);
  svbox = gtk_vbox_new (FALSE, 0);

/*   if (add) */
/*     table = gtk_table_new(12, 2, FALSE); */
/*   else */
/*     table = gtk_table_new(10, 2, FALSE); */

  row=0;

  if (add) {

    frame=gtk_frame_new(gettext(headers[TRIP_HDR_CARDRIVER]));
    gtk_box_pack_start (GTK_BOX(svbox), frame, TRUE, TRUE, 5);
    gtk_widget_show(frame);
    
    uvbox = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER(frame), uvbox);

    /* Car button */
    pui->logcarbutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
    hildon_button_set_title (HILDON_BUTTON (pui->logcarbutton), gettext(labels[TRIP_EDIT_CAR]));
    gtk_box_pack_start (GTK_BOX(uvbox), pui->logcarbutton, TRUE, TRUE, 0);
    gtk_widget_show(pui->logcarbutton);
    
    pui->logcarcombo = create_car_combo(pui, pui->logcarbutton);

    /* Driver button */
    row++;
    pui->logdriverbutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
    hildon_button_set_title (HILDON_BUTTON (pui->logdriverbutton), gettext(labels[TRIP_EDIT_DRIVER]));
    gtk_box_pack_start (GTK_BOX(uvbox), pui->logdriverbutton, TRUE, TRUE, 0);
    gtk_widget_show(pui->logdriverbutton);

    pui->logdrivercombo = create_driver_combo(pui, pui->logdriverbutton);

    gtk_widget_show(uvbox);
  
  }

  /* Explanation entry */
  frame=gtk_frame_new(gettext(headers[TRIP_HDR_EXPLANATION]));
  gtk_box_pack_start (GTK_BOX(svbox), frame, TRUE, TRUE, 5);
  gtk_widget_show(frame);

  row++;
  pui->logexlanationentry=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  if (add) {
    hildon_gtk_entry_set_placeholder_text(GTK_ENTRY(pui->logexlanationentry),
					  gettext(labels[TRIP_EDIT_DRIVER]));
  }
  gtk_container_add (GTK_CONTAINER(frame), pui->logexlanationentry);
  gtk_widget_show(pui->logexlanationentry);

  /* Start place entry */
  frame=gtk_frame_new(gettext(headers[TRIP_HDR_STARTPLACE]));
  gtk_box_pack_start (GTK_BOX(svbox), frame, TRUE, TRUE, 5);
  gtk_widget_show(frame);

  table = gtk_table_new(3, 2, FALSE);
  gtk_container_add (GTK_CONTAINER(frame), table);

  row=0;
  pui->logstartplaceentry=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_table_attach(GTK_TABLE(table), pui->logstartplaceentry, 0, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->logstartplaceentry);

  if (!add) {
    row++;
    button = hildon_button_new_with_text(HILDON_SIZE_FINGER_HEIGHT,
					 HILDON_BUTTON_ARRANGEMENT_HORIZONTAL,
					 gettext(labels[TRIP_EDIT_ADDTOLOCATIONS]),
					 NULL);
    gtk_table_attach(GTK_TABLE(table), button, 0, 1, row, row+1,
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    g_signal_connect (GTK_OBJECT (button), "clicked",
		      G_CALLBACK(start_to_cache), pui);
    gtk_widget_show(button);

    button = hildon_button_new_with_text(HILDON_SIZE_FINGER_HEIGHT,
					 HILDON_BUTTON_ARRANGEMENT_HORIZONTAL,
					 gettext(labels[TRIP_EDIT_GETFROMLOCATIONS]),
					 NULL);
    gtk_table_attach(GTK_TABLE(table), button, 1, 2, row, row+1,
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    g_signal_connect (GTK_OBJECT (button), "clicked",
		      G_CALLBACK(start_from_cache), pui);
    gtk_widget_show(button);

    row++;
    button = hildon_button_new_with_text(HILDON_SIZE_FINGER_HEIGHT,
					 HILDON_BUTTON_ARRANGEMENT_HORIZONTAL,
					 gettext(labels[TRIP_EDIT_GETFROMNETWORK]),
					 NULL);
    gtk_table_attach(GTK_TABLE(table), button, 0, 2, row, row+1,
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    g_signal_connect (GTK_OBJECT (button), "clicked",
		      G_CALLBACK(start_from_network), pui);
    gtk_widget_show(button);

    row++;
    button = hildon_button_new_with_text(HILDON_SIZE_FINGER_HEIGHT,
					 HILDON_BUTTON_ARRANGEMENT_HORIZONTAL,
					 gettext(labels[TRIP_EDIT_SHOWONMAP]),
					 NULL);
    gtk_table_attach(GTK_TABLE(table), button, 0, 2, row, row+1,
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    g_signal_connect (GTK_OBJECT (button), "clicked",
		      G_CALLBACK(show_start_on_map), pui);
    gtk_widget_show(button);
  }
  gtk_widget_show(table);

  /* End place entry */
  frame=gtk_frame_new(gettext(headers[TRIP_HDR_ENDPLACE]));
  gtk_box_pack_start (GTK_BOX(svbox), frame, TRUE, TRUE, 5);
  gtk_widget_show(frame);

  table = gtk_table_new(3, 2, FALSE);
  gtk_container_add (GTK_CONTAINER(frame), table);

  row=0;
  pui->logendplaceentry=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_table_attach(GTK_TABLE(table), pui->logendplaceentry, 0, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->logendplaceentry);

  if (!add) {
    row++;
    button = hildon_button_new_with_text(HILDON_SIZE_FINGER_HEIGHT,
					 HILDON_BUTTON_ARRANGEMENT_HORIZONTAL,
					 gettext(labels[TRIP_EDIT_ADDTOLOCATIONS]),
					 NULL);
    gtk_table_attach(GTK_TABLE(table), button, 0, 1, row, row+1,
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    g_signal_connect (GTK_OBJECT (button), "clicked",
		      G_CALLBACK(end_to_cache), pui);
    gtk_widget_show(button);

    button = hildon_button_new_with_text(HILDON_SIZE_FINGER_HEIGHT,
					 HILDON_BUTTON_ARRANGEMENT_HORIZONTAL,
					 gettext(labels[TRIP_EDIT_GETFROMLOCATIONS]),
					 NULL);
    gtk_table_attach(GTK_TABLE(table), button, 1, 2, row, row+1,
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    g_signal_connect (GTK_OBJECT (button), "clicked",
		      G_CALLBACK(end_from_cache), pui);
    gtk_widget_show(button);

    row++;
    button = hildon_button_new_with_text(HILDON_SIZE_FINGER_HEIGHT,
					 HILDON_BUTTON_ARRANGEMENT_HORIZONTAL,
					 gettext(labels[TRIP_EDIT_GETFROMNETWORK]),
					 NULL);
    gtk_table_attach(GTK_TABLE(table), button, 0, 2, row, row+1,
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    g_signal_connect (GTK_OBJECT (button), "clicked",
		      G_CALLBACK(end_from_network), pui);
    gtk_widget_show(button);

    row++;
    button = hildon_button_new_with_text(HILDON_SIZE_FINGER_HEIGHT,
					 HILDON_BUTTON_ARRANGEMENT_HORIZONTAL,
					 gettext(labels[TRIP_EDIT_SHOWONMAP]),
					 NULL);
    gtk_table_attach(GTK_TABLE(table), button, 0, 2, row, row+1,
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    g_signal_connect (GTK_OBJECT (button), "clicked",
		      G_CALLBACK(show_end_on_map), pui);
    gtk_widget_show(button);
  }
  gtk_widget_show(table);

  /* Distance entry */
  frame=gtk_frame_new(gettext(headers[TRIP_HDR_DISTANCE]));
  gtk_box_pack_start (GTK_BOX(svbox), frame, TRUE, TRUE, 5);
  gtk_widget_show(frame);

  row++;
  pui->logdistanceentry=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_container_add (GTK_CONTAINER(frame), pui->logdistanceentry);
  gtk_widget_show(pui->logdistanceentry);

  gtk_widget_show(svbox);
  gtk_box_pack_start (GTK_BOX(GTK_DIALOG (dialog)->vbox), scrollwin, TRUE, TRUE, 0);
  gtk_widget_show(scrollwin);
  hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA(scrollwin),svbox);

#else /* MAEMO_VERSION_MAJOR != 5 */

/** \todo: Maemo 4 version of trip editing window */

#endif

}


/**
 * \fn void callback_edittrip( GtkAction * action, AppUIData *pui )
 * \brief Callback for editing a record
 *
 * This callback creates the trip editing dialog
 *
 * \todo This should be changed so that fuelpad.c has a function for
 * obtaining the record data (i.e. sqlite would not be used here)
 *
 */
static
void callback_edittrip( GtkAction * action, AppUIData *pui )
{
  GtkWidget *dialog;
  GtkWidget *label;
  gchar *header[] = {gettext_noop("Edit a trip")};

  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreeModel *model;

  char* starttime;
  char* endtime;
  char* startplace;
  char* endplace;
  char* explanation;
  double distance;
  double startlat;
  double startlon;
  double endlat;
  double endlon;
  char* effectivetime;
  GString *dist;

  if (db_open()) {
    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->logview));
    if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
      dialog = gtk_dialog_new_with_buttons(gettext(header[0]),
					   GTK_WINDOW(pui->app->drivinglogwindow),
					   GTK_DIALOG_MODAL,
					   GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
					   GTK_STOCK_CANCEL,
					   GTK_RESPONSE_REJECT,
					   NULL);
      create_trip_editwin(pui, dialog, 0);

      /* Note: data in model is already in user units, no need to use conversion functions */
      gtk_tree_model_get(model, &iter, 
			 LOG_COL_STARTTIME, &starttime, LOG_COL_ENDTIME, &endtime, 
			 LOG_COL_STARTLAT, &startlat, LOG_COL_STARTLON, &startlon,
			 LOG_COL_STARTPLACE, &startplace,
			 LOG_COL_ENDLAT, &endlat, LOG_COL_ENDLON, &endlon,
			 LOG_COL_ENDPLACE, &endplace, LOG_COL_DISTANCE, &distance,
			 LOG_COL_EFFECTIVETIME, &effectivetime, LOG_COL_EXPLANATION, &explanation,
			 -1);

      dist=g_string_new(NULL);
      g_string_printf(dist,"%.1f",distance);

      gtk_entry_set_text(GTK_ENTRY(pui->logexlanationentry), explanation);
      gtk_entry_set_text(GTK_ENTRY(pui->logstartplaceentry), startplace);
      gtk_entry_set_text(GTK_ENTRY(pui->logendplaceentry), endplace);
      gtk_entry_set_text(GTK_ENTRY(pui->logdistanceentry), dist->str);

      g_string_free(dist, TRUE);

      g_free(starttime);
      g_free(endtime);
      g_free(startplace);
      g_free(endplace);
      g_free(explanation);
      g_free(effectivetime);

#if LIBHELP == 1
	  help_dialog_help_enable(GTK_DIALOG(dialog),
					 HELP_ID_ADDRECORD,
					 pui->app->osso);
#endif

	  g_signal_connect (dialog, "response",
			    G_CALLBACK (edit_trip_response),
			    &ui);
    }
    else {
      dialog = gtk_dialog_new_with_buttons(gettext(header[0]),
					   GTK_WINDOW(pui->app->drivinglogwindow),
					   GTK_DIALOG_MODAL,
					   GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
					   NULL);

      label=gtk_label_new(_("Select a trip first"));
      gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label, TRUE, TRUE, 5);
      gtk_widget_show(label);

      g_signal_connect_swapped (dialog, "response",
				G_CALLBACK (gtk_widget_destroy),
				dialog);
    }

  }
  else {
    dialog = gtk_dialog_new_with_buttons(gettext(header[0]),
					 GTK_WINDOW(pui->app->drivinglogwindow),
					 GTK_DIALOG_MODAL,
					 GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
					 NULL);

    label=gtk_label_new(_("Can't access database - editing records not possible"));
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label, TRUE, TRUE, 5);
    gtk_widget_show(label);

    /* Destroy the dialog when the user responds to it (e.g. clicks a button) */
    g_signal_connect_swapped (dialog, "response",
			      G_CALLBACK (gtk_widget_destroy),
			      dialog);
  }

  gtk_widget_show(dialog);
}

/**
 * \fn void callback_deletetrip( GtkAction * action, AppUIData *pui )
 * \brief Callback for deleting a trip
 *
 * This callback deletes the currently selected trip. The user is
 * asked for a confirmation.
 *
 */
static
void callback_deletetrip( GtkAction * action, AppUIData *pui )
{
  GtkWidget *confirmdlg;
  GtkWidget *separator;
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreeIter storeiter;
  GtkListStore *store;
  GtkTreeModel *model;
  gint64 id;
  gint confirm;

  if (db_open()) { /* delete */

    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->logview));
    if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
      gtk_tree_model_get (model, &iter, LOG_COL_ID, &id, -1);

#if PLAINGTK == 0
      confirmdlg = hildon_note_new_confirmation(GTK_WINDOW(pui->app->drivinglogwindow), 
						_("Are you sure you want to delete the selected trip"));
#else
      confirmdlg = gtk_message_dialog_new(GTK_WINDOW(pui->app->drivinglogwindow),
					  GTK_DIALOG_MODAL,
					  GTK_MESSAGE_QUESTION,
					  GTK_BUTTONS_OK_CANCEL,
					  _("Are you sure you want to delete the selected trip"));
      gtk_widget_show(confirmdlg);
#endif

      confirm = gtk_dialog_run(GTK_DIALOG(confirmdlg));
      gtk_widget_destroy(GTK_WIDGET(confirmdlg));

      if(confirm == GTK_RESPONSE_OK) {

	if (DB_DONE != db_delete_trip(id)) {
	  PDEBUG("Problem deleting the trip");
	}
	else { /* remove the record if succesfully removed from db */

	  get_store_and_iter(model, pui->logview, &iter, &store, &storeiter);
	  gtk_list_store_remove(store, &storeiter);

	}
      } /* if (confirm == GTK_RESPONSE_OK) */
    }
    else
      PDEBUG("Something wrong with finding the selection\n");
  }
  else { /* nothing to delete */
    PDEBUG("Database not open, can't delete anything\n");
  }
}

static
void callback_startgps( GtkAction * action, AppUIData *pui )
{
  ui_init_location(pui);
}

static
void callback_stopgps( GtkAction * action, AppUIData *pui )
{
  ui_stop_location();
}

#if PLAINGTK == 1
static void create_secondary_toolbar(GtkWidget * main_window, GtkWidget *vbox, AppUIData *pui)
#else
  static void create_secondary_toolbar(HildonWindow * main_window, AppUIData *pui)
#endif
{
  GtkWidget *toolbar;

  PangoAttrList *attrs;
  PangoAttribute *attr;

  /* Create toolbar */
  toolbar = gtk_toolbar_new();

  /* Create toolbar items */

  /* Scale texts smaller: create pango attributes */
  attrs = pango_attr_list_new();
  attr = pango_attr_scale_new(PANGO_SCALE_X_SMALL);
  attr->start_index = 0;
  attr->end_index = -1;
  pango_attr_list_insert(attrs, attr);

  /* Total distance */
  pui->stb_logdistance = gtk_tool_item_new();
  pui->stb_logdistancelabel=gtk_label_new("Status:");
  gtk_label_set_selectable(GTK_LABEL(pui->stb_logdistancelabel), TRUE);
  gtk_label_set_attributes(GTK_LABEL(pui->stb_logdistancelabel), attrs);
  gtk_container_add(GTK_CONTAINER(pui->stb_logdistance), GTK_WIDGET(pui->stb_logdistancelabel));

  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->stb_logdistance),TRUE);

  /* Add all items to toolbar */
  gtk_toolbar_insert(GTK_TOOLBAR(toolbar), pui->stb_logdistance, -1);

  /* Add toolbar HildonWindow */
#if PLAINGTK == 1
  gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(toolbar), FALSE, FALSE, 5);
#else
  hildon_window_add_toolbar(main_window, GTK_TOOLBAR(toolbar));
#endif
}

#if PLAINGTK == 1
static void create_logwin_menu(GtkWidget *window, GtkWidget *root, AppUIData *pui)
#else
static void create_logwin_menu(HildonWindow *window, AppUIData *pui)
#endif
{
#if MAEMO_VERSION_MAJOR == 5
  HildonAppMenu *main_menu;
#else
  GtkWidget *main_menu;
#endif
  GtkWidget *item_edittrip;
  GtkWidget *item_deletetrip;
  GtkWidget *item_editlocations;
  GtkWidget *item_startgps;
  GtkWidget *item_stopgps;
#if PLAINGTK == 1
  /** \todo Driving log menus for PLAINGTK */
#endif

#if MAEMO_VERSION_MAJOR == 5
  GtkWidget * button;

  main_menu = HILDON_APP_MENU (hildon_app_menu_new ());

  item_edittrip = hildon_button_new_with_text(HILDON_SIZE_AUTO,
					      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
					      _("Edit trip"),
					      NULL);
  hildon_app_menu_append (main_menu, GTK_BUTTON (item_edittrip));

  item_deletetrip = hildon_button_new_with_text(HILDON_SIZE_AUTO,
					      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
					      _("Delete trip"),
					      NULL);
  hildon_app_menu_append (main_menu, GTK_BUTTON (item_deletetrip));

  item_editlocations = hildon_button_new_with_text(HILDON_SIZE_AUTO,
					      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
					      _("Edit locations"),
					      NULL);
  hildon_app_menu_append (main_menu, GTK_BUTTON (item_editlocations));

  item_startgps = hildon_button_new_with_text(HILDON_SIZE_AUTO,
					      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
					      _("Start GPS"),
					      NULL);
  hildon_app_menu_append (main_menu, GTK_BUTTON (item_startgps));

  item_stopgps = hildon_button_new_with_text(HILDON_SIZE_AUTO,
					      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
					      _("Stop GPS"),
					      NULL);
  hildon_app_menu_append (main_menu, GTK_BUTTON (item_stopgps));


  g_signal_connect(G_OBJECT(item_edittrip), "clicked",
		   G_CALLBACK(callback_edittrip), pui);
  g_signal_connect(G_OBJECT(item_deletetrip), "clicked",
		   G_CALLBACK(callback_deletetrip), pui);
  g_signal_connect(G_OBJECT(item_editlocations), "clicked",
		   G_CALLBACK(callback_editlocations), pui);
  g_signal_connect(G_OBJECT(item_startgps), "clicked",
		   G_CALLBACK(callback_startgps), pui);
  g_signal_connect(G_OBJECT(item_stopgps), "clicked",
		   G_CALLBACK(callback_stopgps), pui);

  hildon_window_set_app_menu(HILDON_WINDOW(window), main_menu);

#else

  /** \todo: Driving main menus for Maemo 4 */

#endif

  /* Make all menu widgets visible */
  gtk_widget_show_all(GTK_WIDGET(main_menu));
}

/*******************************************************************
 * Log control functions
 *******************************************************************/

static
time_t get_effective_time(void)
{
  time_t efftime;

  if (driving_log_logging() || driving_log_paused()) {
    if (driving_log_paused()) {
      efftime = currenttrip.pausestart-currenttrip.starttime;
    }
    else {
      efftime = getmktime()-currenttrip.starttime-currenttrip.pausetime;
    }
  }
  else {
    efftime = 0;
  }

  return efftime;
}

static
void init_trip(void)
{
  currenttrip.distance = 0.0;
  currenttrip.starttime = 0;
  currenttrip.lat_s = 0.0;
  currenttrip.lon_s = 0.0;
  if (currenttrip.startplace != NULL) {
    free(currenttrip.startplace);
    currenttrip.startplace = NULL;
  }
  currenttrip.endtime = 0;
  currenttrip.lat_e = 0.0;
  currenttrip.lon_e = 0.0;
  if (currenttrip.endplace != NULL) {
    free(currenttrip.endplace);
    currenttrip.endplace = NULL;
  }
  currenttrip.effectivetime = 0;
  currenttrip.pausetime = 0;
  currenttrip.pausestart = 0;
  if (currenttrip.explanation != NULL) {
    free(currenttrip.explanation);
    currenttrip.explanation = NULL;
  }
}

static
void trip_save_start_point(time_t time, double lat, double lon)
{
  gboolean foundincache;
  char *address;

  currenttrip.starttime = time;
  currenttrip.lat_s = lat;
  currenttrip.lon_s = lon;

  /* Find the location from locationalias */
  foundincache = get_from_locations_cache(lat, lon, &address);

  if (foundincache) {
    currenttrip.startplace = address;
  }
  else {
    (void)ui_connection_ensure();
    g_idle_add(reverse_geocode_idle, (gpointer)LOG_START);	
  }
}

static
void trip_save_end_point(time_t time, double lat, double lon)
{
  gboolean foundincache;
  char *address;

  currenttrip.endtime = time;
  currenttrip.lat_e = lat;
  currenttrip.lon_e = lon;
  currenttrip.distance = distance/1000; /* Saved in kilometres */

  /* Find the location from locationalias */
  foundincache = get_from_locations_cache(lat, lon, &address);

  if (foundincache) {
    currenttrip.endplace = address;
  }
  else {
    (void)ui_connection_ensure();
    currenttrip.endplace = reverse_geocode(lat, lon); 
  }
}

static
void trip_debug_show_data(void)
{
  char *aika;

  unixtimefmt(currenttrip.starttime, &aika, "%Y-%m-%dT%H:%M.%S");
  g_print("Trip started %s\n",aika);
  free(aika);

  g_print("from %s (lat %f, lon %f)\n",
	  currenttrip.startplace, 
	  currenttrip.lat_s,
	  currenttrip.lon_s);

  unixtimefmt(currenttrip.endtime, &aika, "%Y-%m-%dT%H:%M.%S");
  g_print("Trip ended %s\n",aika);
  free(aika);

  g_print("to %s (lat %f, lon %f)\n",
	  currenttrip.endplace, 
	  currenttrip.lat_e,
	  currenttrip.lon_e);
  g_print("Distance: %.3f metres\n", distance);

}

static
gboolean toggle_log_paused(AppUIData *pui)
{
  gps_set_logging(!driving_log_logging());
  paused = !paused;
  update_start_button_icon(pui);
  if (paused) {
    currenttrip.pausestart=getmktime();
  }
  else {
    currenttrip.pausetime=getmktime()-currenttrip.pausestart;
  }

  return paused;
}

static
void driving_log_clear( GtkAction * action, AppUIData *pui )
{
  GtkWidget *confirmdlg;
  gint confirm;

#if PLAINGTK == 0
  confirmdlg = hildon_note_new_confirmation(GTK_WINDOW(pui->app->drivinglogwindow), 
					    _("Are you sure you want to clear the distance?"));
#else
  confirmdlg = gtk_message_dialog_new(GTK_WINDOW(pui->app->drivinglogwindow),
				      GTK_DIALOG_MODAL,
				      GTK_MESSAGE_QUESTION,
				      GTK_BUTTONS_OK_CANCEL,
				      _("Are you sure you want to clear the distance?"));
  gtk_widget_show(confirmdlg);
#endif
  confirm = gtk_dialog_run(GTK_DIALOG(confirmdlg));
  gtk_widget_destroy(GTK_WIDGET(confirmdlg));
  if(confirm == GTK_RESPONSE_OK) {
    distance = 0.0;
    ui_show_banner(_("Distance cleared."));
    g_idle_add(update_distance_ui, pui); 
  }
}


static
void driving_log_pause( GtkAction * action, AppUIData *pui )
{
  if (toggle_log_paused(pui)) {
    ui_show_banner(_("Pausing logging"));
  }
  else {
    ui_show_banner(_("Continuing logging"));
  }
}

static
void driving_log_start( GtkAction * action, AppUIData *pui )
{
  time_t aika;
  char aikastr[22];

  if (driving_log_logging() || driving_log_paused()) {
    driving_log_pause(action, pui);
  }
  else {
    ui_init_location(pui);

    init_trip();
    trip_save_start_point(getmktime(), device->fix->latitude, device->fix->longitude);

    distance=0.0;
    gps_set_logging(TRUE);
    paused=FALSE;

    ui_show_banner(_("Logging started."));
    update_start_button_icon(pui);
  }
}

static
void driving_log_stop( GtkAction * action, AppUIData *pui )
{
  GtkWidget *dialog;
  gchar *header[] = {gettext_noop("Add a trip")};
  GString *str;
  GString *dist;

  GtkWidget *confirmdlg;
  gint confirm;

  if (driving_log_logging() || driving_log_paused()) {
#if PLAINGTK == 0
    confirmdlg = hildon_note_new_confirmation(GTK_WINDOW(pui->app->drivinglogwindow), 
					    _("Are you sure you want stop the trip?"));
#else
    confirmdlg = gtk_message_dialog_new(GTK_WINDOW(pui->app->drivinglogwindow),
					GTK_DIALOG_MODAL,
					GTK_MESSAGE_QUESTION,
					GTK_BUTTONS_OK_CANCEL,
					_("Are you sure you want stop the trip?"));
    gtk_widget_show(confirmdlg);
#endif
    confirm = gtk_dialog_run(GTK_DIALOG(confirmdlg));
    gtk_widget_destroy(GTK_WIDGET(confirmdlg));
    if(confirm == GTK_RESPONSE_OK) {
      if (driving_log_paused()) {
	toggle_log_paused(pui);
      }
      str = g_string_new(NULL);
      g_string_printf(str,_("Stopping GPS. Distance traveled %.1f km"),distance/1000.0);
      ui_show_banner(str->str);
      g_string_free(str, TRUE);
      
      trip_save_end_point(getmktime(), device->fix->latitude, device->fix->longitude);

      ui_stop_location();
      gps_set_logging(FALSE);
      update_start_button_icon(pui);

      dialog = gtk_dialog_new_with_buttons(gettext(header[0]),
					   GTK_WINDOW(pui->app->drivinglogwindow),
					   GTK_DIALOG_MODAL,
					   GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
					   GTK_STOCK_CANCEL,
					   GTK_RESPONSE_REJECT,
					   NULL);
      create_trip_editwin(pui, dialog, 1);

      dist=g_string_new(NULL);
      g_string_printf(dist,"%.1f", currenttrip.distance);

      gtk_entry_set_text(GTK_ENTRY(pui->logstartplaceentry), currenttrip.startplace);
      gtk_entry_set_text(GTK_ENTRY(pui->logendplaceentry), currenttrip.endplace);
      gtk_entry_set_text(GTK_ENTRY(pui->logdistanceentry), dist->str);

      g_string_free(dist, TRUE);

      g_signal_connect (dialog, "response",
			G_CALLBACK (add_trip_response),
			&ui);

      gtk_widget_show(dialog);

    }
  }
}

/*******************************************************************
 *
 * Public functions
 *
 *******************************************************************/

gboolean driving_log_logging(void)
{
  return gps_get_logging();
}

gboolean driving_log_paused(void)
{
  return paused;
}

void driving_log_callback( GtkAction * action, AppUIData *pui )
{
  GtkWidget *vbox;
  GtkWidget *table;
  GtkWidget *button;
  GtkWidget *image;
  GtkWidget *scrollwin;

  if (autostartGPS) {
    ui_init_location(pui);
  }

  connect_init();

  if (autostartnetwork) {
    if (!connect_request()) {
      g_print("Connection request failed\n");
    }
  }

  if (pui->app->drivinglogwindow != NULL) {
    gtk_window_present(GTK_WINDOW(pui->app->drivinglogwindow));
    return;
  }
#if PLAINGTK == 1
  pui->app->drivinglogwindow=gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(pui->app->drivinglogwindow), "Driving log");
  gtk_widget_set_usize(pui->app->drivinglogwindow,800,400);
  gtk_container_border_width(GTK_CONTAINER(pui->app->drivinglogwindow),0);
#else
#if MAEMO_VERSION_MAJOR == 5
  pui->app->drivinglogwindow = hildon_stackable_window_new();
#else
  pui->app->drivinglogwindow = HILDON_WINDOW(hildon_window_new());
#endif
  hildon_program_add_window(pui->app->program, pui->app->drivinglogwindow);
#endif

  g_signal_connect(G_OBJECT(pui->app->drivinglogwindow), "destroy",
		   G_CALLBACK(destroy_logwin), pui);

  vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (pui->app->drivinglogwindow), vbox);

  table = gtk_table_new(2, 2, FALSE);
  gtk_box_pack_start (GTK_BOX(vbox), table, FALSE, FALSE, 0);

#if MAEMO_VERSION_MAJOR == 5
  pui->logstartbutton = hildon_button_new(HILDON_SIZE_FINGER_HEIGHT,
					  HILDON_BUTTON_ARRANGEMENT_HORIZONTAL);
  hildon_button_set_image_position (HILDON_BUTTON (pui->logstartbutton), GTK_POS_LEFT);
  update_start_button_icon(pui);
  gtk_table_attach(GTK_TABLE(table), pui->logstartbutton, 0, 1, 0, 1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  g_signal_connect (GTK_OBJECT (pui->logstartbutton), "clicked",
		    G_CALLBACK(driving_log_start), &ui);
  gtk_widget_show(pui->logstartbutton);

  button = hildon_button_new_with_text(HILDON_SIZE_FINGER_HEIGHT,
				       HILDON_BUTTON_ARRANGEMENT_HORIZONTAL,
				       _("Stop"),
				       NULL);
  image = gtk_image_new_from_icon_name ("camera_video_stop", GTK_ICON_SIZE_BUTTON);
  hildon_button_set_image (HILDON_BUTTON (button), image);
  hildon_button_set_image_position (HILDON_BUTTON (button), GTK_POS_LEFT);
  gtk_table_attach(GTK_TABLE(table), button, 1, 2, 0, 1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  g_signal_connect (GTK_OBJECT (button), "clicked",
		    G_CALLBACK(driving_log_stop), &ui);
  gtk_widget_show(button);

#else
  /** \todo Maemo 4 UI for driving log */
#endif

#if MAEMO_VERSION_MAJOR == 5
  scrollwin = hildon_pannable_area_new();
  g_object_set(G_OBJECT(scrollwin), "mov-mode", HILDON_MOVEMENT_MODE_BOTH, NULL);
#else
  scrollwin = gtk_scrolled_window_new(NULL, NULL);
#endif
  gtk_box_pack_start (GTK_BOX(vbox), scrollwin, TRUE, TRUE, 0);

  pui->logview = create_logview_and_model ();
  gtk_container_add (GTK_CONTAINER (scrollwin), pui->logview);
  gtk_widget_show(pui->logview);
  gtk_widget_show(scrollwin);

  gtk_widget_show_all(GTK_WIDGET(vbox));

#if PLAINGTK == 1
  create_logwin_menu(pui->app->drivinglogwindow, vbox, pui);
#else
  create_logwin_menu(pui->app->drivinglogwindow, pui);
#endif

#if PLAINGTK == 1
   create_secondary_toolbar(pui->app->drivinglogwindow, vbox, pui);
#else
   create_secondary_toolbar(pui->app->drivinglogwindow, pui);
#endif

  gtk_widget_show_all(GTK_WIDGET(pui->app->drivinglogwindow));

  if(!gnome_vfs_init()) {
    g_print("Failed to initialize GnomeVFS-libraries, exiting\n");
  }

}
