/* 
 * This file is part of fuelpad, the fuel diary
 *
 * Copyright (c) 2007-2010 Julius Luukko <julle.luukko@quicknet.inet.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

/**
 *
 * \file
 * \brief  GTK+/Hildon based user interface to fuelpad
 * \author Julius Luukko
 *
 *
 */

#include "ui.h"
#include "uiview.h"
#include "uisettings.h"
#include "uiconfig.h"
#include "uifilter.h"
#include "uireport.h"
#include "uistat.h"
#include "uiexpimp.h"
#include "uialarm.h"
#include "uidrivinglog.h"
#include "uiconnection.h"
#include <assert.h>
#include <math.h>
#include <libgnomevfs/gnome-vfs.h>


/*******************************************************************
 *
 * Private type definition
 *
 *******************************************************************/

typedef struct {
  double km;
  double trip;
  double fill;
  double price;
  double service;
  double oil;
  double tires;
} entry_t;


/*******************************************************************
 *
 * Public variables
 *
 *******************************************************************/

AppData app;
AppUIData ui;

gboolean carchanged=FALSE;

/* Font scale factors corresponding to enum fontscale */
double fontscalefactors[]={PANGO_SCALE_MEDIUM,
			   PANGO_SCALE_X_SMALL,
			   PANGO_SCALE_SMALL,
			   PANGO_SCALE_MEDIUM,
			   PANGO_SCALE_LARGE};

/**********************************************************************
 *
 * Variables stored in gconf
 *
 *******************************************************************/

/* Main table font size */
enum fontscale fontsize=MEDIUM;

/* Toolbar visibility */
gboolean main_toolbar_visible;
gboolean secondary_toolbar_visible;

/* Main table sort order */
gboolean maintablesorted=TRUE;
gint maintablesortcol=COL_DAY;
GtkSortType maintablesortorder=GTK_SORT_ASCENDING;

/**********************************************************************/



/*******************************************************************
 *
 * Private variables
 *
 *******************************************************************/

enum dbtimespan stbstattime=OVERALL;

/*******************************************************************
 *
 * Private function prototypes
 *
 *******************************************************************/
#if PLAINGTK == 0
static
gint dbus_callback (const gchar *interface, const gchar *method,
		    GArray *arguments, gpointer data,
		    osso_rpc_t *retval);
static void osso_hw_state_callback(osso_hw_state_t *state, gpointer data);
#if MAEMO_VERSION_MAJOR == 5
static DBusHandlerResult
mce_filter_func (DBusConnection * connection,
                 DBusMessage * message, AppData *app);
#endif // MAEMO_VERSION_MAJOR == 5
#endif // PLAINGTK == 0
static
gboolean delete_event(GtkWidget *widget, GdkEvent *event, gpointer data);
static
void main_window_fullscreen(gpointer data);
static
void cb_main_fullscreen( GtkAction * action, gpointer data );
#if PLAINGTK == 0
static
gboolean key_press_cb(GtkWidget * widget, GdkEventKey * event,
                      AppUIData *pui);
#endif
static
void cb_main_toolbar( GtkAction * action, AppUIData *pui);
static
void cb_secondary_toolbar( GtkAction * action, AppUIData *pui);
static
void update_font_scale(GtkTreeView *view);
static
void cb_fontsize_x_small( GtkAction * action, AppUIData *pui);
static
void cb_fontsize_small( GtkAction * action, AppUIData *pui);
static
void cb_fontsize_medium( GtkAction * action, AppUIData *pui);
static
void cb_fontsize_large( GtkAction * action, AppUIData *pui);
static
void driver_changed_cb (GtkWidget *widget, gpointer   callback_data );
static
void car_changed_cb (GtkWidget *widget, gpointer   callback_data );
static
void car_apply_cb (GtkWidget *widget, AppUIData *pui );
static
void fill_driver_combo(GtkWidget *combo);
static
void fill_car_combo(GtkWidget *combo);
GtkTreeModel *get_filter(GtkWidget *view);
gboolean ui_find_iter(GtkListStore *store,
		      GtkTreeIter *iter,
		      sqlite_int64 id);
void ui_update_row_data(GtkListStore  *store,
			GtkTreeIter   *iter,
			const char *date,
			double km,
			double trip,
			double fill,
			double consum,
			double price,
			double priceperlitre,
			double pricepertrip,
			double service,
			double oil,
			double tires,
			const char *notes,
			sqlite_int64 id,
			gboolean visible);
void update_totalkm(AppUIData *pui);
#if PLAINGTK == 0
static HildonFileSystemModel* get_file_system_model(GtkWidget *ref_widget);
#endif
static
void internal_close_db(AppUIData *pui);
static
void callback_close_db( GtkAction * action, AppUIData *pui );
static
void callback_open_db( GtkAction * action, AppUIData *pui );
static
void add_record_response (GtkDialog *dialog, gint arg1, AppUIData *pui);
static
void edit_record_response (GtkDialog *dialog, gint arg1, AppUIData *pui);
static
gboolean callback_kmadded (GtkWidget     *widget,
                           GdkEventFocus *event,
                           AppUIData     *pui);
static
gboolean callback_tripadded (GtkWidget     *widget,
			     GdkEventFocus *event,
			     AppUIData     *pui);
static
void create_record_editwin(AppUIData *pui, GtkWidget *dialog, gint add);
static
void callback_newrecord( GtkAction * action, AppUIData *pui );
static
void callback_prevrecord( GtkAction * action, AppUIData *pui );
static
void callback_nextrecord( GtkAction * action, AppUIData *pui );
#if PLAINGTK == 1
static void create_mainwin_toolbar(GtkWidget * main_window, GtkWidget *vbox, AppUIData *pui);
#else
static void create_mainwin_toolbar(HildonWindow * main_window, AppUIData *pui);
#endif
#if PLAINGTK == 1
static void create_secondary_toolbar(GtkWidget * main_window, GtkWidget *vbox, AppUIData *pui);
#else
static void create_secondary_toolbar(HildonWindow * main_window, AppUIData *pui);
#endif
static
void enable_mainmenu_items(AppUIData *pui);
static
void create_mainwin_menu(GtkWidget *root, AppUIData *pui);

/*******************************************************************
 *
 * Private functions
 *
 *******************************************************************/

#if PLAINGTK == 0
static
gint dbus_callback (const gchar *interface, const gchar *method,
		    GArray *arguments, gpointer data,
		    osso_rpc_t *retval)
{
  PDEBUG ("fuelpad dbus: %s, %s\n", interface, method);

  if (!strcmp (method, "top_application"))
      gtk_window_present (GTK_WINDOW (data));

  retval->type = DBUS_TYPE_INVALID;
  return OSSO_OK;
}

/**
 * \fn void osso_hw_state_callback(osso_hw_state_t *state, gpointer data)
 * \brief Osso hardware state callback
 *
 * This functions is a callback for hardware state changes. It
 * implements only the shutdown_ind state, which calls fuelpad_exit()
 * and exits with exit(1).
 *
 */
static void osso_hw_state_callback(osso_hw_state_t *state, gpointer data)
{
  if(state->shutdown_ind) {
    fuelpad_exit();
    exit(1);
  }
}

#if MAEMO_VERSION_MAJOR == 5
static DBusHandlerResult
mce_filter_func (DBusConnection * connection,
                 DBusMessage * message, AppData *app)
{
  DBusError error;
  dbus_error_init (&error);

  if (dbus_message_is_signal (message, MCE_SIGNAL_IF,
                              MCE_DEVICE_ORIENTATION_SIG))
  {
    char *rotation, *stand, *face;
    int x, y, z;

    /* Retrieve arguments from a message */
    if (!dbus_message_get_args (message,
                                &error,
                                DBUS_TYPE_STRING, &rotation,
                                DBUS_TYPE_STRING, &stand,
                                DBUS_TYPE_STRING, &face,
                                DBUS_TYPE_INT32, &x,
                                DBUS_TYPE_INT32, &y,
                                DBUS_TYPE_INT32, &z, DBUS_TYPE_INVALID))
    {
      g_printerr ("DBus error: %s (%s)\n", error.name, error.message);
      dbus_error_free (&error);
    }
    else
    {
      /* Rotate main window */
      if (hildon_window_get_is_topmost(HILDON_WINDOW(app->mainwindow)))
	if (!strcmp (rotation, MCE_ORIENTATION_PORTRAIT))
	  hildon_gtk_window_set_portrait_flags (GTK_WINDOW(app->mainwindow), HILDON_PORTRAIT_MODE_REQUEST);
	else hildon_gtk_window_set_portrait_flags (GTK_WINDOW(app->mainwindow),  ~HILDON_PORTRAIT_MODE_REQUEST);
    }
  }

  return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

}
#endif

#endif

/**
 * \fn gboolean delete_event( GtkWidget *widget, GdkEvent  *event, gpointer   data )
 * \brief Main window delete_event callback
 * \return FALSE
 */
static
gboolean delete_event( GtkWidget *widget,
		       GdkEvent  *event,
		       gpointer   data )
{
  GtkWidget *confirmdlg;
  gint confirm;
  AppUIData *pui;

  pui = (AppUIData *)data;

  if (driving_log_logging() || driving_log_paused()) {
#if PLAINGTK == 0
    confirmdlg = hildon_note_new_confirmation(GTK_WINDOW(pui->app->drivinglogwindow), 
					      _("Driving log is active. Are you sure you want to quit?"));
#else
    confirmdlg = gtk_message_dialog_new(GTK_WINDOW(pui->app->drivinglogwindow),
					GTK_DIALOG_MODAL,
					GTK_MESSAGE_QUESTION,
					GTK_BUTTONS_OK_CANCEL,
					_("Driving log is active. Are you sure you want to quit?"));
    gtk_widget_show(confirmdlg);
#endif
    confirm = gtk_dialog_run(GTK_DIALOG(confirmdlg));
    gtk_widget_destroy(GTK_WIDGET(confirmdlg));
    if(confirm != GTK_RESPONSE_OK) {
      return TRUE;
    }
  }

  gtk_main_quit ();
#if PLAINGTK == 0

#if MAEMO_VERSION_MAJOR == 5
  if (osso_rpc_run_system(app.osso, MCE_SERVICE, MCE_REQUEST_PATH,
          MCE_REQUEST_IF, MCE_ACCELEROMETER_DISABLE_REQ, NULL, DBUS_TYPE_INVALID) == OSSO_OK) {
	g_printerr("INFO: Accelerometer disabled\n");
  } else {
        g_printerr("WARN: Cannot disable accelerometer\n");
  }

  /* Remove match rule and filter, finalize connection */
  if (app.system_bus) {
    if (dbus_connection_get_is_connected (app.system_bus))
      dbus_bus_remove_match (app.system_bus, MCE_SIGNAL_MATCH, NULL);
    dbus_connection_remove_filter (app.system_bus,
                                   (DBusHandleMessageFunction) mce_filter_func,
                                   &app);

    dbus_connection_unref(app.system_bus);
    app.system_bus = NULL;
  }
#endif

  osso_deinitialize(app.osso);
#endif
  return FALSE;
}

/**
 * \fn void main_window_fullscreen(gpointer window)
 * \brief Puts the window to full screen mode and scales if necessary
 *
 * Calls gtk_window_fullscreen() or gtk_window_unfullscreen(). If
 * statistics window is present, scale the canvas.
 *
 */
static
void main_window_fullscreen(gpointer data)
{
  AppUIData *a;

  a=(AppUIData *)data;
  a->mainfullscreen = !a->mainfullscreen;
  if (a->mainfullscreen) {
    gtk_window_fullscreen(GTK_WINDOW(a->app->mainwindow));
  } else {
    gtk_window_unfullscreen(GTK_WINDOW(a->app->mainwindow));
  }
}

static
void cb_main_fullscreen( GtkAction * action, gpointer data )
{
  main_window_fullscreen(data);
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(((AppUIData *)(data))->main_menu_item_fullscreen),
				 ((AppUIData *)(data))->mainfullscreen);
}

/* Callback for hardware keys */
/* See maemo tutorial */
#if PLAINGTK == 0
static
gboolean key_press_cb(GtkWidget * widget, GdkEventKey * event,
                      AppUIData *pui)
{
  switch (event->keyval) {

    case GDK_F6:
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pui->main_menu_item_fullscreen),
				     !pui->mainfullscreen);
      return TRUE;
  }

  return FALSE;
}
#endif

static
void cb_main_toolbar( GtkAction * action, AppUIData *pui)
{
  main_toolbar_visible = !main_toolbar_visible;
  main_toolbar_show_hide(pui);
}

static
void cb_secondary_toolbar( GtkAction * action, AppUIData *pui)
{
  secondary_toolbar_visible = !secondary_toolbar_visible;
  secondary_toolbar_show_hide(pui);
}

/* void update_font_scale(AppUIData *pui) */
static
void update_font_scale(GtkTreeView *view)
{
  gint i;
  GtkTreeViewColumn   *col;
  GtkWidget           *label;
  PangoAttrList       *attrs;
  PangoAttribute      *attr;
  GList               *rendererlist;

  for (i=COL_DAY; i<COL_ID; i++) {
    col=gtk_tree_view_get_column(view, i);
    label = gtk_tree_view_column_get_widget(col); 
    attrs=gtk_label_get_attributes(GTK_LABEL(label));
    attr = pango_attr_scale_new(fontscalefactors[fontsize]);
    attr->start_index = 0;
    attr->end_index = -1;
    pango_attr_list_change(attrs, attr);

    rendererlist=gtk_tree_view_column_get_cell_renderers(col);
    g_object_set(rendererlist->data, "scale", fontscalefactors[fontsize], NULL);
    g_list_free(rendererlist);
  }
}

static
void cb_fontsize_x_small( GtkAction * action, AppUIData *pui)
{
  fontsize=XSMALL;
  update_font_scale(GTK_TREE_VIEW(pui->view));
  update_record_model(pui);
}

static
void cb_fontsize_small( GtkAction * action, AppUIData *pui)
{
  fontsize=SMALL;
  update_font_scale(GTK_TREE_VIEW(pui->view));
}

static
void cb_fontsize_medium( GtkAction * action, AppUIData *pui)
{
  fontsize=MEDIUM;
  update_font_scale(GTK_TREE_VIEW(pui->view));
}

static
void cb_fontsize_large( GtkAction * action, AppUIData *pui)
{
  fontsize=LARGE;
  update_font_scale(GTK_TREE_VIEW(pui->view));
}

/**
 * \fn void driver_changed_cb (GtkWidget *widget, gpointer   callback_data )
 * \brief Callback for driver selection combo box
 *
 */
static
void driver_changed_cb (GtkWidget *widget, gpointer   callback_data )
{
  unsigned int index;
  unsigned int i;

#if MAEMO_VERSION_MAJOR == 5
  index=hildon_touch_selector_get_active(HILDON_TOUCH_SELECTOR(widget), 0);
#else
  index = gtk_combo_box_get_active(GTK_COMBO_BOX(widget));
#endif
  currentdriver=db_get_driverid(index);
}

/**
 * \fn void car_changed_cb (GtkWidget *widget, gpointer   callback_data )
 * \brief Callback for car selection combo box
 *
 * This callback creates the contents of the record table after car
 * has been changed.
 *
 */
static
void car_changed_cb (GtkWidget *widget, gpointer   callback_data )
{
  unsigned int index;
  unsigned int i;

#if MAEMO_VERSION_MAJOR == 5
  index=hildon_touch_selector_get_active(HILDON_TOUCH_SELECTOR(widget), 0);
#else
  index=gtk_combo_box_get_active(GTK_COMBO_BOX(widget));
#endif
  currentcar=db_get_carid(index);

  carchanged=TRUE;
}


static
void car_apply_cb (GtkWidget *widget, AppUIData *pui )
{
  gboolean warn;

  update_car_changed(pui);

  /* Update the next event estimates */
  pui->alarmview = create_alarmview_and_model (pui);
  pui->warn = update_next_event(gtk_tree_view_get_model(GTK_TREE_VIEW(pui->alarmview)));
  update_reminder_toolbutton (pui, pui->warn);
  /* Show or hide toolbars */
  main_toolbar_show_hide(pui);
  secondary_toolbar_show_hide(pui);
}

static
void fill_driver_combo(GtkWidget *combo)
{
  gint i;
  gint active;
  gint64 id;

  active=0;
  i=0;
  if (db_open()) {

    /* We'll need to block, otherwise driver_changed_cb would be fired in between */
    g_signal_handlers_block_by_func(combo, G_CALLBACK(driver_changed_cb), NULL);
    if (DB_OK==db_init_driver_query()) {
      while(DB_ROW == db_step_driver_query()) {
#if MAEMO_VERSION_MAJOR == 5
	hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (combo), db_driver_column_text(0));
#else
	gtk_combo_box_append_text(GTK_COMBO_BOX (combo),db_driver_column_text(0));
#endif
	id=db_driver_column_int64(2);
	if (id==currentdriver)
	  active=i;
	i++;
      }
    }
    g_signal_handlers_unblock_by_func(combo, G_CALLBACK(driver_changed_cb), NULL);
    db_reset_driver_query();

#if MAEMO_VERSION_MAJOR == 5
    hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (combo), 0, active);
#else
    gtk_combo_box_set_active(GTK_COMBO_BOX (combo),active);
#endif
  }
}


/**
 * \fn void fill_car_combo(GtkWidget *combo)
 * \brief Fills the car combo with all the cars in the database
 *
 * This callback fills the car combo with all the cars found in the
 * database.
 *
 */
static
void fill_car_combo(GtkWidget *combo)
{
  gint i;
  gint active;
  gint64 id;
  GString *listitemtext;

  active=0;
  i=0;
  if (db_open()) {
    listitemtext=g_string_new(NULL);

    /* We'll need to block, otherwise car_changed_cb would be fired in between */
    g_signal_handlers_block_by_func(combo, G_CALLBACK(car_changed_cb), NULL);
    if (DB_OK==db_init_car_query()) {
      while(DB_ROW == db_step_car_query()) {
	g_string_printf(listitemtext, "%s %s",db_car_column_text(0),db_car_column_text(1));
      
#if MAEMO_VERSION_MAJOR == 5
	hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (combo), listitemtext->str);
#else
	gtk_combo_box_append_text(GTK_COMBO_BOX (combo),listitemtext->str);
#endif

	id=db_car_column_int64(2);
	if (id==currentcar)
	  active=i;
	i++;
      }
    }
    g_signal_handlers_unblock_by_func(combo, G_CALLBACK(car_changed_cb), NULL);
    g_string_free(listitemtext, TRUE);
    db_reset_car_query();

#if MAEMO_VERSION_MAJOR == 5
    hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (combo), 0, active);
#else
    gtk_combo_box_set_active(GTK_COMBO_BOX (combo),active);
#endif
  }
}

GtkTreeModel *get_filter(GtkWidget *view)
{
  GtkTreeModel *sortable;
  GtkTreeModel *filter;

  sortable=gtk_tree_view_get_model(GTK_TREE_VIEW(view));
  if (sortable != NULL)
    filter=gtk_tree_model_sort_get_model(GTK_TREE_MODEL_SORT(sortable));
  else
    filter=NULL;

  return filter;
}


gboolean ui_find_iter(GtkListStore *store,
		      GtkTreeIter *iter,
		      sqlite_int64 id)
{
  sqlite_int64 curid;
  gboolean ret;

  ret=FALSE;
  if (gtk_tree_model_get_iter_first(GTK_TREE_MODEL(store), iter)) {
    gtk_tree_model_get(GTK_TREE_MODEL(store), iter, COL_ID, &curid, -1);
    while (iter != NULL && curid != id) {
      gtk_tree_model_iter_next(GTK_TREE_MODEL(store), iter);
      gtk_tree_model_get(GTK_TREE_MODEL(store), iter, COL_ID, &curid, -1);
    }
    if (curid==id)
      ret=TRUE;
  }
  return ret;
}



void ui_update_row_data(GtkListStore  *store,
			GtkTreeIter   *iter,
			const char *date,
			double km,
			double trip,
			double fill,
			double consum,
			double price,
			double priceperlitre,
			double pricepertrip,
			double service,
			double oil,
			double tires,
			const char *notes,
			sqlite_int64 id,
			gboolean visible)
{
  char *userdate;

  if (date != NULL) {
    convdate(&userdate, datefmtstr[curunit.dateformat], "%Y-%m-%d", date);
    gtk_list_store_set (store, iter, COL_DAY,  userdate, -1);
    free(userdate);
  }

  if (!(km < 0.0)) gtk_list_store_set (store, iter, COL_KM, SIlength2user(km), -1);
  if (!(trip < 0.0)) gtk_list_store_set (store, iter, COL_TRIP, SIlength2user(trip), -1);
  if (!(fill < 0.0)) gtk_list_store_set (store, iter, COL_FILL, SIvolume2user(fill), -1);
  if (!(consum < 0.0)) gtk_list_store_set (store, iter, COL_CONSUM, doubleornothing(SIconsumption2user(consum)), -1);
  if (!(price < 0.0)) gtk_list_store_set (store, iter, COL_PRICE, price, -1);
  if (!(pricepertrip < 0.0)) gtk_list_store_set (store, iter, COL_PRICEPERTRIP, doubleornothing(SIpricepertrip2user(pricepertrip)), -1);
  if (!(priceperlitre < 0.0)) gtk_list_store_set (store, iter, COL_PRICEPERLITRE, doubleornothing(SIppl2user(priceperlitre)), -1);
  if (!(service < 0.0)) gtk_list_store_set (store, iter, COL_SERVICE, service, -1);
  if (!(oil < 0.0)) gtk_list_store_set (store, iter, COL_OIL, oil, -1);
  if (!(tires < 0.0)) gtk_list_store_set (store, iter, COL_TIRES, tires, -1);
				/* 			COL_INSURANCE, sqlite3_column_double(ppStmtRecords,5), */
				/* 			COL_OTHER, sqlite3_column_double(ppStmtRecords,5), */
  if (!(consum < 0.0)) gtk_list_store_set (store, iter, COL_CO2EMISSION, SIemission2user(calc_co2_emission(consum,currentcar)), -1);
  if (notes != NULL) gtk_list_store_set (store, iter, COL_NOTES, notes, -1);
  gtk_list_store_set (store, iter, COL_ID, id, COL_VISIBLE, visible, -1);
}




void update_totalkm(AppUIData *pui)
{
  GString *str;
  float totalkm;
  float totalfillkm;
  float totalfill;

  str=g_string_new(NULL);

  totalkm=db_get_totalkm(stbstattime);
  totalfillkm=db_get_totalfillkm(stbstattime);
  totalfill=db_get_totalfill(stbstattime);
  g_string_printf(str, "%.0f ", SIlength2user(totalkm));
  if (curunit.lengthunit==SI)
    g_string_append_printf(str, _("km"));
  else
    g_string_append_printf(str, _("miles"));
  gtk_label_set_text(GTK_LABEL(pui->stb_totalkmlabel),str->str);

  if (totalfillkm != 0.0)
    g_string_printf(str, "%.1f ",SIconsumption2user(totalfill/totalfillkm*100));
  else
    g_string_printf(str, "%.1f ",0.0);
  if (curunit.consumeunit==SI)
    g_string_append_printf(str, _("l/100 km"));
  else
    g_string_append_printf(str, _("MPG"));
  gtk_label_set_text(GTK_LABEL(pui->stb_avgconsumlabel),str->str);

  g_string_printf(str, "%.0f", db_get_totalcost());
  g_string_append_printf(str, " %s", curunit.currency->str);
  gtk_label_set_text(GTK_LABEL(pui->stb_totcostlabel),str->str);

  g_string_free(str, TRUE);

}

#if PLAINGTK == 0
static HildonFileSystemModel* get_file_system_model(GtkWidget *ref_widget)
{
 return HILDON_FILE_SYSTEM_MODEL(g_object_new(HILDON_TYPE_FILE_SYSTEM_MODEL,
					      "ref_widget", ref_widget, NULL));
}
#endif

static
void internal_close_db(AppUIData *pui)
{
  GtkListStore *store;

  db_close();
  store = (GtkListStore *)gtk_tree_view_get_model(GTK_TREE_VIEW(pui->view));
  gtk_list_store_clear(store);
}

static
void callback_close_db( GtkAction * action, AppUIData *pui )
{
  internal_close_db(pui);
  enable_mainmenu_items(pui);
}

static
void callback_open_db( GtkAction * action, AppUIData *pui )
{
  gchar *filename = NULL;

  filename = interface_file_chooser ("Open database",
				     GTK_FILE_CHOOSER_ACTION_OPEN,
				     "db",
				     HELP_ID_DATABASE);

  if (filename == NULL) {
    filename = "NULL";
  }
  else {
    internal_close_db(pui);
    db_setfilename(filename);
    db_connect();
/*     update_record_model(); */
    update_car_changed(pui);
    enable_mainmenu_items(pui);
  }
}

/**
 * \fn void add_record_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
 * \brief Response callback for add record dialog
 *
 * This callback adds the data entered in the add record dialog to the
 * database.
 *
 * \todo This should be changed so that fuelpad.c has a function
 * edit_driver which is called with the data (i.e. sqlite would not be
 * used here)
 *
 */
static
void add_record_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  GtkListStore  *store;
  GtkTreeIter    storeiter;
#if MAEMO_VERSION_MAJOR == 5
  GtkTreePath *path;
  GdkRectangle *rect;
  gint y;
#else
  GtkAdjustment *vertical;
#endif

  GString *str;
  GString *notes;
  char *date;
  char *userdate;
  gdouble km;
  gdouble trip;
  gdouble fill;
  gdouble price;
  gdouble consum;
  gdouble pricepertrip;
  gdouble service;
  gdouble oil;
  gdouble tires;
#if PLAINGTK == 0
  guint year;
  guint month;
  guint day;
#endif
  sqlite_int64 recordid;
  sqlite_int64 fullid;
  gdouble fullconsum;
  gdouble fullfill;
  gdouble fullkm;
  gdouble fullprice;
  gdouble fullpricepertrip;

  if (!db_open()) {
       gtk_widget_destroy(GTK_WIDGET(dialog));
  }
  else {
    switch (arg1){
    case GTK_RESPONSE_ACCEPT: 
      if (carchanged)
	update_car_changed(pui);

#if PLAINGTK == 0
#if MAEMO_VERSION_MAJOR == 5
      hildon_date_button_get_date(HILDON_DATE_BUTTON(pui->entrydate), &year, &month, &day);  /* Month is betweewn 0 and 11 */
      month++; 
#else
      hildon_date_editor_get_date(HILDON_DATE_EDITOR(pui->entrydate), &year, &month, &day);
#endif
      date=(char *)malloc(ENTRYDATEMAX*sizeof(char));
      g_snprintf(date, ENTRYDATEMAX, "%d-%02d-%02d",year, month, day);
#else
      date2sqlite(&date, datefmtstr[curunit.dateformat], 
		  gtk_entry_get_text(GTK_ENTRY(pui->entrydate)));
#endif

/*       km = user2SIlength(atof(gtk_entry_get_text(GTK_ENTRY(pui->entrykm)))); */
      str = get_entry(pui->entrykm);
      km  = user2SIlength(atof(str->str));
      g_string_free(str, TRUE);

/*       trip = user2SIlength(atof(gtk_entry_get_text(GTK_ENTRY(pui->entrytrip)))); */
      str = get_entry(pui->entrytrip);
      trip = user2SIlength(atof(str->str));
      g_string_free(str, TRUE);

/*       fill = user2SIvolume(atof(gtk_entry_get_text(GTK_ENTRY(pui->entryfill)))); */
      str = get_entry(pui->entryfill);
      fill  = user2SIvolume(atof(str->str));
      g_string_free(str, TRUE);

/*       price = atof(gtk_entry_get_text(GTK_ENTRY(pui->entryprice))); */
      str = get_entry(pui->entryprice);
      price  = atof(str->str);
      g_string_free(str, TRUE);

/*       service = atof(gtk_entry_get_text(GTK_ENTRY(pui->entryservice))); */
      str = get_entry(pui->entryservice);
      service = atof(str->str);
      g_string_free(str, TRUE);

/*       oil=atof(gtk_entry_get_text(GTK_ENTRY(pui->entryoil))); */
      str = get_entry(pui->entryoil);
      oil = atof(str->str);
      g_string_free(str, TRUE);

/*       tires=atof(gtk_entry_get_text(GTK_ENTRY(pui->entrytires))); */
      str = get_entry(pui->entrytires);
      tires = atof(str->str);
      g_string_free(str, TRUE);

      notes = get_entry(pui->entrynotes);

      /* Take not full tanks into account: calculate the consumption for full record */
#if MAEMO_VERSION_MAJOR == 5
      if (hildon_check_button_get_active (HILDON_CHECK_BUTTON (pui->buttonnotfull))) { 
#else
      if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (pui->buttonnotfull))) {
#endif

	/* For this record */
	consum = 0.0;
	pricepertrip = 0.0;

	/* Find next full record */
	fullid=db_find_next_full(km, &fullfill, &fullkm, &fullprice);
	if (fullid>=0) {
	  fullconsum=(fullfill+fill)/(fullkm+trip)*100;
	  fullpricepertrip=(fullprice+price)/(fullkm+trip);

	  /* Update now the full record consum and tree view also */
	  db_update_record(fullid, NULL, -1.0, -1.0, -1.0, fullconsum, -1.0, -1.0, fullpricepertrip, -1.0, -1.0, -1.0, NULL);

	  get_store_and_iter(NULL, pui->view, NULL, &store, NULL);
	  if (ui_find_iter(store, &storeiter,fullid)) {
	    ui_update_row_data(store, &storeiter,
			       NULL, /* date */
			       -1.0, /* km */
			       -1.0, /* trip */
			       -1.0, /* fill */
			       fullconsum, /* consum */
			       -1.0, /* price */
			       -1.0, /* price/litre */
			       fullpricepertrip, /* price/trip */
			       -1.0, /* service */
			       -1.0, /* oil */
			       -1.0, /* tires */
			       NULL, /* notes */
			       fullid,
			       TRUE);
	  }
	  else
	    PDEBUG("First iter not found when trying to update full fill record\n");
	}
	else
	  PDEBUG("Full fill not found\n");
      }
      else { // Full fill
	/* Find if there are any not full fills before this record */
	db_find_prev_full(km, &fullfill, &fullkm, &fullprice);

	consum = (fullfill+fill)/(fullkm+trip)*100;
	pricepertrip = (fullprice+price)/(fullkm+trip);
      }
      if (db_open()) {
	recordid = db_add_record(date, km, trip, fill, consum, price, 
				 price/fill, pricepertrip, service, oil, tires, notes->str);
	if (recordid<0) {
	  /*     ui_show_banner("Problem saving result"); */
	  PDEBUG("Problem saving result");
	}
	else { /* record succesfully inserted */
	  get_store_and_iter(NULL, pui->view, NULL, &store, NULL);
	  gtk_list_store_append (store, &storeiter);
	  ui_update_row_data(store, &storeiter,
			     date,
			     km,
			     trip,
			     fill,
			     consum,
			     price,
			     price/fill,
			     pricepertrip,
			     service,
			     oil,
			     tires,
			     notes->str,
			     recordid,
			     TRUE);

	  update_totalkm(pui);
	  g_string_free(notes, TRUE);
	  
	  /* Scroll to the end of list */
#if MAEMO_VERSION_MAJOR == 5
#if 0
	  path = gtk_tree_model_get_path (store, &storeiter);
	  gtk_tree_view_get_background_area (GTK_TREE_VIEW (pui->view),
                                    path, NULL, &rect);
	  gtk_tree_view_convert_bin_window_to_tree_coords (GTK_TREE_VIEW (pui->view),
							   0, rect->y, NULL, &y);
	  hildon_pannable_area_scroll_to (HILDON_PANNABLE_AREA(pui->datascrollwin), -1, y);
	  gtk_tree_path_free (path);
#endif
#else
	  if (maintablesorted) {
	    vertical = gtk_scrolled_window_get_vadjustment((GtkScrolledWindow *)pui->datascrollwin);
	    if (maintablesortorder==GTK_SORT_ASCENDING) {
	      vertical->value = vertical->upper;
	    }
	    else {
	      vertical->value = vertical->lower;
	    }
	    gtk_scrolled_window_set_vadjustment((GtkScrolledWindow *)pui->datascrollwin,vertical);
	    gtk_adjustment_value_changed(vertical);
	  }
#endif
	  
	}
      }
      else {
	PDEBUG("Can't save new record - no database open");
      }
      free(date);
      gtk_widget_destroy(GTK_WIDGET(dialog));
      break;
    case GTK_RESPONSE_REJECT:
      gtk_widget_destroy(GTK_WIDGET(dialog));
      break;
    }
  }
}

/**
 * \fn void edit_record_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
 * \brief Response callback for edit record dialog
 *
 * This callback alters the data edited in the edit record dialog to
 * the database.
 *
 * \todo This should be changed so that fuelpad.c has a function
 * edit_driver which is called with the data (i.e. sqlite would not be
 * used here)
 *
 */
static
void edit_record_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkListStore  *store;
  GtkListStore  *fullstore;
  GtkTreeIter    iter;
  GtkTreeIter    storeiter;
  GtkTreeIter    fullstoreiter;

  gint64 id;
  GString *str;
  GString *notes;
  char *date;
  char *userdate;
  gdouble km;
  gdouble trip;
  gdouble fill;
  gdouble price;
  gdouble consum;
  gdouble pricepertrip;
  gdouble service;
  gdouble oil;
  gdouble tires;
#if PLAINGTK == 0
  guint year;
  guint month;
  guint day;
#endif
  gboolean notfull;
  gboolean oldnotfull;
  sqlite_int64 fullid;
  gdouble fullconsum;
  gdouble fullfill;
  gdouble fullkm;
  gdouble fullprice;
  gdouble fullpricepertrip;
  gdouble oldfill;
  gdouble oldtrip;
  gdouble oldconsum;
  gdouble oldprice;

  if (!db_open()) {
       gtk_widget_destroy(GTK_WIDGET(dialog));
  }
  else {
    switch (arg1) {
    case GTK_RESPONSE_ACCEPT:
      selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->view));
      if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
	gtk_tree_model_get (model, &iter, COL_ID, &id, -1);

#if PLAINGTK == 0
#if MAEMO_VERSION_MAJOR == 5
      hildon_date_button_get_date(HILDON_DATE_BUTTON(pui->entrydate), &year, &month, &day);  /* Month is betweewn 0 and 11 */
      month++; 
#else
	hildon_date_editor_get_date(HILDON_DATE_EDITOR(pui->entrydate), &year, &month, &day);
#endif
	date=(char *)malloc(ENTRYDATEMAX*sizeof(char));
	g_snprintf(date, ENTRYDATEMAX, "%d-%02d-%02d",year, month, day);
#else
	date2sqlite(&date, datefmtstr[curunit.dateformat], 
		    gtk_entry_get_text(GTK_ENTRY(pui->entrydate)));
#endif

/* 	km = user2SIlength(atof(gtk_entry_get_text(GTK_ENTRY(pui->entrykm)))); */
      str = get_entry(pui->entrykm);
      km  = user2SIlength(atof(str->str));
      g_string_free(str, TRUE);

/* 	trip = user2SIlength(atof(gtk_entry_get_text(GTK_ENTRY(pui->entrytrip)))); */
      str = get_entry(pui->entrytrip);
      trip = user2SIlength(atof(str->str));
      g_string_free(str, TRUE);

/* 	fill = user2SIvolume(atof(gtk_entry_get_text(GTK_ENTRY(pui->entryfill)))); */
      str = get_entry(pui->entryfill);
      fill  = user2SIvolume(atof(str->str));
      g_string_free(str, TRUE);

/* 	price = atof(gtk_entry_get_text(GTK_ENTRY(pui->entryprice))); */
      str = get_entry(pui->entryprice);
      price  = atof(str->str);
      g_string_free(str, TRUE);

/* 	service = atof(gtk_entry_get_text(GTK_ENTRY(pui->entryservice))); */
      str = get_entry(pui->entryservice);
      service = atof(str->str);
      g_string_free(str, TRUE);

/* 	oil=atof(gtk_entry_get_text(GTK_ENTRY(pui->entryoil))); */
      str = get_entry(pui->entryoil);
      oil = atof(str->str);
      g_string_free(str, TRUE);

/* 	tires=atof(gtk_entry_get_text(GTK_ENTRY(pui->entrytires))); */
      str = get_entry(pui->entrytires);
      tires = atof(str->str);
      g_string_free(str, TRUE);

      notes = get_entry(pui->entrynotes);

      oldnotfull=FALSE;
      /* 
       * Well need to obtain the unmodified data to be excluded from the new 
       * consumption calculations 
       */
      if (SQLITE_OK == sqlite3_bind_int(ppStmtOneRecord, 1, id)) {
	while (SQLITE_ROW == sqlite3_step(ppStmtOneRecord)) {
	  oldfill=sqlite3_column_double(ppStmtOneRecord,3);
	  oldtrip=sqlite3_column_double(ppStmtOneRecord,2);
	  oldconsum=sqlite3_column_double(ppStmtOneRecord,10);
	  oldprice=sqlite3_column_double(ppStmtOneRecord,4);
	  oldnotfull=(oldfill>0.0) && (abs(oldconsum)<1e-5);
	}
	sqlite3_reset(ppStmtOneRecord);
      }
      

      /* Take not full tanks into account: calculate the consumption for full record */
      notfull = 
#if MAEMO_VERSION_MAJOR == 5
	hildon_check_button_get_active (HILDON_CHECK_BUTTON (pui->buttonnotfull));
#else
        gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (pui->buttonnotfull));
#endif
      if (notfull) {

	/* For this record */
	consum = 0.0;

	/* Find next full record */
	fullid=db_find_next_full(km, &fullfill, &fullkm, &fullprice);

	if (fullid>=0) {
	  if (!oldnotfull) {
	    oldfill=0.0;
	    oldtrip=0.0;
	    oldprice=0.0;
	  }
	  fullconsum=(fullfill+fill-oldfill)/(fullkm+trip-oldtrip)*100;
	  fullpricepertrip=(fullprice+price-oldprice)/(fullkm+trip-oldtrip);

	  /* Update now the full record consum and tree view also */
	  db_update_record(fullid, NULL, -1.0, -1.0, -1.0, fullconsum, -1.0, -1.0, fullpricepertrip, -1.0, -1.0, -1.0, NULL);

	}
	else
	  PDEBUG("Full fill not found\n");
      }
      else { // Full fill

	if (oldnotfull) {

	  /* Find next full record */
	  fullid=db_find_next_full(km, &fullfill, &fullkm, &fullprice);

	  if (fullid>=0) {

	    fullconsum=(fullfill-oldfill)/(fullkm-oldtrip)*100;
	    fullpricepertrip=(fullprice-oldprice)/(fullkm-oldtrip);

	    /* Update now the full record consum and tree view also */
	    db_update_record(fullid, NULL, -1.0, -1.0, -1.0, fullconsum, -1.0, -1.0, fullpricepertrip, -1.0, -1.0, -1.0, NULL);
	  }
	}

	/* Find if there are any not full fills before this record */
	db_find_prev_full(km, &fullfill, &fullkm, &fullprice);

	if (oldnotfull) {
	  oldfill=0.0;
	  oldtrip=0.0;
	}
/* 	consum = (fullfill+fill-oldfill)/(fullkm+trip-oldtrip)*100; */
	consum = (fullfill+fill)/(fullkm+trip)*100;
	pricepertrip = (fullprice+price)/(fullkm+trip);
      }

	if (db_open()) {
	  if (db_update_record(id, date, km, trip, fill, consum, price, price/fill,
			       pricepertrip,
			       service, oil, tires, notes->str) != id) {
	    PDEBUG("Problem saving result");
	  }
	  else {
	    get_store_and_iter(model, pui->view, &iter, &store, &storeiter);
	    ui_update_row_data(store, &storeiter,
			       date,
			       km,
			       trip,
			       fill,
			       consum,
			       price,
			       price/fill,
			       pricepertrip,
			       service,
			       oil,
			       tires,
			       notes->str,
			       id,
			       TRUE);

	    /* Update the data for the full fill */
	    if (notfull || (notfull!=oldnotfull)) { /* not enough to test notfull, but when?*/
	      if (fullid>0) {
		get_store_and_iter(NULL, pui->view, NULL, &fullstore, NULL);
		if (ui_find_iter(fullstore, &fullstoreiter,fullid)) {
		  ui_update_row_data(fullstore, &fullstoreiter,
				     NULL, /* date */
				     -1.0, /* km */
				     -1.0, /* trip */
				     -1.0, /* fill */
				     fullconsum, /* consum */
				     -1.0, /* price */
				     -1.0, /* price/fill */
				     fullpricepertrip, /* price/trip */
				     -1.0, /* service */
				     -1.0, /* oil */
				     -1.0, /* tires */
				     NULL, /* notes */
				     fullid,
				     TRUE);
		}
	      }
	    }
	    else
	      PDEBUG("First iter not found when trying to update full fill record\n");
	    update_totalkm(pui);
	    g_string_free(notes, TRUE);

	  }
	  
	}
	else {
	  PDEBUG("Can't save new driver - no database open");
	}
      }
      else {
	PDEBUG("editrecord: Something wrong with finding the selection\n");
      }
      free(date);
      gtk_widget_destroy(GTK_WIDGET(dialog));
      break;
    case GTK_RESPONSE_REJECT:
      gtk_widget_destroy(GTK_WIDGET(dialog));
      break;
    }
  }
}

static
gboolean callback_kmadded (GtkWidget     *widget,
                           GdkEventFocus *event,
                           AppUIData     *pui)
{
  float trip;
  float newkm;
  float lastkm;
  GString *buf;

  buf=g_string_new(NULL);
  trip=atof(gtk_entry_get_text(GTK_ENTRY(pui->entrytrip)));
  newkm=user2SIlength(atof(gtk_entry_get_text(GTK_ENTRY(pui->entrykm))));
  if (trip<0.1 && newkm>0) {
    lastkm=db_get_last_refill(newkm);
    if (lastkm<0.1)
      lastkm=db_get_last_km();
    g_string_printf(buf, "%.1f",SIlength2user(newkm-lastkm));
    gtk_entry_set_text(GTK_ENTRY(pui->entrytrip),buf->str);
  }
  g_string_free(buf, TRUE);
  return FALSE;
}

static
gboolean callback_tripadded (GtkWidget     *widget,
			     GdkEventFocus *event,
			     AppUIData     *pui)
{
  float trip;
  float newkm;
  float lastkm;
  GString *buf;

  buf=g_string_new(NULL);
  newkm=atof(gtk_entry_get_text(GTK_ENTRY(pui->entrykm)));
  if (newkm<0.1) {
    trip=user2SIlength(atof(gtk_entry_get_text(GTK_ENTRY(pui->entrytrip))));
    lastkm=db_get_last_refill(newkm);
    if (lastkm<0.1)
      lastkm=db_get_last_km();
    g_string_printf(buf, "%.1f",SIlength2user(lastkm+trip));
    gtk_entry_set_text(GTK_ENTRY(pui->entrykm),buf->str);
  }
  g_string_free(buf, TRUE);
  return FALSE;
}

/**
 * \fn void create_record_editwin(AppUIData *pui, GtkWidget *dialog, gint add)
 * \brief Creates the contents of a record adding or editing dialog
 * \param *pui Pointer to the UI data structure
 * \param *dialog Pointer to the already created dialog
 * \param add Set this to 1 if an add dialog is needed
 *
 * This function is used for creating an edit window both for adding a
 * new record and editing an existing one.
 *
 */
static
void create_record_editwin(AppUIData *pui, GtkWidget *dialog, gint add)
{
  enum
  {
    RECORD_EDIT_DATE=0,
    RECORD_EDIT_KM,
    RECORD_EDIT_MILES,
    RECORD_EDIT_TRIP,
    RECORD_EDIT_FILL,
    RECORD_EDIT_NOTFULL,
    RECORD_EDIT_PRICE,
    RECORD_EDIT_NOTES,
    RECORD_EDIT_SERVICE,
    RECORD_EDIT_OIL,
    RECORD_EDIT_TIRES,
    RECORD_EDIT_CAR,
    RECORD_EDIT_DRIVER
  };

  GtkWidget *scrollwin;
#if MAEMO_VERSION_MAJOR == 5
  gint       row;
#else
  GtkWidget *notebook;
#endif
  GtkWidget *table;
  GtkWidget *label;
#if PLAINGTK == 1
  char date[ENTRYDATEMAX];
  char *aika;
#endif
  GtkListStore *store;
  GtkEntryCompletion *completion;
  gchar *labels[] = {gettext_noop("Date"),
		     gettext_noop("Km"),
		     gettext_noop("Miles"),
		     gettext_noop("Trip"),
		     gettext_noop("Fill"),
		     gettext_noop("Not full tank"),
		     gettext_noop("Price"),
		     gettext_noop("Notes"),
		     gettext_noop("Service"),
		     gettext_noop("Oil"),
		     gettext_noop("Tires"),
		     gettext_noop("Car"),
		     gettext_noop("Driver")
  };

#if MAEMO_VERSION_MAJOR == 5
  scrollwin = hildon_pannable_area_new();
  gtk_widget_set_size_request(scrollwin, -1, DIALOG_MIN_HEIGHTMAX);

  if (add)
    table = gtk_table_new(12, 2, FALSE);
  else
    table = gtk_table_new(10, 2, FALSE);

  row=0;

  if (add) {

    /* First row, first entry */
  pui->carbutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->carbutton), gettext(labels[RECORD_EDIT_CAR]));
  gtk_table_attach(GTK_TABLE(table), pui->carbutton, 0, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->carbutton);

  pui->carcombo = create_car_combo(pui, pui->carbutton);

  /* First row, second entry */
  row++;
  pui->driverbutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->driverbutton), gettext(labels[RECORD_EDIT_DRIVER]));
  gtk_table_attach(GTK_TABLE(table), pui->driverbutton, 0, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->driverbutton);

  pui->drivercombo = create_driver_combo(pui, pui->driverbutton);
/*     gtk_table_attach(GTK_TABLE(table), pui->drivercombo, 3, 4, 3, 4,  */
/* 		     GTK_EXPAND|GTK_FILL, */
/* 		     0, 0, 5); */
/*     gtk_widget_show(pui->drivercombo); */

  }

  /* First row, first entry */
  row++;
  pui->entrydate = hildon_date_button_new(HILDON_SIZE_FINGER_HEIGHT,
					  HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  gtk_table_attach(GTK_TABLE(table), pui->entrydate, 0, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrydate);
  
  /* First row, second entry */
  row++;
  if (curunit.lengthunit==SI)
    label=gtk_label_new(gettext(labels[RECORD_EDIT_KM]));
  else
    label=gtk_label_new(gettext(labels[RECORD_EDIT_MILES]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1,
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
  gtk_widget_show(label);

/*   pui->entrykm = gtk_entry_new();  */
  pui->entrykm=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrykm), ENTRYKMMAX);
#if PLAINGTK == 0
  g_object_set(G_OBJECT(pui->entrykm),
#if HILDON == 1
              "hildon-input-mode",
		 HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
		 NULL);
#else
		 "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
  g_object_set(G_OBJECT(pui->entrykm), "autocap", FALSE, NULL);
#endif
#endif
  if (add) {
    g_signal_connect(pui->entrykm, "focus-out-event", 
		     G_CALLBACK(callback_kmadded), &ui);
  }

  gtk_table_attach(GTK_TABLE(table), pui->entrykm, 1, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrykm);

  /* Second row, first entry */
  row++;
  label=gtk_label_new(gettext(labels[RECORD_EDIT_TRIP]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1,
	    GTK_EXPAND|GTK_FILL,
		    0, 0, 5);
  gtk_widget_show(label);

/*   pui->entrytrip = gtk_entry_new();  */
  pui->entrytrip=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrytrip), ENTRYTRIPMAX);
#if PLAINGTK == 0
  g_object_set(G_OBJECT(pui->entrytrip),
#if HILDON == 1
              "hildon-input-mode",
	       HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
	       NULL);
#else
               "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
  g_object_set(G_OBJECT(pui->entrytrip), "autocap", FALSE, NULL);
#endif
#endif
  if (add) {
    g_signal_connect(pui->entrytrip, "focus-out-event", 
		     G_CALLBACK(callback_tripadded), &ui);
  }
  gtk_table_attach(GTK_TABLE(table), pui->entrytrip, 1, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrytrip);

  /* Second row, second entry */
  row++;
  label=gtk_label_new(gettext(labels[RECORD_EDIT_FILL]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

/*   pui->entryfill = gtk_entry_new();  */
  pui->entryfill=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryfill), ENTRYFILLMAX);
#if PLAINGTK == 0
  g_object_set(G_OBJECT(pui->entryfill),
#if HILDON == 1
	       "hildon-input-mode",
	       HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
	       NULL);
#else
               "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
  g_object_set(G_OBJECT(pui->entryfill), "autocap", FALSE, NULL);
#endif
#endif
  gtk_table_attach(GTK_TABLE(table), pui->entryfill, 1, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryfill);

  row++;
  pui->buttonnotfull = hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label (HILDON_CHECK_BUTTON (pui->buttonnotfull), gettext(labels[RECORD_EDIT_NOTFULL]));
  gtk_table_attach(GTK_TABLE(table), pui->buttonnotfull, 1, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->buttonnotfull);

  /* Third row, first entry */
  row++;
  label=gtk_label_new(gettext(labels[RECORD_EDIT_PRICE]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

/*   pui->entryprice = gtk_entry_new();  */
  pui->entryprice=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryprice), ENTRYPRICEMAX);
#if PLAINGTK == 0
  g_object_set(G_OBJECT(pui->entryprice),
#if HILDON == 1
	       "hildon-input-mode",
	       HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
	       NULL);
#else
               "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
  g_object_set(G_OBJECT(pui->entryprice), "autocap", FALSE, NULL);
#endif
#endif
  gtk_table_attach(GTK_TABLE(table), pui->entryprice, 1, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryprice);

  /* Third row, second entry */
  row++;
  label=gtk_label_new(gettext(labels[RECORD_EDIT_NOTES]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

/*   pui->entrynotes = gtk_entry_new();  */
  pui->entrynotes=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrynotes), ENTRYNOTESMAX);
  gtk_table_attach(GTK_TABLE(table), pui->entrynotes, 1, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrynotes);

  completion = gtk_entry_completion_new();
  store = (GtkListStore *)gtk_tree_view_get_model(GTK_TREE_VIEW(pui->view));
  gtk_entry_completion_set_model(completion, GTK_TREE_MODEL(store));
  gtk_entry_completion_set_text_column(completion, COL_NOTES);
  gtk_entry_set_completion(GTK_ENTRY(pui->entrynotes), completion);

  /* First row, first entry */
  row++;
  label=gtk_label_new(gettext(labels[RECORD_EDIT_SERVICE]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1,
	    GTK_EXPAND|GTK_FILL,
	    0, 0, 5);
  gtk_widget_show(label);

/*     pui->entryservice = gtk_entry_new();  */
  pui->entryservice=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryservice), ENTRYSERVICEMAX);
#if PLAINGTK == 0
  g_object_set(G_OBJECT(pui->entryservice),
#if HILDON == 1
                "hildon-input-mode",
		 HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
		 NULL);
#else
                 "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
  g_object_set(G_OBJECT(pui->entryservice), "autocap", FALSE, NULL);
#endif
#endif
  gtk_table_attach(GTK_TABLE(table), pui->entryservice, 1, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryservice);

    /* Second row, first entry */
  row++;
  label=gtk_label_new(gettext(labels[RECORD_EDIT_OIL]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
    gtk_widget_show(label);

/*     pui->entryoil = gtk_entry_new();  */
  pui->entryoil=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryoil), ENTRYOILMAX);
#if PLAINGTK == 0
  g_object_set(G_OBJECT(pui->entryoil),
#if HILDON == 1
              "hildon-input-mode",
	       HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
	       NULL);
#else
               "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
  g_object_set(G_OBJECT(pui->entryoil), "autocap", FALSE, NULL);
#endif
#endif
  gtk_table_attach(GTK_TABLE(table), pui->entryoil, 1, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryoil);

    /* Third row, first entry */
  row++;
  label=gtk_label_new(gettext(labels[RECORD_EDIT_TIRES]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

/*     pui->entrytires = gtk_entry_new();  */
  pui->entrytires=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrytires), ENTRYTIRESMAX);
#if PLAINGTK == 0
  g_object_set(G_OBJECT(pui->entrytires),
#if HILDON == 1
              "hildon-input-mode",
	       HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
	       NULL);
#else
               "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
  g_object_set(G_OBJECT(pui->entrytires), "autocap", FALSE, NULL);
#endif
#endif
  gtk_table_attach(GTK_TABLE(table), pui->entrytires, 1, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrytires);

  gtk_widget_show(table);
  gtk_box_pack_start (GTK_BOX(GTK_DIALOG (dialog)->vbox), scrollwin, TRUE, TRUE, 0);
  gtk_widget_show(scrollwin);
  hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA(scrollwin),table);

#else /* MAEMO_VERSION_MAJOR != 5 */
  notebook = gtk_notebook_new ();
  gtk_notebook_set_tab_pos (GTK_NOTEBOOK (notebook), GTK_POS_TOP);
  gtk_box_pack_start (GTK_BOX(GTK_DIALOG (dialog)->vbox), notebook, TRUE, TRUE, 0);
  gtk_widget_show (notebook);

  scrollwin = gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_set_size_request(scrollwin, -1, DIALOG_MIN_HEIGHT1);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollwin), 
				 GTK_POLICY_NEVER, 
				 GTK_POLICY_AUTOMATIC);

/*   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), scrollwin, TRUE, TRUE, 0); */
  table = gtk_table_new(4, 4, FALSE);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrollwin),table);

  /* First row, first entry */
  label=gtk_label_new(gettext(labels[RECORD_EDIT_DATE]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

#if PLAINGTK == 1
  pui->entrydate = gtk_entry_new(); 
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrydate), ENTRYDATEMAX);
  gettimefmt(&aika, datefmtstr[curunit.dateformat]);
  gtk_entry_set_text(GTK_ENTRY(pui->entrydate),aika);
  free(aika);
#else
  pui->entrydate = hildon_date_editor_new();
#endif
  gtk_table_attach(GTK_TABLE(table), pui->entrydate, 1, 2, 0, 1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrydate);

  /* First row, second entry */
  if (curunit.lengthunit==SI)
    label=gtk_label_new(gettext(labels[RECORD_EDIT_KM]));
  else
    label=gtk_label_new(gettext(labels[RECORD_EDIT_MILES]));
  gtk_table_attach(GTK_TABLE(table), label, 2, 3, 0, 1, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
  gtk_widget_show(label);

  pui->entrykm = gtk_entry_new(); 
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrykm), ENTRYKMMAX);
#if PLAINGTK == 0
  g_object_set(G_OBJECT(pui->entrykm),
#if HILDON == 1
              "hildon-input-mode",
		 HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
		 NULL);
#else
		 "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
  g_object_set(G_OBJECT(pui->entrykm), "autocap", FALSE, NULL);
#endif
#endif
  if (add) {
    g_signal_connect(pui->entrykm, "focus-out-event", 
		     G_CALLBACK(callback_kmadded), &ui);
  }

  gtk_table_attach(GTK_TABLE(table), pui->entrykm, 3, 4, 0, 1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrykm);

  /* Second row, first entry */
  label=gtk_label_new(gettext(labels[RECORD_EDIT_TRIP]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, 
	    GTK_EXPAND|GTK_FILL,
		    0, 0, 5);
  gtk_widget_show(label);

  pui->entrytrip = gtk_entry_new(); 
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrytrip), ENTRYTRIPMAX);
#if PLAINGTK == 0
  g_object_set(G_OBJECT(pui->entrytrip),
#if HILDON == 1
              "hildon-input-mode",
	       HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
	       NULL);
#else
               "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
  g_object_set(G_OBJECT(pui->entrytrip), "autocap", FALSE, NULL);
#endif
#endif
  if (add) {
    g_signal_connect(pui->entrytrip, "focus-out-event", 
		     G_CALLBACK(callback_tripadded), &ui);
  }
  gtk_table_attach(GTK_TABLE(table), pui->entrytrip, 1, 2, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrytrip);

  /* Second row, second entry */
  label=gtk_label_new(gettext(labels[RECORD_EDIT_FILL]));
  gtk_table_attach(GTK_TABLE(table), label, 2, 3, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entryfill = gtk_entry_new(); 
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryfill), ENTRYFILLMAX);
#if PLAINGTK == 0
  g_object_set(G_OBJECT(pui->entryfill),
#if HILDON == 1
	       "hildon-input-mode",
	       HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
	       NULL);
#else
               "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
  g_object_set(G_OBJECT(pui->entryfill), "autocap", FALSE, NULL);
#endif
#endif
  gtk_table_attach(GTK_TABLE(table), pui->entryfill, 3, 4, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryfill);

  /* Not full button */
  pui->buttonnotfull = gtk_check_button_new_with_label(gettext(labels[RECORD_EDIT_NOTFULL]));
  gtk_table_attach(GTK_TABLE(table), pui->buttonnotfull, 3, 4, 2, 3,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->buttonnotfull);

  /* Third row, first entry */
  label=gtk_label_new(gettext(labels[RECORD_EDIT_PRICE]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 3, 4, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entryprice = gtk_entry_new(); 
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryprice), ENTRYPRICEMAX);
#if PLAINGTK == 0
  g_object_set(G_OBJECT(pui->entryprice),
#if HILDON == 1
	       "hildon-input-mode",
	       HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
	       NULL);
#else
               "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
  g_object_set(G_OBJECT(pui->entryprice), "autocap", FALSE, NULL);
#endif
#endif
  gtk_table_attach(GTK_TABLE(table), pui->entryprice, 1, 2, 3, 4, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryprice);

  /* Third row, second entry */
  label=gtk_label_new(gettext(labels[RECORD_EDIT_NOTES]));
  gtk_table_attach(GTK_TABLE(table), label, 2, 3, 3, 4,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entrynotes = gtk_entry_new(); 
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrynotes), ENTRYNOTESMAX);
  gtk_table_attach(GTK_TABLE(table), pui->entrynotes, 3, 4, 3, 4, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrynotes);

  completion = gtk_entry_completion_new();
  store = (GtkListStore *)gtk_tree_view_get_model(GTK_TREE_VIEW(pui->view));
  gtk_entry_completion_set_model(completion, GTK_TREE_MODEL(store));
  gtk_entry_completion_set_text_column(completion, COL_NOTES);
  gtk_entry_set_completion(GTK_ENTRY(pui->entrynotes), completion);

  /* Table ready - show it */
  gtk_widget_show (table);
  gtk_widget_show(scrollwin);

  label = gtk_label_new (_("Fuel fill"));
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), scrollwin, label);

  /* Service etc. notebook */
  table = gtk_table_new(1, 3, FALSE);

    /* First row, first entry */
    label=gtk_label_new(gettext(labels[RECORD_EDIT_SERVICE]));
    gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(label);

    pui->entryservice = gtk_entry_new(); 
    gtk_entry_set_max_length(GTK_ENTRY(pui->entryservice), ENTRYSERVICEMAX);
#if PLAINGTK == 0
    g_object_set(G_OBJECT(pui->entryservice),
#if HILDON == 1
                "hildon-input-mode",
		 HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
		 NULL);
#else
                 "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
    g_object_set(G_OBJECT(pui->entryservice), "autocap", FALSE, NULL);
#endif
#endif
    gtk_table_attach(GTK_TABLE(table), pui->entryservice, 1, 2, 0, 1, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(pui->entryservice);

    /* Second row, first entry */
    label=gtk_label_new(gettext(labels[RECORD_EDIT_OIL]));
    gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(label);

    pui->entryoil = gtk_entry_new(); 
    gtk_entry_set_max_length(GTK_ENTRY(pui->entryoil), ENTRYOILMAX);
#if PLAINGTK == 0
    g_object_set(G_OBJECT(pui->entryoil),
#if HILDON == 1
                "hildon-input-mode",
		 HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
		 NULL);
#else
                 "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
    g_object_set(G_OBJECT(pui->entryoil), "autocap", FALSE, NULL);
#endif
#endif
    gtk_table_attach(GTK_TABLE(table), pui->entryoil, 1, 2, 1, 2, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(pui->entryoil);

    /* Third row, first entry */
    label=gtk_label_new(gettext(labels[RECORD_EDIT_TIRES]));
    gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3,
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(label);

    pui->entrytires = gtk_entry_new(); 
    gtk_entry_set_max_length(GTK_ENTRY(pui->entrytires), ENTRYTIRESMAX);
#if PLAINGTK == 0
    g_object_set(G_OBJECT(pui->entrytires),
#if HILDON == 1
                "hildon-input-mode",
		 HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
		 NULL);
#else
                 "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
    g_object_set(G_OBJECT(pui->entrytires), "autocap", FALSE, NULL);
#endif
#endif
    gtk_table_attach(GTK_TABLE(table), pui->entrytires, 1, 2, 2, 3, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(pui->entrytires);

    /* Table ready - show it */
    gtk_widget_show (table);

  label = gtk_label_new (_("Service/oil/tires"));
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), table, label);

  if (add) {

    table = gtk_table_new(2, 2, FALSE);

    /* First row, first entry */
    label=gtk_label_new(gettext(labels[RECORD_EDIT_CAR]));
    gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(label);

    pui->carcombo = create_car_combo(pui);
    gtk_table_attach(GTK_TABLE(table), pui->carcombo, 1, 2, 0, 1, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(pui->carcombo);

    /* First row, second entry */
    label=gtk_label_new(gettext(labels[RECORD_EDIT_DRIVER]));
    gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(label);
    
    pui->drivercombo = create_driver_combo(pui);
    gtk_table_attach(GTK_TABLE(table), pui->drivercombo, 1, 2, 1, 2, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(pui->drivercombo);

    gtk_widget_show (table);

    label = gtk_label_new (_("Driver and car"));
    gtk_notebook_append_page (GTK_NOTEBOOK (notebook), table, label);

  }
#endif

}

static
void callback_newrecord( GtkAction * action, AppUIData *pui )
{
  GtkWidget *dialog;
  GtkWidget *label;
  gchar *header[] = {gettext_noop("Add a new record")};

  if (db_open()) {
    dialog = gtk_dialog_new_with_buttons(gettext(header[0]),
					 GTK_WINDOW(pui->app->mainwindow),
					 GTK_DIALOG_MODAL,
					 GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
					 GTK_STOCK_CANCEL,
					 GTK_RESPONSE_REJECT,
					 NULL);
    create_record_editwin(pui, dialog, 1);

#if LIBHELP == 1
    help_dialog_help_enable(GTK_DIALOG(dialog),
				   HELP_ID_ADDRECORD,
				   pui->app->osso);
#endif

/*     g_signal_connect_swapped (dialog, "response", */
/* 			      G_CALLBACK (add_record_response), */
/* 			      dialog); */
    g_signal_connect (dialog, "response",
		      G_CALLBACK (add_record_response),
		      &ui);
  }
  else {
    dialog = gtk_dialog_new_with_buttons(gettext(header[0]),
					 GTK_WINDOW(pui->app->mainwindow),
					 GTK_DIALOG_MODAL,
					 GTK_STOCK_OK, 
					 GTK_RESPONSE_REJECT,
					 NULL);

    label=gtk_label_new(_("Can't access database - adding records not possible"));
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label, TRUE, TRUE, 5);
    gtk_widget_show(label);

    g_signal_connect_swapped (dialog, "response",
			      G_CALLBACK (gtk_widget_destroy),
			      dialog);
  }


  gtk_widget_show(dialog);
}

static
void callback_prevrecord( GtkAction * action, AppUIData *pui )
{
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter    iter;
  GtkTreePath   *path;
  GtkAdjustment *vertical;

  if (db_open()) {
      selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->view));
      if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
	vertical = gtk_scrolled_window_get_vadjustment((GtkScrolledWindow *)pui->datascrollwin);
	vertical->value -= vertical->step_increment;
	if (vertical->value<vertical->lower)
	  vertical->value = vertical->lower;
	/* 	  gtk_tree_model_iter_prev(model,&iter); */
	path = gtk_tree_model_get_path(model, &iter);
	if (gtk_tree_path_prev(path)) {
	  gtk_tree_model_get_iter(model, &iter, path);
	  gtk_tree_selection_select_iter(selection, &iter);
	}
	gtk_tree_path_free(path);
	gtk_tree_selection_select_iter(selection, &iter);
	gtk_scrolled_window_set_vadjustment((GtkScrolledWindow *)pui->datascrollwin,vertical);
	gtk_adjustment_value_changed(vertical);
      }
  }
}

static
void callback_nextrecord( GtkAction * action, AppUIData *pui )
{
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter    iter;
  GtkAdjustment *vertical;

  if (db_open()) {
      selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->view));
      if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
	vertical = gtk_scrolled_window_get_vadjustment((GtkScrolledWindow *)pui->datascrollwin);
	vertical->value += vertical->step_increment;
	if (vertical->value>vertical->upper)
	  vertical->value = vertical->upper;
	gtk_tree_model_iter_next(model,&iter);
	gtk_tree_selection_select_iter(selection, &iter);
	gtk_scrolled_window_set_vadjustment((GtkScrolledWindow *)pui->datascrollwin,vertical);
	gtk_adjustment_value_changed(vertical);
      }
  }
}




/**
 * \fn void callback_deleterecord( GtkAction * action, AppUIData *pui )
 * \brief Callback for deleting a record
 *
 * This callback deletes the currently selected record. The user is
 * asked for a confirmation. If there is also a reminder event
 * associated with this record, it may be also optionally deleted.
 *
 * \todo This should be changed so that fuelpad.c has a function for
 * obtaining the needed data (i.e. sqlite would not be used here)
 *
 */
void callback_deleterecord( GtkAction * action, AppUIData *pui )
{
  GtkWidget *confirmdlg;
  GtkWidget *check;
  GtkWidget *separator;
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreeIter storeiter;
  GtkTreeIter fullstoreiter;
  GtkListStore *store;
  GtkListStore *fullstore;
  GtkTreeModel *model;
  gint64 id;
  gint confirm;
  gint numevents;
  gboolean delevent;
  gdouble km;
  gdouble fill;
  gdouble trip;
  gdouble consum;
  sqlite_int64 fullid;
  gdouble fullconsum;
  gdouble fullfill;
  gdouble fullkm;
  gdouble fullprice;
  gdouble fullpricepertrip;

  if (db_open()) { /* delete */

    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->view));
    if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
      gtk_tree_model_get (model, &iter, COL_ID, &id, -1);

#if PLAINGTK == 0
      confirmdlg = hildon_note_new_confirmation(GTK_WINDOW(pui->app->mainwindow), 
						_("Are you sure you want to delete the current record"));
#else
      confirmdlg = gtk_message_dialog_new(GTK_WINDOW(pui->app->mainwindow),
					  GTK_DIALOG_MODAL,
					  GTK_MESSAGE_QUESTION,
					  GTK_BUTTONS_OK_CANCEL,
					  _("Are you sure you want to delete the current record"));
      gtk_widget_show(confirmdlg);
#endif
      numevents=db_get_numof_events_with_record(id);
      if (numevents>0) {
	separator=gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(confirmdlg)->vbox), separator, TRUE, TRUE, 0);
	gtk_widget_show(separator);

	check=gtk_check_button_new_with_label(_("Delete the corresponding reminder event"));
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (check), TRUE);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(confirmdlg)->vbox), check, TRUE, TRUE, 0);
	gtk_widget_show(check);
      }

      confirm = gtk_dialog_run(GTK_DIALOG(confirmdlg));
      if (numevents>0)
	delevent = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check));
      else
	delevent = FALSE;
      gtk_widget_destroy(GTK_WIDGET(confirmdlg));

      if(confirm == GTK_RESPONSE_OK) {

	/* 
	 * Well need to obtain the unmodified data to be excluded from the new 
	 * consumption calculations 
	 */
	if (SQLITE_OK == sqlite3_bind_int(ppStmtOneRecord, 1, id)) {
	  while (SQLITE_ROW == sqlite3_step(ppStmtOneRecord)) {
	    km=sqlite3_column_double(ppStmtOneRecord,1);
	    fill=sqlite3_column_double(ppStmtOneRecord,3);
	    trip=sqlite3_column_double(ppStmtOneRecord,2);
	    consum=sqlite3_column_double(ppStmtOneRecord,9);
	  }
	  sqlite3_reset(ppStmtOneRecord);
	}
	else
	  PDEBUG("Something wrong with ppStmtOneRecord\n");

	if (DB_DONE != db_delete_record(id)) {
	  PDEBUG("Problem deleting the record");
	}
	else { /* remove the record if succesfully removed from db */

	  get_store_and_iter(model, pui->view, &iter, &store, &storeiter);

	  gtk_list_store_remove(store, &storeiter);

	  if (fill>0.0)
	    if (abs(consum)<1e-5) { /* Not full fill */ 

	      /* Find next full record */
	      fullid=db_find_next_full(km, &fullfill, &fullkm, &fullprice);
	      if (fullid>=0) {
		fullconsum=fullfill/fullkm*100;
		fullpricepertrip=fullprice/fullkm;

		/* Update now the full record consum and tree view also */
		db_update_record(fullid, NULL, -1.0, -1.0, -1.0, fullconsum, -1.0, -1.0, fullpricepertrip, -1.0, -1.0, -1.0, NULL);

		/* Update the data for the full fill */
		get_store_and_iter(NULL, pui->view, NULL, &fullstore, NULL);
		if (ui_find_iter(fullstore, &fullstoreiter,fullid)) {
		  ui_update_row_data(fullstore, &fullstoreiter,
				     NULL, /* date */
				     -1.0, /* km */
				     -1.0, /* trip */
				     -1.0, /* fill */
				     fullconsum, /* consum */
				     -1.0, /* price */
				     -1.0, /* price/litre */
				     fullpricepertrip, /* pricepertrip */
				     -1.0, /* service */
				     -1.0, /* oil */
				     -1.0, /* tires */
				     NULL, /* notes */
				     fullid,
				     TRUE);
		}
		
	      }
	      else
		PDEBUG("Full fill not found\n");
	    }

	  update_totalkm(pui);

	  /* Delete the corresponding alarmevent */
	  if (numevents > 0 && delevent == TRUE) {
/* 	    if (SQLITE_OK != sqlite3_bind_int(ppStmtDeleteEventwithRecordid, 1, id) */
/* 		||SQLITE_DONE != sqlite3_step(ppStmtDeleteEventwithRecordid)) { */
/* 	      PDEBUG("Problem deleting the event"); */
/* 	    } */
/* 	    else { /\* Alarm event deleted, we should update */
/* 		    the next event estimates if event window open*\/ */
/* 	      PDEBUG("Alarm event deleted\n"); */
/* 	    } */
/* 	    sqlite3_reset(ppStmtDeleteEventwithRecordid); */
	    if (!db_delete_event_with_record_id(id)) {
	      PDEBUG("Problem deleting the event");
	    }
	    else { /* Alarm event deleted, we should update
		      the next event estimates if event window open*/
	      PDEBUG("Alarm event deleted\n");
	    }
	  }

	}
      } /* if (confirm == GTK_RESPONSE_OK) */
    }
    else
      PDEBUG("deleterecord: Something wrong with finding the selection\n");
  }
  else { /* nothing to delete */
    PDEBUG("Database not open, can't delete anything\n");
  }
}

#if PLAINGTK == 1
static void create_mainwin_toolbar(GtkWidget * main_window, GtkWidget *vbox, AppUIData *pui)
#else
  static void create_mainwin_toolbar(HildonWindow * main_window, AppUIData *pui)
#endif
{
  /* Create toolbar */
  pui->main_toolbar = gtk_toolbar_new();

  /* Create toolbar button items */
  pui->mtb_add = gtk_tool_button_new_from_stock(GTK_STOCK_ADD);
  pui->mtb_edit = gtk_tool_button_new_from_stock(GTK_STOCK_EDIT);
  pui->mtb_up = gtk_tool_button_new_from_stock(GTK_STOCK_GO_UP);
  pui->mtb_down = gtk_tool_button_new_from_stock(GTK_STOCK_GO_DOWN);
  pui->mtb_delete = gtk_tool_button_new_from_stock(GTK_STOCK_DELETE);
  pui->mtb_reminder = gtk_tool_button_new(NULL, NULL);
  pui->mtb_close = gtk_tool_button_new_from_stock(GTK_STOCK_QUIT);

  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->mtb_add),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->mtb_edit),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->mtb_up),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->mtb_down),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->mtb_delete),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->mtb_reminder),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->mtb_close),TRUE);

  /* Add all items to toolbar */
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), pui->mtb_add, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), gtk_separator_tool_item_new(), -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), pui->mtb_edit, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), gtk_separator_tool_item_new(), -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), pui->mtb_up, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), gtk_separator_tool_item_new(), -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), pui->mtb_down, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), gtk_separator_tool_item_new(), -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), pui->mtb_delete, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), gtk_separator_tool_item_new(), -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), pui->mtb_reminder, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), gtk_separator_tool_item_new(), -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), pui->mtb_close, -1);

  g_signal_connect(G_OBJECT(pui->mtb_add), "clicked",
		   G_CALLBACK(callback_newrecord), pui);
  g_signal_connect(G_OBJECT(pui->mtb_edit), "clicked",
		   G_CALLBACK(callback_editrecord), pui);
  g_signal_connect(G_OBJECT(pui->mtb_up), "clicked",
		   G_CALLBACK(callback_prevrecord), pui);
  g_signal_connect(G_OBJECT(pui->mtb_down), "clicked",
		   G_CALLBACK(callback_nextrecord), pui);
  g_signal_connect(G_OBJECT(pui->mtb_delete), "clicked",
		   G_CALLBACK(callback_deleterecord), pui);
  g_signal_connect(G_OBJECT(pui->mtb_reminder), "clicked",
		   G_CALLBACK(callback_alarm), pui);
  g_signal_connect(G_OBJECT(pui->mtb_close), "clicked",
		   G_CALLBACK(delete_event), NULL);

  /* Add toolbar HildonWindow */
#if PLAINGTK == 1
  gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(pui->main_toolbar), FALSE, FALSE, 5);
#else
  hildon_window_add_toolbar(main_window, GTK_TOOLBAR(pui->main_toolbar));
#endif
}

#if PLAINGTK == 1
static void create_secondary_toolbar(GtkWidget * main_window, GtkWidget *vbox, AppUIData *pui)
#else
  static void create_secondary_toolbar(HildonWindow * main_window, AppUIData *pui)
#endif
{
  GString *str;
  float totalkm;
  float totalfillkm;
  float totalfill;

  PangoAttrList *attrs;
  PangoAttribute *attr;


  /* Create toolbar */
  pui->secondary_toolbar = gtk_toolbar_new();

  /* Create toolbar items */

  /* Scale texts smaller: create pango attributes */
  attrs = pango_attr_list_new();
  attr = pango_attr_scale_new(PANGO_SCALE_X_SMALL);
  attr->start_index = 0;
  attr->end_index = -1;
  pango_attr_list_insert(attrs, attr);

  /* Car combo */
  pui->stb_car = gtk_tool_item_new();
#if MAEMO_VERSION_MAJOR == 5
  pui->stb_carbutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->stb_carbutton), _("Car"));
  pui->stb_carcombo = create_car_combo(pui, pui->stb_carbutton);
#endif
#if MAEMO_VERSION_MAJOR == 5
  gtk_container_add(GTK_CONTAINER(pui->stb_car), GTK_WIDGET(pui->stb_carbutton));
#else
  pui->stb_carcombo = create_car_combo(pui);
  gtk_container_add(GTK_CONTAINER(pui->stb_car), GTK_WIDGET(pui->stb_carcombo));

  /* Apply car combo selection */
  pui->stb_carapply = gtk_tool_button_new_from_stock(GTK_STOCK_OK);
#endif

  str=g_string_new(NULL);

  /* Total distance */
  totalkm=db_get_totalkm(stbstattime);
  g_string_printf(str, "%.0f ", SIlength2user(totalkm));
  if (curunit.lengthunit==SI)
    g_string_append_printf(str, _("km"));
  else
    g_string_append_printf(str, _("miles"));
  pui->stb_totalkm = gtk_tool_item_new();
  pui->stb_totalkmlabel=gtk_label_new(str->str);
  gtk_label_set_selectable(GTK_LABEL(pui->stb_totalkmlabel), TRUE);
  gtk_label_set_attributes(GTK_LABEL(pui->stb_totalkmlabel), attrs);
  gtk_container_add(GTK_CONTAINER(pui->stb_totalkm), GTK_WIDGET(pui->stb_totalkmlabel));

  /* Average consumption */
  totalfill=db_get_totalfill(stbstattime);
  totalfillkm=db_get_totalfillkm(stbstattime);
  if (totalfillkm != 0.0)
    g_string_printf(str, "%.1f ",SIconsumption2user(totalfill/totalfillkm*100));
  else
    g_string_printf(str, "%.1f ",0.0);
  if (curunit.consumeunit==SI)
    g_string_append_printf(str, _("l/100 km"));
  else
    g_string_append_printf(str, _("MPG"));
  pui->stb_avgconsum = gtk_tool_item_new();
  pui->stb_avgconsumlabel=gtk_label_new(str->str);
  gtk_label_set_selectable(GTK_LABEL(pui->stb_avgconsumlabel), TRUE);
  gtk_label_set_attributes(GTK_LABEL(pui->stb_avgconsumlabel), attrs);
  gtk_container_add(GTK_CONTAINER(pui->stb_avgconsum), GTK_WIDGET(pui->stb_avgconsumlabel));

  /* Total cost */
  g_string_printf(str, "%.0f %s", db_get_totalcost(), curunit.currency->str);
  pui->stb_totcost = gtk_tool_item_new();
  pui->stb_totcostlabel=gtk_label_new(str->str);
  gtk_label_set_selectable(GTK_LABEL(pui->stb_totcostlabel), TRUE);
  gtk_label_set_attributes(GTK_LABEL(pui->stb_totcostlabel), attrs);
  gtk_container_add(GTK_CONTAINER(pui->stb_totcost), GTK_WIDGET(pui->stb_totcostlabel));

  g_string_free(str, TRUE);

  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->stb_car),FALSE);
#if MAEMO_VERSION_MAJOR == 5
#else
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->stb_carapply),FALSE);
#endif
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->stb_totalkm),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->stb_avgconsum),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->stb_totcost),TRUE);

  /* Add all items to toolbar */
  gtk_toolbar_insert(GTK_TOOLBAR(pui->secondary_toolbar), pui->stb_car, -1);
#if MAEMO_VERSION_MAJOR == 5
#else
  gtk_toolbar_insert(GTK_TOOLBAR(pui->secondary_toolbar), pui->stb_carapply, -1);
#endif
  gtk_toolbar_insert(GTK_TOOLBAR(pui->secondary_toolbar), pui->stb_totalkm, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->secondary_toolbar), pui->stb_avgconsum, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->secondary_toolbar), pui->stb_totcost, -1);

#if MAEMO_VERSION_MAJOR == 5
  g_signal_connect(G_OBJECT(pui->stb_carbutton), "value-changed",
		   G_CALLBACK(car_apply_cb), pui);
#else
  g_signal_connect(G_OBJECT(pui->stb_carapply), "clicked",
		   G_CALLBACK(car_apply_cb), pui);
#endif

  /* Add toolbar HildonWindow */
#if PLAINGTK == 1
  gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(pui->secondary_toolbar), FALSE, FALSE, 5);
#else
  hildon_window_add_toolbar(main_window, GTK_TOOLBAR(pui->secondary_toolbar));
#endif
}

static
void enable_mainmenu_items(AppUIData *pui)
{
  gboolean dbopen;

  dbopen=db_open();

#if MAEMO_VERSION_MAJOR == 5
#else
  gtk_widget_set_sensitive(pui->mm_item_new, dbopen);
  gtk_widget_set_sensitive(pui->mm_item_edit, dbopen);
  gtk_widget_set_sensitive(pui->mm_item_delete, dbopen);
  gtk_widget_set_sensitive(pui->mm_item_monthly, dbopen);
  gtk_widget_set_sensitive(pui->mm_item_report, dbopen);
  gtk_widget_set_sensitive(pui->mm_item_close, dbopen);
  gtk_widget_set_sensitive(pui->mm_item_import, dbopen);
  gtk_widget_set_sensitive(pui->mm_item_export, dbopen);
#endif
  gtk_widget_set_sensitive(GTK_WIDGET(pui->mtb_add), dbopen);
  gtk_widget_set_sensitive(GTK_WIDGET(pui->mtb_edit), dbopen);
  gtk_widget_set_sensitive(GTK_WIDGET(pui->mtb_up), dbopen);
  gtk_widget_set_sensitive(GTK_WIDGET(pui->mtb_down), dbopen);
  gtk_widget_set_sensitive(GTK_WIDGET(pui->mtb_delete), dbopen);
}

static
void create_mainwin_menu(GtkWidget *root, AppUIData *pui)
{ 
#if MAEMO_VERSION_MAJOR == 5
  GtkWidget * button;

  pui->main_menu = HILDON_APP_MENU (hildon_app_menu_new ());

  pui->mm_item_new = hildon_button_new_with_text(HILDON_SIZE_AUTO,
						      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
						      _("Add record"),
						      NULL);
  hildon_app_menu_append (pui->main_menu, GTK_BUTTON (pui->mm_item_new));

  pui->mm_item_edit = hildon_button_new_with_text(HILDON_SIZE_AUTO,
						      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
						      _("Edit record"),
						      NULL);
  hildon_app_menu_append (pui->main_menu, GTK_BUTTON (pui->mm_item_edit));

  pui->mm_item_delete = hildon_button_new_with_text(HILDON_SIZE_AUTO,
						      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
						      _("Delete record"),
						      NULL);
  hildon_app_menu_append (pui->main_menu, GTK_BUTTON (pui->mm_item_delete));

  pui->mm_item_filter = hildon_button_new_with_text(HILDON_SIZE_AUTO,
						      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
						      _("Filter records"),
						    (filter_filterqueue_length() > 0)?
						    _("Active") : _("Not active"));
  hildon_app_menu_append (pui->main_menu, GTK_BUTTON (pui->mm_item_filter));

  pui->mm_item_stat = hildon_button_new_with_text(HILDON_SIZE_AUTO,
						      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
						      _("Statistics"),
						      NULL);
  hildon_app_menu_append (pui->main_menu, GTK_BUTTON (pui->mm_item_stat));

  pui->mm_item_report = hildon_button_new_with_text(HILDON_SIZE_AUTO,
						      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
						      _("Report"),
						      NULL);
  hildon_app_menu_append (pui->main_menu, GTK_BUTTON (pui->mm_item_report));

  pui->mm_item_driving_log = hildon_button_new_with_text(HILDON_SIZE_AUTO,
						      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
						      _("Driving log"),
							 (driving_log_logging() || driving_log_paused()) ? _("Logging") : _("Idle"));
  hildon_app_menu_append (pui->main_menu, GTK_BUTTON (pui->mm_item_driving_log));


  pui->mm_item_alarm = hildon_button_new_with_text(HILDON_SIZE_AUTO,
						      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
						      _("Reminders"),
						   pui->warn ? _("Active") : _("Not active"));
  hildon_app_menu_append (pui->main_menu, GTK_BUTTON (pui->mm_item_alarm));

  pui->mm_item_sortcolumn = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->mm_item_sortcolumn), _("Sort by"));
  pui->mm_combo_sortcolumn = create_column_combo(pui, pui->mm_item_sortcolumn);
  hildon_app_menu_append (pui->main_menu, GTK_BUTTON (pui->mm_item_sortcolumn));

  pui->mm_item_settings = hildon_button_new_with_text(HILDON_SIZE_AUTO,
						      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
						      _("Settings"),
						      NULL);
  hildon_app_menu_append (pui->main_menu, GTK_BUTTON (pui->mm_item_settings));

  button = hildon_gtk_radio_button_new (HILDON_SIZE_AUTO, NULL);
  gtk_button_set_label (GTK_BUTTON (button), _("Sort ascending"));
  g_signal_connect_after (button, "clicked", G_CALLBACK (sort_order_ascending_callback), pui);
  hildon_app_menu_add_filter (pui->main_menu, GTK_BUTTON (button));
  gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (button), FALSE);
  gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(button), maintablesortorder==GTK_SORT_ASCENDING);
 
  button = hildon_gtk_radio_button_new_from_widget (HILDON_SIZE_AUTO,
                                                    GTK_RADIO_BUTTON (button));
  gtk_button_set_label (GTK_BUTTON (button), _("Sort descending"));
  g_signal_connect_after (button, "clicked", G_CALLBACK (sort_order_descending_callback), pui);
  hildon_app_menu_add_filter (pui->main_menu, GTK_BUTTON (button));
  gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (button), FALSE);
  gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(button), maintablesortorder==GTK_SORT_DESCENDING);


  g_signal_connect(G_OBJECT(pui->mm_item_new), "clicked",
		   G_CALLBACK(callback_newrecord), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_edit), "clicked",
		   G_CALLBACK(callback_editrecord), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_alarm), "clicked",
		   G_CALLBACK(callback_alarm), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_driving_log), "clicked",
		   G_CALLBACK(driving_log_callback), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_sortcolumn), "value-changed",
		   G_CALLBACK(callback_select_sortcolumn), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_settings), "clicked",
		   G_CALLBACK(callback_settings), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_delete), "clicked",
		   G_CALLBACK(callback_deleterecord), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_filter), "clicked",
		   G_CALLBACK(callback_filter_records), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_stat), "clicked",
		   G_CALLBACK(callback_stat), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_report), "clicked",
		   G_CALLBACK(callback_report), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_alarm), "clicked",
		   G_CALLBACK(callback_alarm), pui);

  hildon_window_set_app_menu(HILDON_WINDOW(pui->app->mainwindow), pui->main_menu);

#else
  GSList *radio_menu_group = NULL;

  pui->main_menu = gtk_menu_new();
  pui->mm_menu_db = gtk_menu_new();
  pui->mm_menu_record = gtk_menu_new();
  pui->mm_menu_stat = gtk_menu_new();
  pui->mm_menu_view = gtk_menu_new();
  pui->mm_menu_toolbar = gtk_menu_new();
  pui->mm_menu_fontsize = gtk_menu_new();

  /* Create main menu items */
  pui->mm_item_db = gtk_menu_item_new_with_label(_("Database"));
  pui->mm_item_record = gtk_menu_item_new_with_label(_("Record"));
  pui->mm_item_stat = gtk_menu_item_new_with_label(_("Statistics"));
  pui->mm_item_alarm = gtk_menu_item_new_with_label(_("Reminders..."));
  pui->mm_item_view = gtk_menu_item_new_with_label(_("View"));
  pui->mm_item_settings = gtk_menu_item_new_with_label(_("Settings..."));
#if LIBHELP == 1
  pui->mm_item_help = gtk_menu_item_new_with_label(_("Help"));
#endif
  pui->mm_item_about = gtk_menu_item_new_with_label(_("About"));
  pui->mm_item_exit = gtk_menu_item_new_with_label(_("Exit"));

  /* Create database menu items */
  pui->mm_item_open = gtk_menu_item_new_with_label(_("Open..."));
  pui->mm_item_close = gtk_menu_item_new_with_label(_("Close"));
  pui->mm_item_import = gtk_menu_item_new_with_label(_("Import..."));
  pui->mm_item_export = gtk_menu_item_new_with_label(_("Export..."));

  /* Create record menu items */
  pui->mm_item_new = gtk_menu_item_new_with_label(_("New"));
  pui->mm_item_edit = gtk_menu_item_new_with_label(_("Edit"));
  pui->mm_item_delete = gtk_menu_item_new_with_label(_("Delete"));

  /* Create statistics menu items */
  pui->mm_item_quick = gtk_menu_item_new_with_label(_("Quick"));
  pui->mm_item_monthly = gtk_menu_item_new_with_label(_("Graphical"));
  pui->mm_item_report = gtk_menu_item_new_with_label(_("Yearly report"));

  /* Create view menu items */
  pui->main_menu_item_fullscreen = gtk_check_menu_item_new_with_label(_("Full screen"));
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pui->main_menu_item_fullscreen),
				 pui->mainfullscreen);
  pui->mm_item_toolbar = gtk_menu_item_new_with_label(_("Toolbars"));
  pui->mm_item_fontsize = gtk_menu_item_new_with_label(_("Table font size"));
  pui->mm_item_columnselect = gtk_menu_item_new_with_label(_("Select columns..."));
  pui->mm_item_filter = gtk_menu_item_new_with_label(_("Filter records..."));

  /* Create toolbar menu items */
  pui->mm_item_toolbar_main = gtk_check_menu_item_new_with_label(_("Buttons"));
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pui->mm_item_toolbar_main),
				 main_toolbar_visible);
  pui->mm_item_toolbar_secondary = gtk_check_menu_item_new_with_label(_("Information"));
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pui->mm_item_toolbar_secondary),
				 secondary_toolbar_visible);

  /* Create fontsize menu items */
  pui->mm_item_fontsize_x_small = gtk_radio_menu_item_new_with_label(radio_menu_group,_("Extra small"));
  radio_menu_group = gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM (pui->mm_item_fontsize_x_small));
  if (fontsize == XSMALL)
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pui->mm_item_fontsize_x_small), TRUE);
  pui->mm_item_fontsize_small = gtk_radio_menu_item_new_with_label(radio_menu_group,_("Small"));
  radio_menu_group = gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM (pui->mm_item_fontsize_small));
  if (fontsize == SMALL)
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pui->mm_item_fontsize_small), TRUE);
  pui->mm_item_fontsize_medium = gtk_radio_menu_item_new_with_label(radio_menu_group,_("Medium"));
  radio_menu_group = gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM (pui->mm_item_fontsize_medium));
  if (fontsize == MEDIUM)
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pui->mm_item_fontsize_medium), TRUE);
  pui->mm_item_fontsize_large = gtk_radio_menu_item_new_with_label(radio_menu_group,_("Large"));
  if (fontsize == LARGE)
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pui->mm_item_fontsize_large), TRUE);

  /* Add menu items to right menus */
  /* Main menu */
  gtk_menu_append(pui->main_menu, pui->mm_item_db);
  gtk_menu_append(pui->main_menu, pui->mm_item_record);
  gtk_menu_append(pui->main_menu, pui->mm_item_stat);
  gtk_menu_append(pui->main_menu, gtk_separator_menu_item_new());
  gtk_menu_append(pui->main_menu, pui->mm_item_alarm);
  gtk_menu_append(pui->main_menu, gtk_separator_menu_item_new());
  gtk_menu_append(pui->main_menu, pui->mm_item_view);
  gtk_menu_append(pui->main_menu, gtk_separator_menu_item_new());
  gtk_menu_append(pui->main_menu, pui->mm_item_settings);
  gtk_menu_append(pui->main_menu, gtk_separator_menu_item_new());
#if LIBHELP == 1
  gtk_menu_append(pui->main_menu, pui->mm_item_help);
#endif
  gtk_menu_append(pui->main_menu, pui->mm_item_about);
  gtk_menu_append(pui->main_menu, pui->mm_item_exit);

  /* Database menu */
  gtk_menu_append(pui->mm_menu_db, pui->mm_item_open);
  gtk_menu_append(pui->mm_menu_db, pui->mm_item_close);
  gtk_menu_append(pui->mm_menu_db, pui->mm_item_import);
  gtk_menu_append(pui->mm_menu_db, pui->mm_item_export);

  /* Record menu */
  gtk_menu_append(pui->mm_menu_record, pui->mm_item_new);
  gtk_menu_append(pui->mm_menu_record, pui->mm_item_edit);
  gtk_menu_append(pui->mm_menu_record, pui->mm_item_delete);

  /* Statistics menu */
  gtk_menu_append(pui->mm_menu_stat, pui->mm_item_quick);
  gtk_menu_append(pui->mm_menu_stat, pui->mm_item_monthly);
  gtk_menu_append(pui->mm_menu_stat, pui->mm_item_report);

  /* View menu */
  gtk_menu_append(pui->mm_menu_view, pui->main_menu_item_fullscreen);
  gtk_menu_append(pui->mm_menu_view, pui->mm_item_toolbar);
  gtk_menu_append(pui->mm_menu_view, pui->mm_item_fontsize);
  gtk_menu_append(pui->mm_menu_view, pui->mm_item_columnselect);
  gtk_menu_append(pui->mm_menu_view, pui->mm_item_filter);

  /* Toolbars menu */
  gtk_menu_append(pui->mm_menu_toolbar, pui->mm_item_toolbar_main);
  gtk_menu_append(pui->mm_menu_toolbar, pui->mm_item_toolbar_secondary);

  /* Font size menu */
  gtk_menu_append(pui->mm_menu_fontsize, pui->mm_item_fontsize_x_small);
  gtk_menu_append(pui->mm_menu_fontsize, pui->mm_item_fontsize_small);
  gtk_menu_append(pui->mm_menu_fontsize, pui->mm_item_fontsize_medium);
  gtk_menu_append(pui->mm_menu_fontsize, pui->mm_item_fontsize_large);

#if PLAINGTK == 1
  pui->mm_menubar = gtk_menu_bar_new();
  gtk_box_pack_start (GTK_BOX (root), pui->mm_menubar, FALSE, FALSE, 2);
  gtk_widget_show (pui->mm_menubar);
  pui->mm_item_file = gtk_menu_item_new_with_label ("fuelpad");
  gtk_widget_show (pui->mm_item_file);
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (pui->mm_item_file), pui->main_menu);
  gtk_menu_bar_append (GTK_MENU_BAR (pui->mm_menubar), pui->mm_item_file);
#else
  hildon_window_set_menu(HILDON_WINDOW(pui->app->mainwindow), GTK_MENU(pui->main_menu));
#endif
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(pui->mm_item_db), pui->mm_menu_db);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(pui->mm_item_record), pui->mm_menu_record);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(pui->mm_item_stat), pui->mm_menu_stat);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(pui->mm_item_view), pui->mm_menu_view);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(pui->mm_item_toolbar), pui->mm_menu_toolbar);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(pui->mm_item_fontsize), pui->mm_menu_fontsize);

  /* Attach the callback functions to the activate signal */
  g_signal_connect(G_OBJECT(pui->mm_item_alarm), "activate",
		   G_CALLBACK(callback_alarm), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_settings), "activate",
		   G_CALLBACK(callback_settings), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_quick), "activate",
		   G_CALLBACK(callback_quick_stat), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_monthly), "activate",
		   G_CALLBACK(callback_stat), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_report), "activate",
		   G_CALLBACK(callback_report), pui);
#if LIBHELP == 1
  g_signal_connect(G_OBJECT(pui->mm_item_help), "activate",
		   G_CALLBACK(callback_help), NULL);
#endif
  g_signal_connect(G_OBJECT(pui->mm_item_about), "activate",
		   G_CALLBACK(callback_about), NULL);
  g_signal_connect(G_OBJECT(pui->mm_item_exit), "activate",
		   G_CALLBACK(delete_event), pui);

  g_signal_connect(G_OBJECT(pui->mm_item_open), "activate",
		   G_CALLBACK(callback_open_db), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_close), "activate",
		   G_CALLBACK(callback_close_db), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_import), "activate",
		   G_CALLBACK(callback_import), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_export), "activate",
		   G_CALLBACK(callback_export), NULL);

  g_signal_connect(G_OBJECT(pui->mm_item_new), "activate",
		   G_CALLBACK(callback_newrecord), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_edit), "activate",
		   G_CALLBACK(callback_editrecord), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_delete), "activate",
		   G_CALLBACK(callback_deleterecord), pui);

  g_signal_connect(G_OBJECT(pui->main_menu_item_fullscreen), "toggled",
		   G_CALLBACK(cb_main_fullscreen), (gpointer)pui);
  g_signal_connect(G_OBJECT(pui->mm_item_columnselect), "activate",
		   G_CALLBACK(callback_select_columns), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_filter), "activate",
		   G_CALLBACK(callback_filter_records), pui);

  g_signal_connect(G_OBJECT(pui->mm_item_toolbar_main), "toggled",
		   G_CALLBACK(cb_main_toolbar), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_toolbar_secondary), "toggled",
		   G_CALLBACK(cb_secondary_toolbar), pui);

  g_signal_connect(G_OBJECT(pui->mm_item_fontsize_x_small), "toggled",
			   G_CALLBACK(cb_fontsize_x_small), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_fontsize_small), "toggled",
			   G_CALLBACK(cb_fontsize_small), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_fontsize_medium), "toggled",
			   G_CALLBACK(cb_fontsize_medium), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_fontsize_large), "toggled",
			   G_CALLBACK(cb_fontsize_large), pui);

#endif

  /* Make all menu widgets visible */
  gtk_widget_show_all(GTK_WIDGET(pui->main_menu));
}

static
gboolean timeout_func(gpointer data)
{
  *((gboolean *)data) = TRUE;
  
  return FALSE;
}

/*******************************************************************
 *
 * Public functions
 *
 *******************************************************************/

/**
 * \fn void get_store_and_iter(GtkTreeModel *model,
			GtkWidget *view,
			GtkTreeIter *iter,
			GtkListStore **store,
			GtkTreeIter *storeiter)
 * \brief Obtains the list store and iter of a view
 * \param model if model available or NULL
 * \param view the outermost tree view
 * \param iter if iter available or NULL
 * \param store pointer to a list store pointer
 * \param storeiter pointer to a list iter pointer (or NULL)
 *
 * This functions obtains the list store embedded inside a
 * GtkTreeModelSort and a GtkTreeModelFilter. It also converts the
 * iter of the outermost model to the iter of the innermost iter
 * corresponding to the list store.
 *
 */
void get_store_and_iter(GtkTreeModel *model,
			GtkWidget *view,
			GtkTreeIter *iter,
			GtkListStore **store,
			GtkTreeIter *storeiter)
{
  GtkTreeModel *sortable;
  GtkTreeModel *filter;
  GtkTreeIter filteriter;

  sortable=gtk_tree_view_get_model(GTK_TREE_VIEW(view));
  filter=gtk_tree_model_sort_get_model(GTK_TREE_MODEL_SORT(sortable));
  *store = GTK_LIST_STORE(gtk_tree_model_filter_get_model(GTK_TREE_MODEL_FILTER(filter)));
  if (model==NULL || iter==NULL) 
    storeiter=NULL;
  else
  {
    gtk_tree_model_sort_convert_iter_to_child_iter(GTK_TREE_MODEL_SORT(model), &filteriter, iter);
    gtk_tree_model_filter_convert_iter_to_child_iter(GTK_TREE_MODEL_FILTER(filter), storeiter, &filteriter);
  }
}

gchar *interface_file_chooser (gchar *header, 
				GtkFileChooserAction action, 
				const gchar *ext,
				gchar *help_id)
{
    GtkWidget *dialog;
    gchar *filename = NULL;
#if PLAINGTK == 0
    HildonFileSystemModel *model = NULL;
#endif

#if PLAINGTK == 1
    dialog = gtk_file_chooser_dialog_new (header,
					  GTK_WINDOW(app.mainwindow),
					  action,
					  GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					  action == GTK_FILE_CHOOSER_ACTION_SAVE ? GTK_STOCK_SAVE : GTK_STOCK_OPEN, 
					  GTK_RESPONSE_OK,
					  NULL);
#else
    if( (model = get_file_system_model(GTK_WIDGET(app.mainwindow)) ) == NULL)
    {
       PDEBUG("could not get file system model\n\n");
       return NULL;
    }
    dialog = hildon_file_chooser_dialog_new_with_properties
                    (GTK_WINDOW(app.mainwindow),
		             "file_system_model", model,
		             "action", action,
	         	     NULL);
#if HILDON == 1
    hildon_file_chooser_dialog_set_extension(HILDON_FILE_CHOOSER_DIALOG(dialog), (gchar *)ext);
#endif
#endif
#if LIBHELP == 1
    help_dialog_help_enable(GTK_DIALOG(dialog),
				   help_id,
				   app.osso);
#endif
    gtk_widget_show_all (GTK_WIDGET (dialog));

    if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK) {
        filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
    }
    gtk_widget_destroy(dialog);
    return filename;
}

GString *get_entry(GtkWidget *entry)
{
  GString *text;

  text=g_string_new(NULL);
  g_string_assign(text,gtk_entry_get_text(GTK_ENTRY(entry)));

  return text;
}

gboolean get_check_button(GtkWidget *check)
{
  gboolean checked;

#if MAEMO_VERSION_MAJOR == 5
  checked = hildon_check_button_get_active (HILDON_CHECK_BUTTON (check));
#else
  checked = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check));
#endif

  return checked;
}

void update_filter_button(AppUIData *pui, gboolean active)
{
#if MAEMO_VERSION_MAJOR == 5
  hildon_button_set_value(HILDON_BUTTON(pui->mm_item_filter), (active)?
						    _("Active") : _("Not active"));
#else
  /** \todo: update_filter_button for maemo 4 */
#endif
}

/**
 * \fn void update_reminder_toolbutton(AppUIData *pui, gboolean warn)
 * \brief Display the reminder icon
 * \param warn If TRUE, display "no active reminders" icon, otherwise
 * "active reminders" icon
 *
 * This callback displays a different icon depending on the parameter warn.
 *
 */
void update_reminder_toolbutton(AppUIData *pui, gboolean warn)
{
  static GdkPixbuf *pixbufred=NULL;
  static GdkPixbuf *pixbufgreen=NULL;
  static GtkWidget *imagered=NULL;
  static GtkWidget *imagegreen=NULL;
  GtkWidget *image;

#if MAEMO_VERSION_MAJOR == 5
  hildon_button_set_value(HILDON_BUTTON(pui->mm_item_alarm),pui->warn ? _("Active") : _("Not active"));
#else
  if (pixbufred == NULL) {
    pixbufred = gdk_pixbuf_new_from_xpm_data((const char **)&exclamation_red_xpm);
  }
  if (pixbufgreen == NULL) {
    pixbufgreen = gdk_pixbuf_new_from_xpm_data((const char **)&exclamation_green_xpm);
  }
  if (!GTK_IS_WIDGET(imagered)) {
    imagered = gtk_image_new_from_pixbuf(pixbufred);
  }
  if (!GTK_IS_WIDGET(imagegreen)) {
    imagegreen = gtk_image_new_from_pixbuf(pixbufgreen);
  }

  if (warn)
    image = imagered;
  else
    image = imagegreen;

  gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON(pui->mtb_reminder),image);
  gtk_widget_show_all(GTK_WIDGET(pui->app->mainwindow));

  /* Show or hide toolbars */
  main_toolbar_show_hide(pui);
  secondary_toolbar_show_hide(pui);
#endif
}

void update_driving_log_button(AppUIData *pui)
{
#if MAEMO_VERSION_MAJOR == 5
  hildon_button_set_value(HILDON_BUTTON(pui->mm_item_driving_log),(driving_log_logging() || driving_log_paused()) ? _("Logging") : _("Idle"));
#else
  /** \todo: update_drivinglog_button for Maemo 4 */
#endif
}

void main_toolbar_show_hide(AppUIData *pui)
{
  if (main_toolbar_visible)
    gtk_widget_show(GTK_WIDGET(pui->main_toolbar));
  else
    gtk_widget_hide(GTK_WIDGET(pui->main_toolbar));
}

void secondary_toolbar_show_hide(AppUIData *pui)
{
  if (secondary_toolbar_visible)
    gtk_widget_show(GTK_WIDGET(pui->secondary_toolbar));
  else
    gtk_widget_hide(GTK_WIDGET(pui->secondary_toolbar));
}


/**
 * \fn void update_car_changed(AppUIData *pui)
 * \brief Creates a combo box for selecting a unit
 * \param *pui Pointer to the UI data structure
 *
 * This function can be used after the current driver has been changed
 * to save the driver to gconf and update the labels in the main window.
 *
 */
void update_car_changed(AppUIData *pui)
{
  config_save();
  update_record_model(pui);
}

/**
 * \fn void update_driver_combo(gpointer combo)
 * \brief Update the contents of driver combo
 * \param combo Pointer to the already existing combo vox
 *
 * This function can be used to update the driver selection combo box
 * after adding a new driver or deleting one.
 *
 */
void update_driver_combo(gpointer combo)
{
#if MAEMO_VERSION_MAJOR == 5
  HildonTouchSelectorColumn *column;
  GtkTreeIter iter;
#endif
  GtkTreeModel *model;
  gint i, count;

#if MAEMO_VERSION_MAJOR == 5
  model = hildon_touch_selector_get_model(HILDON_TOUCH_SELECTOR(combo), 0);
  gtk_list_store_clear(GTK_LIST_STORE(model));
#else
  model = gtk_combo_box_get_model(GTK_COMBO_BOX(combo));
  count = gtk_tree_model_iter_n_children(model, NULL);
  for (i=0; i < count; i++)
    gtk_combo_box_remove_text(GTK_COMBO_BOX(combo), 0);
#endif
  fill_driver_combo(GTK_WIDGET(combo));
}


#if MAEMO_VERSION_MAJOR == 5
GtkWidget *create_driver_combo(AppUIData *pui, GtkWidget *button)
#else
GtkWidget *create_driver_combo(AppUIData *pui)
#endif
{
  GtkWidget *combo;

#if MAEMO_VERSION_MAJOR == 5
  combo = hildon_touch_selector_new_text ();
#else
  combo = gtk_combo_box_new_text();
#endif
  fill_driver_combo(combo);

#if MAEMO_VERSION_MAJOR == 5
  hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (button),
                                     HILDON_TOUCH_SELECTOR (combo));
#endif

  g_signal_connect (GTK_OBJECT (combo), "changed",
		    G_CALLBACK(driver_changed_cb), NULL);

  return combo;
}

/**
 * \fn void update_car_combo(gpointer widget)
 * \brief Update the contents of car combo
 * \param combo Pointer to the already existing combo vox
 *
 * This function can be used to update the car selection combo box
 * after adding a new car or deleting one.
 *
 */
void update_car_combo(gpointer widget)
{
#if MAEMO_VERSION_MAJOR == 5
  HildonTouchSelectorColumn *column;
  HildonTouchSelector *selector;
#endif
  GtkTreeModel *model;
  gint i, count;

#if MAEMO_VERSION_MAJOR == 5
  selector = hildon_picker_button_get_selector(HILDON_PICKER_BUTTON(widget));

  /* We'll need to block callbacks, otherwise they could be fired in between */
  g_signal_handlers_block_by_func(GTK_WIDGET(widget), G_CALLBACK(car_apply_cb), &ui);
  g_signal_handlers_block_by_func(GTK_WIDGET(selector), G_CALLBACK(car_changed_cb), NULL);

  model = hildon_touch_selector_get_model(HILDON_TOUCH_SELECTOR(selector), 0);
  gtk_list_store_clear(GTK_LIST_STORE(model));
  fill_car_combo(GTK_WIDGET(selector));

  g_signal_handlers_unblock_by_func(GTK_WIDGET(selector), G_CALLBACK(car_changed_cb), NULL);
  g_signal_handlers_unblock_by_func(GTK_WIDGET(widget), G_CALLBACK(car_apply_cb), &ui);

/*   hildon_touch_selector_remove_column (HILDON_TOUCH_SELECTOR(combo), 0);   */
/*   column = hildon_touch_selector_append_text_column (HILDON_TOUCH_SELECTOR(combo), */
/* 						     model, */
/* 						     TRUE); */
/*   hildon_touch_selector_column_set_text_column(column, 0); */
#else
  model = gtk_combo_box_get_model(GTK_COMBO_BOX(widget));
  count = gtk_tree_model_iter_n_children(model, NULL);
  for (i=0; i < count; i++)
    gtk_combo_box_remove_text(GTK_COMBO_BOX(widget), 0);
  fill_car_combo(GTK_WIDGET(widget));
#endif
}

#if MAEMO_VERSION_MAJOR == 5
GtkWidget *create_car_combo(AppUIData *pui, GtkWidget *button)
#else
GtkWidget *create_car_combo(AppUIData *pui)
#endif
{
  GtkWidget *combo;
  
#if MAEMO_VERSION_MAJOR == 5
  combo = hildon_touch_selector_new_text ();
#else
  combo = gtk_combo_box_new_text();
#endif
  fill_car_combo(combo);

#if MAEMO_VERSION_MAJOR == 5
  hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (button),
                                     HILDON_TOUCH_SELECTOR (combo));
#endif

  g_signal_connect (GTK_OBJECT (combo), "changed",
		    G_CALLBACK(car_changed_cb), NULL);

  return combo;
}

#if MAEMO_VERSION_MAJOR == 5
GtkWidget *create_andor_combo(GtkWidget *button)
#else
GtkWidget *create_andor_combo(void)
#endif
{
  enum
  {
    ANDOR_AND=0,
    ANDOR_OR
  };
  GtkWidget *combo;
  gchar *labels[] = {gettext_noop("Mandatory"),
		     gettext_noop("Optional")
  };

#if MAEMO_VERSION_MAJOR == 5
  combo = hildon_touch_selector_new_text ();
  hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (combo), gettext(labels[ANDOR_AND]));
  hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (combo), gettext(labels[ANDOR_OR]));
  hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR (combo), 0, 1);
  hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (button),
                                     HILDON_TOUCH_SELECTOR (combo));
#else
  combo = gtk_combo_box_new_text();
  gtk_combo_box_append_text(GTK_COMBO_BOX (combo), gettext(labels[ANDOR_AND]));
  gtk_combo_box_append_text(GTK_COMBO_BOX (combo), gettext(labels[ANDOR_OR]));
  gtk_combo_box_set_active(GTK_COMBO_BOX (combo), 1);
#endif
  return combo;
}

#if MAEMO_VERSION_MAJOR == 5
GtkWidget *create_column_combo(AppUIData *pui, GtkWidget *button)
#else
GtkWidget *create_column_combo(AppUIData *pui)
#endif
{
  GtkWidget *combo;
  gint i;
  GString *header;

#if MAEMO_VERSION_MAJOR == 5
  combo = hildon_touch_selector_new_text ();
  for (i=COL_DAY;i<=COL_NOTES;i++) {
    header = get_column_header_one_line(i);
    hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (combo), header->str);
    g_string_free(header, TRUE);
  }
  hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR (combo), 0, maintablesortcol);
  hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (button),
                                     HILDON_TOUCH_SELECTOR (combo));
#else
  combo = gtk_combo_box_new_text();
  for (col=0,j=0,i=COL_DAY;i<=COL_NOTES;i++,j++) {
    header = get_column_header_one_line(i);
    gtk_combo_box_append_text(GTK_COMBO_BOX (combo), header->str);
    g_string_free(header, TRUE);
  }
  gtk_combo_box_set_active(GTK_COMBO_BOX (combo), maintablesortcol);
#endif
  return combo;
}

/**
 * \fn void callback_editrecord( GtkAction * action, AppUIData *pui )
 * \brief Callback for editing a record
 *
 * This callback creates the record editing dialog
 *
 * \todo This should be changed so that fuelpad.c has a function for
 * obtaining the record data (i.e. sqlite would not be used here)
 *
 */
void callback_editrecord( GtkAction * action, AppUIData *pui )
{
  GtkWidget *dialog;
  GtkWidget *label;
  gchar *header[] = {gettext_noop("Edit a record")};

  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreeModel *model;
  gint64 id;
  char buf[BUFMAX];
#if PLAINGTK == 0
  guint year;
  guint month;
  guint day;
#else
  char *date;
#endif
  char notes[ENTRYNOTESMAX];
  gchar *convnotes;

  if (db_open()) {
    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->view));
    if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
      dialog = gtk_dialog_new_with_buttons(gettext(header[0]),
					   GTK_WINDOW(pui->app->mainwindow),
					   GTK_DIALOG_MODAL,
					   GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
					   GTK_STOCK_CANCEL,
					   GTK_RESPONSE_REJECT,
					   NULL);
      create_record_editwin(pui, dialog, 0);

      gtk_tree_model_get (model, &iter, COL_ID, &id, -1);
      /** \todo Data should be read from the store, not from database */
      if (SQLITE_OK == sqlite3_bind_int(ppStmtOneRecord, 1, id)) {
	while (SQLITE_ROW == sqlite3_step(ppStmtOneRecord)) {
#if PLAINGTK == 1
	  convdate(&date, datefmtstr[curunit.dateformat], "%Y-%m-%d", sqlite3_column_text(ppStmtOneRecord,0));

	  gtk_entry_set_text(GTK_ENTRY(pui->entrydate),date);
	  free(date);
#else
	  getyearmonthday(sqlite3_column_text(ppStmtOneRecord,0), 
			  &year,
			  &month,
			  &day);
#if MAEMO_VERSION_MAJOR == 5
	  hildon_date_button_set_date(HILDON_DATE_BUTTON(pui->entrydate), year, month-1, day); /* Month is betweewn 0 and 11 */
#else
	  hildon_date_editor_set_date(HILDON_DATE_EDITOR(pui->entrydate), year, month, day);
#endif
#endif

	  g_snprintf(buf,BUFMAX,"%.1f",SIlength2user(sqlite3_column_double(ppStmtOneRecord,1)));
	  gtk_entry_set_text(GTK_ENTRY(pui->entrykm),buf);

	  g_snprintf(buf,BUFMAX,"%.1f",SIlength2user(sqlite3_column_double(ppStmtOneRecord,2)));
	  gtk_entry_set_text(GTK_ENTRY(pui->entrytrip),buf);

	  g_snprintf(buf,BUFMAX,"%.2f",SIvolume2user(sqlite3_column_double(ppStmtOneRecord,3)));
	  gtk_entry_set_text(GTK_ENTRY(pui->entryfill),buf);

	  if ( (sqlite3_column_double(ppStmtOneRecord,3)>0) && 
	       ( abs(sqlite3_column_double(ppStmtOneRecord,10))<1e-3) ) {
#if MAEMO_VERSION_MAJOR == 5
	    hildon_check_button_set_active(HILDON_CHECK_BUTTON (pui->buttonnotfull), TRUE);
#else
	    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (pui->buttonnotfull), TRUE);
#endif
	  }

	  g_snprintf(buf,BUFMAX,"%.2f",sqlite3_column_double(ppStmtOneRecord,4));
	  gtk_entry_set_text(GTK_ENTRY(pui->entryprice),buf);

	  g_snprintf(buf,BUFMAX,"%.2f",sqlite3_column_double(ppStmtOneRecord,6));
	  gtk_entry_set_text(GTK_ENTRY(pui->entryservice),buf);

	  g_snprintf(buf,BUFMAX,"%.2f",sqlite3_column_double(ppStmtOneRecord,7));
	  gtk_entry_set_text(GTK_ENTRY(pui->entryoil),buf);

	  g_snprintf(buf,BUFMAX,"%.2f",sqlite3_column_double(ppStmtOneRecord,8));
	  gtk_entry_set_text(GTK_ENTRY(pui->entrytires),buf);

	/* Convert notes to utf8 (earlier versions might have imported wrongly encoded data */
	  g_snprintf(notes, ENTRYNOTESMAX, "%s", sqlite3_column_text(ppStmtOneRecord,9));
	  if (!g_utf8_validate(notes, -1, NULL))
	    convnotes=g_convert(notes, -1, "UTF-8", "ISO-8859-15", NULL, NULL, NULL);
	  else
	    convnotes=g_strndup(notes, strlen(notes));
	  gtk_entry_set_text(GTK_ENTRY(pui->entrynotes),convnotes);
	  g_free(convnotes);

#if LIBHELP == 1
	  help_dialog_help_enable(GTK_DIALOG(dialog),
					 HELP_ID_ADDRECORD,
					 pui->app->osso);
#endif

/* 	  g_signal_connect_swapped (dialog, "response", */
/* 				    G_CALLBACK (edit_record_response), */
/* 				    dialog); */
	  g_signal_connect (dialog, "response",
			    G_CALLBACK (edit_record_response),
			    &ui);
	}
      }
      else {
	PDEBUG("Problem obtaining record\n");
      }
    }
    else {
      dialog = gtk_dialog_new_with_buttons(gettext(header[0]),
					   GTK_WINDOW(pui->app->mainwindow),
					   GTK_DIALOG_MODAL,
					   GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
					   NULL);

      label=gtk_label_new(_("Select a record first"));
      gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label, TRUE, TRUE, 5);
      gtk_widget_show(label);

      g_signal_connect_swapped (dialog, "response",
				G_CALLBACK (gtk_widget_destroy),
				dialog);
    }
    sqlite3_reset(ppStmtOneRecord);

  }
  else {
    dialog = gtk_dialog_new_with_buttons(gettext(header[0]),
					 GTK_WINDOW(pui->app->mainwindow),
					 GTK_DIALOG_MODAL,
					 GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
					 NULL);

    label=gtk_label_new(_("Can't access database - editing records not possible"));
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label, TRUE, TRUE, 5);
    gtk_widget_show(label);

    /* Destroy the dialog when the user responds to it (e.g. clicks a button) */
    g_signal_connect_swapped (dialog, "response",
			      G_CALLBACK (gtk_widget_destroy),
			      dialog);
  }

  gtk_widget_show(dialog);
}

void callback_about( GtkAction * action, gpointer data )
{
  GtkWidget *dialog;
  gchar *authors[] = {"Julius Luukko", NULL};

  dialog = gtk_about_dialog_new ();
  gtk_about_dialog_set_name(GTK_ABOUT_DIALOG(dialog), "Fuelpad");
  gtk_about_dialog_set_version(GTK_ABOUT_DIALOG(dialog), VERSION); 
  gtk_about_dialog_set_copyright(GTK_ABOUT_DIALOG(dialog), 
				 "Copyright (C) 2007-2011 Julius Luukko");

  gtk_about_dialog_set_license(GTK_ABOUT_DIALOG(dialog),
			       "Licensed under GNU GPL v. 2");
  gtk_about_dialog_set_website(GTK_ABOUT_DIALOG(dialog),
			       "http://fuelpad.garage.maemo.org/");
  gtk_about_dialog_set_authors(GTK_ABOUT_DIALOG(dialog),
			       &authors);
  gtk_about_dialog_set_translator_credits(GTK_ABOUT_DIALOG(dialog),
			       _("translator-credits"));
  gtk_dialog_run(GTK_DIALOG (dialog));
  gtk_widget_destroy(dialog);
}

#if LIBHELP == 1
/**
 * \fn void help_dialog_help_enable(GtkDialog *dialog, const gchar *topic, osso_context_t *osso)
 * \brief Help enable wrapper
 *
 * Wrapper for enabling dialog help.
 *
 */
void help_dialog_help_enable(GtkDialog *dialog,
				 const gchar *topic,
				 osso_context_t *osso)
{
#if HILDON == 1
  hildon_help_dialog_help_enable(dialog, topic, osso);
#else
  ossohelp_dialog_help_enable(dialog, topic, osso);
#endif
}
#endif

#if LIBHELP == 1
void help_activated(GtkWidget *win, gchar *help_id)
{
   osso_return_t retval;

   if (!help_id)
      return;

#if HILDON == 1
   retval = hildon_help_show(
      app.osso, help_id, 0);
#else
   retval = ossohelp_show(
      app.osso, help_id, 0);
#endif
}
#endif

#if LIBHELP == 1
void callback_help( GtkAction * action, gpointer data )
{
#if HILDON == 1
/*   hildon_help_show(app.osso, HELP_ID_INTRO, HILDON_HELP_SHOW_DIALOG); */
  hildon_help_show(app.osso, HELP_ID_INTRO, 0);
#else
  ossohelp_show(app.osso, HELP_ID_INTRO, 0);
#endif
}
#endif

void create_mainwin_widgets(AppUIData *pui)
{
  GtkWidget *vbox;

  pui->mainvbox = gtk_vbox_new (FALSE, 0);
  vbox = pui->mainvbox;
  gtk_container_add (GTK_CONTAINER (app.mainwindow), vbox);

  create_mainwin_menu(vbox, pui);

#if MAEMO_VERSION_MAJOR == 5
  pui->datascrollwin = hildon_pannable_area_new();
  g_object_set(G_OBJECT(pui->datascrollwin), "mov-mode", HILDON_MOVEMENT_MODE_BOTH, NULL);
#else
  pui->datascrollwin = gtk_scrolled_window_new(NULL, NULL);
#endif
  gtk_box_pack_start (GTK_BOX(vbox), pui->datascrollwin, TRUE, TRUE, 0);

  pui->view = create_view_and_model ();
  gtk_container_add (GTK_CONTAINER (pui->datascrollwin), pui->view);

#if PLAINGTK == 1
  create_mainwin_toolbar(app.mainwindow, vbox, pui);
#else
  create_mainwin_toolbar(app.mainwindow, pui);
#endif

#if PLAINGTK == 1
  create_secondary_toolbar(app.mainwindow, vbox, pui);
#else
  create_secondary_toolbar(app.mainwindow, pui);
#endif

  enable_mainmenu_items(pui);

  gtk_widget_show_all(GTK_WIDGET(vbox));

  /*----------------------------------------------------------------*/
}


void ui_show_animation(GtkWidget **banner, const char *s)
{
#if PLAINGTK == 0
  *banner = hildon_banner_show_animation(GTK_WIDGET(app.mainwindow), NULL, s);
/*   PDEBUG(s); */
#else
  fputs(s,stderr);
#endif
}

void ui_show_banner(const char *s)
{
#if PLAINGTK == 0
  hildon_banner_show_information(GTK_WIDGET(app.mainwindow), NULL, s);
  PDEBUG(s);
#else
  fputs(s,stderr);
#endif
}

void ui_remove_banner(GtkWidget *banner)
{
#if PLAINGTK == 0
  gtk_widget_destroy(GTK_WIDGET(banner));
#endif
}

gboolean ui_connection_ensure(void)
{
  gboolean timedout;

  if (!netconnected) {
    ui_show_banner(_("Establishing network connection"));
    timedout = FALSE;
    g_timeout_add_full(G_PRIORITY_DEFAULT, 60000, timeout_func, &timedout, NULL);
    connect_ensure();
    while (netconnecting && !timedout) {
      if (gtk_events_pending()) {
	gtk_main_iteration();
      }
    }
    if (timedout) ui_show_banner("Establishing connection timed out");
  }

  return netconnected;
}


void ui_create_mainwin(void)
{
#if PLAINGTK == 0
  osso_return_t ret;
#endif

#if PLAINGTK == 0
  app.osso = osso_initialize ("org.luukko.fuelpad", VERSION, TRUE, NULL);
  if (app.osso == NULL) {
    fprintf (stderr, "osso_initialize failed.\n");
    exit (1);
  }

/*   ret = osso_rpc_set_default_cb_f (app.osso, dbus_callback, NULL); */
/*   if (ret != OSSO_OK) */
/*     { */
/*       fprintf (stderr, "osso_rpc_set_default_cb_f failed: %d.\n", ret); */
/*       exit (1); */
/*     } */

    osso_hw_set_event_cb(app.osso, NULL, osso_hw_state_callback, NULL);

#if MAEMO_VERSION_MAJOR == 5

    if (osso_rpc_run_system(app.osso, MCE_SERVICE, MCE_REQUEST_PATH,
			    MCE_REQUEST_IF, MCE_ACCELEROMETER_ENABLE_REQ, NULL, DBUS_TYPE_INVALID) == OSSO_OK) {
      g_printerr("INFO: Accelerometer enabled\n");
    } else {
      g_printerr("WARN: Cannot enable accelerometer\n");
    }

    /* Connect to session bus, add a match rule, install filter callback */
    app.system_bus = dbus_bus_get (DBUS_BUS_SYSTEM, NULL);
    if (app.system_bus) {
      dbus_bus_add_match (app.system_bus, MCE_SIGNAL_MATCH, NULL);
      dbus_connection_add_filter (app.system_bus,
				  (DBusHandleMessageFunction) mce_filter_func,
				  &app, NULL);
    }
#endif
    app.session_bus = dbus_bus_get (DBUS_BUS_SESSION, NULL);

#endif

#if PLAINGTK == 0
  /* Create the hildon program and setup the title */
  app.program = HILDON_PROGRAM(hildon_program_get_instance());
  g_set_application_name("Fuelpad");

  /* Create HildonWindow and set it to HildonProgram */
#if MAEMO_VERSION_MAJOR == 5
  app.mainwindow = hildon_stackable_window_new();
#else
  app.mainwindow = HILDON_WINDOW(hildon_window_new());
#endif

  hildon_program_add_window(app.program, app.mainwindow);

  /* Link app data to ui data */
  ui.app = &app;

  /* Callback for hardware keys */
  g_signal_connect(G_OBJECT(app.mainwindow), "key_press_event",
		   G_CALLBACK(key_press_cb), &ui);

#else
  /* Create the main window */
  app.mainwindow = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (app.mainwindow), "fuelpad");

  /* Link app data to ui data */
  ui.app = &app;
#endif

  g_signal_connect (G_OBJECT (app.mainwindow), "delete_event",
		    G_CALLBACK (delete_event), &ui);


  create_mainwin_widgets(&ui);

  /* View the reminder system toolbar item */
  ui.alarmview = create_alarmview_and_model (&ui);
  ui.warn = update_next_event(gtk_tree_view_get_model(GTK_TREE_VIEW(ui.alarmview)));
  update_reminder_toolbutton (&ui, ui.warn);


  /* Begin the main application */
  gtk_widget_show_all(GTK_WIDGET(app.mainwindow));

  /* Show or hide toolbars */
  main_toolbar_show_hide(&ui);
  secondary_toolbar_show_hide(&ui);

}

/**
 * \fn void ui_init(void)
 * \param argc argc from main()
 * \param argv argv from main()
 * \brief The first function to be called before any other ui functions
 *
 * This function initializes localization, gtk and reads gconf data.
 *
 */
void ui_init(int   argc,
	     char *argv[])
{
  /* Initialize localization. */
  setlocale(LC_ALL, "");
  bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
  bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
  textdomain(GETTEXT_PACKAGE);

#if MAEMO_VERSION_MAJOR == 5
  hildon_gtk_init (&argc, &argv);
#else
  gtk_init (&argc, &argv);
#endif
  config_init();

  if(!gnome_vfs_init()) {
    g_print("Failed to initialize GnomeVFS-libraries\n");
  }

  filter_init(&recfilter);
}

/**
 * \fn void ui_main_loop(void)
 * \brief User interface main loop
 *
 * This functions calls gtk_main() and when gtk_main() exists, saves
 * the gconf data.
 *
 */
void ui_main_loop(void)
{
  gtk_main ();
  config_save();
  filter_clear(&recfilter);
}
