/*
 * This file is part of fuelpad, the fuel diary
 *
 * Copyright (c) 2007-2010 Julius Luukko <julle.luukko@quicknet.inet.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <location/location-gps-device.h>
#include <location/location-gpsd-control.h>
#include <location/location-distance-utils.h>

#include <glib.h>

#include "gpsutil.h"

/*******************************************************************
 *
 * Public variables
 *
 *******************************************************************/

LocationGPSDControl *control;
LocationGPSDevice *device;
volatile double distance;

/*******************************************************************
 *
 * Private definitions
 *
 *******************************************************************/

/*******************************************************************
 *
 * Private enumerations
 *
 *******************************************************************/

/*******************************************************************
 *
 * Private variables
 *
 *******************************************************************/

gboolean logging=FALSE;

/*******************************************************************
 *
 * Prototypes of private functions
 *
 *******************************************************************/
static
void location_on_error(LocationGPSDControl *control, LocationGPSDControlError error, gpointer user_data);
static
void location_on_changed(LocationGPSDevice *device, AppUIData *pui);
static
gboolean start_location(gpointer data);

/*******************************************************************
 *
 * Private functions
 *
 *******************************************************************/

/*******************************************************************
 * Interface to the location framework
 *******************************************************************/

static
void location_on_error(LocationGPSDControl *control, LocationGPSDControlError error, gpointer user_data)
{ 
  switch (error) {
  case LOCATION_ERROR_USER_REJECTED_DIALOG:
    g_print("User didn't enable requested methods\n");
    break;
  case LOCATION_ERROR_USER_REJECTED_SETTINGS:
    g_print("User changed settings, which disabled location\n");
    break;
  case LOCATION_ERROR_BT_GPS_NOT_AVAILABLE:
    g_print("Problems with BT GPS\n");
    break;
  case LOCATION_ERROR_METHOD_NOT_ALLOWED_IN_OFFLINE_MODE:
    g_print("Requested method is not allowed in offline mode\n");
    break;
  case LOCATION_ERROR_SYSTEM:
    g_print("System error\n");
    break;
  }
}
 
static
void location_on_changed(LocationGPSDevice *device, AppUIData *pui)
{
  static double prev_time=0.0;
  static double prev_lat=0.0;
  static double prev_lon=0.0;
  static gboolean firstfix=TRUE;
  char *aika;

  if (!device)
    return;
 
  if (device->fix) {
    if (device->fix->fields & LOCATION_GPS_DEVICE_SPEED_SET) { // we'll rely on the result if speed set
      if (!firstfix) {
	if (logging) {
	  distance += 1000.0*location_distance_between(prev_lat, prev_lon, 
						       device->fix->latitude, 
						       device->fix->longitude);
	  prev_time = device->fix->time;
	  prev_lon = device->fix->longitude;
	  prev_lat = device->fix->latitude;
	}
      }

      if (firstfix==TRUE) {
	unixtimefmt((time_t)device->fix->time, &aika, "%Y-%m-%dT%H:%M.%S");
	g_print("First fix at %s\n",aika);
	free(aika);
	prev_time = device->fix->time;
	prev_lon = device->fix->longitude;
	prev_lat = device->fix->latitude;
	firstfix=FALSE;

      }

/*       g_idle_add(update_distance_ui,pui);  */
    }
  }
 
}

static
gboolean start_location(gpointer data)
{
  location_gpsd_control_start((LocationGPSDControl *) data);
  return FALSE;
}

/*******************************************************************
 *
 * Public functions
 *
 *******************************************************************/

void gps_init_location(AppUIData *pui)
{
  if (control == NULL) {
    control = location_gpsd_control_get_default();
  }
  if (device == NULL) {
    device = g_object_new(LOCATION_TYPE_GPS_DEVICE, NULL);

    g_signal_connect(control, "error-verbose", G_CALLBACK(location_on_error), NULL);
    g_signal_connect(device,  "changed", G_CALLBACK(location_on_changed), pui); 

    g_idle_add(start_location, control);

  }
  else {
    g_print("Already connected\n");
  }
}

void gps_stop_location(void)
{
  location_gpsd_control_stop(control);
  g_object_unref(device);
  g_object_unref(control);
  device = NULL;
  control = NULL;
}

gboolean gps_get_logging(void)
{
  return logging;
}

void gps_set_logging(gboolean set)
{
  logging = set;
}
