/*
 * This file is part of fuelpad, the fuel diary
 *
 * Copyright (c) 2007-2010 Julius Luukko <julle.luukko@quicknet.inet.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include <string.h>
#include <locale.h>
#include <libxml/parser.h>
#include <libgnomevfs/gnome-vfs.h>
#include <glib.h>

#include "geocode.h"
#include "util.h"

/*******************************************************************
 *
 * Public variables
 *
 *******************************************************************/

/*******************************************************************
 *
 * Private definitions
 *
 *******************************************************************/

#define SERVICE_ADDRESS "http://nominatim.openstreetmap.org/reverse?format=xml&lat=%f&lon=%f&zoom=18&addressdetails=1"

/*******************************************************************
 *
 * Private variables
 *
 *******************************************************************/

/*******************************************************************
 *
 * Prototypes of private functions
 *
 *******************************************************************/
static gboolean
retrieve_result(char **result, xmlDoc *doc, xmlNode *a_node);
static
gboolean nominatim_parse_xml(char **result, const char *buffer);

/*******************************************************************
 *
 * Private functions
 *
 *******************************************************************/

static gboolean
retrieve_result(char **result, xmlDoc *doc, xmlNode *a_node)
{
  xmlNode *cur_node = NULL;
  xmlChar *key;
  gboolean ret = FALSE;

  for (cur_node = a_node->xmlChildrenNode; cur_node; cur_node = cur_node->next) {
    if (cur_node->type == XML_ELEMENT_NODE) {
      if ((!xmlStrcmp(cur_node->name, (const xmlChar *)"result"))){
	ret = TRUE;
	key = xmlNodeListGetString(doc, cur_node->xmlChildrenNode, 1);
	*result = (char *)malloc((xmlStrlen(key)+1)*sizeof(char));
	strncpy(*result, key, xmlStrlen(key)+1);
      }

    }

  }

  return ret;
}

static
gboolean nominatim_parse_xml(char **result, const char *buffer)
{
  xmlDoc *doc = NULL;
  xmlNode *root_element = NULL;
  gboolean ret = FALSE;

  doc = xmlReadMemory(buffer, strlen(buffer), 
		      "http://nominatim.openstreetmap.org/",
		      NULL, 0);
  if (doc == NULL) {
    g_printerr("error: could not parse buffer\n");
    ret=FALSE;
  }
  else {
    root_element = xmlDocGetRootElement(doc);

    if (xmlStrcmp(root_element->name, (const xmlChar *) "reversegeocode")){
      g_printerr("Query result not well formed\n");
      ret=FALSE;
    }
    else { // query ok
      ret = retrieve_result(result, doc, root_element);
    }
  }

  xmlFreeDoc(doc);
  xmlCleanupParser();

  return ret;
}

/*******************************************************************
 *
 * Public functions
 *
 *******************************************************************/

gboolean nominatim_reverse_geocode(char **address, double lat, double lon)
{
  gboolean ret = FALSE;
  char *saved_locale;
  GString *query;
  gchar *inbuffer = NULL;
  gint size;
  GnomeVFSResult vfs_result;

  query = g_string_new(NULL);

  /* We'll need to ensure that decimal separator is . */
  saved_locale=store_locale(LC_NUMERIC);
  setlocale(LC_NUMERIC, "C");
  g_string_printf(query, SERVICE_ADDRESS, lat, lon);
/*   g_print("query: %s\n",query->str); */
  restore_locale(LC_NUMERIC, saved_locale);

  vfs_result = gnome_vfs_read_entire_file(query->str, &size, &inbuffer);

  if (vfs_result != GNOME_VFS_OK) {
    g_printerr("Network error in reverse geocoding\n!");
    ret = FALSE;
  }
  else {
    ret = nominatim_parse_xml(address, inbuffer);
  }

  g_string_free(query, TRUE);

  return ret;
  
}
