/* 
 * This file is part of fuelpad, the fuel diary
 *
 * Copyright (c) 2007, 2008 Julius Luukko <julle.luukko@quicknet.inet.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "ui.h"
#include "uialarm.h"

enum
{
  ALM_COL_DESC = 0,
  ALM_COL_KMLIM,
  ALM_COL_TIMELIM,
  ALM_COL_NEXTKM,
  ALM_COL_NEXTDATE,
  ALM_COL_LASTKM,
  ALM_COL_LASTDATE,
  ALM_COL_COLOR,
  ALM_COL_ID,
  ALM_NUM_COLS
};

enum
{
  EVENT_COL_DAY = 0,
  EVENT_COL_KM,
  EVENT_COL_ID,
  EVENT_COL_RECID,
  EVENT_NUM_COLS
};

#define RESPONSE_EDIT 1
#define RESPONSE_DELETE 2


extern char *datefmtstr[];
extern unit_t curunit;
extern double warnkm;
extern double warndays;
extern enum fontscale fontsize;
extern double fontscalefactors[];

gchar *alm_col_headers[] = {gettext_noop("Description"),
			   gettext_noop("Km limit"),
			   gettext_noop("Time limit"),
			   gettext_noop("Next km"),
			   gettext_noop("Next date"),
			   gettext_noop("Last km"),
			   gettext_noop("Last date")
};

gchar *alm_col_headers_non_SI[] = {gettext_noop("Description"),
				   gettext_noop("Miles limit"),
				   gettext_noop("Time limit"),
				   gettext_noop("Next miles"),
				   gettext_noop("Next date"),
				   gettext_noop("Last miles"),
				   gettext_noop("Last date")
};

/* Local function prototypes */
void update_alarm_headers(GtkWidget *view);
void update_event_headers(GtkWidget *view);
void deleteevent(AppUIData *pui, GtkDialog *parent);
void editevent(AppUIData *pui, GtkDialog *parent);
void create_alarm_editwin(AppUIData *pui, GtkWidget *box, gboolean add);
void callback_listevents( GtkAction * action, AppUIData *pui );
void callback_newevent( GtkAction * action, AppUIData *pui );
GtkResponseType callback_newalarm( GtkAction * action, AppUIData *pui );
void callback_editalarm( GtkAction * action, AppUIData *pui );
void callback_deletealarm( GtkAction * action, AppUIData *pui );
void newalarm_add(AppUIData *pui);
void callback_reminderactivated (GtkTreeView *treeview, GtkTreePath *path, GtkTreeViewColumn  *col, AppUIData *pui);
void callback_eventactivated (GtkTreeView *treeview, GtkTreePath *path, GtkTreeViewColumn  *col, AppUIData *pui);

/**
 * \fn gboolean delete_alarmwin( GtkWidget *widget, GdkEvent  *event, gpointer   data )
 * \brief Statistics window delete_event callback
 * \return FALSE
 */
static gboolean delete_alarmwin( GtkWidget *widget,
				 GdkEvent  *event,
				 gpointer   data )
{
  extern AppUIData ui;

  gtk_widget_destroy(GTK_WIDGET(ui.app->alarmwindow));
  return FALSE;
}

void destroy_alarmwin( GtkWidget *widget,
		      gpointer   data )
{
  AppUIData *a;

  a=(AppUIData *)data;

  gtk_widget_destroy(GTK_WIDGET(a->app->alarmwindow));
  a->app->alarmwindow=NULL;
}

void alarm_window_fullscreen(gpointer data)
{
  AppUIData *a;

  a=(AppUIData *)data;
  a->alarmfullscreen = !a->alarmfullscreen;
  if (a->alarmfullscreen) {
    gtk_window_fullscreen(GTK_WINDOW(a->app->alarmwindow));
  } else {
    gtk_window_unfullscreen(GTK_WINDOW(a->app->alarmwindow));
  }
}

void cb_alarm_fullscreen( GtkAction * action, gpointer data )
{
  alarm_window_fullscreen(data);
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(((AppUIData *)(data))->alarm_menu_item_fullscreen),
				 ((AppUIData *)(data))->alarmfullscreen);
}


#if PLAINGTK == 0
gboolean alarm_key_press_cb(GtkWidget * widget, GdkEventKey * event,
			    AppUIData *pui)
{
  switch (event->keyval) {

    case GDK_F6:
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pui->alarm_menu_item_fullscreen),
				     !pui->alarmfullscreen);
      return TRUE;
    }

    return FALSE;
}
#endif

static void
cell_data_func (GtkTreeViewColumn *col,
		      GtkCellRenderer   *renderer,
		      GtkTreeModel      *model,
		      GtkTreeIter       *iter,
		      gpointer           user_data)
{
/*   gfloat  speed; */
  double  speed;
  gchar   buf[20];

  gtk_tree_model_get(model, iter, ((data_func_t *)user_data)->col, &speed, -1);
  g_snprintf(buf, sizeof(buf), ((data_func_t *)user_data)->format, speed);

  g_object_set(renderer, "text", buf, NULL);
}

gint
sort_alarmiter_compare_func (GtkTreeModel *model,
                        GtkTreeIter  *a,
                        GtkTreeIter  *b,
                        gpointer      userdata)
{
  gint sortcol = GPOINTER_TO_INT(userdata); /* not used yet */
  gint ret = 0;

  switch (sortcol)
    {
    case ALM_COL_KMLIM:
    case ALM_COL_TIMELIM:
    case ALM_COL_LASTKM:
    case ALM_COL_NEXTKM: {
      
      double result1, result2;

      gtk_tree_model_get(model, a, sortcol, &result1, -1);
      gtk_tree_model_get(model, b, sortcol, &result2, -1);
    
      if (result1 != result2)
	{
	  if (!isnan(result1) && !isnan(result2))
	    ret = (result1 > result2) ? 1 : -1;
	  else {
	    if (isnan(result1) && !isnan(result2)) {
		  ret = -1;
	    }
	    else if (!isnan(result1) && isnan(result2)) {
	      ret = 1;
	    }
	  }
	}
      /* else both equal => ret = 0 */
    }
      break;
    case ALM_COL_LASTDATE:
    case ALM_COL_NEXTDATE: {
      char *date1, *date2;
      gchar *userdate1, *userdate2;

      gtk_tree_model_get(model, a, sortcol, &userdate1, -1);
      gtk_tree_model_get(model, b, sortcol, &userdate2, -1);

      date2sqlite(&date1, datefmtstr[curunit.dateformat], userdate1);
      date2sqlite(&date2, datefmtstr[curunit.dateformat], userdate2);

      ret=strcmp(date1,date2);

      free(date1);
      free(date2);
      g_free(userdate1);
      g_free(userdate2);

    }
      break;
    case ALM_COL_DESC:{

        gchar *name1, *name2;
	
        gtk_tree_model_get(model, a, sortcol, &name1, -1);
        gtk_tree_model_get(model, b, sortcol, &name2, -1);
	
        if (name1 == NULL || name2 == NULL)
	  {
	    if (name1 == NULL && name2 == NULL)
	      break; /* both equal => ret = 0 */
	    
	    ret = (name1 == NULL) ? -1 : 1;
	  }
        else
	  {
	    ret = g_utf8_collate(name1,name2);
	  }
	
        g_free(name1);
        g_free(name2);
      }
      break;
    }

  return ret;
}

void newevent_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  GtkListStore  *store;
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreeIter storeiter;
  GtkTreeModel *model;
  gint64 id;
  sqlite_int64 neweventid;
  sqlite_int64 recordid;

  GString *str;
  GString *notes;
  char *date;
  char *viewdate;
#if PLAINGTK == 0
  guint year;
  guint month;
  guint day;
#endif
  gdouble km;
  gdouble service;
  gdouble oil;
  gdouble tires;
  gboolean warn;

  if (!db_open()) {
       gtk_widget_destroy(GTK_WIDGET(dialog));
  }
  else {
    switch (arg1){
    case GTK_RESPONSE_ACCEPT: 
     selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->alarmview));
     if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
       gtk_tree_model_get (model, &iter, ALM_COL_ID, &id, -1);

#if PLAINGTK == 0
#if MAEMO_VERSION_MAJOR == 5
      hildon_date_button_get_date(HILDON_DATE_BUTTON(pui->entryeventdate), &year, &month, &day); /* Month is betweewn 0 and 11 */
      month++; 
#else
       hildon_date_editor_get_date(HILDON_DATE_EDITOR(pui->entryeventdate), &year, &month, &day);
#endif
       date=(char *)malloc(ENTRYDATEMAX*sizeof(char));
       g_snprintf(date, ENTRYDATEMAX, "%d-%02d-%02d",year, month, day);
#else
       date2sqlite(&date, datefmtstr[curunit.dateformat], 
		   gtk_entry_get_text(GTK_ENTRY(pui->entryeventdate)));
#endif
/*        km = user2SIlength(atof(gtk_entry_get_text(GTK_ENTRY(pui->entryeventkm)))); */
       str = get_entry(pui->entryeventkm);
       km  = user2SIlength(atof(str->str));
       g_string_free(str, TRUE);

/*        service = atof(gtk_entry_get_text(GTK_ENTRY(pui->entryeventservice))); */
       str = get_entry(pui->entryeventservice);
       service  = user2SIlength(atof(str->str));
       g_string_free(str, TRUE);

/*        oil = atof(gtk_entry_get_text(GTK_ENTRY(pui->entryeventoil))); */
       str = get_entry(pui->entryeventoil);
       oil  = user2SIlength(atof(str->str));
       g_string_free(str, TRUE);

/*        tires = atof(gtk_entry_get_text(GTK_ENTRY(pui->entryeventtires))); */
       str = get_entry(pui->entryeventtires);
       tires  = user2SIlength(atof(str->str));
       g_string_free(str, TRUE);

       notes = get_entry(pui->entryeventnotes);

       /* Save first to the record table */
       recordid = db_add_record(date, km, 0.0, 0.0, 0.0, 0.0, 0.0,
				service, oil, tires, notes->str);
       if (recordid<0) {
	  PDEBUG("Problem saving result");
	  free(date);
	}
	else { /* record succesfully inserted */

	  /* And then save to event table */
	  if (SQLITE_OK != sqlite3_bind_int64(ppStmtAddEvent, 1, id) /* alarmid */
	      || SQLITE_OK != sqlite3_bind_int(ppStmtAddEvent, 2, currentcar) /* carid */
	      || SQLITE_OK != sqlite3_bind_int(ppStmtAddEvent, 3, currentdriver) /* driverid */
	      || SQLITE_OK != sqlite3_bind_int64(ppStmtAddEvent, 4, recordid) /* recordid */
	      || SQLITE_OK != sqlite3_bind_text(ppStmtAddEvent, 5, date,
						-1, SQLITE_STATIC)
	      || SQLITE_OK != sqlite3_bind_int(ppStmtAddEvent, 6, km)
	      || SQLITE_DONE != sqlite3_step(ppStmtAddEvent)) {
	    /*     ui_show_banner("Problem saving result"); */
	    PDEBUG("Problem saving result\n");
	    free(date);
	  }
	  else { /* record succesfully inserted */
	    neweventid = db_get_last_insert_rowid();
	    convdate(&viewdate, datefmtstr[curunit.dateformat], "%Y-%m-%d", date);

	    /* Add to the event list, if open */
/* 	    if (pui->eventview != NULL) { */
	    if (GTK_IS_WIDGET(pui->eventview)) {
	      store = (GtkListStore *)gtk_tree_view_get_model(GTK_TREE_VIEW(pui->eventview));
	      gtk_list_store_append (store, &iter);
	      gtk_list_store_set (store, &iter,
				  EVENT_COL_DAY,viewdate,
				  EVENT_COL_KM,SIlength2user(km),
				  EVENT_COL_ID,neweventid,
				  EVENT_COL_RECID,recordid,
				  -1);
	    }

	    /* Add to the main view */
	    get_store_and_iter(NULL, pui->view, NULL, &store, NULL);
	    gtk_list_store_append (store, &storeiter);
	    gtk_list_store_set (store, &storeiter,
				COL_DAY, viewdate,
				COL_KM, SIlength2user(km),
				COL_TRIP, 0.0,
				COL_FILL, 0.0,
				COL_CONSUM, 0.0,
				COL_PRICE, 0.0,
				COL_PRICEPERLITRE, 0.0,
				COL_SERVICE, service,
				COL_OIL, oil,
				COL_TIRES, tires,
/* 			      /\* 			COL_INSURANCE, sqlite3_column_double(ppStmtRecords,5), *\/ */
/* 			      /\* 			COL_OTHER, sqlite3_column_double(ppStmtRecords,5), *\/ */
/* 				COL_NOTES, gtk_entry_get_text(GTK_ENTRY(pui->entryeventnotes)), */
				COL_NOTES, notes->str,
				COL_ID, recordid,
				COL_VISIBLE, TRUE,
				-1);
	  
	    update_totalkm(pui);
	    free(viewdate);
	    free(date);

	  }
	  sqlite3_reset(ppStmtAddEvent);
	}

       /* Calculate the new next event estimate */
       pui->warn = update_next_event(gtk_tree_view_get_model(GTK_TREE_VIEW(pui->alarmview)));
       update_reminder_toolbutton (pui, pui->warn);

       g_string_free(notes, TRUE);

       gtk_widget_destroy(GTK_WIDGET(dialog));
     }
     else
       {
	 PDEBUG("No alarm type selected\n");
       }
     break;
    case GTK_RESPONSE_REJECT:
      gtk_widget_destroy(GTK_WIDGET(dialog));
      break;
    }
  }
}

void editevent_response (GtkDialog *dialog, gint arg1, AppUIData *pui, 
			 sqlite_int64 recordid)
{
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkListStore  *store;
  GtkTreeIter    iter;
  gint64 id;
  sqlite_int64 curid;

  GString *str;
  GString *notes;
  char *date;
  char *viewdate;
#if PLAINGTK == 0
  guint year;
  guint month;
  guint day;
#endif
  gdouble km;
  gdouble service;
  gdouble oil;
  gdouble tires;

  switch (arg1) {
  case GTK_RESPONSE_ACCEPT:
      selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->eventview));
      if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
	gtk_tree_model_get (model, &iter, EVENT_COL_ID, &id, -1);

#if PLAINGTK == 0
#if MAEMO_VERSION_MAJOR == 5
      hildon_date_button_get_date(HILDON_DATE_BUTTON(pui->entryeventdate), &year, &month, &day); /* Month is betweewn 0 and 11 */
      month++; 
#else
       hildon_date_editor_get_date(HILDON_DATE_EDITOR(pui->entryeventdate), &year, &month, &day);
#endif
       date=(char *)malloc(ENTRYDATEMAX*sizeof(char));
       g_snprintf(date, ENTRYDATEMAX, "%d-%02d-%02d",year, month, day);
#else
       date2sqlite(&date, datefmtstr[curunit.dateformat], 
		   gtk_entry_get_text(GTK_ENTRY(pui->entryeventdate)));
#endif
/*        km = user2SIlength(atof(gtk_entry_get_text(GTK_ENTRY(pui->entryeventkm)))); */
       str = get_entry(pui->entryeventkm);
       km  = user2SIlength(atof(str->str));
       g_string_free(str, TRUE);

/*        service = atof(gtk_entry_get_text(GTK_ENTRY(pui->entryeventservice))); */
       str = get_entry(pui->entryeventservice);
       service  = user2SIlength(atof(str->str));
       g_string_free(str, TRUE);

/*        oil = atof(gtk_entry_get_text(GTK_ENTRY(pui->entryeventoil))); */
       str = get_entry(pui->entryeventoil);
       oil  = user2SIlength(atof(str->str));
       g_string_free(str, TRUE);

/*        tires = atof(gtk_entry_get_text(GTK_ENTRY(pui->entryeventtires))); */
       str = get_entry(pui->entryeventtires);
       tires  = user2SIlength(atof(str->str));
       g_string_free(str, TRUE);

       notes = get_entry(pui->entryeventnotes);

       /* Edit first the record table */
       if (db_update_record(recordid, date, km, -1.0, -1.0, -1.0, -1.0, -1.0,
			    service, oil, tires, notes->str) != recordid) {
	 PDEBUG("Problem saving result");
	 free(date);
       }
       else {
	 /* Store to event view */
	 convdate(&viewdate, datefmtstr[curunit.dateformat], "%Y-%m-%d", date);
	 store = (GtkListStore *)gtk_tree_view_get_model(GTK_TREE_VIEW(pui->eventview));
	 gtk_list_store_set (store, &iter,
			     EVENT_COL_DAY,viewdate,
			     EVENT_COL_KM,SIlength2user(km),
			     -1);

	 /* Store to the fuel database view */
	 get_store_and_iter(NULL, pui->view, NULL, &store, NULL);
	 if (gtk_tree_model_get_iter_first(GTK_TREE_MODEL(store), &iter)) {
	   gtk_tree_model_get(GTK_TREE_MODEL(store), &iter, COL_ID, &curid, -1);
	   while (&iter != NULL && curid != recordid) {
	     gtk_tree_model_iter_next(GTK_TREE_MODEL(store), &iter);
	     gtk_tree_model_get(GTK_TREE_MODEL(store), &iter, COL_ID, &curid, -1);
	   }

	   /* Store */
	   gtk_list_store_set (store, &iter,
			       COL_DAY, viewdate,
			       COL_KM, SIlength2user(km),
			       COL_SERVICE, service,
			       COL_OIL, oil,
			       COL_TIRES, tires,
			       COL_NOTES, notes->str,
			       COL_ID, recordid,
			       -1);
	    
	   update_totalkm(pui);
	 }
	 else
	   PDEBUG("Something wrong with finding the first iter\n");

	 free(viewdate);
	
       }
       sqlite3_reset(ppStmtUpdateRecord);

       
       /* Update event table */
       if (SQLITE_OK != sqlite3_bind_text(ppStmtUpdateEvent, 1,
					  date,
					  -1, SQLITE_STATIC)
	   || SQLITE_OK != sqlite3_bind_double(ppStmtUpdateEvent, 2,
					       km)
	   || SQLITE_OK != sqlite3_bind_int64(ppStmtUpdateEvent, 3,
					    id)
	   || SQLITE_DONE != sqlite3_step(ppStmtUpdateEvent)) {
	 PDEBUG("Problem updating event");
       }
       free(date);
       sqlite3_reset(ppStmtUpdateEvent);
       g_string_free(notes, TRUE);
      }
      else {
	PDEBUG("editevent: Something wrong with finding the selection\n");
      }
    break;
  case RESPONSE_EDIT:
    callback_listevents(NULL, pui);
    break;
  case GTK_RESPONSE_REJECT:
    break;
  }
  gtk_widget_destroy(GTK_WIDGET(dialog));
}

void listevents_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  switch (arg1) {
  case GTK_RESPONSE_CLOSE:
    break;
  case RESPONSE_EDIT:
    editevent(pui, dialog);
    break;
  case RESPONSE_DELETE:
    deleteevent(pui, dialog);
    break;
  }
  gtk_widget_destroy(GTK_WIDGET(dialog));
}

static GtkTreeModel *
create_and_fill_eventmodel (AppUIData *pui)
{
  GtkListStore  *store;
  GtkTreeIter    iter;
  GtkTreeSortable     *sortable;
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  gint64 id;
  char *date;

  if (db_open()) {
    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->alarmview));
    if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
      gtk_tree_model_get (model, &iter, ALM_COL_ID, &id, -1);

      store = gtk_list_store_new (EVENT_NUM_COLS, 
				  G_TYPE_STRING,  /* EVENT_COL_DAY */
				  G_TYPE_DOUBLE,  /* EVENT_COL_KM */
				  G_TYPE_INT64,    /* EVENT_COL_ID */
				  G_TYPE_INT64    /* EVENT_COL_RECID */
				  );

      if (SQLITE_OK == sqlite3_bind_int64(ppStmtGetEvents, 1, id)) {
	while(SQLITE_ROW == sqlite3_step(ppStmtGetEvents)) {

	  convdate(&date, datefmtstr[curunit.dateformat], "%Y-%m-%d", sqlite3_column_text(ppStmtGetEvents,0));

	  gtk_list_store_append (store, &iter);
	  gtk_list_store_set (store, &iter,
			      EVENT_COL_DAY,date,
			      EVENT_COL_KM,SIlength2user(sqlite3_column_double(ppStmtGetEvents,1)),
			      EVENT_COL_ID,sqlite3_column_int64(ppStmtGetEvents,2),
			      EVENT_COL_RECID,sqlite3_column_int64(ppStmtGetEvents,3),
			      -1);
	  free(date);

	}
      }

      sqlite3_reset(ppStmtGetEvents);

	  
      sortable = GTK_TREE_SORTABLE(store);

      gtk_tree_sortable_set_sort_column_id(sortable, EVENT_COL_DAY, GTK_SORT_DESCENDING);
    
      return GTK_TREE_MODEL (store);
    }
  }
  else
    return NULL;
}


static GtkWidget *
create_event_view_and_model (AppUIData *pui)
{
  GtkWidget *view;
  static data_func_t dfdata1;

  GtkTreeViewColumn   *col;
  GtkCellRenderer     *renderer;
  GtkTreeModel        *model;
  GtkTreeSelection    *select;

  model = create_and_fill_eventmodel (pui);

#if MAEMO_VERSION_MAJOR == 5
  view = hildon_gtk_tree_view_new_with_model (HILDON_UI_MODE_EDIT, 
					      model);
#else
  view = gtk_tree_view_new_with_model (model);
#endif

  /* --- Column #1 --- */
  col = gtk_tree_view_column_new();

  gtk_tree_view_column_set_title(col, _("Date"));

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", EVENT_COL_DAY);
  gtk_tree_view_column_set_resizable(col, TRUE);
  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), EVENT_COL_DAY);

  /* --- Column #2 --- */
  col = gtk_tree_view_column_new();

  gtk_tree_view_column_set_title(col, _("Km"));

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", EVENT_COL_KM);
  gtk_tree_view_column_set_resizable(col, TRUE);
  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), EVENT_COL_KM);

  dfdata1.col = EVENT_COL_KM;
  strncpy(dfdata1.format,"%.0f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata1, NULL);


  /* Invisible column to distinguish the results */
  /* --- Column #last-1 --- */

  col = gtk_tree_view_column_new();

  gtk_tree_view_column_set_title(col, "Id");

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", EVENT_COL_ID);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),EVENT_COL_ID), FALSE);

  /* Invisible column to link to fuel database */
  /* --- Column #last --- */

  col = gtk_tree_view_column_new();

  gtk_tree_view_column_set_title(col, "RecId");

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", EVENT_COL_RECID);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),EVENT_COL_RECID), FALSE);


  /* Change the unit dependent headers */
  update_event_headers(view);

  gtk_tree_view_set_model (GTK_TREE_VIEW (view), model);
  gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(view), TRUE);

  /* The tree view has acquired its own reference to the
   *  model, so we can drop ours. That way the model will
   *  be freed automatically when the tree view is destroyed */

  g_object_unref (model);

  select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
  gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);
/*   g_signal_connect (G_OBJECT (select), "changed", */
/* 		    G_CALLBACK (list_selection_changed_cb), */
/* 		    pui); */

  return view;

}

void callback_editevents( GtkAction * action, AppUIData *pui )
{
  editevent(pui, GTK_DIALOG(pui->app->alarmwindow));
}

void callback_deleteevents( GtkAction * action, AppUIData *pui )
{
  deleteevent(pui, GTK_DIALOG(pui->app->alarmwindow));
}

gint confirm_newreminder(AppUIData *pui )
{
  gint ret;
  GtkWidget *confirmdlg;
  gint confirm;
  gint added;

  ret=1;
  if (db_get_numof_alarmtypes(currentcar) == 0) {
#if PLAINGTK == 0
    confirmdlg = hildon_note_new_confirmation(GTK_WINDOW(pui->app->alarmwindow), 
		 _("You haven't defined any reminders yet. You will have to have a reminder before adding any events. Choose OK to add a reminder, Cancel to close the dialog without adding a reminder."));
#else
    confirmdlg = gtk_message_dialog_new(GTK_WINDOW(pui->app->alarmwindow),
					GTK_DIALOG_MODAL,
					GTK_MESSAGE_QUESTION,
					GTK_BUTTONS_OK_CANCEL,
					_("You haven't defined any reminders yet. You will have to have a reminder before adding any events. Choose OK to add a reminder, Cancel to close the dialog without adding a reminder."));
    gtk_widget_show(confirmdlg);
#endif
    confirm = gtk_dialog_run(GTK_DIALOG(confirmdlg));
    gtk_widget_destroy(GTK_WIDGET(confirmdlg));
    
    if(confirm == GTK_RESPONSE_OK) {
      added=callback_newalarm(NULL, pui);
      if (added != GTK_RESPONSE_OK)
	ret=0;
    }
    else
      ret=0;
  }

  return ret;
}

void callback_listevents( GtkAction * action, AppUIData *pui )
{
  GtkWidget *dialog;
  GtkWidget *vbox;
  GtkWidget *notebook;
  GtkWidget *vbox1;
  GtkWidget *hbox;
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreeModel *model;
  gchar *desc;
  GtkWidget *label;
  GtkWidget *button;
  GtkWidget *scrolledwin;

  if (!confirm_newreminder(pui))
    return;

  dialog = gtk_dialog_new_with_buttons(_("Events"),
				       GTK_WINDOW(pui->app->alarmwindow),
				       GTK_DIALOG_MODAL,
				       GTK_STOCK_CLOSE, GTK_RESPONSE_ACCEPT,
/* 				       GTK_STOCK_EDIT, RESPONSE_EDIT, */
/* 				       GTK_STOCK_DELETE, RESPONSE_DELETE, */
				       NULL);

  vbox = GTK_WIDGET(GTK_DIALOG(dialog)->vbox);
  vbox1 = gtk_vbox_new(FALSE, 0);

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->alarmview));
  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
    gtk_tree_model_get (model, &iter, ALM_COL_DESC, &desc, -1);
    label=gtk_label_new(desc);
  }
  else
    label=gtk_label_new(_("No reminder selected"));
  gtk_widget_show(label);
  gtk_box_pack_start(GTK_BOX(vbox1), label, TRUE, TRUE,0);

#if MAEMO_VERSION_MAJOR == 5
  scrolledwin = hildon_pannable_area_new();
#else
  scrolledwin = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwin), 
				 GTK_POLICY_NEVER, 
				 GTK_POLICY_AUTOMATIC);
#endif
  gtk_widget_set_size_request(scrolledwin, -1, DIALOG_MIN_HEIGHT0);
  gtk_box_pack_start (GTK_BOX(vbox1), scrolledwin, TRUE, TRUE, 0);

  pui->eventview = create_event_view_and_model (pui);
  g_signal_connect(pui->eventview, "row-activated",
		   G_CALLBACK (callback_eventactivated),
		   pui);
  gtk_container_add (GTK_CONTAINER (scrolledwin), pui->eventview);

  gtk_widget_show(pui->eventview);
  gtk_widget_show(scrolledwin);

  hbox = gtk_hbox_new(FALSE, 0); 
  gtk_box_pack_start (GTK_BOX(vbox1), hbox, FALSE, FALSE, 0);

  button = gtk_button_new_with_label(_("New"));
  gtk_box_pack_start(GTK_BOX(hbox),button, TRUE, TRUE,0);
  g_signal_connect (button, "clicked",
		    G_CALLBACK (callback_newevent),
		    pui);
  gtk_widget_show(button);

  button = gtk_button_new_with_label(_("Edit"));
  gtk_box_pack_start(GTK_BOX(hbox),button, TRUE, TRUE,0);
  g_signal_connect (button, "clicked",
		    G_CALLBACK (callback_editevents),
		    pui);
  gtk_widget_show(button);

  button = gtk_button_new_with_label(_("Delete"));
  gtk_box_pack_start(GTK_BOX(hbox),button, TRUE, TRUE,0);
  g_signal_connect (button, "clicked",
		    G_CALLBACK (callback_deleteevents),
		    pui);
  gtk_widget_show(button);

  gtk_widget_show(hbox);

  gtk_box_pack_start (GTK_BOX(vbox), vbox1, TRUE, TRUE, 0);
  gtk_widget_show(vbox1);

/*   label = gtk_label_new (_("Events")); */
/*   gtk_notebook_append_page (GTK_NOTEBOOK (notebook), vbox1, label); */

#if LIBHELP == 1
  help_dialog_help_enable(GTK_DIALOG(dialog),
				 HELP_ID_EDITEVENT,
				 pui->app->osso);
#endif

  g_signal_connect (dialog, "response",
		    G_CALLBACK (listevents_response),
		    pui);

  gtk_widget_show(dialog);

}

void create_event_editwin(AppUIData *pui, GtkWidget *dialog)
{
  enum
  {
    EVENT_EDIT_DATE=0,
    EVENT_EDIT_KM,
    EVENT_EDIT_MILES,
    EVENT_EDIT_SERVICE,
    EVENT_EDIT_OIL,
    EVENT_EDIT_TIRES,
    EVENT_EDIT_NOTES,
  };

#if MAEMO_VERSION_MAJOR == 5
  GtkWidget *scrollwin;
  GtkWidget *svbox;
#endif
  GtkWidget *vbox;
  GtkWidget *table;
  GtkWidget *label;
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreeModel *model;
  gchar *desc;
#if PLAINGTK == 1
  char *timebuf;
#endif
  gchar *labels[] = {gettext_noop("Date"),
		     gettext_noop("Km"),
		     gettext_noop("Miles"),
		     gettext_noop("Service"),
		     gettext_noop("Oil"),
		     gettext_noop("Tires"),
		     gettext_noop("Notes")
  };

  vbox = GTK_WIDGET(GTK_DIALOG(dialog)->vbox);

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->alarmview));
  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
    gtk_tree_model_get (model, &iter, ALM_COL_DESC, &desc, -1);
    label=gtk_label_new(desc);
  }
  else
    label=gtk_label_new(_("No reminder selected"));

#if MAEMO_VERSION_MAJOR == 5
  scrollwin = hildon_pannable_area_new();
  gtk_widget_set_size_request(scrollwin, -1, DIALOG_MIN_HEIGHTMAX);
  svbox = gtk_vbox_new (FALSE, 0);

  gtk_box_pack_start (GTK_BOX(svbox), label, TRUE, TRUE, 0);
  gtk_widget_show(label);

  table = gtk_table_new(5, 2, FALSE);
  gtk_box_pack_start (GTK_BOX (svbox), table, FALSE, FALSE, 5);

  pui->entryeventdate = hildon_date_button_new(HILDON_SIZE_FINGER_HEIGHT,
					  HILDON_BUTTON_ARRANGEMENT_VERTICAL);
/*   gtk_box_pack_start (GTK_BOX(svbox), pui->entryeventdate, TRUE, TRUE, 0); */
  gtk_table_attach(GTK_TABLE(table), pui->entryeventdate, 0, 2, 0, 1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryeventdate);

  /* First row, second entry */
  if (curunit.lengthunit==SI)
    label=gtk_label_new(gettext(labels[EVENT_EDIT_KM]));
  else
    label=gtk_label_new(gettext(labels[EVENT_EDIT_MILES]));
/*   gtk_box_pack_start (GTK_BOX(svbox), label, TRUE, TRUE, 0); */
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entryeventkm = hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryeventkm), ENTRYKMMAX);
  g_object_set(G_OBJECT(pui->entryeventkm),
	       "hildon-input-mode",
	       HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
	       NULL);
/*   gtk_box_pack_start (GTK_BOX(svbox), pui->entryeventkm, TRUE, TRUE, 0); */
  gtk_table_attach(GTK_TABLE(table), pui->entryeventkm, 1, 2, 1, 2,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryeventkm);

  label=gtk_label_new(gettext(labels[EVENT_EDIT_SERVICE]));
/*   gtk_box_pack_start (GTK_BOX(svbox), label, TRUE, TRUE, 0); */
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entryeventservice = hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryeventservice), ENTRYSERVICEMAX);
  g_object_set(G_OBJECT(pui->entryeventservice),
              "hildon-input-mode",
		 HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
		 NULL);
/*   gtk_box_pack_start (GTK_BOX(svbox), pui->entryeventservice, TRUE, TRUE, 0); */
  gtk_table_attach(GTK_TABLE(table), pui->entryeventservice, 1, 2, 2, 3,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryeventservice);

  label=gtk_label_new(gettext(labels[EVENT_EDIT_OIL]));
/*   gtk_box_pack_start (GTK_BOX(svbox), label, TRUE, TRUE, 0); */
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 3, 4,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entryeventoil = hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryeventoil), ENTRYOILMAX);
  g_object_set(G_OBJECT(pui->entryeventoil),
                "hildon-input-mode",
		 HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
		 NULL);
/*   gtk_box_pack_start (GTK_BOX(svbox), pui->entryeventoil, TRUE, TRUE, 0); */
  gtk_table_attach(GTK_TABLE(table), pui->entryeventoil, 1, 2, 3, 4,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryeventoil);

  label=gtk_label_new(gettext(labels[EVENT_EDIT_TIRES]));
/*   gtk_box_pack_start (GTK_BOX(svbox), label, TRUE, TRUE, 0); */
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 4, 5,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entryeventtires = hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryeventtires), ENTRYTIRESMAX);
  g_object_set(G_OBJECT(pui->entryeventtires),
                "hildon-input-mode",
		 HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
		 NULL);
/*   gtk_box_pack_start (GTK_BOX(svbox), pui->entryeventtires, TRUE, TRUE, 0); */
  gtk_table_attach(GTK_TABLE(table), pui->entryeventtires, 1, 2, 4, 5,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryeventtires);

  label=gtk_label_new(gettext(labels[EVENT_EDIT_NOTES]));
/*   gtk_box_pack_start (GTK_BOX(svbox), label, TRUE, TRUE, 0); */
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 5, 6,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entryeventnotes = hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryeventnotes), ENTRYNOTESMAX);
/*   gtk_box_pack_start (GTK_BOX(svbox), pui->entryeventnotes, TRUE, TRUE, 0); */
  gtk_table_attach(GTK_TABLE(table), pui->entryeventnotes, 1, 2, 5, 6,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryeventnotes);

  gtk_widget_show(table);

  gtk_widget_show(svbox);
  gtk_box_pack_start (GTK_BOX(vbox), scrollwin, TRUE, TRUE, 0);
  gtk_widget_show(scrollwin);
  hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA(scrollwin),svbox);

#else
  table = gtk_table_new(4, 4, FALSE);
  gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, FALSE, 5);

  /* Header row */
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);


  /* First row, first entry */
  label=gtk_label_new(gettext(labels[EVENT_EDIT_DATE]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

#if PLAINGTK == 1
  pui->entryeventdate = gtk_entry_new(); 
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryeventdate), ENTRYDATEMAX);
  gettimefmt(&timebuf, datefmtstr[curunit.dateformat]);
  gtk_entry_set_text(GTK_ENTRY(pui->entryeventdate),timebuf);
  free(timebuf);
#else
  pui->entryeventdate = hildon_date_editor_new();
#endif
  gtk_table_attach(GTK_TABLE(table), pui->entryeventdate, 1, 2, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryeventdate);

  /* First row, second entry */
  if (curunit.lengthunit==SI)
    label=gtk_label_new(gettext(labels[EVENT_EDIT_KM]));
  else
    label=gtk_label_new(gettext(labels[EVENT_EDIT_MILES]));
  gtk_table_attach(GTK_TABLE(table), label, 2, 3, 1, 2,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entryeventkm = gtk_entry_new(); 
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryeventkm), ENTRYKMMAX);
#if PLAINGTK == 0
  g_object_set(G_OBJECT(pui->entryeventkm),
#if HILDON == 1
              "hildon-input-mode",
		 HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
		 NULL);
#else
		 "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
  g_object_set(G_OBJECT(pui->entryeventkm), "autocap", FALSE, NULL);
#endif
#endif
  gtk_table_attach(GTK_TABLE(table), pui->entryeventkm, 3, 4, 1, 2,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryeventkm);

  /* Second row, first entry */
  label=gtk_label_new(gettext(labels[EVENT_EDIT_SERVICE]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entryeventservice = gtk_entry_new(); 
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryeventservice), ENTRYSERVICEMAX);
#if PLAINGTK == 0
  g_object_set(G_OBJECT(pui->entryeventservice),
#if HILDON == 1
              "hildon-input-mode",
		 HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
		 NULL);
#else
                 "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
  g_object_set(G_OBJECT(pui->entryeventservice), "autocap", FALSE, NULL);
#endif
#endif
  gtk_table_attach(GTK_TABLE(table), pui->entryeventservice, 1, 2, 2, 3, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
  gtk_widget_show(pui->entryeventservice);

  /* Second row, second entry */
  label=gtk_label_new(gettext(labels[EVENT_EDIT_OIL]));
  gtk_table_attach(GTK_TABLE(table), label, 2, 3, 2, 3, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
  gtk_widget_show(label);

  pui->entryeventoil = gtk_entry_new(); 
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryeventoil), ENTRYOILMAX);
#if PLAINGTK == 0
  g_object_set(G_OBJECT(pui->entryeventoil),
#if HILDON == 1
                "hildon-input-mode",
		 HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
		 NULL);
#else
                 "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
  g_object_set(G_OBJECT(pui->entryeventoil), "autocap", FALSE, NULL);
#endif
#endif
  gtk_table_attach(GTK_TABLE(table), pui->entryeventoil, 3, 4, 2, 3, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
  gtk_widget_show(pui->entryeventoil);

  /* Third row, first entry */
  label=gtk_label_new(gettext(labels[EVENT_EDIT_TIRES]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 3, 4, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
  gtk_widget_show(label);

  pui->entryeventtires = gtk_entry_new(); 
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryeventtires), ENTRYTIRESMAX);
#if PLAINGTK == 0
  g_object_set(G_OBJECT(pui->entryeventtires),
#if HILDON == 1
                "hildon-input-mode",
		 HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
		 NULL);
#else
                 "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
  g_object_set(G_OBJECT(pui->entryeventtires), "autocap", FALSE, NULL);
#endif
#endif
  gtk_table_attach(GTK_TABLE(table), pui->entryeventtires, 1, 2, 3, 4, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
  gtk_widget_show(pui->entryeventtires);

  /* Third row, second entry */
  label=gtk_label_new(gettext(labels[EVENT_EDIT_NOTES]));
  gtk_table_attach(GTK_TABLE(table), label, 2, 3, 3, 4, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entryeventnotes = gtk_entry_new(); 
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryeventnotes), ENTRYNOTESMAX);
  gtk_table_attach(GTK_TABLE(table), pui->entryeventnotes, 3, 4, 3, 4, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryeventnotes);


  gtk_widget_show(table);
#endif
}

void callback_newevent( GtkAction * action, AppUIData *pui )
{
  GtkWidget *dialog;
  GtkWidget *vbox;
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreeModel *model;
  gchar *desc;

  if (confirm_newreminder(pui)) {

    dialog = gtk_dialog_new_with_buttons(_("Add a new event"),
					 GTK_WINDOW(pui->app->alarmwindow),
					 GTK_DIALOG_MODAL,
					 GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
					 GTK_STOCK_CANCEL,
					 GTK_RESPONSE_REJECT,
					 NULL);

    vbox = GTK_WIDGET(GTK_DIALOG(dialog)->vbox);

    create_event_editwin(pui, dialog);

    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->alarmview));
    if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
      gtk_tree_model_get (model, &iter, ALM_COL_DESC, &desc, -1);
      gtk_entry_set_text(GTK_ENTRY(pui->entryeventnotes),desc);
    }

#if LIBHELP == 1
    help_dialog_help_enable(GTK_DIALOG(dialog),
				   HELP_ID_ADDEVENT,
				   pui->app->osso);
#endif

    g_signal_connect (dialog, "response",
		      G_CALLBACK (newevent_response),
		      pui);

    /*   gtk_widget_show_all (GTK_WIDGET (dialog)); */
    gtk_widget_show (dialog);
  }

}

void deleteevent(AppUIData *pui, GtkDialog *parent)
{
  GtkWidget *confirmdlg;
  GtkWidget *check;
  GtkWidget *separator;
  gint confirm;
  gboolean delrecord;
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreeModel *model;
  GtkListStore *store;
  gint64 id;
  gint64 recid;
  gboolean warn;

  if (db_open()) { /* delete */

#if PLAINGTK == 0
    confirmdlg = hildon_note_new_confirmation(GTK_WINDOW(parent), 
		 _("Are you sure you want to delete the current event"));
#else
    confirmdlg = gtk_message_dialog_new(GTK_WINDOW(parent),
					GTK_DIALOG_MODAL,
					GTK_MESSAGE_QUESTION,
					GTK_BUTTONS_OK_CANCEL,
				       _("Are you sure you want to delete the current event"));

    gtk_widget_show(confirmdlg);
#endif

    separator=gtk_hseparator_new();
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(confirmdlg)->vbox), separator, TRUE, TRUE, 0);
    gtk_widget_show(separator);

    check=gtk_check_button_new_with_label(_("Delete the corresponding record in the fuel database"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (check), TRUE);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(confirmdlg)->vbox), check, TRUE, TRUE, 0);
    gtk_widget_show(check);

    confirm = gtk_dialog_run(GTK_DIALOG(confirmdlg));

    delrecord=gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check));
    gtk_widget_destroy(GTK_WIDGET(confirmdlg));
    
    if(confirm == GTK_RESPONSE_OK) {
      selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->eventview));
      if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
	gtk_tree_model_get (model, &iter, EVENT_COL_ID, &id, EVENT_COL_RECID, &recid, -1);
	if (SQLITE_OK != sqlite3_bind_int64(ppStmtDeleteEvent, 1, id)
	    ||SQLITE_DONE != sqlite3_step(ppStmtDeleteEvent)) {
	  PDEBUG("Problem deleting the event");
	  sqlite3_reset(ppStmtDeleteEvent);
	}
	else { /* remove the record if succesfully removed from db */
	  sqlite3_reset(ppStmtDeleteEvent);

	  /* Remove from the eventview */
	  store = (GtkListStore *)gtk_tree_view_get_model(GTK_TREE_VIEW(pui->eventview));
	  gtk_list_store_remove(store, &iter);

	  /* Update the next event estimates */
	  pui->warn = update_next_event(gtk_tree_view_get_model(GTK_TREE_VIEW(pui->alarmview)));
	  update_reminder_toolbutton (pui, pui->warn);
	  if (SQLITE_OK != sqlite3_bind_int64(ppStmtDeleteRecord, 1, recid)
	      ||SQLITE_DONE != sqlite3_step(ppStmtDeleteRecord)) {
	    PDEBUG("Problem deleting the record, maybe it has been already deleted?");
	  }
	  else { /* remove the record if succesfully removed from db */
	    update_record_model(pui);
	  }
	  sqlite3_reset(ppStmtDeleteRecord);
	}
      }

    }
  }
}

void editevent(AppUIData *pui, GtkDialog *parent)
{
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreeModel *model;
  GtkWidget *dialog;
  GtkWidget *vbox;
  GtkWidget *label;
  gint64 id;
  sqlite_int64 recordid;
  char *date;
  char buf[BUFMAX];
#if PLAINGTK == 0
  guint year;
  guint month;
  guint day;
#endif
  gint rsponse;
  gboolean warn;
  gchar *header[] = {gettext_noop("Edit an event")};

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->eventview));
  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
    dialog = gtk_dialog_new_with_buttons(gettext(header[0]),
					 GTK_WINDOW(parent),
					 GTK_DIALOG_MODAL,
					 GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
					 GTK_STOCK_CANCEL,
					 GTK_RESPONSE_REJECT,
					 NULL);

    vbox = GTK_WIDGET(GTK_DIALOG(dialog)->vbox);

    create_event_editwin(pui, dialog);

    gtk_tree_model_get (model, &iter, EVENT_COL_ID, &id, -1);

      if (SQLITE_OK == sqlite3_bind_int64(ppStmtGetOneEvent, 1, id)) {
	while (SQLITE_ROW == sqlite3_step(ppStmtGetOneEvent)) {
#if PLAINGTK == 1
	  convdate(&date, datefmtstr[curunit.dateformat], "%Y-%m-%d", sqlite3_column_text(ppStmtGetOneEvent,0));

	  gtk_entry_set_text(GTK_ENTRY(pui->entryeventdate),date);
	  free(date);
#else
	  convdate(&date, "%Y", "%Y-%m-%d", sqlite3_column_text(ppStmtGetOneEvent,0));
	  year=atoi(date);
	  free(date);

	  convdate(&date, "%m", "%Y-%m-%d", sqlite3_column_text(ppStmtGetOneEvent,0));
	  month=atoi(date);
	  free(date);

	  convdate(&date, "%d", "%Y-%m-%d", sqlite3_column_text(ppStmtGetOneEvent,0));
	  day=atoi(date);
	  free(date);

#if MAEMO_VERSION_MAJOR == 5
	  hildon_date_button_set_date(HILDON_DATE_BUTTON(pui->entryeventdate), year, month-1, day); /* Month is betweewn 0 and 11 */
#else
	  hildon_date_editor_set_date(HILDON_DATE_EDITOR(pui->entryeventdate), year, month, day);
#endif
#endif

	  g_snprintf(buf,BUFMAX,"%.1f",SIlength2user(sqlite3_column_double(ppStmtGetOneEvent,1)));
	  gtk_entry_set_text(GTK_ENTRY(pui->entryeventkm),buf);

	  recordid=sqlite3_column_double(ppStmtGetOneEvent,2);

	}
      }

      sqlite3_reset(ppStmtGetOneEvent);

      if (SQLITE_OK == sqlite3_bind_int64(ppStmtOneRecord, 1, recordid)) {
	while (SQLITE_ROW == sqlite3_step(ppStmtOneRecord)) {

	  g_snprintf(buf,BUFMAX,"%.2f",sqlite3_column_double(ppStmtOneRecord,5));
	  gtk_entry_set_text(GTK_ENTRY(pui->entryeventservice),buf);

	  g_snprintf(buf,BUFMAX,"%.2f",sqlite3_column_double(ppStmtOneRecord,6));
	  gtk_entry_set_text(GTK_ENTRY(pui->entryeventoil),buf);

	  g_snprintf(buf,BUFMAX,"%.2f",sqlite3_column_double(ppStmtOneRecord,7));
	  gtk_entry_set_text(GTK_ENTRY(pui->entryeventtires),buf);

	  gtk_entry_set_text(GTK_ENTRY(pui->entryeventnotes),sqlite3_column_text(ppStmtOneRecord,8));

	}
      }

      sqlite3_reset(ppStmtOneRecord);

#if LIBHELP == 1
      help_dialog_help_enable(GTK_DIALOG(dialog),
				     HELP_ID_EDITEVENT,
				     pui->app->osso);
#endif

      gtk_widget_show (dialog);
      rsponse = gtk_dialog_run(GTK_DIALOG(dialog));
      editevent_response(GTK_DIALOG(dialog), rsponse, pui, recordid);
      pui->warn = update_next_event(gtk_tree_view_get_model(GTK_TREE_VIEW(pui->alarmview)));
      update_reminder_toolbutton (pui, pui->warn);

  }
  else {
    dialog = gtk_dialog_new_with_buttons(gettext(header[0]),
					 GTK_WINDOW(parent),
					 GTK_DIALOG_MODAL,
					 GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
					 GTK_STOCK_CANCEL,
					 GTK_RESPONSE_REJECT,
					 NULL);

    label=gtk_label_new(_("Select an event first"));
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label, TRUE, TRUE, 5);
    gtk_widget_show(label);

#if LIBHELP == 1
    help_dialog_help_enable(GTK_DIALOG(dialog),
				   HELP_ID_EDITEVENT,
				   pui->app->osso);
#endif

    g_signal_connect_swapped (dialog, "response",
			      G_CALLBACK (gtk_widget_destroy),
			      dialog);
    gtk_widget_show (dialog);
  }


}

/**
 * \fn GString *get_alarm_column_header(gint col)
 * \brief Returns the header of column col
 * \param col the column number
 * \return Pointer to a newly allocated column header string
 *
 * This functions returns the column header taking the used unit into
 * account. The string is allocated from the heap. After usage, the
 * string must be freed with g_string_free.
 *
 */
GString *get_alarm_column_header(gint col)
{
  GString *header;

  header = g_string_new(NULL);

  switch (col) {
  case ALM_COL_DESC:
  case ALM_COL_TIMELIM:
  case ALM_COL_NEXTDATE:
  case ALM_COL_LASTDATE:
    g_string_assign(header, gettext(alm_col_headers[col]));
    break;
  case ALM_COL_KMLIM:
  case ALM_COL_LASTKM:
  case ALM_COL_NEXTKM:
    if (curunit.lengthunit==SI)
      g_string_assign(header, gettext(alm_col_headers[col]));
    else
      g_string_assign(header, gettext(alm_col_headers_non_SI[col]));
    break;
  }
}

void update_alarm_headers(GtkWidget *view)
{
  GtkTreeViewColumn   *col;
  GtkWidget           *label;
  GString *header;
  gint i;

  for (i=0;i<ALM_COL_COLOR;i++) {
    col = gtk_tree_view_get_column(GTK_TREE_VIEW(view), i);
    label = gtk_tree_view_column_get_widget(col); 
    header = get_alarm_column_header(i);
    gtk_label_set_text(GTK_LABEL(label), header->str);
    g_string_free(header, TRUE);
  }
}

void update_event_headers(GtkWidget *view)
{
  GtkTreeViewColumn   *col;

  col = gtk_tree_view_get_column(GTK_TREE_VIEW(view), ALM_COL_KMLIM);
  if (curunit.lengthunit==SI)
    gtk_tree_view_column_set_title(col, _("Km"));
  else
    gtk_tree_view_column_set_title(col, _("Miles"));
}

gboolean update_next_event(GtkTreeModel *store)
{
  GtkTreeIter    iter;
  gint64 id;
  char *curdate;
  char *lastdate;
  char *userlastdate;
  char *estdate;
  char *userestdate;
  char warnbuf[ENTRYDATEMAX];
  char *estdays;
  double km;
  double distance;
  double interval;
  double lastkm;
  double estkm;
  gboolean expired;
  gboolean warn;
  gboolean anywarnings;
  glong warnday;

  anywarnings=FALSE;
  if (gtk_tree_model_get_iter_first(GTK_TREE_MODEL(store), &iter)) {
    lastkm=db_get_last_km();
    gettime(&curdate);
    do {
      gtk_tree_model_get(GTK_TREE_MODEL(store), &iter, 
			 ALM_COL_ID, &id,
			 ALM_COL_KMLIM, &distance,
			 ALM_COL_TIMELIM, &interval,
			 -1);
      distance=user2SIlength(distance);
      db_get_last_event(id, &lastdate, &km);
      if (distance > 0.0) { /* Distance limit given */
	db_estimate_date(km+distance,&estdate,lastdate,interval);
	if (lastdate != NULL && estdate != NULL && strlen(lastdate)>0) {
	  expired=strcmp(curdate,estdate) >= 0;

	  /* Determine the day when warning given */
	  convdate(&estdays, "%s", "%Y-%m-%d",estdate);
	  warnday=atof(estdays)-warndays*3600*24;
	  free(estdays);

	  g_snprintf(warnbuf, ENTRYDATEMAX, "%ld", warnday);
	  convdate(&estdays, "%Y-%m-%d", "%s", warnbuf);
	  warn=(strcmp(curdate,estdays) >=0) || (lastkm+warnkm > km+distance);
	  free(estdays);
	  
	  convdate(&userlastdate, datefmtstr[curunit.dateformat], "%Y-%m-%d",lastdate);
	  convdate(&userestdate, datefmtstr[curunit.dateformat], "%Y-%m-%d",estdate);
	  
	  gtk_list_store_set(GTK_LIST_STORE(store), &iter,
			     ALM_COL_LASTKM, SIlength2user(km),
			     ALM_COL_LASTDATE, userlastdate,
			     ALM_COL_NEXTKM, SIlength2user(km+distance),
			     ALM_COL_NEXTDATE, userestdate,
			     ALM_COL_COLOR, expired ? "red" : warn ? "yellow" : "green",
			     -1);
	  free(userestdate);
	  free(userlastdate);
	  anywarnings = anywarnings || expired || warn;
	}
	else {
	  gtk_list_store_set(GTK_LIST_STORE(store), &iter,
			     ALM_COL_LASTKM, 0.0,
			     ALM_COL_LASTDATE, "",
			     ALM_COL_NEXTKM, 0.0,
			     ALM_COL_NEXTDATE, "",
			     ALM_COL_COLOR, NULL,
			     -1);
	}
	free(estdate);
      } /* Distance limit given */
      else { /* Only time limit given */
	if (lastdate != NULL && strlen(lastdate)>0) {
	  db_calc_date(&estdate, lastdate, interval, "months");
	  expired=strcmp(curdate,estdate) >= 0;
	  estkm=db_estimate_km(lastdate, interval);

	  /* Determine the day when warning given */
	  convdate(&estdays, "%s", "%Y-%m-%d",estdate);
	  warnday=atof(estdays)-warndays*3600*24;
	  free(estdays);

	  g_snprintf(warnbuf, ENTRYDATEMAX, "%ld", warnday);
	  convdate(&estdays, "%Y-%m-%d", "%s", warnbuf);
	  warn=(strcmp(curdate,estdays) >=0);
	  free(estdays);

	  convdate(&userlastdate, datefmtstr[curunit.dateformat], "%Y-%m-%d",lastdate);
	  convdate(&userestdate, datefmtstr[curunit.dateformat], "%Y-%m-%d",estdate);

	  gtk_list_store_set(GTK_LIST_STORE(store), &iter,
			     ALM_COL_LASTKM, SIlength2user(km),
			     ALM_COL_LASTDATE, userlastdate,
			     ALM_COL_NEXTKM, SIlength2user(estkm),
			     ALM_COL_NEXTDATE, userestdate,
			     ALM_COL_COLOR, expired ? "red" : warn ? "yellow" : "green",
			     -1);
	  free(userestdate);
	  free(userlastdate);
	  free(estdate);
	  anywarnings = anywarnings || expired || warn;

	}
	else {
	  gtk_list_store_set(GTK_LIST_STORE(store), &iter,
			     ALM_COL_LASTKM, 0.0,
			     ALM_COL_LASTDATE, "",
			     ALM_COL_NEXTKM, 0.0,
			     ALM_COL_NEXTDATE, "",
			     ALM_COL_COLOR, NULL,
			     -1);
	}
      }
      free(lastdate);
    } while (gtk_tree_model_iter_next(GTK_TREE_MODEL(store), &iter));
    free(curdate);
  }

  return anywarnings;
}

static GtkTreeModel *
create_and_fill_alarmmodel (AppUIData *pui)
{
  GtkListStore  *store;
  GtkTreeIter    iter;
  GtkTreeSortable     *sortable;

  if (db_open()) {
    store = gtk_list_store_new (ALM_NUM_COLS, 
				G_TYPE_STRING,  /* ALM_COL_DESC */
				G_TYPE_DOUBLE,  /* ALM_COL_KMLIM */
				G_TYPE_DOUBLE,  /* ALM_COL_TIMELIM */
				G_TYPE_DOUBLE,  /* ALM_COL_NEXTKM */
				G_TYPE_STRING,  /* ALM_COL_NEXTDATE */
				G_TYPE_DOUBLE,  /* ALM_COL_LASTKM */
				G_TYPE_STRING,  /* ALM_COL_LASTDATE */
				G_TYPE_STRING,  /* ALM_COL_COLOR */
				G_TYPE_INT64    /* ALM_COL_ID */
				);

    if (SQLITE_OK == sqlite3_bind_int(ppStmtGetAlarmtype, 1, currentcar)) {
      while(SQLITE_ROW == sqlite3_step(ppStmtGetAlarmtype)) {

	gtk_list_store_append (store, &iter);
	gtk_list_store_set (store, &iter,
			    ALM_COL_DESC, sqlite3_column_text(ppStmtGetAlarmtype,0),
			    ALM_COL_KMLIM, SIlength2user(sqlite3_column_double(ppStmtGetAlarmtype,1)),
			    ALM_COL_TIMELIM, sqlite3_column_double(ppStmtGetAlarmtype,2),
			    ALM_COL_NEXTKM, 0.0,
			    ALM_COL_NEXTDATE, "",
			    ALM_COL_LASTKM, 0.0,
			    ALM_COL_LASTDATE, "",
			    ALM_COL_COLOR, NULL,
			    ALM_COL_ID, sqlite3_column_int64(ppStmtGetAlarmtype,3),
			    -1);    
      }
    }

    sqlite3_reset(ppStmtGetAlarmtype);

    /* Fill the last event columns */
    update_next_event(GTK_TREE_MODEL(store));

    sortable = GTK_TREE_SORTABLE(store);
    gtk_tree_sortable_set_sort_func(sortable, ALM_COL_DESC, sort_alarmiter_compare_func,
                                    GINT_TO_POINTER(ALM_COL_DESC), NULL);

    gtk_tree_sortable_set_sort_func(sortable, ALM_COL_KMLIM, sort_alarmiter_compare_func,
                                    GINT_TO_POINTER(ALM_COL_KMLIM), NULL);

    gtk_tree_sortable_set_sort_func(sortable, ALM_COL_TIMELIM, sort_alarmiter_compare_func,
                                    GINT_TO_POINTER(ALM_COL_TIMELIM), NULL);

    gtk_tree_sortable_set_sort_func(sortable, ALM_COL_LASTKM, sort_alarmiter_compare_func,
                                    GINT_TO_POINTER(ALM_COL_LASTKM), NULL);
    gtk_tree_sortable_set_sort_func(sortable, ALM_COL_LASTDATE, sort_alarmiter_compare_func,
                                    GINT_TO_POINTER(ALM_COL_LASTDATE), NULL);
    gtk_tree_sortable_set_sort_func(sortable, ALM_COL_NEXTDATE, sort_alarmiter_compare_func,
                                    GINT_TO_POINTER(ALM_COL_NEXTDATE), NULL);

    /* set initial sort order */
    gtk_tree_sortable_set_sort_column_id(sortable, ALM_COL_NEXTDATE, GTK_SORT_ASCENDING);

    return GTK_TREE_MODEL (store);
  }
  else
    return NULL;
}

#define MAXSTRLEN 30
static void
list_selection_changed_cb (GtkTreeSelection *selection, AppUIData *pui)
{
  GtkTreeIter iter;
  GtkTreeModel *model;
/*         gchar *colstring; */
  gint64 id;
  double value;
  GString *str;
  gchar *d;

  str=g_string_new(NULL);

  if (gtk_tree_selection_get_selected (selection, &model, &iter))
    {
      gtk_tree_model_get (model, &iter, ALM_COL_ID, &id, -1);
      
      gtk_tree_model_get (model, &iter, ALM_COL_DESC, &d,-1);
      gtk_label_set_text(GTK_LABEL(pui->stb_alarmdesclabel),d);
      g_free(d);

      gtk_tree_model_get (model, &iter, ALM_COL_KMLIM, &value,-1);
      g_string_printf(str,"%.0f ",value);
      if (curunit.lengthunit==SI)
	g_string_append_printf(str, _("km"));
      else
	g_string_append_printf(str, _("miles"));
      gtk_label_set_text(GTK_LABEL(pui->stb_alarmkmlabel),str->str);

      gtk_tree_model_get (model, &iter, ALM_COL_TIMELIM, &value,-1);
      g_string_printf(str,"%.0f ",value);
      g_string_append_printf(str,_("months"));
      gtk_label_set_text(GTK_LABEL(pui->stb_alarmtimelabel),str->str);
    }

  g_string_free(str, TRUE);

}

GtkWidget *create_alarmview_and_model (AppUIData *pui)
{
  GtkWidget *view;
  static data_func_t dfdata1;
  static data_func_t dfdata2;
  static data_func_t dfdata3;
  static data_func_t dfdata4;
  static data_func_t dfdata6;

  GtkTreeViewColumn   *col;
  GtkCellRenderer     *renderer;
  GtkTreeModel        *model;
  GtkTreeSelection    *select;

  GtkWidget           *label;
  PangoAttrList       *attrs;
  PangoAttribute      *attr;

  /* Scale texts smaller: create pango attributes */
  attrs = pango_attr_list_new();
  attr = pango_attr_scale_new(fontscalefactors[fontsize]);
  attr->start_index = 0;
  attr->end_index = -1;
  pango_attr_list_insert(attrs, attr);

  model = create_and_fill_alarmmodel (pui);

#if MAEMO_VERSION_MAJOR == 5
  view = hildon_gtk_tree_view_new_with_model (HILDON_UI_MODE_EDIT, 
					      model);
#else
  view = gtk_tree_view_new_with_model (model);
/*   view = gtk_tree_view_new_with_model (liststore); */
/*   view = gtk_tree_view_new (); */
#endif

  /* --- Column #1 --- */

  col = gtk_tree_view_column_new();

/*   gtk_tree_view_column_set_title(col, _("Description")); */
  label=gtk_label_new(gettext(alm_col_headers[ALM_COL_DESC]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", ALM_COL_DESC);
  gtk_tree_view_column_add_attribute(col, renderer, "background", ALM_COL_COLOR);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  dfdata1.col = ALM_COL_DESC;
  strncpy(dfdata1.format,"%s",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata1, NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), ALM_COL_DESC);

  /* --- Column #2 --- */

  col = gtk_tree_view_column_new();

/*   gtk_tree_view_column_set_title(col, _("Km limit")); */
  label=gtk_label_new(gettext(alm_col_headers[ALM_COL_KMLIM]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", ALM_COL_KMLIM);
  gtk_tree_view_column_add_attribute(col, renderer, "background", ALM_COL_COLOR);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  dfdata2.col = ALM_COL_KMLIM;
  strncpy(dfdata2.format,"%.0f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata2, NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), ALM_COL_KMLIM);
  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),ALM_COL_KMLIM), FALSE);

  /* --- Column #3 --- */

  col = gtk_tree_view_column_new();

/*   gtk_tree_view_column_set_title(col, _("Time limit")); */
  label=gtk_label_new(gettext(alm_col_headers[ALM_COL_TIMELIM]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", ALM_COL_TIMELIM);
  gtk_tree_view_column_add_attribute(col, renderer, "background", ALM_COL_COLOR);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  dfdata3.col = ALM_COL_TIMELIM;
  strncpy(dfdata3.format,"%.0f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata3, NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), ALM_COL_TIMELIM);
  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),ALM_COL_TIMELIM), FALSE);


  /* --- Column #4 --- */

  col = gtk_tree_view_column_new();

/*   gtk_tree_view_column_set_title(col, _("Next km")); */
  label=gtk_label_new(gettext(alm_col_headers[ALM_COL_NEXTKM]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", ALM_COL_NEXTKM);
  gtk_tree_view_column_add_attribute(col, renderer, "background", ALM_COL_COLOR);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), ALM_COL_NEXTKM);
  dfdata4.col = ALM_COL_NEXTKM;
  strncpy(dfdata4.format,"%.0f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata4, NULL);

  /* --- Column #5 --- */

  col = gtk_tree_view_column_new();

/*   gtk_tree_view_column_set_title(col, _("Next date")); */
  label=gtk_label_new(gettext(alm_col_headers[ALM_COL_NEXTDATE]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", ALM_COL_NEXTDATE);
  gtk_tree_view_column_add_attribute(col, renderer, "background", ALM_COL_COLOR);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), ALM_COL_NEXTDATE);

/*   dfdata7.col = COL_PRICEPERLITRE; */
/*   strncpy(dfdata7.format,"%.2f",DFFORMATLEN); */
/*   gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata7, NULL); */

  /* --- Column #6 --- */

  col = gtk_tree_view_column_new();

/*   gtk_tree_view_column_set_title(col, _("Last km")); */
  label=gtk_label_new(gettext(alm_col_headers[ALM_COL_LASTKM]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", ALM_COL_LASTKM);
  gtk_tree_view_column_add_attribute(col, renderer, "background", ALM_COL_COLOR);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), ALM_COL_LASTKM);

  dfdata6.col = ALM_COL_LASTKM;
  strncpy(dfdata6.format,"%.0f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata6, NULL);

  /* --- Column #6 --- */

  col = gtk_tree_view_column_new();

/*   gtk_tree_view_column_set_title(col, _("Last date")); */
  label=gtk_label_new(gettext(alm_col_headers[ALM_COL_LASTDATE]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", ALM_COL_LASTDATE);
  gtk_tree_view_column_add_attribute(col, renderer, "background", ALM_COL_COLOR);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), ALM_COL_LASTDATE);

/*   dfdata5.col = ALM_COL_LASTDATE; */
/*   strncpy(dfdata5.format,"%.1f",DFFORMATLEN); */
/*   gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata5, NULL); */

  /* Invisible column for color */
  /* --- Column #8 --- */

  col = gtk_tree_view_column_new();

  gtk_tree_view_column_set_title(col, "Color");

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", ALM_COL_COLOR);

/*   renderer = gtk_cell_renderer_text_new (); */
/*   gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view), */
/*                                                -1,       */
/*                                                "Id",   */
/*                                                renderer, */
/*                                                "text", COL_ID, */
/*                                                NULL); */
  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),ALM_COL_COLOR), FALSE);

  /* Invisible column to distinguish the results */
  /* --- Column #last --- */

  col = gtk_tree_view_column_new();

  gtk_tree_view_column_set_title(col, "Id");

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", ALM_COL_ID);

/*   renderer = gtk_cell_renderer_text_new (); */
/*   gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view), */
/*                                                -1,       */
/*                                                "Id",   */
/*                                                renderer, */
/*                                                "text", COL_ID, */
/*                                                NULL); */
  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),ALM_COL_ID), FALSE);

/*   model = create_and_fill_alarmmodel (); */

  /* Change the unit dependent headers */
  update_alarm_headers(view);

  gtk_tree_view_set_model (GTK_TREE_VIEW (view), model);
  gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(view), TRUE);

  /* The tree view has acquired its own reference to the
   *  model, so we can drop ours. That way the model will
   *  be freed automatically when the tree view is destroyed */

  g_object_unref (model);

  select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
  gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);
  g_signal_connect (G_OBJECT (select), "changed",
		    G_CALLBACK (list_selection_changed_cb),
		    pui);
  g_signal_connect(view, "row-activated",
		   G_CALLBACK (callback_reminderactivated),
		   pui);

  return view;
}

void newalarm_add(AppUIData *pui)
{
  GtkTreeSelection *selection;
  GtkListStore  *store;
  GtkTreeIter    iter;
  GString *str;
  GString *desc;
  GString *longdesc;
  gdouble km;
  gdouble interval;

/*   km = user2SIlength(atof(gtk_entry_get_text(GTK_ENTRY(pui->entryalarmdistance)))); */
  str = get_entry(pui->entryalarmdistance);
  km  = user2SIlength(atof(str->str));
  g_string_free(str, TRUE);

/*   interval = atof(gtk_entry_get_text(GTK_ENTRY(pui->entryalarminterval))); */
  str = get_entry(pui->entryalarminterval);
  interval  = atof(str->str);
  g_string_free(str, TRUE);

  desc = get_entry(pui->entryalarmdesc);
  longdesc = get_entry(pui->entryalarmlongdesc);

  /* Read dialog entries and store to database */ 
  if (SQLITE_OK != sqlite3_bind_int(ppStmtAddAlarmtype, 1, currentcar) /* carid */
      || SQLITE_OK != sqlite3_bind_text(ppStmtAddAlarmtype, 2,
/* 					gtk_entry_get_text(GTK_ENTRY(pui->entryalarmdesc)), */
					desc->str,
					-1, SQLITE_STATIC) /* shotdesc */
      || SQLITE_OK != sqlite3_bind_double(ppStmtAddAlarmtype, 3, 
					  km) /* distance */
      || SQLITE_OK != sqlite3_bind_double(ppStmtAddAlarmtype, 4,
					  interval)
      || SQLITE_OK != sqlite3_bind_text(ppStmtAddAlarmtype, 5,
/* 					gtk_entry_get_text(GTK_ENTRY(pui->entryalarmlongdesc)), */
					longdesc->str,
					-1, SQLITE_STATIC)
      || SQLITE_DONE != sqlite3_step(ppStmtAddAlarmtype)) {
    PDEBUG("Problem saving alarmtype");
  }
  else { /* alarmtype succesfully inserted */
    store = (GtkListStore *)gtk_tree_view_get_model(GTK_TREE_VIEW(pui->alarmview));
    gtk_list_store_append (store, &iter);
    gtk_list_store_set (store, &iter,
/* 			ALM_COL_DESC, gtk_entry_get_text(GTK_ENTRY(pui->entryalarmdesc)), */
			ALM_COL_DESC, desc->str,
			ALM_COL_KMLIM, SIlength2user(km),
			ALM_COL_TIMELIM, interval,
			ALM_COL_LASTKM, 0.0,
			ALM_COL_LASTDATE, "",
			ALM_COL_NEXTKM, 0.0,
			ALM_COL_NEXTDATE, "",
			ALM_COL_ID, db_get_last_insert_rowid(),
			-1);
    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->alarmview));
    gtk_tree_selection_select_iter(selection, &iter);

  }
  g_string_free(desc, TRUE);
  g_string_free(longdesc, TRUE);
  sqlite3_reset(ppStmtAddAlarmtype);
}

void newalarm_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  if (arg1==GTK_RESPONSE_ACCEPT) {
    if (db_open()) {
      newalarm_add(pui);
    }
  }
  gtk_widget_destroy(GTK_WIDGET(dialog));
}

void create_alarm_editwin(AppUIData *pui, GtkWidget *box, gboolean add)
{
  enum
  {
    ALARM_EDIT_CAR=0,
    ALARM_EDIT_SHORT,
    ALARM_EDIT_INTERVAL_KM,
    ALARM_EDIT_INTERVAL_MILES,
    ALARM_EDIT_INTERVAL_MONTHS,
    ALARM_EDIT_LONG
  };
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreeModel *model;
  GtkWidget *table;
  GtkWidget *label;
  GtkWidget *scrollwin;
  gint row;
  gint64 id;
  char buf[BUFMAX];
  gchar *labels[] = {gettext_noop("Car"),
		     gettext_noop("Short description"),
		     gettext_noop("Service interval (km)"),
		     gettext_noop("Service interval (miles)"),
		     gettext_noop("Service interval (months)"),
		     gettext_noop("Long description")
  };

#if MAEMO_VERSION_MAJOR == 5
  scrollwin = hildon_pannable_area_new();
  gtk_widget_set_size_request(scrollwin, -1, DIALOG_MIN_HEIGHTMAX);

  if (add)
    table = gtk_table_new(5, 2, FALSE);
  else
    table = gtk_table_new(4, 2, FALSE);

  row=0;

  if (add) {
    pui->alarmcarbutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
    hildon_button_set_title (HILDON_BUTTON (pui->alarmcarbutton), gettext(labels[ALARM_EDIT_CAR]));
    gtk_table_attach(GTK_TABLE(table), pui->alarmcarbutton, 0, 2, row, row+1, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    pui->alarmcarcombo = create_car_combo(pui, pui->alarmcarbutton);

    row++;
  }

  /* Second row */
  label=gtk_label_new(gettext(labels[ALARM_EDIT_SHORT]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entryalarmdesc = hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryalarmdesc), 30);
  gtk_table_attach(GTK_TABLE(table), pui->entryalarmdesc, 1, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryalarmdesc);
  row++;

  /* Third row */
  if (curunit.lengthunit==SI)
    label=gtk_label_new(gettext(labels[ALARM_EDIT_INTERVAL_KM]));
  else
    label=gtk_label_new(gettext(labels[ALARM_EDIT_INTERVAL_MILES]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entryalarmdistance = hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryalarmdistance), 30);
  g_object_set(G_OBJECT(pui->entryalarmdistance),
	       "hildon-input-mode",
	       HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
	       NULL);
  gtk_table_attach(GTK_TABLE(table), pui->entryalarmdistance, 1, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryalarmdistance);
  row++;

  /* Forth row */
  label=gtk_label_new(gettext(labels[ALARM_EDIT_INTERVAL_MONTHS]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entryalarminterval = hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryalarminterval), 30);
  g_object_set(G_OBJECT(pui->entryalarminterval),
              "hildon-input-mode",
		 HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
		 NULL);
  gtk_table_attach(GTK_TABLE(table), pui->entryalarminterval, 1, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryalarminterval);
  row++;

  /* Fifth row */
  label=gtk_label_new(gettext(labels[ALARM_EDIT_LONG]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entryalarmlongdesc = hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryalarmlongdesc), 60);
  gtk_table_attach(GTK_TABLE(table), pui->entryalarmlongdesc, 1, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryalarmlongdesc);

  gtk_widget_show(table);
  gtk_box_pack_start (GTK_BOX(box), scrollwin, TRUE, TRUE, 0);
  gtk_widget_show(scrollwin);
  hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA(scrollwin),table);

  if (!add) {
    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->alarmview));
    if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
    gtk_tree_model_get (model, &iter, ALM_COL_ID, &id, -1);

    if (SQLITE_OK == sqlite3_bind_int64(ppStmtGetOneAlarmtype, 1, id)) {
      while (SQLITE_ROW == sqlite3_step(ppStmtGetOneAlarmtype)) {
	gtk_entry_set_text(GTK_ENTRY(pui->entryalarmdesc), 
			   sqlite3_column_text(ppStmtGetOneAlarmtype,0));
	g_snprintf(buf,BUFMAX,"%.0f",SIlength2user(sqlite3_column_double(ppStmtGetOneAlarmtype,1)));
	gtk_entry_set_text(GTK_ENTRY(pui->entryalarmdistance), buf);
	gtk_entry_set_text(GTK_ENTRY(pui->entryalarminterval), 
			   sqlite3_column_text(ppStmtGetOneAlarmtype,2));
      }
    }
    sqlite3_reset(ppStmtGetOneAlarmtype);
    }
  }

#else
  scrollwin = gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_set_size_request(scrollwin, -1, DIALOG_MIN_HEIGHT1);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollwin), 
				 GTK_POLICY_NEVER, 
				 GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start(GTK_BOX(box),scrollwin, TRUE, TRUE,0);

  if (add)
    table = gtk_table_new(5, 2, FALSE);
  else
    table = gtk_table_new(4, 2, FALSE);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrollwin),table);

  row=0;

  /* First row */
  if (add) {
    label=gtk_label_new(gettext(labels[ALARM_EDIT_CAR]));
    gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(label);

    pui->alarmcarcombo = create_car_combo(pui);
    gtk_table_attach(GTK_TABLE(table), pui->alarmcarcombo, 1, 2, row, row+1, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(pui->alarmcarcombo);

    row++;
  }

  /* Second row */
  label=gtk_label_new(gettext(labels[ALARM_EDIT_SHORT]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entryalarmdesc = gtk_entry_new();
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryalarmdesc), 30);
  gtk_table_attach(GTK_TABLE(table), pui->entryalarmdesc, 1, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryalarmdesc);
  row++;

  /* Third row */
  if (curunit.lengthunit==SI)
    label=gtk_label_new(gettext(labels[ALARM_EDIT_INTERVAL_KM]));
  else
    label=gtk_label_new(gettext(labels[ALARM_EDIT_INTERVAL_MILES]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entryalarmdistance = gtk_entry_new();
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryalarmdistance), 30);
#if PLAINGTK == 0
  g_object_set(G_OBJECT(pui->entryalarmdistance),
#if HILDON == 1
              "hildon-input-mode",
		 HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
		 NULL);
#else
		 "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
  g_object_set(G_OBJECT(pui->entryalarmdistance), "autocap", FALSE, NULL);
#endif
#endif
  gtk_table_attach(GTK_TABLE(table), pui->entryalarmdistance, 1, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryalarmdistance);
  row++;

  /* Forth row */
  label=gtk_label_new(gettext(labels[ALARM_EDIT_INTERVAL_MONTHS]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entryalarminterval = gtk_entry_new();
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryalarminterval), 30);
#if PLAINGTK == 0
  g_object_set(G_OBJECT(pui->entryalarminterval),
#if HILDON == 1
              "hildon-input-mode",
		 HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
		 NULL);
#else
		 "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
  g_object_set(G_OBJECT(pui->entryalarminterval), "autocap", FALSE, NULL);
#endif
#endif
  gtk_table_attach(GTK_TABLE(table), pui->entryalarminterval, 1, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryalarminterval);
  row++;

  /* Fifth row */
  label=gtk_label_new(gettext(labels[ALARM_EDIT_LONG]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entryalarmlongdesc = gtk_entry_new();
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryalarmlongdesc), 60);
  gtk_table_attach(GTK_TABLE(table), pui->entryalarmlongdesc, 1, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryalarmlongdesc);

  gtk_widget_show(table);
  gtk_widget_show(scrollwin);

  if (!add) {
    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->alarmview));
    if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
    gtk_tree_model_get (model, &iter, ALM_COL_ID, &id, -1);

    if (SQLITE_OK == sqlite3_bind_int64(ppStmtGetOneAlarmtype, 1, id)) {
      while (SQLITE_ROW == sqlite3_step(ppStmtGetOneAlarmtype)) {
	gtk_entry_set_text(GTK_ENTRY(pui->entryalarmdesc), 
			   sqlite3_column_text(ppStmtGetOneAlarmtype,0));
	g_snprintf(buf,BUFMAX,"%.0f",SIlength2user(sqlite3_column_double(ppStmtGetOneAlarmtype,1)));
	gtk_entry_set_text(GTK_ENTRY(pui->entryalarmdistance), buf);
	gtk_entry_set_text(GTK_ENTRY(pui->entryalarminterval), 
			   sqlite3_column_text(ppStmtGetOneAlarmtype,2));
      }
    }
    sqlite3_reset(ppStmtGetOneAlarmtype);
    }
  }
#endif
}

GtkResponseType callback_newalarm( GtkAction * action, AppUIData *pui )
{
  GtkWidget *dialog;
  gint confirm;

  dialog = gtk_dialog_new_with_buttons(_("Add a reminder"),
				       GTK_WINDOW(pui->app->alarmwindow),
				       GTK_DIALOG_MODAL,
				       GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
				       GTK_STOCK_CANCEL,
				       GTK_RESPONSE_REJECT,
				       NULL);

  create_alarm_editwin(pui, GTK_WIDGET(GTK_DIALOG(dialog)->vbox), TRUE);

#if LIBHELP == 1
  help_dialog_help_enable(GTK_DIALOG(dialog),
				 HELP_ID_ADDREMINDER,
				 pui->app->osso);
#endif

/*   g_signal_connect (dialog, "response", */
/* 		    G_CALLBACK (newalarm_response), */
/* 		    pui); */

  gtk_widget_show (dialog);
  confirm = gtk_dialog_run(GTK_DIALOG(dialog));
  newalarm_response(GTK_DIALOG(dialog), confirm, pui);

  return confirm;
}

void editalarm_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreeModel *model;
  GtkListStore  *store;
  gint64 id;
  gdouble kmlim;
  gdouble interval;
  gboolean warn;

  switch (arg1) {
  case GTK_RESPONSE_ACCEPT:
    if (db_open()) {
      selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->alarmview));
      if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
	gtk_tree_model_get (model, &iter, ALM_COL_ID, &id, -1);
	kmlim=user2SIlength(atof(gtk_entry_get_text(GTK_ENTRY(pui->entryalarmdistance))));
	interval=atof(gtk_entry_get_text(GTK_ENTRY(pui->entryalarminterval)));
	if (SQLITE_OK != sqlite3_bind_text(ppStmtUpdateAlarmtype, 1,
					  gtk_entry_get_text(GTK_ENTRY(pui->entryalarmdesc)), -1, SQLITE_STATIC) /* shortdesc */
	    || SQLITE_OK != sqlite3_bind_double(ppStmtUpdateAlarmtype, 2, 
						kmlim) /* distance */
	    || SQLITE_OK != sqlite3_bind_double(ppStmtUpdateAlarmtype, 3,
						interval)
	    || SQLITE_OK != sqlite3_bind_int64(ppStmtUpdateAlarmtype, 4,
					     id)
	    || SQLITE_DONE != sqlite3_step(ppStmtUpdateAlarmtype)) {
	  PDEBUG("Problem saving alarmtype");
	}
	else { /* alarmtype succesfully updated */
	  /* Update the alarm view */
	  store = (GtkListStore *)gtk_tree_view_get_model(GTK_TREE_VIEW(pui->alarmview));
	  gtk_list_store_set (store, &iter,
			      ALM_COL_DESC, gtk_entry_get_text(GTK_ENTRY(pui->entryalarmdesc)),
			      ALM_COL_KMLIM, SIlength2user(kmlim),
			      ALM_COL_TIMELIM, interval,
			      -1);
	  g_signal_emit_by_name((gpointer)gtk_tree_view_get_selection(GTK_TREE_VIEW(pui->alarmview)), "changed");

	  /* Update the next event estimates */
	  pui->warn = update_next_event(gtk_tree_view_get_model(GTK_TREE_VIEW(pui->alarmview)));
	  update_reminder_toolbutton (pui, pui->warn);

	}
	sqlite3_reset(ppStmtUpdateAlarmtype);
      }
    }
    break;
  case RESPONSE_EDIT:
    callback_listevents(NULL, pui);
    break;
  }
  gtk_widget_destroy(GTK_WIDGET(dialog));
}


void callback_editalarm( GtkAction * action, AppUIData *pui )
{
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreeModel *model;
  GtkWidget *dialog;

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->alarmview));
  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {

    dialog = gtk_dialog_new_with_buttons(_("Edit a reminder"),
					 GTK_WINDOW(pui->app->alarmwindow),
					 GTK_DIALOG_MODAL,
					 GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
					 GTK_STOCK_CANCEL,
					 GTK_RESPONSE_REJECT,
					 _("Edit events"),
					 RESPONSE_EDIT,
					 NULL);

    create_alarm_editwin(pui, GTK_WIDGET(GTK_DIALOG(dialog)->vbox), FALSE);

#if LIBHELP == 1
    help_dialog_help_enable(GTK_DIALOG(dialog),
				   HELP_ID_EDITREMINDER,
				   pui->app->osso);
#endif

    g_signal_connect (dialog, "response",
		      G_CALLBACK (editalarm_response),
		      pui);

    /*   gtk_widget_show_all (GTK_WIDGET (dialog)); */
    gtk_widget_show (dialog);
  }
  else {
    PDEBUG("callback_editalarm: Problem finding selection\n");
  }

}

void callback_reminderactivated (GtkTreeView        *treeview,
                               GtkTreePath        *path,
                               GtkTreeViewColumn  *col,
                               AppUIData *pui)
{
  GtkTreeModel *model;
  GtkTreeIter   iter;

  model = gtk_tree_view_get_model(treeview);

  if (gtk_tree_model_get_iter(model, &iter, path)) {
    callback_listevents(NULL, pui);
  }
}

void callback_eventactivated (GtkTreeView        *treeview,
                               GtkTreePath        *path,
                               GtkTreeViewColumn  *col,
                               AppUIData *pui)
{
  GtkTreeModel *model;
  GtkTreeIter   iter;

  model = gtk_tree_view_get_model(treeview);

  if (gtk_tree_model_get_iter(model, &iter, path)) {
    callback_editevents(NULL, pui);
  }
}

void callback_deletealarm( GtkAction * action, AppUIData *pui )
{
  GtkWidget *confirmdlg;
  gint confirm;
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreeModel *model;
  GtkListStore *store;
  gint64 id;
  GString *message;
  int eventsexist;


  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->alarmview));
  if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
    gtk_tree_model_get (model, &iter, ALM_COL_ID, &id, -1);

    message=g_string_new(_("Are you sure you want to delete the reminder"));

    eventsexist=db_events_exist(id);
    if (eventsexist) {
      g_string_append(message, _(". All events will be deleted as well"));
    }

#if PLAINGTK == 0
    confirmdlg = hildon_note_new_confirmation(GTK_WINDOW(pui->app->alarmwindow), 
		 message->str);
#else
    confirmdlg = gtk_message_dialog_new(GTK_WINDOW(pui->app->alarmwindow),
					GTK_DIALOG_MODAL,
					GTK_MESSAGE_QUESTION,
					GTK_BUTTONS_OK_CANCEL,
					message->str);
    gtk_widget_show(confirmdlg);
#endif
    confirm = gtk_dialog_run(GTK_DIALOG(confirmdlg));
    gtk_widget_destroy(GTK_WIDGET(confirmdlg));
    
    if(confirm == GTK_RESPONSE_OK) {
      if (eventsexist) {
	db_delete_events(id);
      }
      db_delete_reminder(id);
      store = (GtkListStore *)gtk_tree_view_get_model(GTK_TREE_VIEW(pui->alarmview));
      gtk_list_store_remove(store, &iter);
    }

    g_string_free(message, TRUE);

  }

  else {
    PDEBUG("callback_deletealarm: Problem finding selection\n");
  }

}

#if PLAINGTK == 1
static void create_alarm_toolbar(GtkWidget *window, GtkWidget *vbox, AppUIData *pui )
#else
static void create_alarm_toolbar(HildonWindow *window, AppUIData *pui )
#endif
{
  GtkWidget *toolbar;
  GtkToolItem *tb_add;
  GtkToolItem *tb_edit;
  GtkToolItem *tb_close;

  /* Create toolbar */
  toolbar = gtk_toolbar_new();

  /* Create toolbar button items */
  tb_add = gtk_tool_button_new_from_stock(GTK_STOCK_ADD);
  tb_edit = gtk_tool_button_new_from_stock(GTK_STOCK_EDIT);
  tb_close = gtk_tool_button_new_from_stock(GTK_STOCK_CLOSE);

  gtk_tool_item_set_expand(GTK_TOOL_ITEM(tb_add),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(tb_edit),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(tb_close),TRUE);
   
  /* Add all items to toolbar */
  gtk_toolbar_insert(GTK_TOOLBAR(toolbar), tb_add, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(toolbar), tb_edit, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(toolbar), gtk_separator_tool_item_new(), -1);
  gtk_toolbar_insert(GTK_TOOLBAR(toolbar), tb_close, -1);

  g_signal_connect(G_OBJECT(tb_add), "clicked",
		   G_CALLBACK(callback_newevent), pui);
  g_signal_connect(G_OBJECT(tb_edit), "clicked",
		   G_CALLBACK(callback_editevents), pui);
  g_signal_connect_swapped(G_OBJECT(tb_close), "clicked",
			   G_CALLBACK(gtk_widget_destroy), G_OBJECT(window));

  /* Add toolbar HildonWindow */
#if PLAINGTK == 1
  gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(toolbar), FALSE, FALSE, 5);
#else
  hildon_window_add_toolbar(window, GTK_TOOLBAR(toolbar));
#endif
}

#if PLAINGTK == 1
static void create_alarmsecondary_toolbar(GtkWidget * main_window, GtkWidget *vbox, AppUIData *pui)
#else
  static void create_alarmsecondary_toolbar(HildonWindow * main_window, AppUIData *pui)
#endif
{
  GtkWidget *toolbar;
  GString *str;

  PangoAttrList *attrs;
  PangoAttribute *attr;

  /* Create toolbar */
  toolbar = gtk_toolbar_new();

  /* Scale texts smaller: create pango attributes */
  attrs = pango_attr_list_new();
  attr = pango_attr_scale_new(PANGO_SCALE_SMALL);
  attr->start_index = 0;
  attr->end_index = -1;
  pango_attr_list_insert(attrs, attr);

  /* Alarm description */
  pui->stb_alarmdesc = gtk_tool_item_new();
  pui->stb_alarmdesclabel = gtk_label_new(NULL);
  gtk_label_set_attributes(GTK_LABEL(pui->stb_alarmdesclabel), attrs);
  gtk_container_add(GTK_CONTAINER(pui->stb_alarmdesc), GTK_WIDGET(pui->stb_alarmdesclabel));

  str=g_string_new(NULL);

  /* Distance limit */
  g_string_printf(str, "%.0f ", 0.0);
  if (curunit.lengthunit==SI)
    g_string_append_printf(str, _("km"));
  else
    g_string_append_printf(str, _("miles"));
  pui->stb_alarmkm = gtk_tool_item_new();
  pui->stb_alarmkmlabel = gtk_label_new(str->str);
  gtk_label_set_attributes(GTK_LABEL(pui->stb_alarmkmlabel), attrs);
  gtk_container_add(GTK_CONTAINER(pui->stb_alarmkm), GTK_WIDGET(pui->stb_alarmkmlabel));

  /* Time limit */
  g_string_printf(str, "%.0f ", 0.0);
  g_string_append_printf(str,_("months"));
  pui->stb_alarmtime = gtk_tool_item_new();
  pui->stb_alarmtimelabel = gtk_label_new(str->str);
  gtk_label_set_attributes(GTK_LABEL(pui->stb_alarmtimelabel), attrs);
  gtk_container_add(GTK_CONTAINER(pui->stb_alarmtime), GTK_WIDGET(pui->stb_alarmtimelabel));

  g_string_free(str, TRUE);

  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->stb_alarmdesc),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->stb_alarmkm),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->stb_alarmtime),TRUE);

  /* Add all items to toolbar */
  gtk_toolbar_insert(GTK_TOOLBAR(toolbar), pui->stb_alarmdesc, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(toolbar), pui->stb_alarmkm, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(toolbar), pui->stb_alarmtime, -1);

  /* Add toolbar HildonWindow */
#if PLAINGTK == 1
  gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(toolbar), FALSE, FALSE, 5);
#else
  hildon_window_add_toolbar(main_window, GTK_TOOLBAR(toolbar));
#endif
}

#if PLAINGTK == 1
void create_alarmwin_menu(GtkWidget *window, GtkWidget *root, AppUIData *pui)
#else
  void create_alarmwin_menu(HildonWindow *window, AppUIData *pui)
#endif
{
  GtkWidget *main_menu;
  GtkWidget *menu_alarm;
  GtkWidget *menu_event;
  GtkWidget *menu_view;
  GtkWidget *item_alarm;
  GtkWidget *item_event;
  GtkWidget *item_view;
  GtkWidget *item_newalarm;
  GtkWidget *item_editalarm;
  GtkWidget *item_deletealarm;
  GtkWidget *item_newevent;
  GtkWidget *item_listevents;
  GtkWidget *item_help;
  GtkWidget *item_exit;
#if PLAINGTK == 1
  GtkWidget *menubar;
  GtkWidget *item_file;
#endif
 
#if MAEMO_VERSION_MAJOR == 5
  GtkWidget * button;
  
  main_menu = HILDON_APP_MENU (hildon_app_menu_new ());

  item_newevent = hildon_button_new_with_text(HILDON_SIZE_AUTO,
					      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
					      _("Add event"),
					      NULL);
  hildon_app_menu_append (main_menu, GTK_BUTTON (item_newevent));

  item_listevents = hildon_button_new_with_text(HILDON_SIZE_AUTO,
					      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
					      _("Edit events"),
					      NULL);
  hildon_app_menu_append (main_menu, GTK_BUTTON (item_listevents));

  item_newalarm = hildon_button_new_with_text(HILDON_SIZE_AUTO,
					      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
					      _("Add reminder"),
					      NULL);
  hildon_app_menu_append (main_menu, GTK_BUTTON (item_newalarm));

  item_editalarm = hildon_button_new_with_text(HILDON_SIZE_AUTO,
					      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
					      _("Edit reminder"),
					      NULL);
  hildon_app_menu_append (main_menu, GTK_BUTTON (item_editalarm));

  item_deletealarm = hildon_button_new_with_text(HILDON_SIZE_AUTO,
					      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
					      _("Delete reminder"),
					      NULL);
  hildon_app_menu_append (main_menu, GTK_BUTTON (item_deletealarm));

  item_exit = hildon_button_new_with_text(HILDON_SIZE_AUTO,
					  HILDON_BUTTON_ARRANGEMENT_VERTICAL,
					  _("Close"),
					  NULL);
  hildon_app_menu_append (main_menu, GTK_BUTTON (item_exit));

  g_signal_connect(G_OBJECT(item_newevent), "clicked",
		   G_CALLBACK(callback_newevent), pui);
  g_signal_connect(G_OBJECT(item_listevents), "clicked",
		   G_CALLBACK(callback_listevents), pui);
  g_signal_connect(G_OBJECT(item_newalarm), "clicked",
		   G_CALLBACK(callback_newalarm), pui);
  g_signal_connect(G_OBJECT(item_editalarm), "clicked",
		   G_CALLBACK(callback_editalarm), pui);
  g_signal_connect(G_OBJECT(item_deletealarm), "clicked",
		   G_CALLBACK(callback_deletealarm), pui);

  g_signal_connect_swapped(G_OBJECT(item_exit), "clicked",
			   G_CALLBACK(delete_alarmwin), G_OBJECT(window));


  hildon_window_set_app_menu(HILDON_WINDOW(window), main_menu);

#else
  main_menu = gtk_menu_new();
  menu_alarm = gtk_menu_new();
  menu_event = gtk_menu_new();
  menu_view = gtk_menu_new();

  /* Create main menu items */
  item_alarm = gtk_menu_item_new_with_label(_("Reminder"));
  item_event = gtk_menu_item_new_with_label(_("Event"));
  item_view = gtk_menu_item_new_with_label(_("View"));
#if LIBHELP == 1
  item_help = gtk_menu_item_new_with_label(_("Help"));
#endif
  item_exit = gtk_menu_item_new_with_label(_("Close window"));

  /* Create reminder menu items */
  item_newalarm = gtk_menu_item_new_with_label(_("New"));
  item_editalarm = gtk_menu_item_new_with_label(_("Edit"));
  item_deletealarm = gtk_menu_item_new_with_label(_("Delete"));

  /* Create event menu items */
  item_newevent = gtk_menu_item_new_with_label(_("New"));
  item_listevents = gtk_menu_item_new_with_label(_("List"));

  /* Create view menu items */
  pui->alarm_menu_item_fullscreen = gtk_check_menu_item_new_with_label(_("Full screen"));
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pui->alarm_menu_item_fullscreen),
				 pui->alarmfullscreen);

  /* Add menu items to right menus */
  /* Main menu */
  gtk_menu_append(main_menu, item_alarm);
  gtk_menu_append(main_menu, item_event);
  gtk_menu_append(main_menu, gtk_separator_menu_item_new());
  gtk_menu_append(main_menu, item_view);
  gtk_menu_append(main_menu, gtk_separator_menu_item_new());
#if LIBHELP == 1
  gtk_menu_append(main_menu, item_help);
#endif
  gtk_menu_append(main_menu, item_exit);

  /* Reminder menu */
  gtk_menu_append(menu_alarm, item_newalarm);
  gtk_menu_append(menu_alarm, item_editalarm);
  gtk_menu_append(menu_alarm, item_deletealarm);

  /* Event menu */
  gtk_menu_append(menu_event, item_newevent);
  gtk_menu_append(menu_event, item_listevents);

  /* View menu */
  gtk_menu_append(menu_view, pui->alarm_menu_item_fullscreen);

#if PLAINGTK == 1
  menubar = gtk_menu_bar_new();
  gtk_box_pack_start (GTK_BOX (root), menubar, FALSE, FALSE, 2);
  gtk_widget_show (menubar);
  item_file = gtk_menu_item_new_with_label ("fuelpad");
  gtk_widget_show (item_file);
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (item_file), main_menu);
  gtk_menu_bar_append (GTK_MENU_BAR (menubar), item_file);
#else
  hildon_window_set_menu(HILDON_WINDOW(window), GTK_MENU(main_menu));
#endif

  gtk_menu_item_set_submenu(GTK_MENU_ITEM(item_alarm), menu_alarm);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(item_event), menu_event);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(item_view), menu_view);

  /* Attach the callback functions to the activate signal */
  g_signal_connect(G_OBJECT(item_newalarm), "activate",
		   G_CALLBACK(callback_newalarm), pui);
  g_signal_connect(G_OBJECT(item_editalarm), "activate",
		   G_CALLBACK(callback_editalarm), pui);
  g_signal_connect(G_OBJECT(item_deletealarm), "activate",
		   G_CALLBACK(callback_deletealarm), pui);
  g_signal_connect(G_OBJECT(item_newevent), "activate",
		   G_CALLBACK(callback_newevent), pui);
  g_signal_connect(G_OBJECT(item_listevents), "activate",
		   G_CALLBACK(callback_listevents), pui);
#if LIBHELP == 1
  g_signal_connect(G_OBJECT(item_help), "activate",
		   G_CALLBACK(help_activated), HELP_ID_ALARM);
#endif
  g_signal_connect_swapped(G_OBJECT(item_exit), "activate",
			   G_CALLBACK(delete_alarmwin), G_OBJECT(window));

  g_signal_connect(G_OBJECT(pui->alarm_menu_item_fullscreen), "toggled",
		   G_CALLBACK(cb_alarm_fullscreen), pui);

#endif

  /* Make all menu widgets visible */
  gtk_widget_show_all(GTK_WIDGET(main_menu));
}


void callback_alarm( GtkAction * action, AppUIData *pui )
{
  GtkWidget *vbox;

  if (pui->app->alarmwindow != NULL) {
    gtk_window_present(GTK_WINDOW(pui->app->alarmwindow));
    return;
  }
#if PLAINGTK == 1
  pui->app->alarmwindow=gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(pui->app->alarmwindow), "Reminders");
  gtk_widget_set_usize(pui->app->alarmwindow,800,400);
  gtk_container_border_width(GTK_CONTAINER(pui->app->alarmwindow),0);
#else
#if MAEMO_VERSION_MAJOR == 5
  pui->app->alarmwindow = hildon_stackable_window_new();
#else
  pui->app->alarmwindow = HILDON_WINDOW(hildon_window_new());
#endif
  hildon_program_add_window(pui->app->program, pui->app->alarmwindow);

  /* Callback for hardware keys */
  g_signal_connect(G_OBJECT(pui->app->alarmwindow), "key_press_event", 
		   G_CALLBACK(alarm_key_press_cb), pui);
#endif

  g_signal_connect(G_OBJECT(pui->app->alarmwindow), "destroy",
		   G_CALLBACK(destroy_alarmwin), pui);

  vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (pui->app->alarmwindow), vbox);

#if PLAINGTK == 1
  create_alarmwin_menu(pui->app->alarmwindow, vbox, pui);
#else
  create_alarmwin_menu(pui->app->alarmwindow, pui);
#endif

#if MAEMO_VERSION_MAJOR == 5
#else
#if PLAINGTK == 1
   create_alarm_toolbar(pui->app->alarmwindow, vbox, pui);
#else
   create_alarm_toolbar(pui->app->alarmwindow, pui);
#endif
#endif

#if MAEMO_VERSION_MAJOR == 5
  pui->alarmscrollwin = hildon_pannable_area_new();
  g_object_set(G_OBJECT(pui->alarmscrollwin), "mov-mode", HILDON_MOVEMENT_MODE_BOTH, NULL);
#else
  pui->alarmscrollwin = gtk_scrolled_window_new(NULL, NULL);
#endif
  gtk_box_pack_start (GTK_BOX(vbox), pui->alarmscrollwin, TRUE, TRUE, 0);

  pui->alarmview = create_alarmview_and_model (pui);
  gtk_container_add (GTK_CONTAINER (pui->alarmscrollwin), pui->alarmview);

#if PLAINGTK == 1
   create_alarmsecondary_toolbar(pui->app->alarmwindow, vbox, pui);
#else
   create_alarmsecondary_toolbar(pui->app->alarmwindow, pui);
#endif

  gtk_widget_show(vbox);

  gtk_widget_show_all(GTK_WIDGET(pui->app->alarmwindow));
  if (pui->mainfullscreen) 
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pui->alarm_menu_item_fullscreen),
				   !pui->alarmfullscreen);

}
