/* 
 * This file is part of fuelpad, the fuel diary
 *
 * Copyright (c) 2007, 2008 Julius Luukko <julle.luukko@quicknet.inet.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

/**
 *
 * \file
 * \brief  GTK+/Hildon based user interface to fuelpad
 * \author Julius Luukko
 *
 *
 */

#include "ui.h"
#include "uireport.h"
#include "uistat.h"
#include "uiexpimp.h"
#include "uialarm.h"
#include "uiicons.h"
#include <assert.h>
#include <math.h>


#define GCONF_KEY_PREFIX "/apps/fuelpad"
#define GCONF_KEY_CURRENT_DRIVER GCONF_KEY_PREFIX"/current_driver"
#define GCONF_KEY_CURRENT_CAR GCONF_KEY_PREFIX"/current_car"
#define GCONF_KEY_CURRENT_UNIT GCONF_KEY_PREFIX"/current_unit"
#define GCONF_KEY_INDIVIDUAL_UNIT GCONF_KEY_PREFIX"/individual_unit"
#define GCONF_KEY_CURRENT_LENGTHUNIT GCONF_KEY_PREFIX"/current_lengthunit"
#define GCONF_KEY_CURRENT_VOLUMEUNIT GCONF_KEY_PREFIX"/current_volumeunit"
#define GCONF_KEY_CURRENT_CONSUMEUNIT GCONF_KEY_PREFIX"/current_consumeunit"
#define GCONF_KEY_CURRENT_MASSUNIT GCONF_KEY_PREFIX"/current_massunit"
#define GCONF_KEY_DATE_FORMAT GCONF_KEY_PREFIX"/date_format"
#define GCONF_KEY_CURRENCY GCONF_KEY_PREFIX"/currency"
#define GCONF_KEY_DATABASE GCONF_KEY_PREFIX"/database"
#define GCONF_KEY_WARNING_KM GCONF_KEY_PREFIX"/warningkm"
#define GCONF_KEY_WARNING_DAYS GCONF_KEY_PREFIX"/warningdays"
#define GCONF_KEY_REPORT_PAPER GCONF_KEY_PREFIX"/reportpaper"
#define GCONF_KEY_SELECT_COLUMNS GCONF_KEY_PREFIX"/mainviewcolumns"
#define GCONF_KEY_MAINVIEW_FONTSIZE GCONF_KEY_PREFIX"/mainviewfontsize"
#define GCONF_KEY_MAINTOOLBAR_NOTVISIBLE GCONF_KEY_PREFIX"/maintoolbar_notvisible"
#define GCONF_KEY_SECTOOLBAR_NOTVISIBLE GCONF_KEY_PREFIX"/secondarytoolbar_notvisible"
#define GCONF_MAINTABLE_SORTED GCONF_KEY_PREFIX"/maintablesorted"
#define GCONF_MAINTABLE_SORTCOLUMN GCONF_KEY_PREFIX"/maintablesortcolumn"
#define GCONF_MAINTABLE_SORTORDER GCONF_KEY_PREFIX"/maintablesortorder"

/* Which columns are displayed by default (a bitfield) */
#define DISPCOLDEFAULT (1<<COL_NOTES | 1<<COL_NOTES | 0 << COL_TIRES | 0 << COL_OIL | 0 << COL_SERVICE | 0 << COL_PRICEPERLITRE | 1 << COL_PRICEPERTRIP | 1 << COL_PRICE | 1 << COL_CONSUM | 1 << COL_FILL | 1 << COL_TRIP | 1 << COL_KM | 1 << COL_DAY)

#define RESPONSE_SELECT_COLUMNS 1

typedef struct {
  double km;
  double trip;
  double fill;
  double price;
  double service;
  double oil;
  double tires;
} entry_t;

typedef struct filterentry {
  GtkWidget *removebutton;
#if MAEMO_VERSION_MAJOR == 5
  GtkWidget *andorbutton;
#else
  GtkWidget *label;
#endif
  GtkWidget *andorcombo;
  GtkWidget *minentry;
  GtkWidget *maxentry;
} filterentry_t;

typedef struct recordfilter {
  filterentry_t entry;
  GString *mindate;
  GString *maxdate;
  GString *notes;
  GPatternSpec *pattern;
  gdouble minval;
  gdouble maxval;
  gboolean and;
  int column;
} recordfilter_t;

recordfilter_t recfilter;

GList *filterqueue=NULL;

AppData app;
AppUIData ui;

unit_t curunit = {
  .mainunit = SI,
  .lengthunit=SI,
  .volumeunit=SI,
  .consumeunit=SI,
  .massunit=SI,
  .individualunit = FALSE
};

/**********************************************************************
 *
 * Variables stored in gconf
 *
 */
double warnkm=1000.0;
double warndays=7.0;

/* Paper used in report and statistics */
enum papertype reportpaper=A4PAPER;

/* Which columns are shown */
gint dispcol=DISPCOLDEFAULT;

/* Main table font size */
enum fontscale fontsize=MEDIUM;

/* Toolbar visibility */
gboolean main_toolbar_visible;
gboolean secondary_toolbar_visible;

/* Main table sort order */
gboolean maintablesorted=TRUE;
gint maintablesortcol=COL_DAY;
GtkSortType maintablesortorder=GTK_SORT_ASCENDING;

/**********************************************************************/

/* Font scale factors corresponding to enum fontscale */
double fontscalefactors[]={PANGO_SCALE_MEDIUM,
			   PANGO_SCALE_X_SMALL,
			   PANGO_SCALE_SMALL,
			   PANGO_SCALE_MEDIUM,
			   PANGO_SCALE_LARGE};

/* Possible date format strings (see strptime(3) for format descri ption) */
char *datefmtstr[]={"%Y-%m-%d", "%d.%m.%Y", "%d/%m/%Y", "%d/%m/%y", 
		    "%d-%m-%Y", "%m/%d/%Y", "%m/%d/%y"};

/* Possible values of warning times in days */
float wtimes[]={1.0, 2.0, 7.0, 14.0, 30.0};

/* Warn times */
gchar *warntexts[] = {gettext_noop("1 day"),
		      gettext_noop("2 days"),
		      gettext_noop("1 week"),
		      gettext_noop("2 weeks"),
		      gettext_noop("1 month")};

gchar *column_headers[] = {gettext_noop("Date"),
			   gettext_noop("Km"),
			   gettext_noop("Trip\n[km]"),
			   gettext_noop("Fill\n[litre]"),
			   gettext_noop("[litre/\n100 km]"),
			   gettext_noop("Price"),
			   gettext_noop("Price/\nkm"),
			   gettext_noop("Price/\nlitre"),
			   gettext_noop("Service"),
			   gettext_noop("Oil"),
			   gettext_noop("Tires"),
			   gettext_noop("CO2 Emissions\n[g/km]"),
			   gettext_noop("Notes"),
};

gchar *column_headers_non_SI[] = {gettext_noop("Date"),
				  gettext_noop("Miles"),
				  gettext_noop("Trip\n[miles]"),
				  gettext_noop("Fill\n[gal.]"),
				  gettext_noop("[Miles/\ngallon]"),
				  gettext_noop("Price"),
				  gettext_noop("Price/\nmile"),
				  gettext_noop("Price/\ngal."),
				  gettext_noop("Service"),
				  gettext_noop("Oil"),
				  gettext_noop("Tires"),
				  gettext_noop("CO2 Emissions\n[lb/100 miles]"),
				  gettext_noop("Notes"),
};

gboolean carchanged=FALSE;
enum dbtimespan stbstattime=OVERALL;

/* Prototypes of some functions */
void newrecord_additionalfields(AppUIData *pui, GtkDialog *parent);
void callback_about( GtkAction * action, gpointer data );
static GtkTreeModel *create_and_fill_model (void);
void internal_close_db(AppUIData *pui);
void callback_close_db( GtkAction * action, AppUIData *pui );
void callback_open_db( GtkAction * action, AppUIData *pui );
void update_car_combo(gpointer widget);
void update_driver_combo(gpointer combo);
GtkWidget *create_fueltype_combo(AppUIData *pui);
void update_warn_label(GtkWidget *label);
void update_warn_entry(GtkWidget *entry, double value);
double get_entrywarnkm(AppUIData *pui);
void enable_mainmenu_items(AppUIData *pui);
void create_mainwin_widgets(AppUIData *pui);
void callback_recordactivated (GtkTreeView *treeview, GtkTreePath *path, GtkTreeViewColumn  *col, AppUIData *pui);
void callback_select_columns( GtkAction * action, AppUIData *pui );
void filter_update(recordfilter_t *filt);
void filter_init(recordfilter_t *filt);
void filter_clear(recordfilter_t *filt);
static gboolean visible_func (GtkTreeModel *model, GtkTreeIter *iter, gpointer data);
guint filter_filterqueue_length(void);
void fill_filter_minmax_entrys(AppUIData *pui, gint active, GtkWidget *minentry, GtkWidget *maxentry);
void draw_filter_selector(AppUIData *pui, GtkWidget *table, gint i, guint nele, recordfilter_t *p);

/*************************************************************
 * Unit conversion functions 
 ************************************************************/

/* lcf = length conversion factor */
/* vcf = volume conversion factor */
/* mcf = mass conversion factor */
double lcf[]={1.0, 1.609344, 1.609344};
double vcf[]={1.0, 3.785411784, 4.54609};
double mcf[]={1.0, 453.59237, 453.59237};

/**
 * \fn double SIlength2user(double length)
 * \brief Converts length from SI units to user units
 * \param length Length in SI units (meters)
 * \return Length in user units
 */
double SIlength2user(double length)
{
  return length/lcf[curunit.lengthunit];
}

/**
 * \fn double user2SIlength(double length)
 * \brief Converts length from user units to SI units
 * \param length Length in user units
 * \return Length in SI units
 */
double user2SIlength(double length)
{
  return length*lcf[curunit.lengthunit];
}

/**
 * \fn double SIvolume2user(double length)
 * \brief Converts volume from SI units to user units
 * \param length Volume in "SI" units (litres)
 * \return Volume in user units
 */
double SIvolume2user(double length)
{
  return length/vcf[curunit.volumeunit];
}

/**
 * \fn double user2SIvolume(double length)
 * \brief Converts volume from user units to SI units
 * \param length Volume in user units
 * \return Volume in SI units
 */
double user2SIvolume(double length)
{
  return length*vcf[curunit.volumeunit];
}

/**
 * \fn double SIconsumption2user(double consum)
 * \brief Converts consumption from SI units to user units
 * \param consum Consumption in litres/100 km
 * \return Consumption in user units
 *
 * If the global consumeunit is other than SI, the consumption
 * is converted from litres/100 km to miles per gallon (MPG).
 *
 */
double SIconsumption2user(double consum)
{
  if (curunit.consumeunit==SI)
    return consum;
  else
    return vcf[curunit.consumeunit]/lcf[curunit.consumeunit]*100.0/consum;
}

/**
 * \fn double user2SIconsumption(double consum)
 * \brief Converts consumption from  user units to SI units
 * \param consum Consumption in user units
 * \return Consumption in litres/100 km
 */
double user2SIconsumption(double consum)
{
  return SIconsumption2user(consum);
}

/**
 * \fn double SIppl2user(double ppl)
 * \brief Converts price per volume from SI units to user units
 * \param ppl Price/litre
 * \return Price/user unit
 */
double SIppl2user(double ppl)
{
  return user2SIvolume(ppl);
}

/**
 * \fn double SImass2user(double mass)
 * \brief Converts mass from SI units to user units
 * \param mass Mass in SI units (g)
 * \return Mass in user units
 */
double SImass2user(double mass)
{
  return mass/mcf[curunit.massunit];
}

/**
 * \fn double SIemission2user(double emission)
 * \brief Converts emission from SI units to user units
 * \param emission Emission in SI units (g/km)
 * \return Mass in user units
 */
double SIemission2user(double emission)
{
  double result;

  if (curunit.massunit==SI)
    result=SImass2user(emission)*lcf[curunit.lengthunit];
  else
    result=SImass2user(emission)*lcf[curunit.lengthunit]*100;
  return result;
}

/**
 * \fn double grams2kg(double grams)
 * \brief Converts grams to kilograms if massunit is SI, otherwise do nothing
 * \param grams Mass in grams
 * \return Kilograms if massunit is SI
 */
double grams2kg(double grams)
{
  double result;

  if (curunit.massunit==SI)
    result=grams/1000;
  else
    result=grams;
  return result;
}

/**
 * \fn void config_save(void)
 * \brief Save all configuration data to gconf.
 *
 * Saves the current car, current driver and current units to gconf.
 *
 */
static
void config_save(void)
{
  GConfClient *gconf_client = gconf_client_get_default();

  if(!gconf_client)
    {
      g_print(_("Failed to initialize GConf.  Settings were not saved."));
      return;
    }
  gconf_client_set_int(gconf_client,GCONF_KEY_CURRENT_CAR,
		       currentcar, NULL);
  gconf_client_set_int(gconf_client,GCONF_KEY_CURRENT_DRIVER, 
		       currentdriver, NULL);
  gconf_client_set_int(gconf_client,GCONF_KEY_CURRENT_UNIT,
		       curunit.mainunit, NULL);
  gconf_client_set_int(gconf_client,GCONF_KEY_CURRENT_LENGTHUNIT,
		       curunit.lengthunit, NULL);
  gconf_client_set_int(gconf_client,GCONF_KEY_CURRENT_VOLUMEUNIT,
		       curunit.volumeunit, NULL);
  gconf_client_set_int(gconf_client,GCONF_KEY_CURRENT_CONSUMEUNIT,
		       curunit.consumeunit, NULL);
  gconf_client_set_int(gconf_client,GCONF_KEY_CURRENT_MASSUNIT,
		       curunit.massunit, NULL);
  gconf_client_set_bool(gconf_client,GCONF_KEY_INDIVIDUAL_UNIT,
			curunit.individualunit, NULL);
  gconf_client_set_int(gconf_client,GCONF_KEY_DATE_FORMAT,
		       curunit.dateformat, NULL);
  gconf_client_set_string(gconf_client,GCONF_KEY_CURRENCY,
			   curunit.currency->str, NULL);
  gconf_client_set_string(gconf_client,GCONF_KEY_DATABASE,
			  db_getfilename(), NULL);
  gconf_client_set_float(gconf_client,GCONF_KEY_WARNING_KM,
		       warnkm, NULL);
  gconf_client_set_float(gconf_client,GCONF_KEY_WARNING_DAYS,
		       warndays, NULL);
  gconf_client_set_int(gconf_client,GCONF_KEY_REPORT_PAPER,
		       reportpaper, NULL);
  gconf_client_set_int(gconf_client,GCONF_KEY_SELECT_COLUMNS,
		       dispcol, NULL);
  gconf_client_set_int(gconf_client,GCONF_KEY_MAINVIEW_FONTSIZE,
		       (gint)fontsize, NULL);
  gconf_client_set_bool(gconf_client,GCONF_KEY_MAINTOOLBAR_NOTVISIBLE,
			!main_toolbar_visible, NULL);
  gconf_client_set_bool(gconf_client,GCONF_KEY_SECTOOLBAR_NOTVISIBLE,
			!secondary_toolbar_visible, NULL);
  gconf_client_set_bool(gconf_client,GCONF_MAINTABLE_SORTED,
			maintablesorted, NULL);
  gconf_client_set_int(gconf_client,GCONF_MAINTABLE_SORTCOLUMN,
		       (gint)maintablesortcol, NULL);
  gconf_client_set_int(gconf_client,GCONF_MAINTABLE_SORTORDER,
		       (gint)maintablesortorder, NULL);

  gconf_client_clear_cache(gconf_client);
  g_object_unref(gconf_client);
}

/**
 * \fn void config_init(void)
 * \brief Initialize all configuration from gconf.
 */
static
void config_init(void)
{
  GConfClient *gconf_client = gconf_client_get_default();
  struct lconv *locdata;
  gchar *tmpcurrency;
  gchar *tmpdatabase;

  if(!gconf_client)
    {
      g_print(_("Failed to initialize GConf.  Settings not read."));
      return;
    }
  currentdriver = gconf_client_get_int(gconf_client, GCONF_KEY_CURRENT_DRIVER,
				       NULL);
  currentcar = gconf_client_get_int(gconf_client, GCONF_KEY_CURRENT_CAR,
				    NULL);
  curunit.mainunit = gconf_client_get_int(gconf_client, GCONF_KEY_CURRENT_UNIT,
				     NULL);
  curunit.lengthunit = gconf_client_get_int(gconf_client,
					   GCONF_KEY_CURRENT_LENGTHUNIT, NULL);
  curunit.volumeunit = gconf_client_get_int(gconf_client,
					   GCONF_KEY_CURRENT_VOLUMEUNIT, NULL);
  curunit.consumeunit = gconf_client_get_int(gconf_client,
					    GCONF_KEY_CURRENT_CONSUMEUNIT, NULL);
  curunit.massunit = gconf_client_get_int(gconf_client,
					    GCONF_KEY_CURRENT_MASSUNIT, NULL);
  curunit.individualunit = gconf_client_get_bool(gconf_client,
					 GCONF_KEY_INDIVIDUAL_UNIT, NULL);

  curunit.dateformat = gconf_client_get_int(gconf_client,
					    GCONF_KEY_DATE_FORMAT, NULL);
  tmpcurrency = gconf_client_get_string(gconf_client,
				     GCONF_KEY_CURRENCY, NULL);
  tmpdatabase = gconf_client_get_string(gconf_client,
					GCONF_KEY_DATABASE, NULL);
  warnkm = gconf_client_get_float(gconf_client,
				  GCONF_KEY_WARNING_KM, NULL);
  warndays = gconf_client_get_float(gconf_client,
				    GCONF_KEY_WARNING_DAYS, NULL);
  reportpaper=gconf_client_get_int(gconf_client,
				   GCONF_KEY_REPORT_PAPER, NULL);
  dispcol=gconf_client_get_int(gconf_client,
				   GCONF_KEY_SELECT_COLUMNS, NULL);
  fontsize=gconf_client_get_int(gconf_client,
				   GCONF_KEY_MAINVIEW_FONTSIZE, NULL);
  main_toolbar_visible=!gconf_client_get_bool(gconf_client,
					      GCONF_KEY_MAINTOOLBAR_NOTVISIBLE,
					      NULL);
  secondary_toolbar_visible=!gconf_client_get_bool(gconf_client,
						   GCONF_KEY_SECTOOLBAR_NOTVISIBLE,
						   NULL);
  maintablesorted = gconf_client_get_bool(gconf_client,
					  GCONF_MAINTABLE_SORTED,
					  NULL);
  maintablesortcol = gconf_client_get_int(gconf_client,
					  GCONF_MAINTABLE_SORTCOLUMN,
					  NULL);
  maintablesortorder = (GtkSortType)gconf_client_get_int(gconf_client,
							 GCONF_MAINTABLE_SORTORDER,
							 NULL);


#if MAEMO_VERSION_MAJOR == 5
  main_toolbar_visible=FALSE;
  secondary_toolbar_visible=TRUE;
#endif

  gconf_client_clear_cache(gconf_client);
  g_object_unref(gconf_client);  

  if (tmpcurrency == NULL) {
    locdata=localeconv();
    curunit.currency=g_string_new(locdata->int_curr_symbol);
  }
  else
    curunit.currency=g_string_new(tmpcurrency);

  db_setfilename(tmpdatabase);

  /* Current car and driver are zero if this is the first time 
   * this program is run
   */
  if (currentcar==0) currentcar=1;
  if (currentdriver==0) currentdriver=1;
  if (curunit.mainunit>IMPERIAL || curunit.mainunit < SI) 
    curunit.mainunit=SI;
  if (curunit.lengthunit>IMPERIAL || curunit.lengthunit < SI) 
    curunit.lengthunit=SI;
  if (curunit.volumeunit>IMPERIAL || curunit.volumeunit < SI) 
    curunit.volumeunit=SI;
  if (curunit.consumeunit>IMPERIAL || curunit.consumeunit < SI) 
    curunit.consumeunit=SI;
  if (curunit.massunit>IMPERIAL || curunit.massunit < SI) 
    curunit.massunit=SI;
  if (curunit.dateformat<0 || curunit.dateformat>=sizeof(datefmtstr)/sizeof(char *))
    curunit.dateformat=0;
  if (warnkm==0.0)
    warnkm=1000;
  if (warndays==0.0)
    warndays=7.0;

  /* Limit reportpaper between 0 and 1 */ 
  if (reportpaper<0) reportpaper=0;
  else if (reportpaper>1) reportpaper=1;

  /* Reset shown columns to default value */
  if (dispcol == 0)
    dispcol=DISPCOLDEFAULT;

  /* Reset fontsize to MEDIUM*/
  if (fontsize<XSMALL || fontsize>LARGE)
    fontsize=MEDIUM;
}


/**
 * \fn void config_init(void)
 * \param argc argc from main()
 * \param argv argv from main()
 * \brief The first function to be called before any other ui functions
 *
 * This function initializes localization, gtk and reads gconf data.
 *
 */
void ui_init(int   argc,
	     char *argv[])
{
  /* Initialize localization. */
  setlocale(LC_ALL, "");
  bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
  bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
  textdomain(GETTEXT_PACKAGE);

#if MAEMO_VERSION_MAJOR == 5
  hildon_gtk_init (&argc, &argv);
#else
  gtk_init (&argc, &argv);
#endif
  config_init();
  filter_init(&recfilter);
}

/**
 * \fn void ui_main_loop(void)
 * \brief User interface main loop
 *
 * This functions calls gtk_main() and when gtk_main() exists, saves
 * the gconf data.
 *
 */
void ui_main_loop(void)
{
  gtk_main ();
  config_save();
  filter_clear(&recfilter);
}

#if PLAINGTK == 0
gint dbus_callback (const gchar *interface, const gchar *method,
		    GArray *arguments, gpointer data,
		    osso_rpc_t *retval)
{
  PDEBUG ("fuelpad dbus: %s, %s\n", interface, method);

  if (!strcmp (method, "top_application"))
      gtk_window_present (GTK_WINDOW (data));

  retval->type = DBUS_TYPE_INVALID;
  return OSSO_OK;
}

/**
 * \fn void osso_hw_state_callback(osso_hw_state_t *state, gpointer data)
 * \brief Osso hardware state callback
 *
 * This functions is a callback for hardware state changes. It
 * implements only the shutdown_ind state, which calls fuelpad_exit()
 * and exits with exit(1).
 *
 */
static void osso_hw_state_callback(osso_hw_state_t *state, gpointer data)
{
  if(state->shutdown_ind) {
    fuelpad_exit();
    exit(1);
  }
}

#if MAEMO_VERSION_MAJOR == 5
static DBusHandlerResult
mce_filter_func (DBusConnection * connection,
                 DBusMessage * message, AppData *app)
{
  DBusError error;
  dbus_error_init (&error);

  if (dbus_message_is_signal (message, MCE_SIGNAL_IF,
                              MCE_DEVICE_ORIENTATION_SIG))
  {
    char *rotation, *stand, *face;
    int x, y, z;

    /* Retrieve arguments from a message */
    if (!dbus_message_get_args (message,
                                &error,
                                DBUS_TYPE_STRING, &rotation,
                                DBUS_TYPE_STRING, &stand,
                                DBUS_TYPE_STRING, &face,
                                DBUS_TYPE_INT32, &x,
                                DBUS_TYPE_INT32, &y,
                                DBUS_TYPE_INT32, &z, DBUS_TYPE_INVALID))
    {
      g_printerr ("DBus error: %s (%s)\n", error.name, error.message);
      dbus_error_free (&error);
    }
    else
    {
      g_printerr ("Device orientation changed\n");
      g_printerr ("Rotation: %s\n", rotation);
      g_printerr ("Face: %s\n", face);
      g_printerr ("Stand: %s\n", stand);
      g_printerr ("X: %d\n", x);
      g_printerr ("Y: %d\n", y);
      g_printerr ("Z: %d\n", z);

  /* Rotate main window */
  if (!strcmp (rotation, MCE_ORIENTATION_PORTRAIT))
    hildon_gtk_window_set_portrait_flags (GTK_WINDOW(app->mainwindow), HILDON_PORTRAIT_MODE_REQUEST);
  else hildon_gtk_window_set_portrait_flags (GTK_WINDOW(app->mainwindow),  ~HILDON_PORTRAIT_MODE_REQUEST);
    }
  }

  return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;

}
#endif

#endif

/**
 * \fn gboolean delete_event( GtkWidget *widget, GdkEvent  *event, gpointer   data )
 * \brief Main window delete_event callback
 * \return FALSE
 */
static gboolean delete_event( GtkWidget *widget,
                              GdkEvent  *event,
                              gpointer   data )
{
  gtk_main_quit ();
#if PLAINGTK == 0

#if MAEMO_VERSION_MAJOR == 5
  if (osso_rpc_run_system(app.osso, MCE_SERVICE, MCE_REQUEST_PATH,
          MCE_REQUEST_IF, MCE_ACCELEROMETER_DISABLE_REQ, NULL, DBUS_TYPE_INVALID) == OSSO_OK) {
	g_printerr("INFO: Accelerometer disabled\n");
  } else {
        g_printerr("WARN: Cannot disable accelerometer\n");
  }

  /* Remove match rule and filter, finalize connection */
  if (app.system_bus) {
    if (dbus_connection_get_is_connected (app.system_bus))
      dbus_bus_remove_match (app.system_bus, MCE_SIGNAL_MATCH, NULL);
    dbus_connection_remove_filter (app.system_bus,
                                   (DBusHandleMessageFunction) mce_filter_func,
                                   &app);

    dbus_connection_unref(app.system_bus);
    app.system_bus = NULL;
  }
#endif

  osso_deinitialize(app.osso);
#endif
  return FALSE;
}

/**
 * \fn void main_window_fullscreen(gpointer window)
 * \brief Puts the window to full screen mode and scales if necessary
 *
 * Calls gtk_window_fullscreen() or gtk_window_unfullscreen(). If
 * statistics window is present, scale the canvas.
 *
 */
void main_window_fullscreen(gpointer data)
{
  AppUIData *a;

  a=(AppUIData *)data;
  a->mainfullscreen = !a->mainfullscreen;
  if (a->mainfullscreen) {
    gtk_window_fullscreen(GTK_WINDOW(a->app->mainwindow));
  } else {
    gtk_window_unfullscreen(GTK_WINDOW(a->app->mainwindow));
  }
}

void cb_main_fullscreen( GtkAction * action, gpointer data )
{
  main_window_fullscreen(data);
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(((AppUIData *)(data))->main_menu_item_fullscreen),
				 ((AppUIData *)(data))->mainfullscreen);
}

/* Callback for hardware keys */
/* See maemo tutorial */
#if PLAINGTK == 0
gboolean key_press_cb(GtkWidget * widget, GdkEventKey * event,
                      AppUIData *pui)
{
  switch (event->keyval) {

    case GDK_F6:
      gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pui->main_menu_item_fullscreen),
				     !pui->mainfullscreen);
      return TRUE;
  }

  return FALSE;
}
#endif

void main_toolbar_show_hide(AppUIData *pui)
{
  if (main_toolbar_visible)
    gtk_widget_show(GTK_WIDGET(pui->main_toolbar));
  else
    gtk_widget_hide(GTK_WIDGET(pui->main_toolbar));
}

void secondary_toolbar_show_hide(AppUIData *pui)
{
  if (secondary_toolbar_visible)
    gtk_widget_show(GTK_WIDGET(pui->secondary_toolbar));
  else
    gtk_widget_hide(GTK_WIDGET(pui->secondary_toolbar));
}

void cb_main_toolbar( GtkAction * action, AppUIData *pui)
{
  main_toolbar_visible = !main_toolbar_visible;
  main_toolbar_show_hide(pui);
}

void cb_secondary_toolbar( GtkAction * action, AppUIData *pui)
{
  secondary_toolbar_visible = !secondary_toolbar_visible;
  secondary_toolbar_show_hide(pui);
}

/* void update_font_scale(AppUIData *pui) */
void update_font_scale(GtkTreeView *view)
{
  gint i;
  GtkTreeViewColumn   *col;
  GtkWidget           *label;
  PangoAttrList       *attrs;
  PangoAttribute      *attr;
  GList               *rendererlist;

  for (i=COL_DAY; i<COL_ID; i++) {
    col=gtk_tree_view_get_column(view, i);
    label = gtk_tree_view_column_get_widget(col); 
    attrs=gtk_label_get_attributes(GTK_LABEL(label));
    attr = pango_attr_scale_new(fontscalefactors[fontsize]);
    attr->start_index = 0;
    attr->end_index = -1;
    pango_attr_list_change(attrs, attr);

    rendererlist=gtk_tree_view_column_get_cell_renderers(col);
    g_object_set(rendererlist->data, "scale", fontscalefactors[fontsize], NULL);
    g_list_free(rendererlist);
  }
}

void cb_fontsize_x_small( GtkAction * action, AppUIData *pui)
{
  fontsize=XSMALL;
  update_font_scale(GTK_TREE_VIEW(pui->view));
  update_record_model(pui);
}

void cb_fontsize_small( GtkAction * action, AppUIData *pui)
{
  fontsize=SMALL;
  update_font_scale(GTK_TREE_VIEW(pui->view));
}

void cb_fontsize_medium( GtkAction * action, AppUIData *pui)
{
  fontsize=MEDIUM;
  update_font_scale(GTK_TREE_VIEW(pui->view));
}

void cb_fontsize_large( GtkAction * action, AppUIData *pui)
{
  fontsize=LARGE;
  update_font_scale(GTK_TREE_VIEW(pui->view));
}

void ui_show_animation(GtkWidget **banner, const char *s)
{
#if PLAINGTK == 0
  *banner = hildon_banner_show_animation(GTK_WIDGET(app.mainwindow), NULL, s);
/*   PDEBUG(s); */
#else
  fputs(s,stderr);
#endif
}

void ui_show_banner(const char *s)
{
#if PLAINGTK == 0
  hildon_banner_show_information(GTK_WIDGET(app.mainwindow), NULL, s);
  PDEBUG(s);
#else
  fputs(s,stderr);
#endif
}

void ui_remove_banner(GtkWidget *banner)
{
#if PLAINGTK == 0
  gtk_widget_destroy(GTK_WIDGET(banner));
#endif
}

/**
 * \fn void edit_driver_response (GtkDialog *dialog, gint arg1, gpointer user_data)
 * \brief Response callback for edit car dialog
 *
 * This callback changes the entered driver data to the database.
 *
 * \todo This should be changed so that fuelpad.c has a function
 * edit_driver which is called with the data (i.e. sqlite would not be
 * used here)
 *
 */
void edit_car_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  GString *mark;
  GString *model;
  GString *year;
  GString *regist;
  GString *notes;
  gint fueltype;

  mark=g_string_new(NULL);
  model=g_string_new(NULL);
  year=g_string_new(NULL);
  regist=g_string_new(NULL);
  notes=g_string_new(NULL);
  if (arg1==GTK_RESPONSE_ACCEPT) {
    if (db_open()) {
      g_string_assign(mark,gtk_entry_get_text(GTK_ENTRY(pui->entrymark)));
      g_string_assign(model,gtk_entry_get_text(GTK_ENTRY(pui->entrymodel)));
      g_string_assign(year,gtk_entry_get_text(GTK_ENTRY(pui->entryyear)));
      g_string_assign(regist,gtk_entry_get_text(GTK_ENTRY(pui->entryregister)));
      g_string_assign(notes,gtk_entry_get_text(GTK_ENTRY(pui->entrycarnotes)));
#if MAEMO_VERSION_MAJOR == 5
      fueltype=hildon_touch_selector_get_active(HILDON_TOUCH_SELECTOR(pui->combofueltype), 0);
#else
      fueltype=gtk_combo_box_get_active(GTK_COMBO_BOX(pui->combofueltype));
#endif
      if (SQLITE_OK != sqlite3_bind_text(ppStmtUpdateCar, 1,
					 mark->str,
					 -1, SQLITE_STATIC)
	  || SQLITE_OK != sqlite3_bind_text(ppStmtUpdateCar, 2,
					    model->str,
					    -1, SQLITE_STATIC)
	  || SQLITE_OK != sqlite3_bind_int(ppStmtUpdateCar, 3,
					   atoi(year->str))
	  || SQLITE_OK != sqlite3_bind_text(ppStmtUpdateCar, 4,
					    regist->str,
					    -1, SQLITE_STATIC)
	  || SQLITE_OK != sqlite3_bind_text(ppStmtUpdateCar, 5,
					    notes->str,
					    -1, SQLITE_STATIC)
	  || SQLITE_OK != sqlite3_bind_int(ppStmtUpdateCar, 6,
					   fueltype)
	  || SQLITE_OK != sqlite3_bind_int(ppStmtUpdateCar, 7,
					   currentcar)
	  || SQLITE_DONE != sqlite3_step(ppStmtUpdateCar)) {
	PDEBUG("Problem udpating car properties");
      }
      else {
	PDEBUG("Car data updated\n");
      }
      sqlite3_reset(ppStmtUpdateCar);
#if MAEMO_VERSION_MAJOR == 5
      update_car_combo(pui->carbutton);
      update_car_combo(pui->stb_carbutton);
#else
      update_car_combo(pui->carcombo);
      update_car_combo(pui->stb_carcombo);
#endif
      g_string_free(mark, TRUE);
      g_string_free(model, TRUE);
      g_string_free(year, TRUE);
      g_string_free(regist, TRUE);
      g_string_free(notes, TRUE);
    }
    else {
      PDEBUG("Can't update driver properties - no database open");
    }
  }
  gtk_widget_destroy(GTK_WIDGET(dialog));
}

/**
 * \fn void edit_driver_response (GtkDialog *dialog, gint arg1, gpointer user_data)
 * \brief Response callback editing driver data
 *
 * This callback updates the driver data to the database.
 *
 * \todo This should be changed so that fuelpad.c has a function
 * edit_driver which is called with the data (i.e. sqlite would not be
 * used here)
 *
 */
void edit_driver_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  GString *name;
  GString *nick;

  name=g_string_new(NULL);
  nick=g_string_new(NULL);
  if (arg1==GTK_RESPONSE_ACCEPT) {
    if (db_open()) {
      g_string_assign(name,gtk_entry_get_text(GTK_ENTRY(pui->entryname)));
      g_string_assign(nick,gtk_entry_get_text(GTK_ENTRY(pui->entrynick)));
      if (SQLITE_OK != sqlite3_bind_text(ppStmtUpdateDriver, 1, 
					 name->str, 
					 -1, SQLITE_STATIC)
	  || SQLITE_OK != sqlite3_bind_text(ppStmtUpdateDriver, 2, 
					    nick->str, 
					    -1, SQLITE_STATIC)
	  || SQLITE_OK != sqlite3_bind_int(ppStmtUpdateDriver, 3,
					   currentdriver)
	  || SQLITE_DONE != sqlite3_step(ppStmtUpdateDriver)) {
	PDEBUG("Problem updatig the driver data");
      }
      else { /* New driver added */
/* 	gtk_combo_box_append_text(GTK_COMBO_BOX (pui->drivercombo),gtk_entry_get_text(GTK_ENTRY(pui->entrynick))); */
	PDEBUG("Driver data updated\n");
      }
      sqlite3_reset(ppStmtUpdateDriver);
      update_driver_combo(pui->drivercombo);
      g_string_free(nick, TRUE);
      g_string_free(name, TRUE);
    }
    else {
      PDEBUG("Can't update driver data - no database open");
    }
  }
  gtk_widget_destroy(GTK_WIDGET(dialog));
}

/**
 * \fn void add_driver_response (GtkDialog *dialog, gint arg1, gpointer user_data)
 * \brief Response callback for add car dialog
 *
 * This callback stores the entered new driver data to the database.
 *
 * \todo This should be changed so that fuelpad.c has a function
 * add_driver which is called with the data (i.e. sqlite would not be
 * used here)
 *
 */
void add_driver_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  GString *name;
  GString *nick;

  name=g_string_new(NULL);
  nick=g_string_new(NULL);
  if (arg1==GTK_RESPONSE_ACCEPT) {
    if (db_open()) {
      g_string_assign(name,gtk_entry_get_text(GTK_ENTRY(pui->entryname)));
      g_string_assign(nick,gtk_entry_get_text(GTK_ENTRY(pui->entrynick)));
      if (SQLITE_OK != sqlite3_bind_text(ppStmtAddDriver, 1, 
					 name->str, 
					 -1, SQLITE_STATIC)
	  || SQLITE_OK != sqlite3_bind_text(ppStmtAddDriver, 2, 
					    nick->str, 
					    -1, SQLITE_STATIC)
	  || SQLITE_DONE != sqlite3_step(ppStmtAddDriver)) {
	PDEBUG("Problem saving the new driver");
      }
      else { /* New driver added */
#if MAEMO_VERSION_MAJOR == 5
	hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (pui->drivercombo), nick->str);
#else
	gtk_combo_box_append_text(GTK_COMBO_BOX (pui->drivercombo), nick->str);
#endif
      }
      sqlite3_reset(ppStmtAddDriver);
      g_string_free(nick, TRUE);
      g_string_free(name, TRUE);
    }
    else {
      PDEBUG("Can't save new driver - no database open");
    }
  }
  gtk_widget_destroy(GTK_WIDGET(dialog));
}

/**
 * \fn void create_driver_editwin(AppUIData *pui, GtkWidget *dialog)
 * \brief Creates a driver editing window
 *
 * This function is used for creating an edit window both for adding a
 * new driver and editing an existing one.
 *
 */
void create_driver_editwin(AppUIData *pui, GtkWidget *dialog)
{
  GtkWidget *table;
  GtkWidget *label;

  table = gtk_table_new(2, 2, FALSE);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), table, FALSE, FALSE, 5);

  label=gtk_label_new(_("Full name:"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

#if MAEMO_VERSION_MAJOR == 5
  pui->entryname=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
#else
  pui->entryname = gtk_entry_new();
#endif
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryname), 30);
  gtk_table_attach(GTK_TABLE(table), pui->entryname, 1, 2, 0, 1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryname);

  label=gtk_label_new(_("Short name:"));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

#if MAEMO_VERSION_MAJOR == 5
  pui->entrynick=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
#else
  pui->entrynick = gtk_entry_new();
#endif
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrynick), NICKMAXLEN);
  gtk_table_attach(GTK_TABLE(table), pui->entrynick, 1, 2, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrynick);

  gtk_widget_show(table);

}

/**
 * \fn void add_driver_cb (GtkWidget *widget, gpointer   callback_data )
 * \brief Callback for adding a new driver to the database
 *
 * This callback opens a dialog which allows to add a new driver to
 * the database.
 *
 */
void edit_driver_cb (GtkWidget *widget, AppUIData *pui )
{
  GtkWidget *dialog;

  dialog = gtk_dialog_new_with_buttons(_("Edit driver properties"),
				       GTK_WINDOW(pui->app->mainwindow),
				       GTK_DIALOG_MODAL,
				       GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
				       GTK_STOCK_CANCEL,
				       GTK_RESPONSE_REJECT,
				       NULL);

  create_driver_editwin(pui, dialog);

  if (SQLITE_OK == sqlite3_bind_int(ppStmtOneDriver, 1, currentdriver)) {
    while (SQLITE_ROW == sqlite3_step(ppStmtOneDriver)) {
      gtk_entry_set_text(GTK_ENTRY(pui->entryname),sqlite3_column_text(ppStmtOneDriver,0));
      gtk_entry_set_text(GTK_ENTRY(pui->entrynick),sqlite3_column_text(ppStmtOneDriver,1));
    }
    sqlite3_reset(ppStmtOneDriver);
  }

#if LIBHELP == 1
  help_dialog_help_enable(GTK_DIALOG(dialog),
				 HELP_ID_ADDDRIVER,
				 pui->app->osso);
#endif

  g_signal_connect (dialog, "response",
		    G_CALLBACK (edit_driver_response),
		    &ui);

  gtk_widget_show(dialog);

}

/**
 * \fn void add_driver_cb (GtkWidget *widget, gpointer   callback_data )
 * \brief Callback for adding a new driver to the database
 *
 * This callback opens a dialog which allows to add a new driver to
 * the database.
 *
 */
void add_driver_cb (GtkWidget *widget, AppUIData *pui )
{
  GtkWidget *dialog;

  dialog = gtk_dialog_new_with_buttons(_("Add a new driver"),
				       GTK_WINDOW(pui->app->mainwindow),
				       GTK_DIALOG_MODAL,
				       GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
				       GTK_STOCK_CANCEL,
				       GTK_RESPONSE_REJECT,
				       NULL);

  create_driver_editwin(pui, dialog);

#if LIBHELP == 1
  help_dialog_help_enable(GTK_DIALOG(dialog),
				 HELP_ID_ADDDRIVER,
				 pui->app->osso);
#endif

  g_signal_connect (dialog, "response",
		    G_CALLBACK (add_driver_response),
		    &ui);

  gtk_widget_show(dialog);

}

/**
 * \fn void delete_driver_cb( GtkAction * action, AppUIData *pui )
 * \brief Callback for deleting a driver from the database
 *
 * This callback deletes a driver from the database. All the records
 * and events, which have this driver as a driver are also deleted. The
 * user is asked for a confirmation before deleting.
 *
 */
void delete_driver_cb( GtkAction * action, AppUIData *pui )
{
  GtkWidget *confirmdlg;
  int numrec;
  int numevents;
  GString *message;
  gint confirm;
  char *nickname;

  if (db_open()) { /* delete */
    if (db_get_numof_drivers() <= 1) {
      message=g_string_new(_("The driver can't be deleted. There is only one driver left and the database must contain at least one driver."));
#if PLAINGTK == 0
      confirmdlg = hildon_note_new_information(GTK_WINDOW(pui->app->mainwindow), 
						message->str);
#else
      confirmdlg = gtk_message_dialog_new(GTK_WINDOW(pui->app->mainwindow),
					  GTK_DIALOG_MODAL,
					  GTK_MESSAGE_INFO,
					  GTK_BUTTONS_OK,
					  message->str);
      gtk_widget_show(confirmdlg);
#endif
      confirm = gtk_dialog_run(GTK_DIALOG(confirmdlg));
      gtk_widget_destroy(GTK_WIDGET(confirmdlg));
      g_string_free(message, TRUE);
    }
    else {
      numrec=db_get_numof_records(currentdriver);
      numevents=db_get_numof_events(currentdriver);
      db_get_current_driver(&nickname);
      message=g_string_new(_("Are you sure you want to delete driver"));
      g_string_append(message, " ");
      g_string_append(message, nickname);
      g_string_append(message, ". ");
      if (nickname != NULL)
	free(nickname);
      if (numrec+numevents==0)
	g_string_append(message,_("There are no records or events for this driver."));
      else {
	if (numrec>0 && numevents == 0)
	  g_string_append_printf(message,_("There are %d records for this driver, which will also be deleted."), numrec);
	else if (numrec>0 && numevents > 0)
	  g_string_append_printf(message,_("There are %d records and %d events for this driver, which will also be deleted."), numrec, numevents);
	else if (numrec == 0 && numevents > 0)
	  g_string_append_printf(message,_("There are %d events for this driver, which will also be deleted."), numevents);
      }
      
#if PLAINGTK == 0
      confirmdlg = hildon_note_new_confirmation(GTK_WINDOW(pui->app->mainwindow), 
						message->str);
#else
      confirmdlg = gtk_message_dialog_new(GTK_WINDOW(pui->app->mainwindow),
					  GTK_DIALOG_MODAL,
					  GTK_MESSAGE_QUESTION,
					  GTK_BUTTONS_OK_CANCEL,
					  message->str);
      gtk_widget_show(confirmdlg);
#endif
      confirm = gtk_dialog_run(GTK_DIALOG(confirmdlg));
      gtk_widget_destroy(GTK_WIDGET(confirmdlg));
      
      if(confirm == GTK_RESPONSE_OK) {
	if (!db_delete_driver(currentdriver)) {
	  PDEBUG("Problem deleting the driver");
	}
	else { /* remove the record if succesfully removed from db */
	  update_driver_combo(pui->drivercombo);
	  update_record_model(pui);
	  update_totalkm(pui);
	}
      } /* if (confirm == GTK_RESPONSE_OK) */
      
      g_string_free(message, TRUE);
    }
  }
  else { /* nothing to delete */
    PDEBUG("Database not open, can't delete anything\n");
  }
}

/**
 * \fn void delete_car_cb( GtkAction * action, AppUIData *pui )
 * \brief Callback for deleting a car from the database
 *
 * This callback deletes a car from the database. All the records and
 * events for this car are also deleted. The user is asked for a
 * confirmation before deleting.
 *
 */
void delete_car_cb( GtkAction * action, AppUIData *pui )
{
  GtkWidget *confirmdlg;
  int numrec;
  int numevents;
  GString *message;
  gint confirm;
  char *car;

  if (db_open()) { /* delete */
    if (db_get_numof_cars() <= 1) {
      message=g_string_new(_("The car can't be deleted. There is only one car left and the database must contain at least one car."));
#if PLAINGTK == 0
      confirmdlg = hildon_note_new_information(GTK_WINDOW(pui->app->mainwindow), 
						message->str);
#else
      confirmdlg = gtk_message_dialog_new(GTK_WINDOW(pui->app->mainwindow),
					  GTK_DIALOG_MODAL,
					  GTK_MESSAGE_INFO,
					  GTK_BUTTONS_OK,
					  message->str);
      gtk_widget_show(confirmdlg);
#endif
      confirm = gtk_dialog_run(GTK_DIALOG(confirmdlg));
      gtk_widget_destroy(GTK_WIDGET(confirmdlg));
      g_string_free(message, TRUE);
    }
    else {
      numrec=db_get_numof_carrecords(currentcar);
      numevents=db_get_numof_carevents(currentcar);
      db_get_current_car(&car);
      message=g_string_new(_("Are you sure you want to delete car"));
      g_string_append(message, " ");
      g_string_append(message, car);
      if (car != NULL)
	free(car);
      g_string_append(message, ". ");
      if (numrec+numevents==0)
	g_string_append(message,_("There are no records or events for this car."));
      else {
	if (numrec>0 && numevents == 0)
	  g_string_append_printf(message,_("There are %d records for this car, which will also be deleted."), numrec);
	else if (numrec>0 && numevents > 0)
	  g_string_append_printf(message,_("There are %d records and %d events for this car, which will also be deleted."), numrec, numevents);
	else if (numrec == 0 && numevents > 0)
	  g_string_append_printf(message,_("There are %d events for this car, which will also be deleted."), numevents);
      }
      
#if PLAINGTK == 0
      confirmdlg = hildon_note_new_confirmation(GTK_WINDOW(pui->app->mainwindow), 
						message->str);
#else
      confirmdlg = gtk_message_dialog_new(GTK_WINDOW(pui->app->mainwindow),
					  GTK_DIALOG_MODAL,
					  GTK_MESSAGE_QUESTION,
					  GTK_BUTTONS_OK_CANCEL,
					  message->str);
      gtk_widget_show(confirmdlg);
#endif
      confirm = gtk_dialog_run(GTK_DIALOG(confirmdlg));
      gtk_widget_destroy(GTK_WIDGET(confirmdlg));
      
      if(confirm == GTK_RESPONSE_OK) {
	if (!db_delete_car(currentcar)) {
	  PDEBUG("Problem deleting the car");
	}
	else { /* remove the record if succesfully removed from db */
#if MAEMO_VERSION_MAJOR == 5
      update_car_combo(pui->carbutton);
      update_car_combo(pui->stb_carbutton);
#else
	  update_car_combo(pui->carcombo);
	  update_car_combo(pui->stb_carcombo);
#endif
	  update_record_model(pui);
	  update_totalkm(pui);
	}
      } /* if (confirm == GTK_RESPONSE_OK) */
      
      g_string_free(message, TRUE);
    }
  }
  else { /* nothing to delete */
    PDEBUG("Database not open, can't delete anything\n");
  }
}

/**
 * \fn void add_car_response (GtkDialog *dialog, gint arg1, gpointer user_data)
 * \brief Response callback for add car dialog
 *
 * This callback stores the entered new car data to the database.
 *
 * \todo This should be changed so that fuelpad.c has a function add_car which 
 * is called with the data (i.e. sqlite would not be used here)
 *
 */
void add_car_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  GString *listitemtext;
  GString *mark;
  GString *model;
  GString *year;
  GString *regist;
  GString *notes;
  gint fueltype;

  mark=g_string_new(NULL);
  model=g_string_new(NULL);
  year=g_string_new(NULL);
  regist=g_string_new(NULL);
  notes=g_string_new(NULL);
  if (arg1==GTK_RESPONSE_ACCEPT) {
    if (db_open()) {
      g_string_assign(mark,gtk_entry_get_text(GTK_ENTRY(pui->entrymark)));
      g_string_assign(model,gtk_entry_get_text(GTK_ENTRY(pui->entrymodel)));
      g_string_assign(year,gtk_entry_get_text(GTK_ENTRY(pui->entryyear)));
      g_string_assign(regist,gtk_entry_get_text(GTK_ENTRY(pui->entryregister)));
      g_string_assign(notes,gtk_entry_get_text(GTK_ENTRY(pui->entrycarnotes)));
#if MAEMO_VERSION_MAJOR == 5
      fueltype=hildon_touch_selector_get_active(HILDON_TOUCH_SELECTOR(pui->combofueltype), 0);
#else
      fueltype=gtk_combo_box_get_active(GTK_COMBO_BOX(pui->combofueltype));
#endif
      if (SQLITE_OK != sqlite3_bind_text(ppStmtAddCar, 1, 
					 mark->str,
					 -1, SQLITE_STATIC)
	  || SQLITE_OK != sqlite3_bind_text(ppStmtAddCar, 2,
					    model->str,
					    -1, SQLITE_STATIC)
	  || SQLITE_OK != sqlite3_bind_int(ppStmtAddCar, 3,
					   atoi(year->str))
	  || SQLITE_OK != sqlite3_bind_text(ppStmtAddCar, 4,
					    regist->str,
					    -1, SQLITE_STATIC)
	  || SQLITE_OK != sqlite3_bind_text(ppStmtAddCar, 5,
					    notes->str,
					    -1, SQLITE_STATIC)
	  || SQLITE_OK != sqlite3_bind_int(ppStmtUpdateCar, 6,
					   fueltype)
	  || SQLITE_DONE != sqlite3_step(ppStmtAddCar)) {
	PDEBUG("Problem saving the new car");
      }
      else { /* New car added */
#if MAEMO_VERSION_MAJOR == 5
	update_car_combo(pui->carbutton);
	update_car_combo(pui->stb_carbutton);
#else
	update_car_combo(pui->carcombo);
	update_car_combo(pui->stb_carcombo);
#endif
      }
      sqlite3_reset(ppStmtAddCar);
      g_string_free(mark, TRUE);
      g_string_free(model, TRUE);
      g_string_free(year, TRUE);
      g_string_free(regist, TRUE);
      g_string_free(notes, TRUE);
    }
    else {
      PDEBUG("Can't save new driver - no database open");
    }
  }
  gtk_widget_destroy(GTK_WIDGET(dialog));
}

/**
 * \fn void create_car_editwin(AppUIData *pui, GtkWidget *dialog)
 * \brief Creates a car editing window
 *
 * This function is used for creating an edit window both for adding a
 * new car and editing an existing one.
 *
 */
void create_car_editwin(AppUIData *pui, GtkWidget *dialog)
{
  enum
  {
    CAR_EDIT_MARK=0,
    CAR_EDIT_MODEL,
    CAR_EDIT_YEAR,
    CAR_EDIT_REGISTER,
    CAR_EDIT_FUELTYPE,
    CAR_EDIT_NOTES
  };

  GtkWidget *table;
  GtkWidget *label;
  GtkWidget *scrollwin;
  gchar *labels[] = {gettext_noop("Mark"),
		   gettext_noop("Model"),
		   gettext_noop("Year"),
		   gettext_noop("Register"),
		   gettext_noop("Fuel type"),
		   gettext_noop("Notes")
  };

#if MAEMO_VERSION_MAJOR == 5
  scrollwin = hildon_pannable_area_new();
  gtk_widget_set_size_request(scrollwin, -1, DIALOG_MIN_HEIGHTMAX);
#else
  scrollwin = gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_set_size_request(scrollwin, -1, DIALOG_MIN_HEIGHT1);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollwin), 
				 GTK_POLICY_NEVER, 
				 GTK_POLICY_AUTOMATIC);
#endif
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), scrollwin, TRUE, TRUE, 0);

  /* Mark */
  table = gtk_table_new(6, 2, FALSE);
/*   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), table, FALSE, FALSE, 5); */
#if MAEMO_VERSION_MAJOR == 5
  hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA(scrollwin),table);
#else
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrollwin),table);
#endif

  /* Mark */
  label=gtk_label_new(gettext(labels[CAR_EDIT_MARK]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

#if MAEMO_VERSION_MAJOR == 5
  pui->entrymark=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
#else
  pui->entrymark = gtk_entry_new();
#endif
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrymark), 30);
  gtk_table_attach(GTK_TABLE(table), pui->entrymark, 1, 2, 0, 1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrymark);

  /* Model */
  label=gtk_label_new(gettext(labels[CAR_EDIT_MODEL]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

#if MAEMO_VERSION_MAJOR == 5
  pui->entrymodel=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
#else
  pui->entrymodel = gtk_entry_new();
#endif
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrymodel), 30);
  gtk_table_attach(GTK_TABLE(table), pui->entrymodel, 1, 2, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrymodel);

  /* Year */
  label=gtk_label_new(gettext(labels[CAR_EDIT_YEAR]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

#if MAEMO_VERSION_MAJOR == 5
  pui->entryyear=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
#else
  pui->entryyear = gtk_entry_new();
#endif
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryyear), 30);
  gtk_table_attach(GTK_TABLE(table), pui->entryyear, 1, 2, 2, 3, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryyear);

  /* Register */
  label=gtk_label_new(gettext(labels[CAR_EDIT_REGISTER]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 3, 4, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

#if MAEMO_VERSION_MAJOR == 5
  pui->entryregister=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
#else
  pui->entryregister = gtk_entry_new();
#endif
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryregister), 30);
  gtk_table_attach(GTK_TABLE(table), pui->entryregister, 1, 2, 3, 4, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryregister);

  /* Fueltype */
#if MAEMO_VERSION_MAJOR == 5
  pui->fueltypebutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->fueltypebutton), gettext(labels[CAR_EDIT_FUELTYPE]));
  gtk_table_attach(GTK_TABLE(table), pui->fueltypebutton, 0, 2, 4, 5, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->fueltypebutton);
#else
  label=gtk_label_new(gettext(labels[CAR_EDIT_FUELTYPE]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 4, 5, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);
#endif

  pui->combofueltype = create_fueltype_combo(pui);
#if MAEMO_VERSION_MAJOR == 5
#else
  gtk_table_attach(GTK_TABLE(table), pui->combofueltype, 1, 2, 4, 5, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->combofueltype);
#endif

  /* Notes */
  label=gtk_label_new(gettext(labels[CAR_EDIT_NOTES]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 5, 6, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

#if MAEMO_VERSION_MAJOR == 5
  pui->entrycarnotes=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
#else
  pui->entrycarnotes = gtk_entry_new();
#endif
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrycarnotes), 30);
  gtk_table_attach(GTK_TABLE(table), pui->entrycarnotes, 1, 2, 5, 6, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrycarnotes);

  gtk_widget_show(table);
  gtk_widget_show(scrollwin);
}

/**
 * \fn void add_car_cb (GtkWidget *widget, gpointer   callback_data )
 * \brief Callback for adding a new car to the database
 *
 * This callback opens a dialog which allows to add a new car to the
 * database.
 *
 */
void add_car_cb (GtkWidget *widget, AppUIData *pui )
{
  GtkWidget *dialog;

  dialog = gtk_dialog_new_with_buttons(_("Add a new car"),
				       GTK_WINDOW(pui->app->mainwindow),
				       GTK_DIALOG_MODAL,
				       GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
				       GTK_STOCK_CANCEL,
				       GTK_RESPONSE_REJECT,
				       NULL);

  create_car_editwin(pui, dialog);

#if LIBHELP == 1
  help_dialog_help_enable(GTK_DIALOG(dialog),
				 HELP_ID_ADDCAR,
				 pui->app->osso);
#endif

  g_signal_connect (dialog, "response",
		    G_CALLBACK (add_car_response),
		    &ui);

  gtk_widget_show(dialog);

}

/**
 * \fn void edit_car_cb (GtkWidget *widget, gpointer   callback_data )
 * \brief Callback for edit car properties
 *
 * This callback opens a dialog which allows to edit car properties.
 *
 */
void edit_car_cb (GtkWidget *widget, AppUIData *pui )
{
  GtkWidget *dialog;

  dialog = gtk_dialog_new_with_buttons(_("Edit car properties"),
				       GTK_WINDOW(pui->app->mainwindow),
				       GTK_DIALOG_MODAL,
				       GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
				       GTK_STOCK_CANCEL,
				       GTK_RESPONSE_REJECT,
				       NULL);

  create_car_editwin(pui, dialog);

  if (SQLITE_OK == sqlite3_bind_int(ppStmtOneCar, 1, currentcar)) {
    while (SQLITE_ROW == sqlite3_step(ppStmtOneCar)) {
      gtk_entry_set_text(GTK_ENTRY(pui->entrymark),sqlite3_column_text(ppStmtOneCar,0));
      gtk_entry_set_text(GTK_ENTRY(pui->entrymodel),sqlite3_column_text(ppStmtOneCar,1));
      gtk_entry_set_text(GTK_ENTRY(pui->entryyear),sqlite3_column_text(ppStmtOneCar,2));
      gtk_entry_set_text(GTK_ENTRY(pui->entryregister),sqlite3_column_text(ppStmtOneCar,3));
      gtk_entry_set_text(GTK_ENTRY(pui->entrycarnotes),sqlite3_column_text(ppStmtOneCar,4));
#if MAEMO_VERSION_MAJOR == 5
      hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR (pui->combofueltype), 0, sqlite3_column_int(ppStmtOneCar,5));
#else
      gtk_combo_box_set_active(GTK_COMBO_BOX (pui->combofueltype),sqlite3_column_int(ppStmtOneCar,5));
#endif
    }
    sqlite3_reset(ppStmtOneCar);
  }

#if LIBHELP == 1
  help_dialog_help_enable(GTK_DIALOG(dialog),
				 HELP_ID_ADDCAR,
				 pui->app->osso);
#endif

  g_signal_connect (dialog, "response",
		    G_CALLBACK (edit_car_response),
		    &ui);

  gtk_widget_show(dialog);

}

/**
 * \fn void dateformat_changed_cb (GtkWidget *widget, gpointer
 *                                 callback_data ) 
 * \brief Callback for date format selection combo box
 *
 * This callback changes the dateformat in the curunit structure
 * according to the selection made in the date format combo box.
 *
 */
void dateformat_changed_cb (GtkWidget *widget, gpointer   callback_data )
{
#if MAEMO_VERSION_MAJOR == 5
  curunit.dateformat=hildon_touch_selector_get_active(HILDON_TOUCH_SELECTOR(widget), 0);
#else
  curunit.dateformat = gtk_combo_box_get_active(GTK_COMBO_BOX(widget));
#endif
}

/**
 * \fn void driver_changed_cb (GtkWidget *widget, gpointer   callback_data )
 * \brief Callback for driver selection combo box
 *
 */
void driver_changed_cb (GtkWidget *widget, gpointer   callback_data )
{
  unsigned int index;
  unsigned int i;

#if MAEMO_VERSION_MAJOR == 5
  index=hildon_touch_selector_get_active(HILDON_TOUCH_SELECTOR(widget), 0);
#else
  index = gtk_combo_box_get_active(GTK_COMBO_BOX(widget));
#endif
  currentdriver=db_get_driverid(index);
}

/**
 * \fn void car_changed_cb (GtkWidget *widget, gpointer   callback_data )
 * \brief Callback for car selection combo box
 *
 * This callback creates the contents of the record table after car
 * has been changed.
 *
 */
void car_changed_cb (GtkWidget *widget, gpointer   callback_data )
{
  unsigned int index;
  unsigned int i;

#if MAEMO_VERSION_MAJOR == 5
  index=hildon_touch_selector_get_active(HILDON_TOUCH_SELECTOR(widget), 0);
#else
  index=gtk_combo_box_get_active(GTK_COMBO_BOX(widget));
#endif
  currentcar=db_get_carid(index);

  carchanged=TRUE;
}

/**
 * \fn void warntime_changed_cb (GtkWidget *widget, gpointer   callback_data )
 * \brief Callback for alarm warning time combo changed
 *
 */
void warntime_changed_cb (GtkWidget *widget, gpointer   callback_data )
{
  unsigned int index;

#if MAEMO_VERSION_MAJOR == 5
  index=hildon_touch_selector_get_active(HILDON_TOUCH_SELECTOR(widget), 0);
#else
  index=gtk_combo_box_get_active(GTK_COMBO_BOX(widget));
#endif
  warndays=wtimes[index];
}

/**
 * \fn void paper_changed_cb (GtkWidget *widget, gpointer   callback_data )
 * \brief Callback for report paper combo changed
 *
 */
void paper_changed_cb (GtkWidget *widget, gpointer   callback_data )
{
  unsigned int index;

#if MAEMO_VERSION_MAJOR == 5
  index=hildon_touch_selector_get_active(HILDON_TOUCH_SELECTOR(widget), 0);
#else
  index=gtk_combo_box_get_active(GTK_COMBO_BOX(widget));
#endif
  reportpaper=(enum papertype)index;
}

/**
 * \fn GString *get_column_header(gint col)
 * \brief Returns the header of column col
 * \param col the column number
 * \return Pointer to a newly allocated column header string
 *
 * This functions returns the column header taking the used unit into
 * account. The string is allocated from the heap. After usage, the
 * string must be freed with g_string_free.
 *
 */
GString *get_column_header(gint col)
{
  GString *header;

  header = g_string_new(NULL);

  switch (col) {
  case COL_DAY:
  case COL_PRICE:
  case COL_SERVICE:
  case COL_OIL:
  case COL_TIRES:
  case COL_NOTES:
    g_string_assign(header, gettext(column_headers[col]));
    break;
  case COL_KM:
  case COL_TRIP:
  case COL_PRICEPERTRIP:
    if (curunit.lengthunit==SI)
      g_string_assign(header, gettext(column_headers[col]));
    else
      g_string_assign(header, gettext(column_headers_non_SI[col]));
    break;
  case COL_FILL:
  case COL_PRICEPERLITRE:
    if (curunit.volumeunit==SI)
      g_string_assign(header, gettext(column_headers[col]));
    else
      g_string_assign(header, gettext(column_headers_non_SI[col]));
    break;
  case COL_CONSUM:
    if (curunit.consumeunit==SI)
      g_string_assign(header, gettext(column_headers[col]));
    else
      g_string_assign(header, gettext(column_headers_non_SI[col]));
    break;
  case COL_CO2EMISSION:
    if (curunit.massunit==SI) {
      if (curunit.lengthunit==SI)
	g_string_assign(header, _("CO2 Emissions\n[g/km]"));
      else
	g_string_assign(header, _("CO2 Emissions\n[g/mile]"));
    }
    else {
    if (curunit.lengthunit==SI)
	g_string_assign(header, _("CO2 Emissions\n[lb/100 km]"));
      else
	g_string_assign(header, _("CO2 Emissions\n[lb/100 miles]"));
    }
    break;
  }
}

/**
 * \fn void update_column_headers(GtkWidget *view)
 * \brief Updates the headers to correspond to the current unit
 *
 */
void update_column_headers(GtkWidget *view)
{
  GtkTreeViewColumn   *col;
  GtkWidget           *label;
  GString *header;
  gint i;

  for (i=0;i<COL_ID;i++) {
    col = gtk_tree_view_get_column(GTK_TREE_VIEW(view), i);
    label = gtk_tree_view_column_get_widget(col); 
    header = get_column_header(i);
    gtk_label_set_text(GTK_LABEL(label), header->str);
    g_string_free(header, TRUE);
  }
}

enum unit get_unit(GtkWidget *widget)
{
  enum unit tmpunit;

#if MAEMO_VERSION_MAJOR == 5
  HildonTouchSelector *selector;
  selector = hildon_picker_button_get_selector(HILDON_PICKER_BUTTON(widget));
  tmpunit=(enum unit)hildon_touch_selector_get_active(HILDON_TOUCH_SELECTOR(selector), 0);
#else
  tmpunit = gtk_combo_box_get_active(GTK_COMBO_BOX(widget));
#endif
  if (tmpunit>IMPERIAL || tmpunit < SI) tmpunit=SI;

  return tmpunit;
}

/**
 * \fn void unit_changed_cb (GtkWidget *widget, gpointer   callback_data )
 * \brief Callback for unit selection combo boxes
 * \param callback_data selects between different unit combo boxes
 *
 * This callback sets the corresponding unit variable (mainunit,
 * lengthunit, volumeunit, consumeunit) to the value
 * read from the combo box.
 *
 */
void unit_changed_cb (GtkWidget *widget, AppUIData *pui )
{
  enum unit tmpunit;
  double oldwarnkm;

  if (GTK_IS_WIDGET(pui->entrywarnkm))
    oldwarnkm=get_entrywarnkm(pui);
  else
    oldwarnkm=0.0;
  tmpunit=get_unit(widget);
  curunit.mainunit = tmpunit;
  if (!curunit.individualunit) {
    curunit.lengthunit = tmpunit;
    curunit.volumeunit = tmpunit;
    curunit.consumeunit = tmpunit;
    if (pui->lengthcombo != NULL) {
#if MAEMO_VERSION_MAJOR == 5
      hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (pui->lengthcombo), 0, (gint)tmpunit);
      hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (pui->volumecombo), 0, (gint)tmpunit);
      hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (pui->consumecombo), 0, (gint)tmpunit);
      hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (pui->masscombo), 0, (gint)tmpunit);
#else
      gtk_combo_box_set_active(GTK_COMBO_BOX (pui->lengthcombo),tmpunit);
      gtk_combo_box_set_active(GTK_COMBO_BOX (pui->volumecombo),tmpunit);
      gtk_combo_box_set_active(GTK_COMBO_BOX (pui->consumecombo),tmpunit);
      gtk_combo_box_set_active(GTK_COMBO_BOX (pui->masscombo),tmpunit);
#endif
    }
  }
  if (GTK_IS_WIDGET(pui->entrywarnkm)) {
    update_warn_label(pui->labelentrywarn);
    update_warn_entry(pui->entrywarnkm, oldwarnkm);
  }
}

void l_unit_changed_cb(GtkWidget *widget, AppUIData *pui)
{
  double oldwarnkm;

  if (GTK_IS_WIDGET(pui->entrywarnkm))
    oldwarnkm=get_entrywarnkm(pui);
  else
    oldwarnkm=0.0;
  curunit.lengthunit = get_unit(widget);
  if (GTK_IS_WIDGET(pui->entrywarnkm)) {
    update_warn_label(pui->labelentrywarn);
    update_warn_entry(pui->entrywarnkm, oldwarnkm);
  }
}

void v_unit_changed_cb(GtkWidget *widget, AppUIData *pui)
{
  curunit.volumeunit = get_unit(widget);
}

void c_unit_changed_cb(GtkWidget *widget, AppUIData *pui)
{
  curunit.consumeunit = get_unit(widget);
}

void m_unit_changed_cb(GtkWidget *widget, AppUIData *pui)
{
  curunit.massunit = get_unit(widget);
}

/**
 * \fn void unit_check_cb (GtkWidget *widget, AppUIData *pui )
 * \brief Callback for check button for selecting the unit individually
 *
 * This function either enables or disables the unit selection
 * comboboxes and sets the global variable individualunit according to
 * the state of the check button.
 *
 */
void unit_check_cb (GtkWidget *widget, AppUIData *pui )
{
#if MAEMO_VERSION_MAJOR == 5
if (hildon_check_button_get_active (HILDON_CHECK_BUTTON (widget))) 
#else
if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget))) 
#endif
  {
    /* If control reaches here, the toggle button is down */
    curunit.individualunit = TRUE;
#if MAEMO_VERSION_MAJOR == 5
       gtk_widget_set_sensitive(pui->unitsystembutton, FALSE);
       gtk_widget_set_sensitive(pui->lengthunitbutton, TRUE);
       gtk_widget_set_sensitive(pui->volumeunitbutton, TRUE);
       gtk_widget_set_sensitive(pui->consumeunitbutton, TRUE);
       gtk_widget_set_sensitive(pui->massunitbutton, TRUE);
       gtk_widget_show(pui->lengthunitbutton);
       gtk_widget_show(pui->volumeunitbutton);
       gtk_widget_show(pui->consumeunitbutton);
       gtk_widget_show(pui->massunitbutton);
#else
    gtk_widget_set_sensitive(pui->unitcombo, FALSE);
    gtk_widget_set_sensitive(pui->lengthcombo, TRUE);
    gtk_widget_set_sensitive(pui->volumecombo, TRUE);
    gtk_widget_set_sensitive(pui->consumecombo, TRUE);
    gtk_widget_set_sensitive(pui->masscombo, TRUE);
#endif
  } else {  
  /* If control reaches here, the toggle button is up */
    curunit.individualunit = FALSE;
#if MAEMO_VERSION_MAJOR == 5
       gtk_widget_set_sensitive(pui->unitsystembutton, TRUE);
       gtk_widget_set_sensitive(pui->lengthunitbutton, FALSE);
       gtk_widget_set_sensitive(pui->volumeunitbutton, FALSE);
       gtk_widget_set_sensitive(pui->consumeunitbutton, FALSE);
       gtk_widget_set_sensitive(pui->massunitbutton, FALSE);
       gtk_widget_hide(pui->lengthunitbutton);
       gtk_widget_hide(pui->volumeunitbutton);
       gtk_widget_hide(pui->consumeunitbutton);
       gtk_widget_hide(pui->massunitbutton);
       unit_changed_cb(pui->unitsystembutton, pui);
#else
    gtk_widget_set_sensitive(pui->unitcombo, TRUE);
    gtk_widget_set_sensitive(pui->lengthcombo, FALSE);
    gtk_widget_set_sensitive(pui->volumecombo, FALSE);
    gtk_widget_set_sensitive(pui->consumecombo, FALSE);
    gtk_widget_set_sensitive(pui->masscombo, FALSE);
    unit_changed_cb(pui->unitcombo, pui);
#endif
 }
}

/**
 * \fn void update_car_changed(AppUIData *pui)
 * \brief Creates a combo box for selecting a unit
 * \param *pui Pointer to the UI data structure
 *
 * This function can be used after the current driver has been changed
 * to save the driver to gconf and update the labels in the main window.
 *
 */
void update_car_changed(AppUIData *pui)
{
  config_save();
  update_record_model(pui);
}

double get_entrywarnkm(AppUIData *pui)
{
  double warn;

  warn = atof(gtk_entry_get_text(GTK_ENTRY(pui->entrywarnkm)));
  return user2SIlength(warn);
}

void settings_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  gboolean warn;
  switch (arg1) {
  case GTK_RESPONSE_ACCEPT:
    g_string_assign(curunit.currency, gtk_entry_get_text(GTK_ENTRY(pui->entrycurrency)));
    warnkm=get_entrywarnkm(pui);
    update_car_changed(pui);
    gtk_widget_destroy(GTK_WIDGET(dialog));

    /* Update the next event estimates */
    pui->alarmview = create_alarmview_and_model (pui);
    pui->warn = update_next_event(gtk_tree_view_get_model(GTK_TREE_VIEW(pui->alarmview)));
    update_reminder_toolbutton (pui, pui->warn);

    /* Show or hide toolbars */
    main_toolbar_show_hide(pui);
    secondary_toolbar_show_hide(pui);

#if MAEMO_VERSION_MAJOR == 5
    update_car_combo(pui->stb_carbutton);
#else
    update_car_combo(pui->stb_carcombo);
#endif

    break;
  case GTK_RESPONSE_HELP:
    callback_about(NULL, NULL);
    break;
  case RESPONSE_SELECT_COLUMNS:
    callback_select_columns(NULL, pui);
    break;
  }
}

void car_apply_cb (GtkWidget *widget, AppUIData *pui )
{
  gboolean warn;

  update_car_changed(pui);

  /* Update the next event estimates */
  pui->alarmview = create_alarmview_and_model (pui);
  pui->warn = update_next_event(gtk_tree_view_get_model(GTK_TREE_VIEW(pui->alarmview)));
  update_reminder_toolbutton (pui, pui->warn);
  /* Show or hide toolbars */
  main_toolbar_show_hide(pui);
  secondary_toolbar_show_hide(pui);
}

/**
 * \fn GtkWidget *create_unit_combo(gint whichunit, AppUIData *pui)
 * \brief Creates a combo box for selecting a unit
 * \param whichunit UNITCOMBO, LENGTHCOMBO, VOLUMECOMBO, CONSUMECOMBO or MASSCOMBO
 * \param pui pointer to user interface data
 * \return Pointer to the created combo box
 *
 * This function is used to create a unit selection combo box. The
 * unit selection combo box containts selections SI, US and
 * Imperial. The default value is read from mainunit,
 * lengthunit, volumeunit or consumeunit
 * depending on the value of the parameter whichunit.
 *
 */
#if MAEMO_VERSION_MAJOR == 5
GtkWidget *create_unit_combo(gint whichunit, AppUIData *pui, GtkWidget *button)
#else
GtkWidget *create_unit_combo(gint whichunit, AppUIData *pui)
#endif
{
  GtkWidget *combo;
  enum unit tmpunit=curunit.mainunit;

#if MAEMO_VERSION_MAJOR == 5
  combo = hildon_touch_selector_new_text ();
#else
  combo = gtk_combo_box_new_text();
#endif
  
#if MAEMO_VERSION_MAJOR == 5
  hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (combo), "SI");
  hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (combo), "US");
  hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (combo), "Imperial");
#else
  gtk_combo_box_append_text(GTK_COMBO_BOX (combo),"SI");
  gtk_combo_box_append_text(GTK_COMBO_BOX (combo),"US");
  gtk_combo_box_append_text(GTK_COMBO_BOX (combo),"Imperial");
#endif

  switch (whichunit) {
  case UNITCOMBO: 
    tmpunit = curunit.mainunit;
#if MAEMO_VERSION_MAJOR == 5
    hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (combo), 0, tmpunit);
    hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (button),
				       HILDON_TOUCH_SELECTOR (combo));
#else
    g_signal_connect (GTK_OBJECT (combo), "changed",
		      G_CALLBACK(unit_changed_cb), pui);
#endif
    break;
  case LENGTHCOMBO:
    tmpunit = curunit.lengthunit;
#if MAEMO_VERSION_MAJOR == 5
    hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (combo), 0, tmpunit);
    hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (button),
				       HILDON_TOUCH_SELECTOR (combo));
#else
    g_signal_connect (GTK_OBJECT (combo), "changed",
		      G_CALLBACK(l_unit_changed_cb), pui);
#endif
    break;
  case VOLUMECOMBO:
    tmpunit = curunit.volumeunit;
#if MAEMO_VERSION_MAJOR == 5
    hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (combo), 0, tmpunit);
    hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (button),
				       HILDON_TOUCH_SELECTOR (combo));
#else
    g_signal_connect (GTK_OBJECT (combo), "changed",
		      G_CALLBACK(v_unit_changed_cb), pui);
#endif
    break;
  case CONSUMECOMBO:
    tmpunit = curunit.consumeunit;
#if MAEMO_VERSION_MAJOR == 5
    hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (combo), 0, tmpunit);
    hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (button),
				       HILDON_TOUCH_SELECTOR (combo));
#else
    g_signal_connect (GTK_OBJECT (combo), "changed",
		      G_CALLBACK(c_unit_changed_cb), pui);
#endif
    break;
  case MASSCOMBO:
    tmpunit = curunit.massunit;
#if MAEMO_VERSION_MAJOR == 5
    hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (combo), 0, tmpunit);
    hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (button),
				       HILDON_TOUCH_SELECTOR (combo));
#else
    g_signal_connect (GTK_OBJECT (combo), "changed",
		      G_CALLBACK(m_unit_changed_cb), pui);
#endif
    break;
  }
#if MAEMO_VERSION_MAJOR == 5
/*     hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (combo), 0, tmpunit); */
#else
  gtk_combo_box_set_active(GTK_COMBO_BOX (combo),tmpunit);
#endif


  return combo;
}

void fill_driver_combo(GtkWidget *combo)
{
  gint i;
  gint active;
  gint64 id;

  active=0;
  i=0;
  if (db_open()) {

    /* We'll need to block, otherwise driver_changed_cb would be fired in between */
    g_signal_handlers_block_by_func(combo, G_CALLBACK(driver_changed_cb), NULL);
    while(SQLITE_ROW == sqlite3_step(ppStmtDriver)) {
#if MAEMO_VERSION_MAJOR == 5
      hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (combo), sqlite3_column_text(ppStmtDriver,0));
#else
      gtk_combo_box_append_text(GTK_COMBO_BOX (combo),sqlite3_column_text(ppStmtDriver,0));
#endif
      id=sqlite3_column_int64(ppStmtDriver,2);
      if (id==currentdriver)
	active=i;
      i++;
    }
    g_signal_handlers_unblock_by_func(combo, G_CALLBACK(driver_changed_cb), NULL);
    sqlite3_reset(ppStmtDriver);
#if MAEMO_VERSION_MAJOR == 5
    hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (combo), 0, active);
#else
    gtk_combo_box_set_active(GTK_COMBO_BOX (combo),active);
#endif
  }
}

/**
 * \fn void update_driver_combo(gpointer combo)
 * \brief Update the contents of driver combo
 * \param combo Pointer to the already existing combo vox
 *
 * This function can be used to update the driver selection combo box
 * after adding a new driver or deleting one.
 *
 */
void update_driver_combo(gpointer combo)
{
#if MAEMO_VERSION_MAJOR == 5
  HildonTouchSelectorColumn *column;
  GtkTreeIter iter;
#endif
  GtkTreeModel *model;
  gint i, count;

#if MAEMO_VERSION_MAJOR == 5
  model = hildon_touch_selector_get_model(HILDON_TOUCH_SELECTOR(combo), 0);
  gtk_list_store_clear(GTK_LIST_STORE(model));
#else
  model = gtk_combo_box_get_model(GTK_COMBO_BOX(combo));
  count = gtk_tree_model_iter_n_children(model, NULL);
  for (i=0; i < count; i++)
    gtk_combo_box_remove_text(GTK_COMBO_BOX(combo), 0);
#endif
  fill_driver_combo(GTK_WIDGET(combo));
}


#if MAEMO_VERSION_MAJOR == 5
GtkWidget *create_driver_combo(AppUIData *pui, GtkWidget *button)
#else
GtkWidget *create_driver_combo(AppUIData *pui)
#endif
{
  GtkWidget *combo;

#if MAEMO_VERSION_MAJOR == 5
  combo = hildon_touch_selector_new_text ();
#else
  combo = gtk_combo_box_new_text();
#endif
  fill_driver_combo(combo);

#if MAEMO_VERSION_MAJOR == 5
  hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (button),
                                     HILDON_TOUCH_SELECTOR (combo));
#endif

  g_signal_connect (GTK_OBJECT (combo), "changed",
		    G_CALLBACK(driver_changed_cb), NULL);

  return combo;
}

/**
 * \fn void fill_car_combo(GtkWidget *combo)
 * \brief Fills the car combo with all the cars in the database
 *
 * This callback fills the car combo with all the cars found in the
 * database.
 *
 * \todo This should be changed so that fuelpad.c has a function which
 * is called to obtain the data (i.e. sqlite would not be used here)
 *
 */
void fill_car_combo(GtkWidget *combo)
{
  gint i;
  gint active;
  gint64 id;
  GString *listitemtext;

  active=0;
  i=0;
  if (db_open()) {
    listitemtext=g_string_new(NULL);

    /* We'll need to block, otherwise car_changed_cb would be fired in between */
    g_signal_handlers_block_by_func(combo, G_CALLBACK(car_changed_cb), NULL);
    while(SQLITE_ROW == sqlite3_step(ppStmtCar)) {
      
      g_string_printf(listitemtext, "%s %s",sqlite3_column_text(ppStmtCar,0),sqlite3_column_text(ppStmtCar,1));

#if MAEMO_VERSION_MAJOR == 5
      hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (combo), listitemtext->str);
#else
      gtk_combo_box_append_text(GTK_COMBO_BOX (combo),listitemtext->str);
#endif
      id=sqlite3_column_int64(ppStmtCar,2);
      if (id==currentcar)
	active=i;
      i++;
    }
    g_signal_handlers_unblock_by_func(combo, G_CALLBACK(car_changed_cb), NULL);
    g_string_free(listitemtext, TRUE);
    sqlite3_reset(ppStmtCar);
#if MAEMO_VERSION_MAJOR == 5
    hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (combo), 0, active);
#else
    gtk_combo_box_set_active(GTK_COMBO_BOX (combo),active);
#endif
  }
}

/**
 * \fn void update_car_combo(gpointer widget)
 * \brief Update the contents of car combo
 * \param combo Pointer to the already existing combo vox
 *
 * This function can be used to update the car selection combo box
 * after adding a new car or deleting one.
 *
 */
void update_car_combo(gpointer widget)
{
#if MAEMO_VERSION_MAJOR == 5
  HildonTouchSelectorColumn *column;
  HildonTouchSelector *selector;
#endif
  GtkTreeModel *model;
  gint i, count;

#if MAEMO_VERSION_MAJOR == 5
  selector = hildon_picker_button_get_selector(HILDON_PICKER_BUTTON(widget));

  /* We'll need to block callbacks, otherwise they could be fired in between */
  g_signal_handlers_block_by_func(GTK_WIDGET(widget), G_CALLBACK(car_apply_cb), &ui);
  g_signal_handlers_block_by_func(GTK_WIDGET(selector), G_CALLBACK(car_changed_cb), NULL);

  model = hildon_touch_selector_get_model(HILDON_TOUCH_SELECTOR(selector), 0);
  gtk_list_store_clear(GTK_LIST_STORE(model));
  fill_car_combo(GTK_WIDGET(selector));

  g_signal_handlers_unblock_by_func(GTK_WIDGET(selector), G_CALLBACK(car_changed_cb), NULL);
  g_signal_handlers_unblock_by_func(GTK_WIDGET(widget), G_CALLBACK(car_apply_cb), &ui);

/*   hildon_touch_selector_remove_column (HILDON_TOUCH_SELECTOR(combo), 0);   */
/*   column = hildon_touch_selector_append_text_column (HILDON_TOUCH_SELECTOR(combo), */
/* 						     model, */
/* 						     TRUE); */
/*   hildon_touch_selector_column_set_text_column(column, 0); */
#else
  model = gtk_combo_box_get_model(GTK_COMBO_BOX(widget));
  count = gtk_tree_model_iter_n_children(model, NULL);
  for (i=0; i < count; i++)
    gtk_combo_box_remove_text(GTK_COMBO_BOX(widget), 0);
  fill_car_combo(GTK_WIDGET(combo));
#endif
}

#if MAEMO_VERSION_MAJOR == 5
GtkWidget *create_car_combo(AppUIData *pui, GtkWidget *button)
#else
GtkWidget *create_car_combo(AppUIData *pui)
#endif
{
  GtkWidget *combo;
  
#if MAEMO_VERSION_MAJOR == 5
  combo = hildon_touch_selector_new_text ();
#else
  combo = gtk_combo_box_new_text();
#endif
  fill_car_combo(combo);

#if MAEMO_VERSION_MAJOR == 5
  hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (button),
                                     HILDON_TOUCH_SELECTOR (combo));
#endif

  g_signal_connect (GTK_OBJECT (combo), "changed",
		    G_CALLBACK(car_changed_cb), NULL);

  return combo;
}

#if MAEMO_VERSION_MAJOR == 5
GtkWidget *create_andor_combo(GtkWidget *button)
#else
GtkWidget *create_andor_combo(void)
#endif
{
  enum 
  {
    ANDOR_AND=0,
    ANDOR_OR
  };
  GtkWidget *combo;
  gchar *labels[] = {gettext_noop("Mandatory"),
		     gettext_noop("Optional")
  };

#if MAEMO_VERSION_MAJOR == 5
  combo = hildon_touch_selector_new_text ();
  hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (combo), gettext(labels[ANDOR_AND]));
  hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (combo), gettext(labels[ANDOR_OR]));
  hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR (combo), 0, 1);
  hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (button),
                                     HILDON_TOUCH_SELECTOR (combo));
#else
  combo = gtk_combo_box_new_text();
  gtk_combo_box_append_text(GTK_COMBO_BOX (combo), gettext(labels[ANDOR_AND]));
  gtk_combo_box_append_text(GTK_COMBO_BOX (combo), gettext(labels[ANDOR_OR]));
  gtk_combo_box_set_active(GTK_COMBO_BOX (combo), 1);
#endif
  return combo;
}

GtkWidget *create_dateformat_combo(AppUIData *pui)
{
  GtkWidget *combo;
  gint       i;
  char *timestr;

#if MAEMO_VERSION_MAJOR == 5
  combo = hildon_touch_selector_new_text ();
#else
  combo = gtk_combo_box_new_text();
#endif
  g_signal_connect (GTK_OBJECT (combo), "changed",
		    G_CALLBACK(dateformat_changed_cb), NULL);
  for (i=0;i<sizeof(datefmtstr)/sizeof(char *);i++) {
    gettimefmt(&timestr, datefmtstr[i]);
#if MAEMO_VERSION_MAJOR == 5
    hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (combo), timestr);
#else
    gtk_combo_box_append_text(GTK_COMBO_BOX (combo),timestr);
#endif
    free(timestr);
  }

#if MAEMO_VERSION_MAJOR == 5
  hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (combo), 0, curunit.dateformat);
#else
  gtk_combo_box_set_active(GTK_COMBO_BOX (combo),curunit.dateformat);
#endif

#if MAEMO_VERSION_MAJOR == 5
  hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (pui->datebutton),
                                     HILDON_TOUCH_SELECTOR (combo));
#endif

  return combo;
}

GtkWidget *create_warntime_combo(AppUIData *pui)
{
  GtkWidget *combo;
  gint i;

#if MAEMO_VERSION_MAJOR == 5
  combo = hildon_touch_selector_new_text ();
#else
  combo = gtk_combo_box_new_text();
#endif

  for (i=0;i<sizeof(warntexts)/sizeof(char *);i++) {
#if MAEMO_VERSION_MAJOR == 5
    hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (combo), gettext(warntexts[i]));
#else
    gtk_combo_box_append_text(GTK_COMBO_BOX (combo),gettext(warntexts[i]));
#endif
  }
  i=0;
  while(i<sizeof(wtimes)/sizeof(float) && warndays!=wtimes[i])
    i++;
#if MAEMO_VERSION_MAJOR == 5
  hildon_touch_selector_set_active (HILDON_TOUCH_SELECTOR (combo), 0, i);
#else
  gtk_combo_box_set_active(GTK_COMBO_BOX (combo),i);
#endif
  g_signal_connect (GTK_OBJECT (combo), "changed",
		    G_CALLBACK(warntime_changed_cb), NULL);

#if MAEMO_VERSION_MAJOR == 5
  hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (pui->remindbutton),
                                     HILDON_TOUCH_SELECTOR (combo));
#endif

  return combo;
}

GtkWidget *create_paper_combo(AppUIData *pui)
{
  GtkWidget *combo;

#if MAEMO_VERSION_MAJOR == 5
  combo = hildon_touch_selector_new_text ();
  hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (combo),"A4");
  hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (combo),"Letter");
  hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(combo),0,0);

  hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (pui->paperbutton),
                                     HILDON_TOUCH_SELECTOR (combo));
#else
  combo = gtk_combo_box_new_text();
  gtk_combo_box_append_text(GTK_COMBO_BOX (combo),"A4");
  gtk_combo_box_append_text(GTK_COMBO_BOX (combo),"Letter");
  gtk_combo_box_set_active(GTK_COMBO_BOX (combo),0);
#endif

  g_signal_connect (GTK_OBJECT (combo), "changed",
		    G_CALLBACK(paper_changed_cb), NULL);

  switch (reportpaper) {
  case A4PAPER:
#if MAEMO_VERSION_MAJOR == 5
    hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(combo),0,0);
#else
    gtk_combo_box_set_active(GTK_COMBO_BOX (combo),0);
#endif
    break;
  case LETTERPAPER:
#if MAEMO_VERSION_MAJOR == 5
    hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(combo),0,1);
#else
    gtk_combo_box_set_active(GTK_COMBO_BOX (combo),1);
#endif
  }

  return combo;
}

GtkWidget *create_fueltype_combo(AppUIData *pui)
{
  enum
  {
    FUELTYPE_PETROL=0,
    FUELTYPE_DIESEL
  };

  GtkWidget *combo;
  gchar *labels[] = {gettext_noop("Petrol"),
		   gettext_noop("Diesel")
  };

#if MAEMO_VERSION_MAJOR == 5
  combo = hildon_touch_selector_new_text ();
  hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (combo), gettext(labels[FUELTYPE_PETROL]));
  hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (combo), gettext(labels[FUELTYPE_DIESEL]));
  hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR (combo), 0, 0);
  hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (pui->fueltypebutton),
                                     HILDON_TOUCH_SELECTOR (combo));
#else
  combo = gtk_combo_box_new_text();
  gtk_combo_box_append_text(GTK_COMBO_BOX (combo), gettext(labels[FUELTYPE_PETROL]));
  gtk_combo_box_append_text(GTK_COMBO_BOX (combo), gettext(labels[FUELTYPE_DIESEL]));
  gtk_combo_box_set_active(GTK_COMBO_BOX (combo), 0);
#endif
  return combo;
}


#define MAXWARNLEN 10
void update_warn_label(GtkWidget *label)
{
  if (curunit.lengthunit==SI)
    gtk_label_set_text(GTK_LABEL(label), _("km before"));
  else
    gtk_label_set_text(GTK_LABEL(label), _("miles before"));
}

void update_warn_entry(GtkWidget *entry, double value)
{
  GString *s;

  s=g_string_new(NULL);
  g_string_printf(s, "%.0f", SIlength2user(value));
  gtk_entry_set_text(GTK_ENTRY(entry),s->str);
  g_string_free(s, TRUE);
}

void callback_settings( GtkAction * action, AppUIData *pui )
{
  enum
  {
    SETTINGS_TITLE=0,
    SETTINGS_DATABASE,
    SETTINGS_OPEN,
    SETTINGS_IMPORT,
    SETTINGS_EXPORT,
    SETTINGS_DRIVER,
    SETTINGS_EDIT,
    SETTINGS_NEW,
    SETTINGS_DELETE,
    SETTINGS_CAR,
    SETTINGS_DATEFORMAT,
    SETTINGS_WARN,
    SETTINGS_REMIND,
    SETTINGS_CURRENCY,
    SETTINGS_UNITSYSTEM,
    SETTINGS_INDIVIDUAL,
    SETTINGS_LENGTHUNIT,
    SETTINGS_VOLUMEUNIT,
    SETTINGS_CONSUMEUNIT,
    SETTINGS_MASSUNIT,
    SETTINGS_PAPER
  };

  enum 
  {
    SETTINGS_HDR_DATABASE=0,
    SETTINGS_HDR_CARDRIVER,
    SETTINGS_HDR_REMINDERS,
    SETTINGS_HDR_UNITS,
    SETTINGS_HDR_REPORT
  };


#if MAEMO_VERSION_MAJOR == 5
  GtkWidget *scrollwin;
  GtkWidget *hbox;
  GtkWidget *svbox;
  GtkWidget *uvbox;
  GtkWidget *frame;
#else
  GtkWidget *notebook;
#endif
  GtkWidget *dialog;
  GtkWidget *vbox;
  GtkWidget *table;
  GtkWidget *label;
  GtkWidget *button;
  GtkWidget *unitcheck;
  GtkWidget *unitscrollwin;
  gint       i;
  gint       row;
  gchar *labels[] = {gettext_noop("Settings"),
		   gettext_noop("Database"),
		   gettext_noop("Open"),
		   gettext_noop("Import"),
		   gettext_noop("Export"),
		   gettext_noop("Driver"),
		   gettext_noop("Edit"),
		   gettext_noop("New"),
		   gettext_noop("Delete"),
		   gettext_noop("Car"),
		   gettext_noop("Date format"),
		   gettext_noop("Warn"),
		   gettext_noop("Remind before"),
		   gettext_noop("Currency"),
		   gettext_noop("Unit system"),
		   gettext_noop("Select individually"),
		   gettext_noop("Length unit"),
		   gettext_noop("Volume unit"),
		   gettext_noop("Fuel consumption unit"),
		   gettext_noop("Mass unit"),
		   gettext_noop("Paper"),
  };
  gchar *headers[] = {gettext_noop("Database"),
		      gettext_noop("Driver and car"),
		      gettext_noop("Reminders"),
		      gettext_noop("Units"),
		      gettext_noop("Report")
  };


#if MAEMO_VERSION_MAJOR == 5
  dialog = gtk_dialog_new_with_buttons(gettext(labels[SETTINGS_TITLE]),
				       GTK_WINDOW(pui->app->mainwindow),
				       GTK_DIALOG_MODAL,
				       _("About"),
				       GTK_RESPONSE_HELP,
				       _("Select columns"),
				       RESPONSE_SELECT_COLUMNS,
				       GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
				       NULL);
#else
  dialog = gtk_dialog_new_with_buttons(gettext(labels[SETTINGS_TITLE]),
				       GTK_WINDOW(pui->app->mainwindow),
				       GTK_DIALOG_MODAL,
				       GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
				       NULL);
#endif

  vbox = GTK_WIDGET(GTK_DIALOG(dialog)->vbox);
#if MAEMO_VERSION_MAJOR == 5
  scrollwin = hildon_pannable_area_new();
  gtk_widget_set_size_request(scrollwin, -1, DIALOG_MIN_HEIGHTMAX);
  svbox = gtk_vbox_new (FALSE, 0);
  row=0;
#else
  notebook = gtk_notebook_new ();
  gtk_notebook_set_tab_pos (GTK_NOTEBOOK (notebook), GTK_POS_TOP);
  gtk_box_pack_start (GTK_BOX(vbox), notebook, TRUE, TRUE, 0);
  gtk_widget_show (notebook);
#endif

  /** Driver and car notebook page **/

#if MAEMO_VERSION_MAJOR == 5

/*   table = gtk_table_new(16, 3, FALSE); */

  frame=gtk_frame_new(gettext(headers[SETTINGS_HDR_CARDRIVER]));
  gtk_box_pack_start (GTK_BOX(svbox), frame, TRUE, TRUE, 5);
  gtk_widget_show(frame);

  row++;
  /* First row */

  table = gtk_table_new(4, 2, FALSE);
  gtk_container_add (GTK_CONTAINER(frame), table);

  pui->driverbutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->driverbutton), gettext(labels[SETTINGS_DRIVER]));
  gtk_table_attach(GTK_TABLE(table), pui->driverbutton, 0, 1, 0, 1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->driverbutton);

  pui->drivercombo = create_driver_combo(pui, pui->driverbutton);

  button = hildon_gtk_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label(GTK_BUTTON(button), gettext(labels[SETTINGS_EDIT]));
  gtk_widget_set_sensitive(button, db_open());
  gtk_table_attach(GTK_TABLE(table), button, 1, 2, 0, 1,
		   GTK_EXPAND|GTK_FILL, 
		   0, 0, 5);
  g_signal_connect (GTK_OBJECT (button), "clicked",
		    G_CALLBACK(edit_driver_cb), &ui);
  gtk_widget_show(button);

  row++;
  button = hildon_gtk_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label(GTK_BUTTON(button), gettext(labels[SETTINGS_NEW]));
  gtk_widget_set_sensitive(button, db_open());
  gtk_table_attach(GTK_TABLE(table), button, 2, 3, 0, 1,
		   GTK_EXPAND|GTK_FILL, 
		   0, 0, 5);
  g_signal_connect (GTK_OBJECT (button), "clicked",
		    G_CALLBACK(add_driver_cb), &ui);
  gtk_widget_show(button);

  button = hildon_gtk_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label(GTK_BUTTON(button), gettext(labels[SETTINGS_DELETE]));
  gtk_widget_set_sensitive(button, db_open());
  gtk_table_attach(GTK_TABLE(table), button, 3, 4, 0, 1,
		   GTK_EXPAND|GTK_FILL, 
		   0, 0, 5);
  g_signal_connect (GTK_OBJECT (button), "clicked",
		    G_CALLBACK(delete_driver_cb), &ui);
  gtk_widget_show(button);

/*   gtk_widget_show(table); */

  row++;
  /* Second row */

/*   table = gtk_table_new(4, 1, FALSE); */
/*   gtk_box_pack_start (GTK_BOX(svbox), table, TRUE, TRUE, 0); */

  pui->carbutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->carbutton), gettext(labels[SETTINGS_CAR]));
  gtk_table_attach(GTK_TABLE(table), pui->carbutton, 0, 1, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->carbutton);

  pui->carcombo = create_car_combo(pui, pui->carbutton);

  button = hildon_gtk_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label(GTK_BUTTON(button), gettext(labels[SETTINGS_EDIT]));
  gtk_widget_set_sensitive(button, db_open());
  gtk_table_attach(GTK_TABLE(table), button, 1, 2, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  g_signal_connect (GTK_OBJECT (button), "clicked",
		    G_CALLBACK(edit_car_cb), pui);
  gtk_widget_show(button);

  row++;
  button = hildon_gtk_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label(GTK_BUTTON(button), gettext(labels[SETTINGS_NEW]));
  gtk_widget_set_sensitive(button, db_open());
  gtk_table_attach(GTK_TABLE(table), button, 2, 3, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  g_signal_connect (GTK_OBJECT (button), "clicked",
		    G_CALLBACK(add_car_cb), pui);
  gtk_widget_show(button);

  button = hildon_gtk_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label(GTK_BUTTON(button), gettext(labels[SETTINGS_DELETE]));
  gtk_widget_set_sensitive(button, db_open());
  gtk_table_attach(GTK_TABLE(table), button, 3, 4, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  g_signal_connect (GTK_OBJECT (button), "clicked",
		    G_CALLBACK(delete_car_cb), pui);
  gtk_widget_show(button);

  gtk_widget_show(table);

  row++;
  /* Third row */

/*   gtk_widget_show(table); */

  row++;

  /* First row */

  frame=gtk_frame_new(gettext(headers[SETTINGS_HDR_REMINDERS]));
  gtk_box_pack_start (GTK_BOX(svbox), frame, TRUE, TRUE, 5);
  gtk_widget_show(frame);

  hbox=gtk_hbox_new(FALSE, 0);
  gtk_container_add (GTK_CONTAINER(frame), hbox);

  label=gtk_label_new(gettext(labels[SETTINGS_WARN]));
  gtk_box_pack_start (GTK_BOX(hbox), label, TRUE, TRUE, 0);
  gtk_widget_show(label);

  pui->entrywarnkm=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrywarnkm), MAXWARNLEN);
  update_warn_entry(pui->entrywarnkm, warnkm);
  gtk_box_pack_start (GTK_BOX(hbox), pui->entrywarnkm, TRUE, TRUE, 0);
  gtk_widget_show(pui->entrywarnkm);

  pui->labelentrywarn=gtk_label_new(NULL);
  update_warn_label(pui->labelentrywarn);
  gtk_box_pack_start (GTK_BOX(hbox), pui->labelentrywarn, TRUE, TRUE, 0);
  gtk_widget_show(pui->labelentrywarn);

  gtk_widget_show(hbox);

  row++;
  /* Second row */
  pui->remindbutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->remindbutton), gettext(labels[SETTINGS_REMIND]));
  gtk_box_pack_start (GTK_BOX(svbox), pui->remindbutton, TRUE, TRUE, 0);
  gtk_widget_show(pui->remindbutton);

  pui->warntimecombo=create_warntime_combo(pui);
    
  //  gtk_widget_show(table);

  row++;

  frame=gtk_frame_new(gettext(headers[SETTINGS_HDR_UNITS]));
  gtk_box_pack_start (GTK_BOX(svbox), frame, TRUE, TRUE, 5);
  gtk_widget_show(frame);

  uvbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER(frame), uvbox);

  /* First row */
  pui->datebutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->datebutton), gettext(labels[SETTINGS_DATEFORMAT]));
  gtk_box_pack_start (GTK_BOX(uvbox), pui->datebutton, TRUE, TRUE, 0);
  gtk_widget_show(pui->datebutton);

  pui->datecombo = create_dateformat_combo(pui);

  hbox=gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start (GTK_BOX(uvbox), hbox, TRUE, TRUE, 0);

  label=gtk_label_new(gettext(labels[SETTINGS_CURRENCY]));
  gtk_box_pack_start (GTK_BOX(hbox), label, TRUE, TRUE, 0);
  gtk_widget_show(label);

  pui->entrycurrency=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrycurrency), 30);
  gtk_entry_set_text(GTK_ENTRY(pui->entrycurrency),curunit.currency->str);
  gtk_box_pack_start (GTK_BOX(hbox), pui->entrycurrency, TRUE, TRUE, 0);
  gtk_widget_show(pui->entrycurrency);

  gtk_widget_show(hbox);

  row++;
  pui->unitsystembutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->unitsystembutton), gettext(labels[SETTINGS_UNITSYSTEM]));
  hildon_button_set_value_alignment(HILDON_BUTTON (pui->unitsystembutton),0.5,0.5);
  gtk_box_pack_start (GTK_BOX(uvbox), pui->unitsystembutton, TRUE, TRUE, 0);
  gtk_widget_show(pui->unitsystembutton);

  unitcheck = hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label (HILDON_CHECK_BUTTON (unitcheck), gettext(labels[SETTINGS_INDIVIDUAL]));
  g_signal_connect (GTK_OBJECT (unitcheck), "toggled",
		    G_CALLBACK(unit_check_cb), pui);
  gtk_box_pack_start (GTK_BOX(uvbox), unitcheck, TRUE, TRUE, 0);
  gtk_widget_show(unitcheck);

  row=row+2;
  pui->lengthunitbutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->lengthunitbutton), gettext(labels[SETTINGS_LENGTHUNIT]));
  hildon_button_set_value_alignment(HILDON_BUTTON (pui->lengthunitbutton),0.5,0.5);
  gtk_box_pack_start (GTK_BOX(uvbox), pui->lengthunitbutton, TRUE, TRUE, 0);
  if (curunit.individualunit)
    gtk_widget_show(pui->lengthunitbutton);

  pui->lengthcombo = create_unit_combo(LENGTHCOMBO, pui, pui->lengthunitbutton);
  g_signal_connect(G_OBJECT(pui->lengthunitbutton), "value-changed",
		   G_CALLBACK(l_unit_changed_cb), pui);

  row++;
  pui->volumeunitbutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->volumeunitbutton), gettext(labels[SETTINGS_VOLUMEUNIT]));
  hildon_button_set_value_alignment(HILDON_BUTTON (pui->volumeunitbutton),0.5,0.5);
  gtk_box_pack_start (GTK_BOX(uvbox), pui->volumeunitbutton, TRUE, TRUE, 0);
  if (curunit.individualunit)
    gtk_widget_show(pui->volumeunitbutton);

  pui->volumecombo = create_unit_combo(VOLUMECOMBO, pui, pui->volumeunitbutton);
  g_signal_connect(G_OBJECT(pui->volumeunitbutton), "value-changed",
		   G_CALLBACK(v_unit_changed_cb), pui);

  row++;
  pui->consumeunitbutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->consumeunitbutton), gettext(labels[SETTINGS_CONSUMEUNIT]));
  hildon_button_set_value_alignment(HILDON_BUTTON (pui->consumeunitbutton),0.5,0.5);
  gtk_box_pack_start (GTK_BOX(uvbox), pui->consumeunitbutton, TRUE, TRUE, 0);
  if (curunit.individualunit)
    gtk_widget_show(pui->consumeunitbutton);

  pui->consumecombo = create_unit_combo(CONSUMECOMBO, pui, pui->consumeunitbutton);
  g_signal_connect(G_OBJECT(pui->consumeunitbutton), "value-changed",
		   G_CALLBACK(c_unit_changed_cb), pui);

  row++;
  pui->massunitbutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->massunitbutton), gettext(labels[SETTINGS_MASSUNIT]));
  hildon_button_set_value_alignment(HILDON_BUTTON (pui->massunitbutton),0.5,0.5);
  gtk_box_pack_start (GTK_BOX(uvbox), pui->massunitbutton, TRUE, TRUE, 0);
  if (curunit.individualunit)
    gtk_widget_show(pui->massunitbutton);

  pui->masscombo = create_unit_combo(MASSCOMBO, pui, pui->massunitbutton);
  g_signal_connect(G_OBJECT(pui->massunitbutton), "value-changed",
		   G_CALLBACK(m_unit_changed_cb), pui);

  row=row-5;
  /* Must be the last combo since the toggle callback uses the other combos */
  pui->unitcombo = create_unit_combo(UNITCOMBO, pui, pui->unitsystembutton);
  g_signal_connect(G_OBJECT(pui->unitsystembutton), "value-changed",
		   G_CALLBACK(unit_changed_cb), pui);

  row++;
  /* Must be the last since the toggle callback uses the combos */
  g_signal_connect (GTK_OBJECT (unitcheck), "toggled",
		    G_CALLBACK(unit_check_cb), pui);
  hildon_check_button_set_active(HILDON_CHECK_BUTTON (unitcheck), curunit.individualunit);
  unit_check_cb (unitcheck, pui);

  row=row+5;

  gtk_widget_show(uvbox);

  frame=gtk_frame_new(gettext(headers[SETTINGS_HDR_REPORT]));
  gtk_box_pack_start (GTK_BOX(svbox), frame, TRUE, TRUE, 5);
  gtk_widget_show(frame);

  /* First row */
  pui->paperbutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->paperbutton), gettext(labels[SETTINGS_PAPER]));
  gtk_container_add (GTK_CONTAINER(frame), pui->paperbutton);
  gtk_widget_show(pui->paperbutton);

  pui->papercombo = create_paper_combo(pui);

  frame=gtk_frame_new(gettext(headers[SETTINGS_HDR_DATABASE]));
  gtk_box_pack_start (GTK_BOX(svbox), frame, TRUE, TRUE, 5);
  gtk_widget_show(frame);

  hbox=gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start (GTK_BOX(svbox), hbox, TRUE, TRUE, 0);

  pui->databasebutton = hildon_button_new_with_text (HILDON_SIZE_AUTO, 
						     HILDON_BUTTON_ARRANGEMENT_VERTICAL,
						     gettext(labels[SETTINGS_DATABASE]),
						     db_getfilename());
/*   gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1,  */
/* 		   GTK_EXPAND|GTK_FILL, */
/* 		   0, 0, 5); */
  gtk_box_pack_start (GTK_BOX(hbox), pui->databasebutton, TRUE, TRUE, 0);
  gtk_widget_show(pui->databasebutton);

  gtk_widget_show(hbox);

  /* Temporarily disabled */
/*   button = hildon_gtk_button_new(HILDON_SIZE_FINGER_HEIGHT); */
/*   gtk_button_set_label(GTK_BUTTON(button), gettext(labels[SETTINGS_OPEN])); */
/*   gtk_box_pack_start (GTK_BOX(svbox), button, TRUE, TRUE, 0); */
/*   g_signal_connect(G_OBJECT(button), "clicked", */
/* 		   G_CALLBACK(callback_open_db), pui); */
/*   gtk_widget_show(button); */

  hbox=gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start (GTK_BOX(svbox), hbox, TRUE, TRUE, 0);

  button = hildon_gtk_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label(GTK_BUTTON(button), gettext(labels[SETTINGS_IMPORT]));
  gtk_box_pack_start (GTK_BOX(hbox), button, TRUE, TRUE, 0);
  g_signal_connect(G_OBJECT(button), "clicked",
		   G_CALLBACK(callback_import), pui);
  gtk_widget_show(button);

  button = hildon_gtk_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label(GTK_BUTTON(button), gettext(labels[SETTINGS_EXPORT]));
  gtk_box_pack_start (GTK_BOX(hbox), button, TRUE, TRUE, 0);
  g_signal_connect(GTK_OBJECT(button), "clicked",
		   G_CALLBACK(callback_export), pui);
  gtk_widget_show(button);

  gtk_widget_show(hbox);

/*   gtk_box_pack_start (GTK_BOX(scrollwin), svbox, TRUE, TRUE, 0); */
  gtk_widget_show(svbox);
  gtk_box_pack_start (GTK_BOX(vbox), scrollwin, TRUE, TRUE, 0);
  gtk_widget_show(scrollwin);
  hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA(scrollwin),svbox);

#if LIBHELP == 1
  help_dialog_help_enable(GTK_DIALOG(dialog),
				 HELP_ID_SETTINGS,
				 pui->app->osso);
#endif

#else // MAEMO_VERSION_MAJOR < 5
  /** Driver and car notebook page **/

  table = gtk_table_new(6, 4, FALSE);
/*     gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, FALSE, 5); */

  label=gtk_label_new(gettext(labels[SETTINGS_DATABASE]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  label=gtk_entry_new();
  gtk_entry_set_text(GTK_ENTRY(label), db_getfilename());
  gtk_editable_set_editable(GTK_EDITABLE(label), FALSE);
  gtk_table_attach(GTK_TABLE(table), label, 1, 3, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  row++;
  /* First row */
  label=gtk_label_new(gettext(labels[SETTINGS_DRIVER]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->drivercombo = create_driver_combo(pui);
  gtk_table_attach(GTK_TABLE(table), pui->drivercombo, 1, 2, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->drivercombo);

  button = gtk_button_new_with_label(gettext(labels[SETTINGS_EDIT]));
  gtk_widget_set_sensitive(button, db_open());
  gtk_table_attach(GTK_TABLE(table), button, 2, 3, row, row+1,
		   GTK_EXPAND|GTK_FILL, 
		   0, 0, 5);
  g_signal_connect (GTK_OBJECT (button), "clicked",
		    G_CALLBACK(edit_driver_cb), &ui);
  gtk_widget_show(button);

  row++;
  button = gtk_button_new_with_label(gettext(labels[SETTINGS_NEW]));
  gtk_widget_set_sensitive(button, db_open());
  gtk_table_attach(GTK_TABLE(table), button, 1, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL, 
		   0, 0, 5);
  g_signal_connect (GTK_OBJECT (button), "clicked",
		    G_CALLBACK(add_driver_cb), &ui);
  gtk_widget_show(button);

  button = gtk_button_new_with_label(gettext(labels[SETTINGS_DELETE]));
  gtk_widget_set_sensitive(button, db_open());
  gtk_table_attach(GTK_TABLE(table), button, 2, 3, row, row+1,
		   GTK_EXPAND|GTK_FILL, 
		   0, 0, 5);
  g_signal_connect (GTK_OBJECT (button), "clicked",
		    G_CALLBACK(delete_driver_cb), &ui);
  gtk_widget_show(button);

  row++;
  /* Second row */
  label=gtk_label_new(gettext(labels[SETTINGS_CAR]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->carcombo = create_car_combo(pui);
  gtk_table_attach(GTK_TABLE(table), pui->carcombo, 1, 2, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);

  gtk_widget_show(pui->carcombo);

  button = gtk_button_new_with_label(gettext(labels[SETTINGS_EDIT]));
  gtk_widget_set_sensitive(button, db_open());
  gtk_table_attach(GTK_TABLE(table), button, 2, 3, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  g_signal_connect (GTK_OBJECT (button), "clicked",
		    G_CALLBACK(edit_car_cb), pui);
  gtk_widget_show(button);

  row++;
  button = gtk_button_new_with_label(gettext(labels[SETTINGS_NEW]));
  gtk_widget_set_sensitive(button, db_open());
  gtk_table_attach(GTK_TABLE(table), button, 1, 2, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  g_signal_connect (GTK_OBJECT (button), "clicked",
		    G_CALLBACK(add_car_cb), pui);
  gtk_widget_show(button);

  button = gtk_button_new_with_label(gettext(labels[SETTINGS_DELETE]));
  gtk_widget_set_sensitive(button, db_open());
  gtk_table_attach(GTK_TABLE(table), button, 2, 3, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  g_signal_connect (GTK_OBJECT (button), "clicked",
		    G_CALLBACK(delete_car_cb), pui);
  gtk_widget_show(button);

  row++;
  /* Third row */
  label=gtk_label_new(gettext(labels[SETTINGS_DATEFORMAT]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->datecombo = create_dateformat_combo(pui);
  gtk_table_attach(GTK_TABLE(table), pui->datecombo, 1, 2, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->datecombo);

  gtk_widget_show(table);

  label = gtk_label_new (gettext(headers[SETTINGS_HDR_DATABASE]));
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), table, label);

  /** Reminders notebook page **/

  table = gtk_table_new(2, 3, FALSE);
  row=0;

  /* First row */
  label=gtk_label_new(gettext(labels[SETTINGS_WARN]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entrywarnkm=gtk_entry_new();
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrywarnkm), MAXWARNLEN);
  update_warn_entry(pui->entrywarnkm, warnkm);
  gtk_table_attach(GTK_TABLE(table), pui->entrywarnkm, 1, 2, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrywarnkm);

  pui->labelentrywarn=gtk_label_new(NULL);
  update_warn_label(pui->labelentrywarn);
  gtk_table_attach(GTK_TABLE(table), pui->labelentrywarn, 2, 3, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->labelentrywarn);

  row++;
  /* Second row */
  label=gtk_label_new(gettext(labels[SETTINGS_REMIND]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->warntimecombo=create_warntime_combo(pui);
  gtk_table_attach(GTK_TABLE(table), pui->warntimecombo, 1, 3, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->warntimecombo);

/*   label=gtk_label_new(_("before")); */
/*   gtk_table_attach(GTK_TABLE(table), label, 2, 3, row, row+1,  */
/* 		   GTK_EXPAND|GTK_FILL, */
/* 		   0, 0, 5); */
/*   gtk_widget_show(label); */
    
  gtk_widget_show(table);

  label = gtk_label_new (gettext(headers[SETTINGS_HDR_REMINDERS]));
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), table, label);

  /** Units notebook page **/

  unitscrollwin = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(unitscrollwin), 
				 GTK_POLICY_NEVER, 
				 GTK_POLICY_AUTOMATIC);
  table = gtk_table_new(7, 3, FALSE);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(unitscrollwin),table);

  row=0;

  /* First row */
  label=gtk_label_new(gettext(labels[SETTINGS_CURRENCY]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entrycurrency = gtk_entry_new();
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrycurrency), 30);
  gtk_entry_set_text(GTK_ENTRY(pui->entrycurrency),curunit.currency->str);
  gtk_table_attach(GTK_TABLE(table), pui->entrycurrency, 1, 2, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrycurrency);

  row++;
  label=gtk_label_new(gettext(labels[SETTINGS_UNITSYSTEM]));
  /*     gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2); */
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  row=row+2;
  label=gtk_label_new(gettext(labels[SETTINGS_LENGTHUNIT]));
  /*     gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2); */
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->lengthcombo = create_unit_combo(LENGTHCOMBO, pui);
  /*     gtk_table_attach_defaults(GTK_TABLE(table), pui->drivercombo, 1, 2, 0, 1); */
  gtk_table_attach(GTK_TABLE(table), pui->lengthcombo, 1, 2, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_set_sensitive(pui->lengthcombo, FALSE);
  gtk_widget_show(pui->lengthcombo);

  row++;
  /*     gtk_table_attach_defaults(GTK_TABLE(table), pui->drivercombo, 1, 2, 0, 1); */
  gtk_table_attach(GTK_TABLE(table), pui->volumecombo, 1, 2, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_set_sensitive(pui->volumecombo, FALSE);
  gtk_widget_show(pui->volumecombo);

  row++;
  label=gtk_label_new(gettext(labels[SETTINGS_CONSUMEUNIT]));
  /*     gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2); */
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->consumecombo = create_unit_combo(CONSUMECOMBO, pui);
  /*     gtk_table_attach_defaults(GTK_TABLE(table), pui->drivercombo, 1, 2, 0, 1); */
  gtk_table_attach(GTK_TABLE(table), pui->consumecombo, 1, 2, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_set_sensitive(pui->consumecombo, FALSE);
  gtk_widget_show(pui->consumecombo);

  row++;
  label=gtk_label_new(gettext(labels[SETTINGS_MASSUNIT]));
  /*     gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2); */
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->masscombo = create_unit_combo(MASSCOMBO, pui);
  /*     gtk_table_attach_defaults(GTK_TABLE(table), pui->drivercombo, 1, 2, 0, 1); */
  gtk_table_attach(GTK_TABLE(table), pui->masscombo, 1, 2, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_set_sensitive(pui->masscombo, FALSE);
  gtk_widget_show(pui->masscombo);

  row=row-5;
  /* Must be the last combo since the toggle callback uses the other combos */
  pui->unitcombo = create_unit_combo(UNITCOMBO, pui);
  /*     gtk_table_attach_defaults(GTK_TABLE(table), pui->drivercombo, 1, 2, 0, 1); */
  gtk_table_attach(GTK_TABLE(table), pui->unitcombo, 1, 2, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->unitcombo);

  row++;
  /* Must be the last since the toggle callback uses the combos */
  unitcheck = gtk_check_button_new_with_label(gettext(labels[SETTINGS_INDIVIDUAL]));
  g_signal_connect (GTK_OBJECT (unitcheck), "toggled",
		    G_CALLBACK(unit_check_cb), pui);
  /*     gtk_table_attach_defaults(GTK_TABLE(table), unitcheck, 0, 1, 1, 2); */
  gtk_table_attach(GTK_TABLE(table), unitcheck, 1, 2, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(unitcheck);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (unitcheck), curunit.individualunit);

  gtk_widget_show(table);
  gtk_widget_show(unitscrollwin);

  label = gtk_label_new (gettext(headers[SETTINGS_HDR_UNITS]));
  /*     gtk_notebook_append_page (GTK_NOTEBOOK (notebook), table, label); */
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), unitscrollwin, label);

  /** Reports notebook page **/

  table = gtk_table_new(1, 2, FALSE);
  row=0;

  /* First row */
  label=gtk_label_new(gettext(labels[SETTINGS_PAPER]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->papercombo = create_paper_combo(pui);
  gtk_table_attach(GTK_TABLE(table), pui->papercombo, 1, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->papercombo);

  gtk_widget_show(table);

  label = gtk_label_new (gettext(headers[SETTINGS_HDR_REPORT]));
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), table, label);

  /* Notebooks done */


#if LIBHELP == 1
  help_dialog_help_enable(GTK_DIALOG(dialog),
				 HELP_ID_SETTINGS,
				 pui->app->osso);
#endif
#endif

/*   g_signal_connect_swapped (dialog, "response", */
/* 			    G_CALLBACK (settings_response), */
/* 			    dialog); */
  g_signal_connect (dialog, "response",
		    G_CALLBACK (settings_response),
		    pui);

  gtk_widget_show(dialog);
}

static void
cell_data_func (GtkTreeViewColumn *col,
		      GtkCellRenderer   *renderer,
		      GtkTreeModel      *model,
		      GtkTreeIter       *iter,
		      gpointer           user_data)
{
  double  value;
  gchar   buf[20];

  gtk_tree_model_get(model, iter, ((data_func_t *)user_data)->col, &value, -1);

  switch (((data_func_t *)user_data)->col) {
  case COL_CONSUM:
  case COL_CO2EMISSION:
    if (abs(value)<1e-5)
      g_snprintf(buf, sizeof(buf), "%s", "-");
    else
      g_snprintf(buf, sizeof(buf), ((data_func_t *)user_data)->format, value);
    break;
  default:
    g_snprintf(buf, sizeof(buf), ((data_func_t *)user_data)->format, value);
  }

  g_object_set(renderer, "text", buf, NULL);
}

void sort_column_changed(GtkTreeSortable *sortable,
			  gpointer         pui)
{
  gint colid;
  GtkSortType sortorder;

  if (gtk_tree_sortable_get_sort_column_id(sortable, &colid, &sortorder)) {
    maintablesorted=TRUE;
    maintablesortorder=sortorder;
    maintablesortcol=colid;
  }
  else {
    maintablesorted=FALSE;
  }
}

gint
sort_iter_compare_func (GtkTreeModel *model,
                        GtkTreeIter  *a,
                        GtkTreeIter  *b,
                        gpointer      userdata)
{
  gint sortcol = GPOINTER_TO_INT(userdata); /* not used yet */
  gint ret = 0;

  switch (sortcol)
    {
    case COL_KM:
    case COL_TRIP:
    case COL_FILL:
    case COL_CONSUM:
    case COL_PRICE:
    case COL_PRICEPERTRIP:
    case COL_PRICEPERLITRE:
    case COL_SERVICE:
    case COL_OIL:
    case COL_TIRES: {
      
      double result1, result2;

      gtk_tree_model_get(model, a, sortcol, &result1, -1);
      gtk_tree_model_get(model, b, sortcol, &result2, -1);
    
      if (result1 != result2)
	{
	  if (!isnan(result1) && !isnan(result2))
	    ret = (result1 > result2) ? 1 : -1;
	  else {
	    if (isnan(result1) && !isnan(result2)) {
		  ret = -1;
	    }
	    else if (!isnan(result1) && isnan(result2)) {
	      ret = 1;
	    }
	  }
	}
      /* else both equal => ret = 0 */
    }
      break;
    case COL_DAY: {
      char *date1, *date2;
      gchar *userdate1, *userdate2;

      gtk_tree_model_get(model, a, sortcol, &userdate1, -1);
      gtk_tree_model_get(model, b, sortcol, &userdate2, -1);

      date2sqlite(&date1, datefmtstr[curunit.dateformat], userdate1);
      date2sqlite(&date2, datefmtstr[curunit.dateformat], userdate2);

      ret=strcmp(date1,date2);

      free(date1);
      free(date2);
      g_free(userdate1);
      g_free(userdate2);

    }
      break;
    case COL_NOTES:{

        gchar *name1, *name2;
	
        gtk_tree_model_get(model, a, sortcol, &name1, -1);
        gtk_tree_model_get(model, b, sortcol, &name2, -1);
	
        if (name1 == NULL || name2 == NULL)
	  {
	    if (name1 == NULL && name2 == NULL)
	      break; /* both equal => ret = 0 */
	    
	    ret = (name1 == NULL) ? -1 : 1;
	  }
        else
	  {
	    ret = g_utf8_collate(name1,name2);
	  }
	
        g_free(name1);
        g_free(name2);
      }
      break;
    }

  return ret;
}

void update_record_model(AppUIData *pui)
{
  GtkTreeModel        *model;

  ui_show_banner(_("Updating, please wait"));
  /* Update the UI */
  while (gtk_events_pending())
    gtk_main_iteration();

  carchanged=FALSE;

  model = create_and_fill_model();
  gtk_tree_view_set_model(GTK_TREE_VIEW(pui->view),model);

  update_column_headers(pui->view);
  update_totalkm(pui);
}

GtkTreeModel *get_filter(GtkWidget *view)
{
  GtkTreeModel *sortable;
  GtkTreeModel *filter;

  sortable=gtk_tree_view_get_model(GTK_TREE_VIEW(view));
  if (sortable != NULL)
    filter=gtk_tree_model_sort_get_model(GTK_TREE_MODEL_SORT(sortable));
  else
    filter=NULL;

  return filter;
}

/**
 * \fn void get_store_and_iter(GtkTreeModel *model,
			GtkWidget *view,
			GtkTreeIter *iter,
			GtkListStore **store,
			GtkTreeIter *storeiter)
 * \brief Obtains the list store and iter of a view
 * \param model if model available or NULL
 * \param view the outermost tree view
 * \param iter if iter available or NULL
 * \param store pointer to a list store pointer
 * \param storeiter pointer to a list iter pointer (or NULL)
 *
 * This functions obtains the list store embedded inside a
 * GtkTreeModelSort and a GtkTreeModelFilter. It also converts the
 * iter of the outermost model to the iter of the innermost iter
 * corresponding to the list store.
 *
 */
void get_store_and_iter(GtkTreeModel *model,
			GtkWidget *view,
			GtkTreeIter *iter,
			GtkListStore **store,
			GtkTreeIter *storeiter)
{
  GtkTreeModel *sortable;
  GtkTreeModel *filter;
  GtkTreeIter filteriter;

  sortable=gtk_tree_view_get_model(GTK_TREE_VIEW(view));
  filter=gtk_tree_model_sort_get_model(GTK_TREE_MODEL_SORT(sortable));
  *store = GTK_LIST_STORE(gtk_tree_model_filter_get_model(GTK_TREE_MODEL_FILTER(filter)));
  if (model==NULL || iter==NULL) 
    storeiter=NULL;
  else
  {
    gtk_tree_model_sort_convert_iter_to_child_iter(GTK_TREE_MODEL_SORT(model), &filteriter, iter);
    gtk_tree_model_filter_convert_iter_to_child_iter(GTK_TREE_MODEL_FILTER(filter), storeiter, &filteriter);
  }
}

gboolean ui_find_iter(GtkListStore *store,
		      GtkTreeIter *iter,
		      sqlite_int64 id)
{
  sqlite_int64 curid;
  gboolean ret;

  ret=FALSE;
  if (gtk_tree_model_get_iter_first(GTK_TREE_MODEL(store), iter)) {
    gtk_tree_model_get(GTK_TREE_MODEL(store), iter, COL_ID, &curid, -1);
    while (iter != NULL && curid != id) {
      gtk_tree_model_iter_next(GTK_TREE_MODEL(store), iter);
      gtk_tree_model_get(GTK_TREE_MODEL(store), iter, COL_ID, &curid, -1);
    }
    if (curid==id)
      ret=TRUE;
  }
  return ret;
}
		     


void ui_update_row_data(GtkListStore  *store,
			GtkTreeIter   *iter,
			const char *date,
			double km,
			double trip,
			double fill,
			double consum,
			double price,
			double priceperlitre,
			double service,
			double oil,
			double tires,
			const char *notes,
			sqlite_int64 id,
			gboolean visible)
{
  char *userdate;

  if (date != NULL) {
    convdate(&userdate, datefmtstr[curunit.dateformat], "%Y-%m-%d", date);
    gtk_list_store_set (store, iter, COL_DAY,  userdate, -1);
    free(userdate);
  }

  if (!(km < 0.0)) gtk_list_store_set (store, iter, COL_KM, SIlength2user(km), -1);
  if (!(trip < 0.0)) gtk_list_store_set (store, iter, COL_TRIP, SIlength2user(trip), -1);
  if (!(fill < 0.0)) gtk_list_store_set (store, iter, COL_FILL, SIvolume2user(fill), -1);
  if (!(consum < 0.0)) gtk_list_store_set (store, iter, COL_CONSUM, doubleornothing(SIconsumption2user(consum)), -1);
  if (!(price < 0.0)) gtk_list_store_set (store, iter, COL_PRICE, price, -1);
  if (!(price < 0.0 || trip < 0.0)) gtk_list_store_set (store, iter, COL_PRICEPERTRIP, doubleornothing(price/SIlength2user(trip)), -1);
  if (!(priceperlitre < 0.0)) gtk_list_store_set (store, iter, COL_PRICEPERLITRE, doubleornothing(SIppl2user(priceperlitre)), -1);
  if (!(service < 0.0)) gtk_list_store_set (store, iter, COL_SERVICE, service, -1);
  if (!(oil < 0.0)) gtk_list_store_set (store, iter, COL_OIL, oil, -1);
  if (!(tires < 0.0)) gtk_list_store_set (store, iter, COL_TIRES, tires, -1);
				/* 			COL_INSURANCE, sqlite3_column_double(ppStmtRecords,5), */
				/* 			COL_OTHER, sqlite3_column_double(ppStmtRecords,5), */
  if (!(consum < 0.0)) gtk_list_store_set (store, iter, COL_CO2EMISSION, SIemission2user(calc_co2_emission(consum,currentcar)), -1);
  if (notes != NULL) gtk_list_store_set (store, iter, COL_NOTES, notes, -1);
  gtk_list_store_set (store, iter, COL_ID, id, COL_VISIBLE, visible, -1);
}

static GtkTreeModel *
create_and_fill_model (void)
{
  GtkListStore  *store;
  GtkTreeIter    iter;
  GtkTreeModel        *filter;
  GtkTreeSortable     *sortable;
  char *date;
  char notes[ENTRYNOTESMAX];
  gchar *convnotes;
  double fill;
  double trip;
  double price;
  double consum;

  if (db_open()) {
    store = gtk_list_store_new (NUM_COLS, G_TYPE_STRING, G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_DOUBLE, G_TYPE_STRING, G_TYPE_INT64, G_TYPE_BOOLEAN);

    if (SQLITE_OK == sqlite3_bind_int(ppStmtRecords, 1, currentcar)) {
      while(SQLITE_ROW == sqlite3_step(ppStmtRecords)) {

	convdate(&date, datefmtstr[curunit.dateformat], "%Y-%m-%d", sqlite3_column_text(ppStmtRecords,0));

	/* Convert notes to utf8 (earlier versions might have imported wrongly encoded data */
	g_snprintf(notes, ENTRYNOTESMAX, "%s", sqlite3_column_text(ppStmtRecords,10));
	if (!g_utf8_validate(notes, -1, NULL))
	  convnotes=g_convert(notes, -1, "UTF-8", "ISO-8859-15", NULL, NULL, NULL);
	else
	  convnotes=g_strndup(notes, strlen(notes));
	trip=sqlite3_column_double(ppStmtRecords,2);
	fill=sqlite3_column_double(ppStmtRecords,3);
	price=sqlite3_column_double(ppStmtRecords,5);
	consum=sqlite3_column_double(ppStmtRecords,4);
	gtk_list_store_append (store, &iter);
	gtk_list_store_set (store, &iter,
			    COL_DAY, date,
			    COL_KM, SIlength2user(sqlite3_column_double(ppStmtRecords,1)),
			    COL_TRIP, SIlength2user(trip),
			    COL_FILL, SIvolume2user(fill),
			    COL_CONSUM, doubleornothing(SIconsumption2user(consum)),
			    COL_PRICE, price,
			    COL_PRICEPERTRIP, doubleornothing(price/SIlength2user(trip)),
			    COL_PRICEPERLITRE, doubleornothing(SIppl2user(sqlite3_column_double(ppStmtRecords,6))),
			    COL_SERVICE, sqlite3_column_double(ppStmtRecords,7),
			    COL_OIL, sqlite3_column_double(ppStmtRecords,8),
			    COL_TIRES, sqlite3_column_double(ppStmtRecords,9),
			    /* 			COL_INSURANCE, sqlite3_column_double(ppStmtRecords,5), */
			    /* 			COL_OTHER, sqlite3_column_double(ppStmtRecords,5), */
			    COL_CO2EMISSION, SIemission2user(calc_co2_emission(consum,currentcar)),
			    COL_NOTES, convnotes,
			    COL_ID, sqlite3_column_int64(ppStmtRecords,11),
			    COL_VISIBLE, TRUE,
			    -1);
	g_free(convnotes);
    
	free(date);

      }
    }

    sqlite3_reset(ppStmtRecords);

    filter = gtk_tree_model_filter_new(GTK_TREE_MODEL(store), NULL);
/*     gtk_tree_model_filter_set_visible_column(GTK_TREE_MODEL_FILTER(filter), */
/* 					     COL_VISIBLE); */

    gtk_tree_model_filter_set_visible_func(GTK_TREE_MODEL_FILTER(filter),
					   visible_func,
                                           (gpointer)&filterqueue,
					   NULL);
    sortable = GTK_TREE_SORTABLE(gtk_tree_model_sort_new_with_model(filter));

    gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(sortable), COL_DAY, 
				    sort_iter_compare_func,
				    GINT_TO_POINTER(COL_DAY), NULL);

    gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(sortable), COL_TRIP,
				    sort_iter_compare_func,
				    GINT_TO_POINTER(COL_TRIP), NULL);

    gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(sortable), COL_CONSUM,
				    sort_iter_compare_func,
				    GINT_TO_POINTER(COL_CONSUM), NULL);

    gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(sortable), COL_PRICEPERTRIP,
				    sort_iter_compare_func,
				    GINT_TO_POINTER(COL_PRICEPERTRIP), NULL);

    gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(sortable), COL_PRICEPERLITRE,
				    sort_iter_compare_func,
				    GINT_TO_POINTER(COL_PRICEPERLITRE), NULL);

    /* set initial sort order */
/*     gtk_tree_sortable_set_sort_column_id(sortable, COL_DAY, GTK_SORT_ASCENDING); */
    gtk_tree_sortable_set_sort_column_id(sortable, maintablesortcol, maintablesortorder);

    g_signal_connect(sortable, "sort-column-changed",
		     G_CALLBACK(sort_column_changed), &ui);

    return GTK_TREE_MODEL (sortable);
  }
  else
    return NULL;
}

static gboolean
visible_func (GtkTreeModel *model,
              GtkTreeIter  *iter,
              gpointer      data)
{
  gchar *str;
  char *date;
  gdouble colval;
  gboolean visible = FALSE;
  gboolean compres;
  GList *filt;
  GList *lp;
  recordfilter_t *p;
  gboolean first;
  gboolean and;

  filt=*(GList **)data;

  gtk_tree_model_get (model, iter, COL_DAY, &str, -1);

  if (str) {
    visible = TRUE;

    if (filter_filterqueue_length() > 0) {

      lp=g_list_first(filt);
      first=TRUE;
      while (lp != NULL) {
	p=(recordfilter_t *)lp->data;
	and=TRUE;
	compres=TRUE;
	if (p->column==COL_DAY) {
	  date2sqlite(&date, datefmtstr[curunit.dateformat], str);
	  compres = strcmp(p->mindate->str, date) <= 0 && strcmp(p->maxdate->str, date) >= 0;
	  and=p->and;
	  free(date);
	  g_free(str);
	}
	else if (p->column==COL_NOTES) {
	  gtk_tree_model_get (model, iter, COL_NOTES, &str, -1);
	  if (str) {
	    compres = g_pattern_match_string(p->pattern, str);
	    and=p->and;
	    g_free(str);
	  }
	}
	else {
	  gtk_tree_model_get (model, iter, p->column, &colval, -1);
	  compres = colval >= p->minval && colval <= p->maxval;
	  and=p->and;
	}

	if (!first) {
	  if (and)
	    visible &= compres;
	  else
	    visible |= compres;
	}
	else {
	  first=FALSE;
	  visible = compres;
	}

	lp=g_list_next(lp);
      }

    }
  }

  return visible;
}

static GtkWidget *
create_view_and_model ()
{
  GtkWidget *view;
/*   static data_func_t dfdata1; */
  static data_func_t dfdata2;
  static data_func_t dfdata3;
  static data_func_t dfdata4;
  static data_func_t dfdata5;
  static data_func_t dfdata6;
  static data_func_t dfdata7;
  static data_func_t dfdata8;
  static data_func_t dfdata9;
  static data_func_t dfdata10;
  static data_func_t dfdata11;
  static data_func_t dfdata12;

  GtkTreeViewColumn   *col;
  GtkCellRenderer     *renderer;
  GtkTreeModel        *model;
  GtkTreeSelection    *select;

  GtkWidget           *label;
  PangoAttrList       *attrs;
  PangoAttribute      *attr;

  /* Scale texts smaller: create pango attributes */
  attrs = pango_attr_list_new();
  attr = pango_attr_scale_new(fontscalefactors[fontsize]);
  attr->start_index = 0;
  attr->end_index = -1;
  pango_attr_list_insert(attrs, attr);

  model = create_and_fill_model ();
#if MAEMO_VERSION_MAJOR == 5
  view = hildon_gtk_tree_view_new_with_model (HILDON_UI_MODE_EDIT, 
					      model);
#else
  view = gtk_tree_view_new_with_model (model);
#endif

  /* --- Column #1 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_DAY]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_DAY);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

/*   dfdata1.col = COL_DAY; */
/*   strncpy(dfdata1.format,"%.0f",DFFORMATLEN); */
/*   gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata1, NULL); */

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_DAY);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_DAY), dispcol & (1<<COL_DAY));

  /* --- Column #2 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_KM]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_KM);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  dfdata2.col = COL_KM;
  strncpy(dfdata2.format,"%.0f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata2, NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_KM);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_KM), dispcol & (1<<COL_KM));

  /* --- Column #3 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_TRIP]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_TRIP);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  dfdata3.col = COL_TRIP;
  strncpy(dfdata3.format,"%.1f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata3, NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_TRIP);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_TRIP), dispcol & (1<<COL_TRIP));

  /* --- Column #4 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_FILL]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_FILL);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_FILL);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_FILL), dispcol & (1<<COL_FILL));

  dfdata4.col = COL_FILL;
  strncpy(dfdata4.format,"%.2f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata4, NULL);

  /* --- Column #5 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_CONSUM]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_CONSUM);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_CONSUM);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_CONSUM), dispcol & (1<<COL_CONSUM));

  dfdata5.col = COL_CONSUM;
  strncpy(dfdata5.format,"%.1f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata5, NULL);


  /* --- Column #6 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_PRICE]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_PRICE);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_PRICE);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_PRICE), dispcol & (1<<COL_PRICE));

  dfdata6.col = COL_PRICE;
  strncpy(dfdata6.format,"%.2f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata6, NULL);

  /* --- Column #7 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_PRICEPERTRIP]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_PRICEPERTRIP);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_PRICEPERTRIP);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_PRICEPERTRIP), dispcol & (1<<COL_PRICEPERTRIP));

  dfdata7.col = COL_PRICEPERTRIP;
  strncpy(dfdata7.format,"%.2f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata7, NULL);

  /* --- Column #8 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_PRICEPERLITRE]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_PRICEPERLITRE);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_PRICEPERLITRE);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_PRICEPERLITRE), dispcol & (1<<COL_PRICEPERLITRE));

  dfdata8.col = COL_PRICEPERLITRE;
  strncpy(dfdata8.format,"%.2f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata8, NULL);

  /* --- Column #9 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_SERVICE]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_SERVICE);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_SERVICE);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_SERVICE), dispcol & (1<<COL_SERVICE));

  dfdata9.col = COL_SERVICE;
  strncpy(dfdata9.format,"%.2f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata9, NULL);

  /* --- Column #10 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_OIL]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_OIL);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_OIL);

  dfdata10.col = COL_OIL;
  strncpy(dfdata10.format,"%.2f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata10, NULL);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_OIL), dispcol & (1<<COL_OIL));

  /* --- Column #11 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_TIRES]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_TIRES);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_TIRES);

  dfdata11.col = COL_TIRES;
  strncpy(dfdata11.format,"%.2f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata11, NULL);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_TIRES), dispcol & (1<<COL_TIRES));

  /* --- Column #12 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_CO2EMISSION]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_CO2EMISSION);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_CO2EMISSION);

  dfdata12.col = COL_CO2EMISSION;
  strncpy(dfdata12.format,"%.0f",DFFORMATLEN);
  gtk_tree_view_column_set_cell_data_func(col, renderer, cell_data_func, &dfdata12, NULL);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_CO2EMISSION), dispcol & (1<<COL_CO2EMISSION));

  /* --- Column #13 --- */

  col = gtk_tree_view_column_new();

  label=gtk_label_new(gettext(column_headers[COL_NOTES]));
  gtk_tree_view_column_set_widget(col, GTK_WIDGET(label));
  gtk_label_set_attributes(GTK_LABEL(label), attrs);
  gtk_widget_show(label);

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_NOTES);
  gtk_tree_view_column_set_resizable(col, TRUE);
  g_object_set(renderer, "scale", fontscalefactors[fontsize], NULL);

  gtk_tree_view_column_set_sort_column_id(GTK_TREE_VIEW_COLUMN(col), COL_NOTES);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_NOTES), dispcol & (1<<COL_NOTES));

  /* Invisible column to distinguish the results */
  /* --- Column #last --- */

  col = gtk_tree_view_column_new();

  gtk_tree_view_column_set_title(col, "Id");

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_ID);

/*   renderer = gtk_cell_renderer_text_new (); */
/*   gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (view), */
/*                                                -1,       */
/*                                                "Id",   */
/*                                                renderer, */
/*                                                "text", COL_ID, */
/*                                                NULL); */
  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_ID), FALSE);

  /* Invisible visibility column */
  /* --- Column #last+1 --- */

  col = gtk_tree_view_column_new();

  gtk_tree_view_column_set_title(col, "Visible");

  /* pack tree view column into tree view */
  gtk_tree_view_append_column(GTK_TREE_VIEW(view), col);

  renderer = gtk_cell_renderer_text_new();

  /* pack cell renderer into tree view column */
  gtk_tree_view_column_pack_start(col, renderer, TRUE);

  gtk_tree_view_column_add_attribute(col, renderer, "text", COL_VISIBLE);

  gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (view),COL_VISIBLE), FALSE);

/*   model = create_and_fill_model (); */

  /* Change the unit dependent headers */
  update_column_headers(view);

/*   gtk_tree_view_set_model (GTK_TREE_VIEW (view), model); */
  gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(view), TRUE);

  /* The tree view has acquired its own reference to the
   *  model, so we can drop ours. That way the model will
   *  be freed automatically when the tree view is destroyed */

  g_object_unref (model);

  select = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));
  gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);
  g_signal_connect(view, "row-activated",
		   G_CALLBACK (callback_recordactivated),
		   &ui);

  return view;
}

void select_columns_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  gint i;
  gboolean checked;

  for (i=0;i<(NUM_COLS-2);i++) {
#if MAEMO_VERSION_MAJOR == 5
    checked = hildon_check_button_get_active (HILDON_CHECK_BUTTON (pui->columncheck[i]));
#else
    checked = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (pui->columncheck[i]));
#endif
    if (checked)
      dispcol |= (1<<i);
    else
      dispcol &= ~(1<<i);
    gtk_tree_view_column_set_visible (gtk_tree_view_get_column(GTK_TREE_VIEW (pui->view),i), dispcol & (1<<i));
  }


  /* Note! Was this really needed in Diablo? */
#if MAEMO_VERSION_MAJOR == 5
#else
  gtk_widget_destroy(GTK_WIDGET(pui->mainvbox));
  create_mainwin_widgets(pui);
#endif

  gtk_widget_destroy(GTK_WIDGET(dialog));
}


#if MAEMO_VERSION_MAJOR == 5
#define SELCOLLINES (float)(COL_ID)
#else
#define SELCOLLINES 4.0
#endif
void callback_select_columns( GtkAction * action, AppUIData *pui )
{
#if MAEMO_VERSION_MAJOR == 5
  GtkWidget *scrollwin;
  GRegex *regex;
  gchar *pattern="\\n";
  gchar *newstr;
#endif
  GString *header;
  GtkWidget *dialog;
  GtkWidget *vbox;
  GtkWidget *table;
  gint i;
  gint j;
  gint col;
  gfloat numcols;

  dialog = gtk_dialog_new_with_buttons(_("Select columns"),
				       GTK_WINDOW(pui->app->mainwindow),
				       GTK_DIALOG_MODAL,
				       GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
				       GTK_STOCK_CANCEL,
				       GTK_RESPONSE_REJECT,
				       NULL);

  vbox = GTK_WIDGET(GTK_DIALOG(dialog)->vbox);

  numcols=ceilf(COL_ID/SELCOLLINES);

#if MAEMO_VERSION_MAJOR == 5
  scrollwin = hildon_pannable_area_new();
  gtk_widget_set_size_request(scrollwin, -1, DIALOG_MIN_HEIGHTMAX);
  gtk_box_pack_start (GTK_BOX (vbox), scrollwin, TRUE, TRUE, 0);
#endif
  table = gtk_table_new(numcols, COL_ID/numcols, FALSE);
#if MAEMO_VERSION_MAJOR == 5
  hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA(scrollwin),table);
#else
  gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, FALSE, 5);
#endif

#if MAEMO_VERSION_MAJOR == 5
  regex = g_regex_new(pattern, G_REGEX_DOTALL, 0, NULL);
#endif

  for (col=0,j=0,i=0;i<COL_ID;i++,j++) {
    if (j==(int)SELCOLLINES) {
      j=0;
      col++;
    }
    header = get_column_header(i);
#if MAEMO_VERSION_MAJOR == 5
    pui->columncheck[i]=hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
    newstr = g_regex_replace(regex, header->str, -1, 1, " ", 0, NULL);
    gtk_button_set_label (GTK_BUTTON(pui->columncheck[i]), newstr);
    hildon_check_button_set_active(HILDON_CHECK_BUTTON (pui->columncheck[i]), dispcol & (1<<i));
#else
    pui->columncheck[i]=gtk_check_button_new_with_label (header->str);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (pui->columncheck[i]), dispcol & (1<<i));
#endif
    g_string_free(header, TRUE);
    gtk_table_attach(GTK_TABLE(table), pui->columncheck[i], col, col+1, j, j+1, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show (pui->columncheck[i]);

#if MAEMO_VERSION_MAJOR == 5
    g_free(newstr);
#endif

  }

#if MAEMO_VERSION_MAJOR == 5
    g_regex_unref(regex);
#endif

  gtk_widget_show(table);
#if MAEMO_VERSION_MAJOR == 5
  gtk_widget_show(scrollwin);
#endif

#if LIBHELP == 1
    help_dialog_help_enable(GTK_DIALOG(dialog),
				   HELP_ID_MAINWIN,
				   pui->app->osso);
#endif

  g_signal_connect (dialog, "response",
		    G_CALLBACK (select_columns_response),
		    pui);
  gtk_widget_show(dialog);
  
}

/**
 * \fn GString *get_filter_string(GList *flist)
 * \brief Returns the current filter as a string
 * \param *flist pointer to the filter list
 * \return Pointer to a newly allocated filter string
 *
 * This functions returns the filter list as a filter string. The
 * string is allocated from the heap. After usage, the string must be
 * freed with g_string_free.
 *
 */
GString *get_filter_string(GList *flist)
{
  GString *filtstring;
  GList *lp;
  recordfilter_t *p;
  gboolean first;

  filtstring=g_string_new(NULL);
  lp=g_list_first(flist);
  first=TRUE;
  while (lp != NULL) {
    p=(recordfilter_t *)lp->data;
    if (!first) {
      if (p->and) 
	g_string_append_printf(filtstring, " AND "); 
      else
	g_string_append_printf(filtstring, " OR ");
    }
    else
      first=FALSE;
    if (p->column==COL_DAY)
      g_string_append_printf(filtstring, "(%s < Date < %s)",p->mindate->str, p->maxdate->str);
    else if (p->column==COL_NOTES)
      g_string_append_printf(filtstring,"(Notes==%s)",p->notes->str);
    else
      g_string_append_printf(filtstring,"(%.2f < Column %d < %.2f)",p->minval, p->column, p->maxval);
    lp=g_list_next(lp);
  };

  return (filtstring);

}

void add_filter_cb (GtkWidget *widget, gpointer data )
{
  gint active;
  gint andactive;
  recordfilter_t *p;
  AppUIData *pui;
  char *mindate;
  char *maxdate;
#if PLAINGTK == 0
  guint year;
  guint month;
  guint day;
#endif
  GString *filtstring;
  GString *s;

  pui=&ui;

#if MAEMO_VERSION_MAJOR == 5
  active=hildon_touch_selector_get_active(HILDON_TOUCH_SELECTOR(pui->filtercombo), 0);
#else
  active=gtk_combo_box_get_active(GTK_COMBO_BOX(pui->filtercombo));
#endif

  p=(recordfilter_t *)malloc(sizeof(recordfilter_t));
  filter_init(p);
  draw_filter_selector(pui, pui->filtertable, active, filter_filterqueue_length(), p);

  if (filterqueue != NULL) {
#if MAEMO_VERSION_MAJOR == 5
    andactive = hildon_touch_selector_get_active (HILDON_TOUCH_SELECTOR (pui->filteraddandorcombo), 0);
    p->and= andactive == 0;
    hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR (p->entry.andorcombo), 0, andactive);
#else
    andactive = gtk_combo_box_get_active (GTK_COMBO_BOX (pui->filteraddandorcombo));
    p->and= andactive == 0;
    gtk_combo_box_set_active(GTK_COMBO_BOX (p->entry.andorcombo), andactive);
#endif
  }
  p->column=active;
  filterqueue=g_list_append(filterqueue, p);
  if (active==COL_DAY) {

      /* Get the minimum date */
#if PLAINGTK == 0
    /* Missing: reading gtk_entry and converting to year, month, day */
#if MAEMO_VERSION_MAJOR == 5
      hildon_date_button_get_date(HILDON_DATE_BUTTON(pui->filteraddminentry), &year, &month, &day); /* Month is betweewn 0 and 11 */
      month++; 
      hildon_date_button_set_date(HILDON_DATE_BUTTON(p->entry.minentry), year, month-1, day);
#else
      hildon_date_editor_get_date(HILDON_DATE_EDITOR(pui->filteraddminentry), &year, &month, &day);
      hildon_date_editor_set_date(HILDON_DATE_EDITOR(p->entry.minentry), year, month, day);
#endif
      mindate=(char *)malloc(ENTRYDATEMAX*sizeof(char));
      g_snprintf(mindate, ENTRYDATEMAX, "%d-%02d-%02d",year, month, day);
#else
      date2sqlite(&mindate, datefmtstr[curunit.dateformat], 
		      gtk_entry_get_text(GTK_ENTRY(pui->filteraddminentry)));
#endif
      g_string_assign(p->mindate, mindate);
      free(mindate);

      /* Get the maximum date */
#if PLAINGTK == 0
    /* Missing: reading gtk_entry and converting to year, month, day */
#if MAEMO_VERSION_MAJOR == 5
      hildon_date_button_get_date(HILDON_DATE_BUTTON(pui->filteraddmaxentry), &year, &month, &day); /* Month is betweewn 0 and 11 */
      month++; 
      hildon_date_button_set_date(HILDON_DATE_BUTTON(p->entry.maxentry), year, month-1, day);
#else
      hildon_date_editor_get_date(HILDON_DATE_EDITOR(pui->filteraddmaxentry), &year, &month, &day);
      hildon_date_editor_set_date(HILDON_DATE_EDITOR(p->entry.maxentry), year, month, day);
#endif
      maxdate=(char *)malloc(ENTRYDATEMAX*sizeof(char));
      g_snprintf(maxdate, ENTRYDATEMAX, "%d-%02d-%02d",year, month, day);
#else
      date2sqlite(&maxdate, datefmtstr[curunit.dateformat], 
		      gtk_entry_get_text(GTK_ENTRY(pui->filteraddmaxentry)));
#endif
      g_string_assign(p->maxdate, maxdate);
      free(maxdate);

  }
  else if (active==COL_NOTES) {
    /* Get the notes filter */
    g_string_assign(p->notes, gtk_entry_get_text(GTK_ENTRY(pui->filteraddminentry)));
    gtk_entry_set_text(GTK_ENTRY(p->entry.minentry),p->notes->str);
    g_pattern_spec_free(p->pattern);
    p->pattern=g_pattern_spec_new(p->notes->str);
  }
  else {
    /* Get the other fields */
    s=g_string_new("");
    g_string_assign(s, gtk_entry_get_text(GTK_ENTRY(pui->filteraddminentry)));
    gtk_entry_set_text(GTK_ENTRY(p->entry.minentry),s->str);
    p->minval=atof(s->str);

    g_string_assign(s, gtk_entry_get_text(GTK_ENTRY(pui->filteraddmaxentry)));
    gtk_entry_set_text(GTK_ENTRY(p->entry.maxentry),s->str);
    p->maxval=atof(s->str);
    g_string_free(s, TRUE);
  }

#if MAEMO_VERSION_MAJOR == 5
  /* This is a work around since the apply button has some issues */
  gtk_tree_model_filter_refilter(GTK_TREE_MODEL_FILTER(get_filter(pui->view)));
#endif
}

void filter_update(recordfilter_t *filt)
{
  char *date;

  db_get_min_date(&date);
  if (date != NULL) {
    g_string_append_printf(filt->mindate,"%s",date);
    free(date);
  }
  db_get_max_date(&date);
  if (date != NULL) {
    g_string_append_printf(filt->maxdate,"%s",date);
    free(date);
  }
}

void filter_init(recordfilter_t *filt)
{
  filt->mindate=g_string_new("");
  filt->maxdate=g_string_new("");
  filt->notes=g_string_new("*");
  filt->pattern=g_pattern_spec_new(filt->notes->str);
  filt->column=0;

  filter_update(filt);

  filt->and = FALSE;

}

void filter_clear(recordfilter_t *filt)
{
  g_string_free(filt->mindate, TRUE);
  g_string_free(filt->maxdate, TRUE);
  g_string_free(filt->notes, TRUE);
  g_pattern_spec_free(filt->pattern);
}

void filter_free (gpointer data, gpointer user_data)
{
  recordfilter_t *p;

  p=(recordfilter_t *)data;
  filter_clear(p);
  free(p);
}

void filter_filterqueue_empty(GList **list)
{
  g_list_foreach(*list, filter_free, NULL);
  g_list_free(*list);
  *list=NULL;
}

guint filter_filterqueue_length(void)
{
  return g_list_length(filterqueue);
}

void callback_clear_filter( GtkAction * action, AppUIData *pui )
{
#if MAEMO_VERSION_MAJOR == 5
  GtkTextBuffer *textbuffer;
#endif
  filter_filterqueue_empty(&filterqueue);
/* #if MAEMO_VERSION_MAJOR == 5 */
/*   textbuffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(pui->filterentry)); */
/*   gtk_text_buffer_set_text(textbuffer, "", -1); */
/* #else */
/*   gtk_entry_set_text(GTK_ENTRY(pui->filterentry), ""); */
/* #endif */
#if MAEMO_VERSION_MAJOR == 5
  /* This is a work around since the apply button has some issues */
  gtk_tree_model_filter_refilter(GTK_TREE_MODEL_FILTER(get_filter(pui->view)));
#endif
}

void filter_records_update(void)
{
  gint active;
  gint andactive;
  GList *lp;
  recordfilter_t *p;
  char *mindate;
  char *maxdate;
#if PLAINGTK == 0
  guint year;
  guint month;
  guint day;
#endif
  GString *s;

  lp=g_list_first(filterqueue);
  while (lp != NULL) {
    p=(recordfilter_t *)lp->data;

    /* Mandatory or optional */
#if MAEMO_VERSION_MAJOR == 5
    andactive = hildon_touch_selector_get_active (HILDON_TOUCH_SELECTOR (p->entry.andorcombo), 0);
#else
    andactive = gtk_combo_box_get_active (GTK_COMBO_BOX (p->entry.andorcombo));
#endif
    p->and= andactive == 0;

    if (p->column==COL_DAY) {
      /* Get the minimum date */
#if PLAINGTK == 0
    /* Missing: reading gtk_entry and converting to year, month, day */
#if MAEMO_VERSION_MAJOR == 5
      hildon_date_button_get_date(HILDON_DATE_BUTTON(p->entry.minentry), &year, &month, &day); /* Month is betweewn 0 and 11 */
      month++; 
#else
      hildon_date_editor_get_date(HILDON_DATE_EDITOR(p->entry.minentry), &year, &month, &day);
#endif
      mindate=(char *)malloc(ENTRYDATEMAX*sizeof(char));
      g_snprintf(mindate, ENTRYDATEMAX, "%d-%02d-%02d",year, month, day);
#else
      date2sqlite(&mindate, datefmtstr[curunit.dateformat], 
		  gtk_entry_get_text(GTK_ENTRY(p->entry.minentry)));
#endif
      g_string_assign(p->mindate, mindate);
      free(mindate);

      /* Get the maximum date */
#if PLAINGTK == 0
    /* Missing: reading gtk_entry and converting to year, month, day */
#if MAEMO_VERSION_MAJOR == 5
      hildon_date_button_get_date(HILDON_DATE_BUTTON(p->entry.maxentry), &year, &month, &day); /* Month is betweewn 0 and 11 */
      month++; 
#else
      hildon_date_editor_get_date(HILDON_DATE_EDITOR(p->entry.maxentry), &year, &month, &day);
#endif
      maxdate=(char *)malloc(ENTRYDATEMAX*sizeof(char));
      g_snprintf(maxdate, ENTRYDATEMAX, "%d-%02d-%02d",year, month, day);
#else
      date2sqlite(&maxdate, datefmtstr[curunit.dateformat], 
		  gtk_entry_get_text(GTK_ENTRY(p->entry.maxentry)));
#endif
      g_string_assign(p->maxdate, maxdate);
      free(maxdate);
    }
    else if (active==COL_NOTES) {
      /* Get the notes filter */
      g_string_assign(p->notes, gtk_entry_get_text(GTK_ENTRY(p->entry.minentry)));
      g_pattern_spec_free(p->pattern);
      p->pattern=g_pattern_spec_new(p->notes->str);
    }
    else {
      /* Get the other fields */
      s=g_string_new("");
      g_string_assign(s, gtk_entry_get_text(GTK_ENTRY(p->entry.minentry)));
      p->minval=atof(s->str);

      g_string_assign(s, gtk_entry_get_text(GTK_ENTRY(p->entry.maxentry)));
      p->maxval=atof(s->str);
      g_string_free(s, TRUE);
    }

    lp=g_list_next(lp);
  };
}

void filter_records_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  filter_records_update();

  gtk_tree_model_filter_refilter(GTK_TREE_MODEL_FILTER(get_filter(pui->view)));

  gtk_widget_destroy(GTK_WIDGET(pui->filterwindow));
}

void remove_filter_callback( GtkAction * action, recordfilter_t *p )
{
  gtk_widget_destroy(p->entry.removebutton);
#if MAEMO_VERSION_MAJOR == 5
  gtk_widget_destroy(p->entry.andorbutton);
#else
  gtk_widget_destroy(p->entry.andorcombo);
  gtk_widget_destroy(p->entry.label);
#endif
  gtk_widget_destroy(p->entry.minentry);
  gtk_widget_destroy(p->entry.maxentry);
  filterqueue = g_list_remove(filterqueue, p);
  filter_clear(p);
  free(p);

  /* The following is propably not necessary */
  if (filter_filterqueue_length()==0) {
    filter_filterqueue_empty(&filterqueue);
  }
}

void filter_andor_changed_cb( GtkAction * action, recordfilter_t *p )
{
  gint andactive;

#if MAEMO_VERSION_MAJOR == 5
  HildonTouchSelector *selector;
  selector = hildon_picker_button_get_selector(HILDON_PICKER_BUTTON(p->entry.andorbutton));
  andactive = hildon_touch_selector_get_active (HILDON_TOUCH_SELECTOR (selector), 0);
#else
  andactive = gtk_combo_box_get_active (GTK_COMBO_BOX (p->entry.andorcombo));
#endif
  p->and=andactive == 0;
}

/**
 * \fn GString* get_filter_entry_userunit(gdouble value, gint col)
 * \brief Returns the value as a string in user unit
 * \param value the value
 * \param col the column number
 * \return Pointer to a newly allocated string
 *
 * This functions returns the given value in the appropriate user
 * unit. The string is allocated from the heap. After usage, the
 * string must be freed with g_string_free.
 *
 */
GString* get_filter_entry_userunit(gdouble value, gint col)
{
  GString *text;

  text=g_string_new(NULL);

  switch (col) {
  case COL_PRICE:
  case COL_SERVICE:
  case COL_OIL:
  case COL_TIRES:
    g_string_printf(text, "%.2f",value);
    break;
  case COL_KM:
  case COL_TRIP:
    g_string_printf(text, "%.2f",SIlength2user(value));
    break;
  case COL_PRICEPERTRIP:
    g_string_printf(text, "%.2f",value/SIlength2user(1.0));
    break;
  case COL_PRICEPERLITRE:
    g_string_printf(text, "%.2f",SIppl2user(value));
    break;
  case COL_FILL:
    g_string_printf(text, "%.2f",SIvolume2user(value));
    break;
  case COL_CONSUM:
    g_string_printf(text, "%.2f",SIconsumption2user(value));
    break;
  case COL_CO2EMISSION:
    g_string_printf(text, "%.2f",SIemission2user(value));
    break;
  }
  
  return text;
}

void fill_filter_entrys(AppUIData *pui, recordfilter_t *p)
{
  gint andactive;
  guint year;
  guint month;
  guint day;
  GString *s1;
  GString *s2;

  if (p->and)
    andactive = 0;
  else
    andactive = 1;
#if MAEMO_VERSION_MAJOR == 5
    hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR (p->entry.andorcombo), 0, andactive);
#else
    gtk_combo_box_set_active(GTK_COMBO_BOX (p->entry.andorcombo), andactive);
#endif

    if (p->column == COL_DAY) {
      getyearmonthday(p->mindate->str,
		      &year,
		      &month,
		      &day);
#if MAEMO_VERSION_MAJOR == 5
      hildon_date_button_set_date(HILDON_DATE_BUTTON(p->entry.minentry), year, month-1, day);
#else
      hildon_date_editor_set_date(HILDON_DATE_EDITOR(p->entry.minentry), year, month, day);
#endif

      getyearmonthday(p->maxdate->str,
		      &year,
		      &month,
		      &day);
#if MAEMO_VERSION_MAJOR == 5
      hildon_date_button_set_date(HILDON_DATE_BUTTON(p->entry.maxentry), year, month-1, day);
#else
      hildon_date_editor_set_date(HILDON_DATE_EDITOR(p->entry.maxentry), year, month, day);
#endif
    }
    else if (p->column == COL_NOTES) {
      gtk_entry_set_text(GTK_ENTRY(p->entry.minentry),p->notes->str);
    }
    else { /* Normal fields */
      s1=g_string_new("");
      s2=g_string_new("");
      g_string_printf(s1, "%.2f",p->minval);
      g_string_printf(s2, "%.2f",p->maxval);
      gtk_entry_set_text(GTK_ENTRY(p->entry.minentry),s1->str);
      gtk_entry_set_text(GTK_ENTRY(p->entry.maxentry),s2->str);
      g_string_free(s2, TRUE);
      g_string_free(s1, TRUE);
    }

}

void draw_filter_selector(AppUIData *pui, GtkWidget *table, gint i, guint nele, recordfilter_t *p)
{
#if MAEMO_VERSION_MAJOR == 5
  GRegex *regex;
  gchar *pattern="\\n";
#else
  GtkWidget *label;
#endif
  gchar *newstr;
  GString *header;

#if MAEMO_VERSION_MAJOR == 5
  regex = g_regex_new(pattern, G_REGEX_DOTALL, 0, NULL);
#endif

    header = get_column_header(i);
#if MAEMO_VERSION_MAJOR == 5
    newstr = g_regex_replace(regex, header->str, -1, 1, " ", 0, NULL);
#else
    newstr = gettext(header->str);
#endif

#if MAEMO_VERSION_MAJOR == 5
    p->entry.removebutton=hildon_button_new_with_text(HILDON_SIZE_AUTO,
						  HILDON_BUTTON_ARRANGEMENT_VERTICAL,
						  _("Remove"),
						  newstr);
#else
    p->entry.removebutton=gtk_button_new_from_stock (GTK_STOCK_REMOVE);
#endif
    gtk_table_attach(GTK_TABLE(table), p->entry.removebutton, 0, 1, nele, nele+1,
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    g_signal_connect (p->entry.removebutton, "clicked",
		      G_CALLBACK (remove_filter_callback),
		      p);
    gtk_widget_show (p->entry.removebutton);

#if MAEMO_VERSION_MAJOR == 5
#else
    p->entry.label = gtk_label_new(newstr);
    gtk_table_attach(GTK_TABLE(table), p->entry.label, 1, 2, nele, nele+1,
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show (p->entry.label);
#endif

    g_string_free(header, TRUE);
#if MAEMO_VERSION_MAJOR == 5
    g_free(newstr);
#endif

#if MAEMO_VERSION_MAJOR == 5
    p->entry.andorbutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
    hildon_button_set_title (HILDON_BUTTON (p->entry.andorbutton), _("Necessity"));
    gtk_table_attach(GTK_TABLE(table), p->entry.andorbutton, 2, 3, nele, nele+1,
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show (p->entry.andorbutton);
    p->entry.andorcombo=create_andor_combo (p->entry.andorbutton);
/*     g_signal_connect (GTK_OBJECT (p->entry.andorbutton), "value-changed", */
/* 		      G_CALLBACK(filter_andor_changed_cb), p); */
#else
    p->entry.andorcombo=create_andor_combo ();
/*     g_signal_connect (GTK_OBJECT (p->entry.andorcombo), "changed", */
/* 		      G_CALLBACK(filter_andor_changed_cb), p); */
    gtk_table_attach(GTK_TABLE(table), p->entry.andorcombo, 2, 3, nele, nele+1,
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show (p->entry.andorcombo);
#endif

    if (i==COL_DAY) {
#if PLAINGTK == 1
	  p->entry.minentry=gtk_entry_new();
#else
#if MAEMO_VERSION_MAJOR == 5
	  p->entry.minentry = hildon_date_button_new(HILDON_SIZE_FINGER_HEIGHT,
						     HILDON_BUTTON_ARRANGEMENT_VERTICAL);
#else
	  p->entry.minentry = hildon_date_editor_new();
#endif
#endif
    }
    else if (i!=COL_NOTES) {
#if MAEMO_VERSION_MAJOR == 5
      p->entry.minentry=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
#else
      p->entry.minentry=gtk_entry_new();
#endif
    }
    else {
#if MAEMO_VERSION_MAJOR == 5
      p->entry.minentry=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
#else
      p->entry.minentry=gtk_entry_new();
#endif
    }
    gtk_table_attach(GTK_TABLE(table), p->entry.minentry, 3, 4, nele, nele+1,
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(p->entry.minentry);

    if (i!=COL_NOTES) {
      if (i==COL_DAY) {
#if PLAINGTK == 1

#if MAEMO_VERSION_MAJOR == 5
	  p->entry.maxentry=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
#else
	  p->entry.maxentry=gtk_entry_new();
#endif
#else
#if MAEMO_VERSION_MAJOR == 5
	  p->entry.maxentry = hildon_date_button_new(HILDON_SIZE_FINGER_HEIGHT,
						     HILDON_BUTTON_ARRANGEMENT_VERTICAL);
#else
	  p->entry.maxentry = hildon_date_editor_new();
#endif
#endif

      }
      else {
#if MAEMO_VERSION_MAJOR == 5
	p->entry.maxentry=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
#else
	p->entry.maxentry=gtk_entry_new();
#endif
      }
      gtk_table_attach(GTK_TABLE(table), p->entry.maxentry, 4, 5, nele, nele+1,
		       GTK_EXPAND|GTK_FILL,
		       0, 0, 5);
      gtk_widget_show(p->entry.maxentry);
    }

#if MAEMO_VERSION_MAJOR == 5
    g_regex_unref(regex);
#endif

}

GtkWidget *create_filter_combo(AppUIData *pui)
{
  GtkWidget *combo;
  gint i;
#if MAEMO_VERSION_MAJOR == 5
  GRegex *regex;
  gchar *pattern="\\n";
#endif
  gchar *newstr;
  GString *header;

#if MAEMO_VERSION_MAJOR == 5
  combo = hildon_touch_selector_new_text ();
#else
  combo = gtk_combo_box_new_text();
#endif

#if MAEMO_VERSION_MAJOR == 5
  regex = g_regex_new(pattern, G_REGEX_DOTALL, 0, NULL);
#endif

  for (i=0;i<COL_ID;i++) {

    header = get_column_header(i);
#if MAEMO_VERSION_MAJOR == 5
    newstr = g_regex_replace(regex, header->str, -1, 1, " ", 0, NULL);
#else
    newstr = gettext(header->str);
#endif

#if MAEMO_VERSION_MAJOR == 5
    hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR (combo), newstr);
#else
    gtk_combo_box_append_text(GTK_COMBO_BOX (combo), newstr);
#endif
    g_string_free(header, TRUE);

#if MAEMO_VERSION_MAJOR == 5
    g_free(newstr);
#endif
  }

#if MAEMO_VERSION_MAJOR == 5
  hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR (combo), 0, 0);
#else
  gtk_combo_box_set_active(GTK_COMBO_BOX (combo), 0);
#endif

#if MAEMO_VERSION_MAJOR == 5
  g_regex_unref(regex);
#endif

  return combo;

}

void fill_filter_minmax_entrys(AppUIData *pui, gint active, GtkWidget *minentry, GtkWidget *maxentry)
{
  GString *s;
  GString *value;
  char *newdate;
#if PLAINGTK == 0
  guint year;
  guint month;
  guint day;
#else
  char *date;
#endif
  int fueltype;
  gfloat emisperlitre;
  gfloat minval;
  gfloat maxval;
  gfloat temp;

  if (active==COL_CO2EMISSION) {
    fueltype = db_get_fueltype(currentcar);
    emisperlitre = get_emission_per_litre(fueltype);
  }

  s=g_string_new("");
  if (active==COL_DAY) {
    db_get_min_date(&newdate);
#if PLAINGTK == 1
    convdate(&date, datefmtstr[curunit.dateformat], "%Y-%m-%d", newdate);

    gtk_entry_set_text(GTK_ENTRY(minentry),date);
    free(date);
#else
    getyearmonthday(newdate, 
		    &year,
		    &month,
		    &day);
#if MAEMO_VERSION_MAJOR == 5
    hildon_date_button_set_date(HILDON_DATE_BUTTON(minentry), year, month-1, day); /* Month is betweewn 0 and 11 */
#else
    hildon_date_editor_set_date(HILDON_DATE_EDITOR(minentry), year, month, day);
#endif
#endif
    free(newdate);

      db_get_max_date(&newdate);
#if PLAINGTK == 1
      convdate(&date, datefmtstr[curunit.dateformat], "%Y-%m-%d", newdate);
      gtk_entry_set_text(GTK_ENTRY(maxentry),date);
      free(date);
#else
      getyearmonthday(newdate, 
		      &year,
		      &month,
		      &day);
#if MAEMO_VERSION_MAJOR == 5
      hildon_date_button_set_date(HILDON_DATE_BUTTON(maxentry), year, month-1, day); /* Month is betweewn 0 and 11 */
#else
      hildon_date_editor_set_date(HILDON_DATE_EDITOR(maxentry), year, month, day);
#endif
#endif
      free(newdate);

  }
  else if (active!=COL_NOTES) {
    minval=db_get_min(active);
    if (active==COL_CO2EMISSION)
      minval=minval*emisperlitre;
    maxval=db_get_max(active);
    if (active==COL_CO2EMISSION)
      maxval=maxval*emisperlitre;

    if (active==COL_CONSUM && (curunit.consumeunit==US || curunit.consumeunit==IMPERIAL)) {
      temp=maxval;
      maxval=minval;
      minval=temp;
    }

    value=get_filter_entry_userunit(minval, active);
    gtk_entry_set_text(GTK_ENTRY(minentry),value->str);
    g_string_free(value, TRUE);

    value=get_filter_entry_userunit(maxval, active);
    gtk_entry_set_text(GTK_ENTRY(maxentry),value->str);
    g_string_free(value, TRUE);
  }
  else {
    gtk_entry_set_text(GTK_ENTRY(minentry),recfilter.notes->str);
  }
  g_string_free(s, TRUE);
  
}

void show_filter_minmax_entrys(AppUIData *pui, GtkWidget *vbox)
{
  GtkWidget *hbox;
  GtkWidget *label;
  gint active;

#if MAEMO_VERSION_MAJOR == 5
  active=hildon_touch_selector_get_active(HILDON_TOUCH_SELECTOR(pui->filtercombo), 0);
#else
  active=gtk_combo_box_get_active(GTK_COMBO_BOX(pui->filtercombo));
#endif

#if MAEMO_VERSION_MAJOR == 5
  pui->filteraddandorbutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->filteraddandorbutton), _("Necessity"));
  gtk_box_pack_start (GTK_BOX(vbox), pui->filteraddandorbutton, TRUE, TRUE, 0);
  gtk_widget_show (pui->filteraddandorbutton);
  pui->filteraddandorcombo=create_andor_combo (pui->filteraddandorbutton);
#else
  pui->filteraddandorcombo=create_andor_combo ();
  gtk_box_pack_start (GTK_BOX(vbox), pui->filteraddandorcombo, TRUE, TRUE, 0);
  gtk_widget_show (pui->filteraddandorcombo);
#endif

  hbox=gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start (GTK_BOX(vbox), hbox, TRUE, TRUE, 0);

if (active!=COL_NOTES)
  label=gtk_label_new(_("Minimum value"));
 else
  label=gtk_label_new(_("Search pattern"));

  gtk_box_pack_start (GTK_BOX(hbox), label, TRUE, TRUE, 0);
  gtk_widget_show(label);

  if (active==COL_DAY) {
#if PLAINGTK == 1
    pui->filteraddminentry=gtk_entry_new();
#else
#if MAEMO_VERSION_MAJOR == 5
    pui->filteraddminentry = hildon_date_button_new(HILDON_SIZE_FINGER_HEIGHT,
							  HILDON_BUTTON_ARRANGEMENT_VERTICAL);
#else
    pui->filteraddminentry = hildon_date_editor_new();
#endif
#endif
  }
  else if (active!=COL_NOTES) {
#if MAEMO_VERSION_MAJOR == 5
    pui->filteraddminentry=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
#else
    pui->filteraddminentry=gtk_entry_new();
#endif
  }
  else {
#if MAEMO_VERSION_MAJOR == 5
    pui->filteraddminentry=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
#else
    pui->filteraddminentry=gtk_entry_new();
#endif
  }

  gtk_box_pack_start (GTK_BOX(hbox), pui->filteraddminentry, TRUE, TRUE, 0);
  gtk_widget_show(pui->filteraddminentry);

  gtk_widget_show(hbox);

  hbox=gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start (GTK_BOX(vbox), hbox, TRUE, TRUE, 0);


  if (active!=COL_NOTES) {
    label=gtk_label_new(_("Maximum value"));
    gtk_box_pack_start (GTK_BOX(hbox), label, TRUE, TRUE, 0);
    gtk_widget_show(label);
    if (active==COL_DAY) {
#if PLAINGTK == 1

#if MAEMO_VERSION_MAJOR == 5
      pui->filteraddmaxentry=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
#else
      pui->filteraddmaxentry=gtk_entry_new();
#endif
#else
#if MAEMO_VERSION_MAJOR == 5
      pui->filteraddmaxentry = hildon_date_button_new(HILDON_SIZE_FINGER_HEIGHT,
							  HILDON_BUTTON_ARRANGEMENT_VERTICAL);
#else
      pui->filteraddmaxentry = hildon_date_editor_new();
#endif
#endif

    }
    else {
#if MAEMO_VERSION_MAJOR == 5
      pui->filteraddmaxentry=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
#else
      pui->filteraddmaxentry=gtk_entry_new();
#endif
    }

  }
  else { /* active==COL_NOTES */
    /* Create an empty label to keep widget the packing ok */ 
    label=gtk_label_new("");
    gtk_box_pack_start (GTK_BOX(hbox), label, TRUE, TRUE, 0);
    gtk_widget_show(label);
  }

  gtk_box_pack_start (GTK_BOX(hbox), pui->filteraddmaxentry, TRUE, TRUE, 0);
  gtk_widget_show(pui->filteraddmaxentry);
      
  gtk_widget_show(hbox);

  fill_filter_minmax_entrys(pui, active, pui->filteraddminentry, pui->filteraddmaxentry);
}

void filter_changed_callback( GtkAction * action, AppUIData *pui)
{
  GtkWidget *vbox;

  vbox = GTK_WIDGET(GTK_DIALOG(pui->addfilterdialog)->vbox);

  gtk_widget_destroy(pui->filtervbox);
  pui->filtervbox=gtk_vbox_new(FALSE, 0);
  gtk_box_pack_start (GTK_BOX(vbox), pui->filtervbox, TRUE, TRUE, 0);
  show_filter_minmax_entrys(pui, pui->filtervbox);
  gtk_widget_show(pui->filtervbox);
}

void add_filter_callback( GtkAction * action, AppUIData *pui)
{
  GtkWidget *dialog;
  GtkWidget *vbox;
  GtkWidget *label;
  GtkWidget *hbox;

  pui->addfilterdialog = gtk_dialog_new_with_buttons(_("Add a new filter"),
						     GTK_WINDOW(pui->filterwindow),
						     GTK_DIALOG_MODAL,
						     GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
						     GTK_STOCK_CANCEL,
						     GTK_RESPONSE_REJECT,
						     NULL);

  vbox = GTK_WIDGET(GTK_DIALOG(pui->addfilterdialog)->vbox);

#if MAEMO_VERSION_MAJOR == 5
  pui->filterbutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->filterbutton), _("Filter"));
  gtk_box_pack_start (GTK_BOX(vbox), pui->filterbutton, TRUE, TRUE, 0);
  gtk_widget_show(pui->filterbutton);
#else
  hbox=gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start (GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
  label=gtk_label_new(_("Filter"));
  gtk_box_pack_start (GTK_BOX(hbox), label, TRUE, TRUE, 0);
  gtk_widget_show(label);
#endif

  pui->filtercombo = create_filter_combo(pui);
#if MAEMO_VERSION_MAJOR == 5
  hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (pui->filterbutton),
                                     HILDON_TOUCH_SELECTOR (pui->filtercombo));
#else
  gtk_box_pack_start (GTK_BOX(hbox), pui->filtercombo, TRUE, TRUE, 0);
  gtk_widget_show(pui->filtercombo);
  gtk_widget_show(hbox);
#endif

  pui->filtervbox=gtk_vbox_new(FALSE, 0);
  gtk_box_pack_start (GTK_BOX(vbox), pui->filtervbox, TRUE, TRUE, 0);
  show_filter_minmax_entrys(pui, pui->filtervbox);
  gtk_widget_show(pui->filtervbox);

#if MAEMO_VERSION_MAJOR == 5
  g_signal_connect(G_OBJECT(pui->filterbutton), "value-changed",
		   G_CALLBACK(filter_changed_callback), pui);
#else
  g_signal_connect (GTK_OBJECT (pui->filtercombo), "changed",
		    G_CALLBACK(filter_changed_callback), pui);
#endif

  gtk_widget_show(pui->addfilterdialog);

  if (gtk_dialog_run(GTK_DIALOG(pui->addfilterdialog))  == GTK_RESPONSE_ACCEPT) {
    add_filter_cb(pui->addfilterdialog, 0);
  }
  gtk_widget_destroy(GTK_WIDGET(pui->addfilterdialog));
}

void delete_filter_records( GtkWidget *widget,
			    GdkEvent  *event,
			    AppUIData *pui )
{

  filter_records_update();

#if MAEMO_VERSION_MAJOR == 5
  /* This is a work around since the apply button has some issues */
  gtk_tree_model_filter_refilter(GTK_TREE_MODEL_FILTER(get_filter(pui->view)));
#endif
  gtk_widget_destroy(GTK_WIDGET(widget));

  return FALSE;
}


void callback_filter_records( GtkAction * action, AppUIData *pui )
{
#if MAEMO_VERSION_MAJOR == 5
  GtkTextBuffer *textbuffer;
#endif
  gchar *newstr;
  GtkWidget *vbox;
  GtkWidget *scrollwin;
  GtkWidget *label;
  GtkWidget *image;
  GtkWidget *button;
  gint i;
  GList *lp;
  recordfilter_t *p;

#if MAEMO_VERSION_MAJOR == 5
  pui->filterwindow = hildon_stackable_window_new();
  hildon_program_add_window(pui->app->program, pui->filterwindow);
  g_signal_connect(G_OBJECT(pui->filterwindow), "destroy",
		   G_CALLBACK(gtk_widget_destroy), NULL);
  g_signal_connect(G_OBJECT(pui->filterwindow), "delete_event",
		   G_CALLBACK(delete_filter_records), pui);
  vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (pui->filterwindow), vbox);
#else
  pui->filterwindow = gtk_dialog_new_with_buttons(_("Filter records"),
				       GTK_WINDOW(pui->app->mainwindow),
				       GTK_DIALOG_MODAL,
				       GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
				       GTK_STOCK_CANCEL,
				       GTK_RESPONSE_REJECT,
				       NULL);

  vbox = GTK_WIDGET(GTK_DIALOG(pui->filterwindow)->vbox);
#endif

  label=gtk_label_new(_("Active filters"));
  gtk_box_pack_start (GTK_BOX(vbox), label, FALSE, TRUE, 0);
  gtk_widget_show(label);

#if MAEMO_VERSION_MAJOR == 5
  button = hildon_button_new_with_text(HILDON_SIZE_AUTO,
				       HILDON_BUTTON_ARRANGEMENT_HORIZONTAL,
				       _("Add new"),
				       NULL);
#else
  button = gtk_button_new_with_label(_("Add new"));
#endif
#if MAEMO_VERSION_MAJOR == 5
  image = gtk_image_new_from_icon_name ("general_add", GTK_ICON_SIZE_BUTTON);
  hildon_button_set_image (HILDON_BUTTON (button), image);
  hildon_button_set_image_position (HILDON_BUTTON (button), GTK_POS_LEFT);
#else
  image = gtk_image_new_from_stock (GTK_STOCK_ADD, GTK_ICON_SIZE_BUTTON);
  gtk_button_set_image (GTK_BUTTON (button), image);
  gtk_button_set_image_position (GTK_BUTTON (button), GTK_POS_LEFT);
#endif
  gtk_box_pack_start (GTK_BOX(vbox), button, FALSE, TRUE, 0);
  g_signal_connect (button, "clicked",
		    G_CALLBACK (add_filter_callback),
		    pui);
  gtk_widget_show(button);

#if MAEMO_VERSION_MAJOR == 5
  scrollwin = hildon_pannable_area_new();
  g_object_set(G_OBJECT(scrollwin), "mov-mode", HILDON_MOVEMENT_MODE_BOTH, NULL);
  gtk_widget_set_size_request(scrollwin, DIALOG_MIN_WIDTH1, DIALOG_MIN_HEIGHT0);
#else
  scrollwin=gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_set_size_request(scrollwin, DIALOG_MIN_WIDTH1, DIALOG_MIN_HEIGHT2);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollwin), 
				 GTK_POLICY_AUTOMATIC, 
				 GTK_POLICY_AUTOMATIC);
#endif
/*   gtk_box_pack_start (GTK_BOX(vbox), scrollwin, TRUE, TRUE, 0); */
  gtk_box_pack_start (GTK_BOX(vbox), scrollwin, FALSE, TRUE, 0);

  pui->filtertable = gtk_table_new(5, COL_ID+1, FALSE);
#if MAEMO_VERSION_MAJOR == 5
  hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA(scrollwin),pui->filtertable);
#else
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrollwin),pui->filtertable);
#endif

  i=0;
  lp=g_list_first(filterqueue);
  while (lp != NULL) {
    p=(recordfilter_t *)lp->data;
    draw_filter_selector(pui, pui->filtertable, p->column, i, p);
    fill_filter_entrys(pui, p);
    i++;
    lp=g_list_next(lp);
  };

  gtk_widget_show(pui->filtertable);
  gtk_widget_show(scrollwin);

#if LIBHELP == 1
    help_dialog_help_enable(GTK_DIALOG(pui->filterwindow),
				   HELP_ID_FILTER,
				   pui->app->osso);
#endif

#if MAEMO_VERSION_MAJOR == 5
/*     button = hildon_gtk_button_new(HILDON_SIZE_FINGER_HEIGHT); */
/*     gtk_button_set_label(GTK_BUTTON(button), _("Apply")); */
/*     gtk_box_pack_start (GTK_BOX(vbox), button, TRUE, TRUE, 0); */
/*     g_signal_connect(G_OBJECT(button), "clicked", */
/* 		     G_CALLBACK(filter_records_response), */
/* 		     pui); */
/*     gtk_widget_show(button); */

    gtk_widget_show(vbox);
    gtk_widget_show_all(GTK_WIDGET(pui->filterwindow));
#else
    g_signal_connect (G_OBJECT(pui->filterwindow), "response",
		      G_CALLBACK (filter_records_response),
		      pui);
    gtk_widget_show(pui->filterwindow);
#endif
  
}


void update_totalkm(AppUIData *pui)
{
  GString *str;
  float totalkm;
  float totalfillkm;
  float totalfill;

  str=g_string_new(NULL);

  totalkm=db_get_totalkm(stbstattime);
  totalfillkm=db_get_totalfillkm(stbstattime);
  totalfill=db_get_totalfill(stbstattime);
  g_string_printf(str, "%.0f ", SIlength2user(totalkm));
  if (curunit.lengthunit==SI)
    g_string_append_printf(str, _("km"));
  else
    g_string_append_printf(str, _("miles"));
  gtk_label_set_text(GTK_LABEL(pui->stb_totalkmlabel),str->str);

  if (totalfillkm != 0.0)
    g_string_printf(str, "%.1f ",SIconsumption2user(totalfill/totalfillkm*100));
  else
    g_string_printf(str, "%.1f ",0.0);
  if (curunit.consumeunit==SI)
    g_string_append_printf(str, _("l/100 km"));
  else
    g_string_append_printf(str, _("MPG"));
  gtk_label_set_text(GTK_LABEL(pui->stb_avgconsumlabel),str->str);

  g_string_printf(str, "%.0f", db_get_totalcost());
  g_string_append_printf(str, " %s", curunit.currency->str);
  gtk_label_set_text(GTK_LABEL(pui->stb_totcostlabel),str->str);

  g_string_free(str, TRUE);

}

#if PLAINGTK == 0
static HildonFileSystemModel* get_file_system_model(GtkWidget *ref_widget)
{
 return HILDON_FILE_SYSTEM_MODEL(g_object_new(HILDON_TYPE_FILE_SYSTEM_MODEL,
					      "ref_widget", ref_widget, NULL));
}
#endif

gchar * interface_file_chooser (gchar *header, 
				GtkFileChooserAction action, 
				const gchar *ext,
				gchar *help_id)
{
    GtkWidget *dialog;
    gchar *filename = NULL;
#if PLAINGTK == 0
    HildonFileSystemModel *model = NULL;
#endif

#if PLAINGTK == 1
    dialog = gtk_file_chooser_dialog_new (header,
					  GTK_WINDOW(app.mainwindow),
					  action,
					  GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					  action == GTK_FILE_CHOOSER_ACTION_SAVE ? GTK_STOCK_SAVE : GTK_STOCK_OPEN, 
					  GTK_RESPONSE_OK,
					  NULL);
#else
    if( (model = get_file_system_model(GTK_WIDGET(app.mainwindow)) ) == NULL)
    {
       PDEBUG("could not get file system model\n\n");
       return NULL;
    }
    dialog = hildon_file_chooser_dialog_new_with_properties
                    (GTK_WINDOW(app.mainwindow),
		             "file_system_model", model,
		             "action", action,
	         	     NULL);
#if HILDON == 1
    hildon_file_chooser_dialog_set_extension(HILDON_FILE_CHOOSER_DIALOG(dialog), (gchar *)ext);
#endif
#endif
#if LIBHELP == 1
    help_dialog_help_enable(GTK_DIALOG(dialog),
				   help_id,
				   app.osso);
#endif
    gtk_widget_show_all (GTK_WIDGET (dialog));

    if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK) {
        filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
    }
    gtk_widget_destroy(dialog);
    return filename;
}

void internal_close_db(AppUIData *pui)
{
  GtkListStore *store;

  db_close();
  store = (GtkListStore *)gtk_tree_view_get_model(GTK_TREE_VIEW(pui->view));
  gtk_list_store_clear(store);
}

void callback_close_db( GtkAction * action, AppUIData *pui )
{
  internal_close_db(pui);
  enable_mainmenu_items(pui);
}

void callback_open_db( GtkAction * action, AppUIData *pui )
{
  gchar *filename = NULL;

  filename = interface_file_chooser ("Open database",
				     GTK_FILE_CHOOSER_ACTION_OPEN,
				     "db",
				     HELP_ID_DATABASE);

  if (filename == NULL) {
    filename = "NULL";
  }
  else {
    internal_close_db(pui);
    db_setfilename(filename);
    db_connect();
/*     update_record_model(); */
    update_car_changed(pui);
    enable_mainmenu_items(pui);
  }
}

GString *get_entry(GtkWidget *entry)
{
  GString *text;

  text=g_string_new(NULL);
  g_string_assign(text,gtk_entry_get_text(GTK_ENTRY(entry)));

  return text;
}

/**
 * \fn void add_record_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
 * \brief Response callback for add record dialog
 *
 * This callback adds the data entered in the add record dialog to the
 * database.
 *
 * \todo This should be changed so that fuelpad.c has a function
 * edit_driver which is called with the data (i.e. sqlite would not be
 * used here)
 *
 */
void add_record_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  GtkListStore  *store;
  GtkTreeIter    storeiter;
#if MAEMO_VERSION_MAJOR == 5
  GtkTreePath *path;
  GdkRectangle *rect;
  gint y;
#else
  GtkAdjustment *vertical;
#endif

  GString *str;
  GString *notes;
  char *date;
  char *userdate;
  gdouble km;
  gdouble trip;
  gdouble fill;
  gdouble price;
  gdouble consum;
  gdouble service;
  gdouble oil;
  gdouble tires;
#if PLAINGTK == 0
  guint year;
  guint month;
  guint day;
#endif
  sqlite_int64 recordid;
  sqlite_int64 fullid;
  gdouble fullconsum;
  gdouble fullfill;
  gdouble fullkm;

  if (!db_open()) {
       gtk_widget_destroy(GTK_WIDGET(dialog));
  }
  else {
    switch (arg1){
    case GTK_RESPONSE_ACCEPT: 
      if (carchanged)
	update_car_changed(pui);

#if PLAINGTK == 0
#if MAEMO_VERSION_MAJOR == 5
      hildon_date_button_get_date(HILDON_DATE_BUTTON(pui->entrydate), &year, &month, &day);  /* Month is betweewn 0 and 11 */
      month++; 
#else
      hildon_date_editor_get_date(HILDON_DATE_EDITOR(pui->entrydate), &year, &month, &day);
#endif
      date=(char *)malloc(ENTRYDATEMAX*sizeof(char));
      g_snprintf(date, ENTRYDATEMAX, "%d-%02d-%02d",year, month, day);
#else
      date2sqlite(&date, datefmtstr[curunit.dateformat], 
		  gtk_entry_get_text(GTK_ENTRY(pui->entrydate)));
#endif

/*       km = user2SIlength(atof(gtk_entry_get_text(GTK_ENTRY(pui->entrykm)))); */
      str = get_entry(pui->entrykm);
      km  = user2SIlength(atof(str->str));
      g_string_free(str, TRUE);

/*       trip = user2SIlength(atof(gtk_entry_get_text(GTK_ENTRY(pui->entrytrip)))); */
      str = get_entry(pui->entrytrip);
      trip = user2SIlength(atof(str->str));
      g_string_free(str, TRUE);

/*       fill = user2SIvolume(atof(gtk_entry_get_text(GTK_ENTRY(pui->entryfill)))); */
      str = get_entry(pui->entryfill);
      fill  = user2SIvolume(atof(str->str));
      g_string_free(str, TRUE);

/*       price = atof(gtk_entry_get_text(GTK_ENTRY(pui->entryprice))); */
      str = get_entry(pui->entryprice);
      price  = atof(str->str);
      g_string_free(str, TRUE);

/*       service = atof(gtk_entry_get_text(GTK_ENTRY(pui->entryservice))); */
      str = get_entry(pui->entryservice);
      service = atof(str->str);
      g_string_free(str, TRUE);

/*       oil=atof(gtk_entry_get_text(GTK_ENTRY(pui->entryoil))); */
      str = get_entry(pui->entryoil);
      oil = atof(str->str);
      g_string_free(str, TRUE);

/*       tires=atof(gtk_entry_get_text(GTK_ENTRY(pui->entrytires))); */
      str = get_entry(pui->entrytires);
      tires = atof(str->str);
      g_string_free(str, TRUE);

      notes = get_entry(pui->entrynotes);

      /* Take not full tanks into account: calculate the consumption for full record */
#if MAEMO_VERSION_MAJOR == 5
      if (hildon_check_button_get_active (HILDON_CHECK_BUTTON (pui->buttonnotfull))) { 
#else
      if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (pui->buttonnotfull))) {
#endif

	/* For this record */
	consum = 0.0;

	/* Find next full record */
	fullid=db_find_next_full(km, &fullfill, &fullkm);
	if (fullid>=0) {
	  fullconsum=(fullfill+fill)/(fullkm+trip)*100;

	  /* Update now the full record consum and tree view also */
	  db_update_record(fullid, NULL, -1.0, -1.0, -1.0, fullconsum, -1.0, -1.0, -1.0, -1.0, -1.0, NULL);

	  get_store_and_iter(NULL, pui->view, NULL, &store, NULL);
	  if (ui_find_iter(store, &storeiter,fullid)) {
	    ui_update_row_data(store, &storeiter,
			       NULL, /* date */
			       -1.0, /* km */
			       -1.0, /* trip */
			       -1.0, /* fill */
			       fullconsum, /* consum */
			       -1.0, /* price */
			       -1.0, /* price/trip */
			       -1.0, /* service */
			       -1.0, /* oil */
			       -1.0, /* tires */
			       NULL, /* notes */
			       fullid,
			       TRUE);
	  }
	  else
	    PDEBUG("First iter not found when trying to update full fill record\n");
	}
	else
	  PDEBUG("Full fill not found\n");
      }
      else { // Full fill
	/* Find if there are any not full fills before this record */
	db_find_prev_full(km, &fullfill, &fullkm);

	consum = (fullfill+fill)/(fullkm+trip)*100;
      }
      if (db_open()) {
	recordid = db_add_record(date, km, trip, fill, consum, price, 
				 price/fill, service, oil, tires, notes->str);
	if (recordid<0) {
	  /*     ui_show_banner("Problem saving result"); */
	  PDEBUG("Problem saving result");
	}
	else { /* record succesfully inserted */
	  get_store_and_iter(NULL, pui->view, NULL, &store, NULL);
	  gtk_list_store_append (store, &storeiter);
	  ui_update_row_data(store, &storeiter,
			     date,
			     km,
			     trip,
			     fill,
			     consum,
			     price,
			     price/fill,
			     service,
			     oil,
			     tires,
			     notes->str,
			     recordid,
			     TRUE);
	  
	  update_totalkm(pui);
	  g_string_free(notes, TRUE);

	  /* Scroll to the end of list */
#if MAEMO_VERSION_MAJOR == 5
#if 0
	  path = gtk_tree_model_get_path (store, &storeiter);
	  gtk_tree_view_get_background_area (GTK_TREE_VIEW (pui->view),
                                    path, NULL, &rect);
	  gtk_tree_view_convert_bin_window_to_tree_coords (GTK_TREE_VIEW (pui->view),
							   0, rect->y, NULL, &y);
	  hildon_pannable_area_scroll_to (HILDON_PANNABLE_AREA(pui->datascrollwin), -1, y);
	  gtk_tree_path_free (path);
#endif
#else
	  if (maintablesorted) {
	    vertical = gtk_scrolled_window_get_vadjustment((GtkScrolledWindow *)pui->datascrollwin);
	    if (maintablesortorder==GTK_SORT_ASCENDING) {
	      vertical->value = vertical->upper;
	    }
	    else {
	      vertical->value = vertical->lower;
	    }
	    gtk_scrolled_window_set_vadjustment((GtkScrolledWindow *)pui->datascrollwin,vertical);
	    gtk_adjustment_value_changed(vertical);
	  }
#endif
	  
	}
	sqlite3_reset(ppStmtAddRecord);
      }
      else {
	PDEBUG("Can't save new record - no database open");
      }
      free(date);
      gtk_widget_destroy(GTK_WIDGET(dialog));
      break;
    case GTK_RESPONSE_REJECT:
      gtk_widget_destroy(GTK_WIDGET(dialog));
      break;
    }
  }
}

/**
 * \fn void edit_record_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
 * \brief Response callback for edit record dialog
 *
 * This callback alters the data edited in the edit record dialog to
 * the database.
 *
 * \todo This should be changed so that fuelpad.c has a function
 * edit_driver which is called with the data (i.e. sqlite would not be
 * used here)
 *
 */
void edit_record_response (GtkDialog *dialog, gint arg1, AppUIData *pui)
{
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkListStore  *store;
  GtkListStore  *fullstore;
  GtkTreeIter    iter;
  GtkTreeIter    storeiter;
  GtkTreeIter    fullstoreiter;

  gint64 id;
  GString *str;
  GString *notes;
  char *date;
  char *userdate;
  gdouble km;
  gdouble trip;
  gdouble fill;
  gdouble price;
  gdouble consum;
  gdouble service;
  gdouble oil;
  gdouble tires;
#if PLAINGTK == 0
  guint year;
  guint month;
  guint day;
#endif
  gboolean notfull;
  gboolean oldnotfull;
  sqlite_int64 fullid;
  gdouble fullconsum;
  gdouble fullfill;
  gdouble fullkm;
  gdouble oldfill;
  gdouble oldtrip;
  gdouble oldconsum;

  if (!db_open()) {
       gtk_widget_destroy(GTK_WIDGET(dialog));
  }
  else {
    switch (arg1) {
    case GTK_RESPONSE_ACCEPT:
      selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->view));
      if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
	gtk_tree_model_get (model, &iter, COL_ID, &id, -1);

#if PLAINGTK == 0
#if MAEMO_VERSION_MAJOR == 5
      hildon_date_button_get_date(HILDON_DATE_BUTTON(pui->entrydate), &year, &month, &day);  /* Month is betweewn 0 and 11 */
      month++; 
#else
	hildon_date_editor_get_date(HILDON_DATE_EDITOR(pui->entrydate), &year, &month, &day);
#endif
	date=(char *)malloc(ENTRYDATEMAX*sizeof(char));
	g_snprintf(date, ENTRYDATEMAX, "%d-%02d-%02d",year, month, day);
#else
	date2sqlite(&date, datefmtstr[curunit.dateformat], 
		    gtk_entry_get_text(GTK_ENTRY(pui->entrydate)));
#endif

/* 	km = user2SIlength(atof(gtk_entry_get_text(GTK_ENTRY(pui->entrykm)))); */
      str = get_entry(pui->entrykm);
      km  = user2SIlength(atof(str->str));
      g_string_free(str, TRUE);

/* 	trip = user2SIlength(atof(gtk_entry_get_text(GTK_ENTRY(pui->entrytrip)))); */
      str = get_entry(pui->entrytrip);
      trip = user2SIlength(atof(str->str));
      g_string_free(str, TRUE);

/* 	fill = user2SIvolume(atof(gtk_entry_get_text(GTK_ENTRY(pui->entryfill)))); */
      str = get_entry(pui->entryfill);
      fill  = user2SIvolume(atof(str->str));
      g_string_free(str, TRUE);

/* 	price = atof(gtk_entry_get_text(GTK_ENTRY(pui->entryprice))); */
      str = get_entry(pui->entryprice);
      price  = atof(str->str);
      g_string_free(str, TRUE);

/* 	service = atof(gtk_entry_get_text(GTK_ENTRY(pui->entryservice))); */
      str = get_entry(pui->entryservice);
      service = atof(str->str);
      g_string_free(str, TRUE);

/* 	oil=atof(gtk_entry_get_text(GTK_ENTRY(pui->entryoil))); */
      str = get_entry(pui->entryoil);
      oil = atof(str->str);
      g_string_free(str, TRUE);

/* 	tires=atof(gtk_entry_get_text(GTK_ENTRY(pui->entrytires))); */
      str = get_entry(pui->entrytires);
      tires = atof(str->str);
      g_string_free(str, TRUE);

      notes = get_entry(pui->entrynotes);

      oldnotfull=FALSE;
      /* 
       * Well need to obtain the unmodified data to be excluded from the new 
       * consumption calculations 
       */
      if (SQLITE_OK == sqlite3_bind_int(ppStmtOneRecord, 1, id)) {
	while (SQLITE_ROW == sqlite3_step(ppStmtOneRecord)) {
	  oldfill=sqlite3_column_double(ppStmtOneRecord,3);
	  oldtrip=sqlite3_column_double(ppStmtOneRecord,2);
	  oldconsum=sqlite3_column_double(ppStmtOneRecord,9);
	  oldnotfull=(oldfill>0.0) && (abs(oldconsum)<1e-5);
	}
	sqlite3_reset(ppStmtOneRecord);
      }
      

      /* Take not full tanks into account: calculate the consumption for full record */
      notfull = 
#if MAEMO_VERSION_MAJOR == 5
	hildon_check_button_get_active (HILDON_CHECK_BUTTON (pui->buttonnotfull));
#else
        gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (pui->buttonnotfull));
#endif
      if (notfull) {

	/* For this record */
	consum = 0.0;

	/* Find next full record */
	fullid=db_find_next_full(km, &fullfill, &fullkm);

	if (fullid>=0) {
	  if (!oldnotfull) {
	    oldfill=0.0;
	    oldtrip=0.0;
	  }
	  fullconsum=(fullfill+fill-oldfill)/(fullkm+trip-oldtrip)*100;

	  /* Update now the full record consum and tree view also */
	  db_update_record(fullid, NULL, -1.0, -1.0, -1.0, fullconsum, -1.0, -1.0, -1.0, -1.0, -1.0, NULL);

	}
	else
	  PDEBUG("Full fill not found\n");
      }
      else { // Full fill

	if (oldnotfull) {

	  /* Find next full record */
	  fullid=db_find_next_full(km, &fullfill, &fullkm);

	  if (fullid>=0) {

	    fullconsum=(fullfill-oldfill)/(fullkm-oldtrip)*100;

	    /* Update now the full record consum and tree view also */
	    db_update_record(fullid, NULL, -1.0, -1.0, -1.0, fullconsum, -1.0, -1.0, -1.0, -1.0, -1.0, NULL);
	  }
	}

	/* Find if there are any not full fills before this record */
	db_find_prev_full(km, &fullfill, &fullkm);

	if (oldnotfull) {
	  oldfill=0.0;
	  oldtrip=0.0;
	}
/* 	consum = (fullfill+fill-oldfill)/(fullkm+trip-oldtrip)*100; */
	consum = (fullfill+fill)/(fullkm+trip)*100;
      }

	if (db_open()) {
	  if (db_update_record(id, date, km, trip, fill, consum, price, price/fill,
			       service, oil, tires, notes->str) != id) {
	    PDEBUG("Problem saving result");
	  }
	  else {
	    get_store_and_iter(model, pui->view, &iter, &store, &storeiter);
	    ui_update_row_data(store, &storeiter,
			       date,
			       km,
			       trip,
			       fill,
			       consum,
			       price,
			       price/fill,
			       service,
			       oil,
			       tires,
			       notes->str,
			       id,
			       TRUE);

	    /* Update the data for the full fill */
	    if (notfull || (notfull!=oldnotfull)) { /* not enough to test notfull, but when?*/
	      if (fullid>0) {
		get_store_and_iter(NULL, pui->view, NULL, &fullstore, NULL);
		if (ui_find_iter(fullstore, &fullstoreiter,fullid)) {
		  ui_update_row_data(fullstore, &fullstoreiter,
				     NULL, /* date */
				     -1.0, /* km */
				     -1.0, /* trip */
				     -1.0, /* fill */
				     fullconsum, /* consum */
				     -1.0, /* price */
				     -1.0, /* price/trip */
				     -1.0, /* service */
				     -1.0, /* oil */
				     -1.0, /* tires */
				     NULL, /* notes */
				     fullid,
				     TRUE);
		}
	      }
	    }
	    else
	      PDEBUG("First iter not found when trying to update full fill record\n");
	    update_totalkm(pui);
	    g_string_free(notes, TRUE);
	
	  }
	  
	  sqlite3_reset(ppStmtUpdateRecord);
	}
	else {
	  PDEBUG("Can't save new driver - no database open");
	}
      }
      else {
	PDEBUG("editrecord: Something wrong with finding the selection\n");
      }
      free(date);
      gtk_widget_destroy(GTK_WIDGET(dialog));
      break;
    case GTK_RESPONSE_REJECT:
      gtk_widget_destroy(GTK_WIDGET(dialog));
      break;
    }
  }
}

gboolean callback_kmadded (GtkWidget     *widget,
                           GdkEventFocus *event,
                           AppUIData     *pui)
{
  float trip;
  float newkm;
  float lastkm;
  GString *buf;

  buf=g_string_new(NULL);
  trip=atof(gtk_entry_get_text(GTK_ENTRY(pui->entrytrip)));
  newkm=user2SIlength(atof(gtk_entry_get_text(GTK_ENTRY(pui->entrykm))));
  if (trip<0.1 && newkm>0) {
    lastkm=db_get_last_refill(newkm);
    if (lastkm<0.1)
      lastkm=db_get_last_km();
    g_string_printf(buf, "%.1f",SIlength2user(newkm-lastkm));
    gtk_entry_set_text(GTK_ENTRY(pui->entrytrip),buf->str);
  }
  g_string_free(buf, TRUE);
  return FALSE;
}

gboolean callback_tripadded (GtkWidget     *widget,
			     GdkEventFocus *event,
			     AppUIData     *pui)
{
  float trip;
  float newkm;
  float lastkm;
  GString *buf;

  buf=g_string_new(NULL);
  newkm=atof(gtk_entry_get_text(GTK_ENTRY(pui->entrykm)));
  if (newkm<0.1) {
    trip=user2SIlength(atof(gtk_entry_get_text(GTK_ENTRY(pui->entrytrip))));
    lastkm=db_get_last_refill(newkm);
    if (lastkm<0.1)
      lastkm=db_get_last_km();
    g_string_printf(buf, "%.1f",SIlength2user(lastkm+trip));
    gtk_entry_set_text(GTK_ENTRY(pui->entrykm),buf->str);
  }
  g_string_free(buf, TRUE);
  return FALSE;
}

/**
 * \fn void create_record_editwin(AppUIData *pui, GtkWidget *dialog, gint add)
 * \brief Creates the contents of a record adding or editing dialog
 * \param *pui Pointer to the UI data structure
 * \param *dialog Pointer to the already created dialog
 * \param add Set this to 1 if an add dialog is needed
 *
 * This function is used for creating an edit window both for adding a
 * new record and editing an existing one.
 *
 */
void create_record_editwin(AppUIData *pui, GtkWidget *dialog, gint add)
{
  enum
  {
    RECORD_EDIT_DATE=0,
    RECORD_EDIT_KM,
    RECORD_EDIT_MILES,
    RECORD_EDIT_TRIP,
    RECORD_EDIT_FILL,
    RECORD_EDIT_NOTFULL,
    RECORD_EDIT_PRICE,
    RECORD_EDIT_NOTES,
    RECORD_EDIT_SERVICE,
    RECORD_EDIT_OIL,
    RECORD_EDIT_TIRES,
    RECORD_EDIT_CAR,
    RECORD_EDIT_DRIVER
  };

  GtkWidget *scrollwin;
#if MAEMO_VERSION_MAJOR == 5
  gint       row;
#else
  GtkWidget *notebook;
#endif
  GtkWidget *table;
  GtkWidget *label;
#if PLAINGTK == 1  
  char date[ENTRYDATEMAX];
  char *aika;
#endif
  GtkListStore *store;
  GtkEntryCompletion *completion;
  gchar *labels[] = {gettext_noop("Date"),
		     gettext_noop("Km"),
		     gettext_noop("Miles"),
		     gettext_noop("Trip"),
		     gettext_noop("Fill"),
		     gettext_noop("Not full tank"),
		     gettext_noop("Price"),
		     gettext_noop("Notes"),
		     gettext_noop("Service"),
		     gettext_noop("Oil"),
		     gettext_noop("Tires"),
		     gettext_noop("Car"),
		     gettext_noop("Driver")
  };

#if MAEMO_VERSION_MAJOR == 5
  scrollwin = hildon_pannable_area_new();
  gtk_widget_set_size_request(scrollwin, -1, DIALOG_MIN_HEIGHTMAX);

  if (add)
    table = gtk_table_new(12, 2, FALSE);
  else
    table = gtk_table_new(10, 2, FALSE);

  row=0;

  if (add) {

    /* First row, first entry */
  pui->carbutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->carbutton), gettext(labels[RECORD_EDIT_CAR]));
  gtk_table_attach(GTK_TABLE(table), pui->carbutton, 0, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->carbutton);

  pui->carcombo = create_car_combo(pui, pui->carbutton);

  /* First row, second entry */
  row++;
  pui->driverbutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->driverbutton), gettext(labels[RECORD_EDIT_DRIVER]));
  gtk_table_attach(GTK_TABLE(table), pui->driverbutton, 0, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->driverbutton);

  pui->drivercombo = create_driver_combo(pui, pui->driverbutton);
/*     gtk_table_attach(GTK_TABLE(table), pui->drivercombo, 3, 4, 3, 4,  */
/* 		     GTK_EXPAND|GTK_FILL, */
/* 		     0, 0, 5); */
/*     gtk_widget_show(pui->drivercombo); */

  }

  /* First row, first entry */
  row++;
  pui->entrydate = hildon_date_button_new(HILDON_SIZE_FINGER_HEIGHT,
					  HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  gtk_table_attach(GTK_TABLE(table), pui->entrydate, 0, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrydate);
  
  /* First row, second entry */
  row++;
  if (curunit.lengthunit==SI)
    label=gtk_label_new(gettext(labels[RECORD_EDIT_KM]));
  else
    label=gtk_label_new(gettext(labels[RECORD_EDIT_MILES]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1,
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
  gtk_widget_show(label);

/*   pui->entrykm = gtk_entry_new();  */
  pui->entrykm=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrykm), ENTRYKMMAX);
#if PLAINGTK == 0
  g_object_set(G_OBJECT(pui->entrykm),
#if HILDON == 1
              "hildon-input-mode",
		 HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
		 NULL);
#else
		 "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
  g_object_set(G_OBJECT(pui->entrykm), "autocap", FALSE, NULL);
#endif
#endif
  if (add) {
    g_signal_connect(pui->entrykm, "focus-out-event", 
		     G_CALLBACK(callback_kmadded), &ui);
  }

  gtk_table_attach(GTK_TABLE(table), pui->entrykm, 1, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrykm);

  /* Second row, first entry */
  row++;
  label=gtk_label_new(gettext(labels[RECORD_EDIT_TRIP]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1,
	    GTK_EXPAND|GTK_FILL,
		    0, 0, 5);
  gtk_widget_show(label);

/*   pui->entrytrip = gtk_entry_new();  */
  pui->entrytrip=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrytrip), ENTRYTRIPMAX);
#if PLAINGTK == 0
  g_object_set(G_OBJECT(pui->entrytrip),
#if HILDON == 1
              "hildon-input-mode",
	       HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
	       NULL);
#else
               "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
  g_object_set(G_OBJECT(pui->entrytrip), "autocap", FALSE, NULL);
#endif
#endif
  if (add) {
    g_signal_connect(pui->entrytrip, "focus-out-event", 
		     G_CALLBACK(callback_tripadded), &ui);
  }
  gtk_table_attach(GTK_TABLE(table), pui->entrytrip, 1, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrytrip);

  /* Second row, second entry */
  row++;
  label=gtk_label_new(gettext(labels[RECORD_EDIT_FILL]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

/*   pui->entryfill = gtk_entry_new();  */
  pui->entryfill=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryfill), ENTRYFILLMAX);
#if PLAINGTK == 0
  g_object_set(G_OBJECT(pui->entryfill),
#if HILDON == 1
	       "hildon-input-mode",
	       HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
	       NULL);
#else
               "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
  g_object_set(G_OBJECT(pui->entryfill), "autocap", FALSE, NULL);
#endif
#endif
  gtk_table_attach(GTK_TABLE(table), pui->entryfill, 1, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryfill);

  row++;
  pui->buttonnotfull = hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_button_set_label (HILDON_CHECK_BUTTON (pui->buttonnotfull), gettext(labels[RECORD_EDIT_NOTFULL]));
  gtk_table_attach(GTK_TABLE(table), pui->buttonnotfull, 1, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->buttonnotfull);

  /* Third row, first entry */
  row++;
  label=gtk_label_new(gettext(labels[RECORD_EDIT_PRICE]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

/*   pui->entryprice = gtk_entry_new();  */
  pui->entryprice=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryprice), ENTRYPRICEMAX);
#if PLAINGTK == 0
  g_object_set(G_OBJECT(pui->entryprice),
#if HILDON == 1
	       "hildon-input-mode",
	       HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
	       NULL);
#else
               "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
  g_object_set(G_OBJECT(pui->entryprice), "autocap", FALSE, NULL);
#endif
#endif
  gtk_table_attach(GTK_TABLE(table), pui->entryprice, 1, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryprice);

  /* Third row, second entry */
  row++;
  label=gtk_label_new(gettext(labels[RECORD_EDIT_NOTES]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

/*   pui->entrynotes = gtk_entry_new();  */
  pui->entrynotes=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrynotes), ENTRYNOTESMAX);
  gtk_table_attach(GTK_TABLE(table), pui->entrynotes, 1, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrynotes);

  completion = gtk_entry_completion_new();
  store = (GtkListStore *)gtk_tree_view_get_model(GTK_TREE_VIEW(pui->view));
  gtk_entry_completion_set_model(completion, GTK_TREE_MODEL(store));
  gtk_entry_completion_set_text_column(completion, COL_NOTES);
  gtk_entry_set_completion(GTK_ENTRY(pui->entrynotes), completion);

  /* First row, first entry */
  row++;
  label=gtk_label_new(gettext(labels[RECORD_EDIT_SERVICE]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1,
	    GTK_EXPAND|GTK_FILL,
	    0, 0, 5);
  gtk_widget_show(label);

/*     pui->entryservice = gtk_entry_new();  */
  pui->entryservice=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryservice), ENTRYSERVICEMAX);
#if PLAINGTK == 0
  g_object_set(G_OBJECT(pui->entryservice),
#if HILDON == 1
                "hildon-input-mode",
		 HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
		 NULL);
#else
                 "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
  g_object_set(G_OBJECT(pui->entryservice), "autocap", FALSE, NULL);
#endif
#endif
  gtk_table_attach(GTK_TABLE(table), pui->entryservice, 1, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryservice);

    /* Second row, first entry */
  row++;
  label=gtk_label_new(gettext(labels[RECORD_EDIT_OIL]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
    gtk_widget_show(label);

/*     pui->entryoil = gtk_entry_new();  */
  pui->entryoil=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryoil), ENTRYOILMAX);
#if PLAINGTK == 0
  g_object_set(G_OBJECT(pui->entryoil),
#if HILDON == 1
              "hildon-input-mode",
	       HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
	       NULL);
#else
               "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
  g_object_set(G_OBJECT(pui->entryoil), "autocap", FALSE, NULL);
#endif
#endif
  gtk_table_attach(GTK_TABLE(table), pui->entryoil, 1, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryoil);

    /* Third row, first entry */
  row++;
  label=gtk_label_new(gettext(labels[RECORD_EDIT_TIRES]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

/*     pui->entrytires = gtk_entry_new();  */
  pui->entrytires=hildon_entry_new(HILDON_SIZE_FINGER_HEIGHT);
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrytires), ENTRYTIRESMAX);
#if PLAINGTK == 0
  g_object_set(G_OBJECT(pui->entrytires),
#if HILDON == 1
              "hildon-input-mode",
	       HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
	       NULL);
#else
               "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
  g_object_set(G_OBJECT(pui->entrytires), "autocap", FALSE, NULL);
#endif
#endif
  gtk_table_attach(GTK_TABLE(table), pui->entrytires, 1, 2, row, row+1,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrytires);

  gtk_widget_show(table);
  gtk_box_pack_start (GTK_BOX(GTK_DIALOG (dialog)->vbox), scrollwin, TRUE, TRUE, 0);
  gtk_widget_show(scrollwin);
  hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA(scrollwin),table);

#else /* MAEMO_VERSION_MAJOR != 5 */
  notebook = gtk_notebook_new ();
  gtk_notebook_set_tab_pos (GTK_NOTEBOOK (notebook), GTK_POS_TOP);
  gtk_box_pack_start (GTK_BOX(GTK_DIALOG (dialog)->vbox), notebook, TRUE, TRUE, 0);
  gtk_widget_show (notebook);

  scrollwin = gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_set_size_request(scrollwin, -1, DIALOG_MIN_HEIGHT1);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollwin), 
				 GTK_POLICY_NEVER, 
				 GTK_POLICY_AUTOMATIC);

/*   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), scrollwin, TRUE, TRUE, 0); */
  table = gtk_table_new(4, 4, FALSE);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrollwin),table);

  /* First row, first entry */
  label=gtk_label_new(gettext(labels[RECORD_EDIT_DATE]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

#if PLAINGTK == 1  
  pui->entrydate = gtk_entry_new(); 
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrydate), ENTRYDATEMAX);
  gettimefmt(&aika, datefmtstr[curunit.dateformat]);
  gtk_entry_set_text(GTK_ENTRY(pui->entrydate),aika);
  free(aika);
#else
  pui->entrydate = hildon_date_editor_new();
#endif
  gtk_table_attach(GTK_TABLE(table), pui->entrydate, 1, 2, 0, 1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrydate);
  
  /* First row, second entry */
  if (curunit.lengthunit==SI)
    label=gtk_label_new(gettext(labels[RECORD_EDIT_KM]));
  else
    label=gtk_label_new(gettext(labels[RECORD_EDIT_MILES]));
  gtk_table_attach(GTK_TABLE(table), label, 2, 3, 0, 1, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
  gtk_widget_show(label);

  pui->entrykm = gtk_entry_new(); 
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrykm), ENTRYKMMAX);
#if PLAINGTK == 0
  g_object_set(G_OBJECT(pui->entrykm),
#if HILDON == 1
              "hildon-input-mode",
		 HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
		 NULL);
#else
		 "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
  g_object_set(G_OBJECT(pui->entrykm), "autocap", FALSE, NULL);
#endif
#endif
  if (add) {
    g_signal_connect(pui->entrykm, "focus-out-event", 
		     G_CALLBACK(callback_kmadded), &ui);
  }

  gtk_table_attach(GTK_TABLE(table), pui->entrykm, 3, 4, 0, 1, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrykm);

  /* Second row, first entry */
  label=gtk_label_new(gettext(labels[RECORD_EDIT_TRIP]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, 
	    GTK_EXPAND|GTK_FILL,
		    0, 0, 5);
  gtk_widget_show(label);

  pui->entrytrip = gtk_entry_new(); 
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrytrip), ENTRYTRIPMAX);
#if PLAINGTK == 0
  g_object_set(G_OBJECT(pui->entrytrip),
#if HILDON == 1
              "hildon-input-mode",
	       HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
	       NULL);
#else
               "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
  g_object_set(G_OBJECT(pui->entrytrip), "autocap", FALSE, NULL);
#endif
#endif
  if (add) {
    g_signal_connect(pui->entrytrip, "focus-out-event", 
		     G_CALLBACK(callback_tripadded), &ui);
  }
  gtk_table_attach(GTK_TABLE(table), pui->entrytrip, 1, 2, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrytrip);

  /* Second row, second entry */
  label=gtk_label_new(gettext(labels[RECORD_EDIT_FILL]));
  gtk_table_attach(GTK_TABLE(table), label, 2, 3, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entryfill = gtk_entry_new(); 
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryfill), ENTRYFILLMAX);
#if PLAINGTK == 0
  g_object_set(G_OBJECT(pui->entryfill),
#if HILDON == 1
	       "hildon-input-mode",
	       HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
	       NULL);
#else
               "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
  g_object_set(G_OBJECT(pui->entryfill), "autocap", FALSE, NULL);
#endif
#endif
  gtk_table_attach(GTK_TABLE(table), pui->entryfill, 3, 4, 1, 2, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryfill);

  /* Not full button */
  pui->buttonnotfull = gtk_check_button_new_with_label(gettext(labels[RECORD_EDIT_NOTFULL]));
  gtk_table_attach(GTK_TABLE(table), pui->buttonnotfull, 3, 4, 2, 3,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->buttonnotfull);

  /* Third row, first entry */
  label=gtk_label_new(gettext(labels[RECORD_EDIT_PRICE]));
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 3, 4, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entryprice = gtk_entry_new(); 
  gtk_entry_set_max_length(GTK_ENTRY(pui->entryprice), ENTRYPRICEMAX);
#if PLAINGTK == 0
  g_object_set(G_OBJECT(pui->entryprice),
#if HILDON == 1
	       "hildon-input-mode",
	       HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
	       NULL);
#else
               "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
  g_object_set(G_OBJECT(pui->entryprice), "autocap", FALSE, NULL);
#endif
#endif
  gtk_table_attach(GTK_TABLE(table), pui->entryprice, 1, 2, 3, 4, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entryprice);

  /* Third row, second entry */
  label=gtk_label_new(gettext(labels[RECORD_EDIT_NOTES]));
  gtk_table_attach(GTK_TABLE(table), label, 2, 3, 3, 4,
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(label);

  pui->entrynotes = gtk_entry_new(); 
  gtk_entry_set_max_length(GTK_ENTRY(pui->entrynotes), ENTRYNOTESMAX);
  gtk_table_attach(GTK_TABLE(table), pui->entrynotes, 3, 4, 3, 4, 
		   GTK_EXPAND|GTK_FILL,
		   0, 0, 5);
  gtk_widget_show(pui->entrynotes);

  completion = gtk_entry_completion_new();
  store = (GtkListStore *)gtk_tree_view_get_model(GTK_TREE_VIEW(pui->view));
  gtk_entry_completion_set_model(completion, GTK_TREE_MODEL(store));
  gtk_entry_completion_set_text_column(completion, COL_NOTES);
  gtk_entry_set_completion(GTK_ENTRY(pui->entrynotes), completion);

  /* Table ready - show it */
  gtk_widget_show (table);
  gtk_widget_show(scrollwin);

  label = gtk_label_new (_("Fuel fill"));
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), scrollwin, label);

  /* Service etc. notebook */
  table = gtk_table_new(1, 3, FALSE);

    /* First row, first entry */
    label=gtk_label_new(gettext(labels[RECORD_EDIT_SERVICE]));
    gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(label);

    pui->entryservice = gtk_entry_new(); 
    gtk_entry_set_max_length(GTK_ENTRY(pui->entryservice), ENTRYSERVICEMAX);
#if PLAINGTK == 0
    g_object_set(G_OBJECT(pui->entryservice),
#if HILDON == 1
                "hildon-input-mode",
		 HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
		 NULL);
#else
                 "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
    g_object_set(G_OBJECT(pui->entryservice), "autocap", FALSE, NULL);
#endif
#endif
    gtk_table_attach(GTK_TABLE(table), pui->entryservice, 1, 2, 0, 1, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(pui->entryservice);

    /* Second row, first entry */
    label=gtk_label_new(gettext(labels[RECORD_EDIT_OIL]));
    gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(label);

    pui->entryoil = gtk_entry_new(); 
    gtk_entry_set_max_length(GTK_ENTRY(pui->entryoil), ENTRYOILMAX);
#if PLAINGTK == 0
    g_object_set(G_OBJECT(pui->entryoil),
#if HILDON == 1
                "hildon-input-mode",
		 HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
		 NULL);
#else
                 "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
    g_object_set(G_OBJECT(pui->entryoil), "autocap", FALSE, NULL);
#endif
#endif
    gtk_table_attach(GTK_TABLE(table), pui->entryoil, 1, 2, 1, 2, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(pui->entryoil);

    /* Third row, first entry */
    label=gtk_label_new(gettext(labels[RECORD_EDIT_TIRES]));
    gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(label);

    pui->entrytires = gtk_entry_new(); 
    gtk_entry_set_max_length(GTK_ENTRY(pui->entrytires), ENTRYTIRESMAX);
#if PLAINGTK == 0
    g_object_set(G_OBJECT(pui->entrytires),
#if HILDON == 1
                "hildon-input-mode",
		 HILDON_GTK_INPUT_MODE_NUMERIC|HILDON_GTK_INPUT_MODE_SPECIAL,
		 NULL);
#else
                 "input-mode", HILDON_INPUT_MODE_HINT_NUMERICSPECIAL, NULL);
    g_object_set(G_OBJECT(pui->entrytires), "autocap", FALSE, NULL);
#endif
#endif
    gtk_table_attach(GTK_TABLE(table), pui->entrytires, 1, 2, 2, 3, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(pui->entrytires);

    /* Table ready - show it */
    gtk_widget_show (table);

  label = gtk_label_new (_("Service/oil/tires"));
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook), table, label);

  if (add) {

    table = gtk_table_new(2, 2, FALSE);

    /* First row, first entry */
    label=gtk_label_new(gettext(labels[RECORD_EDIT_CAR]));
    gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(label);

    pui->carcombo = create_car_combo(pui);
    gtk_table_attach(GTK_TABLE(table), pui->carcombo, 1, 2, 0, 1, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(pui->carcombo);

    /* First row, second entry */
    label=gtk_label_new(gettext(labels[RECORD_EDIT_DRIVER]));
    gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(label);
    
    pui->drivercombo = create_driver_combo(pui);
    gtk_table_attach(GTK_TABLE(table), pui->drivercombo, 1, 2, 1, 2, 
		     GTK_EXPAND|GTK_FILL,
		     0, 0, 5);
    gtk_widget_show(pui->drivercombo);

    gtk_widget_show (table);

    label = gtk_label_new (_("Driver and car"));
    gtk_notebook_append_page (GTK_NOTEBOOK (notebook), table, label);

  }
#endif

}


void callback_newrecord( GtkAction * action, AppUIData *pui )
{
  GtkWidget *dialog;
  GtkWidget *label;
  gchar *header[] = {gettext_noop("Add a new record")};

  if (db_open()) {
    dialog = gtk_dialog_new_with_buttons(gettext(header[0]),
					 GTK_WINDOW(pui->app->mainwindow),
					 GTK_DIALOG_MODAL,
					 GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
					 GTK_STOCK_CANCEL,
					 GTK_RESPONSE_REJECT,
					 NULL);
    create_record_editwin(pui, dialog, 1);

#if LIBHELP == 1
    help_dialog_help_enable(GTK_DIALOG(dialog),
				   HELP_ID_ADDRECORD,
				   pui->app->osso);
#endif

/*     g_signal_connect_swapped (dialog, "response", */
/* 			      G_CALLBACK (add_record_response), */
/* 			      dialog); */
    g_signal_connect (dialog, "response",
		      G_CALLBACK (add_record_response),
		      &ui);
  }
  else {
    dialog = gtk_dialog_new_with_buttons(gettext(header[0]),
					 GTK_WINDOW(pui->app->mainwindow),
					 GTK_DIALOG_MODAL,
					 GTK_STOCK_OK, 
					 GTK_RESPONSE_REJECT,
					 NULL);

    label=gtk_label_new(_("Can't access database - adding records not possible"));
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label, TRUE, TRUE, 5);
    gtk_widget_show(label);

    g_signal_connect_swapped (dialog, "response",
			      G_CALLBACK (gtk_widget_destroy),
			      dialog);
  }


  gtk_widget_show(dialog);
}

void callback_prevrecord( GtkAction * action, AppUIData *pui )
{
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter    iter;
  GtkTreePath   *path;
  GtkAdjustment *vertical;

  if (db_open()) {
      selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->view));
      if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
	vertical = gtk_scrolled_window_get_vadjustment((GtkScrolledWindow *)pui->datascrollwin);
	vertical->value -= vertical->step_increment;
	if (vertical->value<vertical->lower)
	  vertical->value = vertical->lower;
	/* 	  gtk_tree_model_iter_prev(model,&iter); */
	path = gtk_tree_model_get_path(model, &iter);
	if (gtk_tree_path_prev(path)) {
	  gtk_tree_model_get_iter(model, &iter, path);
	  gtk_tree_selection_select_iter(selection, &iter);
	}
	gtk_tree_path_free(path);
	gtk_tree_selection_select_iter(selection, &iter);
	gtk_scrolled_window_set_vadjustment((GtkScrolledWindow *)pui->datascrollwin,vertical);
	gtk_adjustment_value_changed(vertical);
      }
  }
}

void callback_nextrecord( GtkAction * action, AppUIData *pui )
{
  GtkTreeSelection *selection;
  GtkTreeModel *model;
  GtkTreeIter    iter;
  GtkAdjustment *vertical;

  if (db_open()) {
      selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->view));
      if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
	vertical = gtk_scrolled_window_get_vadjustment((GtkScrolledWindow *)pui->datascrollwin);
	vertical->value += vertical->step_increment;
	if (vertical->value>vertical->upper)
	  vertical->value = vertical->upper;
	gtk_tree_model_iter_next(model,&iter);
	gtk_tree_selection_select_iter(selection, &iter);
	gtk_scrolled_window_set_vadjustment((GtkScrolledWindow *)pui->datascrollwin,vertical);
	gtk_adjustment_value_changed(vertical);
      }
  }
}

/**
 * \fn void callback_editrecord( GtkAction * action, AppUIData *pui )
 * \brief Callback for editing a record
 *
 * This callback creates the record editing dialog
 *
 * \todo This should be changed so that fuelpad.c has a function for
 * obtaining the record data (i.e. sqlite would not be used here)
 *
 */
void callback_editrecord( GtkAction * action, AppUIData *pui )
{
  GtkWidget *dialog;
  GtkWidget *label;
  gchar *header[] = {gettext_noop("Edit a record")};

  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreeModel *model;
  gint64 id;
  char buf[BUFMAX];
#if PLAINGTK == 0
  guint year;
  guint month;
  guint day;
#else
  char *date;
#endif
  char notes[ENTRYNOTESMAX];
  gchar *convnotes;

  if (db_open()) {
    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->view));
    if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
      dialog = gtk_dialog_new_with_buttons(gettext(header[0]),
					   GTK_WINDOW(pui->app->mainwindow),
					   GTK_DIALOG_MODAL,
					   GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
					   GTK_STOCK_CANCEL,
					   GTK_RESPONSE_REJECT,
					   NULL);
      create_record_editwin(pui, dialog, 0);

      gtk_tree_model_get (model, &iter, COL_ID, &id, -1);
      /** \todo Data should be read from the store, not from database */
      if (SQLITE_OK == sqlite3_bind_int(ppStmtOneRecord, 1, id)) {
	while (SQLITE_ROW == sqlite3_step(ppStmtOneRecord)) {
#if PLAINGTK == 1
	  convdate(&date, datefmtstr[curunit.dateformat], "%Y-%m-%d", sqlite3_column_text(ppStmtOneRecord,0));

	  gtk_entry_set_text(GTK_ENTRY(pui->entrydate),date);
	  free(date);
#else
	  getyearmonthday(sqlite3_column_text(ppStmtOneRecord,0), 
			  &year,
			  &month,
			  &day);
#if MAEMO_VERSION_MAJOR == 5
	  hildon_date_button_set_date(HILDON_DATE_BUTTON(pui->entrydate), year, month-1, day); /* Month is betweewn 0 and 11 */
#else
	  hildon_date_editor_set_date(HILDON_DATE_EDITOR(pui->entrydate), year, month, day);
#endif
#endif

	  g_snprintf(buf,BUFMAX,"%.1f",SIlength2user(sqlite3_column_double(ppStmtOneRecord,1)));
	  gtk_entry_set_text(GTK_ENTRY(pui->entrykm),buf);

	  g_snprintf(buf,BUFMAX,"%.1f",SIlength2user(sqlite3_column_double(ppStmtOneRecord,2)));
	  gtk_entry_set_text(GTK_ENTRY(pui->entrytrip),buf);

	  g_snprintf(buf,BUFMAX,"%.2f",SIvolume2user(sqlite3_column_double(ppStmtOneRecord,3)));
	  gtk_entry_set_text(GTK_ENTRY(pui->entryfill),buf);

	  if ( (sqlite3_column_double(ppStmtOneRecord,3)>0) && 
	       ( abs(sqlite3_column_double(ppStmtOneRecord,9))<1e-3) ) {
#if MAEMO_VERSION_MAJOR == 5
	    hildon_check_button_set_active(HILDON_CHECK_BUTTON (pui->buttonnotfull), TRUE);
#else
	    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (pui->buttonnotfull), TRUE);
#endif
	  }

	  g_snprintf(buf,BUFMAX,"%.2f",sqlite3_column_double(ppStmtOneRecord,4));
	  gtk_entry_set_text(GTK_ENTRY(pui->entryprice),buf);

	  g_snprintf(buf,BUFMAX,"%.2f",sqlite3_column_double(ppStmtOneRecord,5));
	  gtk_entry_set_text(GTK_ENTRY(pui->entryservice),buf);

	  g_snprintf(buf,BUFMAX,"%.2f",sqlite3_column_double(ppStmtOneRecord,6));
	  gtk_entry_set_text(GTK_ENTRY(pui->entryoil),buf);

	  g_snprintf(buf,BUFMAX,"%.2f",sqlite3_column_double(ppStmtOneRecord,7));
	  gtk_entry_set_text(GTK_ENTRY(pui->entrytires),buf);

	/* Convert notes to utf8 (earlier versions might have imported wrongly encoded data */
	  g_snprintf(notes, ENTRYNOTESMAX, "%s", sqlite3_column_text(ppStmtOneRecord,8));
	  if (!g_utf8_validate(notes, -1, NULL))
	    convnotes=g_convert(notes, -1, "UTF-8", "ISO-8859-15", NULL, NULL, NULL);
	  else
	    convnotes=g_strndup(notes, strlen(notes));
	  gtk_entry_set_text(GTK_ENTRY(pui->entrynotes),convnotes);
	  g_free(convnotes);

#if LIBHELP == 1
	  help_dialog_help_enable(GTK_DIALOG(dialog),
					 HELP_ID_ADDRECORD,
					 pui->app->osso);
#endif

/* 	  g_signal_connect_swapped (dialog, "response", */
/* 				    G_CALLBACK (edit_record_response), */
/* 				    dialog); */
	  g_signal_connect (dialog, "response",
			    G_CALLBACK (edit_record_response),
			    &ui);
	}
      }
      else {
	PDEBUG("Problem obtaining record\n");
      }
    }
    else {
      dialog = gtk_dialog_new_with_buttons(gettext(header[0]),
					   GTK_WINDOW(pui->app->mainwindow),
					   GTK_DIALOG_MODAL,
					   GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
					   NULL);

      label=gtk_label_new(_("Select a record first"));
      gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label, TRUE, TRUE, 5);
      gtk_widget_show(label);

      g_signal_connect_swapped (dialog, "response",
				G_CALLBACK (gtk_widget_destroy),
				dialog);
    }
    sqlite3_reset(ppStmtOneRecord);

  }
  else {
    dialog = gtk_dialog_new_with_buttons(gettext(header[0]),
					 GTK_WINDOW(pui->app->mainwindow),
					 GTK_DIALOG_MODAL,
					 GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
					 NULL);

    label=gtk_label_new(_("Can't access database - editing records not possible"));
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label, TRUE, TRUE, 5);
    gtk_widget_show(label);

    /* Destroy the dialog when the user responds to it (e.g. clicks a button) */
    g_signal_connect_swapped (dialog, "response",
			      G_CALLBACK (gtk_widget_destroy),
			      dialog);
  }

  gtk_widget_show(dialog);
}

void callback_recordactivated (GtkTreeView        *treeview,
                               GtkTreePath        *path,
                               GtkTreeViewColumn  *col,
                               AppUIData *pui)
{
  GtkTreeModel *model;
  GtkTreeIter   iter;

  model = gtk_tree_view_get_model(treeview);

  if (gtk_tree_model_get_iter(model, &iter, path)) {
    callback_editrecord(NULL, pui);
  }
}


/**
 * \fn void callback_deleterecord( GtkAction * action, AppUIData *pui )
 * \brief Callback for deleting a record
 *
 * This callback deletes the currently selected record. The user is
 * asked for a confirmation. If there is also a reminder event
 * associated with this record, it may be also optionally deleted.
 *
 * \todo This should be changed so that fuelpad.c has a function for
 * obtaining the needed data (i.e. sqlite would not be used here)
 *
 */
void callback_deleterecord( GtkAction * action, AppUIData *pui )
{
  GtkWidget *confirmdlg;
  GtkWidget *check;
  GtkWidget *separator;
  GtkTreeSelection *selection;
  GtkTreeIter iter;
  GtkTreeIter storeiter;
  GtkTreeIter fullstoreiter;
  GtkListStore *store;
  GtkListStore *fullstore;
  GtkTreeModel *model;
  gint64 id;
  gint confirm;
  gint numevents;
  gboolean delevent;
  gdouble km;
  gdouble fill;
  gdouble trip;
  gdouble consum;
  sqlite_int64 fullid;
  gdouble fullconsum;
  gdouble fullfill;
  gdouble fullkm;

  if (db_open()) { /* delete */

    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (pui->view));
    if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
      gtk_tree_model_get (model, &iter, COL_ID, &id, -1);

#if PLAINGTK == 0
      confirmdlg = hildon_note_new_confirmation(GTK_WINDOW(pui->app->mainwindow), 
						_("Are you sure you want to delete the current record"));
#else
      confirmdlg = gtk_message_dialog_new(GTK_WINDOW(pui->app->mainwindow),
					  GTK_DIALOG_MODAL,
					  GTK_MESSAGE_QUESTION,
					  GTK_BUTTONS_OK_CANCEL,
					  _("Are you sure you want to delete the current record"));
      gtk_widget_show(confirmdlg);
#endif
      numevents=db_get_numof_events_with_record(id);
      if (numevents>0) {
	separator=gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(confirmdlg)->vbox), separator, TRUE, TRUE, 0);
	gtk_widget_show(separator);

	check=gtk_check_button_new_with_label(_("Delete the corresponding reminder event"));
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON (check), TRUE);
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(confirmdlg)->vbox), check, TRUE, TRUE, 0);
	gtk_widget_show(check);
      }

      confirm = gtk_dialog_run(GTK_DIALOG(confirmdlg));
      if (numevents>0)
	delevent = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check));
      else
	delevent = FALSE;
      gtk_widget_destroy(GTK_WIDGET(confirmdlg));
    
      if(confirm == GTK_RESPONSE_OK) {

	/* 
	 * Well need to obtain the unmodified data to be excluded from the new 
	 * consumption calculations 
	 */
	if (SQLITE_OK == sqlite3_bind_int(ppStmtOneRecord, 1, id)) {
	  while (SQLITE_ROW == sqlite3_step(ppStmtOneRecord)) {
	    km=sqlite3_column_double(ppStmtOneRecord,1);
	    fill=sqlite3_column_double(ppStmtOneRecord,3);
	    trip=sqlite3_column_double(ppStmtOneRecord,2);
	    consum=sqlite3_column_double(ppStmtOneRecord,9);
	  }
	  sqlite3_reset(ppStmtOneRecord);
	}
	else
	  PDEBUG("Something wrong with ppStmtOneRecord\n");

	if (SQLITE_DONE != db_delete_record(id)) {
	  PDEBUG("Problem deleting the record");
	  sqlite3_reset(ppStmtDeleteRecord);
	}
	else { /* remove the record if succesfully removed from db */
	  sqlite3_reset(ppStmtDeleteRecord);
	  get_store_and_iter(model, pui->view, &iter, &store, &storeiter);

	  gtk_list_store_remove(store, &storeiter);

	  if (fill>0.0)
	    if (abs(consum)<1e-5) { /* Not full fill */ 

	      /* Find next full record */
	      fullid=db_find_next_full(km, &fullfill, &fullkm);
	      if (fullid>=0) {
		fullconsum=fullfill/fullkm*100;
		
		/* Update now the full record consum and tree view also */
		db_update_record(fullid, NULL, -1.0, -1.0, -1.0, fullconsum, -1.0, -1.0, -1.0, -1.0, -1.0, NULL);

		/* Update the data for the full fill */
		get_store_and_iter(NULL, pui->view, NULL, &fullstore, NULL);
		if (ui_find_iter(fullstore, &fullstoreiter,fullid)) {
		  ui_update_row_data(fullstore, &fullstoreiter,
				     NULL, /* date */
				     -1.0, /* km */
				     -1.0, /* trip */
				     -1.0, /* fill */
				     fullconsum, /* consum */
				     -1.0, /* price */
				     -1.0, /* price/trip */
				     -1.0, /* service */
				     -1.0, /* oil */
				     -1.0, /* tires */
				     NULL, /* notes */
				     fullid,
				     TRUE);
		}
		
	      }
	      else
		PDEBUG("Full fill not found\n");
	    }

	  update_totalkm(pui);

	  /* Delete the corresponding alarmevent */
	  if (numevents > 0 && delevent == TRUE) {
	    if (SQLITE_OK != sqlite3_bind_int(ppStmtDeleteEventwithRecordid, 1, id)
		||SQLITE_DONE != sqlite3_step(ppStmtDeleteEventwithRecordid)) {
	      PDEBUG("Problem deleting the event");
	    }
	    else { /* Alarm event deleted, we should update
		    the next event estimates if event window open*/
	      PDEBUG("Alarm event deleted\n");
	    }
	    sqlite3_reset(ppStmtDeleteEventwithRecordid);
	  }

	}
      } /* if (confirm == GTK_RESPONSE_OK) */
    }
    else
      PDEBUG("deleterecord: Something wrong with finding the selection\n");
  }
  else { /* nothing to delete */
    PDEBUG("Database not open, can't delete anything\n");
  }
}

#if LIBHELP == 1
/**
 * \fn void help_dialog_help_enable(GtkDialog *dialog, const gchar *topic, osso_context_t *osso)
 * \brief Help enable wrapper
 *
 * Wrapper for enabling dialog help.
 *
 */
void help_dialog_help_enable(GtkDialog *dialog,
				 const gchar *topic,
				 osso_context_t *osso)
{
#if HILDON == 1
  hildon_help_dialog_help_enable(dialog, topic, osso);
#else
  ossohelp_dialog_help_enable(dialog, topic, osso);
#endif
}
#endif

#if LIBHELP == 1
void help_activated(GtkWidget *win, gchar *help_id)
{
   osso_return_t retval;

   if (!help_id)
      return;

#if HILDON == 1
   retval = hildon_help_show(
      app.osso, help_id, 0);
#else
   retval = ossohelp_show(
      app.osso, help_id, 0);
#endif
}
#endif

#if LIBHELP == 1
void callback_help( GtkAction * action, gpointer data )
{
#if HILDON == 1
/*   hildon_help_show(app.osso, HELP_ID_INTRO, HILDON_HELP_SHOW_DIALOG); */
  hildon_help_show(app.osso, HELP_ID_INTRO, 0);
#else
  ossohelp_show(app.osso, HELP_ID_INTRO, 0);
#endif
}
#endif

void callback_about( GtkAction * action, gpointer data )
{
  GtkWidget *dialog;
  gchar *authors[] = {"Julius Luukko", NULL};

  dialog = gtk_about_dialog_new ();
  gtk_about_dialog_set_name(GTK_ABOUT_DIALOG(dialog), "Fuelpad");
  gtk_about_dialog_set_version(GTK_ABOUT_DIALOG(dialog), VERSION); 
  gtk_about_dialog_set_copyright(GTK_ABOUT_DIALOG(dialog), 
				 "Copyright (C) 2007-2010 Julius Luukko");

  gtk_about_dialog_set_license(GTK_ABOUT_DIALOG(dialog),
			       "Licensed under GNU GPL v. 2");
  gtk_about_dialog_set_website(GTK_ABOUT_DIALOG(dialog),
			       "http://fuelpad.garage.maemo.org/");
  gtk_about_dialog_set_authors(GTK_ABOUT_DIALOG(dialog),
			       &authors);
  gtk_about_dialog_set_translator_credits(GTK_ABOUT_DIALOG(dialog),
			       _("translator-credits"));
  gtk_dialog_run(GTK_DIALOG (dialog));
  gtk_widget_destroy(dialog);
}

/**
 * \fn void update_reminder_toolbutton(AppUIData *pui, gboolean warn)
 * \brief Display the reminder icon
 * \param warn If TRUE, display "no active reminders" icon, otherwise
 * "active reminders" icon
 *
 * This callback displays a different icon depending on the parameter warn.
 *
 */
void update_reminder_toolbutton(AppUIData *pui, gboolean warn)
{
  static GdkPixbuf *pixbufred=NULL;
  static GdkPixbuf *pixbufgreen=NULL;
  static GtkWidget *imagered=NULL;
  static GtkWidget *imagegreen=NULL;
  GtkWidget *image;

#if MAEMO_VERSION_MAJOR == 5
  hildon_button_set_value(HILDON_BUTTON(pui->mm_item_alarm),pui->warn ? _("Active") : _("Not active"));
#else
  if (pixbufred == NULL) {
    pixbufred = gdk_pixbuf_new_from_xpm_data((const char **)&exclamation_red_xpm);
  }
  if (pixbufgreen == NULL) {
    pixbufgreen = gdk_pixbuf_new_from_xpm_data((const char **)&exclamation_green_xpm);
  }
  if (!GTK_IS_WIDGET(imagered)) {
    imagered = gtk_image_new_from_pixbuf(pixbufred);
  }
  if (!GTK_IS_WIDGET(imagegreen)) {
    imagegreen = gtk_image_new_from_pixbuf(pixbufgreen);
  }

  if (warn)
    image = imagered;
  else
    image = imagegreen;

  gtk_tool_button_set_icon_widget(GTK_TOOL_BUTTON(pui->mtb_reminder),image);
  gtk_widget_show_all(GTK_WIDGET(pui->app->mainwindow));

  /* Show or hide toolbars */
  main_toolbar_show_hide(pui);
  secondary_toolbar_show_hide(pui);
#endif
}

#if PLAINGTK == 1
static void create_mainwin_toolbar(GtkWidget * main_window, GtkWidget *vbox, AppUIData *pui)
#else
  static void create_mainwin_toolbar(HildonWindow * main_window, AppUIData *pui)
#endif
{
  /* Create toolbar */
  pui->main_toolbar = gtk_toolbar_new();

  /* Create toolbar button items */
  pui->mtb_add = gtk_tool_button_new_from_stock(GTK_STOCK_ADD);
  pui->mtb_edit = gtk_tool_button_new_from_stock(GTK_STOCK_EDIT);
  pui->mtb_up = gtk_tool_button_new_from_stock(GTK_STOCK_GO_UP);
  pui->mtb_down = gtk_tool_button_new_from_stock(GTK_STOCK_GO_DOWN);
  pui->mtb_delete = gtk_tool_button_new_from_stock(GTK_STOCK_DELETE);
  pui->mtb_reminder = gtk_tool_button_new(NULL, NULL);
  pui->mtb_close = gtk_tool_button_new_from_stock(GTK_STOCK_QUIT);

  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->mtb_add),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->mtb_edit),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->mtb_up),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->mtb_down),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->mtb_delete),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->mtb_reminder),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->mtb_close),TRUE);
   
  /* Add all items to toolbar */
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), pui->mtb_add, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), gtk_separator_tool_item_new(), -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), pui->mtb_edit, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), gtk_separator_tool_item_new(), -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), pui->mtb_up, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), gtk_separator_tool_item_new(), -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), pui->mtb_down, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), gtk_separator_tool_item_new(), -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), pui->mtb_delete, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), gtk_separator_tool_item_new(), -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), pui->mtb_reminder, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), gtk_separator_tool_item_new(), -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->main_toolbar), pui->mtb_close, -1);

  g_signal_connect(G_OBJECT(pui->mtb_add), "clicked",
		   G_CALLBACK(callback_newrecord), pui);
  g_signal_connect(G_OBJECT(pui->mtb_edit), "clicked",
		   G_CALLBACK(callback_editrecord), pui);
  g_signal_connect(G_OBJECT(pui->mtb_up), "clicked",
		   G_CALLBACK(callback_prevrecord), pui);
  g_signal_connect(G_OBJECT(pui->mtb_down), "clicked",
		   G_CALLBACK(callback_nextrecord), pui);
  g_signal_connect(G_OBJECT(pui->mtb_delete), "clicked",
		   G_CALLBACK(callback_deleterecord), pui);
  g_signal_connect(G_OBJECT(pui->mtb_reminder), "clicked",
		   G_CALLBACK(callback_alarm), pui);
  g_signal_connect(G_OBJECT(pui->mtb_close), "clicked",
		   G_CALLBACK(delete_event), NULL);

  /* Add toolbar HildonWindow */
#if PLAINGTK == 1
  gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(pui->main_toolbar), FALSE, FALSE, 5);
#else
  hildon_window_add_toolbar(main_window, GTK_TOOLBAR(pui->main_toolbar));
#endif
}

#if PLAINGTK == 1
static void create_secondary_toolbar(GtkWidget * main_window, GtkWidget *vbox, AppUIData *pui)
#else
  static void create_secondary_toolbar(HildonWindow * main_window, AppUIData *pui)
#endif
{
  GString *str;
  float totalkm;
  float totalfillkm;
  float totalfill;

  PangoAttrList *attrs;
  PangoAttribute *attr;


  /* Create toolbar */
  pui->secondary_toolbar = gtk_toolbar_new();

  /* Create toolbar items */

  /* Scale texts smaller: create pango attributes */
  attrs = pango_attr_list_new();
  attr = pango_attr_scale_new(PANGO_SCALE_X_SMALL);
  attr->start_index = 0;
  attr->end_index = -1;
  pango_attr_list_insert(attrs, attr);

  /* Car combo */
  pui->stb_car = gtk_tool_item_new();
#if MAEMO_VERSION_MAJOR == 5
  pui->stb_carbutton = hildon_picker_button_new (HILDON_SIZE_AUTO, HILDON_BUTTON_ARRANGEMENT_VERTICAL);
  hildon_button_set_title (HILDON_BUTTON (pui->stb_carbutton), _("Car"));
  pui->stb_carcombo = create_car_combo(pui, pui->stb_carbutton);
#endif
#if MAEMO_VERSION_MAJOR == 5
  gtk_container_add(GTK_CONTAINER(pui->stb_car), GTK_WIDGET(pui->stb_carbutton));
#else
  pui->stb_carcombo = create_car_combo(pui);
  gtk_container_add(GTK_CONTAINER(pui->stb_car), GTK_WIDGET(pui->stb_carcombo));

  /* Apply car combo selection */
  pui->stb_carapply = gtk_tool_button_new_from_stock(GTK_STOCK_OK);
#endif

  str=g_string_new(NULL);

  /* Total distance */
  totalkm=db_get_totalkm(stbstattime);
  g_string_printf(str, "%.0f ", SIlength2user(totalkm));
  if (curunit.lengthunit==SI)
    g_string_append_printf(str, _("km"));
  else
    g_string_append_printf(str, _("miles"));
  pui->stb_totalkm = gtk_tool_item_new();
  pui->stb_totalkmlabel=gtk_label_new(str->str);
  gtk_label_set_selectable(GTK_LABEL(pui->stb_totalkmlabel), TRUE);
  gtk_label_set_attributes(GTK_LABEL(pui->stb_totalkmlabel), attrs);
  gtk_container_add(GTK_CONTAINER(pui->stb_totalkm), GTK_WIDGET(pui->stb_totalkmlabel));

  /* Average consumption */
  totalfill=db_get_totalfill(stbstattime);
  totalfillkm=db_get_totalfillkm(stbstattime);
  if (totalfillkm != 0.0)
    g_string_printf(str, "%.1f ",SIconsumption2user(totalfill/totalfillkm*100));
  else
    g_string_printf(str, "%.1f ",0.0);
  if (curunit.consumeunit==SI)
    g_string_append_printf(str, _("l/100 km"));
  else
    g_string_append_printf(str, _("MPG"));
  pui->stb_avgconsum = gtk_tool_item_new();
  pui->stb_avgconsumlabel=gtk_label_new(str->str);
  gtk_label_set_selectable(GTK_LABEL(pui->stb_avgconsumlabel), TRUE);
  gtk_label_set_attributes(GTK_LABEL(pui->stb_avgconsumlabel), attrs);
  gtk_container_add(GTK_CONTAINER(pui->stb_avgconsum), GTK_WIDGET(pui->stb_avgconsumlabel));

  /* Total cost */
  g_string_printf(str, "%.0f %s", db_get_totalcost(), curunit.currency->str);
  pui->stb_totcost = gtk_tool_item_new();
  pui->stb_totcostlabel=gtk_label_new(str->str);
  gtk_label_set_selectable(GTK_LABEL(pui->stb_totcostlabel), TRUE);
  gtk_label_set_attributes(GTK_LABEL(pui->stb_totcostlabel), attrs);
  gtk_container_add(GTK_CONTAINER(pui->stb_totcost), GTK_WIDGET(pui->stb_totcostlabel));

  g_string_free(str, TRUE);

  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->stb_car),FALSE);
#if MAEMO_VERSION_MAJOR == 5
#else
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->stb_carapply),FALSE);
#endif
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->stb_totalkm),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->stb_avgconsum),TRUE);
  gtk_tool_item_set_expand(GTK_TOOL_ITEM(pui->stb_totcost),TRUE);

  /* Add all items to toolbar */
  gtk_toolbar_insert(GTK_TOOLBAR(pui->secondary_toolbar), pui->stb_car, -1);
#if MAEMO_VERSION_MAJOR == 5
#else
  gtk_toolbar_insert(GTK_TOOLBAR(pui->secondary_toolbar), pui->stb_carapply, -1);
#endif
  gtk_toolbar_insert(GTK_TOOLBAR(pui->secondary_toolbar), pui->stb_totalkm, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->secondary_toolbar), pui->stb_avgconsum, -1);
  gtk_toolbar_insert(GTK_TOOLBAR(pui->secondary_toolbar), pui->stb_totcost, -1);

#if MAEMO_VERSION_MAJOR == 5
  g_signal_connect(G_OBJECT(pui->stb_carbutton), "value-changed",
		   G_CALLBACK(car_apply_cb), pui);
#else
  g_signal_connect(G_OBJECT(pui->stb_carapply), "clicked",
		   G_CALLBACK(car_apply_cb), pui);
#endif

  /* Add toolbar HildonWindow */
#if PLAINGTK == 1
  gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(pui->secondary_toolbar), FALSE, FALSE, 5);
#else
  hildon_window_add_toolbar(main_window, GTK_TOOLBAR(pui->secondary_toolbar));
#endif
}

void enable_mainmenu_items(AppUIData *pui)
{
  gboolean dbopen;

  dbopen=db_open();

#if MAEMO_VERSION_MAJOR == 5
#else  
  gtk_widget_set_sensitive(pui->mm_item_new, dbopen);
  gtk_widget_set_sensitive(pui->mm_item_edit, dbopen);
  gtk_widget_set_sensitive(pui->mm_item_delete, dbopen);
  gtk_widget_set_sensitive(pui->mm_item_monthly, dbopen);
  gtk_widget_set_sensitive(pui->mm_item_report, dbopen);
  gtk_widget_set_sensitive(pui->mm_item_close, dbopen);
  gtk_widget_set_sensitive(pui->mm_item_import, dbopen);
  gtk_widget_set_sensitive(pui->mm_item_export, dbopen);
#endif
  gtk_widget_set_sensitive(GTK_WIDGET(pui->mtb_add), dbopen);
  gtk_widget_set_sensitive(GTK_WIDGET(pui->mtb_edit), dbopen);
  gtk_widget_set_sensitive(GTK_WIDGET(pui->mtb_up), dbopen);
  gtk_widget_set_sensitive(GTK_WIDGET(pui->mtb_down), dbopen);
  gtk_widget_set_sensitive(GTK_WIDGET(pui->mtb_delete), dbopen);
}

void create_mainwin_menu(GtkWidget *root, AppUIData *pui)
{ 
#if MAEMO_VERSION_MAJOR == 5
  GtkWidget * button;

  pui->main_menu = HILDON_APP_MENU (hildon_app_menu_new ());

  pui->mm_item_new = hildon_button_new_with_text(HILDON_SIZE_AUTO,
						      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
						      _("Add record"),
						      NULL);
  hildon_app_menu_append (pui->main_menu, GTK_BUTTON (pui->mm_item_new));

  pui->mm_item_edit = hildon_button_new_with_text(HILDON_SIZE_AUTO,
						      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
						      _("Edit record"),
						      NULL);
  hildon_app_menu_append (pui->main_menu, GTK_BUTTON (pui->mm_item_edit));

  pui->mm_item_delete = hildon_button_new_with_text(HILDON_SIZE_AUTO,
						      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
						      _("Delete record"),
						      NULL);
  hildon_app_menu_append (pui->main_menu, GTK_BUTTON (pui->mm_item_delete));

  pui->mm_item_filter = hildon_button_new_with_text(HILDON_SIZE_AUTO,
						      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
						      _("Filter records"),
						      NULL);
  hildon_app_menu_append (pui->main_menu, GTK_BUTTON (pui->mm_item_filter));

  pui->mm_item_stat = hildon_button_new_with_text(HILDON_SIZE_AUTO,
						      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
						      _("Statistics"),
						      NULL);
  hildon_app_menu_append (pui->main_menu, GTK_BUTTON (pui->mm_item_stat));

  pui->mm_item_report = hildon_button_new_with_text(HILDON_SIZE_AUTO,
						      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
						      _("Report"),
						      NULL);
  hildon_app_menu_append (pui->main_menu, GTK_BUTTON (pui->mm_item_report));

  pui->mm_item_alarm = hildon_button_new_with_text(HILDON_SIZE_AUTO,
						      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
						      _("Reminders"),
						   pui->warn ? _("Active") : _("Not active"));
  hildon_app_menu_append (pui->main_menu, GTK_BUTTON (pui->mm_item_alarm));

  pui->mm_item_settings = hildon_button_new_with_text(HILDON_SIZE_AUTO,
						      HILDON_BUTTON_ARRANGEMENT_VERTICAL,
						      _("Settings"),
						      NULL);
  hildon_app_menu_append (pui->main_menu, GTK_BUTTON (pui->mm_item_settings));

  g_signal_connect(G_OBJECT(pui->mm_item_new), "clicked",
		   G_CALLBACK(callback_newrecord), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_edit), "clicked",
		   G_CALLBACK(callback_editrecord), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_alarm), "clicked",
		   G_CALLBACK(callback_alarm), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_settings), "clicked",
		   G_CALLBACK(callback_settings), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_delete), "clicked",
		   G_CALLBACK(callback_deleterecord), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_filter), "clicked",
		   G_CALLBACK(callback_filter_records), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_stat), "clicked",
		   G_CALLBACK(callback_stat), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_report), "clicked",
		   G_CALLBACK(callback_report), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_alarm), "clicked",
		   G_CALLBACK(callback_alarm), pui);

  hildon_window_set_app_menu(HILDON_WINDOW(pui->app->mainwindow), pui->main_menu);

#else
  GSList *radio_menu_group = NULL;

  pui->main_menu = gtk_menu_new();
  pui->mm_menu_db = gtk_menu_new();
  pui->mm_menu_record = gtk_menu_new();
  pui->mm_menu_stat = gtk_menu_new();
  pui->mm_menu_view = gtk_menu_new();
  pui->mm_menu_toolbar = gtk_menu_new();
  pui->mm_menu_fontsize = gtk_menu_new();

  /* Create main menu items */
  pui->mm_item_db = gtk_menu_item_new_with_label(_("Database"));
  pui->mm_item_record = gtk_menu_item_new_with_label(_("Record"));
  pui->mm_item_stat = gtk_menu_item_new_with_label(_("Statistics"));
  pui->mm_item_alarm = gtk_menu_item_new_with_label(_("Reminders..."));
  pui->mm_item_view = gtk_menu_item_new_with_label(_("View"));
  pui->mm_item_settings = gtk_menu_item_new_with_label(_("Settings..."));
#if LIBHELP == 1
  pui->mm_item_help = gtk_menu_item_new_with_label(_("Help"));
#endif
  pui->mm_item_about = gtk_menu_item_new_with_label(_("About"));
  pui->mm_item_exit = gtk_menu_item_new_with_label(_("Exit"));

  /* Create database menu items */
  pui->mm_item_open = gtk_menu_item_new_with_label(_("Open..."));
  pui->mm_item_close = gtk_menu_item_new_with_label(_("Close"));
  pui->mm_item_import = gtk_menu_item_new_with_label(_("Import..."));
  pui->mm_item_export = gtk_menu_item_new_with_label(_("Export..."));

  /* Create record menu items */
  pui->mm_item_new = gtk_menu_item_new_with_label(_("New"));
  pui->mm_item_edit = gtk_menu_item_new_with_label(_("Edit"));
  pui->mm_item_delete = gtk_menu_item_new_with_label(_("Delete"));

  /* Create statistics menu items */
  pui->mm_item_quick = gtk_menu_item_new_with_label(_("Quick"));
  pui->mm_item_monthly = gtk_menu_item_new_with_label(_("Graphical"));
  pui->mm_item_report = gtk_menu_item_new_with_label(_("Yearly report"));

  /* Create view menu items */
  pui->main_menu_item_fullscreen = gtk_check_menu_item_new_with_label(_("Full screen"));
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pui->main_menu_item_fullscreen),
				 pui->mainfullscreen);
  pui->mm_item_toolbar = gtk_menu_item_new_with_label(_("Toolbars"));
  pui->mm_item_fontsize = gtk_menu_item_new_with_label(_("Table font size"));
  pui->mm_item_columnselect = gtk_menu_item_new_with_label(_("Select columns..."));
  pui->mm_item_filter = gtk_menu_item_new_with_label(_("Filter records..."));

  /* Create toolbar menu items */
  pui->mm_item_toolbar_main = gtk_check_menu_item_new_with_label(_("Buttons"));
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pui->mm_item_toolbar_main),
				 main_toolbar_visible);
  pui->mm_item_toolbar_secondary = gtk_check_menu_item_new_with_label(_("Information"));
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pui->mm_item_toolbar_secondary),
				 secondary_toolbar_visible);

  /* Create fontsize menu items */
  pui->mm_item_fontsize_x_small = gtk_radio_menu_item_new_with_label(radio_menu_group,_("Extra small"));
  radio_menu_group = gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM (pui->mm_item_fontsize_x_small));
  if (fontsize == XSMALL)
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pui->mm_item_fontsize_x_small), TRUE);
  pui->mm_item_fontsize_small = gtk_radio_menu_item_new_with_label(radio_menu_group,_("Small"));
  radio_menu_group = gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM (pui->mm_item_fontsize_small));
  if (fontsize == SMALL)
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pui->mm_item_fontsize_small), TRUE);
  pui->mm_item_fontsize_medium = gtk_radio_menu_item_new_with_label(radio_menu_group,_("Medium"));
  radio_menu_group = gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM (pui->mm_item_fontsize_medium));
  if (fontsize == MEDIUM)
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pui->mm_item_fontsize_medium), TRUE);
  pui->mm_item_fontsize_large = gtk_radio_menu_item_new_with_label(radio_menu_group,_("Large"));
  if (fontsize == LARGE)
    gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(pui->mm_item_fontsize_large), TRUE);

  /* Add menu items to right menus */
  /* Main menu */
  gtk_menu_append(pui->main_menu, pui->mm_item_db);
  gtk_menu_append(pui->main_menu, pui->mm_item_record);
  gtk_menu_append(pui->main_menu, pui->mm_item_stat);
  gtk_menu_append(pui->main_menu, gtk_separator_menu_item_new());
  gtk_menu_append(pui->main_menu, pui->mm_item_alarm);
  gtk_menu_append(pui->main_menu, gtk_separator_menu_item_new());
  gtk_menu_append(pui->main_menu, pui->mm_item_view);
  gtk_menu_append(pui->main_menu, gtk_separator_menu_item_new());
  gtk_menu_append(pui->main_menu, pui->mm_item_settings);
  gtk_menu_append(pui->main_menu, gtk_separator_menu_item_new());
#if LIBHELP == 1
  gtk_menu_append(pui->main_menu, pui->mm_item_help);
#endif
  gtk_menu_append(pui->main_menu, pui->mm_item_about);
  gtk_menu_append(pui->main_menu, pui->mm_item_exit);

  /* Database menu */
  gtk_menu_append(pui->mm_menu_db, pui->mm_item_open);
  gtk_menu_append(pui->mm_menu_db, pui->mm_item_close);
  gtk_menu_append(pui->mm_menu_db, pui->mm_item_import);
  gtk_menu_append(pui->mm_menu_db, pui->mm_item_export);

  /* Record menu */
  gtk_menu_append(pui->mm_menu_record, pui->mm_item_new);
  gtk_menu_append(pui->mm_menu_record, pui->mm_item_edit);
  gtk_menu_append(pui->mm_menu_record, pui->mm_item_delete);

  /* Statistics menu */
  gtk_menu_append(pui->mm_menu_stat, pui->mm_item_quick);
  gtk_menu_append(pui->mm_menu_stat, pui->mm_item_monthly);
  gtk_menu_append(pui->mm_menu_stat, pui->mm_item_report);

  /* View menu */
  gtk_menu_append(pui->mm_menu_view, pui->main_menu_item_fullscreen);
  gtk_menu_append(pui->mm_menu_view, pui->mm_item_toolbar);
  gtk_menu_append(pui->mm_menu_view, pui->mm_item_fontsize);
  gtk_menu_append(pui->mm_menu_view, pui->mm_item_columnselect);
  gtk_menu_append(pui->mm_menu_view, pui->mm_item_filter);

  /* Toolbars menu */
  gtk_menu_append(pui->mm_menu_toolbar, pui->mm_item_toolbar_main);
  gtk_menu_append(pui->mm_menu_toolbar, pui->mm_item_toolbar_secondary);

  /* Font size menu */
  gtk_menu_append(pui->mm_menu_fontsize, pui->mm_item_fontsize_x_small);
  gtk_menu_append(pui->mm_menu_fontsize, pui->mm_item_fontsize_small);
  gtk_menu_append(pui->mm_menu_fontsize, pui->mm_item_fontsize_medium);
  gtk_menu_append(pui->mm_menu_fontsize, pui->mm_item_fontsize_large);

#if PLAINGTK == 1
  pui->mm_menubar = gtk_menu_bar_new();
  gtk_box_pack_start (GTK_BOX (root), pui->mm_menubar, FALSE, FALSE, 2);
  gtk_widget_show (pui->mm_menubar);
  pui->mm_item_file = gtk_menu_item_new_with_label ("fuelpad");
  gtk_widget_show (pui->mm_item_file);
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (pui->mm_item_file), pui->main_menu);
  gtk_menu_bar_append (GTK_MENU_BAR (pui->mm_menubar), pui->mm_item_file);
#else
  hildon_window_set_menu(HILDON_WINDOW(pui->app->mainwindow), GTK_MENU(pui->main_menu));
#endif
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(pui->mm_item_db), pui->mm_menu_db);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(pui->mm_item_record), pui->mm_menu_record);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(pui->mm_item_stat), pui->mm_menu_stat);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(pui->mm_item_view), pui->mm_menu_view);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(pui->mm_item_toolbar), pui->mm_menu_toolbar);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(pui->mm_item_fontsize), pui->mm_menu_fontsize);

  /* Attach the callback functions to the activate signal */
  g_signal_connect(G_OBJECT(pui->mm_item_alarm), "activate",
		   G_CALLBACK(callback_alarm), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_settings), "activate",
		   G_CALLBACK(callback_settings), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_quick), "activate",
		   G_CALLBACK(callback_quick_stat), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_monthly), "activate",
		   G_CALLBACK(callback_stat), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_report), "activate",
		   G_CALLBACK(callback_report), pui);
#if LIBHELP == 1
  g_signal_connect(G_OBJECT(pui->mm_item_help), "activate",
		   G_CALLBACK(callback_help), NULL);
#endif
  g_signal_connect(G_OBJECT(pui->mm_item_about), "activate",
		   G_CALLBACK(callback_about), NULL);
  g_signal_connect(G_OBJECT(pui->mm_item_exit), "activate",
		   G_CALLBACK(delete_event), NULL);

  g_signal_connect(G_OBJECT(pui->mm_item_open), "activate",
		   G_CALLBACK(callback_open_db), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_close), "activate",
		   G_CALLBACK(callback_close_db), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_import), "activate",
		   G_CALLBACK(callback_import), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_export), "activate",
		   G_CALLBACK(callback_export), NULL);

  g_signal_connect(G_OBJECT(pui->mm_item_new), "activate",
		   G_CALLBACK(callback_newrecord), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_edit), "activate",
		   G_CALLBACK(callback_editrecord), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_delete), "activate",
		   G_CALLBACK(callback_deleterecord), pui);

  g_signal_connect(G_OBJECT(pui->main_menu_item_fullscreen), "toggled",
		   G_CALLBACK(cb_main_fullscreen), (gpointer)pui);
  g_signal_connect(G_OBJECT(pui->mm_item_columnselect), "activate",
		   G_CALLBACK(callback_select_columns), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_filter), "activate",
		   G_CALLBACK(callback_filter_records), pui);

  g_signal_connect(G_OBJECT(pui->mm_item_toolbar_main), "toggled",
		   G_CALLBACK(cb_main_toolbar), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_toolbar_secondary), "toggled",
		   G_CALLBACK(cb_secondary_toolbar), pui);

  g_signal_connect(G_OBJECT(pui->mm_item_fontsize_x_small), "toggled",
			   G_CALLBACK(cb_fontsize_x_small), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_fontsize_small), "toggled",
			   G_CALLBACK(cb_fontsize_small), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_fontsize_medium), "toggled",
			   G_CALLBACK(cb_fontsize_medium), pui);
  g_signal_connect(G_OBJECT(pui->mm_item_fontsize_large), "toggled",
			   G_CALLBACK(cb_fontsize_large), pui);

#endif

  /* Make all menu widgets visible */
  gtk_widget_show_all(GTK_WIDGET(pui->main_menu));
}

void create_mainwin_widgets(AppUIData *pui)
{
  GtkWidget *vbox;

  pui->mainvbox = gtk_vbox_new (FALSE, 0);
  vbox = pui->mainvbox;
  gtk_container_add (GTK_CONTAINER (app.mainwindow), vbox);

  create_mainwin_menu(vbox, pui);

#if MAEMO_VERSION_MAJOR == 5
  pui->datascrollwin = hildon_pannable_area_new();
  g_object_set(G_OBJECT(pui->datascrollwin), "mov-mode", HILDON_MOVEMENT_MODE_BOTH, NULL);
#else
  pui->datascrollwin = gtk_scrolled_window_new(NULL, NULL);
#endif
  gtk_box_pack_start (GTK_BOX(vbox), pui->datascrollwin, TRUE, TRUE, 0);

  pui->view = create_view_and_model ();
  gtk_container_add (GTK_CONTAINER (pui->datascrollwin), pui->view);

#if PLAINGTK == 1
  create_mainwin_toolbar(app.mainwindow, vbox, pui);
#else
  create_mainwin_toolbar(app.mainwindow, pui);
#endif

#if PLAINGTK == 1
  create_secondary_toolbar(app.mainwindow, vbox, pui);
#else
  create_secondary_toolbar(app.mainwindow, pui);
#endif

  enable_mainmenu_items(pui);

  gtk_widget_show_all(GTK_WIDGET(vbox));

  /*----------------------------------------------------------------*/
}

void ui_create_mainwin(void)
{
#if PLAINGTK == 0
  osso_return_t ret;
#endif

#if PLAINGTK == 0
  app.osso = osso_initialize ("org.luukko.fuelpad", VERSION, TRUE, NULL);
  if (app.osso == NULL) {
    fprintf (stderr, "osso_initialize failed.\n");
    exit (1);
  }

/*   ret = osso_rpc_set_default_cb_f (app.osso, dbus_callback, NULL); */
/*   if (ret != OSSO_OK) */
/*     { */
/*       fprintf (stderr, "osso_rpc_set_default_cb_f failed: %d.\n", ret); */
/*       exit (1); */
/*     } */

    osso_hw_set_event_cb(app.osso, NULL, osso_hw_state_callback, NULL);

#if MAEMO_VERSION_MAJOR == 5

    if (osso_rpc_run_system(app.osso, MCE_SERVICE, MCE_REQUEST_PATH,
			    MCE_REQUEST_IF, MCE_ACCELEROMETER_ENABLE_REQ, NULL, DBUS_TYPE_INVALID) == OSSO_OK) {
      g_printerr("INFO: Accelerometer enabled\n");
    } else {
      g_printerr("WARN: Cannot enable accelerometer\n");
    }

    /* Connect to session bus, add a match rule, install filter callback */
    app.system_bus = dbus_bus_get (DBUS_BUS_SYSTEM, NULL);
    if (app.system_bus) {
      dbus_bus_add_match (app.system_bus, MCE_SIGNAL_MATCH, NULL);
      dbus_connection_add_filter (app.system_bus,
				  (DBusHandleMessageFunction) mce_filter_func,
				  &app, NULL);
    }
#endif
#endif

#if PLAINGTK == 0
  /* Create the hildon program and setup the title */
  app.program = HILDON_PROGRAM(hildon_program_get_instance());
  g_set_application_name("Fuelpad");

  /* Create HildonWindow and set it to HildonProgram */
#if MAEMO_VERSION_MAJOR == 5
  app.mainwindow = hildon_stackable_window_new();
#else
  app.mainwindow = HILDON_WINDOW(hildon_window_new());
#endif

  hildon_program_add_window(app.program, app.mainwindow);

  /* Link app data to ui data */
  ui.app = &app;

  /* Callback for hardware keys */
  g_signal_connect(G_OBJECT(app.mainwindow), "key_press_event",
		   G_CALLBACK(key_press_cb), &ui);

#else
  /* Create the main window */
  app.mainwindow = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (app.mainwindow), "fuelpad");

  /* Link app data to ui data */
  ui.app = &app;
#endif

  g_signal_connect (G_OBJECT (app.mainwindow), "delete_event",
		    G_CALLBACK (delete_event), NULL);


  create_mainwin_widgets(&ui);

  /* View the reminder system toolbar item */
  ui.alarmview = create_alarmview_and_model (&ui);
  ui.warn = update_next_event(gtk_tree_view_get_model(GTK_TREE_VIEW(ui.alarmview)));
  update_reminder_toolbutton (&ui, ui.warn);

  /* Begin the main application */
  gtk_widget_show_all(GTK_WIDGET(app.mainwindow));

  /* Show or hide toolbars */
  main_toolbar_show_hide(&ui);
  secondary_toolbar_show_hide(&ui);

}
