/* 
 * This file is part of fuelpad, the fuel diary
 *
 * Copyright (c) 2007, 2008 Julius Luukko <julle.luukko@quicknet.inet.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

/**
 *
 * \file
 * \brief  Main of fuelpad
 * \author Julius Luukko
 *
 */

#include <config.h>

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <time.h>

#include <locale.h>

#include <sqlite3.h>

#include "fuelpad.h"
#include "ui.h"

int debug_level;

/* local function prototypes */
void db_connect(void);

unsigned int currentdriver=1;
unsigned int currentcar=1;

sqlite3 *db;
#if PLAINGTK == 1
char default_db[] = "fuelpad.db";
#else
char default_db[] = DATABASE_FILENAME;
#endif
char *result_db;
sqlite3_stmt *ppStmtSaveRecord;
sqlite3_stmt *ppStmtDriver;
sqlite3_stmt *ppStmtCar;
sqlite3_stmt *ppStmtOneDriver;
sqlite3_stmt *ppStmtCurCar;
sqlite3_stmt *ppStmtOneCar;
sqlite3_stmt *ppStmtAddDriver;
sqlite3_stmt *ppStmtUpdateDriver;
sqlite3_stmt *ppStmtAddCar;
sqlite3_stmt *ppStmtUpdateCar;
sqlite3_stmt *ppStmtOneRecord;
sqlite3_stmt *ppStmtRecords;
sqlite3_stmt *ppStmtExport;
sqlite3_stmt *ppStmtExportCar;
sqlite3_stmt *ppStmtAddRecord;
sqlite3_stmt *ppStmtNextFull;
sqlite3_stmt *ppStmtPrevFull;
sqlite3_stmt *ppStmtDeleteRecord;
sqlite3_stmt *ppStmtUpdateRecord;
sqlite3_stmt *ppStmtGetYears;
sqlite3_stmt *ppStmtGetReport;
sqlite3_stmt *ppStmtAddAlarmtype;
sqlite3_stmt *ppStmtGetAlarmtype;
sqlite3_stmt *ppStmtGetOneAlarmtype;
sqlite3_stmt *ppStmtUpdateAlarmtype;
sqlite3_stmt *ppStmtAddEvent;
sqlite3_stmt *ppStmtGetEvents;
sqlite3_stmt *ppStmtGetOneEvent;
sqlite3_stmt *ppStmtDeleteEvent;
sqlite3_stmt *ppStmtDeleteEventwithRecordid;
sqlite3_stmt *ppStmtUpdateEvent;

/**
 * \fn double atof_l(const char *nptr, const char *locale)
 * \brief Locale independent atof
 * \param *nptr string to be converted
 * \param *locale locale used in conversion
 * \return String as numeric value
 *
 * Locale independent atof. Changes the locale to *locale, makes the
 * conversion and changes the locale back to original.
 *
 */
double atof_l(const char *nptr, const char *locale)
{
  char *currentlocale;
  char *psave;
  size_t len;
  double ret;

  currentlocale=setlocale(LC_NUMERIC, NULL);
  len=strlen(currentlocale);
  psave=(char *)malloc((len+1)*sizeof(char));
  strncpy(psave, currentlocale, len+1);

  setlocale(LC_NUMERIC, locale);
  ret=atof(nptr);

  setlocale(LC_NUMERIC, psave);
  free(psave);

  return ret;
}

/**
 * \fn void db_setfilename(char *fn)
 * \brief Set the database filename
 * \param *fn filename
 *
 * Sets the filename of current database
 *
 */
void db_setfilename(char *fn)
{
  size_t len;

  if (fn==NULL)
    fn=default_db;

  len=strlen(fn);
  result_db=(char *)realloc((void *)result_db, len+1);
  strncpy(result_db, fn, len+1);
}

/**
 * \fn char *db_getfilename(void)
 * \brief Gets the database filename
 * \return A pointer to the filename
 *
 * Gets the filename of current database.
 *
 */
char *db_getfilename(void)
{
  return result_db;
}

/**
 * \fn void db_connect(void)
 * \brief Opens the connection to sqlite3 database
 *
 * Opens the connection to sqlite3 database. If a database does not
 * exist, creates a new one. Leaves the database open.
 *
 */
void db_connect(void)
{
  char *zErrMsg = NULL;
  char **sqlresult;
  int sqlrows;
  int sqlcolumns;
  int rc;
  char *create_sql;
  int maxid;

  if (db) {
    sqlite3_close(db);
    db=NULL;
  }

  if (SQLITE_OK != (sqlite3_open(result_db,&db))) {
    PDEBUG("Problem opening result database.");
    sqlite3_close(db);
    db=NULL;
  }

  if (db) {
    /* Check if database does not exist */
    rc = sqlite3_get_table(db, "SELECT nickname FROM driver LIMIT 1",&sqlresult, &sqlrows, &sqlcolumns, NULL);
    sqlite3_free_table(sqlresult);
    if (rc != SQLITE_OK ) { /* create database if not found */
      create_sql = sqlite3_mprintf(
				   "CREATE TABLE driver ("
				   "id INTEGER PRIMARY KEY AUTOINCREMENT UNIQUE,"
				   "fullname TEXT,"
				   "nickname TEXT);"
				   "CREATE TABLE car ("
				   "id INTEGER PRIMARY KEY AUTOINCREMENT UNIQUE,"
				   "mark TEXT,"
				   "model TEXT,"
				   "year NUMBER,"
				   "register TEXT,"
				   "notes TEXT,"
				   "fueltype INTEGER);"
				   "CREATE TABLE record("
				   "id INTEGER PRIMARY KEY AUTOINCREMENT,"
				   "carid INTEGER,"
				   "driverid INTEGER,"
				   "day TIMESTAMP,"
				   "km REAL,"
				   "trip REAL,"
				   "fill REAL,"
				   "consum REAL,"
				   "price REAL,"
				   "priceperlitre REAL,"
				   "service REAL,"
				   "oil REAL,"
				   "tires REAL,"
				   "insurance REAL,"
				   "other REAL,"
				   "notes TEXT);"
				   "CREATE TABLE alarmtype ("
				   "id INTEGER PRIMARY KEY AUTOINCREMENT UNIQUE,"
				   "carid INTEGER,"
				   "shortdesc TEXT,"
				   "distance NUMBER,"
				   "interval INTEGER,"
				   "longdesc TEXT);"
				   "CREATE TABLE alarmevent ("
				   "id INTEGER PRIMARY KEY AUTOINCREMENT,"
				   "alarmid INTEGER,"
				   "carid INTEGER,"
				   "driverid INTEGER,"
				   "recordid INTEGER,"
				   "day TIMESTAMP,"
				   "km REAL);"
				   "INSERT INTO driver(fullname,nickname)"
				   "VALUES('Default Driver','Default');"
				   "INSERT INTO car(mark,model,year,register,notes)"
				   "VALUES('Default','Model',2007,'ABC-123','');"
				   );

      if (SQLITE_OK != sqlite3_exec(db, create_sql, NULL, NULL, &zErrMsg) &&
	  SQLITE_OK != sqlite3_get_table(db, "SELECT nickname FROM driver LIMIT 1",&sqlresult, &sqlrows, &sqlcolumns, NULL)) {
	PDEBUG("Failed to open or create result database\n");
	PDEBUG("%s\n",sqlite3_errmsg(db));
	sqlite3_close(db);
	db = NULL;
      }
      sqlite3_free(create_sql);
      sqlite3_free_table(sqlresult);
    }
    else { /* Check if database is from version 0.51 or below and in that case
	      create the alarm system tables */
      rc = sqlite3_get_table(db, "SELECT id FROM alarmtype LIMIT 1",&sqlresult, &sqlrows, &sqlcolumns, NULL);
      sqlite3_free_table(sqlresult);
      if (rc != SQLITE_OK ) { /* create new tables if not found */
	PDEBUG("Creating new alarm tables\n");
	create_sql = sqlite3_mprintf(
				   "CREATE TABLE alarmtype ("
				   "id INTEGER PRIMARY KEY AUTOINCREMENT UNIQUE,"
				   "carid INTEGER,"
				   "shortdesc TEXT,"
				   "distance NUMBER,"
				   "interval INTEGER,"
				   "longdesc TEXT);"
				   "CREATE TABLE alarmevent ("
				   "id INTEGER PRIMARY KEY AUTOINCREMENT,"
				   "alarmid INTEGER,"
				   "carid INTEGER,"
				   "driverid INTEGER,"
				   "recordid INTEGER,"
				   "day TIMESTAMP,"
				   "km REAL);"
				   );
	if (SQLITE_OK != sqlite3_exec(db, create_sql, NULL, NULL, &zErrMsg) &&
	    SQLITE_OK != sqlite3_get_table(db, "SELECT id FROM alarmtype LIMIT 1",&sqlresult, &sqlrows, &sqlcolumns, NULL)) {
	  PDEBUG("Failed to open or create database\n");
	  PDEBUG("%s\n",sqlite3_errmsg(db));
	  sqlite3_close(db);
	  db = NULL;
      }
      sqlite3_free(create_sql);
      sqlite3_free_table(sqlresult);
      }

      /* Check if the database contains the fueltype column (0.7 and before did not) */
      rc = sqlite3_get_table(db, "SELECT fueltype FROM car LIMIT 1",&sqlresult, &sqlrows, &sqlcolumns, NULL);
      sqlite3_free_table(sqlresult);
      if (rc != SQLITE_OK ) { /* create new column if not found */
	PDEBUG("Creating new fueltype column to table car\n");
	create_sql = sqlite3_mprintf(
				     "ALTER TABLE car "
				     "ADD COLUMN fueltype INTEGER;"
				     );
	if (SQLITE_OK != sqlite3_exec(db, create_sql, NULL, NULL, &zErrMsg) &&
	    SQLITE_OK != sqlite3_get_table(db, "SELECT fueltype FROM car LIMIT 1",&sqlresult, &sqlrows, &sqlcolumns, NULL)) {
	  PDEBUG("Failed to add fueltype column to database\n");
	  PDEBUG("%s\n",sqlite3_errmsg(db));
	  sqlite3_close(db);
	  db = NULL;
	}
	sqlite3_free(create_sql);
	sqlite3_free_table(sqlresult);
      }
    }

    if (db) {
      /* maximum carid and driverid */
      rc = sqlite3_get_table(db, "SELECT max(id) FROM car", &sqlresult, &sqlrows, &sqlcolumns, NULL);
      if (rc==SQLITE_OK) {
	if ( (sqlcolumns*(sqlrows+1))> 0) {
	  maxid=atoi(sqlresult[sqlcolumns*(sqlrows+1)-1]);
	  if (currentcar > maxid)
	    currentcar=maxid;
	}
	else
	  currentcar=1;
	sqlite3_free_table(sqlresult);
      }
      else
	currentcar=1;

      rc = sqlite3_get_table(db, "SELECT max(id) FROM driver", &sqlresult, &sqlrows, &sqlcolumns, NULL);
      if (rc==SQLITE_OK) {
	if ( (sqlcolumns*(sqlrows+1))> 0) {
	  maxid=atoi(sqlresult[sqlcolumns*(sqlrows+1)-1]);
	  if (currentdriver > maxid)
	    currentdriver=maxid;
	}
	else
	  currentdriver=1;
	sqlite3_free_table(sqlresult);
      }
      else
	currentdriver=1;

      /* prepare sql query for saving the result */
      rc = sqlite3_prepare_v2(db,"INSERT INTO record(carid,driverid,day,km,trip,fill,consum,price,priceperlitre,service,oil,tires,insurance,other,notes) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?);",-1,&ppStmtSaveRecord, NULL);

      /* prepare sql query for driver and car selection dialog */
      rc = sqlite3_prepare_v2(db, "SELECT nickname,fullname,id FROM driver;", -1, &ppStmtDriver, NULL);
      rc = sqlite3_prepare_v2(db, "SELECT mark,register,id FROM car;", -1, &ppStmtCar, NULL);

      /* prepare sql query for obtaining one driver data  */
      rc = sqlite3_prepare_v2(db, "SELECT fullname,nickname FROM driver WHERE id=?;", -1, &ppStmtOneDriver, NULL);

      /* prepare sql query for querying the current car */
      rc = sqlite3_prepare_v2(db, "SELECT mark,model,register,year FROM car where id=?;", -1, &ppStmtCurCar, NULL);

      /* prepare sql query for querying the current car */
      rc = sqlite3_prepare_v2(db, "SELECT mark,model,year,register,notes,fueltype FROM car where id=?;", -1, &ppStmtOneCar, NULL);

      /* prepare sql query for adding a new driver */
      rc = sqlite3_prepare_v2(db, "INSERT INTO driver(fullname,nickname) VALUES(?,?);", -1, &ppStmtAddDriver, NULL);

      /* prepare sql query for changing car properties */
      rc = sqlite3_prepare_v2(db, 
"UPDATE driver "
"SET fullname=?, nickname=? WHERE id=?;",-1, &ppStmtUpdateDriver, NULL);

      /* prepare sql query for adding a new car */
      rc = sqlite3_prepare_v2(db, "INSERT INTO car(mark,model,year,register,notes,fueltype) VALUES(?,?,?,?,?,?);", -1, &ppStmtAddCar, NULL);

      /* prepare sql query for changing car properties */
      rc = sqlite3_prepare_v2(db, 
"UPDATE car "
"SET mark=?, model=?, year=?, register=?, notes=?, fueltype=? WHERE id=?;",-1, &ppStmtUpdateCar, NULL);

      /* prepare sql query for deleting a record */
      rc = sqlite3_prepare_v2(db, "DELETE FROM record WHERE id=?;", -1, &ppStmtDeleteRecord, NULL);

      /* prepare sql query for obtaining one record */
      rc = sqlite3_prepare_v2(db, 
"SELECT day,km,trip,fill,price,service,oil,tires,notes,consum,id "
"FROM record WHERE id=?;", -1, &ppStmtOneRecord, NULL);

      /* prepare sql query for obtaining all the records */
      rc = sqlite3_prepare_v2(db, 
"SELECT day,km,trip,fill,consum,price,"
"priceperlitre,service,oil,tires,notes,"
"id FROM record WHERE carid=? ORDER BY km;", -1, &ppStmtRecords, NULL);

      /* prepare sql query for obtaining all the records (more fields
         for export functionality) */
      rc = sqlite3_prepare_v2(db, 
"SELECT day,km,trip,fill,consum,price,"
"priceperlitre,service,oil,tires,notes,"
"driverid FROM record WHERE carid=? ORDER BY km;", -1, &ppStmtExport, NULL);

      rc = sqlite3_prepare_v2(db, "SELECT mark,model,year,register,notes FROM car WHERE id=?;", -1, &ppStmtExportCar, NULL);

      /* prepare sql query for adding a record */
      rc = sqlite3_prepare_v2(db, 
"INSERT "
"INTO record(carid,driverid,day,km,trip,fill,consum,price,priceperlitre,service,oil,tires,notes) "
"VALUES(?,?,?,?,?,?,?,?,?,?,?,?,?);", -1, &ppStmtAddRecord, NULL);

      /* prepare sql query for calculating the full fill (fill and total km) */
      rc = sqlite3_prepare_v2(db, 
"SELECT "
"km,trip,fill,consum,id FROM record WHERE carid=? AND km>? AND fill>0 AND consum>0 ORDER BY km LIMIT 1;", 
			      -1, &ppStmtNextFull, NULL);

      /* prepare sql query for calculating the full fill (fill and total km) */
      rc = sqlite3_prepare_v2(db, 
"SELECT "
"km,trip,fill,consum,id FROM record WHERE carid=? AND km<? AND fill>0 ORDER BY km DESC;", 
			      -1, &ppStmtPrevFull, NULL);

      /* prepare sql query for changing a record */
      rc = sqlite3_prepare_v2(db, 
"UPDATE record "
"SET day=?, km=?, trip=?, fill=?, consum=?, price=?, priceperlitre=?, service=?, oil=?, tires=?, notes=? WHERE id=?;", -1, &ppStmtUpdateRecord, NULL);

      /* Prepare sql query for obtaining the years that contain data */
      rc = sqlite3_prepare_v2(db,
"SELECT DISTINCT STRFTIME('%Y',date(day)) FROM record WHERE carid=?;", -1,
			      &ppStmtGetYears, NULL);

      /* prepare sql query for adding an alarmtype */
      rc = sqlite3_prepare_v2(db,
"INSERT "
"INTO alarmtype(carid,shortdesc,distance,interval,longdesc) "
				"VALUES(?,?,?,?,?);", -1, &ppStmtAddAlarmtype, NULL);

      /* prepare sql query for obtaining alarmtypes */
      rc = sqlite3_prepare_v2(db,
"SELECT shortdesc,distance,interval,"
"id FROM alarmtype WHERE carid=?;", -1, &ppStmtGetAlarmtype, NULL);

      /* prepare sql query for obtaining one alarmtype */
      rc = sqlite3_prepare_v2(db,
"SELECT shortdesc,distance,interval,"
"id FROM alarmtype WHERE id=?;", -1, &ppStmtGetOneAlarmtype, NULL);

      /* prepare sql query for updating one alarmtype */
      rc = sqlite3_prepare_v2(db,
"UPDATE alarmtype "
"SET shortdesc=?, distance=?, interval=? WHERE id=?", -1, &ppStmtUpdateAlarmtype, NULL);

      /* prepare sql query for adding an alarm event */
      rc = sqlite3_prepare_v2(db,
"INSERT "
"INTO alarmevent(alarmid,carid,driverid,recordid,day,km) "
			      "VALUES(?,?,?,?,?,?);", -1, &ppStmtAddEvent, NULL);

      /* prepare sql query for changing an alarm event */
      rc = sqlite3_prepare_v2(db, 
"UPDATE alarmevent "
"SET day=?, km=? WHERE id=?;", -1, &ppStmtUpdateEvent, NULL);


      /* prepare sql query for obtaining alarm events */
      rc = sqlite3_prepare_v2(db,
"SELECT day,km,id,recordid FROM alarmevent WHERE alarmid=?;",
			      -1, &ppStmtGetEvents, NULL);

      /* prepare sql query for obtaining one alarm event */
      rc = sqlite3_prepare_v2(db,
"SELECT day,km,recordid FROM alarmevent WHERE id=?;",
			      -1, &ppStmtGetOneEvent, NULL);

      /* prepare sql query for deleting an event */
      rc = sqlite3_prepare_v2(db, "DELETE FROM alarmevent WHERE id=?;", -1, &ppStmtDeleteEvent, NULL);

      /* prepare sql query for deleting an event with a given recordid */
      rc = sqlite3_prepare_v2(db, "DELETE FROM alarmevent WHERE recordid=?;", -1, &ppStmtDeleteEventwithRecordid, NULL);

    }
    /*   sqlite3_close(db); */ /* leave db open! */ 
  }
}  

/**
 * \fn void db_close(void)
 * \brief Closes the connection to sqlite3 database
 *
 */
void db_close(void)
{
  sqlite3_finalize(ppStmtSaveRecord);
  sqlite3_finalize(ppStmtDriver);
  sqlite3_finalize(ppStmtCar);
  sqlite3_finalize(ppStmtOneDriver);
  sqlite3_finalize(ppStmtCurCar);
  sqlite3_finalize(ppStmtOneCar);
  sqlite3_finalize(ppStmtAddDriver);
  sqlite3_finalize(ppStmtUpdateDriver);
  sqlite3_finalize(ppStmtAddCar);
  sqlite3_finalize(ppStmtUpdateCar);
  sqlite3_finalize(ppStmtDeleteRecord);
  sqlite3_finalize(ppStmtOneRecord);
  sqlite3_finalize(ppStmtRecords);
  sqlite3_finalize(ppStmtExport);
  sqlite3_finalize(ppStmtExportCar);
  sqlite3_finalize(ppStmtAddRecord);
  sqlite3_finalize(ppStmtNextFull);
  sqlite3_finalize(ppStmtPrevFull);
  sqlite3_finalize(ppStmtUpdateRecord);
  sqlite3_finalize(ppStmtGetYears);
  sqlite3_finalize(ppStmtGetReport);
  sqlite3_finalize(ppStmtAddAlarmtype);
  sqlite3_finalize(ppStmtGetAlarmtype);
  sqlite3_finalize(ppStmtAddEvent);
  sqlite3_finalize(ppStmtGetEvents);
  sqlite3_finalize(ppStmtGetOneEvent);
  sqlite3_finalize(ppStmtDeleteEvent);
  sqlite3_finalize(ppStmtDeleteEventwithRecordid);
  sqlite3_close(db);
  db = NULL;
}

/**
 * \fn int db_open(void)
 * \brief Tells if the database is open
 * \return True if open, false if not
 */
int db_open(void)
{
  return db != NULL;
}

/**
 * \fn float db_get_fueltype(int carid)
 * \brief Finds the fueltype of car
 * \return Fueltype of car (0 = petrol, 1 = diesel)
 */
float db_get_fueltype(int carid)
{
  char **sqlresult;
  char *sqlquery;
  int sqlrows;
  int sqlcolumns;
  int rc;
  float ret;

  ret=0.0;
  if (db) {
    sqlquery = sqlite3_mprintf("SELECT fueltype FROM car WHERE id=%d;",carid);
    rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
/*    sqlite3_free((void *)sqlquery); */
    sqlite3_free(sqlquery);
    if (rc==SQLITE_OK) {
      if (sqlcolumns*(sqlrows+1)>0) {
	if (sqlresult[sqlcolumns*(sqlrows+1)-1] != NULL)
	  ret=atof_l(sqlresult[sqlcolumns*(sqlrows+1)-1],"C");
      }
      sqlite3_free_table(sqlresult);
    }
  }

  return ret;
}

/**
 * \fn float get_emission_per_litre(int fueltype)
 * \brief Returns the CO2 emission in g/l
 * \param fueltype 0 for petrol, 1 for diesel
 * \return CO2 emission in g/l
 *
 */
float get_emission_per_litre(int fueltype)
{
  float emissionperlitre[]={2350.0, 2660.0};

  return emissionperlitre[fueltype];

}
/**
 * \fn float calc_co2_emission(float fill, float trip, int fueltype)
 * \brief Returns the CO2 emission in g/km
 * \param fill The amount of fuel filled
 * \param trip The distance driven
 * \param carid The car id
 * \return CO2 emission in g/km
 *
 */
float calc_co2_emission(float consum, int carid)
{
  float result;
  int fueltype;

  fueltype = db_get_fueltype(carid);
  if (fueltype<0) fueltype=0;
  if (fueltype>1) fueltype=1;

  if (consum>1e-5)
    result=get_emission_per_litre(fueltype)*consum/100.0;
  else
    result=0.0;
  return result;
}

char *fields[]={"day", "km", "trip", "fill", "consum", "price", 
		"price/trip", "priceperlitre", 
		"service", "oil", "tires", "fill/trip", "notes"};

/**
 * \fn float db_get_max(int fieldnum)
 * \brief Finds the maximum value of given field
 * \return Maximum value of the given field
 */
float db_get_max(int fieldnum)
{
  char **sqlresult;
  char *sqlquery;
  int sqlrows;
  int sqlcolumns;
  int rc;
  float ret;

  ret=0.0;
  if (fieldnum<sizeof(fields)/sizeof(char *)) {
    if (db) {
      sqlquery = sqlite3_mprintf("SELECT max(%s) FROM record WHERE carid=%d;",fields[fieldnum],currentcar);
      rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
      /*    sqlite3_free((void *)sqlquery); */
      sqlite3_free(sqlquery);
      if (rc==SQLITE_OK) {
	if (sqlcolumns*(sqlrows+1)>0) {
	  if (sqlresult[sqlcolumns*(sqlrows+1)-1] != NULL)
	    ret=atof_l(sqlresult[sqlcolumns*(sqlrows+1)-1],"C");
	}
	sqlite3_free_table(sqlresult);
      }
    }
  }

  return ret;
}

/**
 * \fn void db_get_max_date(char **date)
 * \brief Finds the maximum value of date
 * \param **date buffer for the maximum value of date
 *
 * Finds the maximum of value dates in the database for
 * currentdriver. The date buffer is allocated and must be freed after
 * usage.
 *
 */
void db_get_max_date(char **date)
{
  char **sqlresult;
  char *sqlquery;
  int sqlrows;
  int sqlcolumns;
  int rc;
  int len;

  *date=NULL;
  if (db) {
    sqlquery = sqlite3_mprintf("SELECT max(day) FROM record WHERE carid=%d;",currentcar);
    rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
    /*    sqlite3_free((void *)sqlquery); */
    sqlite3_free(sqlquery);
    if (rc==SQLITE_OK) {
      if (sqlcolumns*(sqlrows+1)>0) {
	if (sqlresult[sqlcolumns*(sqlrows+1)-1] != NULL) {
	  len=strlen(sqlresult[sqlcolumns*(sqlrows+1)-1]);
	  *date=(char *)malloc((len+1)*sizeof(char));
	  strncpy(*date, sqlresult[sqlcolumns*(sqlrows+1)-1],len+1);
	}
      }
      sqlite3_free_table(sqlresult);
    }
  }
}

/**
 * \fn float db_get_min(int fieldnum)
 * \brief Finds the minimum value of given field
 * \param fieldnum the field number
 * \return Minimum value of the given field
 */
float db_get_min(int fieldnum)
{
  char **sqlresult;
  char *sqlquery;
  int sqlrows;
  int sqlcolumns;
  int rc;
  float ret;

  ret=0.0;
  if (fieldnum<sizeof(fields)/sizeof(char *)) {
    if (db) {
      sqlquery = sqlite3_mprintf("SELECT min(%s) FROM record WHERE carid=%d;",fields[fieldnum],currentcar);
      rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
      /*    sqlite3_free((void *)sqlquery); */
      sqlite3_free(sqlquery);
      if (rc==SQLITE_OK) {
	if (sqlcolumns*(sqlrows+1)>0) {
	  if (sqlresult[sqlcolumns*(sqlrows+1)-1] != NULL)
	    ret=atof_l(sqlresult[sqlcolumns*(sqlrows+1)-1],"C");
	}
	sqlite3_free_table(sqlresult);
      }
    }
  }

  return ret;
}

/**
 * \fn void db_get_min_date(char **date)
 * \brief Finds the minimum value of date
 * \param **date buffer for the maximum value of date
 *
 * Finds the minimum of value dates in the database for
 * currentdriver. The date buffer is allocated and must be freed after
 * usage.
 *
 */
void db_get_min_date(char **date)
{
  char **sqlresult;
  char *sqlquery;
  int sqlrows;
  int sqlcolumns;
  int rc;
  int len;

  *date=NULL;
  if (db) {
    sqlquery = sqlite3_mprintf("SELECT min(day) FROM record WHERE carid=%d;",currentcar);
    rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
    /*    sqlite3_free((void *)sqlquery); */
    sqlite3_free(sqlquery);
    if (rc==SQLITE_OK) {
      if (sqlcolumns*(sqlrows+1)>0) {
	if (sqlresult[sqlcolumns*(sqlrows+1)-1] != NULL) {
	  len=strlen(sqlresult[sqlcolumns*(sqlrows+1)-1]);
	  *date=(char *)malloc((len+1)*sizeof(char));
	  strncpy(*date, sqlresult[sqlcolumns*(sqlrows+1)-1],len+1);
	}
      }
      sqlite3_free_table(sqlresult);
    }
  }
}

/**
 * \fn float db_get_last_km(void)
 * \brief Finds the km of last record
 * \return Last km in SI units
 */
float db_get_last_km(void)
{
  char **sqlresult;
  char *sqlquery;
  int sqlrows;
  int sqlcolumns;
  int rc;
  float ret;

  ret=0.0;
 if (db) {
   sqlquery = sqlite3_mprintf("SELECT max(km) FROM record WHERE carid=%d;",currentcar);
   rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
/*    sqlite3_free((void *)sqlquery); */
   sqlite3_free(sqlquery);
   if (rc==SQLITE_OK) {
     if (sqlcolumns*(sqlrows+1)>0) {
       if (sqlresult[sqlcolumns*(sqlrows+1)-1] != NULL)
	 ret=atof_l(sqlresult[sqlcolumns*(sqlrows+1)-1],"C");
     }
     sqlite3_free_table(sqlresult);
   }
 }

 return ret;
}

/**
 * \fn float db_get_last_refill(double newkm)
 * \brief Finds the km of last refill
 * \param newkm the km of the new record
 * \return Last refill km in SI units
 */
float db_get_last_refill(double newkm)
{
  char **sqlresult;
  char *sqlquery;
  int sqlrows;
  int sqlcolumns;
  int rc;
  float ret;

  ret=0.0;
 if (db) {
   sqlquery = sqlite3_mprintf("SELECT km FROM record WHERE carid=%d AND trip>0 AND km<%f ORDER BY km DESC LIMIT 1;",currentcar,newkm);
   rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
/*    sqlite3_free((void *)sqlquery); */
   sqlite3_free(sqlquery);
   if (rc==SQLITE_OK) {
     if (sqlcolumns*(sqlrows+1)>0) {
       if (sqlresult[sqlcolumns*(sqlrows+1)-1] != NULL)
	 ret=atof_l(sqlresult[sqlcolumns*(sqlrows+1)-1],"C");
     }
     sqlite3_free_table(sqlresult);
   }
 }

 return ret;
}

void db_find_prev_full(double km, double *fullfill, double *fullkm)
{
/*   char **sqlresult; */
/*   char *sqlquery; */
/*   int sqlrows; */
/*   int sqlcolumns; */
/*   int rc; */
  double fill;
  double trip;
  double consum;
  
  *fullfill=0.0;
  *fullkm=0.0;
  if (db) {
    if (!(SQLITE_OK != sqlite3_bind_int(ppStmtPrevFull, 1, currentcar) /* carid */
	  || SQLITE_OK != sqlite3_bind_double(ppStmtPrevFull, 2, km)) ) {
	  while (SQLITE_ROW == sqlite3_step(ppStmtPrevFull)) { /* full tank record succesfully found */
	    trip=sqlite3_column_double(ppStmtPrevFull,1);
	    fill=sqlite3_column_double(ppStmtPrevFull,2);
	    consum=sqlite3_column_double(ppStmtPrevFull,3);
	    if (abs(consum)>1e-3) /* Full fill found*/
	      break;
	    else {
	      *fullfill += fill;
	      *fullkm += trip;
	    }
	
	  }
	  sqlite3_reset(ppStmtPrevFull);
	}
  }
}

sqlite_int64 db_find_next_full(double km, double *fullfill, double *fullkm)
{
  sqlite_int64 id;
  double fill;
  double trip;

  id=-1;
  if (db) {
    if (!(SQLITE_OK != sqlite3_bind_int(ppStmtNextFull, 1, currentcar) /* carid */
	  || SQLITE_OK != sqlite3_bind_double(ppStmtNextFull, 2, km))) {
      if(SQLITE_ROW == sqlite3_step(ppStmtNextFull)) {
	km = sqlite3_column_double(ppStmtNextFull,0);
	trip = sqlite3_column_double(ppStmtNextFull,1);
	fill = sqlite3_column_double(ppStmtNextFull,2);
	id = sqlite3_column_int64(ppStmtNextFull,4);
	sqlite3_reset(ppStmtNextFull);
	db_find_prev_full(km,fullfill,fullkm);
	*fullfill += fill;
	*fullkm += trip;
      }
      else
	sqlite3_reset(ppStmtNextFull);
    }
  }
  return id;
}

int db_create_fillview(void)
{
  char **sqlresult;
  char *sqlquery;
  int sqlrows;
  int sqlcolumns;
  int rc;

  sqlquery = sqlite3_mprintf("CREATE TEMP VIEW fillview AS SELECT * FROM record WHERE carid=%d ORDER BY km LIMIT -1 OFFSET 1;",currentcar);
  rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
  sqlite3_free((void *)sqlquery);
  sqlite3_free_table(sqlresult);

  return rc;
}

void db_drop_fillview(void)
{
  char **sqlresult;
  int sqlrows;
  int sqlcolumns;
  int rc;

  rc = sqlite3_get_table(db, "DROP VIEW fillview;", &sqlresult, &sqlrows, &sqlcolumns, NULL);
  if (rc==SQLITE_OK)
    sqlite3_free_table(sqlresult);
}

/**
 * \fn int db_get_monthly_data(int year, double **x, double **fill, double **trip, double **consum, double **priceperlitre)
 * \brief Get the monthly data of a given year
 * \param year The year the data is requested
 * \param **x   Pointer to month data
 * \param **fill Pointer to monthly fill data
 * \param **trip Pointer to monthly trip data
 * \param **consum Pointer to monthly consumption data
 * \param **priceperlitre Pointer to monthly priceperlitre data
 * \return Number of sql rows obtained
 *
 * Tables are allocated in this function and must be freed after usage.
 *
 */
int db_get_monthly_data(int year, double **x, double **fill, double **trip, double **consum, double **priceperlitre)
{
  int i;
  char **sqlresult;
  char *sqlquery;
  int sqlrows;
  int sqlcolumns;
  int rc;
  int resultsize=0;

  *x=NULL;
  *fill=NULL;
  *trip=NULL;
  *consum=NULL;
  *priceperlitre=NULL;
  if (db) {
    rc = db_create_fillview();
    if (rc==SQLITE_OK) {
      // select * from fillview where day between '2007-01-01' and '2007-12-31';
      // SELECT STRFTIME(\"%m\",DATE(day)),STRFTIME(\"%Y\",DATE(day)),SUM(fill),SUM(trip),SUM(fill)/SUM(trip)*100 FROM fillview WHERE carid=2 AND STRFTIME(\"%Y\",DATE(day))=='2007' GROUP BY STRFTIME(\"%m\",day);
      sqlquery = sqlite3_mprintf("SELECT STRFTIME(\"%s\",DATE(day)),STRFTIME(\"%s\",DATE(day)),SUM(fill),SUM(trip),SUM(fill)/SUM(trip)*100,SUM(price)/SUM(fill) FROM fillview WHERE fill > 0 AND STRFTIME(\"%s\",DATE(day))=='%d' GROUP BY STRFTIME(\"%s\",day);","%m","%Y","%Y",year,"%m");
      rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
      sqlite3_free((void *)sqlquery);
      if (rc==SQLITE_OK) {
	if (sqlresult[sqlcolumns*(sqlrows+1)-1] != NULL) {
	  resultsize=sqlrows;
	  *x=(double *)malloc((resultsize)*sizeof(double));
	  *fill=(double *)malloc((resultsize)*sizeof(double));
	  *trip=(double *)malloc((resultsize)*sizeof(double));
	  *consum=(double *)malloc((resultsize)*sizeof(double));
	  *priceperlitre=(double *)malloc((resultsize)*sizeof(double));
	  for (i=1;i<=sqlrows;i++) {
	    (*x)[i-1]=atof_l(sqlresult[i*sqlcolumns],"C");
	    (*fill)[i-1]=atof_l(sqlresult[i*sqlcolumns+2],"C");
	    (*trip)[i-1]=atof_l(sqlresult[i*sqlcolumns+3],"C");
	    (*consum)[i-1]=atof_l(sqlresult[i*sqlcolumns+4],"C");
	    (*priceperlitre)[i-1]=atof_l(sqlresult[i*sqlcolumns+5],"C");
	  }
	}
	else
	  PDEBUG("No data found\n");
	sqlite3_free_table(sqlresult);

	db_drop_fillview();
      } // if(rc==SQLITE_OK)
      else
	PDEBUG("No data found\n");
    } // if(rc==SQLITE_OK)

    return resultsize;
  } // if (db)
  else
    return 0;
}

/**
 * \fn float db_get_totalkm(enum dbtimespan timespan)
 * \param timespan either OVERALL, LASTMONTH or LASTYEAR
 * \brief Get the total km of currentcar
 * \return Total km
 *
 * Return the total km for the whole data in the database or for last
 * month or last year.
 *
 */
float db_get_totalkm(enum dbtimespan timespan)
{
 static char querystr[]="SELECT %s FROM record WHERE carid=%d%s";
 char **sqlresult;
 char *sqlquery;
 int sqlrows;
 int sqlcolumns;
 int rc;
 float ret;

 ret=0.0;
 if (db) {
   switch (timespan) {
   case OVERALL:
   case SPANEND:
     sqlquery = sqlite3_mprintf(querystr,"MAX(km)-MIN(km)",currentcar,";");
     break;
   case LASTMONTH:
     sqlquery = sqlite3_mprintf(querystr,"SUM(trip)",currentcar," AND day BETWEEN DATE('now','-1 month') AND DATE('now');");
     break;
   case LASTYEAR:
     sqlquery = sqlite3_mprintf(querystr,"SUM(trip)",currentcar," AND day BETWEEN DATE('now','-1 year') AND DATE('now');");
     break;
   default:
    sqlquery = sqlite3_mprintf(querystr,"MAX(km)-MIN(km)",currentcar,";");
   }
   rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
/*    sqlite3_free((void *)sqlquery); */
   sqlite3_free(sqlquery);
   if (rc==SQLITE_OK) {
     if (sqlcolumns*(sqlrows+1)>0) {
       if (sqlresult[sqlcolumns*(sqlrows+1)-1] != NULL)
	 ret=atof_l(sqlresult[sqlcolumns*(sqlrows+1)-1],"C");
     }
     sqlite3_free_table(sqlresult);
   }
 }

 return ret;
}

/**
 * \fn float db_get_totalfillkm(void)
 * \param timespan either OVERALL, LASTMONTH or LASTYEAR
 * \brief Get the total km of currentcar for average fuel consumption calculation
 * \return Total km
 *
 * Return the total km for the whole data in the database or for last
 * month or last year. This takes into account only those records with fill>0.
 *
 */
float db_get_totalfillkm(enum dbtimespan timespan)
{
 static char querystr[]="SELECT sum(trip) FROM fillview WHERE carid=%d AND fill>0%s";
 char **sqlresult;
 char *sqlquery;
 int sqlrows;
 int sqlcolumns;
 int rc;
 float ret;

 ret=0.0;
 if (db) {
    rc = db_create_fillview();
    if (rc==SQLITE_OK) {
      switch (timespan) {
      case OVERALL:
      case SPANEND:
	sqlquery = sqlite3_mprintf(querystr,currentcar,";");
	break;
      case LASTMONTH:
	sqlquery = sqlite3_mprintf(querystr,currentcar," AND day BETWEEN DATE('now','-1 month') AND DATE('now');");
	break;
      case LASTYEAR:
	sqlquery = sqlite3_mprintf(querystr,currentcar," AND day BETWEEN DATE('now','-1 year') AND DATE('now');");
	break;
      default:
	sqlquery = sqlite3_mprintf(querystr,currentcar,";");
      }
      rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
      /*    sqlite3_free((void *)sqlquery); */
      sqlite3_free(sqlquery);
      if (rc==SQLITE_OK) {
	if (sqlcolumns*(sqlrows+1)>0) {
	  if (sqlresult[sqlcolumns*(sqlrows+1)-1] != NULL)
	    ret=atof_l(sqlresult[sqlcolumns*(sqlrows+1)-1],"C");
	}
	sqlite3_free_table(sqlresult);
      }
      db_drop_fillview();
   }
 }

 return ret;
}

/**
 * \fn float db_get_totalcost(void)
 * \brief Get the total cost of currentcar
 * \return Total cost
 */
float db_get_totalcost(void)
{
 char **sqlresult;
 char *sqlquery;
 int sqlrows;
 int sqlcolumns;
 int rc;
 float ret;

 ret=0.0;
 if (db) {
   sqlquery = sqlite3_mprintf("SELECT sum(price)+sum(service)+sum(oil)+sum(tires) FROM record WHERE carid=%d",currentcar);
   rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
/*    sqlite3_free((void *)sqlquery); */
   sqlite3_free(sqlquery);
   if (rc==SQLITE_OK) {
     if (sqlcolumns*(sqlrows+1)>0) {
       if (sqlresult[sqlcolumns*(sqlrows+1)-1] != NULL)
	 ret=atof_l(sqlresult[sqlcolumns*(sqlrows+1)-1],"C");
     }
     sqlite3_free_table(sqlresult);
   }
 }

 return ret;
}


/**
 * \fn float db_get_totalfill(void)
 * \brief Get the total fill of currentcar
 * \return Total fill
 *
 * The total fill is calculated as the sum of all fills minus the
 * first fill. This is based on an assumptation that the amount of
 * first fill is still in the tank, so it has not been consumed yet.
 *
 * \todo Wouldn't it be better to subtract the last fill, not the
 * first one?
 *
 */
float db_get_totalfill(enum dbtimespan timespan)
{
 static char querystr[]="SELECT SUM(fill) FROM fillview WHERE carid=%d%s";
 char **sqlresult;
 char *sqlquery;
 int sqlrows;
 int sqlcolumns;
 int rc;
 float ret;

 ret=0.0;
 if (db) {
    rc = db_create_fillview();
    if (rc==SQLITE_OK) {
      switch (timespan) {
      case OVERALL:
      case SPANEND:
	sqlquery = sqlite3_mprintf(querystr,currentcar,";");
	break;
      case LASTMONTH:
	sqlquery = sqlite3_mprintf(querystr,currentcar," AND day BETWEEN DATE('now','-1 month') AND DATE('now');");
	break;
      case LASTYEAR:
	sqlquery = sqlite3_mprintf(querystr,currentcar," AND day BETWEEN DATE('now','-1 year') AND DATE('now');");
	break;
      default:
	sqlquery = sqlite3_mprintf(querystr,currentcar,";");
      }
      rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
      /*    sqlite3_free((void *)sqlquery); */
      sqlite3_free(sqlquery);
      if (rc==SQLITE_OK) {
	if (sqlcolumns*(sqlrows+1)>0) {
	  if (sqlresult[sqlcolumns*(sqlrows+1)-1] != NULL)
	    ret=atof_l(sqlresult[sqlcolumns*(sqlrows+1)-1],"C");
	}
	sqlite3_free_table(sqlresult);
      }
      db_drop_fillview();
   }
 }

 return ret;
}

/**
 * \fn int db_get_yearly_consum(int **year, int **numfills, double **minconsum, double **avgconsum, double **maxconsum)
 * \brief Get the minimun, maximum and average consumption of currentcar
 * \param **year The year output table
 * \param **numfills How many fill events for given year
 * \param **minconsum Minimum consumption in the given year
 * \param **avgconsum Average consumption in the given year
 * \param **maxconsum Maximum consumption in the given year
 * \return Size of output tables
 *
 * Tables are allocated in this function and must be freed after usage.
 *
 */
int db_get_yearly_consum(int **year, int **numfills, double **minconsum, double **avgconsum, double **maxconsum)
{
  char **sqlresult;
  char *sqlquery;
  int sqlrows;
  int sqlcolumns;
  int rc;
  int i;
  int resultsize=0;

  *year=NULL;
  *numfills=NULL;
  *minconsum=NULL;
  *avgconsum=NULL;
  *maxconsum=NULL;
  if (db) {
    rc = db_create_fillview();
    if (rc==SQLITE_OK) {
      /* consum limit 1e5 cuts the Inf values out */
      sqlquery = sqlite3_mprintf("SELECT STRFTIME(\"%s\",DATE(day)),COUNT(*),MIN(consum),SUM(fill)/SUM(trip)*100,MAX(consum) FROM fillview WHERE consum>0 AND consum<1e5 GROUP BY STRFTIME(\"%s\",DATE(day))","%Y","%Y");
      rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
      sqlite3_free((void *)sqlquery);
      if (rc==SQLITE_OK) {
	if (sqlresult[sqlcolumns*(sqlrows+1)-1] != NULL) {
	  resultsize=sqlrows;
	  *year=(int *)malloc((resultsize)*sizeof(int));
	  *numfills=(int *)malloc((resultsize)*sizeof(int));
	  *minconsum=(double *)malloc((resultsize)*sizeof(double));
	  *avgconsum=(double *)malloc((resultsize)*sizeof(double));
	  *maxconsum=(double *)malloc((resultsize)*sizeof(double));
	  for (i=1;i<=resultsize;i++) {
	    (*year)[i-1]=(int)atof_l(sqlresult[i*sqlcolumns],"C");
	    (*numfills)[i-1]=(int)atof_l(sqlresult[i*sqlcolumns+1],"C");
	    (*minconsum)[i-1]=(double)atof_l(sqlresult[i*sqlcolumns+2],"C");
	    (*avgconsum)[i-1]=(double)atof_l(sqlresult[i*sqlcolumns+3],"C");
	    (*maxconsum)[i-1]=(double)atof_l(sqlresult[i*sqlcolumns+4],"C");
	  } // for
	}
      } // if(rc==SQLITE_OK)
      else
	PDEBUG("No data found\n");

      sqlite3_free_table(sqlresult);

      db_drop_fillview();
    } // if(rc==SQLITE_OK)

  } // if (db)

  return resultsize;
}

/**
 * \fn int db_get_consum_bin(int year, double minconsum, double maxconsum)
 * \brief Get the number of trip consumptions between minconsum and maxconsum
 * \param year The year
 * \param minconsum Lower limit of the bin
 * \param maxconsum Higher limit of the bin
 * \return Number of trip consumptions between minconsum and maxconsum
 *
 */
int db_get_consum_bin(int year, double minconsum, double maxconsum)
{
  char **sqlresult;
  char *sqlquery;
  int sqlrows;
  int sqlcolumns;
  int rc;
  int ret;

  ret=0;
  if (db) {
    sqlquery = sqlite3_mprintf("SELECT COUNT(*) FROM record WHERE carid=%d AND STRFTIME(\"%s\",DATE(day))='%d' AND consum > 0 AND consum BETWEEN %f AND %f",currentcar,"%Y",year,minconsum,maxconsum);
    rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
    sqlite3_free((void *)sqlquery);
    if (rc==SQLITE_OK) {
      if (sqlcolumns*(sqlrows+1)>0) {
	if (sqlresult[sqlcolumns*(sqlrows+1)-1] != NULL)
	  ret=atoi(sqlresult[sqlcolumns*(sqlrows+1)-1]);
      }
      sqlite3_free_table(sqlresult);
    } // if (rc==SQLITE_OK)
  } // if (db)

  return ret;
}

/**
 * \fn int db_get_last_insert_rowid(void)
 * \brief Get the id of the last insert row
 * \return Rowid of the last inserted row
 *
 * This function uses the sqlite3 function last_insert_rowid() to obtain
 * the id of the last inserted row.
 *
 */
sqlite_int64 db_get_last_insert_rowid(void)
{
 char **sqlresult;
 int sqlrows;
 int sqlcolumns;
 int rc;
 sqlite_int64 ret;

 if (db) {
   rc = sqlite3_get_table(db, "SELECT last_insert_rowid()", &sqlresult, &sqlrows, &sqlcolumns, NULL);
   if (rc==SQLITE_OK) {
     ret=atol(sqlresult[sqlcolumns*(sqlrows+1)-1]);
     sqlite3_free_table(sqlresult);
   }
   else
     ret=0;
 }
 else
   ret=0;

 return ret;
}

/**
 * \fn void db_get_driver(char **driver, int id)
 * \brief Get the name of the driver
 * \param **driver pointer for storing the name
 * \param id the rowid of a record where the driver name is requested
 *
 * This function returns the name of the driver of a given record. The
 * driver buffer is allocated and must be freed after usage.
 *
 */
void db_get_driver(char **driver, int id)
{
 char **sqlresult;
 char *sqlquery;
 int sqlrows;
 int sqlcolumns;
 int rc;
 int len;

 *driver=NULL;
 if (db) {
   sqlquery = sqlite3_mprintf("SELECT nickname FROM driver WHERE id=%d",id);
   rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
   if (rc==SQLITE_OK) {
     if ( (sqlcolumns*(sqlrows+1))> 0) {
       len=strlen(sqlresult[sqlcolumns*(sqlrows+1)-1]);
       *driver=(char *)malloc((len+1)*sizeof(char));
       strncpy(*driver,sqlresult[sqlcolumns*(sqlrows+1)-1],len+1);
     }
     sqlite3_free_table(sqlresult);
   }
   sqlite3_free((void *)sqlquery);
 }
/*  else */
/*    strcpy(driver,"db closed"); */
}


/**
 * \fn void db_get_current_driver(char **driver)
 * \brief Get the name of current driver
 * \param **driver pointer the buffer for storing the name
 *
 * This function returns the name of the current driver. The driver
 * buffer is allocated and must be freed after usage.
 *
 */
void db_get_current_driver(char **driver)
{
  db_get_driver(driver, currentdriver);
}

/**
 * \fn void sqlite_int64 db_get_driverid(int index)
 * \brief Get the database id of a driver
 * \param index index in the database
 *
 * This function returns carid of a driver in the database. The parameter
 * index is the ordinal of the driver.
 *
 */
sqlite_int64 db_get_driverid(int index)
{
 char **sqlresult;
 char *sqlquery;
 int sqlrows;
 int sqlcolumns;
 int rc;
 sqlite_int64 ret;

 if (db) {
   sqlquery = sqlite3_mprintf("SELECT id FROM driver LIMIT 1 OFFSET %d",index);
   rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
   if (rc==SQLITE_OK) {
     ret=atol(sqlresult[sqlcolumns*(sqlrows+1)-1]);
     sqlite3_free_table(sqlresult);
   }
   else
     ret=0;
 }
 else
   ret=0;

 return ret;
}


/**
 * \fn int db_get_numof_drivers(void)
 * \brief Get the number of drivers in the database
 * \return The number of drivers in the database
 *
 * This function returns the number of the drivers in the database.
 *
 */
int db_get_numof_drivers(void)
{
 char **sqlresult;
 int sqlrows;
 int sqlcolumns;
 int rc;
 int ret;

 if (db) {
   rc = sqlite3_get_table(db, "SELECT count(*) FROM driver", &sqlresult, &sqlrows, &sqlcolumns, NULL);
   if (rc==SQLITE_OK) {
     ret=atoi(sqlresult[sqlcolumns*(sqlrows+1)-1]);
     sqlite3_free_table(sqlresult);
   }
   else
     ret=0;
 }
 else
   ret=0;

 return ret;
}

/**
 * \fn int db_get_numof_records(int id)
 * \brief Get the number of records in the database with a given driver
 * \param id driver id
 * \return The number of records in the database with a given driver
 *
 * This function returns the number of records in the database with a
 * given driver.
 *
 */
int db_get_numof_records(int id)
{
 char **sqlresult;
 char *sqlquery;
 int sqlrows;
 int sqlcolumns;
 int rc;
 int ret;

 if (db) {
   sqlquery = sqlite3_mprintf("SELECT COUNT(*) FROM record WHERE driverid=%d",id);
   rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
   if (rc==SQLITE_OK) {
     ret=atoi(sqlresult[sqlcolumns*(sqlrows+1)-1]);
     sqlite3_free_table(sqlresult);
   }
   else
     ret=0;
   sqlite3_free((void *)sqlquery);
 }
 else
   ret=0;

 return ret;
}

/**
 * \fn int db_get_numof_events(int id)
 * \brief Get the number of events in the database with a given driver
 * \param id driver id
 * \return The number of events in the database with a given driver
 *
 * This function returns the number of events in the database with a 
 * given driver.
 *
 */
int db_get_numof_events(int id)
{
 char **sqlresult;
 char *sqlquery;
 int sqlrows;
 int sqlcolumns;
 int rc;
 int ret;

 if (db) {
   sqlquery = sqlite3_mprintf("SELECT COUNT(*) FROM alarmevent WHERE driverid=%d",id);
   rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
   if (rc==SQLITE_OK) {
     ret=atoi(sqlresult[sqlcolumns*(sqlrows+1)-1]);
     sqlite3_free_table(sqlresult);
   }
   else
     ret=0;
   sqlite3_free((void *)sqlquery);
 }
 else
   ret=0;

 return ret;
}

/**
 * \fn db_get_numof_events_with_record(int id)
 * \brief Get the number of events in the database with a corresponding record in the fuel database
 * \param id record id
 * \return The number of events in the database with a given driver
 *
 * This function returns the number of events in the database for a 
 * corresponding record in the fuel database.
 *
 */
int db_get_numof_events_with_record(int id)
{
 char **sqlresult;
 char *sqlquery;
 int sqlrows;
 int sqlcolumns;
 int rc;
 int ret;

 if (db) {
   sqlquery = sqlite3_mprintf("SELECT COUNT(*) FROM alarmevent WHERE recordid=%d",id);
   rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
   if (rc==SQLITE_OK) {
     ret=atoi(sqlresult[sqlcolumns*(sqlrows+1)-1]);
     sqlite3_free_table(sqlresult);
   }
   else
     ret=0;
   sqlite3_free((void *)sqlquery);
 }
 else
   ret=0;

 return ret;
}

/**
 * \fn int db_get_numof_cars(void)
 * \brief Get the number of cars in the database
 * \return The number of cars in the database
 *
 * This function returns the number of the cars in the database.
 *
 */
int db_get_numof_cars(void)
{
 char **sqlresult;
 int sqlrows;
 int sqlcolumns;
 int rc;
 int ret;

 if (db) {
   rc = sqlite3_get_table(db, "SELECT count(*) FROM car", &sqlresult, &sqlrows, &sqlcolumns, NULL);
   if (rc==SQLITE_OK) {
     ret=atoi(sqlresult[sqlcolumns*(sqlrows+1)-1]);
     sqlite3_free_table(sqlresult);
   }
   else
     ret=0;
 }
 else
   ret=0;

 return ret;
}

/**
 * \fn int db_get_numof_carrecords(int id)
 * \brief Get the number of records in the database for a given car
 * \param id car id
 * \return The number of records in the database with a given driver
 *
 * This function returns the number of records in the database with a
 * given driver.
 *
 */
int db_get_numof_carrecords(int id)
{
 char **sqlresult;
 char *sqlquery;
 int sqlrows;
 int sqlcolumns;
 int rc;
 int ret;

 if (db) {
   sqlquery = sqlite3_mprintf("SELECT COUNT(*) FROM record WHERE carid=%d",id);
   rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
   if (rc==SQLITE_OK) {
     ret=atoi(sqlresult[sqlcolumns*(sqlrows+1)-1]);
     sqlite3_free_table(sqlresult);
   }
   else
     ret=0;
   sqlite3_free((void *)sqlquery);
 }
 else
   ret=0;

 return ret;
}

/**
 * \fn int db_get_numof_carevents(int id)
 * \brief Get the number of events in the database with a given car
 * \param id car id
 * \return The number of events in the database with a given driver
 *
 * This function returns the number of events in the database with a 
 * given driver.
 *
 */
int db_get_numof_carevents(int id)
{
 char **sqlresult;
 char *sqlquery;
 int sqlrows;
 int sqlcolumns;
 int rc;
 int ret;

 if (db) {
   sqlquery = sqlite3_mprintf("SELECT COUNT(*) FROM alarmevent WHERE carid=%d",id);
   rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
   if (rc==SQLITE_OK) {
     ret=atoi(sqlresult[sqlcolumns*(sqlrows+1)-1]);
     sqlite3_free_table(sqlresult);
   }
   else
     ret=0;
   sqlite3_free((void *)sqlquery);
 }
 else
   ret=0;

 return ret;
}

/**
 * \fn int db_get_numof_alarmtypes(int id)
 * \brief Get the number of alarmtypes in the database with a given car
 * \param id car id
 * \return The number of alarmtypes in the database with a given car
 *
 * This function returns the number of alarmtypes in the database with a 
 * given car.
 *
 */
int db_get_numof_alarmtypes(int id)
{
 char **sqlresult;
 char *sqlquery;
 int sqlrows;
 int sqlcolumns;
 int rc;
 int ret;

 if (db) {
   sqlquery = sqlite3_mprintf("SELECT COUNT(*) FROM alarmtype WHERE carid=%d",id);
   rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
   if (rc==SQLITE_OK) {
     ret=atoi(sqlresult[sqlcolumns*(sqlrows+1)-1]);
     sqlite3_free_table(sqlresult);
   }
   else
     ret=0;
   sqlite3_free((void *)sqlquery);
 }
 else
   ret=0;

 return ret;
}

/**
 * \fn int db_delete_driver(int id)
 * \brief Deletes a driver
 * \param id the id of the driver
 * \return nonzero if succesful
 *
 * This function deletes a driver, id of which is given as a paremeter.
 *
 */
int db_delete_driver(int id)
{
 char *sqlquery;
 int ret;

 ret=0;
 if (db) {
   ret=1;
   sqlquery = sqlite3_mprintf("DELETE FROM driver WHERE id=%d;",id);
   if (SQLITE_OK == sqlite3_exec(db, sqlquery, NULL, NULL, NULL)) {
     sqlite3_free((void *)sqlquery);
     sqlquery = sqlite3_mprintf("DELETE FROM record WHERE driverid=%d;",id);
     if (SQLITE_OK == sqlite3_exec(db, sqlquery, NULL, NULL, NULL)) {
       sqlite3_free((void *)sqlquery);
       sqlquery = sqlite3_mprintf("DELETE FROM alarmevent WHERE driverid=%d;",id);
       if (SQLITE_OK != sqlite3_exec(db, sqlquery, NULL, NULL, NULL)) 
	 ret=0;
     }
     else
       ret=0;
   }
   else
     ret=0;
 }
 return ret;
}

/**
 * \fn int db_delete_car(int id)
 * \brief Deletes a car
 * \param id the id of the car
 * \return nonzero if succesful
 *
 * This function deletes a car, id of which is given as a paremeter.
 *
 */
int db_delete_car(int id)
{
 char *sqlquery;
 int ret;

 ret=0;
 if (db) {
   ret=1;
   sqlquery = sqlite3_mprintf("DELETE FROM car WHERE id=%d;",id);
   if (SQLITE_OK == sqlite3_exec(db, sqlquery, NULL, NULL, NULL)) {
     sqlite3_free((void *)sqlquery);
     sqlquery = sqlite3_mprintf("DELETE FROM record WHERE carid=%d;",id);
     if (SQLITE_OK == sqlite3_exec(db, sqlquery, NULL, NULL, NULL)) {
       sqlite3_free((void *)sqlquery);
       sqlquery = sqlite3_mprintf("DELETE FROM alarmevent WHERE carid=%d;",id);
       if (SQLITE_OK != sqlite3_exec(db, sqlquery, NULL, NULL, NULL)) 
	 ret=0;
     }
     else
       ret=0;
   }
   else
     ret=0;
 }
 return ret;
}

/**
 * \fn void db_get_current_car(char **car)
 * \brief Get the name of current car
 * \param **car pointer the buffer for storing the name
 *
 * This function returns the name of the current car. The car buffer
 * is allocated and must be freed after usage.
 *
 */
void db_get_current_car(char **car)
{
 char **sqlresult;
 char *sqlquery;
 int sqlrows;
 int sqlcolumns;
 int rc;
 int len;

 *car=NULL;
 if (db) {
   sqlquery = sqlite3_mprintf("SELECT mark FROM car WHERE id=%d",currentcar);
   rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
   if (rc==SQLITE_OK) {
     if ( (sqlcolumns*(sqlrows+1))> 0) {
       len=strlen(sqlresult[sqlcolumns*(sqlrows+1)-1]);
       *car=(char *)malloc((len+1)*sizeof(char));
       strncpy(*car,sqlresult[sqlcolumns*(sqlrows+1)-1],len+1);
     }
     sqlite3_free_table(sqlresult);
   }
   sqlite3_free((void *)sqlquery);
 }
/*  else */
/*    strcpy(car,"db closed"); */
}

/**
 * \fn void sqlite_int64 db_get_carid(int index)
 * \brief Get the database id of a car
 * \param index index in the database
 *
 * This function returns carid of a car in the database. The parameter
 * index is the ordinal of the car.
 *
 */
sqlite_int64 db_get_carid(int index)
{
 char **sqlresult;
 char *sqlquery;
 int sqlrows;
 int sqlcolumns;
 int rc;
 sqlite_int64 ret;

 if (db) {
   sqlquery = sqlite3_mprintf("SELECT id FROM car LIMIT 1 OFFSET %d",index);
   rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
   if (rc==SQLITE_OK) {
     ret=atol(sqlresult[sqlcolumns*(sqlrows+1)-1]);
     sqlite3_free_table(sqlresult);
   }
   else
     ret=0;
 }
 else
   ret=0;

 return ret;
}

/**
 * \fn int db_estimate_date(float km, char **estdate, char *lastdate, double interval)
 * \brief Estimate the date when the given km will be reached
 * \param km the given km
 * \param **estdate pointer the buffer for storing the date
 * \param *lastdate date of last event
 * \param interval interval between services in months
 *
 * This function estimates when the given km will be reached. The date
 * buffer is allocated and must be freed after usage.
 *
 */
int db_estimate_date(float km, char **estdate, char *lastdate, double interval)
{
  char **sqlresult;
  char *sqlquery;
  int sqlrows;
  int sqlcolumns;
  int rc;
  int ret;
  int len;

 *estdate=NULL;
  ret = 1;
  if (db) {
    sqlquery = sqlite3_mprintf("CREATE TEMP VIEW kmview AS SELECT * FROM record WHERE carid=%d ORDER BY km DESC LIMIT 5;",currentcar);
    rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
    sqlite3_free((void *)sqlquery);
    sqlite3_free_table(sqlresult);
    if (rc==SQLITE_OK) {
      if (lastdate == NULL || strlen(lastdate)==0 || fabs(interval)<0.01)
	sqlquery = sqlite3_mprintf("SELECT DATE(DATE(MAX(day)), QUOTE((%f-MAX(km))/((MAX(km)-MIN(km))/((STRFTIME(\"%s\",DATE(MAX(day)))-STRFTIME(\"%s\",DATE(MIN(day))))/3600)))||'hours') FROM kmview;",km,"%s","%s");
      else
	sqlquery = sqlite3_mprintf("SELECT MIN(DATE(\"%s\",\"%f\"||\"months\"),DATE(DATE(MAX(day)), QUOTE((%f-MAX(km))/((MAX(km)-MIN(km))/((STRFTIME(\"%s\",DATE(MAX(day)))-STRFTIME(\"%s\",DATE(MIN(day))))/3600)))||'hours')) FROM kmview;",lastdate,interval,km,"%s","%s");
      rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
      sqlite3_free((void *)sqlquery);
      if (rc==SQLITE_OK) {
	if ( (sqlcolumns*(sqlrows+1))> 0) {
	  if (sqlresult[sqlcolumns*(sqlrows+1)-1] != NULL) {
	    len=strlen(sqlresult[sqlcolumns*(sqlrows+1)-1]);
	    *estdate=(char *)malloc((len+1)*sizeof(char));
	    strncpy(*estdate,sqlresult[sqlcolumns*(sqlrows+1)-1],len+1);
	  }
	}
	else {
	  PDEBUG("No data found\n");
	  ret=0;
	}
	sqlite3_free_table(sqlresult);
      }

      rc = sqlite3_get_table(db, "DROP VIEW kmview;", &sqlresult, &sqlrows, &sqlcolumns, NULL);
      if (rc==SQLITE_OK) {
	sqlite3_free_table(sqlresult);
	ret=0;
      }
    }
    else {
      ret=0;
    }
  }

  return ret;
}

/**
 * \fn float db_estimate_km(char *lastdate, double interval)
 * \brief Estimate the date when the given km will be reached
 * \param *lastdate date of last event
 * \param interval interval between services in months
 *
 * This function estimates the km at time estdate + interval months.
 *
 */
float db_estimate_km(char *lastdate, double interval)
{
  char **sqlresult;
  char *sqlquery;
  int sqlrows;
  int sqlcolumns;
  int rc;
  float ret;

  ret = 0.0;
  if (db) {
    sqlquery = sqlite3_mprintf("CREATE TEMP VIEW kmview AS SELECT * FROM record WHERE carid=%d ORDER BY km DESC LIMIT 5;",currentcar);
    rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
    sqlite3_free((void *)sqlquery);
    sqlite3_free_table(sqlresult);
    if (rc==SQLITE_OK) {
      if (lastdate != NULL && interval > 0.0) {
	sqlquery = sqlite3_mprintf("SELECT MIN(km)+(MAX(km)-MIN(km))/(STRFTIME(\"%s\",DATE(MAX(day)))-STRFTIME(\"%s\",DATE(MIN(day))))*(STRFTIME(\"%s\",DATE(\"%s\",'+%d months'))-STRFTIME(\"%s\",DATE(MIN(day)))) FROM kmview;","%s","%s","%s",lastdate,(int)interval,"%s");
	rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
	sqlite3_free((void *)sqlquery);
	if (rc==SQLITE_OK) {
	  if (sqlresult[sqlcolumns*(sqlrows+1)-1] != NULL)
	    ret=atof(sqlresult[sqlcolumns*(sqlrows+1)-1]);
	  sqlite3_free_table(sqlresult);
	}
      }
    }

    rc = sqlite3_get_table(db, "DROP VIEW kmview;", &sqlresult, &sqlrows, &sqlcolumns, NULL);
    if (rc==SQLITE_OK)
      sqlite3_free_table(sqlresult);
  }

  return ret;
}

/**
 * \fn void db_get_last_event(int alarmid, char **day, double *km)
 * \brief Gets the last event of a reminder
 * \param alarmid the id of the reminder
 * \param **day date of last event
 * \param *km overall distance of last event
 * \return nonzero if succesful
 *
 * This function obtains the last event of a reminder. The event data
 * is returned as a date and overall distance is parameters day and
 * km. The size of the string for date is given as parameter n.
 *
 */
void db_get_last_event(int alarmid, char **day, double *km)
{
 char **sqlresult;
 char *sqlquery;
 int sqlrows;
 int sqlcolumns;
 int rc;
 int len;

 *day=NULL;
 if (db) {
   sqlquery = sqlite3_mprintf("SELECT day, km FROM alarmevent WHERE alarmid=%d ORDER BY km DESC LIMIT 1",alarmid);
   rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
   if (rc==SQLITE_OK) {
     if ( (sqlcolumns*(sqlrows+1))> 0) {
       len=strlen(sqlresult[sqlcolumns*(sqlrows+1)-2]);
       *day=(char *)malloc((len+1)*sizeof(char));
       strncpy(*day,sqlresult[sqlcolumns*(sqlrows+1)-2],len+1);
       *km=atof_l(sqlresult[sqlcolumns*(sqlrows+1)-1],"C");
     }
     else {
       *km=0.0;
     }
     sqlite3_free_table(sqlresult);
   }
   sqlite3_free((void *)sqlquery);
 }
 else {
   strcpy(*day,"db closed");
   *km=0.0;
 }
}

/**
 * \fn void db_calc_date(char **newdate, char *date, double num, char *unit)
 * \brief 
 * \param **newdate The calculated date
 * \param *date The original date
 * \param num How many units (days, months) to differ from date
 * \param *unit The name of the unit (e.g. "months")
 *
 * Calculates a date which differes the given number of units from the
 * original date.
 *
 */
void db_calc_date(char **newdate, char *date, double num, char *unit)
{
 char **sqlresult;
 char *sqlquery;
 int sqlrows;
 int sqlcolumns;
 int rc;
 int len;

 *newdate=NULL;
 if (db) {
   sqlquery = sqlite3_mprintf("SELECT DATE(\"%s\",'%d'||'%s');",date,(int)num,unit);
   rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
   if (rc==SQLITE_OK) {
     if ( (sqlcolumns*(sqlrows+1))> 0) {
       len=strlen(sqlresult[sqlcolumns*(sqlrows+1)-1]);
       *newdate=(char *)malloc((len+1)*sizeof(char));
       strncpy(*newdate,sqlresult[sqlcolumns*(sqlrows+1)-1],len+1);
     }
     sqlite3_free_table(sqlresult);
   }
   sqlite3_free((void *)sqlquery);
 }
/*  else { */
/*    strcpy(newdate,"db closed"); */
/*  } */
}

/**
 * \fn int db_events_exist(int alarmid)
 * \brief Checks if the given reminder has events associated with it
 * \param alarmid the id of the reminder
 * \return nonzero if events exist, 0 if not
 *
 */
int db_events_exist(int alarmid)
{
 char **sqlresult;
 char *sqlquery;
 int sqlrows;
 int sqlcolumns;
 int rc;
 int retval;

 retval=0;
 if (db) {
   sqlquery = sqlite3_mprintf("SELECT id FROM alarmevent WHERE alarmid=%d LIMIT 1",alarmid);
   rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
   if (rc==SQLITE_OK) {
     if ( (sqlcolumns*(sqlrows+1))> 0) {
       retval=1;
     }
     sqlite3_free_table(sqlresult);
   }
   sqlite3_free((void *)sqlquery);
 }

 return retval;
}

/**
 * \fn int db_delete_events(int alarmid)
 * \brief Deletes the events of a reminder
 * \param alarmid the id of the reminder
 * \return nonzero if succesful, 0 if not
 *
 */
int db_delete_events(int alarmid)
{
 char **sqlresult;
 char *sqlquery;
 int sqlrows;
 int sqlcolumns;
 int rc;
 int retval;

 retval=0;
 if (db) {
   sqlquery = sqlite3_mprintf("DELETE FROM alarmevent WHERE alarmid=%d",alarmid);
   rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
   if (rc==SQLITE_OK) {
     if ( (sqlcolumns*(sqlrows+1))> 0) {
       retval=1;
     }
     else {
       retval=0;
     }
     sqlite3_free_table(sqlresult);
   }
   sqlite3_free((void *)sqlquery);
 }
 else {
   retval=0;
 }
 return retval;
}

/**
 * \fn int db_delete_reminder(int id)
 * \brief Deletes a reminder
 * \param id the id of the reminder
 * \return nonzero if succesful, 0 if not
 *
 */
int db_delete_reminder(int id)
{
 char **sqlresult;
 char *sqlquery;
 int sqlrows;
 int sqlcolumns;
 int rc;
 int retval;

 retval=0;
 if (db) {
   sqlquery = sqlite3_mprintf("DELETE FROM alarmtype WHERE id=%d",id);
   rc = sqlite3_get_table(db, sqlquery, &sqlresult, &sqlrows, &sqlcolumns, NULL);
   if (rc==SQLITE_OK) {
     if ( (sqlcolumns*(sqlrows+1))> 0) {
       retval=1;
     }
     sqlite3_free_table(sqlresult);
   }
   sqlite3_free((void *)sqlquery);
 }

 return retval;
}

int db_init_report_query(unsigned int carid)
{
  int rc;

  rc = db_create_fillview();
  /* Prepare sql query for obtaining the report data */
  rc = sqlite3_prepare_v2(db,
"SELECT STRFTIME('%Y',day),MIN(km),MAX(km),MAX(km)-MIN(km),"
"SUM(fill),SUM(price),SUM(fill)/SUM(trip)*100,SUM(price)/SUM(fill),"
"SUM(price)/SUM(trip),SUM(oil),SUM(oil)/(MAX(km)-MIN(km)),"
"SUM(service),SUM(service)/(MAX(km)-MIN(km)),SUM(tires),"
"SUM(tires)/(MAX(km)-MIN(km)) "
"FROM fillview WHERE carid=? AND "
"km NOT IN (0) "
"GROUP BY STRFTIME('%Y',DATE(day)) "
"UNION "
"SELECT 'Total',MIN(km),MAX(km),MAX(km)-MIN(km),"
"SUM(fill),SUM(price),SUM(fill)/SUM(trip)*100,SUM(price)/SUM(fill),"
"SUM(price)/SUM(trip),SUM(oil),SUM(oil)/(MAX(km)-MIN(km)),"
"SUM(service),SUM(service)/(MAX(km)-MIN(km)),SUM(tires),"
"SUM(tires)/(MAX(km)-MIN(km)) "
"FROM fillview WHERE carid=? AND "
"km NOT IN (0) ", -1, &ppStmtGetReport, NULL);
  
  if (rc==SQLITE_OK) {
    rc=sqlite3_bind_int(ppStmtGetReport, 1, carid);
    rc=sqlite3_bind_int(ppStmtGetReport, 2, carid);
  }

  return rc;
}

void db_reset_report_query(void)
{
  sqlite3_reset(ppStmtGetReport);
}


int db_step_report_query(void)
{
  return sqlite3_step(ppStmtGetReport);
}

const unsigned char *db_report_column_text(int col)
{
  return sqlite3_column_text(ppStmtGetReport,col);
}

double db_report_column_double(int col)
{
  return sqlite3_column_double(ppStmtGetReport,col);
}

void db_close_report_query(void)
{
  db_drop_fillview();
}

sqlite_int64 db_add_record(char * date,
			   double km,
			   double trip,
			   double fill,
			   double consum,
			   double price,
			   double priceperlitre,
			   double service,
			   double oil,
			   double tires,
			   char *notes)
{
  sqlite_int64 ret;

  ret=-1;

  if (db) {

    	if (SQLITE_OK != sqlite3_bind_int(ppStmtAddRecord, 1, currentcar) /* carid */
	    || SQLITE_OK != sqlite3_bind_int(ppStmtAddRecord, 2, currentdriver) /* driverid */
	    || SQLITE_OK != sqlite3_bind_text(ppStmtAddRecord, 3, 
					      date,
					      -1, SQLITE_STATIC)
	    || SQLITE_OK != sqlite3_bind_double(ppStmtAddRecord, 4,
						km)
	    || SQLITE_OK != sqlite3_bind_double(ppStmtAddRecord, 5,
						trip)
	    || SQLITE_OK != sqlite3_bind_double(ppStmtAddRecord, 6,
						fill)
	    || SQLITE_OK != sqlite3_bind_double(ppStmtAddRecord, 7,
						consum)
	    || SQLITE_OK != sqlite3_bind_double(ppStmtAddRecord, 8,
						price)
	    || SQLITE_OK != sqlite3_bind_double(ppStmtAddRecord, 9,
						priceperlitre)
	    || SQLITE_OK != sqlite3_bind_double(ppStmtAddRecord, 10,
						service)
	    || SQLITE_OK != sqlite3_bind_double(ppStmtAddRecord, 11,
						oil)
	    || SQLITE_OK != sqlite3_bind_double(ppStmtAddRecord, 12,
						tires)
	    || SQLITE_OK != sqlite3_bind_text(ppStmtAddRecord, 13, 
					      notes,
					      -1, SQLITE_STATIC)
	    || SQLITE_DONE != sqlite3_step(ppStmtAddRecord)) {
	  PDEBUG("Problem saving result");
	}
	else
	  ret = db_get_last_insert_rowid();
	sqlite3_reset(ppStmtAddRecord);
  }

  return ret;
}

sqlite_int64 db_update_record(sqlite_int64 id,
			      char * date,
			      double km,
			      double trip,
			      double fill,
			      double consum,
			      double price,
			      double priceperlitre,
			      double service,
			      double oil,
			      double tires,
			      char *notes)
{
  sqlite_int64 ret;

  ret=-1;

  if (db) {

      if (SQLITE_OK == sqlite3_bind_int(ppStmtOneRecord, 1, id)) {
	if (SQLITE_ROW == sqlite3_step(ppStmtOneRecord)) {


	  if ( 
	      (  SQLITE_OK != sqlite3_bind_text(ppStmtUpdateRecord, 1,
						(date == NULL) ? 
						sqlite3_column_text(ppStmtOneRecord,0) 
						: date,
						-1, SQLITE_STATIC) )
	      || ( SQLITE_OK != sqlite3_bind_double(ppStmtUpdateRecord, 2,
						    (km < 0.0) ?
						    sqlite3_column_double(ppStmtOneRecord,1) 
						    : km) )
	      || ( SQLITE_OK != sqlite3_bind_double(ppStmtUpdateRecord, 3,
						    (trip < 0.0) ?
						    sqlite3_column_double(ppStmtOneRecord,2) 
						    : trip) )
	      || ( SQLITE_OK != sqlite3_bind_double(ppStmtUpdateRecord, 4,
						    (fill < 0.0) ?
						    sqlite3_column_double(ppStmtOneRecord,3) 
						    : fill) )
	      || ( SQLITE_OK != sqlite3_bind_double(ppStmtUpdateRecord, 5,
						    (consum < 0.0) ?
						    sqlite3_column_double(ppStmtOneRecord,9)
						    : consum) )
	      || ( SQLITE_OK != sqlite3_bind_double(ppStmtUpdateRecord, 6,
						    (price < 0.0) ?
						    sqlite3_column_double(ppStmtOneRecord,4)
						    : price) )
	      || ( SQLITE_OK != sqlite3_bind_double(ppStmtUpdateRecord, 7,
						    (priceperlitre < 0.0) ? 
						    sqlite3_column_double(ppStmtOneRecord,4)/
						    sqlite3_column_double(ppStmtOneRecord,3)
						    : priceperlitre) )
	      || (  SQLITE_OK != sqlite3_bind_double(ppStmtUpdateRecord, 8,
						     (service < 0.0) ?
						     sqlite3_column_double(ppStmtOneRecord,5)
						     : service) )
	      || ( SQLITE_OK != sqlite3_bind_double(ppStmtUpdateRecord, 9,
						    (oil < 0.0) ?
						    sqlite3_column_double(ppStmtOneRecord,6)
						    : oil) )
	      || ( SQLITE_OK != sqlite3_bind_double(ppStmtUpdateRecord, 10,
						    (tires < 0.0) ?
						    sqlite3_column_double(ppStmtOneRecord,7)
						    : tires) )
	      || ( SQLITE_OK != sqlite3_bind_text(ppStmtUpdateRecord, 11,
						  (notes == NULL) ?
						  sqlite3_column_text(ppStmtOneRecord,8) 
						  : notes,
						  -1, SQLITE_STATIC) )
	      || SQLITE_OK != sqlite3_bind_int64(ppStmtUpdateRecord, 12,
						 id)
	      || SQLITE_DONE != sqlite3_step(ppStmtUpdateRecord)) {
	    PDEBUG("Problem saving result");
	  }
	  else {
	    ret = id;
	  }
	  sqlite3_reset(ppStmtOneRecord);

	}
	sqlite3_reset(ppStmtUpdateRecord);
      }
  }

  return ret;
}

/**
 * \fn int db_delete_record(sqlite_int64 id)
 * \brief Deletes a record
 * \param id the id of the record
 * \return nonzero if succesful, 0 if not
 *
 */
int db_delete_record(sqlite_int64 id)
{
  int ret;

  ret = SQLITE_ERROR;
  if (SQLITE_OK != sqlite3_bind_int(ppStmtDeleteRecord, 1, id)
      ||SQLITE_DONE != (ret=sqlite3_step(ppStmtDeleteRecord))) {
    PDEBUG("Problem deleting the record");
  }
  sqlite3_reset(ppStmtDeleteRecord);

  return ret;
}

/**
 * \fn void fuelpad_exit(void)
 * \brief Does the exit of application cleanly
 *
 * Closes the connection to sqlite3 database.
 *
 */
void fuelpad_exit(void)
{
  db_close();
}

int main( int   argc,
          char *argv[] )
{
  int c;

  while ((c=getopt(argc, argv, "D"))>-1) {
    switch (c) {
    case 'D':
      debug_level = 1;
      break;
    default:
      printf("Invalid option found.\n");
      break;
    }
  }

  db_setfilename(default_db);

  ui_init(argc,argv);

  db_connect();

  ui_create_mainwin();

  ui_main_loop();

  fuelpad_exit();

  return 0;
}
