/**
 * Copyright (C) 2008-09 Tan Miaoqing
 * Contact: Tan Miaoqing <rabbitrun84@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <time.h>
#include <glib/gstdio.h>

#include "eventlogger-util.h"
#include "common.h"

typedef enum {
  FRIENDS_STATUS = 0,
  MY_STATUS
} StatusType;

/**
 * get_full_status_text:
 *
 * eventlogger status plugin limits status text to 100 characters
 * So we need to get the full text string from el.db
 */
static ContactStatus *
get_contact_status (RTComEl *eventlogger,
                    RTComElQuery *query)
{
  RTComElIter *iter;
  GValueArray *values;
  const gchar *free_text;
  ContactStatus *status;

  iter = rtcom_el_get_events (eventlogger, query);

  if(!(iter && rtcom_el_iter_first(iter))) {
      g_debug("No query result");
      if (iter)
        g_object_unref(iter); /* must unref iter, otherwise the app will crash */
      return NULL;
  }

  values = rtcom_el_iter_get_valuearray (iter, "free-text", NULL);
  if (!values) {
    g_debug("Failed to get the free-text");
    g_object_unref (iter);
    return NULL;
  }
  free_text = g_value_get_string (g_value_array_get_nth(values, 0));

  status = eventlogger_util_contact_status_new ();
  status->text = g_strdup (free_text);

  GValue timeval = {0};
  rtcom_el_iter_get_raw (iter, "start-time", &timeval);
  time_t t = g_value_get_int (&timeval);
  g_value_unset (&timeval);
  status->start_time = get_timestamp (t);

  g_value_array_free (values);
  g_object_unref (iter);

  return status;
}

/**
 * eventlogger_util_get_contact_status_by_id:
 *
 * @event_id: id of the event in el.db
 *
 * Get the status of event identified by event_id
 */
ContactStatus *
eventlogger_util_get_contact_status_by_id (RTComEl *eventlogger,
                                           gint event_id)
{
  RTComElQuery *query;
  ContactStatus *status;

  query = rtcom_el_query_new (eventlogger);
  rtcom_el_query_set_limit (query, 1);

  if(!rtcom_el_query_prepare (
              query,
              "service", "RTCOM_EL_SERVICE_STATUS", RTCOM_EL_OP_EQUAL,
              "id", event_id, RTCOM_EL_OP_EQUAL,
              NULL)) {
      g_debug("Failed to get the status message of %d", event_id);
      return NULL;
  }

  status = get_contact_status (eventlogger, query);
  g_object_unref (query);

  return status;
}

/**
 * eventlogger_util_get_last_contact_status:
 *
 * @remote_uid: vcard_field of the contact (aka IM field), e.g. someone@gmail.com
 *
 * Get the last status of a contact
 */
ContactStatus *
eventlogger_util_get_last_contact_status (RTComEl *eventlogger,
                                          const gchar *local_uid,
                                          const gchar *remote_uid)
{
  RTComElQuery *query;
  ContactStatus *status;
  gboolean res;

  query = rtcom_el_query_new (eventlogger);
  rtcom_el_query_set_limit (query, 1);

  if (local_uid) {
    res = rtcom_el_query_prepare (
                query,
                "service", "RTCOM_EL_SERVICE_STATUS", RTCOM_EL_OP_EQUAL,
                "local-uid", local_uid, RTCOM_EL_OP_EQUAL,
                "remote-uid", remote_uid, RTCOM_EL_OP_EQUAL,
                NULL);
  }
  else {
    res = rtcom_el_query_prepare (
                query,
                "service", "RTCOM_EL_SERVICE_STATUS", RTCOM_EL_OP_EQUAL,
                "remote-uid", remote_uid, RTCOM_EL_OP_EQUAL,
                NULL);
  }

  if(!res) {
    g_debug("Failed to get the last status message of %s", remote_uid);
    return NULL;
  }

  status = get_contact_status (eventlogger, query);
  g_object_unref (query);

  return status;
}

ContactStatus *
eventlogger_util_contact_status_new ()
{
  ContactStatus *status;
  status = g_slice_new0 (ContactStatus);
  status->text = NULL;
  status->start_time = NULL;
  return status;
}

void
eventlogger_util_contact_status_free (ContactStatus *status)
{
  if (status) {
    g_free (status->text);
    g_free (status->start_time);
    g_slice_free (ContactStatus, status);
  }
}

gint
eventlogger_util_get_last_status_id (RTComEl *eventlogger,
                                     const gchar *eventtype)
{
  RTComElIter *iter = NULL;
  RTComElQuery *query;
  gint last_status_id = 0;

  query = rtcom_el_query_new (eventlogger);
  rtcom_el_query_set_limit (query, 1);

  if (!rtcom_el_query_prepare (query,
      "service", "RTCOM_EL_SERVICE_STATUS", RTCOM_EL_OP_EQUAL,
      "event-type", eventtype, RTCOM_EL_OP_EQUAL,
      NULL)) {
    goto END;
  }

  iter = rtcom_el_get_events (eventlogger, query);

  if (iter && rtcom_el_iter_first (iter)) {
    GValueArray *values;

    values = rtcom_el_iter_get_valuearray (iter,
        "id", NULL);
    last_status_id = g_value_get_int (g_value_array_get_nth (values, 0));
    g_value_array_free (values);
  }

END:
  if (iter)
    g_object_unref (iter);
  g_object_unref (query);
  return last_status_id;
}

static void
free_statuses_array (GArray *statuses)
{
  guint i;

  for (i = 0; i < statuses->len; i++) {
    gchar *status;

    status = g_array_index (statuses, gchar *, i);
    g_free (status);
  }
  g_array_free (statuses, TRUE);
}

static gchar *
get_statuses_since_id (RTComEl *eventlogger,
                       const gchar *eventtype,
                       gint id)
{
  RTComElIter *iter = NULL;
  RTComElQuery *query;
  gchar *result = NULL;

  g_debug ("%s", G_STRFUNC);

  query = rtcom_el_query_new (eventlogger);

  if (!rtcom_el_query_prepare (query,
      "service", "RTCOM_EL_SERVICE_STATUS", RTCOM_EL_OP_EQUAL,
      "event-type", eventtype, RTCOM_EL_OP_EQUAL,
      "id", id, RTCOM_EL_OP_GREATER,
      NULL)) {
    goto END;
  }

  iter = rtcom_el_get_events (eventlogger, query);

  if (iter && rtcom_el_iter_first (iter)) {
    GArray *statuses;
    guint i, len;

    statuses = g_array_new (FALSE, FALSE, sizeof (gchar *));

    do {
      GValueArray *values;
      const gchar *remote_name;
      const gchar *free_text;
      gint start_time;
      gchar *timestamp;
      gchar *status;

      values = rtcom_el_iter_get_valuearray (iter,
          "remote-name",
          "free-text",
          "start-time",
          NULL);

      remote_name = g_value_get_string (g_value_array_get_nth (values, 0));
      free_text = g_value_get_string (g_value_array_get_nth (values, 1));
      start_time = g_value_get_int (g_value_array_get_nth (values, 2));
      timestamp = get_timestamp_full (start_time);

      status = g_strdup_printf ("%s %s\n%s\n",
          remote_name, timestamp, free_text);

      g_array_append_val (statuses, status);

      g_free (timestamp);
      g_value_array_free (values);
    } while (rtcom_el_iter_next (iter));

    len = statuses->len;
    const gchar *status_str[len+2];

    /* rtcom_el_get_events() returns backwards results
     * e.g. 4 3 2 1 (where number is event_id)
     * But when writing to the file, we want 1 2 3 4
     * So next time when we append more results to the file
     * We get 1 2 3 4 5 6 7
     * Thus, we reverse the array order here */
    for (i = 0; i < len; i++) {
      status_str[len-1-i] = g_array_index (statuses, gchar *, i);
    }
    status_str[len] = "";
    status_str[len+1] = NULL;

    result = g_strjoinv ("\n", (gchar **)status_str);

    /* Free array and each string */
    free_statuses_array (statuses);
  }

END:
  if (iter)
    g_object_unref (iter);
  g_object_unref (query);
  return result;
}

static void
save_statuses_to_storage (RTComEl *eventlogger,
                          StatusType type,
                          gint since_id,
                          const gchar *dir)
{
  gchar *result;
  const gchar *eventtype;
  const gchar *filename;

  if (type == FRIENDS_STATUS) {
    eventtype = "RTCOM_EL_EVENTTYPE_STATUS_FRIEND";
    filename = FRIENDS_FILE;
  }
  else {
    eventtype = "RTCOM_EL_EVENTTYPE_STATUS_MY";
    filename = MY_FILE;
  }

  result = get_statuses_since_id (eventlogger, eventtype, since_id);

  if (result) {
    FILE *pFile;
    gchar *file;

    file = g_build_filename (dir, filename, NULL);
    pFile = g_fopen (file, "a");
    fprintf (pFile, result);

    fclose (pFile);
    g_free (file);
    g_free (result);
  }
  else
    g_debug ("no new statuses to save");
}

void
eventlogger_util_save_statuses_to_storage (RTComEl *eventlogger,
                                           gint last_friend_status_id,
                                           gint last_my_status_id)
{
  gchar *dir;
  gint friends_since_id = last_friend_status_id;
  gint my_since_id = last_my_status_id;

  g_debug ("%s", G_STRFUNC);

  /* check if ~/MyDocs/FriendStatus exists,
   * create it if not. Also, 1st time we want to
   * backup all statuses in el.db */
  dir = g_build_filename (g_get_home_dir (),
      STORAGE_DIR, NULL);

  if (g_file_test (dir, G_FILE_TEST_EXISTS) == FALSE) {
    if (g_mkdir (dir, S_IRWXU) != 0) {
      /* Create dir failed */
      goto END;
    }

    /* Save all statuses from the very first one */
    friends_since_id = 0;
    my_since_id = 0;
  }

  /* Save friends statuses to the file storage */
  save_statuses_to_storage (eventlogger,
      FRIENDS_STATUS, friends_since_id, dir);

  /* Save my statuses to the file storage */
  save_statuses_to_storage (eventlogger,
      MY_STATUS, my_since_id, dir);

END:
  g_free (dir);
}
