/*
 * This file is part of Maemo 5 Office UI for KOffice
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: Manikandaprasad N C <manikandaprasad.chandrasekar@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#ifndef MAINWINDOW_H
#define MAINWINDOW_H

#include <QtGui/QMainWindow>
#include <KoDocument.h>
#include <KWView.h>
#include <styles/KoListStyle.h>

#include "PresentationTool.h"
#include "Splash.h"
class KoTextEditor;
class QPushButton;
class QIcon;
class QTextDocument;
class QToolButton;
class QFrame;
class QLabel;
class PresentationTool;

namespace Ui
{
class MainWindow;
}

class QMessageBox;
class QLineEdit;
class KoCanvasController;
class QTextCursor;
class KoShape;
class KoPAPageBase;
class QShortcut;
class QCheckBox;
class QComboBox;
class QFontComboBox;
class QTextListFormat;
class QTextDocument;
class QGridLayout;
class QDialog;

class OfficeInterface;

/*!
 * \brief Main window of the application. KoCanvasController is set as
 * the central widget. It displays the loaded documents.
 */
class MainWindow : public QMainWindow
{
    Q_OBJECT

public:
    MainWindow(Splash *aSplash, QWidget *parent = 0);
    ~MainWindow();
    QString newdocumentype;
    //void tabletEvent ( QTabletEvent * event );
    void mousePressEvent ( QMouseEvent * event );
    void mouseMoveEvent ( QMouseEvent * event );
    void mouseReleaseEvent ( QMouseEvent * event );
    void paintEvent( QPaintEvent * event );

    void disableFullScreenPresentationNavigation();
    void enableFullScreenPresentationNavigation();

private:
    Ui::MainWindow *m_ui;
    int m_count;
    /*!
     * Format frame declaration
     */
    QFrame * m_formatframe;
    QGridLayout * m_formatframelayout;
    QPushButton * m_bold;
    QPushButton * m_italic;
    QPushButton * m_underline;
    QPushButton * m_alignleft;
    QPushButton * m_alignright;
    QPushButton * m_aligncenter;
    QPushButton * m_numberedlist;
    QPushButton * m_bulletlist;
    QPushButton * m_alignjustify;
    /*!
     * Font style frame declaration
     */
    QFrame * m_fontstyleframe;
    QGridLayout *m_fontstyleframelayout;
    QFontComboBox *m_fontcombobox;
    QComboBox *m_fontsizecombo;
    QPushButton *m_textcolor;
    QPushButton *m_superscript;
    QPushButton *m_subscript;
    QPushButton *m_textbackgroundcolor;
    /*!
     * New document chooser dialog
     */
    QDialog * m_docdialog;
    QGridLayout * m_docdialoglayout;
    QToolButton * m_document;
    QToolButton * m_presenter;
    QToolButton * m_spreadsheet;
    /*!
     * Confirmation dialog for closing new document
     */
    QDialog *m_confirmationdialog;
    QGridLayout * m_confirmationdialoglayout;
    QPushButton *m_yes;
    QPushButton *m_no;
    QPushButton *m_cancel;
    QLabel *m_message;
    /*!
     * line edit for search
     */
    QLineEdit *m_search;
    /*!
     * Checkbox to change between normal and exact match searches
     */
    QCheckBox *m_exactMatchCheckBox;
    /*!
     * Pointer to QTextDocument
     */
    QTextDocument *m_textDocument;
    /*!
     * Pointer to KoDocument
     */
    KoDocument *m_doc;
    /*!
     * Pointer to KoTextEditor
     */
    KoTextEditor *m_editor;
    /*!
     * Pointer to KWView
     */
    KWView * m_kwview;
    /*!
     * Pointer to KoView
     */
    KoView *m_view;
    /*!
     * Pointer to KoCanvasController
     */
    KoCanvasController *m_controller;
    /*!
     * Integers about current page
     */
    int m_vPage, m_hPage;
    /*!
     * Flag for pressed state
     */
    bool m_pressed;
    /*!
     * Flag for seeing which toolbar was active
     */
    bool m_isViewToolBar;
    /*!
     * Timer for hiding full screen button
     */
    QTimer *m_fsTimer;
    /*!
     * Full screen push button
     */
    QPushButton *m_fsButton;
    /*!
     * Icon for full screen button
     */
    const QIcon m_fsIcon;

    /*!
     * Full screen back button for presentations
     */
    QPushButton *m_fsPPTBackButton;

    /*!
     * Full screen forward button for presentations
     */
    QPushButton *m_fsPPTForwardButton;

    /*!
     * Current page number. Saved in MainWindow::resourceChanged() slot.
     */
    int m_currentPage;

    /*!
     * Index for moving between searched strings
     */
    int m_index;
    /*!
     * Positions for found text strings
     */
    QList<QPair<QPair<KoPAPageBase*, KoShape*>, QPair<int, int> > > m_positions;
    /*!
     * Flag for seeing if search is for whole words. false by default
     */
    bool m_wholeWord;
    /*!
     * flag for checking open document type
     */
    enum DocumentType { Text, Presentation, Spreadsheet };
    DocumentType m_type;
    /*!
     * Flag to make higlighting of alignment button enable
     */
    enum AlignmentType {Left=1,Right,Center,Justify};
    AlignmentType m_alignType;
    /*!
     * Pointer to splash class
     */
    Splash *m_splash;
    /*!
     * Pointer to pen draw button
     */
    QPushButton *m_fsPPTDrawPenButton;
    /*!
     * Pointer to highlight draw button
     */
    QPushButton *m_fsPPTDrawHighlightButton;
    /*!
     * Pointer to presentation drawing tools
     */
    PresentationTool *m_pptTool;

    void init();
    /*!
     * close the document if it is open and reinit to NULL
     */
    void closeDocument();
    /*!
     * style formatting function
     */
    void doStyle(KoListStyle::Style);
    /*!
     *opening a new document
     */
    void openNewDocument(DocumentType);
    /*!
     *Function to add formatframe components
     */
    QPushButton *addFormatFrameComponent(QString const& imagepath);
    /*!
     *Function to add fontstyleframe components
     */
    QPushButton *addFontStyleFrameComponent(QString const& imagepath);
    /*!
     *Function to create new document
     */
    QToolButton *addNewDocument(QString const& docname);
    /*!
     * Find string from document
     * /param pointer to QTextDocument
     * /param reference to text to be searched
     */
    void findText(QList<QTextDocument*> docs, QList<QPair<KoPAPageBase*, KoShape*> > shapes, const QString &aText);
    /*!
     * Find string from document
     * /param current index
     */
    void highlightText(int aIndex);

    /*!
     * Trigger an action from the action collection of the current KoView.
     * /param name The name of the action to trigger
     * /return bool Returns false if there was no action with the given name found
     */
    bool triggerAction(const char* name);

    /*!
     * Update the enabled/disabled state of actions depending on if a document is currently
     * loaded.
     */
    void updateActions();

    /*!
     * Event filter to catch all mouse events to be able to properly show and hide the fullscreen
     * button when in fullscreen mode.
     */
    bool eventFilter(QObject *watched, QEvent *event);

    /*!
     * Check filetype
     * /param filename
     * /return true if supported
     */
    bool checkFiletype(const QString &fileName);

    /*!
     * shows back and forward buttons in fullscreen presentation mode
     */
    void showFullScreenPresentationIcons();

    /*!
     * Raise window. For some reason Qt programs are not not raised
     * automatically after D-Bus messages. This low level code sends
     * _NET_ACTIVE_WINDOW message to Matchbox and window is raised.
     */
    void raiseWindow(void);

    /*!
     * counts initial undosteps of KoDocument
     */
    void initialUndoStepsCount();

private slots:

    void menuClicked(QAction* action);
    void pluginOpen(bool newWindow, const QString& path);
    void updateUI();
    void resourceChanged(int key, const QVariant &value);
    /*!
     * Slot to perform UndoAction
     */
    void doUndo();
    /*!
     * Slot to perform RedoAction
     */
    void doRedo();
    /*!
     * Slot to actionSearch toggled signal
     */
    void toggleToolBar(bool);
    /*!
     * Slot to actionEdit toggled signal
     */
    void editToolBar(bool);
    /*!
     * Slot to actionClose signal
     */
    void closeDoc();
    /*!
     *  Slot to convert character into bold
     */
    void doBold();
    /*!
     *  Slot to convert character into italic
     */
    void doItalic();
    /*!
     *  Slot to convert character into underline
     */
    void doUnderLine();
    /*!
     *  Slot for Left Alignment
     */
    void doLeftAlignment();
    /*!
     *  Slot for right Alignment
     */
    void doRightAlignment();
    /*!
     *  Slot for center Alignment
     */
    void doCenterAlignment();
    /*!
     *  Slot for justify Alignment
     */
    void doJustify();
    /*!
     *  Slot for adding Numbers
     */
    void doNumberList();
    /*!
     *  Slot for adding Bullets
     */
    void doBulletList();
    /*!
     *  Slot to perform sub Script action
     */
    void doSubScript();
    /*!
     *  Slot to perform super Script action
     */
    void doSuperScript();
    /*!
     *  Slot for font size Selection
     */
    void selectFontSize();
    /*!
     *  Slot for font type Selection
     */
    void selectFontType();
    /*!
     *  Slot for text color Selection
     */
    void selectTextColor();
    /*!
     *  Slot for text backgroundcolor Selection
     */
    void selectTextBackGroundColor();
    /*!
     * Slot to discard newDocument without performing save operation
     */
    void discardNewDocument();
    /*!
     *    Slot to return back from closing new document
     */
    void returnToDoc();
    /*!
     * Slot to display formatframe with all options
     */
    void openFormatFrame();
    /*!
     * Slot to display fontstyleframe with all options
     */
     void openFontStyleFrame();
    /*!
     * Slot to choose new document
     */
    void chooseDocumentType();
    /*!
     * Slot to find AlignmentType
     */
    void findAlignType();
    /*!
     * Slot for intial AlignmentTypeCheck
     */
    void initialAlignType();
    /*!
     * Slot to open new documnet
     */
    void openNewDoc();
    /*!
     * Slot to open new presenter
     */
    void openNewPresenter();
    /*!
     * Slot to open new spreadsheet
     */
    void openNewSpreadSheet();
     /*!
     * Slot to actionZoomIn triggered signal
     */
    void zoomIn();
    /*!
     * Slot to actionZoomOut triggered signal
     */
    void zoomOut();
    /*!
     * Slot to actionZoomLevel triggered signal
     */
    void zoom();
    void zoomToPage();
    void zoomToPageWidth();
    /*!
     * Slot to actionNextPage triggered signal
     */
    void nextPage();
    /*!
     * Slot to actionPrevPage triggered signal
     */
    void prevPage();
    /*!
     * Slot to fullscreen toolbutton triggered signal
     * Logic for switching from  normal mode to full screen mode
     */
    void fullScreen();
    /*!
     * Slot to mFSTimer timeout signal.
     * Hides full screen button
     */
    void fsTimer();
    /*!
     * Slot to mFSButton clicked signal
     * Deactivates fullscreen mode
     */
    void fsButtonClicked();
    /*!
     * Slot for actionSearch triggered signal
     */
    void startSearch();
    /*!
     * Slot for moving to previous found text string
     */
    void previousWord();
    /*!
     * Slot for moving to next found text string
     */
    void nextWord();
    /*!
     * Slot for toggleing between whole word search and part of word search
     */
    void searchOptionChanged(int aCheckBoxState);
    /*!
     * Slot to perform copy operation on selected text
     */
    void copy();
      /*!
     * Slot to perform cut operation on selected text
     */
    void cut();
    /*!
     * Slot to perform paste operation on selected text
     */
    void paste();
    /*!
     * Slot that shows a hildonized application menu
     */
    void showApplicationMenu();
    /*!
     * Slot that is invoked when the currently active tool changes.
     */
    void activeToolChanged(KoCanvasController *canvas, int uniqueToolId);
    /*!
     * function for opening existing document.
     */
    void doOpenDocument();
    /*!
     * Slot to actionAbout triggered signal
     */
    void openAboutDialog();
    /*!
     * Slot for progress indicator
     */
    //void slotProgress(int value);
public slots:
    /*!
     * Slot to perform save operation
     */
    void saveFile();
    /*!
     * Slot to perform save as operation
     */
    void saveFileAs();
    /*!
     * Slot to  dialog fileSelected signal
     * /param filename
     */
    void openDocument(const QString &fileName);
    /*!
     * Slot to actionOpen triggered signal
     */
    void openFileDialog();
    /*!
     * Slot to check DBus activation, If document is not opened
     * then open filedialog
     */
    void checkDBusActivation();
    /*!
     * Slot to test scrolling of a loaded document and quitting afterwards.
     * This slot checks if a document has been loaded. If so, it calls pagedown and waits
     * until the end of the document has been reached.
     * Then it quits.
     */
    void loadScrollAndQuit();

private:

    QMap<QString, OfficeInterface*> loadedPlugins;

    void setShowProgressIndicator(bool visible);
    /*!
     * Opened file
     */
    QString m_fileName;
    /*!
     * Double click detector
     */
    bool m_doubleClick;
    /*!
     * true if existing document is open
     */
    bool m_openCheck;
    /*!
     * true if new document is open
     */
    bool m_newDocOpen;
    /*!
     * flag for new file to existing file conversion
     */
    bool m_existingFile;
    /*!
     * true if already one document exists
     */
    bool m_docExist;
    /*!
     * QShortcut for copying text with Ctrl-c
     */
    QShortcut *m_copyShortcut;
    /*!
     * Count of mouseMove or tabletMove events after MousePress or tabletPress event
     */
    int m_panningCount;
    /*!
     * Position of last mousePress or tabletPress event
     */
    QPoint m_pressPos;
    /*!
     * True if slide can be changed by panning document
     */
    bool m_slideChangePossible;
    /*!
     * Is document is currently being loaded
     */
    bool m_isLoading;
};

#endif // MAINWINDOW_H
