/*
 * This file is part of Maemo 5 Office UI for KOffice
 *
 * Copyright (C) 2010 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: Gopalakrishna Bhat A <gopalakbhat@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include "FeedBackPlugin.h"
#include "InputUI.h"

#include <QUrl>
#include <QProcess>
#include <QNetworkAccessManager>
#include <QNetworkRequest>
#include <QDebug>
#include <QFileDialog>
#include <QDateTime>

#include <kmimetype.h>

#ifdef Q_WS_MAEMO_5
#include <QtMaemo5/QMaemo5InformationBox>
#endif

#define FEEDBACK_URL "http://212.213.221.115/cgi-bin/feedback/upload.cgi"
#define MAX_POST (1024*10000)

FeedBackPlugin::FeedBackPlugin() :
        inputUI(0),
        accessManager(0),
        networkReply(0),
        networkRequest(0),
        pendingRequest(false)
{
}

FeedBackPlugin::~FeedBackPlugin()
{
    qDebug()<<Q_FUNC_INFO;
    if(inputUI) {
        delete inputUI;
        inputUI=0;
    }
}

void FeedBackPlugin::setDocument(void *doc)
{
    Q_UNUSED(doc);
}

QWidget *FeedBackPlugin::view()
{
    if(inputUI) {
        disconnect(inputUI,SIGNAL(feedBackDone(QString,QString,QStringList)),this,
                   SLOT(doneButtonClicked(QString,QString,QStringList)));
        disconnect(this,SIGNAL(progressPercent(int)),inputUI,SLOT(uploadProgress(int)));
        disconnect(inputUI,SIGNAL(abort()),this,SLOT(abort()));

        delete inputUI;
        inputUI=0;
    }
    inputUI=new InputUI();
    initFeedBackPlugin();

    Q_ASSERT(inputUI);

    connect(inputUI,SIGNAL(feedBackDone(QString,QString,QStringList)),
            SLOT(doneButtonClicked(QString,QString,QStringList)));
    connect(this,SIGNAL(progressPercent(int)),inputUI,SLOT(uploadProgress(int)));
    connect(inputUI,SIGNAL(abort()),this,SLOT(abort()));

    return inputUI;
}


QString FeedBackPlugin::pluginName()
{
    return tr("Feedback");
}


QStringList FeedBackPlugin::pluginSupportTypes()
{
    return QStringList("All");
}

void FeedBackPlugin::doneButtonClicked(QString email, QString comments, QStringList uploadFilePaths)
{
    qDebug()<<Q_FUNC_INFO;
    QUrl *feedBackUrl= new QUrl(FEEDBACK_URL);
    QString url=feedBackUrl->toString();
    dataSent.clear();

    if(pendingRequest) {
#ifdef Q_WS_MAEMO_5
        QMaemo5InformationBox::information(0, tr("Previous request pending."),
                                           QMaemo5InformationBox::DefaultTimeout);
#endif
        return;
    }

    if(comments.isEmpty() && email.isEmpty() && uploadFilePaths.isEmpty()) {
#ifdef Q_WS_MAEMO_5
        QMaemo5InformationBox::information(0, tr("All the fields are empty."),
                                           QMaemo5InformationBox::DefaultTimeout);
#endif
        return;
    }

    if(!comments.isEmpty())
    {
        QString host;
        host=url.right(url.length()-url.indexOf("://")-3);
        host=host.left(host.indexOf("/"));
        QString crlfDelimiter="\r\n";
        qsrand(QDateTime::currentDateTime().toTime_t());
        QString b=QVariant(qrand()).toString()+QVariant(qrand()).toString()+QVariant(qrand()).toString();
        QString boundary="---------------------------"+b;
        QString endBoundary=crlfDelimiter+"--"+boundary+"--"+crlfDelimiter;
        QString contentType="multipart/form-data; boundary="+boundary;
        boundary="--"+boundary+crlfDelimiter;
        QByteArray bond=boundary.toAscii();
        //QByteArray dataSent;

        dataSent.append(bond);

        //only for first field
        boundary=crlfDelimiter+boundary;
        bond=boundary.toAscii();

        dataSent.append(QString("Content-Disposition: form-data; name=\""+QString("email_address")+"\""+crlfDelimiter).toAscii());
        dataSent.append(QString("Content-Transfer-Encoding: 8bit"+crlfDelimiter).toAscii());
        dataSent.append(crlfDelimiter.toAscii());
        dataSent.append(email.toUtf8());

        dataSent.append(bond);
        dataSent.append(QString("Content-Disposition: form-data; name=\""+QString("comments")+"\""+crlfDelimiter).toAscii());
        dataSent.append(QString("Content-Transfer-Encoding: 8bit"+crlfDelimiter).toAscii());
        dataSent.append(crlfDelimiter.toAscii());
        dataSent.append(comments.toUtf8());

        //check the version of the f-office used
        QProcess *determineVersion=new QProcess();
        QStringList parameters;
        parameters<<"-W"<<"-f=${Version}"<<"freoffice";
        determineVersion->start("dpkg-query",parameters);
        determineVersion->waitForFinished();

        if(determineVersion->exitCode()==0)
        {
            dataSent.append(bond);
            dataSent.append(QString("Content-Disposition: form-data; name=\""+QString("version")+"\""+crlfDelimiter).toAscii());
            dataSent.append(QString("Content-Transfer-Encoding: 8bit"+crlfDelimiter).toAscii());
            dataSent.append(crlfDelimiter.toAscii());
            dataSent.append(QString(determineVersion->readAllStandardOutput()).toUtf8());
        } else {
            qDebug()<<"Failure determine version the cause being "<<determineVersion->readAllStandardError();
        }

        delete determineVersion;

        //for all files
        for(int i=0; i<uploadFilePaths.size();i++) {
            if(!uploadFilePaths.value(i).isEmpty()) {
                qDebug()<<"file "<<uploadFilePaths.value(i);
                QFile f(uploadFilePaths.value(i));

                if(!f.exists()) {
#ifdef Q_WS_MAEMO_5
                    QMaemo5InformationBox::information(0, tr("File not found."),
                                                       QMaemo5InformationBox::DefaultTimeout);
#endif
                    dataSent.clear();
                    return;
                }

                QString name=f.fileName();
                dataSent.append(bond);
                qDebug()<<"filename"<<name;
                dataSent.append(QString("Content-Disposition: form-data; name=\""+QString("attachment")+"\"; filename=\""+name+"\""+crlfDelimiter).toAscii());
                dataSent.append(QString("Content-Type: "+KMimeType::findByPath(uploadFilePaths.value(i))+crlfDelimiter+crlfDelimiter).toAscii());

                f.open(QIODevice::ReadOnly);
                dataSent.append(f.readAll());
                f.close();

            }
        }
        dataSent.append(endBoundary.toAscii());

        if(dataSent.size()>=MAX_POST) {
#ifdef Q_WS_MAEMO_5
            QMaemo5InformationBox::information(0, tr("File can be a max of 10MB."),
                                               QMaemo5InformationBox::DefaultTimeout);
#endif
            dataSent.clear();
            return;
        }

        accessManager=new QNetworkAccessManager(this);
        networkRequest = new QNetworkRequest();
        networkRequest->setRawHeader("Host", host.toAscii());
        networkRequest->setRawHeader("User-Agent", QString("FreOffice").toAscii());
        networkRequest->setRawHeader("Referer",QString("http://212.213.221.115/feedback/upload.html").toAscii());
        networkRequest->setHeader(QNetworkRequest::ContentTypeHeader, contentType.toAscii());
        networkRequest->setHeader(QNetworkRequest::ContentLengthHeader, QVariant(dataSent.size()).toString());
        networkRequest->setUrl(*feedBackUrl);

        networkReply =accessManager->post(*networkRequest, dataSent);
        pendingRequest=true;
        connect(networkReply,SIGNAL(finished()),SLOT(finished()));
        connect(networkReply,SIGNAL(error(QNetworkReply::NetworkError)),SLOT(error(QNetworkReply::NetworkError)));
        connect(networkReply,SIGNAL(uploadProgress(qint64,qint64)),SLOT(uploadProgress(qint64,qint64)));


    } else {
#ifdef Q_WS_MAEMO_5
        QMaemo5InformationBox::information(0, tr("Please fill the comments field."),
                                           QMaemo5InformationBox::DefaultTimeout);
#endif
    }
}

void FeedBackPlugin::finished()
{
    qDebug()<<"finished";
    if(networkReply->error()==QNetworkReply::NoError) {
        inputUI->clearDirtyFlag();
        inputUI->close();
        dataSent.clear();

#ifdef Q_WS_MAEMO_5
        QMaemo5InformationBox::information(0, tr("Thank you for your feedback."),
                                           QMaemo5InformationBox::DefaultTimeout);
#endif
    }
    networkReply->deleteLater();
    networkReply=0;
    pendingRequest=false;
}

void FeedBackPlugin::error(QNetworkReply::NetworkError code)
{
    if(code==413) {
#ifdef Q_WS_MAEMO_5
        QMaemo5InformationBox::information(0, tr("File size too large."),
                                           QMaemo5InformationBox::DefaultTimeout);
#endif
    } else {
        qDebug()<<"error "<<networkReply->errorString();
#ifdef Q_WS_MAEMO_5
        QMaemo5InformationBox::information(0, networkReply->errorString().trimmed(),
                                           QMaemo5InformationBox::DefaultTimeout);
#endif
    }
    return;
}

void FeedBackPlugin::uploadProgress(qint64 bytesSent, qint64 bytesTotal)
{
    emit progressPercent(((float)bytesSent)/bytesTotal*100);
    //qDebug()<<"upload progress"<<bytesSent<<" "<<bytesTotal;
}

void FeedBackPlugin::abort()
{
    if(networkReply) {
        networkReply->abort();
        dataSent.clear();
    }
    initFeedBackPlugin();
}

void FeedBackPlugin::initFeedBackPlugin()
{
    networkReply=0;
    networkRequest=0;
    accessManager=0;
    pendingRequest=false;
    dataSent.clear();
}

Q_EXPORT_PLUGIN2(feedbackplugin, FeedBackPlugin)
