from ui.Widget import Widget
from ui.Pixmap import Pixmap, pixmap_for_text, TEMPORARY_PIXMAP
from ui.KineticScroller import KineticScroller
from theme import theme


# width of visible scala portion on screen in KHz
_SCALE_WIDTH = 3000

_FREQ_STEP = 50


class FMScala(Widget):
    """
    Widget for a scrollable FM scala with station labels.
    """

    EVENT_FREQUENCY_CHANGED = "event-frequency-changed"
    EVENT_FREQUENCY_ANNOUNCED = "event-frequency-announced"
    

    def __init__(self):

        # width of one MHz in pixels on the scala
        self.__unit_width = 1
    
        # buffer for offscreen drawings
        self.__buffer = None
        
        # whether the buffer contents are valid
        self.__buffer_invalidated = True
    
        # list of stations (freq, name)
        self.__stations = []
        
        # station pixmaps: freq -> pixmap
        self.__station_cache = {}
        
        # current range of the scala
        self.__scala_range = (86000, 109500)
        self.__freq_range = (87500, 108000)
        
        # offset into the scala (center of the screen) in pixels
        self.__offset = 0
        
        # total width of the scala in pixels
        self.__total_width = 0
        
        # scaling factor for converting screen coordinates to frequency values
        self.__scaling = 1.0
    

        Widget.__init__(self)
        
        # kinetic scrolling
        self.__kscr = KineticScroller(self)
        self.__kscr.set_drag_threshold(0)
        #kscr.connect_clicked(self.__on_click)
        self.__kscr.connect_scrolling_stopped(self.__on_stop_scrolling)

        self.set_size(100, 100)


    def connect_frequency_changed(self, cb, *args):
        
        self._connect(self.EVENT_FREQUENCY_CHANGED, cb, *args)


    def connect_frequency_announced(self, cb, *args):
        
        self._connect(self.EVENT_FREQUENCY_ANNOUNCED, cb, *args)


    def __on_stop_scrolling(self):

        freq = self.__round_to_nearest()
        self.emit_event(self.EVENT_FREQUENCY_CHANGED, freq)
        
        
    def __prepare_scala(self):
        """
        Prepares the scala by precomputing some values.
        """

        r1, r2 = self.__scala_range
        ticks_w = theme.scala_ticks.get_width()
        
        units = (r2 - r1) / 1000.0
        self.__unit_width = ticks_w
        self.__total_width = int(ticks_w * units)

        
    def set_size(self, w, h):
    
        old_w, old_h = self.get_size()

        if ((w, h) != (old_w, old_h)):
            freq = self.get_value()
            Widget.set_size(self, w, h)
            self.__buffer = Pixmap(None, w, h)
            self.__buffer_invalidated = True
            #self.__move_to(freq)

        else:
            Widget.set_size(self, w, h)
        
        
    def render_this(self):

        x, y = self.get_screen_pos()
        w, h = self.get_size()
        screen = self.get_screen()

        if (self.__buffer_invalidated):
            self.__buffer_invalidated = False
            self.__render(0, w)

        # render overlays
        TEMPORARY_PIXMAP.copy_buffer(self.__buffer, 0, 0, 0, 0, w, h)
        self.render_overlays(TEMPORARY_PIXMAP)
        TEMPORARY_PIXMAP.draw_pixbuf(theme.scala_glass_and_needle, 0, 0)

        screen.copy_buffer(TEMPORARY_PIXMAP, 0, 0, x, y, w, h)
        #screen.copy_buffer(self.__buffer, 0, 0, x, y, w, h)



    def __render(self, x1, x2):
        """
        Renders the given portion of the scala. The given x-coordinates specify
        screen positions.
        """

        if (not self.__buffer and not self.__scala_pmap): return

        #x, y = self.get_screen_pos()
        w, h = self.get_size()
        ticks_w = theme.scala_ticks.get_width()
        ticks_h = theme.scala_ticks.get_height()
        rng1, rng2 = self.__freq_range

        #screen = self.get_screen()    

        self.__buffer.set_clip_rect(x1, 0, x2 - x1, h)
        self.__buffer.fill_area(x1, 0, x2 - x1, h, theme.color_scala_background)

        offset = (self.__offset + x1) % ticks_w
        remainder = ticks_w - offset
        
        # render ticks
        for x in range(x1 - offset, x2, ticks_w):
            self.__buffer.draw_pixbuf(theme.scala_ticks, x, h - ticks_h)
        #end if

        # render stations
        for freq, name in self.__stations:
            x = self.__freq_to_px(freq)
            if (self.__offset + x1 - offset <= x <= self.__offset + x2 + 50):
            #if (self.__offset + x1 <= x <= self.__offset + x2 - 27):
                pmap = self.__station_cache.get(freq)
                if (not pmap):
                    pmap = self.__make_station_pmap(x - self.__offset - 13, 0,
                                                    name)
                    if (self.__offset + x1 + 13 <= x <= self.__offset + x2 - 27):
                        self.__station_cache[freq] = pmap
                #else:
                #    print "FROM CACHE", name

                self.__buffer.copy_buffer(pmap,
                                          0, 0,
                                          x - self.__offset - 13, 0,
                                          27, 270)
                
            
                """
                pmap = self.__buffer.subpixmap(x - self.__offset - 13, 0,
                                               27, 270)
                pmap.rotate(270)
                pmap.draw_pixbuf(theme.scala_rdslabel, 0, 0)
                pmap.draw_text(name, theme.font_scala_station,
                               24, 0, theme.color_scala_station)
            
                pmap.rotate(90)
                self.__buffer.copy_buffer(pmap,
                                          0, 0,
                                          x - self.__offset - 13, 0,
                                          27, 270)
                del pmap
                """
            #end if
        #end for
        
        self.__buffer.set_clip_rect(None)

        # render frequency labels
        for x in range(x1 - offset, x2 + 100, ticks_w):
            freq = self.__px_to_freq(self.__offset + x)
            if (rng1 <= freq <= rng2):
                self.__buffer.fill_area(x - 50, 10, 100, 32,
                                        theme.color_scala_background)
                self.__buffer.draw_centered_text(`freq / 1000`,
                                                 theme.font_scala_numbers,
                                                 x - 50, 10, 100, 32,
                                                 theme.color_scala_numbers)
        #end for
            

        #self.__buffer.copy_buffer(self.__pmap_scala,
        #                          self.__offset + x1, 0,
        #                          x1, 0,
        #                          x2 - x1, h)

                
        
    def set_range(self, a, b):
        """
        Sets the range of the frequency band to display.

        @param a: begin of FM band
        @param b: end of FM band
        """
    
        self.__freq_range = (a, b)
        a -= 1500
        a -= a % 1000
        b += 2000
        b -= b % 1000
        self.__scala_range = (a, b)
        
        ticks_w = theme.scala_ticks.get_width()
        units = (b - a) / 1000.0
        self.__unit_width = ticks_w
        self.__total_width = int(ticks_w * units)


    def __make_station_pmap(self, x, y, name):
        """
        Creates and returns a station label pixmap.
        """

        pmap = self.__buffer.subpixmap(x, y, 27, 270)
        pmap.rotate(270)
        pmap.draw_pixbuf(theme.scala_rdslabel, 0, 0)
        pmap.draw_text(name, theme.font_scala_station,
                       24, 0, theme.color_scala_station)
        pmap.rotate(90)

        return pmap


    def set_stations(self, *stations):
        """
        Sets station names for frequencies that will be visible on the scala.
        
        @param stations: variable list of (frequency, name) tuples
        """
    
        self.__stations = stations
        self.__station_cache.clear()
        self.__buffer_invalidated = True
        
        
    def __freq_to_px(self, freq):
    
        r1, r2 = self.__scala_range
        px = int((freq - r1) / 1000.0 * self.__unit_width)
        
        return px
        
        
    def __px_to_freq(self, px):

        r1, r2 = self.__scala_range
        freq = r1 + (px / float(self.__unit_width)) * 1000
        freq = max(r1, min(r2, freq))

        return int(freq)


    def __round_freq(self, freq):
    
        r1, r2 = self.__freq_range
        freq = freq + _FREQ_STEP / 2
        freq -= freq % _FREQ_STEP
        freq = max(r1, min(r2, freq))
        return freq


    def __round_to_nearest(self):
    
        w, h = self.get_size()
        cur_freq = self.__px_to_freq(self.__offset + w / 2)

        freq = self.__round_freq(cur_freq)
        self.__move_to(freq)

        return freq


    def __announce_freq(self):
    
        w, h = self.get_size()
        freq = self.__px_to_freq(self.__offset + w / 2)
        self.emit_event(self.EVENT_FREQUENCY_ANNOUNCED, freq)


    def get_value_at(self, px):
        """
        Returns the frequency value at the given screen position.
        
        @param px: x-position on screen
        @return: frequency value in KHz at that position
        """
    
        return self.__px_to_freq(self.__offset + px)
        
        
        
    def set_value(self, freq, notify = True):
        """
        Sets the current frequency value.
        
        @param freq: frequency value in KHz
        """
    
        #w, h = self.get_size()
        #self.__offset = self.__freq_to_px(freq) - w / 2
    
        self.__move_to(freq)
        #freq = self.__round_to_nearest()
        if (notify):
            self.emit_event(self.EVENT_FREQUENCY_CHANGED, freq)
        
        
    def get_value(self):
        """
        Returns the current frequency value.
        
        @return: frequency value in KHz
        """
    
        w, h = self.get_size()
        
        return self.__round_freq(self.get_value_at(w / 2))


    def step_down(self):
    
        v = self.get_value()
        self.set_value(v - _FREQ_STEP)
        self.emit_event(self.EVENT_FREQUENCY_CHANGED, v - _FREQ_STEP)


    def step_up(self):
    
        v = self.get_value()
        self.set_value(v + _FREQ_STEP)
        self.emit_event(self.EVENT_FREQUENCY_CHANGED, v + _FREQ_STEP)


    def move(self, dx, dy):
    
        x, y = self.get_screen_pos()
        w, h = self.get_size()
        screen = self.get_screen()

        if (dx < 0 and self.__offset + dx >= 0):
            # ->
            self.__offset += dx
            self.__buffer.move_area(0, 0, w - abs(dx), h, abs(dx), 0)
            self.__render(0, abs(dx))
            self.render()
            self.__announce_freq()
            return (dx, 0)
            
        elif (dx > 0 and self.__offset + dx < self.__total_width - w):
            # <-
            self.__offset += dx
            self.__buffer.move_area(dx, 0, w - dx, h, -dx, 0)
            self.__render(w - abs(dx), w)
            self.render()
            self.__announce_freq()
            return (dx, 0)            

        else:
            return (0, 0)



    def __move_to(self, freq):
    
        def fx(params):
            from_x, to_x = params
            dx = (to_x - from_x) / 2

            if (abs(dx) > 0):
                self.move(dx, 0)

                params[0] = from_x + dx
                params[1] = to_x                
                return True

            else:
                dx = to_x - from_x
                self.move(dx, 0)
                return False

        w, h = self.get_size()
        new_pos = self.__freq_to_px(freq) - w / 2
        self.animate_with_events(100, fx, [self.__offset, new_pos])
        self.__offset = new_pos

