from com import Component, msgs
from ui.EventBox import EventBox
from ui.ImageButton import ImageButton
from ui.Label import Label
from ui.itemview import ThumbableGridView
from ui.Slider import VSlider
from ui.dialog import InputDialog
from ui.dialog import OptionDialog
from ui.dialog import ListDialog
from ui.Pixmap import Pixmap, text_extents
from ui.Window import Window
from ui.decorators import Gestures
from StationItem import StationItem
from FMScala import FMScala
from FMRadio import *
from SoundPipe import SoundPipe
import platforms
import values
from theme import theme
import config

import gtk
import gobject


_DEVICE_ERRORS = [
    "",
    "Error: Cannot activate Bluetooth/FM combo chip",
    "Error: Cannot establish communication to the FM receiver",
    "Error: Cannot load FM radio driver module",
    "Error: Cannot power up FM radio",
    "Error: FM radio is not available in offline mode"
]


_APP_MENU_SKEL = """
  <menu>
    <choice id="output" selected="%d">
      <option label="Headphones" icon=""/>
      <option label="Speakers" icon=""/>
    </choice>
    %s
  </menu>
"""

_APP_MENU_ADDING = """
  <item id="add" label="Add Station"/>  
  <!--
  <item id="manual" label="Manual"/>
  <item id="autoscan" label="Autoscan"/>
  -->
  <item id="info" label="About"/>
"""

_APP_MENU_EDITING = """
  <item id="rename" label="Rename"/>
  <item id="info" label="About"/>
  <item id="delete" label="Remove"/>
  <!--
  <item id="autoscan" label="Autoscan"/>
  -->
  <!--
  <item id="manual" label="Manual"/>
  -->
"""


_FREQ_STEP = 50


class FMRadioUI(Component, Window):
    """
    Player for FM radio stations.
    """

    def __init__(self):

        self.__buffer = None

        self.__radio = None
        self.__keep_alive_handler = None
        self.__rds_handler = None
        self.__current_rds_ps = ""
        self.__current_rds_rt = ""
        self.__device = platforms.get_product_code()

        self.__sound_pipe = SoundPipe()
        self.__use_speaker = False
        self.__is_interrupted = False
        #self.__was_playing_before_interruption = False
        self.__have_headphones = False
        
        self.__stations = config.get_stations()
        self.__stations_dirty = False
        

        Component.__init__(self)
        Window.__init__(self, Window.TYPE_TOPLEVEL)
        self.set_size(800, 423)
        self.set_title("FM Radio")
                                      
        self.connect_closed(self.__on_close_window)
        self.connect_key_pressed(self.__on_key_pressed)
        
        
        # [Select] button
        self.__btn_select = EventBox()
        self.__btn_select.connect_clicked(self.__on_btn_select)
        self.add(self.__btn_select)


        # FM scala
        self.__fmscala = FMScala()
        #self.__fmscala.add_overlayer(self.__overlay_glass)
        #self.__fmscala.add_overlayer(self.__overlay_navigation_buttons)
        self.__fmscala.set_range(87500, 108000)
        #self.__fmscala.set_value(config.get_lastfreq())
        self.__fmscala.set_stations(*self.__stations)
        self.__fmscala.connect_frequency_changed(self.__on_tune)
        self.__fmscala.connect_frequency_announced(self.__on_freq_announced)
        #self.__fmscala.connect_button_released(self.__on_click_scala)
        self.add(self.__fmscala)
      
        # label for RDS text
        self.__lbl_rds = Label("", theme.font_rds_text,
                               theme.color_rds_text)
        self.__lbl_rds.set_alignment(Label.CENTERED)
        self.add(self.__lbl_rds)

        # label for station name
        self.__lbl_station = Label("", theme.font_rds_name,
                                   theme.color_rds_name)
        self.__lbl_station.set_alignment(Label.CENTERED)
        self.add(self.__lbl_station)

        # label for frequency
        self.__lbl_freq = Label("", theme.font_frequency,
                                theme.color_frequency)
        self.__lbl_freq.set_alignment(Label.CENTERED)
        self.add(self.__lbl_freq)

        # navigation buttons
        self.__btn_prev = ImageButton(theme.bt_prev,
                                      theme.bt_prev_down)
        self.add(self.__btn_prev)
        gst = Gestures(self.__btn_prev)
        gst.connect_tap(lambda *a:self.__skip_preset(-1))
        gst.connect_hold(lambda *a:self.__scan_previous())

        self.__btn_next = ImageButton(theme.bt_next,
                                      theme.bt_next_down)
        self.add(self.__btn_next)
        gst = Gestures(self.__btn_next)
        gst.connect_tap(lambda *a:self.__skip_preset(1))
        gst.connect_hold(lambda *a:self.__scan_next())


        self.set_visible(True)
        gobject.idle_add(self.__init_app)


    def __init_app(self):
        """
        Initializes the application.
        """
    
        if (self.__assert_headphones()):
            self.__radio_on()
            
        if (self.__radio):
            self.__fmscala.set_value(config.get_lastfreq())
            self.__display_note("The Bluetooth module is responsible for\n" \
                                "FM radio and enabled automatically.")

        self.__update_menu()


    def __display_note(self, msg):
        """
        Displays a banner note with the given message string.
        """
          
        try:
            import hildon
            hildon.hildon_banner_show_information(self.get_gtk_window(),
                                                  "", msg)
        except:
            print msg



    def __prepare_buffer(self):
        """
        Prepares the offscreen rendering buffer after resizing.
        """
    
        w, h = self.get_size()
        if (not self.__buffer):
            self.__buffer = Pixmap(None, w, h)
        else:
            buf_w, buf_h = self.__buffer.get_size()
            if ((buf_w, buf_h) != (w, h)):
                self.__buffer = Pixmap(None, w, h)


       
    def render_this(self):
    
        x, y = self.get_screen_pos()
        w, h = self.get_size()
        screen = self.get_screen()
        
        screen.draw_pixbuf(theme.top, 0, 0)
        self.__lbl_rds.set_geometry(10, 0, w - 20, 0)
        self.__fmscala.set_geometry(81, 28, 638, 270)
        self.__btn_prev.set_geometry(0, 28, 81, 270)
        self.__btn_next.set_geometry(w - 81, 28, 81, 270)
        
        self.__render_bottom()
        
        
    def __render_bottom(self):

        x, y = self.get_screen_pos()
        w, h = self.get_size()
        screen = self.get_screen()
        
        screen.draw_pixbuf(theme.bottom, 0, h - 125)
                
        self.__lbl_station.set_geometry(0, h - 100, w, 0)
        self.__lbl_freq.set_geometry(0, h - 60, w, 0)
        self.__btn_select.set_geometry(0, h - 100, w, 100)

        #t_w, t_h = text_extents(self.__lbl_station.get_text(),
        #                        theme.font_rds_name)
        screen.draw_pixbuf(theme.bt_rdsname_arrow,
                           w - 64, h - 64)

        

    def __render_buffered(self):
    
        if (self.may_render()):
        
            if (not self.__buffer):
                self.__prepare_buffer()
        
            x, y = self.get_screen_pos()
            w, h = self.get_size()
            screen = self.get_screen()
            
            self.render_at(self.__buffer)
            screen.copy_buffer(self.__buffer, 0, 0, x, y, w, h)
        #end if

    
    def __update_menu(self):
        """
        Updates the application menu based on current context.
        """
        
        name = self.__get_current_station_name()
        if (name):
            m = _APP_MENU_EDITING
        else:
            m = _APP_MENU_ADDING

        menu = _APP_MENU_SKEL % (self.__use_speaker and 1 or 0, m)
        self.set_menu_xml(menu, {"output": self.__on_menu_output,
                                 "info": self.__on_menu_info,
                                 "add": self.__on_menu_add,
                                 "delete": self.__on_menu_delete,
                                 "rename": self.__on_menu_rename})


    def __overlay_glass(self, obj, screen):
        """
        Overlays the glass pane.
        """
        
        x, y = obj.get_screen_pos()
        screen.draw_pixbuf(theme.scala_glass, 0, 0)
    


    def __overlay_navigation_buttons(self, obj, screen):
        """
        Overlays the navigation buttons.
        """
        
        x, y = obj.get_screen_pos()
        w, h = obj.get_size()

        screen.draw_pixbuf(theme.mb_btn_previous_1,
                           0, (h - 64) / 2)

        screen.draw_pixbuf(theme.mb_btn_next_1,
                           w - 64, (h - 64) / 2)
                                           


    def __on_close_window(self):
    
        def f():
            if (self.__stations_dirty):
                config.set_stations(self.__stations)

            gtk.main_quit()
    
    
        self.__radio_off()
        if (self.__stations_dirty):
            self.__display_note("Saving Stations")

        gobject.idle_add(f)


    def __on_key_pressed(self, key):
    
        if (key == "Left"):
            self.__fmscala.step_down()
            
        elif (key == "Right"):
            self.__fmscala.step_up()

            

    def __on_menu_add(self):
    
        freq = self.__fmscala.get_value()
        name = ""
        
        # try to retrieve name from RDS
        if (self.__radio):
            pi, ps, rt = self.__radio.get_rds()
            name = ps.strip()

        if (not name):
            dlg = InputDialog("Enter Station Name")
            dlg.add_input("Name", "%0.2f MHz" % (freq / 1000.0))
    
            ret = dlg.run()
            if (ret == 0):                
                name = dlg.get_values()[0] or "%0.2f MHz" % (freq / 1000.0)
            
        if (name):
            self.__stations.append((freq, name))
            self.__fmscala.set_stations(*self.__stations)

            self.__update_menu()
            self.__render_buffered()
            self.__stations_dirty = True
        
        
    def __on_menu_delete(self):
    
        dlg = OptionDialog("Remove Station?")
        dlg.add_option(None, "Remove")
        dlg.add_option(None, "Cancel")
        
        ret = dlg.run()
        if (dlg.get_choice() == 0):
            freq = self.__fmscala.get_value()
            new_stations = []
            for f, n in self.__stations:
                if (freq == f):
                    pass
                else:
                    new_stations.append((f, n))
            #end for
        
            self.__stations = new_stations
            self.__fmscala.set_stations(*self.__stations)

            self.__update_menu()
            self.__render_buffered()
            self.__stations_dirty = True
        #end if
        
        
    def __on_menu_rename(self):

        dlg = InputDialog("Rename Radio Station")
        dlg.add_input("Name", self.__get_current_station_name())
        
        ret = dlg.run()
        if (ret != 0):
            return
        new_name = dlg.get_values()[0]
    
        new_stations = []
        freq = self.__fmscala.get_value()
        for f, n in self.__stations:
            if (freq == f):
                new_stations.append((f, new_name))
            else:
                new_stations.append((f, n))
        #end for
        
        self.__stations = new_stations
        self.__fmscala.set_stations(*self.__stations)

        self.__update_menu()
        self.__stations_dirty = True
        

    def __on_menu_info(self):
    
        self.__display_note("%s - version %s\n\n" \
                            "Written by Martin Grimme\n" \
                            "Artwork by Patricia Montenegro\n" \
                            "Icon by Andrew Zhilin" \
                            % (values.NAME, values.VERSION))
        

    def __on_menu_output(self, choice):
    
        if (choice == 0):
            self.__use_speaker = False
        else:
            self.__use_speaker = True

        if (self.__radio):
            self.__sound_pipe.use_speaker(self.__use_speaker)


    def __on_click_scala(self, px, py):
    
        w, h = self.__fmscala.get_size()
        if (px < 80):
            self.__skip_preset(-1)
        elif (px > w - 80):
            self.__skip_preset(1)

      
    def __on_freq_announced(self, freq):
    
        self.__lbl_freq.set_text("%0.2f MHz" % (freq / 1000.0))
      
        
    def __on_tune(self, v):
    
        if (self.__assert_headphones()):
            if (not self.__radio):
                self.__radio_on()
            else:
                self.__radio.cancel_scanning()
                
            self.__set_frequency(v)
        #end if
        
        self.__update_menu()

        #if (self.__fmscala.get_value() != v):
        #    self.__fmscala.set_value(v)
      

    def __on_btn_select(self):
    
        if (not self.__stations):
            self.__display_note("You don't have stations saved")
            return
    
        self.__stations.sort(lambda a,b:cmp(a[1], b[1]))

        dlg = ListDialog("Stations")
        for freq, name in self.__stations:
            item = StationItem(name, freq)
            dlg.add_item(item)
        #end for
        
        ret = dlg.run()
        if (ret == 0):
            item = dlg.get_choice()
            freq = item.get_frequency()
            self.__fmscala.set_value(freq)
        #end if


    def __get_current_station_name(self):
        """
        Returns the name of the currently selected station, or an empty
        string if there is no station set at the moment.
        """
    
        freq = self.__fmscala.get_value()
        stations = [ n for f, n in self.__stations
                     if f == freq ]

        if (stations):
            return stations[0]
        else:
            return ""



    def __assert_headphones(self):
    
        if (not self.__have_headphones):
            self.__display_note("Please insert a headphones cable for a " \
                                "FM antenna.")
            return False
        else:
            return True


    def __show_error(self, err):
    
        msg = _DEVICE_ERRORS[err]
        self.__display_note(msg)


    def __radio_on(self):

        if (self.__radio): return

        if (self.__device == "RX-51"):
            ret, dev = platforms.request_fmradio()
        else:
            ret = 1

        print "return code from FMRX-Enabler:", ret

        if (ret == 0):
            try:
                self.__radio = FMRadio(self.__device)
            except:
                return
        else:
            self.__show_error(ret)
            return
        #end if
        
        if (self.__device == "RX-51"):
            if (self.__keep_alive_handler):
                gobject.source_remove(self.__keep_alive_handler)
            if (self.__rds_handler):
                gobject.source_remove(self.__rds_handler)
            
            self.__keep_alive_handler = gobject.timeout_add(15000,
                                                    self.__keep_radio_alive)
            self.__rds_handler = gobject.timeout_add(500, self.__update_rds)
        #end if

        self.__sound_pipe.on()
        self.__sound_pipe.use_speaker(self.__use_speaker)
        
        
        
    def __radio_off(self):
    
        self.__sound_pipe.off()
        if (not self.__radio): return
        
        config.set_lastfreq(self.__radio.get_frequency())
        self.__radio.close()
        self.__radio = None


    def __skip_preset(self, direction, take_farthest = False):
        """
        Skips to the next preset in the given direction (-1, or 1).
        """

        if (not self.__stations):
            # no presets available...
            return

        current_freq = self.__fmscala.get_value()
        
        # find stations and distance to each station
        if (direction > 0):
            candidates = [ (freq, freq - current_freq)
                           for freq, name in self.__stations
                           if freq > current_freq + _FREQ_STEP / 2 ]
        else:
            candidates = [ (freq, freq - current_freq)
                           for freq, name in self.__stations
                           if freq < current_freq - _FREQ_STEP / 2 ]

        if (candidates):
            # sort by absolute distance
            candidates.sort(lambda a,b:cmp(abs(a[1]), abs(b[1])))
        
            if (take_farthest):
                # take farthest (for wrapping around)
                new_freq, dist = candidates[-1]
            else:
                # take nearest
                new_freq, dist = candidates[0]
                
            self.__fmscala.set_value(new_freq)

        elif (not take_farthest):
            # try to wrap around
            direction = 0 - direction
            self.__skip_preset(direction, take_farthest = True)

        #end if        


    def __on_scan_progress(self, freq):
    
        if (freq % 200 == 0):
            self.__fmscala.set_value(freq, False)
            self.__lbl_freq.set_text("%0.2f MHz" % (freq / 1000.0))
        
        while (gtk.events_pending()):
            gtk.main_iteration(False)


    def __scan_previous(self):
    
        if (self.__radio):
            self.__lbl_station.set_text("... Scanning ...")
            current = self.__radio.get_frequency()
            freq = self.__radio.scan_previous(self.__on_scan_progress)
            if (freq != current):
                self.__fmscala.set_value(freq)
                self.__update_menu()
        

    def __scan_next(self):
    
        if (self.__radio):
            self.__lbl_station.set_text("... Scanning ...")
            current = self.__radio.get_frequency()
            freq = self.__radio.scan_next(self.__on_scan_progress)
            if (freq != current):
                self.__fmscala.set_value(freq)
                self.__update_menu()


    def __set_frequency(self, freq):
        """
        Sets the radio and scala to the given frequency.
        """
    
        if (self.__radio):
            self.__radio.set_frequency(freq)
            
        self.__current_rds_ps = ""
        name = self.__get_current_station_name()
        self.__lbl_station.set_text(name)
        self.__lbl_rds.set_text("")


    def __update_rds(self):
        """
        Updates RDS display.
        """

        if (self.__radio):
            pi, ps, rt = self.__radio.get_rds()
            ps = ps.strip() or ""
            rt = rt.strip() or ""
            
            if (ps):
                self.__lbl_station.set_text(ps)
                self.__current_rds_ps = ""
                
            if (rt and not self.__current_rds_rt.startswith(rt)):
                self.__lbl_rds.set_text(rt)

                self.__current_rds_rt = rt

            return True

        else:
            self.__rds_handler = None
            return False


    def __keep_radio_alive(self):
        """
        Keeps the radio driver alive. This is necessary with the
        N900-FMRX-Enabler.
        """
    
        if (self.__radio):
            platforms.request_fmradio()
            return True
        else:
            self.__keep_alive_handler = None
            return False


    def __begin_interruption(self):
        """
        Begins an interruption and switches off the radio, if on.
        """
        
        if (self.__is_interrupted): return
        self.__is_interrupted = True
        
        if (self.__radio):
            #self.__was_playing_before_interruption = True
            self.__display_note("FM Radio Interrupted")
            
        #else:
        #    self.__was_playing_before_interruption = False

        self.__radio_off()


    def __end_interruption(self):
        """
        Resumes after an interruption ends.
        """

        if (not self.__is_interrupted or not self.__have_headphones): return
        self.__is_interrupted = False
        
        #if (self.__was_playing_before_interruption):
        self.__radio_on()
 

    def handle_CORE_EV_FOLDER_INVALIDATED(self, f):
    
        self.__browser.reload_current_folder()


    def handle_SYSTEM_EV_HEADPHONES_INSERTED(self):
    
        self.__have_headphones = True
        # become active only after the policy decision was broadcast, or the
        # policy will override the our settings...
        #self.__end_interruption()
        
        
    def handle_SYSTEM_EV_HEADPHONES_REMOVED(self):

        self.__have_headphones = False
        self.__begin_interruption()


    def handle_SYSTEM_EV_SOUND_VOLUME_CHANGED(self, v):
    
        # the hw keys don't affect the mixer for the speaker volume, so we
        # have to set that manually
        self.__sound_pipe.set_speaker_volume(v)


    """
    Phone stuff is well-covered by the policy decisions.
    
    def handle_SYSTEM_EV_PHONE_RINGING(self):
    
        self.__begin_interruption()


    def handle_SYSTEM_EV_PHONE_DIALING(self):

        self.__begin_interruption()


    def handle_SYSTEM_EV_PHONE_HANGUP(self):

        self.__end_interruption()
    """


    def handle_SYSTEM_EV_OFFLINE_MODE(self):
    
        self.__begin_interruption()
        
    
    def handle_SYSTEM_EV_NORMAL_MODE(self):
    
        self.__end_interruption()


    def handle_POLICY_EV_FMRADIO_DISALLOWED(self):
    
        self.__begin_interruption()
        
    
    def handle_POLICY_EV_FMRADIO_ALLOWED(self):
    
        self.__end_interruption()

