/*
 *  Flashlight applet (widget) for Maemo.
 *  Copyright (C) 2009 Roman Moravcik
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/ioctl.h>

#include <asm/types.h>
#include <linux/videodev2.h>

#include "flashlight_lib.h"

int flashlight_get_status (FlashlightContext_t *flashlight, int *status)
{
	struct v4l2_control ctrl;

	printf ("flashlight_get_status()\n");

	if (flashlight == NULL) {
		printf ("flashlight_get_status: flashlight context is not valid\n");
		return ENOCONTEXT;
	}

	if (flashlight->fd == -1) {
		printf ("flashlight_get_status: device not openned\n");
		return ENODEVICE;
	}

	*status = 0;

	/* check short circuit fault */
	ctrl.id = V4L2_CID_FLASH_ADP1653_FAULT_SCP;
	if (ioctl (flashlight->fd, VIDIOC_G_CTRL, &ctrl) == -1) {
		printf ("flashlight_set_intensity: cannot get circuit fault status (%s)\n", strerror (errno));
		return EGENERROR;
	}

	if (ctrl.value)
		*status |= FLASHLIGHT_STATUS_SHORT_CIRCUT_FAULT;
	else
		*status &= ~FLASHLIGHT_STATUS_SHORT_CIRCUT_FAULT;

	/* check overtemperature fault */
	ctrl.id = V4L2_CID_FLASH_ADP1653_FAULT_OT;
	if (ioctl (flashlight->fd, VIDIOC_G_CTRL, &ctrl) == -1) {
		printf ("flashlight_set_intensity: cannot get overtemperature fault status (%s)\n", strerror (errno));
		return EGENERROR;
	}

	if (ctrl.value)
		*status |= FLASHLIGHT_STATUS_OVERTEMPERATURE_FAULT;
	else
		*status &= ~FLASHLIGHT_STATUS_OVERTEMPERATURE_FAULT;

	/* check timeout fault */
	ctrl.id = V4L2_CID_FLASH_ADP1653_FAULT_TMR;
	if (ioctl (flashlight->fd, VIDIOC_G_CTRL, &ctrl) == -1) {
		printf ("flashlight_set_intensity: cannot get timeout fault status (%s)\n", strerror (errno));
		return EGENERROR;
	}

	if (ctrl.value)
		*status |= FLASHLIGHT_STATUS_TIMEOUT_FAULT;
	else
		*status &= ~FLASHLIGHT_STATUS_TIMEOUT_FAULT;

	/* check overtemperature fault */
	ctrl.id = V4L2_CID_FLASH_ADP1653_FAULT_OV;
	if (ioctl (flashlight->fd, VIDIOC_G_CTRL, &ctrl) == -1) {
		printf ("flashlight_set_intensity: cannot get overvoltage fault status (%s)\n", strerror (errno));
		return EGENERROR;
	}

	if (ctrl.value)
		*status |= FLASHLIGHT_STATUS_OVERVOLTAGE_FAULT;
	else
		*status &= ~FLASHLIGHT_STATUS_OVERVOLTAGE_FAULT;

	return ENOERROR;
}

int flashlight_set_intensity (FlashlightContext_t *flashlight, int intensity)
{
	struct v4l2_control ctrl;

	printf ("flashlight_set_intensity(%d)\n", intensity);

	if (flashlight == NULL) {
		printf ("flashlight_set_intensity: flashlight context is not valid\n");
		return ENOCONTEXT;
	}

	if (flashlight->fd == -1) {
		printf ("flashlight_set_intensity: device not openned\n");
		return ENODEVICE;
	}

	if (intensity > flashlight->max_intensity)
		intensity = flashlight->max_intensity;

	ctrl.id = V4L2_CID_TORCH_INTENSITY;
	ctrl.value = intensity;

	if (ioctl (flashlight->fd, VIDIOC_S_CTRL, &ctrl) == -1) {
		printf ("flashlight_set_intensity: cannot set intensity (%s)\n", strerror (errno));
		return EGENERROR;
	}

	return ENOERROR;
}

int flashlight_get_intensity (FlashlightContext_t *flashlight, int *intensity)
{
	struct v4l2_control ctrl;

	printf ("flashlight_get_intensity()\n");

	if (flashlight == NULL) {
		printf ("flashlight_get_intensity: flashlight context is not valid\n");
		return ENOCONTEXT;
	}

	if (flashlight->fd == -1) {
		printf ("flashlight_get_intensity: device not openned\n");
		return ENODEVICE;
	}

	ctrl.id = V4L2_CID_TORCH_INTENSITY;

	if (ioctl (flashlight->fd, VIDIOC_G_CTRL, &ctrl) == -1) {
		printf ("flashlight_get_intensity: cannot get intensity (%s)\n", strerror (errno));
		return EGENERROR;
	}

	*intensity = ctrl.value;
	return ENOERROR;
}

int flashlight_open (FlashlightContext_t *flashlight, const char *device_name)
{
	struct v4l2_queryctrl ctrl;
	struct stat st;

	printf ("flashlight_open(%s)\n", device_name);

	if (flashlight == NULL) {
		printf ("flashlight_open: flashlight context is not valid\n");
		return ENOCONTEXT;
	}

	if (device_name == NULL) {
		printf ("flashlight_open: device name not specified\n");
		return EGENERROR;
	}

	memcpy (flashlight->device_name, device_name, sizeof(flashlight->device_name));

	if (stat (flashlight->device_name, &st) == -1) {
		printf ("flashlight_open: cannot identify '%s' (%s)\n", flashlight->device_name, strerror (errno));
		return EGENERROR;
	}

	/* check it device_name is real device */
	if (!S_ISCHR (st.st_mode)) {
		printf ("flashlight_open: %s is no device\n", flashlight->device_name);
		return EGENERROR;
	}

	flashlight->fd = open (flashlight->device_name, O_RDWR /* required */ | O_NONBLOCK, 0);

	if (flashlight->fd == -1) {
		printf ("flashlight_open: cannot open '%s' (%s)\n", flashlight->device_name, strerror (errno));
		return ENODEVICE;
	}

	/* query from driver minimal and maximal flashlight intensity */
	ctrl.id = V4L2_CID_TORCH_INTENSITY;
	if (ioctl (flashlight->fd, VIDIOC_QUERYCTRL, &ctrl) == -1) {
		printf ("flashlight_open: cannot get minimal and maximal flashlight intensity (%s)\n", strerror (errno));
		return EGENERROR;
	}

	flashlight->min_intensity = ctrl.minimum;
	flashlight->max_intensity = ctrl.maximum;

	return ENOERROR;
}

int flashlight_close (FlashlightContext_t *flashlight)
{
	printf ("flashlight_close()\n");

	if (flashlight == NULL) {
		printf ("flashlight_close: flashlight context is not valid\n");
		return ENOCONTEXT;
	}

	if (flashlight->fd != -1) {
		if (close (flashlight->fd) == -1) {
			printf ("flashlight_close: cannot close device '%s' (%s)\n", flashlight->device_name, strerror (errno));
			return ENODEVICE;
		}
	}

	flashlight->fd = -1;
	return ENOERROR;
}

int flashlight_init (FlashlightContext_t **pRefContext)
{
	FlashlightContext_t *flashlight = NULL;

	printf ("flashlight_init()\n");

	if (*pRefContext != NULL) {
		printf("flashlight_init: expecting zero pointer context '*pRefContext'\n");
		return EGENERROR;
	}

	/* allocate memory for context structure */
	flashlight = malloc (sizeof (FlashlightContext_t));
	if (flashlight == NULL) {
		printf ("flashlight_init: unable to allocate memory for context\n");
		return ENOCONTEXT;
	}

	*pRefContext = flashlight;

	/* initialize default values */
	memset (flashlight, 0x00, sizeof (FlashlightContext_t));
	flashlight->fd = -1;

	/* from adp1653.c */
	flashlight->min_intensity = 0;
	flashlight->max_intensity = 11;

	return ENOERROR;
}

int flashlight_deinit (FlashlightContext_t *flashlight)
{
	int intensity = 0;

	printf ("flashlight_deinit()\n");

	if (flashlight == NULL) {
		printf ("flashlight_deinit: flashlight context is not valid\n");
		return ENOCONTEXT;
	}

	if (flashlight->fd != -1) {
		/* check if flashlight isn't enabled before closing device */
		if (flashlight_get_intensity (flashlight, &intensity) == -1)
			return EGENERROR;

		if (intensity > 0) {
			if (flashlight_set_intensity (flashlight, 0) == -1)
				return EGENERROR;
		}

		if (flashlight_close(flashlight))
			return EGENERROR;
	}

	/* free allocated memory */
	free (flashlight);

	return ENOERROR;
}
