/*
    zimage-get-version.c - Get kernel version string from zImage
    Copyright (C) 2012  Pali Rohár <pali.rohar@gmail.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

*/

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <limits.h>
#include <errno.h>

#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include <linux/fs.h>
#include <mtd/mtd-user.h>

#include <zlib.h>

static const unsigned char match_str[] = "Linux version "; /* version string is after this str */
#define MATCH_SIZE	(sizeof(match_str)-1)

#define MAX_SIZE	(1 << 22)	/* 4 MB */
#define HEAD_SIZE	(1 << 16)	/* 64 kB */
#define BUF_SIZE 	(1 << 16)	/* 64 kB */
#define GZ_SIG		0x8B1F

/*
zImage contains binary code for decompressing GZIP archive and after that is concatenated itself gzipped kernel image.
This program try to find GZIP signature and try to start decompressing image. Signature must be in first 64 kB.
If decompression started, try to find matching Linux string and if success write version to stdout.
*/

/*
This program reject zImage files bigger than 4MB. Program can read zImage from regular file, block or mtd device.
Program does not check for zImage signature, so format can be other too which has concatenated GZIP archive.
*/

/*
This program using external zlib library for GZIP decompression. Compile it as:
gcc zimage-get-version.c -O2 -lz -o zimage-get-version
*/

int extract_version(const char * file, int force, char * version) {

	int fd;

	mtd_info_t mtd_info;
	struct stat st;
	ssize_t msize;
	uint64_t blksize;

	void * mptr;
	void * ptr;

	unsigned char buf[BUF_SIZE];
	unsigned char * bufptr;

	int ret;
	z_stream strm;

	int state;

	char ver[BUF_SIZE];
	char * verptr;

	int end;

	if ( force )
		end = MAX_SIZE;
	else
		end = HEAD_SIZE;

	if ( stat(file, &st) != 0 ) {
		fprintf(stderr, "Cannot stat file %s: %s\n", file, strerror(errno));
		return -1;
	}

	fd = open(file, O_RDONLY);

	if ( ! fd ) {
		fprintf(stderr, "Cannot open file %s: %s\n", file, strerror(errno));
		return -1;
	}

	if ( S_ISREG(st.st_mode) )
		msize = st.st_size;
	else if ( S_ISBLK(st.st_mode) ) {
		if ( ioctl(fd, BLKGETSIZE64, &blksize) != 0 ) {
			fprintf(stderr, "Cannot get size of block device %s\n", file);
			close(fd);
			return -1;
		}
		if ( blksize > SSIZE_MAX ) {
			fprintf(stderr, "Size of block device %s is too big (more then %ld bytes)\n", file, SSIZE_MAX);
			close(fd);
			return -1;
		}
		msize = blksize;
	} else if ( S_ISCHR(st.st_mode) && strncmp(file, "/dev/mtd", strlen("/dev/mtd")) == 0 ) {
		if ( ioctl(fd, MEMGETINFO, &mtd_info) != 0 ) {
			fprintf(stderr, "Cannot get info about mtd device %s\n", file);
			close(fd);
			return -1;
		}
		msize = mtd_info.size;
	} else {
		fprintf(stderr, "%s is not regular file, block device or mtd device\n", file);
		close(fd);
		return -1;
	}

	if ( msize <= 0 ) {
		fprintf(stderr, "Size of %s is 0\n", file);
		close(fd);
		return -1;
	} else if ( msize > MAX_SIZE ) {
		fprintf(stderr, "Size of %s is too big (more than %d bytes)\n", file, MAX_SIZE);
		close(fd);
		return -1;
	}

	mptr = malloc(msize);

	if ( ! mptr ) {
		fprintf(stderr, "Cannot allocate memory\n");
		close(fd);
		return -1;
	}

	if ( read(fd, mptr, msize) != msize ) {
		fprintf(stderr, "Cannot read %s: %s\n", file, strerror(errno));
		free(mptr);
		close(fd);
		return -1;
	}

	for ( ptr = mptr; ptr < mptr + end; ++ptr ) {

		if ( *(uint16_t *)ptr == GZ_SIG ) {

			state = 0;
			verptr = ver;

			strm.zalloc = Z_NULL;
			strm.zfree = Z_NULL;
			strm.opaque = Z_NULL;
			strm.avail_in = 0;
			strm.next_in = Z_NULL;

			ret = inflateInit2(&strm, 16 + MAX_WBITS);

			if ( ret != Z_OK )
				continue;

			strm.avail_in = mptr + msize - ptr;
			strm.next_in = ptr;

			while ( 1 ) {

				strm.avail_out = sizeof(buf);
				strm.next_out = buf;

				ret = inflate(&strm, Z_NO_FLUSH);

				if ( ret != Z_OK )
					break;

				for ( bufptr = buf; bufptr < buf + sizeof(buf) - strm.avail_out; ++bufptr ) {

					if ( state != MATCH_SIZE ) {

						if ( match_str[state] == *bufptr )
							++state;
						else
							state = 0;

					} else {

						if ( *bufptr <= 32 || (size_t)(verptr - ver + 1) >= sizeof(ver) ) {

							*verptr = 0;
							strcpy(version, ver);

							inflateEnd(&strm);
							free(mptr);
							close(fd);
							return 0;

						}

						*(verptr++) = *bufptr;

					}

				}

			}

			inflateEnd(&strm);

		}

	}

	free(mptr);
	close(fd);
	return 1;

}

int main(int argc, char * argv[]) {

	char buf[BUF_SIZE];
	char * file;
	int force;
	int ret;

	if ( argc == 3 && strcmp(argv[1], "-f") == 0 ) {
		force = 1;
		file = argv[2];
	} else if ( argc == 2 ) {
		force = 0;
		file = argv[1];
	} else {
		printf("Get kernel version string from zImage\n");
		printf("Usage: %s [-f] file|blkdev|mtddev\n", argv[0]);
		return 0;
	}

	ret = extract_version(file, force, buf);

	if ( ret < 0 )
		return 1;
	else if ( ret > 0 )
		fprintf(stderr, "Version string not found\n");
	else
		printf("%s\n", buf);

	return 0;

}
