# Copyright (c) 2011 Neal H. Walfield
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import logging
logger = logging.getLogger(__name__)
import traceback

# Don't fail if the Woodchuck modules are not available.  Just disable
# Woodchuck's functionality.

# Whether we imported the woodchuck modules successfully.
woodchuck_imported = True
try:
    import pywoodchuck
    from pywoodchuck import PyWoodchuck
    from pywoodchuck import woodchuck
except ImportError, exception:
    logger.info(
        "Unable to load Woodchuck modules: disabling Woodchuck support: %s"
        % traceback.format_exc ())
    woodchuck_imported = False
    class PyWoodchuck (object):
        def available(self, *args, **kwargs):
            return False
    woodchuck = None

# The default channel refresh interval: 6 hours.
refresh_interval = 6 * 60 * 60

class mywoodchuck (PyWoodchuck):
    def __init__(self, config, listing, human_readable_name, identifier,
                 request_feedback):
        try:
            PyWoodchuck.__init__ (self, human_readable_name, identifier,
                                  request_feedback)
        except Exception, e:
            logger.error(
                "Failed to establish a connection to the Woodchuck server: %s"
                % (str(e),))
            self.available = self.not_available
            return

        self.config = config
        self.listing = listing

        try:
            self.enabled = config.getWoodchuckEnabled()
        except Exception:
            logging.exception("Setting enabled")

    def available(self, check_config=True):
        if not PyWoodchuck.available(self):
            return False
        if check_config:
            return self.config.getWoodchuckEnabled()
        return True

    def not_available(self, *args, **kwargs):
        return False

    # Woodchuck upcalls.
    def stream_update_cb(self, stream, *args, **kwargs):
        logger.debug("stream update called on %s (%s)"
                     % (stream.human_readable_name, stream.identifier,))

        # Make sure no one else is concurrently updating this
        # feed.
        try:
            self.listing.updateFeed(stream.identifier)
        except:
            logger.debug("Updating %s: %s"
                         % (stream.identifier, traceback.format_exc ()))

    def object_transfer_cb(self, stream, object,
                           version, filename, quality,
                           *args, **kwargs):
        logger.debug ("object transfer called on %s (%s) in stream %s (%s)"
                      % (object.human_readable_name, object.identifier,
                         stream.human_readable_name, stream.identifier))
        try:
            self[stream.identifier][object.identifier].dont_transfer = True
        except Exception, e:
            logger.warn("Setting '%s'(%s).'%s'(%s).DontTransfer: %s"
                        % (stream.human_readable_name, stream.identifier,
                           object.human_readable_name, object.identifier,
                           str(e)))

_w = None
def wc_init(config, listing, request_feedback=False):
    """Connect to the woodchuck server and initialize any state."""
    global _w
    assert _w is None
    
    _w = mywoodchuck (config, listing, "FeedingIt", "org.marcoz.feedingit",
                      request_feedback)

    if not woodchuck_imported or not _w.available ():
        logger.info("Unable to contact Woodchuck server.")
    else:
        logger.debug("Woodchuck appears to be available.")

def wc_disable_set(disable=True):
    """Disable Woodchuck."""
    if disable:
        logger.info("Disabling Woodchuck")
    else:
        logger.info("Enabling Woodchuck")

    global _w
    if _w is None:
        logging.info("Woodchuck not loaded.  Not doing anything.")
        return

    if not _w.available(check_config=False):
        logging.info("Woodchuck not available.  Not doing anything.")
        return

    try:
        _w.enabled = not disable
    except Exception:
        logger.exception("Disabling Woodchuck")

def wc():
    """Return the Woodchuck singleton."""
    global _w
    assert _w is not None
    return _w
