#ifndef FCAM_TIFF_H
#define FCAM_TIFF_H

#include <string>
#include <map>
#include <vector>

#include <FCam/Image.h>
#include <FCam/TagValue.h>
#include <FCam/Event.h>

#include "TIFFTags.h"

namespace FCam {
    // Currently only handles first IFD (roughly, first page) of a TIFF file
    class TIFFFile {
    public:

        // A class representing a TIFF directory entry
        // Only reads in its data when asked for, which may require file IO
        class IfdEntry;
        // An object representing a TIFF directory, with accessors for
        // interpreting/constructing them
        class Ifd;

        TIFFFile(const std::string &file);
        TIFFFile();
        ~TIFFFile();

        bool readFrom(const std::string &file);
        bool writeTo(const std::string &file);

        bool valid;

        // Add a new top-level IFD to the file, and return a pointer to it
        Ifd *addIfd();
        // Remove all Ifds from the file
        void eraseIfds();

        // Access the whole set of Ifds in this file
        const std::vector<Ifd *> &ifds() const;
        // Get a pointer to a specific Ifd
        Ifd* ifds(int index);

        Event lastEvent;
    private:
        FILE *fp;
        std::string filename;

        bool littleEndian;
        uint32_t offsetToIfd0;

        std::vector<Ifd *> _ifds;

        uint16_t convShort(void const *src);
        uint32_t convLong(void const *src);
        float convFloat(void const *src);
        double convDouble(void const *src);
        TIFFRational convRational(void const *src);

        // Read an array of bytes from the file.
        bool readByteArray(uint32_t offset, uint32_t count, uint8_t *data);

        // Read an array of shorts (probably image data) from the file into dest
        bool readShortArray(uint32_t offset, uint32_t count, uint16_t *dest);

        bool readHeader();
        bool readIfd(uint32_t offsetToIFD, Ifd *ifd, uint32_t *offsetToNextIFD=NULL);
        bool readSubIfds(Ifd *ifd);

        void setError(std::string module, std::string description) {
            lastEvent.creator = NULL;
            lastEvent.type = Event::Error;
            lastEvent.data = Event::FileLoadError;
            lastEvent.time = Time::now();
            lastEvent.description = "TIFFFile::"+module+":"+filename+": "+description;
        }
    };

    class TIFFFile::IfdEntry {
    public:
        // Construct an IfdEntry from a raw TIFF IFD structure
        // Used when loading a TIFF file
        IfdEntry(const TiffIfdEntry &entry, TIFFFile *parent);
        // Construct an IfdEntry from a tag/value pair
        // Used when creating/modifying a TIFF file
        IfdEntry(uint16_t tag, const TagValue &val, TIFFFile *parent);
        // Construct an IfdEntry with just a tag
        // Used for searching for a matching IfdEntry in an Ifd
        IfdEntry(uint16_t tag, TIFFFile *parent=NULL);

        // Check for entry validity. May not be valid due to
        // type mismatches
        bool valid() const;

        // Retrieve the TIFF tag number of this entry
        uint16_t tag() const;
        // Retrieve the name of this entry
        const char *name() const;
        // Retrieve the current value of this entry
        // May cause file IO if loading a TIFF file and this
        // entry has not been read before
        const TagValue& value() const;

        // Change the value of this entry
        bool setValue(const TagValue &);
        // Writes excess data to file, and updates local offset pointer
        bool writeDataBlock(FILE *fw);
        // Writes entry to file. Assumes writeDataBlock has already been done to update entry offset field.
        bool write(FILE *fw);

        bool operator<(const IfdEntry &other) const;
    private:
        TiffIfdEntry entry;
        const TiffEntryInfo *info;
        TIFFFile *parent;

        TagValue parse() const;

        mutable enum {
            INVALID,
            UNREAD,
            READ,
            WRITTEN
        } state; // Tracks caching state

        mutable TagValue val; // Cached value
    };

    class TIFFFile::Ifd {
    public:
        Ifd(TIFFFile *parent);
        ~Ifd();

        // Return a pointer to the raw IFD entry structure matching tag in the given IFD, if one
        // exists, NULL otherwise
        const IfdEntry *find(uint16_t tag) const;
        IfdEntry *find(uint16_t tag);

        // Adds an entry created from a raw TIFF entry structure
        // Used in reading a TIFF.
        // Does not replace an existing entry if there is one already
        // with the same tag.
        bool add(const TiffIfdEntry &);
        // Adds an entry with a given tag and value.
        // In case of type mismatch, returns false.
        // Replaces the existing tag if there is one.
        // Used when writing a TIFF.
        bool add(uint16_t tag, const TagValue &val);
        // Adds an entry with a given tag name and value. If name
        // is unknown, returns false.  Used when writing a TIFF.
        // Replaces an existing tag if there is one.
        bool add(const std::string &tagName, const TagValue &val);

        // Constructs a new subIfd for this Ifd, and returns a pointer to it
        Ifd* addSubIfd();
        // Erase all subIfds
        void eraseSubIfds();

        // Get a reference to a vector of subIfds
        const std::vector<Ifd *> &subIfds();
        // Get a pointer to a specific subIfd
        Ifd* subIfds(int index);

        // Constructs the image referred to by this Ifd. Will require IO to the file if the image
        // hasn't been read already.
        Image getImage();
        // Sets the image to be saved in this Ifd.
        bool setImage(Image newImg);

        // Write all entries, subIFds, and image data to file
        // Retuns success/failure, and the starting location of the Ifd in
        // the file in offset.
        bool write(FILE *fw, uint32_t prevIfdOffset, uint32_t *offset);
    private:
        TIFFFile * const parent;

        std::vector<Ifd *> _subIfds;

        typedef std::map<int, IfdEntry> entryMap;
        entryMap entries;
        enum {
            UNREAD,
            NONE,
            CACHED
        } imgState;
        Image imgCache;

        // Subfunction to write image data out, and to update the IFD
        // entry offsets for it
        bool writeImage(FILE *fw);
    };

}

#endif
