#include "textprocessor.h"
#include "ui_textprocessor.h"

#define STATE_SPACE 0
#define STATE_SYM 1
#define STATE_WORD 2
#define STATE_UNKNOWN 3
#define STATE_DEL 3
#define STATE_MOVECUR_BUTTON 3
#define STATE_CURMOVED 5
#define STATE_AFTERCURMOVED 6
#define STATE_SEND 8

#define ACTION_INS 0
#define ACTION_DEL 1
#define ACTION_SEND 2

#define SINGLE_CHAR_TIMEOUT 750
#define CAPITALIZATION_CHAR ".!?" //After those character, uppercase is needed


TextProcessor::TextProcessor(QWidget *parent) :
        QWidget(parent),
        ui(new Ui::TextProcessor)
{
    ui->setupUi(this);
    m_sMessage = NULL;
    m_recipient = ui->recipientWidget;
    connect(m_recipient->m_btSend,SIGNAL(clicked()),this,SLOT(sendSms()));
    m_smsEdit = ui->textEdit;
    m_phoneNumber = m_recipient->getPhoneNumberField();

    //Enabling scrolling
#if defined(Q_WS_MAEMO_5)
    QAbstractKineticScroller *scroller = m_smsEdit->property("kineticScroller") .value<QAbstractKineticScroller *>();
    if (scroller)
    {
        scroller->setEnabled(true);
        scroller->setOvershootPolicy(QAbstractKineticScroller::OvershootAlwaysOff);
    }
#endif

    //da controllare
    m_nInputSyle = SettingsMenu::AutoCapitalize;
    //init variables for standard input style
    m_bTimerNewChar = true;
    m_timerNewChar = new QTimer(this);
    connect(m_timerNewChar,SIGNAL(timeout()),this,SLOT(alarmNewChar()));
    m_timerNewChar->setSingleShot(true);
    m_timerNewChar->setInterval(SINGLE_CHAR_TIMEOUT);

    //init variables
    m_nWordIndex=0;

    m_sCompleteValue = QString();
    m_cursorSmsText = m_smsEdit->textCursor(); // retrieve the visible cursor
    m_nMsgLength = 0;
    m_nBeginWord = 0;
    m_nEndWord = 0;
    m_nCharCount = 160;
    m_nSMSCount = 1;
    //init variables for capitalization
    m_sCapitalization = CAPITALIZATION_CHAR;

    m_bFastInput = true;
    m_bRememberUppercase = false;

    //manage autorotate from start
    this->setFocus();

    //Synchronize with active kb
    QSettings settings;
    if(settings.value(Core::SETTING_CURRENT_KEYBOARD) == (Core::Qwerty))
    {
        settingsChanged(SettingsMenu::QwertyOn);
        m_bQwertyEnable = true;
    }
    else
    {
        settingsChanged(SettingsMenu::AlfanumOn);
        m_bQwertyEnable = false;
    }
}

void TextProcessor::setDictionary(dictionary *dict)
{
    m_dict = dict;
}

void TextProcessor::processInput(QString sInput,Core::Keyboards sKeyboard)
{
    bool bFocusPhoneNumber = false;
    if(m_phoneNumber->hasFocus())
        bFocusPhoneNumber = true;

    if(!bFocusPhoneNumber)
    {
        m_smsEdit->setFocus();
        if(sKeyboard == Core::Qwerty)
            processQwerty(sInput);
        else
            processAlfanumeric(sInput);

        //Scroll text edit if necessary
        m_smsEdit->ensureCursorVisible();
    }
    else
    {
        if(sInput.compare(Keyboard::DELETE) == 0)
            m_phoneNumber->backspace();
        else
            m_phoneNumber->insert(sInput);
    }
}

void TextProcessor::addNewWord(QString *sWord)
{
    m_nState = STATE_WORD;

    m_nBeginWord = m_cursorSmsText.anchor();
    m_nEndWord=m_nBeginWord + sWord->length();

    m_sCompleteValue.clear();
    m_sCompleteValue.append(m_dict->calculateKey(sWord));
    m_lWord = m_dict->getWordList(&m_sCompleteValue);
    m_nWordIndex=m_lWord.indexOf(*sWord);

    updateText(*sWord);

    for(int i=0; i<sWord->length();i++)
    {
        updateCharCounter(ACTION_INS);
    }
}

void TextProcessor::sendSms()
{
    if(m_recipient->sendSMS(m_smsEdit->toPlainText()))
    {
        if(!m_bQwertyEnable)
            m_smsEdit->disconnect();

        m_nState = STATE_SEND;
        m_smsEdit->clear();
        m_sCompleteValue.clear();
        m_phoneNumber->clear();
        m_lWord.clear();
        m_nWordIndex=0;

        //update char and sms count
        updateCharCounter(ACTION_SEND);
        //

        if(!m_bQwertyEnable)
        {
            connect(m_smsEdit,SIGNAL(textChanged()),this,SLOT(smsChanged()));
            connect(m_smsEdit,SIGNAL(selectionChanged()),this,SLOT(cursorPosChanged()));
        }

    }
}



TextProcessor::~TextProcessor()
{
    delete ui;
}

/** Update Font size **/
void TextProcessor::setSmsFontSize(int nSize)
{
    QFont font = m_smsEdit->font();
    font.setPointSize(nSize);
    m_smsEdit->setFont(font);
}

void TextProcessor::settingsChanged(int nStyle)
{
    switch (nStyle)
    {
    case SettingsMenu::AutoCapitalize:
        m_nInputSyle = SettingsMenu::AutoCapitalize;
        ui->lbl_format->setText(Core::LBL_AUTOCAPITALIZATION);
        break;
    case SettingsMenu::UpperCase:
        m_nInputSyle = SettingsMenu::UpperCase;
        ui->lbl_format->setText(Core::LBL_UPPERCASE);
        break;
    case SettingsMenu::LowerCase:
        m_nInputSyle = SettingsMenu::LowerCase;
        ui->lbl_format->setText(Core::LBL_LOWERCASE);
        break;
    case SettingsMenu::DigitOnly:
        m_nInputSyle = SettingsMenu::DigitOnly;
        ui->lbl_format->setText(Core::LBL_DIGIT);
        break;
    case SettingsMenu::FastWriteOn:
        m_bFastInput = true;
        //Restore FastWrite state
        processAlfanumeric("0");
        processAlfanumeric(Keyboard::DELETE);
        //

        ui->lbl_fastWrite->setText(Core::LBL_FASTWRITE_ON);
        break;
    case SettingsMenu::FastWriteOff:
        m_bFastInput = false;
        ui->lbl_fastWrite->setText(Core::LBL_FASTWRITE_OFF);
        break;
    case SettingsMenu::QwertyOn:
        m_bQwertyEnable = true;
        disconnect(m_smsEdit,SIGNAL(textChanged()),this,SLOT(smsChanged()));
        disconnect(m_smsEdit,SIGNAL(selectionChanged()),this,SLOT(cursorPosChanged()));
        ui->lbl_fastWrite->setText(Core::LBL_QWERTY);
        m_cursorSmsText.clearSelection();
        m_smsEdit->setTextCursor(m_cursorSmsText);
        break;
    case SettingsMenu::AlfanumOn:
        m_bQwertyEnable = false;
        if(m_bFastInput)
            ui->lbl_fastWrite->setText(Core::LBL_FASTWRITE_ON);
        else
            ui->lbl_fastWrite->setText(Core::LBL_FASTWRITE_OFF);
        connect(m_smsEdit,SIGNAL(textChanged()),this,SLOT(smsChanged()));
        connect(m_smsEdit,SIGNAL(selectionChanged()),this,SLOT(cursorPosChanged()));
        cursorPosChanged();
        break;
    }
}



void TextProcessor::processQwerty(QString sInput)
{
    if(sInput.compare(Keyboard::DELETE) == 0)
    {
        m_smsEdit->textCursor().deletePreviousChar();
        updateCharCounter(ACTION_DEL);
    }
    else
    {
        m_smsEdit->textCursor().insertText(capitalizeIfNeeded(sInput));
        updateCharCounter(ACTION_INS);
    }
}

void TextProcessor::processAlfanumeric(QString sInput)
{
    if(sInput.compare(Keyboard::NEXTWORD) == 0)
        nextWord();
    else if(sInput.compare(Keyboard::PREVWORD) == 0)
        prevWord();
    else if(sInput.compare(Keyboard::DELETE) == 0)
        deleteChar();
    else if(sInput.compare(Keyboard::MOVERIGHT) == 0)
        moveCursorBt(true);
    else if(sInput.compare(Keyboard::MOVELEFT) == 0)
        moveCursorBt(false);
    else
        processDigit(sInput);
}

void TextProcessor::nextWord()
{
    if(m_lWord.count() > 0)
    {
        m_nWordIndex++;

        if(m_nWordIndex >= m_lWord.count())
            m_nWordIndex = 0;

        updateText(m_lWord.at(m_nWordIndex));
    }
}

void TextProcessor::prevWord()
{
    if(m_lWord.count() > 0)
    {
        m_nWordIndex--;

        if(m_nWordIndex < 0)
            m_nWordIndex = m_lWord.count() - 1;

        updateText(m_lWord.at(m_nWordIndex));
    }
}

void TextProcessor::deleteChar()
{
    m_nState = STATE_DEL;
    updateCharCounter(ACTION_DEL);

    m_smsEdit->disconnect();
    m_cursorSmsText.clearSelection();
    m_cursorSmsText.deletePreviousChar();

    m_sCompleteValue.clear();

    cursorPosChanged();

    connect(m_smsEdit,SIGNAL(textChanged()),this,SLOT(smsChanged()));
    connect(m_smsEdit,SIGNAL(selectionChanged()),this,SLOT(cursorPosChanged()));
}

/** Update sms and character counter */
void TextProcessor::updateCharCounter(int action)
{
    if(action==ACTION_SEND)
    {
        m_nCharCount=160;
        m_nSMSCount=1;
    }
    else if(action==ACTION_INS)
    {
        m_nCharCount--;

        if(m_nCharCount == -1)
        {
            m_nSMSCount++;

            if(m_nSMSCount==2)
            {
                m_nCharCount=145;
            }
            else if(m_nSMSCount >= 3)
            {
                m_nCharCount=152;
            }
        }
    }
    else if(m_nCharCount > -1)
    {
        m_nCharCount++;

        if(m_nCharCount == 161)
        {
            m_nCharCount = 160;
        }
        else if(m_nCharCount == 153 && m_nSMSCount >= 3)
        {
            m_nCharCount = 0;
            m_nSMSCount--;
        }
        else if(m_nCharCount == 146 && m_nSMSCount == 2)
        {
            m_nCharCount = 0;
            m_nSMSCount--;
        }
    }

    ui->lblCharCount->setText(QString::number(m_nCharCount,10));
    ui->lblMessageCount->setText(QString::number(m_nSMSCount,10));
}

/** Add sWord in cursor current position, after deselect all **/
void TextProcessor::addText(QString sWord)
{
    disconnect(m_smsEdit,SIGNAL(selectionChanged()),0,0);

    qDebug()<<m_cursorSmsText.position();
    m_cursorSmsText.clearSelection();
    m_cursorSmsText.insertText(capitalizeIfNeeded(sWord));

    connect(m_smsEdit,SIGNAL(selectionChanged()),this,SLOT(cursorPosChanged()));
}

/** Delete cursors selection then insert sWord **/
void TextProcessor::updateText(QString sWord)
{
    disconnect(m_smsEdit,SIGNAL(selectionChanged()),0,0);

    m_cursorSmsText.insertText(capitalizeIfNeeded(sWord));
    connect(m_smsEdit,SIGNAL(selectionChanged()),this,SLOT(cursorPosChanged()));

    qDebug() << "Write sms:" << m_smsEdit->toPlainText();
}

QString TextProcessor::capitalizeIfNeeded(QString sWord)
{
    if(m_nInputSyle == SettingsMenu::AutoCapitalize)
    {
        QString smsText = m_smsEdit->toPlainText();
        int nIdx = m_cursorSmsText.position();

        smsText.length() <= nIdx ? nIdx = smsText.length() -1 :  nIdx = nIdx;

        bool bToUpper = true;
        bool bSpace = false;

        for(int i=nIdx; i>=0; i--)
        {
            if(smsText.at(i) == ' ')
            {
                bSpace = true;
                continue;
            }

            if(bSpace || !m_bFastInput || m_bQwertyEnable)
            {
                if(!m_sCapitalization.contains(smsText.at(i)))
                {
                    bToUpper = false;
                    break;
                }
                else
                {
                    break;
                }
            }
            else if(m_dict->isSymbol(smsText.at(i)))
            {
                bToUpper = false;
                break;
            }
        }

        if(bToUpper || (m_bRememberUppercase && !m_bQwertyEnable))
        {
            sWord[0]=sWord[0].toUpper();

            if(!m_bFastInput)
                m_bRememberUppercase = true;
        }
    }
    else if(m_nInputSyle == SettingsMenu::UpperCase)
        sWord = sWord.toUpper();
    else if(m_nInputSyle == SettingsMenu::LowerCase)
        sWord = sWord.toLower();

    return sWord;
}

void TextProcessor::processDigit(QString sInput)
{
    //FastWrite input style selected
    if(m_bFastInput && m_nInputSyle != SettingsMenu::DigitOnly)
    {
        if(sInput.compare("0") == 0)
        {
            m_nState = STATE_SPACE;
            m_nMsgLength++;
            m_sCompleteValue = sInput;
            m_nWordIndex=0;
            m_lWord.clear();
            m_lWord = m_dict->getWordList(&m_sCompleteValue);

            m_nEndWord ++;
            m_nBeginWord = m_nEndWord-1;
            QString sChoosenWord = m_lWord.at(0);
            addText(sChoosenWord);
            updateCharCounter(ACTION_INS);
            m_sCompleteValue.clear();

        }
        else
        {
            if(m_nState == STATE_SPACE)
            {
                m_cursorSmsText.clearSelection();
                m_nBeginWord = m_nEndWord;
            }

            m_sCompleteValue.append(sInput);

            if(sInput.compare("1") == 0)
                m_nState = STATE_SYM;

            if(m_dict->existsKey(&m_sCompleteValue))
            {
                m_nState = STATE_WORD;
                m_nMsgLength++;
                m_nWordIndex=0;
                m_lWord = m_dict->getWordList(&m_sCompleteValue);

                m_nEndWord++;
                QString sChoosenWord = m_lWord.at(0);
                updateText(sChoosenWord);
                updateCharCounter(ACTION_INS);
            }
            else
            {
                if(sInput.compare("1") == 0)
                {
                    m_nState = STATE_SYM;
                    m_nMsgLength++;
                    m_sCompleteValue.clear();
                    m_sCompleteValue = sInput;

                    m_nWordIndex=0;
                    m_lWord = m_dict->getWordList(&m_sCompleteValue);

                    m_nBeginWord = m_nEndWord;
                    m_nEndWord ++;
                    QString sChoosenWord = m_lWord.at(0);
                    addText(sChoosenWord);
                    updateCharCounter(ACTION_INS);
                }
                else
                {
                    if(m_sCompleteValue.at(m_sCompleteValue.length()-2) == QChar('1'))//if(m_nState == STATE_SYM)
                    {
                        m_nState = STATE_WORD;
                        m_nMsgLength++;
                        m_sCompleteValue.clear();
                        m_sCompleteValue = sInput;

                        m_nWordIndex=0;
                        m_lWord = m_dict->getWordList(&m_sCompleteValue);

                        m_nBeginWord = m_nEndWord;
                        m_nEndWord ++;
                        QString sChoosenWord = m_lWord.at(0);
                        addText(sChoosenWord);
                        updateCharCounter(ACTION_INS);
                    }

                    else
                    {
                        //m_nState = STATE_UNKNOWN;
                        m_sCompleteValue = m_sCompleteValue.left(m_sCompleteValue.length()-1);
                        //TODO: no knowing words
                    }
                }
            }
        }
    }//End FastWrite routine
    else if(m_nInputSyle == SettingsMenu::DigitOnly)//Only-Digit input selected
    {
        m_nState = STATE_SPACE;
        m_nMsgLength++;
        m_sCompleteValue = sInput;
        m_nWordIndex=0;
        m_lWord.clear();
        m_lWord = m_dict->getWordList(&m_sCompleteValue);

        m_nEndWord ++;
        m_nBeginWord = m_nEndWord-1;
        addText(sInput);
        updateCharCounter(ACTION_INS);
        m_sCompleteValue.clear();
    }
    else if(!m_bFastInput) //standard-input selected
    {

        if(!m_bTimerNewChar && sInput.compare(m_sLastPress) == 0)
        {
            nextWord();
            m_timerNewChar->start();
        }
        else
        {
            m_bRememberUppercase = false;
            m_nState = STATE_SYM;
            m_nMsgLength++;
            m_sCompleteValue.clear();
            m_sCompleteValue = sInput;
            m_nWordIndex=0;
            m_lWord = m_dict->getWordList(&m_sCompleteValue,true);

            m_nBeginWord = m_nEndWord;
            m_nEndWord ++;
            QString sChoosenWord = m_lWord.at(0);
            addText(sChoosenWord);
            updateCharCounter(ACTION_INS);

            //reset timer
            m_bTimerNewChar = false;
            m_timerNewChar->start();
        }
    }
    m_sLastPress = sInput;

}

void TextProcessor::moveCursorBt(bool bMoveRight)
{
    m_nState = STATE_MOVECUR_BUTTON;

    if(m_cursorSmsText.hasSelection() ||
       (bMoveRight && m_cursorSmsText.atEnd()) ||
       (!bMoveRight && m_cursorSmsText.atStart()))
    {
        m_smsEdit->disconnect();

        if(bMoveRight)
            m_nBeginWord = m_nEndWord;
        else
            m_nEndWord = m_nBeginWord;

        m_lWord.clear();
        m_sCompleteValue.clear();
        m_nWordIndex=0;

        m_cursorSmsText.setPosition(m_nEndWord, QTextCursor::MoveAnchor);
        m_smsEdit->setTextCursor(m_cursorSmsText);
        connect(m_smsEdit,SIGNAL(textChanged()),this,SLOT(smsChanged()));
        connect(m_smsEdit,SIGNAL(selectionChanged()),this,SLOT(cursorPosChanged()));
    }
    else
    {
        m_smsEdit->disconnect();

        m_lWord.clear();
        m_sCompleteValue.clear();
        m_nWordIndex=0;

        if(bMoveRight)
        {
            m_cursorSmsText.movePosition(QTextCursor::NextCharacter);
            m_nBeginWord = m_cursorSmsText.position();
            m_nEndWord = m_nBeginWord;
        }
        else
        {
            m_cursorSmsText.movePosition(QTextCursor::PreviousCharacter);
            m_nBeginWord = m_cursorSmsText.position();
            m_nEndWord = m_nBeginWord;
        }

        m_cursorSmsText.setPosition(m_cursorSmsText.position()-1,QTextCursor::MoveAnchor);
        m_cursorSmsText.setPosition(m_cursorSmsText.position()+2,QTextCursor::KeepAnchor);
        QString sSel = m_cursorSmsText.selectedText();

        m_cursorSmsText.setPosition(m_cursorSmsText.position()-1,QTextCursor::MoveAnchor);
        m_smsEdit->setTextCursor(m_cursorSmsText);

        if((sSel.isEmpty()) || (!sSel.contains(" ") && !m_dict->isSymbol(sSel.at(0)) && !m_dict->isSymbol(sSel.at(1))))
        {
            cursorPosChanged();
        }

        connect(m_smsEdit,SIGNAL(textChanged()),this,SLOT(smsChanged()));
        connect(m_smsEdit,SIGNAL(selectionChanged()),this,SLOT(cursorPosChanged()));
    }
}

void TextProcessor::cursorPosChanged()
{
    if(m_nState != STATE_AFTERCURMOVED)
    {
        m_smsEdit->disconnect();

        qDebug() << "cursor moved";
        m_cursorSmsText = m_smsEdit->textCursor(); // retrieve the visible cursor
        m_cursorSmsText.select(QTextCursor::WordUnderCursor);
        m_nBeginWord = m_cursorSmsText.anchor();
        m_nEndWord = m_cursorSmsText.position();
        m_smsEdit->setTextCursor(m_cursorSmsText);

        if(m_cursorSmsText.hasSelection())
        {
            qDebug() << "Selected text from Cursor moved" << m_cursorSmsText.selectedText().toAscii();

            QString *sWord = new QString(m_cursorSmsText.selectedText());
            m_sCompleteValue = *m_dict->calculateKey(sWord);
            m_lWord = m_dict->getWordList(&m_sCompleteValue);
            //m_nWordIndex = 0;
            m_nWordIndex = m_lWord.indexOf(*sWord);
            if(m_nWordIndex < 0)
                m_nWordIndex = 0;
            m_nState = STATE_AFTERCURMOVED;

            delete sWord;
        }
        else
        {
            qDebug() << "Nothing selected";
            m_sCompleteValue.clear();
            m_nWordIndex = 0;
            m_lWord.clear();
        }

        connect(m_smsEdit,SIGNAL(textChanged()),this,SLOT(smsChanged()));
        connect(m_smsEdit,SIGNAL(selectionChanged()),this,SLOT(cursorPosChanged()));
    }
    else
    {
        m_nState = STATE_CURMOVED;
    }
}

void TextProcessor::smsChanged()
{
    m_smsEdit->disconnect();

    m_cursorSmsText.setPosition(m_nBeginWord, QTextCursor::MoveAnchor);
    m_cursorSmsText.setPosition(m_nEndWord, QTextCursor::KeepAnchor);

    m_smsEdit->setTextCursor(m_cursorSmsText);

    connect(m_smsEdit,SIGNAL(textChanged()),this,SLOT(smsChanged()));
    connect(m_smsEdit,SIGNAL(selectionChanged()),this,SLOT(cursorPosChanged()));
}

void TextProcessor::alarmNewChar()
{
    m_bTimerNewChar = true;
}
