/****************************************************************************
**
** Copyright (C) 2010  Aimone Davide realsportcars@hotmail.com
**
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see <http://www.gnu.org/licenses/>.
**
****************************************************************************/

#include "fastsms.h"
#include "button.h"
#include "dictionary.h"
#include "busManagerServ.h"
#include <QMessage>
#include <QMessageService>
#include <QMessageAddress>
#include <QMenuBar>
#include <QMainWindow>

#define STATE_SPACE 0
#define STATE_SYM 1
#define STATE_WORD 2
#define STATE_UNKNOWN 3
#define STATE_DEL 3
#define STATE_MOVECUR_BUTTON 3
#define STATE_CURMOVED 5
#define STATE_AFTERCURMOVED 6
#define STATE_SEND 8

#define SETTING_LANGUAGE "language"
#define SETTING_ORIENTATION "orientation"

#define ORIENTATION_AUTO "auto"
#define ORIENTATION_LANDSCAPE "landscape"
#define ORIENTATION_PORTRAIT "portrait"

#define BUTTON_CURNEXT "bt_cur_next"
#define BUTTON_CURPREV "bt_cur_prev"

#define DATA_PATH "/data/"

//using namespace QtMobility;
QTM_USE_NAMESPACE

fastsms::fastsms(QWidget *parent) :
    QWidget(parent)//,    ui(new Ui::fastsms)
{
    new busManagerServ(this);
    registerService();


    //Design UI. TODO:use ui file
    //ui->setupUi(this);
    Button *to = createButton(tr("To:"),SLOT(openPhoneBook()));
    Button *send = createButton(tr("Send"),SLOT(sendSMS()));
    Button *btMoveCurNext = createButton(tr("->"),SLOT(moveCursorBt()));
    btMoveCurNext->setObjectName(BUTTON_CURNEXT);
    Button *btMoveCurPrev = createButton(tr("<-"),SLOT(moveCursorBt()));
    btMoveCurPrev->setObjectName(BUTTON_CURPREV);
    Button *delChar = createButton(tr("C"),SLOT(deleteChar()));
    Button *one = createButton(tr("1 .,!"),SLOT(digitClicked()));
    Button *two = createButton(tr("2 abc"),SLOT(digitClicked()));
    Button *three = createButton(tr("3 def"),SLOT(digitClicked()));
    Button *four = createButton(tr("4 ghi"),SLOT(digitClicked()));
    Button *five = createButton(tr("5 jkl"),SLOT(digitClicked()));
    Button *six = createButton(tr("6 mno"),SLOT(digitClicked()));
    Button *seven = createButton(tr("7 pqrs"),SLOT(digitClicked()));
    Button *eight = createButton(tr("8 tuv"),SLOT(digitClicked()));
    Button *nine = createButton(tr("9 wxyz"),SLOT(digitClicked()));
    Button *zero = createButton(tr("0"),SLOT(digitClicked()));
    Button *number = createButton(tr("Next"),SLOT(nextWord()));
    Button *star = createButton(tr("Prev"),SLOT(prevWord()));

    m_phoneNumber = new QLineEdit();
    m_smsEdit = new QTextEdit();

    QGridLayout *queryLayout = new QGridLayout();
    queryLayout->addWidget(to,0,0);
    queryLayout->addWidget(m_phoneNumber,0,1);
    queryLayout->addWidget(send,0,2);
    queryLayout->addWidget(m_smsEdit,2,0,1,3);

    QGridLayout *keyboard = new QGridLayout();
    keyboard->addWidget(btMoveCurPrev,0,0);
    keyboard->addWidget(delChar,0,1);
    keyboard->addWidget(btMoveCurNext,0,2);
    keyboard->addWidget(one,1,0);
    keyboard->addWidget(two,1,1);
    keyboard->addWidget(three,1,2);
    keyboard->addWidget(four,2,0);
    keyboard->addWidget(five,2,1);
    keyboard->addWidget(six,2,2);
    keyboard->addWidget(seven,3,0);
    keyboard->addWidget(eight,3,1);
    keyboard->addWidget(nine,3,2);
    keyboard->addWidget(number,4,0);
    keyboard->addWidget(zero,4,1);
    keyboard->addWidget(star,4,2);

    QVBoxLayout *mainLayout = new QVBoxLayout(this);

    mainLayout->addLayout(queryLayout);
    mainLayout->addLayout(keyboard);

    setLayout(mainLayout);    //End Design UI

    //load configuration
    m_dict = NULL;
    m_settings = new QSettings("it.rsc.fastsms","fastsms",this);
    loadDict();

    //create dictionary
//    QString sLang = m_settings->value(SETTING_LANGUAGE).toString();
//    qDebug() << "Setting language:" << sLang;
//    if(m_dict == NULL && !sLang.isEmpty())
//        m_dict = new dictionary(new QString(m_settings->value(SETTING_LANGUAGE).toString()),this);

    //init variables
    m_nWordIndex=0;
    m_bNeedSave = false;
    m_sCompleteValue = QString();
    m_cursorSmsText = m_smsEdit->textCursor(); // retrieve the visible cursor
    m_nMsgLength = 0;
    m_nBeginWord = 0;
    m_nEndWord = 0;

    //Menu
    QMenuBar *menu = new QMenuBar(this);
    QMenu *fileMenu = new QMenu("File",menu);

    QAction *aOption = new QAction("Options",fileMenu);
    connect(aOption,SIGNAL(triggered()),this, SLOT(openOptionDialog()));
    fileMenu->addAction(aOption);

    QAction *aNewWord = new QAction("New Word",fileMenu);
    connect(aNewWord,SIGNAL(triggered()),this, SLOT(insertNewWord()));
    fileMenu->addAction(aNewWord);

    QAction *aAbout = new QAction("About",fileMenu);
    fileMenu->addAction(aAbout);
    connect(aAbout,SIGNAL(triggered()),this, SLOT(openAboutDialog()));

    QAction *aQuit = new QAction("Quit",fileMenu);
    connect(aQuit,SIGNAL(triggered()),this, SLOT(close()));
    fileMenu->addAction(aQuit);

    menu->addMenu(fileMenu);

    //manage autorotate from start
    this->setFocus();

    //manage cursor
    connect(m_smsEdit,SIGNAL(textChanged()),this,SLOT(smsChanged()));
    connect(m_smsEdit,SIGNAL(selectionChanged()),this,SLOT(cursorPosChanged()));
}

/** check if configuration and dict file exists. If not prompt to option dialog, else load it**/
void fastsms::loadDict()
{
    bool bOpenOpt = true;

    if(m_settings->contains(SETTING_LANGUAGE))
    {
        QString sPath = QApplication::applicationDirPath() + DATA_PATH + m_settings->value(SETTING_LANGUAGE).toString();
        QFile *dictFile = new QFile(sPath);

        if(dictFile->exists())
            bOpenOpt = false;

        delete dictFile;

        #if defined(Q_WS_MAEMO_5)
            QString sCurrOrient = m_settings->value(SETTING_ORIENTATION).toString();
            if(sCurrOrient.compare(ORIENTATION_LANDSCAPE) == 0)
                setAttribute(Qt::WA_Maemo5LandscapeOrientation, true);
            else if(sCurrOrient.compare(ORIENTATION_PORTRAIT) == 0)
                setAttribute(Qt::WA_Maemo5PortraitOrientation, true);
            else
                setAttribute(Qt::WA_Maemo5AutoOrientation, true);
        #endif
    }

    if(bOpenOpt)
        openOptionDialog();
    else
        m_dict = new dictionary(new QString(m_settings->value(SETTING_LANGUAGE).toString()),this);
}

/** Open and populate option dialog **/
void fastsms::openOptionDialog()
{
    m_dialOptions = new QDialog(this);
    m_dialOptions->setAttribute(Qt::WA_DeleteOnClose);
    m_uiOptions.setupUi(m_dialOptions);

    connect(m_uiOptions.done,SIGNAL(clicked()),this,SLOT(closeOptions()));

    QDir dictDir(QApplication::applicationDirPath() + "/data");
    QStringList dictList = dictDir.entryList(QStringList("*.idc"));

    QString sCurrLang = m_settings->value(SETTING_LANGUAGE).toString();
    for(int i=0; i<dictList.size(); i++)
    {
        m_uiOptions.cbLanguage->addItem(dictList.at(i));

        if(dictList.at(i).compare(sCurrLang) == 0)
            m_uiOptions.cbLanguage->setCurrentIndex(i);
    }

    m_uiOptions.orientationAuto->setChecked(true);

    QString sCurrOrient = m_settings->value(SETTING_ORIENTATION).toString();
    if(sCurrOrient.compare(ORIENTATION_LANDSCAPE) == 0)
        m_uiOptions.orientationLandscape->setChecked(true);
    else if(sCurrOrient.compare(ORIENTATION_PORTRAIT) == 0)
        m_uiOptions.orientationPortrait->setChecked(true);

    m_dialOptions->show();
}

/** Open and populate option dialog **/
void fastsms::openAboutDialog()
{
    QDialog * dlAbout = new QDialog(this);
    dlAbout->setAttribute(Qt::WA_DeleteOnClose);

    dlAbout->setWindowTitle("About");
    QGridLayout *layout = new QGridLayout();
    QString sAbout = QString("Created by: Aimone Davide \n\n For requests,suggestions,bugs write to: fastsms@hotmail.it\n");
    QLabel *msg = new QLabel(sAbout);
    layout->addWidget(msg,0,0);
    dlAbout->setLayout(layout);

    dlAbout->show();
}

/** SLOT: close option dialog and save options in settings file **/
void fastsms::closeOptions()
{
    QString sOldLang = m_settings->value(SETTING_LANGUAGE).toString();

    QString sDictFile = m_uiOptions.cbLanguage->currentText();
    m_settings->setValue(SETTING_LANGUAGE,sDictFile);

    if(m_uiOptions.orientationAuto->isChecked())
    {
        m_settings->setValue(SETTING_ORIENTATION,ORIENTATION_AUTO);
        #if defined(Q_WS_MAEMO_5)
            setAttribute(Qt::WA_Maemo5AutoOrientation, true);
        #endif
    }
    else if(m_uiOptions.orientationLandscape->isChecked())
    {
        m_settings->setValue(SETTING_ORIENTATION,ORIENTATION_LANDSCAPE);
        #if defined(Q_WS_MAEMO_5)
            setAttribute(Qt::WA_Maemo5LandscapeOrientation, true);
        #endif
    }
    else if(m_uiOptions.orientationPortrait->isChecked())
    {
        m_settings->setValue(SETTING_ORIENTATION,ORIENTATION_PORTRAIT);
        #if defined(Q_WS_MAEMO_5)
            setAttribute(Qt::WA_Maemo5PortraitOrientation, true);
        #endif
    }

    if(sOldLang.compare(sDictFile) != 0)
    {
        qDebug() << "Language changed!";
        delete m_dict;
        m_dict = new dictionary(new QString(m_settings->value(SETTING_LANGUAGE).toString()),this);
    }

    qDebug() << "Settings:" << m_settings->value(SETTING_LANGUAGE).toString() << "," << m_settings->value(SETTING_ORIENTATION).toString();

    m_dialOptions->close();
}

void fastsms::moveCursorBt()
{
    bool bMoveNext = false;
    m_nState = STATE_MOVECUR_BUTTON;

    Button *clickedButton = qobject_cast<Button *>(sender());
    QString btPressed = clickedButton->objectName();
    if(btPressed.compare(BUTTON_CURNEXT) == 0)
        bMoveNext = true;

    if(m_cursorSmsText.hasSelection() ||
        (bMoveNext && m_cursorSmsText.atEnd()) ||
        (!bMoveNext && m_cursorSmsText.atStart()))
    {
        m_smsEdit->disconnect();

        if(bMoveNext)
            m_nBeginWord = m_nEndWord;
        else
            m_nEndWord = m_nBeginWord;

        lWord.clear();
        m_sCompleteValue.clear();
        m_nWordIndex=0;

        m_cursorSmsText.setPosition(m_nEndWord, QTextCursor::MoveAnchor);
        m_smsEdit->setTextCursor(m_cursorSmsText);
        connect(m_smsEdit,SIGNAL(textChanged()),this,SLOT(smsChanged()));
        connect(m_smsEdit,SIGNAL(selectionChanged()),this,SLOT(cursorPosChanged()));
    }
    else
    {
        m_smsEdit->disconnect();

        lWord.clear();
        m_sCompleteValue.clear();
        m_nWordIndex=0;

        if(bMoveNext)
        {
            m_cursorSmsText.movePosition(QTextCursor::NextCharacter);
            m_nBeginWord = m_cursorSmsText.position();
            m_nEndWord = m_nBeginWord;
        }
        else
        {
            m_cursorSmsText.movePosition(QTextCursor::PreviousCharacter);
            m_nBeginWord = m_cursorSmsText.position();
            m_nEndWord = m_nBeginWord;
        }

        m_cursorSmsText.setPosition(m_cursorSmsText.position()-1,QTextCursor::MoveAnchor);
        m_cursorSmsText.setPosition(m_cursorSmsText.position()+2,QTextCursor::KeepAnchor);
        QString sSel = m_cursorSmsText.selectedText();

        m_cursorSmsText.setPosition(m_cursorSmsText.position()-1,QTextCursor::MoveAnchor);
        m_smsEdit->setTextCursor(m_cursorSmsText);

        if((sSel.isEmpty()) || (!sSel.contains(" ") && !m_dict->isSymbol(sSel.at(0)) && !m_dict->isSymbol(sSel.at(1))))
        {
            cursorPosChanged();
        }

        connect(m_smsEdit,SIGNAL(textChanged()),this,SLOT(smsChanged()));
        connect(m_smsEdit,SIGNAL(selectionChanged()),this,SLOT(cursorPosChanged()));
    }
}

void fastsms::cursorPosChanged()
{
    if(m_nState != STATE_AFTERCURMOVED)
    {
        m_smsEdit->disconnect();

        qDebug() << "cursor moved";
        m_cursorSmsText = m_smsEdit->textCursor(); // retrieve the visible cursor
        m_cursorSmsText.select(QTextCursor::WordUnderCursor);
        m_nBeginWord = m_cursorSmsText.anchor();
        m_nEndWord = m_cursorSmsText.position();
        m_smsEdit->setTextCursor(m_cursorSmsText);

        if(m_cursorSmsText.hasSelection())
        {
            qDebug() << "Selected text from Cursor moved" << m_cursorSmsText.selectedText().toAscii();

            QString *sWord = new QString(m_cursorSmsText.selectedText());
            m_sCompleteValue = *m_dict->calculateKey(sWord);            
            lWord = m_dict->getWordList(&m_sCompleteValue);
            //m_nWordIndex = 0;
            m_nWordIndex = lWord.indexOf(*sWord);
            if(m_nWordIndex < 0)
                m_nWordIndex = 0;
            m_nState = STATE_AFTERCURMOVED;

            delete sWord;
        }
        else
        {
            qDebug() << "Nothing selected";
            m_sCompleteValue.clear();
            m_nWordIndex = 0;
            lWord.clear();
        }

        connect(m_smsEdit,SIGNAL(textChanged()),this,SLOT(smsChanged()));
        connect(m_smsEdit,SIGNAL(selectionChanged()),this,SLOT(cursorPosChanged()));
    }
    else
    {
        m_nState = STATE_CURMOVED;
    }
}

void fastsms::smsChanged()
{
    m_smsEdit->disconnect();

    m_cursorSmsText.setPosition(m_nBeginWord, QTextCursor::MoveAnchor);
    m_cursorSmsText.setPosition(m_nEndWord, QTextCursor::KeepAnchor);

    m_smsEdit->setTextCursor(m_cursorSmsText);

    connect(m_smsEdit,SIGNAL(textChanged()),this,SLOT(smsChanged()));
    connect(m_smsEdit,SIGNAL(selectionChanged()),this,SLOT(cursorPosChanged()));
}

/**Create button */
Button *fastsms::createButton(const QString &text, const char *member)
{
    Button *button = new Button(text,this);
    connect(button, SIGNAL(clicked()), this, member);
    button->setFocusPolicy(Qt::NoFocus);
    return button;
}


void fastsms::digitClicked()
{
    Button *clickedButton = qobject_cast<Button *>(sender());
    QString lastPress = clickedButton->text().at(0);

    if(m_phoneNumber->hasFocus())
    {
        m_phoneNumber->insert(lastPress.toAscii());
    }
    else
    {
        m_smsEdit->setFocus();

        if(lastPress.compare("0") == 0)
        {
            m_nState = STATE_SPACE;
            m_nMsgLength++;
            m_sCompleteValue = lastPress;
            m_nWordIndex=0;
            lWord.clear();
            lWord = m_dict->getWordList(&m_sCompleteValue);

            m_nEndWord ++;
            m_nBeginWord = m_nEndWord-1;
            QString sChoosenWord = lWord.at(0);
            addText(sChoosenWord);
            m_sCompleteValue.clear();

        }
        else
        {
            if(m_nState == STATE_SPACE)
            {
                m_cursorSmsText.clearSelection();
                m_nBeginWord = m_nEndWord;
            }

            m_sCompleteValue.append(lastPress);

            if(lastPress.compare("1") == 0)
                 m_nState = STATE_SYM;

            if(m_dict->existsKey(&m_sCompleteValue))
            {
                m_nState = STATE_WORD;
                m_nMsgLength++;
                m_nWordIndex=0;
                lWord = m_dict->getWordList(&m_sCompleteValue);

                m_nEndWord++;
                QString sChoosenWord = lWord.at(0);
                updateText(sChoosenWord);
            }
            else
            {
                if(lastPress.compare("1") == 0)
                {
                    m_nState = STATE_SYM;
                    m_nMsgLength++;
                    m_sCompleteValue.clear();
                    m_sCompleteValue = lastPress;

                    m_nWordIndex=0;
                    lWord = m_dict->getWordList(&m_sCompleteValue);

                    m_nBeginWord = m_nEndWord;
                    m_nEndWord ++;
                    QString sChoosenWord = lWord.at(0);
                    addText(sChoosenWord);
                }
                else
                {
                    if(m_sCompleteValue.at(m_sCompleteValue.length()-2) == QChar('1'))//if(m_nState == STATE_SYM)
                    {
                        m_nState = STATE_WORD;
                        m_nMsgLength++;
                        m_sCompleteValue.clear();
                        m_sCompleteValue = lastPress;

                        m_nWordIndex=0;
                        lWord = m_dict->getWordList(&m_sCompleteValue);

                        m_nBeginWord = m_nEndWord;
                        m_nEndWord ++;
                        QString sChoosenWord = lWord.at(0);
                        addText(sChoosenWord);

                    }
                    else
                    {
                        //m_nState = STATE_UNKNOWN;
                        m_sCompleteValue = m_sCompleteValue.left(m_sCompleteValue.length()-1);
                        //TODO: no knowing words
                    }
                }
            }
        }
    }
}

void fastsms::nextWord()
{
    if(lWord.count() > 0)
    {
        m_nWordIndex++;

        if(m_nWordIndex >= lWord.count())
            m_nWordIndex = 0;

        updateText(lWord.at(m_nWordIndex));
    }
}

void fastsms::prevWord()
{
    if(lWord.count() > 0)
    {
        m_nWordIndex--;

        if(m_nWordIndex < 0)
            m_nWordIndex = lWord.count() - 1;

        updateText(lWord.at(m_nWordIndex));
    }
}

void fastsms::insertNewWord()
{
    m_dialNewWord = new QDialog(this);
    m_dialNewWord->setAttribute(Qt::WA_DeleteOnClose);
    inputNewWord = new QLineEdit(m_dialNewWord);
    Button *ok = createButton(tr("Ok"),SLOT(newWordInserted()));
    QGridLayout *insWordLayout = new QGridLayout();
    insWordLayout->addWidget(inputNewWord,0,0);
    insWordLayout->addWidget(ok,1,0);
    inputNewWord->setFocus();

    m_dialNewWord->setWindowTitle("Type new word");
    m_dialNewWord->setLayout(insWordLayout);
    m_dialNewWord->show();
}

void fastsms::newWordInserted()
{
    QString *newWord = new QString(inputNewWord->text().toLower());
    if(!newWord->isEmpty())
    {
        qDebug() << "New word is:" << newWord->toAscii();
        m_bNeedSave = m_dict->addWord(newWord);
        m_nState = STATE_WORD;

        m_nBeginWord = m_cursorSmsText.anchor();
        m_nEndWord=m_nBeginWord + newWord->length();

        m_sCompleteValue.clear();
        m_sCompleteValue.append(m_dict->calculateKey(newWord));
        lWord = m_dict->getWordList(&m_sCompleteValue);
        m_nWordIndex=lWord.indexOf(*newWord);

        updateText(*newWord);
    }

     m_dialNewWord->close();
    //m_bKnowWord=true;

    this->setFocus();
}

/** Delete cursors selection then insert sWord **/
void fastsms::updateText(QString sWord)
{
    disconnect(m_smsEdit,SIGNAL(selectionChanged()),0,0);

    m_cursorSmsText.insertText(sWord);

    connect(m_smsEdit,SIGNAL(selectionChanged()),this,SLOT(cursorPosChanged()));

    qDebug() << "Write sms:" << m_smsEdit->toPlainText();
}

/** Add sWord in cursor current position, after deselect all **/
void fastsms::addText(QString sWord)
{
    disconnect(m_smsEdit,SIGNAL(selectionChanged()),0,0);

    m_cursorSmsText.clearSelection();
    m_cursorSmsText.insertText(sWord);

    connect(m_smsEdit,SIGNAL(selectionChanged()),this,SLOT(cursorPosChanged()));
}

void fastsms::deleteChar()
{
    if(m_phoneNumber->hasFocus())
    {
        m_phoneNumber->backspace();
    }
    else
    {
            m_nState = STATE_DEL;

            m_smsEdit->disconnect();
            m_cursorSmsText.clearSelection();
            m_cursorSmsText.deletePreviousChar();

            m_sCompleteValue.clear();

            cursorPosChanged();

            connect(m_smsEdit,SIGNAL(textChanged()),this,SLOT(smsChanged()));
            connect(m_smsEdit,SIGNAL(selectionChanged()),this,SLOT(cursorPosChanged()));
    }
}

void fastsms::openPhoneBook()
{
    QDBusMessage msg = QDBusMessage::createMethodCall(
        "com.crochik.MyContacts", // --dest
        "/", // destination object path
        "com.crochik.MyContacts",
        "show"
    );
    msg << "Opening Mycontacts";
    msg = QDBusConnection::sessionBus().call(msg);
    qDebug() << msg;
    bool ok = msg.type() != QDBusMessage::ErrorMessage;

    if(!ok)
    {
        QDialog *dlCont = new QDialog(this);
        dlCont->setAttribute(Qt::WA_DeleteOnClose);

        dlCont->setWindowTitle("Can't open Mycontacts");
        QGridLayout *layout = new QGridLayout();
        QString sInfo = QString("\nYou need to install MyContacts\nto use this feature!!!\n");
        QLabel *msg = new QLabel(sInfo);
        layout->addWidget(msg,0,0);
        dlCont->setLayout(layout);
        dlCont->show();
    }
}

void fastsms::sendSMS()
{
    QMessageService service;
    QMessageManager manager;

    //Find right account for SMS: default one doesnt work well
    QMessageAccountIdList lstMsgAccountId = manager.queryAccounts();

    if(!m_smsEdit->toPlainText().isEmpty())
    {
        if(!lstMsgAccountId.isEmpty())
        {
            qDebug() << "SMS text" << m_smsEdit->toPlainText();
            for(int i = 0; i < lstMsgAccountId.count(); ++i)
            {
                QMessageAccount account(lstMsgAccountId[i]);
                bool isSmsAccount = (account.messageTypes() & QMessage::Sms) > 0;

                if(isSmsAccount)
                {
                    QMessage myMessage;
                    myMessage.setType(QMessage::Sms);
                    myMessage.setParentAccountId(lstMsgAccountId[i]);
                    myMessage.setTo(QMessageAddress(QMessageAddress::Phone, m_phoneNumber->text()));
                    myMessage.setBody(m_smsEdit->toPlainText());

                    qDebug() << "state:" << service.send(myMessage);
                    break;
                }
            }
        }
    }

    m_smsEdit->disconnect();

    m_nState = STATE_SEND;
    //m_cursorSmsText.select(QTextCursor::Document);
    m_smsEdit->clear();
    m_sCompleteValue.clear();
    m_phoneNumber->clear();
    lWord.clear();
    m_nWordIndex=0;

    connect(m_smsEdit,SIGNAL(textChanged()),this,SLOT(smsChanged()));
    connect(m_smsEdit,SIGNAL(selectionChanged()),this,SLOT(cursorPosChanged()));
}

void fastsms::saveDictionaryOnExit()
{
    if(m_bNeedSave)
    {
        qDebug() << "Saving dictionary";
        m_dict->saveDictionaryToFile();
    }
}

bool fastsms::addRecipient(const QString &phone, const QString &name)
{
    m_phoneNumber->setText(phone);
    return true;
}

bool fastsms::registerService()
{
    QDBusConnection connection = QDBusConnection::sessionBus();
    if ( !connection.registerService("it.rsc.fastsms") ) {
        qDebug() << "error registering service.";
        return false;
    }
    if ( !connection.registerObject("/", this) ) {
        qDebug() << "error registering object";
        return false;
    }

    return true;
}

void fastsms::doNothing()
{

}

fastsms::~fastsms()
{
    //delete ui;
    //delete m_sCompleteValue;

}

void fastsms::changeEvent(QEvent *e)
{
    QWidget::changeEvent(e);
    switch (e->type())
    {
    case QEvent::LanguageChange:
        //ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

