/*
	This file is part of Faster Application Manager.

	Faster Application Manager is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Faster Application Manager is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Faster Application Manager.  If not, see <http://www.gnu.org/licenses/>.

	(C) Heikki Holstila 2010
*/

#include <QtGui>
#include "package.h"
#include "repository.h"
#include "aaptinterface.h"
#include "apt-src/debversion.h"
#include "blacklistselect.h"

Package::Package(QByteArray name_, AAptInterface *apt_):
		iAptInterface(apt_), iName(name_), iIsInstalled(false), iMarkedForOperation(false),
		iPkgStatus(PkgStatUnknown), iSize(0), iInstalledSize(0), iMarkedOperation(PkgOpNone),
		iBlacklist(BlacklistSelect::BlacklistNone), iIcon(0), iPinned(false)
{
}

Package::~Package()
{
	if( iIcon != 0 )
		delete iIcon;
}

QString Package::displayName()
{
	QString pkgname = name();
	if( !iMaemoDisplayName.isEmpty() )
		pkgname = maemoDisplayName();
	QString n( pkgname.at(0) );
	n = n.toUpper();
	pkgname.replace(0,1,n);
	return pkgname;
}

QString Package::fileName()
{
	if( iFullFileNames.count()>0 ) {
		return iFullFileNames.at(0).mid( iFullFileNames.at(0).lastIndexOf('/')+1 );
	} else {
		qDebug() << "Warning: package has no file name information";
		return "unknown_filename";
	}
}

void Package::setMarkedForOperation(operation op_)
{
	if( iMarkedOperation != op_ ) {
		if( op_==PkgOpNone ) {
			if( iAptInterface )
				iAptInterface->setNumSelectedPackages( iAptInterface->numSelectedPackages()-1 );
		} else if( iMarkedOperation==PkgOpNone ) {
			if( iAptInterface )
				iAptInterface->setNumSelectedPackages( iAptInterface->numSelectedPackages()+1 );
		}
	}

	iMarkedOperation = op_;

	if( op_ == PkgOpNone )
		iMarkedForOperation = false;
	else
		iMarkedForOperation = true;
}

void Package::convertIcon()
{
	if( iIconData.length() > 0 && iIcon == 0 ) {
		iIcon = new QPixmap();
		if( !iIcon->loadFromData(QByteArray::fromBase64(iIconData)) ) {
			qDebug() << "Warning: Package" << iName << "has invalid icon data";
		}
	}
}

bool Package::isUpgradeable()
{
	if( iSection=="user/hidden" || iPinned )
		return false;

	if( iIsInstalled )
	{
		QString newer = upgradeableVersion();
		//qDebug() << newer << iVersion << versionCompare(newer,iVersion);
		return versionCompare(newer,iVersion);
	}

	return false;
}

QString Package::upgradeableVersion()
{
	QString ver;
	if( isInstalled() ) {
		Package* newer = iAptInterface->packagesAvailable()->value(iName,0);
		if( newer )
			ver = newer->version();
	}
	return ver;
}

Package* Package::availablePackage()
{
	if( !isInstalled() )
		return 0;

	Package* newer = iAptInterface->packagesAvailable()->value(iName,0);

	return newer;
}

void Package::updateStatus()
{
	if( iIsInstalled ) {
		iPkgStatus = PkgStatInstalled;
		if( isUpgradeable() )
			iPkgStatus = PkgStatUpgradeable;
	} else {
		iPkgStatus = PkgStatNotInstalled;
	}
}

Package::packageStatus Package::status()
{
	updateStatus(); // just in case
	return iPkgStatus;
}

bool Package::hasIconData()
{
	if( iIconData.length()>0 )
		return true;
	else
		return false;
}

bool Package::versionCompare(QString isNewer, QString compare)
{
	//int res=0;
	int res = debVS.CmpVersion(isNewer.toStdString(), compare.toStdString());
	//qDebug() << isNewer << compare << res;

	if( res > 0 )
		return true;

	return false;
}

QStringList Package::toTrimmedRichTextList(QList<QByteArray> list_in)
{
	QStringList list_out;

	for(int i=0; i<list_in.count(); i++)
	{
		QString s = list_in.at(i).trimmed();
		s.replace('<',"&lt;");
		s.replace('>',"&gt;");
		list_out << s;
	}

	return list_out;
}

QStringList Package::checkConflicts_RichText()
{
	QStringList list;

	if( iConflicts.count() == 0 )
		return list;

	for(int i=0; i<iConflicts.count(); i++)
	{
		QString s = iConflicts.at(i).trimmed();
		QString name = s;
		QString ver = "";
		int pos = s.indexOf('(');
		bool confl = false;
		if( pos != -1 ) {
			name = s.left(pos).trimmed();
			ver = s.mid(pos).trimmed();
		}
		Package* p_inst = iAptInterface->packagesInstalled()->value(name,0);
		Package* p_avail = iAptInterface->packagesAvailable()->value(name,0);

		if( p_inst && p_inst->isInstalled() ) {
			confl = versionConflicts(ver, p_inst->version());
		}
		if( p_avail && p_avail->markedOperation()==Package::PkgOpInstallUpgrade ) {
			if( !confl )
				confl = versionConflicts(ver, p_avail->version());
		}

		if( confl )
		{
			qDebug() << "package" << iName << "conflicts:" << name << ver;
			s.replace('<',"&lt;");
			s.replace('>',"&gt;");
			list << s;
		}
	}

	return list;
}

bool Package::versionConflicts(QString conflictVer, QString operVer)
{
	if( conflictVer.isEmpty() )
		return true;

	bool confl = false;
	conflictVer.remove('(');
	conflictVer.remove(')');

	if( conflictVer.startsWith("<<") ) {
		conflictVer = conflictVer.mid(2).trimmed();
		int res = debVS.CmpVersion(conflictVer.toStdString(), operVer.toStdString());
		if( res < 0 )
			confl = true;
	} else if( conflictVer.startsWith("<=") ) {
		conflictVer = conflictVer.mid(2).trimmed();
		int res = debVS.CmpVersion(conflictVer.toStdString(), operVer.toStdString());
		if( res <= 0 )
			confl = true;
	} else if( conflictVer.startsWith("=") ) {
		conflictVer = conflictVer.mid(1).trimmed();
		int res = debVS.CmpVersion(conflictVer.toStdString(), operVer.toStdString());
		if( res == 0 )
			confl = true;
	} else if( conflictVer.startsWith(">=") ) {
		conflictVer = conflictVer.mid(2).trimmed();
		int res = debVS.CmpVersion(conflictVer.toStdString(), operVer.toStdString());
		if( res >= 0 )
			confl = true;
	} else if( conflictVer.startsWith(">>") ) {
		conflictVer = conflictVer.mid(2).trimmed();
		int res = debVS.CmpVersion(conflictVer.toStdString(), operVer.toStdString());
		if( res > 0 )
			confl = true;
	}

	return confl;
}
