/*
	This file is part of Faster Application Manager.

	Faster Application Manager is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Faster Application Manager is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Faster Application Manager.  If not, see <http://www.gnu.org/licenses/>.

	(C) Heikki Holstila 2010
*/

#ifdef MYDEF_GTK_EXISTS
#include <gdk/gdk.h>
#include <gtk/gtk.h>
#endif

#include <QtCore>
#include <QtGui>
#include <QDBusConnection>
#include <QDBusInterface>
#include <phonon/AudioOutput>
#include <phonon/MediaObject>

#ifdef Q_WS_MAEMO_5
#include <QtMaemo5>
#endif

#include "mainwindow.h"
#include "version.h"
#include "ui_mainwindow.h"
#include "aaptinterface.h"
#include "packageview.h"
#include "confirmdialog.h"
#include "dimmer.h"
#include "repoview.h"
#include "help.h"
#include "settings.h"
#include "logview.h"
#include "rotatingbackground.h"
#include "dpkginterface.h"


MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    ui->setupUi(this);

	iAptInterface = new AAptInterface(this);
	iWinPackageView = new PackageView(this);
	iWinPackageView->setAptInterface(iAptInterface);
	iWinRepoView = new RepoView(this);
	iWinRepoView->setAptInterface(iAptInterface);
	iSettings = new Settings(this);
	iSettings->setAptInterface(iAptInterface);
	iSettings->setPackageView(iWinPackageView);
	iWinPackageView->setSettings(iSettings);
	iAptInterface->setSettings(iSettings);
	iDpkgInterface = new DpkgInterface(this);

	iWinPackageView->setSortOrder( (PackageView::sortOrder)iSettings->qsettings()->value("default_sort_order",0).toInt() );

	iWinPackageView->setSearchOptions( iSettings->qsettings()->value("search_pkgnames",true).toBool(),
									   iSettings->qsettings()->value("search_displaynames",true).toBool(),
									   iSettings->qsettings()->value("search_descshort",true).toBool(),
									   iSettings->qsettings()->value("search_desclong",false).toBool() );

#ifdef Q_WS_MAEMO_5
	this->setAttribute(Qt::WA_Maemo5StackedWindow);
	if( !iSettings->qsettings()->value("disable_autorotation",false).toBool() ) {
		this->setAttribute(Qt::WA_Maemo5AutoOrientation);
	} else {
		this->setAttribute(Qt::WA_Maemo5LandscapeOrientation);
	}
#endif

	iDimmer = new dimmer(this);

	iReposAutoUpdating = false;
	iUpgradeAutoUpdate = true;
	iNextOperation = OpNone;

	ui->centralWidget->loadWallpaper();

	QString stylesheet_mainscreen =
			"QPushButton {"
			"border-radius: 16px;"
			"border-width: 1px;"
			"border-color: palette(light);"
			"border-style: outset;"
			"padding-right: 10px;"
			"padding-left: 10px;"
			"color: palette(buttontext);"
			"background: rgba(255,255,255,80);"
			"}"
			"QPushButton:pressed {"
			"border-style: inset;"
			"background-color: rgba(255,255,255,150);"
			"}";

	if( ((QApplication*)QApplication::instance())->styleSheet().isEmpty() )
	{
		QString stylesheet_file;
		QFile f("/root/.fapman/style.css");
		if( f.open(QIODevice::ReadOnly | QIODevice::Text ) )
		{
			while(!f.atEnd()) {
				stylesheet_file += f.readLine().trimmed();
			}
			f.close();
		}

		if( stylesheet_file.isEmpty() ) {
			ui->centralWidget->setStyleSheet(stylesheet_mainscreen);
		} else {
			((QApplication*)QApplication::instance())->setStyleSheet(stylesheet_file);
		}
	}


	/*
	// does not work

	QDBusConnection conn = QDBusConnection::connectToBus(QDBusConnection::SystemBus, "faster_application_manager");

	QString service = "com.nokia.icd";
	QString path = "/com/nokia/icd";
	QString method = "connect";

	QDBusInterface net(service, path, service, conn, this);
	net.call(method,"[ANY]",0);
	*/

	iMediaObject = new Phonon::MediaObject(this);
	Phonon::AudioOutput* aout = new Phonon::AudioOutput(Phonon::NotificationCategory, this);
	Phonon::createPath(iMediaObject, aout);
}

MainWindow::~MainWindow()
{
	// save "need repo refresh" status
	iSettings->qsettings()->setValue("need_repo_refresh", iAptInterface->needRepoRefresh());

	delete iWinPackageView;
	delete iWinRepoView;
	delete iAptInterface;
	delete iDpkgInterface;
	delete iDimmer;
	delete iSettings;
    delete ui;
}

void MainWindow::changeEvent(QEvent *e)
{
    QMainWindow::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void MainWindow::on_btnRepos_clicked()
{
	iWinRepoView->openWin();
}

void MainWindow::on_btnUpdate_clicked()
{	
	// update catalogs

	busyDialog(true, tr("Operation in progress"), tr("Updating catalogs"));

	iAptInterface->addQueuedOperation(AAptInterface::ModeAptGetUpdate);
	iAptInterface->run(iDimmer);
}

#ifdef Q_WS_MAEMO_5
int MainWindow::top_application()
{
	show();
	activateWindow();
	raise();
	return 0;
}
#endif

void MainWindow::dateFetchAsk()
{
	if( !iSettings->qsettings()->value("firstrun_asked_fetch_dates",false).toBool() &&
		!iSettings->qsettings()->value("fetch_dates",false).toBool() )
	{
		iSettings->qsettings()->setValue("firstrun_asked_fetch_dates", true);
		ConfirmDialog d(true, this);
		d.setText("Fetch date information?","Enable date information fetching for packages? You have to enable it in order to be "
				  "able to sort packages by date. The first fetch can be slow but the dates are cached for later use. You can later "
				  "change this selection from the options menu.");
		if( d.exec() )
			iSettings->qsettings()->setValue("fetch_dates", true);
	}
	if( iSettings->qsettings()->value("fetch_dates",false).toBool() )
	{
		iSettings->qsettings()->setValue("firstrun_asked_fetch_dates", true);	// don't ask if the option has already been enabled
		if( !iAptInterface->dateCacheExists() )
		{
			ConfirmDialog d(false, this);
			d.setText("Notice","Date information will be fetched only for packages from maemo.org and only for user categories.");
			d.exec();
		}
		iAptInterface->addQueuedOperation(AAptInterface::ModeFetchDates);
	}
}

void MainWindow::on_btnListInstallable_clicked()
{
	//install

	int listupd = -1;
	int dpkgupd = -1;
	if( iAptInterface->lastListUpdate() < QDateTime::currentDateTime().addSecs(-KListExpireTime) )
		listupd = 1;
	if( iAptInterface->lastDpkgUpdate() < QDateTime::currentDateTime().addSecs(-KListExpireTime) )
		dpkgupd = 1;
	iAptInterface->setNeedRefresh(-1,listupd,dpkgupd,listupd);

	iWinPackageView->setStatFilter( Package::PkgStatNotInstalled );

	if( iAptInterface->needRepoRefresh() && !iSettings->qsettings()->value("no_catalogs_autoupdate",false).toBool() )
		iAptInterface->addQueuedOperation(AAptInterface::ModeAptGetUpdate);

	busyDialog(true, tr("Operation in progress"), tr("Reading package lists"));

	iNextOperation = OpOpenPkgView;
	iAptInterface->addQueuedOperation(AAptInterface::ModeReadPackages);

	dateFetchAsk();

	iAptInterface->run(iDimmer);
}

void MainWindow::on_btnUpgrade_clicked()
{
	// upgrade

	int listupd = -1;
	int dpkgupd = -1;
	if( iAptInterface->lastListUpdate() < QDateTime::currentDateTime().addSecs(-KListExpireTime) )
		listupd = 1;
	if( iAptInterface->lastDpkgUpdate() < QDateTime::currentDateTime().addSecs(-KListExpireTime) )
		dpkgupd = 1;
	iAptInterface->setNeedRefresh(-1,listupd,dpkgupd,listupd);

	iWinPackageView->setStatFilter( Package::PkgStatUpgradeable );

	if( iAptInterface->needRepoRefresh() && !iSettings->qsettings()->value("no_catalogs_autoupdate",false).toBool() )
		iAptInterface->addQueuedOperation(AAptInterface::ModeAptGetUpdate);

	busyDialog(true, tr("Operation in progress"), tr("Reading package lists"));

	iNextOperation = OpOpenPkgView;
	iAptInterface->addQueuedOperation(AAptInterface::ModeReadPackages);

	dateFetchAsk();

	iAptInterface->run(iDimmer);
}

void MainWindow::on_btnListInstalled_clicked()
{
	//remove

	if( !iSettings->qsettings()->value("remove_readfull",false).toBool() )
		iAptInterface->setSkipListAndDates();

	int dpkgupd = -1;
	if( iAptInterface->lastDpkgUpdate() < QDateTime::currentDateTime().addSecs(-KListExpireTime) )
		dpkgupd = 1;
	iAptInterface->setNeedRefresh(-1,-1,dpkgupd,-1);

	iWinPackageView->setStatFilter( Package::PkgStatInstalled );

	busyDialog(true, tr("Operation in progress"), tr("Reading package lists"));

	iNextOperation = OpOpenPkgView;
	iAptInterface->addQueuedOperation(AAptInterface::ModeReadPackages);

	dateFetchAsk();

	iAptInterface->run(iDimmer);
}

void MainWindow::operationQueueFinished(QList<AAptInterface::interfaceMode> lastModes, bool success, QString title, QStringList msgs)
{
	if( lastModes.contains(AAptInterface::ModeAptGetSimulate) && success ) {
		iNextOperation = OpPromptSimulated;
	}
	if( lastModes.contains(AAptInterface::ModeAptGetInstall) && success ) {
		iWinPackageView->clearSelections();
		busyDialog(false);
		iWinPackageView->close();
		iNextOperation = OpNone;

		GdkEventIconThemeReload();
	}

	if( iNextOperation == OpOpenPkgView && success )
	{
		iWinPackageView->openWin();
		busyDialog(false);
		iNextOperation = OpNone;
	} else if( iNextOperation == OpPromptSimulated && success )
	{
		QStringList inst,remv,instver,remvver;
		QStringList all = iAptInterface->processPackages();
		QStringList vers = iAptInterface->processPackageVersions();
		for(int zyx=0; zyx<all.count(); zyx++)
		{
			if( all.at(zyx).endsWith('-') )
			{
				remv.append( all.at(zyx).left( all.at(zyx).size()-1 ) );
				if( vers.count()>zyx )
					remvver.append( vers.at(zyx) );
			} else {
				inst.append( all.at(zyx) );
				if( vers.count()>zyx )
					instver.append( vers.at(zyx) );
			}
		}

		int total_dl_size = 0;
		for( int i=0; i<inst.count(); i++ ) {
			Package* pkg = iAptInterface->packagesAvailable()->value(inst.at(i),0);
			if( pkg ) {
				total_dl_size += pkg->size()/1024;
			}
		}

		QString pkglist;
		pkglist = QString("<b>The following operations will be performed:</b><br>"
				  "%1 to install/upgrade, %2 to remove<br>").arg(inst.count()).arg(remv.count());
		if( inst.count()>0 )
			pkglist += QString("Total download size: %L1 kB<br>").arg(total_dl_size);
		bool mismatch = false;

		if( remv.count()>0 ) {
			pkglist += "<br><b><u>REMOVE:</u></b><br><font size=\"-1\">";
			for( int i=0; i<remv.count(); i++ ) {
				pkglist += "<b>" + remv.at(i) + "</b>";
				Package* pkg = iAptInterface->packagesInstalled()->value(remv.at(i),0);
				if( !pkg ) {
					qDebug() << "Warning: unknown package" << remv.at(i);
					pkglist += "<font color=\"red\">***UNKNOWN***</font>";
					mismatch = true;
				}
				if( remvver.count()>i ) {
					pkglist += " " + remvver.at(i);
					if( pkg && remvver.at(i) != pkg->version() ) {
						qDebug() << "Version mismatch, database version is" << pkg->version() << "but removing" << remvver.at(i);
						mismatch = true;
						pkglist += " <font color=\"red\">***TRYING TO REMOVE " + pkg->version() + "***</font> ";
					}
				}
				if( pkg && pkg->installedSize()>0 )
					pkglist += QString(" (%L1 kB)").arg(pkg->installedSize());
				pkglist += "<br>";
			}
		}
		pkglist += "</font>";

		bool installing_blacklisted = false;
		if( inst.count()>0 ) {
			pkglist += "<br><b><u>INSTALL/UPGRADE:</u></b><br><font size=\"-1\">";
			for( int i=0; i<inst.count(); i++ ) {
				pkglist += "<b>" + inst.at(i) + "</b>";
				Package* pkg = iAptInterface->packagesAvailable()->value(inst.at(i),0);
				if( !pkg ) {
					qDebug() << "Warning: unknown package" << inst.at(i);
					pkglist += "<font color=\"red\">***NEW/UNKNOWN***</font>";
					mismatch = true;
				}
				if( pkg && pkg->isBlacklisted() ) {
					qDebug() << "Warning: installing blacklisted package" << inst.at(i);
					pkglist += "<font color=\"red\">***BLACKLISTED***</font>";
					installing_blacklisted = true;
				}
				if( instver.count()>i ) {
					pkglist += " " + instver.at(i);
					if( pkg && instver.at(i) != pkg->version() ) {
						qDebug() << "Version mismatch, database version is" << pkg->version() << "but installing" << instver.at(i);
						mismatch = true;
						pkglist += " <font color=\"red\">***TRYING TO INSTALL " + pkg->version() + "***</font> ";
					}
				}
				if( pkg && pkg->size()>0 ) {
					pkglist += QString(" (%L1 kB)").arg(pkg->size()/1024);
				}
				pkglist += "<br>";
			}
		}
		pkglist += "</font>";

		if( mismatch ) {
			ConfirmDialog m(false, this);
			m.setText("Warning", "There is a version mismatch between your original package selections and some of the packages being installed " \
					  "from the repositories. This could be due to your application catalogs being out of date.");
			m.exec();
		}
		if( installing_blacklisted ) {
			ConfirmDialog b(false, this);
			b.setText("Warning","Blacklisted package(s) will be installed");
			b.exec();
		}

		busyDialog(false);
		ConfirmDialog d(true, this);
		d.setText("Confirmation",pkglist);
		if( d.exec() ) {
			iAptInterface->addQueuedOperation(AAptInterface::ModeAptGetInstall);

			if( iSettings->qsettings()->value("enable_autoclean",true).toBool() && inst.count()>0 )
				iAptInterface->addQueuedOperation(AAptInterface::ModeAptGetClean);

			QString busytext;
			if( remv.count() > 0 )
				busytext += QString("Remove %1 package(s)<br>").arg(remv.count());
			if( inst.count() > 0 )
				busytext += QString("Install %1 package(s)<br>").arg(inst.count());
			busytext += "Preparing...";
			busyDialog(true, "Operation in progress", busytext);

			// "run" really does nothing here since the previous item should still be running
			if( iWinPackageView->isVisible() ) {
				iAptInterface->run(iWinPackageView->mydimmer());
			} else {
				iAptInterface->run(iDimmer);
			}
		}
		iNextOperation = OpNone;
		return;
	} else {
		busyDialog(false);
		iNextOperation = OpNone;

		if( iSettings->qsettings()->value("sound_notify",false).toBool() )
		{
			qDebug() << "playing sound";
			iMediaObject->setCurrentSource( Phonon::MediaSource(iSettings->qsettings()->value("sound_file","/usr/share/sounds/ui-operation_ready.wav").toString()) );
			iMediaObject->play();
		}

		QString text = "<br><b><u>Faster Application Manager</u></b><br>"
					   "<b>"+title+"</b><br>" + msgs.join("<br>") + "<br>";

		QRect r = QApplication::desktop()->rect();
		if(r.width() < r.height()) {
			ConfirmDialog d(false, this);
			d.setText(title,msgs.join("<br>"));
			d.exec();
		} else {
#ifdef Q_WS_MAEMO_5
			QMaemo5InformationBox::information(0, text, QMaemo5InformationBox::NoTimeout);
#endif
		}

	}

}

void MainWindow::busyDialog(bool show_, QString title, QString text)
{
	if( show_ ) {
		iDimmer->setProgress(-1);
		ui->menuMenu->setEnabled(false);
		ui->centralWidget->setEnabled(false);
		iWinPackageView->disableMenu();
		iDimmer->resizeEvent(0);
		iDimmer->dim(title, text);
		iWinPackageView->mydimmer()->resizeEvent(0);
		iWinPackageView->mydimmer()->dim(title, text);
	} else {
		iDimmer->undim();
		iWinPackageView->mydimmer()->undim();
		ui->menuMenu->setEnabled(true);
		ui->centralWidget->setEnabled(true);
		iWinPackageView->enableMenu();
	}
}

void MainWindow::on_actionAbout_triggered()
{
	ConfirmDialog d(false, this);
	d.setText("About","Faster Application Manager<br>"
			  "<font size=\"-1\">Version " + PROGRAM_VERSION + "</font><br><br>"
			  "(C) Heikki Holstila 2010<br>Donate using "
			  "<a href=\"https://www.paypal.com/cgi-bin/webscr?cmd=_donations&business=6ZKRY5QFHL42A&lc=FI&item_name=Faster%20Application%20Manager"
			  "%20for%20Maemo5&currency_code=EUR&bn=PP%2dDonationsBF%3abtn_donate_SM%2egif%3aNonHosted\">PayPal</a>");
	d.exec();
}

void MainWindow::on_actionClean_triggered()
{
	//if( iOperation != OpNone ) return;
	//iOperation = OpClean;
	iAptInterface->addQueuedOperation(AAptInterface::ModeAptGetClean);
	iAptInterface->run(iDimmer);
}

void MainWindow::closeEvent(QCloseEvent *event)
{
	if( iDimmer->busy() ) {
		iAptInterface->cancel();
		event->ignore();
	} else {
		event->accept();
	}
}

void MainWindow::on_actionView_log_triggered()
{
	QByteArray log = iAptInterface->readLogFile();
	LogView l(log, this);
	l.exec();
}

void MainWindow::on_actionOptions_triggered()
{
	iSettings->openWin();
}

void MainWindow::notifyDialog(QString title, QString msg)
{
	ConfirmDialog d(false, this);
	d.setText(title, msg);
	d.exec();
}

bool MainWindow::confirmDialog(QString title, QString msg)
{
	ConfirmDialog d(true, this);
	d.setText(title, msg);
	return d.exec();
}

void MainWindow::GdkEventIconThemeReload()
{
	// DOES NOT EVEN WORK (at least not reliably) - disabled from the project file

#ifdef MYDEF_GTK_EXISTS
	qDebug() << "Sending GDK icon theme reload event";

	gdk_init((int*)QApplication::argc(),(gchar***)QApplication::argv());

	// taken from hildon application manager
	GdkEventClient ev;
	ev.type = GDK_CLIENT_EVENT;
	ev.window = NULL;
	ev.send_event = TRUE;
	ev.message_type = gdk_atom_intern_static_string("_GTK_LOAD_ICONTHEMES");
	ev.data_format = 32;
	gdk_event_send_clientmessage_toall((GdkEvent*)&ev);

	while(gdk_events_pending()) {
		g_main_context_iteration(NULL, true);
	}

#endif
}

void MainWindow::on_actionLoad_file_triggered()
{
	QStringList files = QFileDialog::getOpenFileNames(this, "Open files", "/", "Files (*.deb *.install)");
	if( files.count() > 0 ) {
		QStringList debs = files.filter(QRegExp(".*\\.deb$"));
		QStringList installs = files.filter(QRegExp(".*\\.install$"));
		if( debs.count()>0 && installs.count()>0 ) {
			ConfirmDialog d(false, this);
			d.setText("Error", "You can't mix different file types in your selection");
			d.exec();
			return;
		}
		if( debs.count()>0 )
			iDpkgInterface->loadDebFiles(debs);
		else if( installs.count()>0 )
			iAptInterface->loadInstallFiles(installs);
	}
}
