/*
	This file is part of Faster Application Manager.

	Faster Application Manager is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Faster Application Manager is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Faster Application Manager.  If not, see <http://www.gnu.org/licenses/>.

	(C) Heikki Holstila 2010
*/

#include <QtGui>
#include "package.h"
#include "repository.h"
#include "aaptinterface.h"
#include "apt-src/debversion.h"
#include "blacklistselect.h"

Package::Package(QByteArray name_, AAptInterface *apt_)
{
	iAptInterface = apt_;
	iName = name_;
	iIsInstalled = false;
	iMarkedForOperation = false;
	iSize = 0;
	iInstalledSize = 0;
	iMarkedOperation = PkgOpNone;
	iPkgStatus = PkgStatUnknown;
	iIcon = 0;
	iBlacklist = BlacklistSelect::BlacklistNone;
}

Package::~Package()
{
	if( iIcon != 0 )
		delete iIcon;
}

QString Package::displayName()
{
	QString pkgname = name();
	if( iMaemoDisplayName!="" )
		pkgname = maemoDisplayName();
	QString n( pkgname.at(0) );
	n = n.toUpper();
	pkgname.replace(0,1,n);
	return pkgname;
}

QString Package::fileName()
{
	if( iFullFileNames.count()>0 ) {
		return iFullFileNames.at(0).mid( iFullFileNames.at(0).lastIndexOf('/')+1 );
	} else {
		qDebug() << "Warning: package has no file name information";
		return "unknown_filename";
	}
}

void Package::setMarkedForOperation(operation op_)
{
	if( iMarkedOperation != op_ ) {
		if( op_==PkgOpNone ) {
			if( iAptInterface )
				iAptInterface->setNumSelectedPackages( iAptInterface->numSelectedPackages()-1 );
		} else if( iMarkedOperation==PkgOpNone ) {
			if( iAptInterface )
				iAptInterface->setNumSelectedPackages( iAptInterface->numSelectedPackages()+1 );
		}
	}

	iMarkedOperation = op_;

	if( op_ == PkgOpNone )
		iMarkedForOperation = false;
	else
		iMarkedForOperation = true;
}

void Package::convertIcon()
{
	if( iIconData.length() > 0 && iIcon == 0 ) {
		iIcon = new QPixmap();
		iIconData = QByteArray::fromBase64( iIconData );
		if( !iIcon->loadFromData( iIconData ) ) {
			qDebug() << "Warning: Package" << iName << "has invalid icon data";
		}
	}

}

bool Package::isUpgradeable()
{
	if( iIsInstalled )
	{
		QString newer = upgradeableVersion();
		//qDebug() << newer << iVersion << versionCompare(newer,iVersion);
		return versionCompare(newer,iVersion);
	}

	return false;
}

QString Package::upgradeableVersion()
{
	QString ver="";

	if( isInstalled() ) {
		Package* newer = iAptInterface->packagesAvailable()->value(iName,0);
		if( newer )
			ver = newer->version();
	}
	return ver;
}

Package* Package::availablePackage()
{
	if( !isInstalled() )
		return 0;

	Package* newer = iAptInterface->packagesAvailable()->value(iName,0);

	return newer;
}

void Package::updateStatus()
{
	if( iIsInstalled ) {
		iPkgStatus = PkgStatInstalled;
		if( isUpgradeable() )
			iPkgStatus = PkgStatUpgradeable;
	} else {
		iPkgStatus = PkgStatNotInstalled;
	}
}

Package::packageStatus Package::status()
{
	updateStatus(); // not optimal, but it's here just in case
	return iPkgStatus;
}

bool Package::hasIconData()
{
	if( iIconData.length()>0 )
		return true;
	else
		return false;
}

bool Package::versionCompare(QString isNewer, QString compare)
{
	//int res=0;
	int res = debVS.CmpVersion(isNewer.toStdString(), compare.toStdString());
	//qDebug() << isNewer << compare << res;

	if( res > 0 )
		return true;

	return false;
}

void Package::setDepends(QByteArray depends_)
{
	iDepends = depends_.split(',');
}

void Package::setConflicts(QByteArray conflicts_)
{
	iConflicts = conflicts_.split(',');
}

QStringList Package::depends()
{
	//qDebug() << iName;
	//qDebug() << iDepends;

	QStringList list;

	for(int i=0; i<iDepends.count(); i++)
	{
		/*
		QList<QByteArray> parts = iConflicts.at(i).split('|');
		for(int j=0; j<parts.count(); j++)
		{
			qDebug() << parts.at(j);
			int pos = parts.at(j).trimmed().indexOf(' ');
			QString s;
			if( pos==-1 )
				s = parts.at(j).trimmed();
			else
				s = parts.at(j).left(pos).trimmed();
			list << s;
		}
		*/
		list << iDepends.at(i).trimmed();
	}

	return list;
}

QStringList Package::conflicts()
{
	QStringList list;

	for(int i=0; i<iConflicts.count(); i++)
	{
		list << iConflicts.at(i).trimmed();
	}

	return list;
}
