/*
	This file is part of Faster Application Manager.

	Faster Application Manager is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Faster Application Manager is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Faster Application Manager.  If not, see <http://www.gnu.org/licenses/>.

	(C) Heikki Holstila 2010
*/

#include <QtGui>
#include <QtNetwork>

#include "aaptinterface.h"
#include "package.h"
#include "dimmer.h"
#include "repository.h"
#include "mainwindow.h"
#include "settings.h"

AAptInterface::AAptInterface(QObject* parent = 0) : QObject(parent)
{
	iMode = ModeNone;
	iMainWindow = dynamic_cast<MainWindow*>(parent);
	iUiDimmer = 0;
	iSettings = 0;
	iTerminated = false;
	iCanCancel = false;
	iNumSelectedPackages = 0;
	iErrorDone = false;
	iNeedRepoRefresh = false;

	iProcAptGetUpdate = new QProcess(this);
	iProcAptGetSimulate = new QProcess(this);
	iProcAptGetInstall = new QProcess(this);
	iProcAptGetClean = new QProcess(this);

	iProcAptGetUpdate->setProcessChannelMode(QProcess::MergedChannels);
	iProcAptGetSimulate->setProcessChannelMode(QProcess::MergedChannels);
	iProcAptGetInstall->setProcessChannelMode(QProcess::MergedChannels);
	iProcAptGetClean->setProcessChannelMode(QProcess::MergedChannels);

	connect(iProcAptGetUpdate,SIGNAL(error(QProcess::ProcessError)),this,SLOT(errorAptGetUpdate(QProcess::ProcessError)));
	connect(iProcAptGetUpdate,SIGNAL(finished(int,QProcess::ExitStatus)),this,SLOT(finishedAptGetUpdate(int,QProcess::ExitStatus)));
	connect(iProcAptGetUpdate,SIGNAL(readyRead()),this,SLOT(uiUpdaterAptGetUpdate()));

	connect(iProcAptGetInstall,SIGNAL(error(QProcess::ProcessError)),this,SLOT(errorAptGetInstall(QProcess::ProcessError)));
	connect(iProcAptGetInstall,SIGNAL(finished(int,QProcess::ExitStatus)),this,SLOT(finishedAptGetInstall(int,QProcess::ExitStatus)));
	connect(iProcAptGetInstall,SIGNAL(readyRead()),this,SLOT(uiUpdaterAptGetInstall()));

	connect(iProcAptGetSimulate,SIGNAL(error(QProcess::ProcessError)),this,SLOT(errorAptGetSimulate(QProcess::ProcessError)));
	connect(iProcAptGetSimulate,SIGNAL(finished(int,QProcess::ExitStatus)),this,SLOT(finishedAptGetSimulate(int,QProcess::ExitStatus)));

	connect(iProcAptGetClean,SIGNAL(error(QProcess::ProcessError)),this,SLOT(errorAptGetClean(QProcess::ProcessError)));
	connect(iProcAptGetClean,SIGNAL(finished(int,QProcess::ExitStatus)),this,SLOT(finishedAptGetClean(int,QProcess::ExitStatus)));

	iProcessPackages.clear();
	iProcessPackagesOrig.clear();
	iProcessPackageVersions.clear();

	QDir logdir(KLogFileDir);
	logdir.mkpath(KLogFileDir);

	readRepositoryInfo();
}

AAptInterface::~AAptInterface()
{
	if( iProcAptGetUpdate->state() != QProcess::NotRunning )
		iProcAptGetUpdate->kill();
	if( iProcAptGetSimulate->state() != QProcess::NotRunning )
		iProcAptGetSimulate->kill();
	if( iProcAptGetInstall->state() != QProcess::NotRunning )
		iProcAptGetInstall->kill();
	if( iProcAptGetClean->state() != QProcess::NotRunning )
		iProcAptGetClean->kill();
}

void AAptInterface::addQueuedOperation(interfaceMode mode_)
{
	iOperationsQueue.append( mode_ );
}

bool AAptInterface::run(dimmer* uiDimmer)
{
	if( iMode != ModeNone ) {
		//qDebug() << "Can't run: not ModeNone";
		return false;
	}

	if( iOperationsQueue.count() == 0 ) {
		qDebug() << "Can't run: Queue empty";
		return false;
	}

	iUiDimmer = uiDimmer;
	iQueueMessages.clear();
	iModeLog.clear();

	runNext();

	return true;
}

void AAptInterface::runNext()
{
	if( iOperationsQueue.count()==0 ) {
		cleanAfterRunAll();
		return;
	}
	if( iTerminated ) {
		cleanAfterError();
		return;
	}

	cleanAfterRunEach();

	iMode = iOperationsQueue.takeAt(0);
	iModeLog.append(iMode);

	if( iMode == ModeAptGetUpdate ) {
		if( !startAptGetUpdate() )
			errorAptGetUpdate( QProcess::FailedToStart );
	}
	if( iMode == ModeAptGetInstall ) {
		if( !startAptGetInstall() )
			errorAptGetInstall( QProcess::FailedToStart );
	}
	if( iMode == ModeAptGetSimulate ) {
		if( !startAptGetSimulate() )
			errorAptGetSimulate( QProcess::FailedToStart );
	}
	if( iMode == ModeAptGetClean ) {
		if( !startAptGetClean() )
			errorAptGetClean( QProcess::FailedToStart );
	}
	if( iMode == ModeFetchDates )
		startFetchDates();

	if( iMode == ModeReadPackages )
		startPkgListRead();
}

void AAptInterface::cleanAfterRunEach()
{
	iMode = ModeNone;
	iTerminated = false;
	iErrorDone = false;
	iCanCancel = false;
}

void AAptInterface::cleanAfterRunAll()
{
	cleanAfterRunEach();
	iUiDimmer = 0;
}

void AAptInterface::cleanAfterError()
{
	cleanAfterRunAll();
	iOperationsQueue.clear();
	iProcessPackages.clear();
	iProcessPackagesOrig.clear();
	iProcessPackageVersions.clear();
}

bool AAptInterface::running()
{
	if( iMode == ModeNone )
		return false;
	return true;
}

bool AAptInterface::cancel()
{
	// should return false if can't cancel, or terminate the running process (and clear queue) otherwise
	if( iMode == ModeNone )
		return false;

	if( !iCanCancel ) {
		return false;
	}

	if( iMode == ModeAptGetUpdate ) {
		if( iProcAptGetUpdate->state() == QProcess::Running )
			iProcAptGetUpdate->terminate();
		else
			return false;
		cleanAfterError();
		iTerminated = true;
		iNeedRepoRefresh = true;
		return true;
	}
	if( iMode == ModeAptGetSimulate ) {
		if( iProcAptGetSimulate->state() == QProcess::Running )
			iProcAptGetSimulate->terminate();
		else
			return false;
		cleanAfterError();
		iTerminated = true;
		return true;
	}
	if( iMode == ModeAptGetInstall ) {
		if( iProcAptGetInstall->state() == QProcess::Running )
			iProcAptGetInstall->terminate();
		else
			return false;
		cleanAfterError();
		iTerminated = true;
		return true;
	}
	if( iMode == ModeReadPackages ) {
		iTerminated = true;
		return true;
	}
	if( iMode == ModeFetchDates ) {
		iTerminated = true;
		return true;
	}
	return false;
}

bool AAptInterface::startAptGetUpdate()
{
	iCanCancel = true;
	iProcAptGetUpdateOutput.clear();

	if( !this->writeRepositories() )
		return false;

	if( iUiDimmer ) {
		iUiDimmer->setProgress(0);
		iUiDimmer->updateText("Updating catalogs");		
	}

	iCatalogCounter = 0;
	iCatalogsTotal = 0;
	for( int i=0; i<iRepositories.count(); i++ ) {
		if( iRepositories.at(i) && iRepositories.at(i)->enabled() ) {
			iCatalogsTotal += 2;
			QStringList comp = iRepositories.at(i)->components().split(' ');
			iCatalogsTotal += comp.count();
		}
	}

	bool useproxy = iSettings->qsettings()->value("use_proxies",false).toBool();
	QString http_proxy = iSettings->qsettings()->value("http_proxy","").toString();
	QString https_proxy = iSettings->qsettings()->value("https_proxy","").toString();
	QProcessEnvironment env = QProcessEnvironment::systemEnvironment();
	if( useproxy ) {
		if( http_proxy != "" )
			env.insert("http_proxy", http_proxy);
		if( https_proxy != "" )
			env.insert("https_proxy", https_proxy);
	}
	iProcAptGetUpdate->setProcessEnvironment(env);

	QString runBinary = "/usr/bin/apt-get";
	QStringList runParameters;
	runParameters << "-q" << "update";

	logToFile( runBinary + " " + runParameters.join(" ") );
	iProcAptGetUpdate->start(runBinary,runParameters);

	return true;
}

bool AAptInterface::startAptGetSimulate()
{
	if( iProcessPackages.count()==0 )
		return false;
	iCanCancel = true;

	if( iUiDimmer ) {
		iUiDimmer->updateText("Reading dependencies");
	}

	QString runBinary = "/usr/bin/apt-get";
	QStringList runParameters;
	runParameters << "-qsy" << "--allow-unauthenticated";
	if( iSettings->qsettings()->value("enable_autoremove", false).toBool() )
		runParameters << "--auto-remove";
	runParameters << "install";
	runParameters << iProcessPackages;

	logToFile( runBinary + " " + runParameters.join(" ") );
	iProcAptGetSimulate->start(runBinary,runParameters);

	return true;
}

bool AAptInterface::startAptGetInstall()
{
	if( iProcessPackages.count()==0 )
		return false;

	iProcAptGetInstallOutput.clear();

	qDebug() << "running apt-get install";

	QString runBinary = "/usr/bin/apt-get";
	QStringList runParameters;
	runParameters << "-qy" << "--allow-unauthenticated";
	if( iSettings->qsettings()->value("enable_autoremove", false).toBool() )
		runParameters << "--auto-remove";
	runParameters << "install";
	runParameters << iProcessPackagesOrig;

	int inst_count = 0;
	int remv_count = 0;
	for( int i=0; i<iProcessPackages.count(); i++) {
		if( iProcessPackages.at(i).endsWith('-') )
			remv_count++;
		else
			inst_count++;
	}

	iAptGetDownloadCount = 0;
	iAptGetInstallCount = 0;
	iAptGetRemoveCount = 0;
	iAptGetInstallTotal = inst_count;
	iAptGetRemoveTotal = remv_count;
	iAptGetCurrentFileDownloadSize = 0;
	iAptGetCurrentFileTotalSize = 0;
	iAptGetCurrentDownloadFileName = "";
	iProgressCheckTimer = 0;

	bool useproxy = iSettings->qsettings()->value("use_proxies",false).toBool();
	QString http_proxy = iSettings->qsettings()->value("http_proxy","").toString();
	QString https_proxy = iSettings->qsettings()->value("https_proxy","").toString();
	QProcessEnvironment env = QProcessEnvironment::systemEnvironment();
	if( useproxy ) {
		if( http_proxy != "" )
			env.insert("http_proxy", http_proxy);
		if( https_proxy != "" )
			env.insert("https_proxy", https_proxy);
	}
	iProcAptGetUpdate->setProcessEnvironment(env);

	logToFile( runBinary + " " + runParameters.join(" ") );
	iProcAptGetInstall->start(runBinary,runParameters);

	return true;
}

bool AAptInterface::startAptGetClean()
{
	QString runBinary = "/usr/bin/apt-get";
	QStringList runParameters;
	runParameters << "clean";

	if( iUiDimmer ) {
		iUiDimmer->updateText("Cleaning package cache");
	}

	logToFile( runBinary + " " + runParameters.join(" ") );
	iProcAptGetClean->start(runBinary,runParameters);

	return true;
}

QString AAptInterface::setQProcessErrorMessage(QProcess::ProcessError error)
{
	if( iTerminated ) {
		logToFile(QString("Cancelled by user - terminating process"));
		return "Cancelled by user";
	}
	else if( error == QProcess::FailedToStart )
		return "Process failed to start";
	else if( error == QProcess::Crashed )
		return "Process crashed";
	else if( error == QProcess::ReadError )
		return "QProcess read error";
	else if( error == QProcess::UnknownError )
		return "QProcess unknown error";

	return "Unknown error";
}

void AAptInterface::errorAptGetUpdate(QProcess::ProcessError error)
{
	QString msg = setQProcessErrorMessage(error);

	if( iUiDimmer )
		iUiDimmer->setProgress(-1);

	communicateStatusToUi(false, "Error", msg);
	cleanAfterError();
	iNeedRepoRefresh = true;
	iErrorDone = true;
}

void AAptInterface::errorAptGetSimulate(QProcess::ProcessError error)
{
	QString msg = setQProcessErrorMessage(error);
	iProcessPackages.clear();
	iProcessPackagesOrig.clear();
	iProcessPackageVersions.clear();

	communicateStatusToUi(false, "Error", msg);
	cleanAfterError();
	iErrorDone = true;
}

void AAptInterface::errorAptGetInstall(QProcess::ProcessError error)
{
	QString msg = setQProcessErrorMessage(error);
	iProcessPackages.clear();
	iProcessPackagesOrig.clear();
	iProcessPackageVersions.clear();

	if( iProgressCheckTimer ) {
		iProgressCheckTimer->stop();
		delete iProgressCheckTimer;
		iProgressCheckTimer = 0;
	}
	if( iUiDimmer ) {
		iUiDimmer->setProgress(-1);
	}

	communicateStatusToUi(false, "Error", msg);
	cleanAfterError();
	iErrorDone = true;
}

void AAptInterface::errorAptGetClean(QProcess::ProcessError error)
{
	QString msg = setQProcessErrorMessage(error);

	communicateStatusToUi(false, "Error", msg);
	cleanAfterError();
	iErrorDone = true;
}

QString AAptInterface::finishProcessCommonErrorMessages(QByteArray& output)
{
	QString msg = "Unknown error - see the log for details";

	if( output.contains("Could not get lock") || output.contains("Could not open lock file") ) {
		msg = "The package management system is locked by another process or permission was denied";
	} else if( output.contains("E: Unable to fetch some archives") ) {
		msg = "Failed to fetch packages - Your network connection might be down or your catalogs could be out of date";
	} else if( output.contains("E: Couldn't find package") ) {
		msg = "Missing package, see the log for details - Your catalogs might be out of date";
	} else if( output.contains("E: Broken packages") ) {
		msg = "Your system has broken packages, see the log for details";
	} else if( output.contains("E: Handler silently failed") ) {
		msg = "Handler silently failed - This can happen if you try to install from the forbidden user/hidden category";
	} else if( iTerminated ) {
		msg = "The operation was cancelled by user";
	} else if( output.contains("Temporary failure resolving") || output.contains("Could not resolve host") ) {
		msg = "DNS errors were reported, check your network connection and/or repository configuration";
	} else if( output.contains("E: dpkg was interrupted") ) {
		msg = "Your system has partially installed or broken packages. You'll have to fix this manually. Try dpkg --configure -a";
	} else if( output.contains("dpkg: error processing") || output.contains("Errors were encountered while processing:") ) {
		msg = "dpkg reported errors while processing a package - see the log for details";
	}

	return msg;
}

void AAptInterface::finishedAptGetUpdate(int exitCode, QProcess::ExitStatus exitStatus)
{
	//QByteArray output = iProcAptGetUpdate->readAllStandardOutput();
	//logToFile( "Output from last process:\n---\n"+output );

	if( iErrorDone ) {
		iErrorDone = false;
		iProcAptGetUpdate->close();
		return;
	}

	bool success = true;
	QString title = "Operation finished";
	QString msg = "Catalogs updated";
	if( exitCode != 0 || exitStatus == QProcess::CrashExit )
	{
		success = false;
		title = "Error";
		msg = finishProcessCommonErrorMessages(iProcAptGetUpdateOutput);
	}

	if( iUiDimmer )
		iUiDimmer->setProgress(-1);

	if( iProcAptGetUpdateOutput.contains("Could not resolve ") || iProcAptGetUpdateOutput.contains("W: Failed to fetch") ||
		iProcAptGetUpdateOutput.contains("Temporary failure resolving") ) {
		success = false;
		title = "Error";
		msg = "Failed to update some or all of the catalogs. Check your network connection and/or repository configuration";
	}

	if( success ) {
		iNeedRepoRefresh = false;

		QFile lastupdate("/root/.fapman/lastupdate");   // create an empty file and/or update the modification time
		if( lastupdate.open(QIODevice::WriteOnly) )
			lastupdate.close();

		int pos = iProcAptGetUpdateOutput.indexOf("\nFetched ");
		if( pos!=-1 ) {
			msg += "<br>apt-get: ";
			msg += iProcAptGetUpdateOutput.mid(pos+1, iProcAptGetUpdateOutput.indexOf('\n', pos+1)-pos ).trimmed();
		}

	} else {
		cleanAfterError();
	}

	iProcAptGetUpdate->close();
	communicateStatusToUi(success, title, msg);
	runNext();
}

void AAptInterface::finishedAptGetSimulate(int exitCode, QProcess::ExitStatus exitStatus)
{
	QByteArray output = iProcAptGetSimulate->readAllStandardOutput();
	logToFile( "Output from last process:\n---\n"+output );

	if( iErrorDone ) {
		iErrorDone = false;
		iProcAptGetSimulate->close();
		return;
	}

	bool success = true;
	QString title = "Operation finished";
	QString msg = "";
	if( exitCode != 0 || exitStatus == QProcess::CrashExit )
	{
		success = false;
		title = "Error";
		msg = finishProcessCommonErrorMessages(output);
	}

	iProcessPackages.clear();
	iProcessPackageVersions.clear();
	if( success )
	{
		QList<QByteArray> lines = output.split('\n');

		for( int i=0; i<lines.count(); i++)
		{
			QString s = lines.at(i);
			if( s.startsWith("Inst ") )
			{
				iProcessPackages <<  s.section(' ',1,1);

				QString vs="";
				/*
				int a1 = s.indexOf('[');
				int a2 = s.indexOf(']');
				if( a1!=-1 && a2!=-1 && a2>a1) {
					vs = s.mid(a1+1, a2-a1-1) + " -> ";
				}*/
				int b1 = s.indexOf('(');
				int b2 = s.indexOf(' ',b1);
				if( b1!=-1 && b2!=-1 && b2>b1) {
					vs += s.mid(b1+1, b2-b1-1);
				}
				//qDebug() << vs;
				iProcessPackageVersions << vs;
			}
			if( s.startsWith("Remv ") )
			{
				iProcessPackages << s.section(' ',1,1) + "-";

				QString vs="";
				int a1 = s.indexOf('[');
				int a2 = s.indexOf(']');
				if( a1!=-1 && a2!=-1 && a2>a1) {
					vs = s.mid(a1+1, a2-a1-1);
				}
				//qDebug() << vs;
				iProcessPackageVersions << vs;
			}
		}
	}
	if( !success )
		cleanAfterError();

	iProcAptGetSimulate->close();
	communicateStatusToUi(success, title, msg);
	runNext();
}

void AAptInterface::finishedAptGetInstall(int exitCode, QProcess::ExitStatus exitStatus)
{
	//QByteArray output = iProcAptGetInstall->readAllStandardOutput();
	//logToFile( "Output from last process:\n---\n"+output );

	if( iErrorDone ) {
		iProcAptGetInstall->close();
		iErrorDone = false;
		return;
	}

	bool success = true;
	QString title = "Operation finished";
	QString msg = "Package operations finished successfully";
	if( exitCode != 0 || exitStatus == QProcess::CrashExit || iTerminated )
	{
		success = false;
		title = "Error";
		msg = finishProcessCommonErrorMessages(iProcAptGetInstallOutput);
	}

	if( iProgressCheckTimer ) {
		iProgressCheckTimer->stop();
		delete iProgressCheckTimer;
		iProgressCheckTimer = 0;
	}
	if( iUiDimmer ) {
		iUiDimmer->setProgress(-1);
	}
	iProcessPackages.clear();
	iProcessPackagesOrig.clear();
	iProcessPackageVersions.clear();

	if( !success )
		cleanAfterError();

	iProcAptGetInstall->close();
	communicateStatusToUi(success, title, msg);
	runNext();
}

void AAptInterface::finishedAptGetClean(int exitCode, QProcess::ExitStatus exitStatus)
{
	QByteArray output = iProcAptGetClean->readAllStandardOutput();
	// this should produce no output
	//logToFile( "Output from last process:\n---\n"+output );

	if( iErrorDone ) {
		iErrorDone = false;
		iProcAptGetClean->close();
		return;
	}

	bool success = true;
	QString title = "Operation finished";
	QString msg = "Package cache cleaned";
	if( exitCode != 0 || exitStatus == QProcess::CrashExit )
	{
		success = false;
		title = "Error";
		msg = finishProcessCommonErrorMessages(output);
	}
	if( !success )
		cleanAfterError();

	iProcAptGetClean->close();
	communicateStatusToUi(success, title, msg);
	runNext();
}


void AAptInterface::uiUpdaterAptGetUpdate()
{
	QByteArray data = iProcAptGetUpdate->read( iProcAptGetUpdate->bytesAvailable() );
	logToFile( data, false );
	iProcAptGetUpdateOutput.append(data);

	if( !iUiDimmer )
		return;

	QStringList lines = QString( data.trimmed() ).split('\n');

	for( int i=0; i<lines.count(); i++ )
	{
		if( lines.at(i).startsWith("Get:") || lines.at(i).startsWith("Hit ") )
			iCatalogCounter++;
	}
	//iUiDimmer->updateText( QString("Updating catalogs (%1)").arg(iCatalogCounter) );
	//qDebug() << iCatalogCounter << iCatalogsTotal;
	iUiDimmer->setProgress( iCatalogCounter*100/iCatalogsTotal );
}

void AAptInterface::uiUpdaterAptGetInstall()
{
	QByteArray data = iProcAptGetInstall->read( iProcAptGetInstall->bytesAvailable() );
	logToFile( data, false );
	iProcAptGetInstallOutput.append(data);

	if( !iUiDimmer )
		return;

	QStringList lines = QString( data.trimmed() ).split('\n');

	bool update = false;
	bool resetprogress = true;
	QString oper = "";
	QString pkgname = "";
	iCanCancel = false;
	for( int i=0; i<lines.count(); i++ )
	{
		QStringList l = lines.at(i).split(' ');

		if( l.count()>=4 && l.at(0).startsWith("Get:") ) {
			oper = "Downloading";
			iCanCancel = true;
			pkgname = l.at(3);
			Package* pkg = iPackagesAvailable.value(pkgname,0);
			if( pkg!=0 ) {
				iAptGetCurrentDownloadFileName = pkg->fileName();
				iAptGetCurrentFileTotalSize = pkg->size()/1024;
				pkgname += QString(" (%1 kB)").arg(iAptGetCurrentFileTotalSize);
			}
			iAptGetDownloadCount++;
			oper += QString(" %1/%2").arg(iAptGetDownloadCount).arg(iAptGetInstallTotal);
			update = true;
			if( !iProgressCheckTimer ) {
				iProgressCheckTimer = new QTimer(this);
				connect(iProgressCheckTimer,SIGNAL(timeout()),this,SLOT(progressCheckTimerCallback()));
				iProgressCheckTimer->start(500);
			}
			resetprogress = false;
		} else if( l.count()>=2 && l.at(0)=="Unpacking") {
			oper = "Installing";
			if( l.count()>=3 && l.at(1)=="replacement" )
				pkgname = l.at(2);
			else
				pkgname = l.at(1);
			iAptGetInstallCount++;
			oper += QString(" %1/%2").arg(iAptGetInstallCount).arg(iAptGetInstallTotal);
			update = true;
		} else if( l.count()>=3 && l.at(0)=="Setting" && l.at(1)=="up") {
			oper = "Setting up";
			pkgname = l.at(2);
		} else if( l.count()>=2 && l.at(0)=="Removing") {
			oper = "Removing";
			pkgname = l.at(1);
			iAptGetRemoveCount++;
			oper += QString(" %1/%2").arg(iAptGetRemoveCount).arg(iAptGetRemoveTotal);
			update = true;
		} else if( l.count()>=1 && l.at(0)=="Done!") {
			oper = "Setting up...";
			pkgname = "";
			update = true;
		}
	}

	if( update && iUiDimmer && iUiDimmer->busy() ) {
		iUiDimmer->updateText( oper + "<br>" + pkgname );
		if( resetprogress ) {
			iUiDimmer->setProgress(-1);
			if( iProgressCheckTimer ) {
				iProgressCheckTimer->stop();
				delete iProgressCheckTimer;
				iProgressCheckTimer = 0;
			}
		}
	}


}

void AAptInterface::progressCheckTimerCallback()
{
	if( iAptGetCurrentDownloadFileName=="" )
		return;

	QFile pkgfile("/var/cache/apt/archives/partial/" + iAptGetCurrentDownloadFileName);
	iAptGetCurrentFileDownloadSize = pkgfile.size()/1024;

	if( iUiDimmer && iUiDimmer->busy() ) {
		int p = iAptGetCurrentFileDownloadSize*100/iAptGetCurrentFileTotalSize;
		if( iAptGetDownloadCount > 0 && iAptGetCurrentFileDownloadSize==0 )
			p = 100;
		iUiDimmer->setProgress( p );
	}
}


void AAptInterface::communicateStatusToUi(bool success, QString title, QString msg)
{
	qDebug() << title << msg;
	iQueueMessages.append(msg);

	if( iMainWindow && iOperationsQueue.count()==0 )
	{
		// title comes from the last finished operation only
		iMainWindow->operationQueueFinished(iModeLog, success, title, iQueueMessages);
	}
}

QByteArray AAptInterface::readLogFile()
{
	QByteArray log = "";

	QFile own(KLogFileName);
	if( own.open(QIODevice::ReadOnly | QIODevice::Text ) )
	{
		while(!own.atEnd())
		{
			QByteArray line = own.readLine();
			log.append( line );
		}
		own.close();
	}

	if( log=="" )
		log = "The log is empty";

	return log;
}

void AAptInterface::logToFile( QString data, bool logtime )
{
	logToFile( data.toAscii(), logtime );
}

void AAptInterface::logToFile( QByteArray data, bool logtime )
{
	QFile f(KLogFileName);

	if( f.open( QIODevice::Append | QIODevice::WriteOnly | QIODevice::Text ) )
	{
		QTextStream out(&f);
		if( logtime )
			out << "--- " << QDateTime::currentDateTime().toString("yyyy-MM-dd hh:mm:ss") << " ---\n";

		out << data;

		if( logtime )
			out << "\n";
		f.close();
	}
}

void AAptInterface::readRepositoryInfo()
{
	for(int i=0; i<iRepositories.count(); i++) {
		if( iRepositories.at(i) )
			delete iRepositories.at(i);
	}
	iRepositories.clear();
	bool ownFound = false;

	QFile own("/root/.fapman/repos.list");
	if( own.open(QIODevice::ReadOnly | QIODevice::Text ) )
	{
		Repository* r = 0;
		while(!own.atEnd())
		{
			QString line = own.readLine().trimmed();
			if( line.startsWith("deb ") || line.startsWith("#deb ") )
			{
				r = new Repository();
				if( r->setFromString(line) ) {
					iRepositories.append(r);
					//qDebug() << r->toListFileNames();
				} else {
					delete r;
				}
			}
		}
		own.close();
		if( iRepositories.count() > 0 )
			ownFound = true;
	}

	QFile names("/root/.fapman/repos.names");
	if( names.open(QIODevice::ReadOnly | QIODevice::Text ) )
	{
		int c=0;
		while(!names.atEnd() && c<iRepositories.count())
		{
			QString line = names.readLine().trimmed();
			if( line.trimmed()!="" )
			iRepositories.at(c)->setName( line.trimmed() );
			c++;
		}
		names.close();
		return;
	}

	if( ownFound )
		return;

	QFile ham("/etc/apt/sources.list.d/hildon-application-manager.list");
	if( ham.open(QIODevice::ReadOnly | QIODevice::Text ) )
	{
		while(!ham.atEnd())
		{
			QString line = ham.readLine();
			Repository* r = new Repository();
			if( r->setFromString(line) ) {
				iRepositories.append(r);
			} else {
				delete r;
			}
		}
		ham.close();
	}
}

bool AAptInterface::writeRepositories()
{
#ifndef Q_WS_MAEMO_5	// for simulator
	return true;
#endif

	QFile own("/root/.fapman/repos.list");
	if( own.open(QIODevice::WriteOnly | QIODevice::Text ) )
	{
		QTextStream out(&own);
		for( int i=0; i<iRepositories.count(); i++ )
			out << iRepositories.at(i)->toString() << "\n";
		own.close();
	}

	QFile names("/root/.fapman/repos.names");
	if( names.open(QIODevice::WriteOnly | QIODevice::Text ) )
	{
		QTextStream out(&names);
		for( int i=0; i<iRepositories.count(); i++ )
			out << iRepositories.at(i)->name() << "\n";
		names.close();
	}

	QFile ham("/etc/apt/sources.list.d/hildon-application-manager.list");
	if( ham.open(QIODevice::WriteOnly | QIODevice::Text ) )
	{
		QTextStream out(&ham);
		for( int i=0; i<iRepositories.count(); i++ )
			out << iRepositories.at(i)->toString() << "\n";
		ham.close();

		return true;
	} else {
		qDebug() << "Failed to write repository list to sources.d!";
	}

	return false;
}

bool AAptInterface::needRepoRefresh()
{
#ifndef Q_WS_MAEMO_5	// for simulator
	return false;
#endif

	if( iNeedRepoRefresh || iSettings->qsettings()->value("need_repo_refresh", false).toBool() ) {
		iNeedRepoRefresh = false;
		iSettings->qsettings()->setValue("need_repo_refresh", false);
		qDebug() << "repo update required, debug 1";
		return true;
	}

	QFile own("/root/.fapman/repos.list");
	QFile ham("/etc/apt/sources.list.d/hildon-application-manager.list");

	if( own.exists() && ham.exists() )
	{
		if( own.size() != ham.size() ) {
			qDebug() << "repo update required, debug 2";
			return true;
		}
	} else {
		qDebug() << "repo update required, debug 3";
		return true;
	}

	QFileInfo a("/root/.fapman/lastupdate");
	QFileInfo b("/home/user/.hildon-application-manager/last-update");
	QDateTime aDate;
	QDateTime bDate;
	aDate.setTime_t(0);
	bDate.setTime_t(0);
	if( a.exists() )
		aDate = a.lastModified();
	if( b.exists() )
		bDate = b.lastModified();
	aDate = aDate.addSecs(24*60*60); //24h
	bDate = bDate.addSecs(24*60*60); //24h

	if( aDate < QDateTime::currentDateTime() && bDate < QDateTime::currentDateTime() ) {
		qDebug() << "repo update required, debug 4";
		return true;
	}

	qDebug() << "repo update not required";
	return false;
}



void AAptInterface::startPkgListRead()
{
	logToFile( QString("Start reading package lists") );
	qDebug() << "reading package list files";

	if( iUiDimmer ) {
		iUiDimmer->updateText("Reading package lists<br>");
		iUiDimmer->setProgress(0);
	}

// clear packages lists
	QHashIterator<QString, Package*> a( iPackagesAvailable );
	while (a.hasNext())
	{
		a.next();
		delete a.value();
	}
	iPackagesAvailable.clear();

	QHashIterator<QString, Package*> i( iPackagesInstalled );
	while (i.hasNext())
	{
		i.next();
		delete i.value();
	}
	iPackagesInstalled.clear();

	iCanCancel = true;

// read apt database (available packages)
	int pkgcount_apt = 0;
	QDir dir("/var/lib/apt/lists/");
	QFileInfoList files = dir.entryInfoList();

	quint64 totaldatasize = 0;
	quint64 currentreaddata = 0;
	quint64 lastupdatedata = 0;
	quint64 updateinterval = 1000000;
	for( int i=0; i<files.count(); i++ )
	{
		if( files.at(i).fileName().endsWith("_Packages"))
			totaldatasize += files.at(i).size();
	}
	QFileInfo dbinfo("/var/lib/dpkg/status");
	totaldatasize += dbinfo.size();
	int filecount = 0;

	for( int i=0; i<files.count(); i++ )
	{
		Repository* currentRepo = 0;
		if( files.at(i).absoluteFilePath().endsWith("_Packages") )
		{
			filecount++;
			for(int x=0; x<iRepositories.count(); x++) {
				if( iRepositories.at(x)->toListFileNames().contains( files.at(i).fileName() ) ) {
					currentRepo = iRepositories.at(x);
				}
			}

			if( iUiDimmer && currentRepo ) {
				iUiDimmer->updateText("Reading package lists<br><font size=\"-1\">" + currentRepo->name() + "</font>");
			}

			//qDebug() << files.at(i).fileName();

			QFile db( files.at(i).absoluteFilePath() );
			if (!db.open(QIODevice::ReadOnly | QIODevice::Text)) {
				qDebug() << "FAIL: Unable to read apt database";
				communicateStatusToUi(false, "Error", "Unable to read package lists");
				cleanAfterError();
				return;
			}

			while (!db.atEnd() && !iTerminated) {
				Package* newpkg = ReadNextPackage(db, currentreaddata);
				//qDebug() << "read" << currentreaddata << "of" << totaldatasize;
				if( iUiDimmer && currentreaddata >= lastupdatedata+updateinterval ) {
					iUiDimmer->setProgress( currentreaddata*100/totaldatasize );
					lastupdatedata = currentreaddata;
				}
				QApplication::processEvents();
				pkgcount_apt++;
				if( newpkg )
				{
					newpkg->addRepository( currentRepo );

					Package* exists = iPackagesAvailable.value(newpkg->name(),0);
					if( !exists ) {
						iPackagesAvailable.insert(newpkg->name(), newpkg);
					} else {
						if( Package::versionCompare(newpkg->version(),exists->version()) )
						{
							iPackagesAvailable.remove(exists->name());
							delete exists;
							exists=0;
							iPackagesAvailable.insert(newpkg->name(), newpkg);
						} else {
							if( newpkg->version() == exists->version() )
								exists->addRepository( currentRepo );
							delete newpkg;
							newpkg=0;
						}
					}					
				}
			}
			db.close();
		}
	}

	qDebug() << "Processed" << filecount << "package list files";

	if( iTerminated ) {
		if( iUiDimmer )
			iUiDimmer->setProgress(-1);
		cleanAfterError();
		communicateStatusToUi(false, "Operation cancelled", "The operation was cancelled by user");		
		return;
	}

// read dpkg database (installed packages)
	int pkgcount_dpkg = 0;
	QFile db("/var/lib/dpkg/status");
	if (!db.open(QIODevice::ReadOnly | QIODevice::Text)) {
		qDebug() << "FAIL: Unable to read dpkg database";
		communicateStatusToUi(false, "Error", "Unable to read package database");
		cleanAfterError();
		return;
	}

	if( iUiDimmer ) {
		iUiDimmer->updateText("Reading package lists<br><font size=\"-1\">dpkg database</font>");
	}

	while (!db.atEnd() && !iTerminated) {
		Package* newpkg = ReadNextPackage(db, currentreaddata);
		if( iUiDimmer && currentreaddata >= lastupdatedata+updateinterval ) {
			iUiDimmer->setProgress( currentreaddata*100/totaldatasize );
			lastupdatedata = currentreaddata;
		}
		QApplication::processEvents();
		pkgcount_dpkg++;
		if( newpkg ) {
			if( newpkg->isInstalled() ) {
				iPackagesInstalled.insert(newpkg->name(), newpkg);
			} else {
				delete newpkg;
				newpkg=0;
			}
		}
	}
	db.close();

	qDebug() << "Processed" << pkgcount_apt << "(apt) and" << pkgcount_dpkg << "(dpkg) package entries";
	qDebug() << "In DB:" << iPackagesAvailable.count() << "packages available,"
			<< iPackagesInstalled.count() << "installed";

	logToFile( QString("Finished reading package lists") );
	if( iUiDimmer ) {
		iUiDimmer->updateText("Reading package lists<br><font size=\"-1\">Creating package view</font>");
		QApplication::processEvents();
		iUiDimmer->setProgress(-1);
	}

	if( iTerminated ) {
		cleanAfterError();
		communicateStatusToUi(false, "Operation cancelled", "The operation was cancelled by user");
		return;
	}

	readBlacklist();

	communicateStatusToUi(true, "Operation finished", "Package data read");
	iCanCancel = false;
	runNext();
}

Package* AAptInterface::ReadNextPackage(QFile& f, quint64& currentreaddata)
{
	iMultiLine=MultiLineNone;

	Package* pkg = new Package("", this);

	bool pkgready = false;
	while( !pkgready && !f.atEnd() ) {
		QByteArray line = f.readLine();
		currentreaddata += line.size();
		if( processPackageDataLine(pkg,line) ) {
			pkgready = true;
		}
	}

	if( pkg->name() != "" && pkg->isInstalled() ) {
		QFileInfo f( "/var/lib/dpkg/info/" + pkg->name() + ".list" );
		if( f.exists() )
			pkg->setDate( f.lastModified() );
	}

	pkg->updateStatus();

	if( pkg->name() == "" ) {
		delete pkg;
		pkg = 0;
	}
	return pkg;
}

bool AAptInterface::processPackageDataLine(Package*& pkg, QByteArray& line)
{
	if( !line.startsWith(' ') && !line.startsWith('\t') )
		line = line.trimmed();
	if( line.trimmed()=="" )
		return true;

	if( line.startsWith("Package: ") )
	{
		pkg->setName( line.mid(9) );
		iMultiLine=MultiLineNone;
	}
	else if( line.startsWith("Status: ") )
	{
		if( line.mid(8) == "install ok installed" )
			pkg->setInstalled(true);
		else
			pkg->setInstalled(false);
	}
	else if( line.startsWith("Section: ") )
	{
		pkg->setSection( line.mid(9) );
	}
	else if( line.startsWith("Version: ") )
	{
		pkg->setVersion( line.mid(9) );
	}
	else if( line.startsWith("Filename: ") )
	{
		pkg->setFileNameFull( line.mid(10) );
	}
	else if( line.startsWith("Size: ") )
	{
		pkg->setSize( line.mid(6).toInt() );
	}
	else if( line.startsWith("Installed-Size: ") )
	{
		pkg->setInstalledSize( line.mid(16).toInt() );
	}
	else if( line.startsWith("Maemo-Display-Name: ") )
	{
		pkg->setMaemoDisplayName( QString::fromUtf8(line.mid(20)) );
	}

	if( iMultiLine == MultiLineDesc ) {
		if( (line.startsWith(' ') || line.startsWith('\t')) && line.trimmed()!="" ) {
			if( line.trimmed()!="." )
				pkg->appendDescLong( QString::fromUtf8(line.trimmed()) + "\n" );
			else
				pkg->appendDescLong( "\n" );
		} else {
			iMultiLine = MultiLineNone;
		}
	}
	else if( iMultiLine == MultiLineIcon ) {
		if( (line.startsWith(' ') || line.startsWith('\t')) && line.trimmed()!="" ) {
			pkg->appendIconData( line.trimmed() );
		} else {
			iMultiLine = MultiLineNone;
		}
	}

	if( line.startsWith("Description: ") )
	{
		pkg->setDescShort( QString::fromUtf8(line.mid(13).trimmed()) );
		iMultiLine = MultiLineDesc;
	}
	else if( line.startsWith("Maemo-Icon-26:") )
	{
		if( line.mid(15).trimmed() != "" ) {
			pkg->appendIconData( line.mid(15).trimmed() );
		}
		iMultiLine = MultiLineIcon;
	}

	return false;
}



void AAptInterface::writeBlacklist()
{
	QHashIterator<QString, Package*> i( iPackagesAvailable );
	while (i.hasNext())
	{
		i.next();

		if( i.value()->blacklisted() == BlacklistSelect::BlacklistAll ) {
			iBlacklist << i.value()->name();
		}
		else if( i.value()->blacklisted() == BlacklistSelect::BlacklistThis ) {
			iBlacklist << (i.value()->name() + " " + i.value()->version());
		}
	}

	iBlacklist.removeDuplicates();

	QFile f("/root/.fapman/black.list");
	if( f.open(QIODevice::WriteOnly | QIODevice::Text ) )
	{
		QTextStream out(&f);
		for( int i=0; i<iBlacklist.count(); i++ )
			out << iBlacklist.at(i) << "\n";
		f.close();
	}

	qDebug() << "blacklist: wrote" << iBlacklist.count() << "entries";
}

void AAptInterface::readBlacklist()
{
	QFile f("/root/.fapman/black.list");
	if( f.open(QIODevice::ReadOnly | QIODevice::Text ) )
	{
		while( !f.atEnd() ) {
			QString line = f.readLine().trimmed();
			if( line != "" )
				iBlacklist.append(line);
		}
		f.close();
	}

	iBlacklist.removeDuplicates();

	qDebug() << "blacklist: read" << iBlacklist.count() << "entries";

	for( int i=0; i<iBlacklist.count(); i++ )
	{
		QStringList parts = iBlacklist.at(i).split(' ');
		if( parts.count()==1 ) {
			Package* pkg = iPackagesAvailable.value(parts.at(0).trimmed(), 0);
			if( pkg ) {
				pkg->setBlacklisted(BlacklistSelect::BlacklistAll);
			}
		} else if( parts.count()==2 ) {
			Package* pkg = iPackagesAvailable.value(parts.at(0).trimmed(), 0);
			if( pkg && pkg->version()==parts.at(1) ) {
				pkg->setBlacklisted(BlacklistSelect::BlacklistThis);
			}
		} else {
			qDebug() << "Warning: invalid blacklist entry:" << iBlacklist.at(i);
		}
	}
}

void AAptInterface::removeFromBlacklist(Package *pkg, BlacklistSelect::blackList oldstate)
{
	if( !pkg ) {
		qDebug() << "Warning: trying to remove null package from blacklist";
		return;
	}

	QStringList newlist;
	bool removed = false;

	for( int i=0; i<iBlacklist.count(); i++ )
	{
		if( oldstate == BlacklistSelect::BlacklistAll )
		{
			if( !(iBlacklist.at(i) == pkg->name()) ) {
				newlist << iBlacklist.at(i);
			} else removed = true;
		} else if( oldstate == BlacklistSelect::BlacklistThis ) {
			if( !(iBlacklist.at(i) == (pkg->name()+" "+pkg->version())) ) {
				newlist << iBlacklist.at(i);
			} else removed = true;
		}
	}

	if( removed )
		qDebug() << "blacklist: removed" << pkg->name();
	else
		qDebug() << "blacklist:" << pkg->name() << "not in saved list";

	iBlacklist = newlist;
}


void AAptInterface::startFetchDates()
{
	logToFile( QString("Start fetching package dates") );
	qDebug() << "start fetching package dates";

	if( iUiDimmer ) {
		iUiDimmer->updateText("Reading date cache");
		iUiDimmer->setProgress(0);
		QApplication::processEvents();
	}

	readDateCache();

	if( iUiDimmer ) {
		iUiDimmer->updateText("Fetching package date information");
		QApplication::processEvents();
	}

	QNetworkAccessManager* nam = new QNetworkAccessManager(this);
	iCanCancel = true;

	int count = 0;
	int updProgress = 0;

	QHash<QString, Package*> fetchable;
	QHashIterator<QString, Package*> i( iPackagesAvailable );
	while (i.hasNext() )
	{
		i.next();
		if( !i.value()->date().isValid() && i.value()->section().startsWith("user/") && !i.value()->isBlacklisted() )
		{
			Repository* repo = 0;
			for( int x=0; x<i.value()->repositories().count(); x++ )
				if( i.value()->repositories().at(x) && i.value()->repositories().at(x)->url().startsWith("http://repository.maemo.org") )
				{
				repo = i.value()->repositories().at(x);
				break;
			}
			if( repo ) {
				fetchable.insert(i.value()->name(), i.value());
			}
		}
	}

	connect(nam, SIGNAL(finished(QNetworkReply*)), this, SLOT(dateFetchNetworkReply(QNetworkReply*)));
	iDateRequestsWaiting = 0;
	iDateRequestsSent = 0;
	iDateRequestsReceived = 0;
	iNetworkError = QNetworkReply::NoError;

	qDebug() << "package dates to fetch:" << fetchable.count();

	QHashIterator<QString, Package*> fe( fetchable );
	while (fe.hasNext() && !iTerminated)
	{
		fe.next();

		if( updProgress >=10 ) {
			iUiDimmer->setProgress( count*100/fetchable.count() );
			updProgress=0;
		}

		if( !fe.value()->date().isValid() && fe.value()->section().startsWith("user/") )
		{
			QString url;
			Repository* repo = 0;
			for( int x=0; x<fe.value()->repositories().count(); x++ )
				if( fe.value()->repositories().at(x) && fe.value()->repositories().at(x)->url().startsWith("http://repository.maemo.org") )
				{
				repo = fe.value()->repositories().at(x);
				break;
			}

			if( repo )
			{
				url = repo->url() + repo->dir() + fe.value()->fileNameFull();

				//qDebug() << "getting date for" << fe.value()->name();
				//qDebug() << url;

				QUrl u(url);
				QNetworkRequest r(u);

				if( iDateRequestsReceived == 0 ) {
					while( iDateRequestsWaiting>0 ) {
						QApplication::processEvents(QEventLoop::WaitForMoreEvents);
					}
				} else {
					while( iDateRequestsWaiting>50 ) {
						QApplication::processEvents(QEventLoop::WaitForMoreEvents);
					}
				}

				if( iDateRequestsReceived>0 && iNetworkError != QNetworkReply::NoError ) {
					qDebug() << "network error" << iNetworkError;
					break;
				} else {
					nam->head(r);
					iDateRequestsSent++;
					iDateRequestsWaiting++;
				}
			}
			count = iDateRequestsReceived;
			updProgress++;
		}
	}
	while( iDateRequestsWaiting>0 ) {
		if( updProgress >=10 ) {
			iUiDimmer->setProgress( count*100/fetchable.count() );
			updProgress=0;
		}
		QApplication::processEvents(QEventLoop::WaitForMoreEvents);
		count = iDateRequestsReceived;
		updProgress++;
	}
	delete nam;

	if( iNetworkError != QNetworkReply::NoError )
	{
		// don't stop on this error, only inform the user
		iMainWindow->notifyDialog("Network error", "There was a network error while fetching date information");
	}

	if( iTerminated ) {
		if( iUiDimmer )
			iUiDimmer->setProgress(-1);
		cleanAfterError();
		communicateStatusToUi(false, "Operation cancelled", "The operation was cancelled by user");
		return;
	}
	iCanCancel = false;

	if( iUiDimmer ) {
		iUiDimmer->setProgress(100);
		QApplication::processEvents();
	}

	qDebug() << "sent" << iDateRequestsSent << "requests, received" << iDateRequestsReceived << "replies with"
			<< iDateRequestErrors << "errors";

	logToFile( QString("Finished fetching package dates") );

	if( fetchable.count()>0 ) {
		iUiDimmer->updateText("Writing date cache");
		QApplication::processEvents();
		writeDateCache();
	}

	if( iUiDimmer ) {
		iUiDimmer->updateText("Creating package view");
		QApplication::processEvents();
		iUiDimmer->setProgress(-1);
	}

	communicateStatusToUi(true, "Operation finished", "Package dates fetched");
	runNext();
}

void AAptInterface::dateFetchNetworkReply(QNetworkReply* reply)
{
	iDateRequestsWaiting--;
	iDateRequestsReceived++;
	//qDebug() << "reply" << reply->header(QNetworkRequest::LastModifiedHeader).toDateTime();
	iNetworkError = reply->error();

	if( reply->header(QNetworkRequest::LastModifiedHeader).toDateTime().isValid() &&
		iNetworkError == QNetworkReply::NoError)
	{
		QString path = reply->url().path();
		int pos = path.lastIndexOf('/');
		int pos2 = path.indexOf('_', pos);
		if( pos!=-1 && pos2!=-1 ) {
			QString pkgname = path.mid(pos+1,pos2-pos-1);
			Package* pkg = iPackagesAvailable.value(pkgname, 0);
			if( pkg ) {
				pkg->setDate( reply->header(QNetworkRequest::LastModifiedHeader).toDateTime() );
			}
		}
	}

	if( iNetworkError != QNetworkReply::NoError )
		iDateRequestErrors++;
	reply->deleteLater();
}

void AAptInterface::writeDateCache()
{
	qDebug() << "writing date cache";

	QFile f("/root/.fapman/dates.cache");
	if( f.open(QIODevice::WriteOnly | QIODevice::Text ) )
	{
		QTextStream out(&f);
		QHashIterator<QString, Package*> i( iPackagesAvailable );
		while (i.hasNext() )
		{
			i.next();
			if( i.value()->date().isValid() && i.value()->section().startsWith("user/") )
				out << i.value()->name() << " " << i.value()->version() << " " << i.value()->date().toString(Qt::ISODate) << "\n";
		}
		f.close();
	} else {
		qDebug() << "Warning: failed to write date cache";
	}
}

void AAptInterface::readDateCache()
{
	qDebug() << "reading date cache";

	QFile f("/root/.fapman/dates.cache");
	if( f.open(QIODevice::ReadOnly | QIODevice::Text ) )
	{
		while( !f.atEnd() ) {
			QString line = f.readLine().trimmed();
			QStringList parts = line.split(' ');
			if( parts.count()==3 ) {
				Package* pkg = iPackagesAvailable.value(parts.at(0),0);
				if( pkg && pkg->section().startsWith("user/") && pkg->version()==parts.at(1) )
				{
					QDateTime dt = QDateTime::fromString( parts.at(2), Qt::ISODate);
					if( dt.isValid() ) {
						pkg->setDate( dt );
					} else {
						qDebug() << "Warning: Invalid date in date cache";
					}
				}
			} else {
				qDebug() << "Warning: error in date cache:" << line;
			}
		}
		f.close();
	} else {
		qDebug() << "date cache does not exist";
	}
}

bool AAptInterface::dateCacheExists()
{
	QFileInfo f("/root/.fapman/dates.cache");

	return f.exists();
}
