/*
	This file is part of Faster Application Manager.

	Faster Application Manager is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Faster Application Manager is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Faster Application Manager.  If not, see <http://www.gnu.org/licenses/>.

	(C) Heikki Holstila 2010
*/

#include <iostream>
#include "packageview.h"
#include "ui_packageview.h"
#include "package.h"
#include "filterselect.h"
#include "confirmdialog.h"
#include "dimmer.h"
#include "packageselector.h"
#include "help.h"
#include "aaptinterface.h"
#include "logview.h"

PackageListWidgetItem::PackageListWidgetItem(Package* p_, QString name_) : QListWidgetItem(name_)
{
	iPackage = p_;
}


void ListItemDelegate::paint(QPainter *painter, const QStyleOptionViewItem &option, const QModelIndex &index) const
{
	QString name = index.data(UserRoleName).toString();
	QString version = index.data(UserRoleVersion).toString();
	QString desc = index.data(UserRoleDescShort).toString();
	Package::operation marked = (Package::operation) index.data(UserRoleMarked).toInt();
	bool installed = index.data(UserRoleInstalled).toBool();
	bool upgradeable = index.data(UserRoleUpgradeable).toBool();
	int statfilter = index.data(UserRoleCurrentStatFilter).toInt();
	QString upg_version = index.data(UserRoleAvailVersion).toString();

	painter->save();
	QRect r = option.rect;

	QLinearGradient gradientBase(r.topLeft(), r.bottomLeft());
	QColor base = option.palette.color(QPalette::Window);
	QColor base2 = base;
	base.setRgb( base.red()+15, base.green()+15, base.blue()+15 );
	base2.setRgb( base2.red()-20, base2.green()-20, base2.blue()-20 );
	gradientBase.setColorAt(0, base);
	gradientBase.setColorAt(1, base2);

	painter->fillRect(r, gradientBase);

	painter->drawLine(QPoint(r.left(),r.bottom()), QPoint(r.right(),r.bottom()));

	QPixmap icon = index.data(Qt::DecorationRole).value<QPixmap>();
	if( icon.isNull() ) {
		// use default icon
		painter->drawPixmap( r.left(), r.top()+4, 48, 48, iDefaultIcon );
	} else {
		painter->drawPixmap( r.left(), r.top()+4, 48, 48, icon );
	}

	QPixmap statusicon;
	if( marked == Package::PkgOpNone )
	{
		if( installed && upgradeable )
			statusicon = iIconPkgNoOpInstalledUpgradeable;
		else if( installed )
			statusicon = iIconPkgNoOpInstalled;
		else if( !installed )
			statusicon = iIconPkgNoOpNotInstalled;
	} else if( marked == Package::PkgOpInstallUpgrade ) {
		if( upgradeable )
			statusicon = iIconPkgUpgrade;
		else
			statusicon = iIconPkgInstall;
	} else if( marked == Package::PkgOpRemove ) {
		statusicon = iIconPkgRemove;
	}

	QString showVer = "";
	if( upgradeable && statfilter==Package::PkgStatUpgradeable )
		showVer = upg_version;
	else
		showVer = version;

	int ver_w = 0;
	if( QApplication::desktop()->width() > QApplication::desktop()->height() )
	{
		r = option.rect;
		r.setRight( r.right()-statusicon.width()-4 );
		painter->drawText(r, Qt::AlignTop|Qt::AlignRight, showVer, &r);
		ver_w = r.width();
	}

	r = option.rect;
	r.setRight( r.right()-statusicon.width()-4-ver_w );  //does not work as it should?
	QFont f = painter->font();
	f.setBold(true);
	painter->setFont(f);
	painter->drawText(r.left()+iDefaultIcon.width()+2, r.top(), r.width(), r.height(), Qt::AlignTop|Qt::AlignLeft, name, &r);
	f.setBold(false);
	painter->setFont(f);

	f.setPointSize( f.pointSize()-4 );
	painter->setFont(f);
	r = option.rect;
	painter->drawText(r.left()+iDefaultIcon.width()+2, r.top(), r.width(), r.height(), Qt::AlignBottom|Qt::AlignLeft, desc, &r);

	r = option.rect;
	painter->drawPixmap(r.right()-statusicon.width()-2, r.top()+4, 24, 24, statusicon);

	painter->restore();
}

void ListItemDelegate::loadIcons()
{
	iDefaultIcon = QPixmap(":/icons/icons/appdefault.png");
	iIconPkgInstall = QPixmap(":/icons/icons/pkg_install.png");
	iIconPkgUpgrade = QPixmap(":/icons/icons/pkg_upgrade.png");
	iIconPkgRemove = QPixmap(":/icons/icons/pkg_remove.png");
	iIconPkgNoOpInstalled = QPixmap(":/icons/icons/pkg_nop_installed.png");
	iIconPkgNoOpNotInstalled = QPixmap(":/icons/icons/pkg_nop_notinstalled.png");
	iIconPkgNoOpInstalledUpgradeable = QPixmap(":/icons/icons/pkg_nop_instupgr.png");
}


QSize ListItemDelegate::sizeHint(const QStyleOptionViewItem&, const QModelIndex&) const
{
	return QSize(400, 60);
}


PackageView::PackageView(QWidget *parent) : QMainWindow(parent), ui(new Ui::PackageView)
{
	iMainWindow = dynamic_cast<MainWindow*>(parent);
	ui->setupUi(this);
#ifdef Q_WS_MAEMO_5
	this->setAttribute(Qt::WA_Maemo5StackedWindow);
	this->setWindowFlags(Qt::Window);
	this->setAttribute(Qt::WA_Maemo5AutoOrientation);
#endif

	connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(orientationChanged()));

	iCatFilterLabels
			<< tr("All marked packages")	// a special case
			<< tr("All user categories")
			<< tr("\tDesktop")
			<< tr("\tEducation")
			<< tr("\tGames")
			<< tr("\tGraphics")
			<< tr("\tInternet & Networking")
			<< tr("\tLocation & Navigation")
			<< tr("\tMultimedia")
			<< tr("\tOffice")
			<< tr("\tOther")
			<< tr("\tProgramming")
			<< tr("\tScience")
			<< tr("\tSystem")
			<< tr("\tUtilities")
			<< tr("All packages (ADVANCED)");

	iCatFilterStrings
			<< ""
			<< "user/"
			<< "user/desktop"
			<< "user/education"
			<< "user/games"
			<< "user/graphics"
			<< "user/network"
			<< "user/navigation"
			<< "user/multimedia"
			<< "user/office"
			<< "user/other"
			<< "user/development"
			<< "user/science"
			<< "user/system"
			<< "user/utilities"
			<< "";

	iDefaultCatFilter = CatFilterAllUser;
	iSelectedCatFilter = iDefaultCatFilter;

	iStatFilterLabels
			<< tr("All")
			<< tr("Not installed")
			<< tr("Upgradeable")
			<< tr("Installed");

	iSelectedStatFilter = Package::PkgStatUnknown;
	iSortOrder = SortAlpha;

	iDimmer = new dimmer(this);

	ui->searchBar->hide();

	iListCoverLabel = new QLabel(ui->listWidget);
	iListCoverLabel->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
	iListCoverLabel->setAlignment(Qt::AlignCenter);

	iKeyFilter = new KeyEventGrabber(this);
	ui->listWidget->installEventFilter(iKeyFilter);
}

PackageView::~PackageView()
{
	delete iListCoverLabel;
	delete iKeyFilter;
	delete iDimmer;
    delete ui;
}

void PackageView::orientationChanged()
{
	ui->listWidget->scroll(1,1);	// this causes all items to be repainted
	iListCoverLabel->setGeometry( ui->listWidget->rect() );
}

void PackageView::resizeEvent(QResizeEvent* event)
{
	iListCoverLabel->setGeometry( ui->listWidget->rect() );
	QWidget::resizeEvent(event);
}

bool PackageView::doFilterCategory(Package* pkg)
{
	if( pkg->section()=="user/hidden" )
		return false;

	if( iSelectedCatFilter==CatFilterAllMarked ) {
		if( pkg->isMarkedForOperation() )
			return true;
		else
			return false;
	}
	if( pkg->section().startsWith( iCatFilterStrings.at(iSelectedCatFilter) ) )
		return true;

	return false;
}

QString PackageView::generateSortString(Package* pkg)
{
	// default sort order is alphabetical by name
	QString sortstr = pkg->displayName();

	if( iSortOrder==SortDateDesc )
		sortstr = pkg->date().toString("yyyy-MM-dd hh:mm");
	if( iSortOrder==SortSizeDesc ) {
		if( pkg->isInstalled() )
			sortstr = QString("%1").arg(pkg->installedSize()*1024, 12 );
		else
			sortstr = QString("%1").arg(pkg->size(), 12 );
	}

	//qDebug() << sortstr;

	return sortstr;
}

void PackageView::openWin()
{
	ui->listWidget->clear();
	ui->listWidget->setSortingEnabled(true);

	if( iSortOrder==SortDateDesc || iSortOrder==SortSizeDesc )
		ui->listWidget->sortItems(Qt::DescendingOrder);
	else
		ui->listWidget->sortItems(Qt::AscendingOrder);
	ui->btn_Sort->hide();

	delete ui->listWidget->itemDelegate();
	ListItemDelegate* delegate = new ListItemDelegate(ui->listWidget);
	delegate->loadIcons();
	ui->listWidget->setItemDelegate( delegate );

	if( !ui->searchBar->isVisible() )
	{
		if( iSelectedStatFilter == Package::PkgStatNotInstalled || iSelectedStatFilter == Package::PkgStatUnknown ||
			iSelectedCatFilter == CatFilterAllMarked )
		{
			QHashIterator<QString, Package*> i( *iAptInterface->packagesAvailable() );
			while (i.hasNext())
			{
				i.next();
				Package* inst = iAptInterface->packagesInstalled()->value(i.value()->name(),0);
				if( doFilterCategory(i.value()) && !inst )
					addListItem(i.value(), generateSortString(i.value()));
			}
		}
		if( iSelectedStatFilter == Package::PkgStatInstalled || iSelectedStatFilter == Package::PkgStatUpgradeable ||
			iSelectedStatFilter == Package::PkgStatUnknown || iSelectedCatFilter == CatFilterAllMarked )
		{
			QHashIterator<QString, Package*> i( *iAptInterface->packagesInstalled() );
			while (i.hasNext())
			{
				i.next();
				if( iSelectedStatFilter == Package::PkgStatUpgradeable ) {
					if( i.value()->isUpgradeable() && doFilterCategory(i.value()) )
						addListItem(i.value(), generateSortString(i.value()));
				} else {
					if( doFilterCategory(i.value()) )
						addListItem(i.value(), generateSortString(i.value()));
				}
			}
		}
	} else {
		for( int j=0; j<iSearchResults.count(); j++ )
		{
			addListItem(iSearchResults.at(j), generateSortString( iSearchResults.at(j) ));
		}
	}

	iListCoverLabel->hide();
	iListCoverLabel->setAutoFillBackground(false);

	updateLabel();

	if( iSelectedStatFilter==Package::PkgStatUpgradeable && ui->listWidget->count()>0 )
		ui->actionUpgrade_all->setVisible(true);
	else
		ui->actionUpgrade_all->setVisible(false);

	show();

	if( !ui->searchBar->isVisible() ) {
		ui->listWidget->setFocusPolicy(Qt::StrongFocus);
		ui->listWidget->setFocus();
	} else {
		ui->listWidget->setFocusPolicy(Qt::NoFocus);
		ui->lineEdit->setFocus();
	}

	if( ui->listWidget->count() == 0 )
	{
		iListCoverLabel->setGeometry( ui->listWidget->rect() );
		iListCoverLabel->setText("No packages");
		iListCoverLabel->show();
	}
}

void PackageView::enableMenu()
{
	ui->menuMenu->setEnabled(true);
}

void PackageView::disableMenu()
{
	ui->menuMenu->setEnabled(false);
}

void PackageView::addListItem(Package* pkg_, QString listname_)
{
	PackageListWidgetItem* p = new PackageListWidgetItem( pkg_, listname_ );

	if( pkg_ != 0 )
	{
		QString name = pkg_->name();
		if( pkg_->maemoDisplayName()!="" )
			name = pkg_->maemoDisplayName();
		p->setData(UserRoleName, name);
	} else {
		p->setData(UserRoleName, listname_);
	}

	if( pkg_ != 0 )
	{
		p->setData(UserRoleDescShort, pkg_->descShort());
		p->setData(UserRoleVersion, pkg_->version());
		p->setData(UserRoleMarked, (int)pkg_->markedOperation());
		p->setData(UserRoleInstalled, pkg_->isInstalled());
		p->setData(UserRoleUpgradeable, pkg_->isUpgradeable());
		p->setData(UserRoleAvailVersion, pkg_->upgradeableVersion());
		p->setData(UserRoleCurrentStatFilter, (int)iSelectedStatFilter);

		//qDebug()<<pkg_->name();

		pkg_->convertIcon();
		p->setData(Qt::DecorationRole, *pkg_->icon());
	}
	ui->listWidget->addItem( p );
}

void PackageView::closeEvent(QCloseEvent *event)
{
	if( !iAptInterface ) {
		resetWindow();
		event->accept();
	}

	if( iDimmer->busy() )
	{
		iAptInterface->cancel();
		event->ignore();
		return;
	}

#ifdef Q_WS_MAEMO_5
	if( iAptInterface->numSelectedPackages() == 0 )
	{		
		resetWindow();
		event->accept();
	} else {
		QString c;
		c.setNum( iAptInterface->numSelectedPackages() );
		ConfirmDialog d(true, this);
		d.setText("Returning to main menu", QString("Clear %1 package selection(s) and lose all the pending changes?").arg(iAptInterface->numSelectedPackages()));
		if( d.exec() ) {
			resetWindow();
			QWidget::closeEvent(event);
			event->accept();
		} else {
			event->ignore();
		}
	}
#else	// for simulator
	resetWindow();
	event->accept();
#endif
}

void PackageView::changeEvent(QEvent *e)
{
    QMainWindow::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void PackageView::on_btn_Commit_clicked()
{
	QStringList pkgnames;

	QHashIterator<QString, Package*> i( *iAptInterface->packagesAvailable() );
	while (i.hasNext())
	{
		i.next();

		if( i.value()->markedOperation() == Package::PkgOpInstallUpgrade )
			pkgnames << i.value()->name();
		if( i.value()->markedOperation() == Package::PkgOpRemove ) {
			qDebug() << "warning: trying to add package marked from the wrong list";
			//pkgnames << i.value()->name() + "-";
		}
	}

	QHashIterator<QString, Package*> r( *iAptInterface->packagesInstalled() );
	while (r.hasNext())
	{
		r.next();

		if( r.value()->markedOperation() == Package::PkgOpInstallUpgrade )
			pkgnames << r.value()->name();
		if( r.value()->markedOperation() == Package::PkgOpRemove )
			pkgnames << r.value()->name() + "-";
	}

	iMainWindow->busyDialog(true, "Operation in progress", "Reading dependencies");

	iAptInterface->setProcessPackages(pkgnames);
	iAptInterface->addQueuedOperation(AAptInterface::ModeAptGetSimulate);
	iAptInterface->run(iDimmer);
}

void PackageView::on_actionClear_selections_triggered()
{
	QString c;
	c.setNum( iAptInterface->numSelectedPackages() );
	ConfirmDialog d(true, this);
	d.setText(tr("Confirmation"), tr("Clear ") + c + tr(" package selection(s) and lose all the pending changes?"));
	if( d.exec() )
	{
		clearSelections();
		openWin();
	}
}

void PackageView::clearSelections()
{
	QHashIterator<QString, Package*> i( *iAptInterface->packagesInstalled() );
	while (i.hasNext())
	{
		i.next();
		i.value()->setMarkedForOperation(Package::PkgOpNone);
	}
	QHashIterator<QString, Package*> a( *iAptInterface->packagesAvailable() );
	while (a.hasNext())
	{
		a.next();
		a.value()->setMarkedForOperation(Package::PkgOpNone);
	}
	iAptInterface->setNumSelectedPackages(0);
}

void PackageView::on_listWidget_itemClicked(QListWidgetItem* item)
{
	PackageSelector s(dynamic_cast<PackageListWidgetItem*>(item)->package(), this);
	s.exec();

	dynamic_cast<PackageListWidgetItem*>(item)->package()->setMarkedForOperation( s.selectedOperation() );
	item->setData(UserRoleMarked, (int)s.selectedOperation());

	updateLabel();
}

void PackageView::updateLabel()
{
	QString s;
	s.setNum( iAptInterface->numSelectedPackages() );
	QString s2;
	s2.setNum( ui->listWidget->count() );
	QString statlabel = iStatFilterLabels.at(iSelectedStatFilter);
	if( iSelectedCatFilter == CatFilterAllMarked )
		statlabel = "All";
	ui->label->setText("<font size=\"-2\"><b>" + s + "</b> package(s) marked<br>"
					   + "Showing: <b>" + statlabel + "</b><br>"
					   + "Filter: " + iCatFilterLabels.at(iSelectedCatFilter) + " - " + s2 + " package(s)</font>");

	if( iAptInterface->numSelectedPackages()==0 ) {
		ui->btn_Commit->setEnabled(false);
		ui->actionClear_selections->setVisible(false);
	} else {
		ui->btn_Commit->setEnabled(true);
		ui->actionClear_selections->setVisible(true);
	}
}

void PackageView::on_btn_CategoryFilter_clicked()
{
	FilterSelect f("Category filter", this);
	f.setList( iCatFilterLabels, iSelectedCatFilter );

	bool s = f.exec();

	if( s )
		iSelectedCatFilter = f.selection();

	if( iSelectedCatFilter == CatFilterAllMarked ) {
		ui->btn_StatusFilter->setEnabled(false);
	} else {
		ui->btn_StatusFilter->setEnabled(true);
	}

	if( s ) {
		iListCoverLabel->setText("Loading...");
		iListCoverLabel->setAutoFillBackground(true);
		iListCoverLabel->show();
		QApplication::processEvents();

		openWin();
	}
}

void PackageView::setStatFilter(Package::packageStatus f_)
{
	iSelectedStatFilter = f_;
}

void PackageView::on_btn_StatusFilter_clicked()
{
	FilterSelect f("Status filter", this);
	f.setList( iStatFilterLabels, iSelectedStatFilter );

	bool s = f.exec();

	if( s ) {
		iSelectedStatFilter = (Package::packageStatus)f.selection();

		iListCoverLabel->setText("Loading...");
		iListCoverLabel->setAutoFillBackground(true);
		iListCoverLabel->show();
		QApplication::processEvents();

		openWin();
	}
}

void PackageView::resetWindow()
{
	iSelectedCatFilter = iDefaultCatFilter;
	ui->btn_StatusFilter->setEnabled(true);

	clearSelections();
	//if( iMainWindow )
	//	iMainWindow->setOperation(MainWindow::OpNone);

	clearSearch();
}

void PackageView::on_actionHelp_triggered()
{
	Help h(this);
	h.exec();
}

void PackageView::on_btn_searchClose_clicked()
{
	clearSearch();

	iListCoverLabel->setText("Loading...");
	iListCoverLabel->setAutoFillBackground(true);
	iListCoverLabel->show();
	QApplication::processEvents();

	openWin();
}

void PackageView::clearSearch()
{
	ui->lineEdit->clear();
	ui->searchBar->hide();
	ui->toolBar->show();
	ui->listWidget->setFocusPolicy(Qt::StrongFocus);

	iSearchResults.clear();
}

void PackageView::on_actionSearch_triggered()
{
	if( ui->searchBar->isVisible() )
		return;

	ui->listWidget->setFocusPolicy(Qt::NoFocus);
	ui->searchLabel->setText( QString("%1 results").arg(ui->listWidget->count()) );
	ui->toolBar->hide();
	ui->searchBar->show();
	ui->lineEdit->setFocus();
	iPrevSearchText = "";
}

void PackageView::on_lineEdit_textEdited(QString text)
{
	if( text=="" ) {
		on_btn_searchClose_clicked();
		return;
	}

	if( iPrevSearchText.length() > text.length() )
	{
		iListCoverLabel->setText("Loading...");
		iListCoverLabel->setAutoFillBackground(true);
		iListCoverLabel->show();
		QApplication::processEvents();

		ui->searchBar->hide();
		openWin();
		ui->searchBar->show();
		ui->lineEdit->setFocus();
	}

	iPrevSearchText = text;
	QList<QListWidgetItem*> results;
	if( text.startsWith(":") )
		results = ui->listWidget->findItems( text.mid(1), Qt::MatchStartsWith );
	else
		results = ui->listWidget->findItems( text, Qt::MatchContains );

	ui->searchLabel->setText( QString("%1 results").arg(results.count()) );

	iSearchResults.clear();
	for( int i=0; i<results.count(); i++ )
	{
		iSearchResults.append( dynamic_cast<PackageListWidgetItem*>(results.at(i))->package() );
	}
	openWin();
}

void PackageView::setSearchText(QString text)
{
	ui->lineEdit->setText(text);
	on_lineEdit_textEdited(text);
}

KeyEventGrabber::KeyEventGrabber(QObject* parent) : QObject(parent)
{
}

bool KeyEventGrabber::eventFilter(QObject *obj, QEvent *event)
{
	if( event->type() == QEvent::KeyPress ) {
		QString text = dynamic_cast<QKeyEvent*>(event)->text();
		int key = dynamic_cast<QKeyEvent*>(event)->key();
		if( (text.trimmed() != "" || text==" ") && key!=Qt::Key_Backspace ) {
			dynamic_cast<PackageView*>(this->parent())->on_actionSearch_triggered();
			dynamic_cast<PackageView*>(this->parent())->setSearchText( text );
			return true;
		}
	}
	return QObject::eventFilter(obj, event);
}

void PackageView::on_actionUpgrade_all_triggered()
{
	for( int i=0; i<ui->listWidget->count(); i++ )
	{
		Package* pkg = dynamic_cast<PackageListWidgetItem*>(ui->listWidget->item(i))->package();
		if( pkg->isUpgradeable() ) {
			pkg->setMarkedForOperation( Package::PkgOpInstallUpgrade );
			ui->listWidget->item(i)->setData(UserRoleMarked, (int)Package::PkgOpInstallUpgrade);
		}
	}
	updateLabel();
}

void PackageView::on_btn_Sort_clicked()
{
	if( iSortOrder==SortAlpha )
		iSortOrder = SortDateDesc;
	else if( iSortOrder==SortDateDesc )
		iSortOrder = SortSizeDesc;
	else if( iSortOrder==SortSizeDesc )
		iSortOrder = SortAlpha;

	openWin();
}

void PackageView::on_actionView_log_triggered()
{
	QByteArray log = iAptInterface->readLogFile();
	LogView l(log, this);
	l.exec();
}
