/*
	This file is part of Faster Application Manager.

	Faster Application Manager is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Faster Application Manager is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Faster Application Manager.  If not, see <http://www.gnu.org/licenses/>.

	(C) Heikki Holstila 2010
*/

#include "packageselector.h"
#include "ui_packageselector.h"
#include "package.h"
#include "packageview.h"
#include "repository.h"

PackageSelector::PackageSelector(Package* pkg, QWidget *parent) :
    QDialog(parent),
    ui(new Ui::PackageSelector)
{
    ui->setupUi(this);

	if( pkg->icon() && !pkg->icon()->isNull() ) {
		ui->label_appicon->setPixmap( *pkg->icon() );
	} else {
		ui->label_appicon->setPixmap( QPixmap(":/icons/icons/appdefault.png") );
	}
	this->setWindowTitle( pkg->name() );

	if( pkg->maemoDisplayName() != "" )
		ui->label_header->setText("<b>" + pkg->maemoDisplayName() + "</b>");
	else
		ui->label_header->setText("<b>" + pkg->name() + "</b>");

	QString longtext;

	longtext += "Category: " + pkg->section() + "<br>";

	if( !pkg->isInstalled() )
	{
		longtext += "Repositories: ";
		if( pkg->repositories().count()>0 ) {
			for(int i=0; i<pkg->repositories().count(); i++ ) {
				if( pkg->repositories().at(i) )
					longtext += pkg->repositories().at(i)->name();
				else
					longtext += "unknown";
				if( i<pkg->repositories().count()-1 )
					longtext += ", ";
			}
			longtext += "<br>";
		} else {
			longtext += "unknown<br>";
		}
	}

	if( pkg->isInstalled() ) {
		longtext += "Package is installed, version <b>" + pkg->version() + "</b>";
		if( pkg->installedSize() > 0 )
			longtext += QString(" (%L1 kB)").arg(pkg->installedSize() );
	} else {
		longtext += "Package is not installed";
	}
	longtext += "<br>";

	if( pkg->isInstalled() ) {
		longtext += "Installation time: ";
		if( pkg->date().isValid() )
			longtext += pkg->date().toString("yyyy-MM-dd hh:mm");
		else
			longtext += "Unknown";
		longtext += "<br>";
	}

	if( !pkg->isInstalled() ){
		longtext += "Available version <b>" + pkg->version() + "</b>";
		if( pkg->size() > 0 )
			longtext += QString(" (%L1 kB)").arg(pkg->size()/1024 );
		longtext += "<br>";
	}
	else if( pkg->isUpgradeable() ){
		longtext += "Available version <b>" + pkg->upgradeableVersion() + "</b>";
		if( pkg->size() > 0 )
			longtext += QString(" (%L1 kB)").arg(pkg->size()/1024 );
		longtext += "<br>";
	}

	if( pkg->isUpgradeable() )
	{
		longtext += "Repositories: ";
		Package* upg_pkg = pkg->availablePackage();
		if( upg_pkg && upg_pkg->repositories().count()>0 ) {
			for(int i=0; i<upg_pkg->repositories().count(); i++ ) {
				if( upg_pkg->repositories().at(i) )
					longtext += upg_pkg->repositories().at(i)->name();
				else
					longtext += "unknown";
				if( i<upg_pkg->repositories().count()-1 )
					longtext += ", ";
			}
			longtext += "<br>";
		} else {
			longtext += "unknown<br>";
		}
	}

	if( pkg->markedOperation() == Package::PkgOpInstallUpgrade )
	{
		if( pkg->isUpgradeable() )
			longtext += "Package has been marked for <b>upgrade</b><br>";
		else
			longtext += "Package has been marked for <b>installation</b><br>";
	} else if(pkg->markedOperation() == Package::PkgOpRemove)
	{
		longtext += "Package has been marked for <b>removal</b><br>";
	}

	longtext += "<br>";
	longtext += pkg->descShort();
	QString descLong = pkg->descLong();
	if( descLong.length()>0 ) {
		descLong.replace('\n',"<br>");
		longtext += "<br><font size=\"-1\"><br>" + descLong + "</font>";
	}

	ui->label_text->setText( longtext );

	if( pkg->isUpgradeable() )
		ui->radioInstall->setText("Upgrade");
	else
		ui->radioInstall->setText("Install");

	if( pkg->markedOperation() == Package::PkgOpNone )
	{
		if( pkg->isInstalled() && pkg->isUpgradeable() )
			ui->label_statusicon->setPixmap(QPixmap(":/icons/icons/pkg_nop_instupgr.png"));
		else if( pkg->isInstalled() )
			ui->label_statusicon->setPixmap(QPixmap(":/icons/icons/pkg_nop_installed.png"));
		else if( !pkg->isInstalled() )
			ui->label_statusicon->setPixmap(QPixmap(":/icons/icons/pkg_nop_notinstalled.png"));

		ui->radioNothing->setChecked(true);
	} else if( pkg->markedOperation() == Package::PkgOpInstallUpgrade ) {
		if( pkg->isUpgradeable() )
			ui->label_statusicon->setPixmap(QPixmap(":/icons/icons/pkg_upgrade.png"));
		else
			ui->label_statusicon->setPixmap(QPixmap(":/icons/icons/pkg_install.png"));
		ui->radioInstall->setChecked(true);
	} else if( pkg->markedOperation() == Package::PkgOpRemove ) {
		ui->label_statusicon->setPixmap(QPixmap(":/icons/icons/pkg_remove.png"));
		ui->radioRemove->setChecked(true);
	}

	if( pkg->isInstalled() && !pkg->isUpgradeable() )
	{
		ui->radioInstall->setEnabled(false);
		ui->radioInstall->hide();
	}
	if( !pkg->isInstalled() ) {
		ui->radioRemove->setEnabled(false);
		ui->radioRemove->hide();
	}

#ifndef Q_WS_MAEMO_5	// for simulator
	ui->radioInstall->setChecked(true);
#endif
}

PackageSelector::~PackageSelector()
{
    delete ui;
}

void PackageSelector::changeEvent(QEvent *e)
{
    QDialog::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

Package::operation PackageSelector::selectedOperation()
{
	if( ui->radioInstall->isChecked() )
		return Package::PkgOpInstallUpgrade;
	if( ui->radioRemove->isChecked() )
		return Package::PkgOpRemove;

	return Package::PkgOpNone;
}
