/*
	This file is part of Faster Application Manager.

	Faster Application Manager is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Faster Application Manager is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Faster Application Manager.  If not, see <http://www.gnu.org/licenses/>.

	(C) Heikki Holstila 2010
*/

#include <QtCore>
#include <iostream>
#include "aptinterface.h"
#include "mainwindow.h"
#include "packageview.h"
#include "package.h"

AptInterface::AptInterface(MainWindow* w_)
{
	iMainWindow = w_;
	iRunProc = new QProcess();
	connect(iRunProc,SIGNAL(error(QProcess::ProcessError)),this,SLOT(procError(QProcess::ProcessError)));
	connect(iRunProc,SIGNAL(finished(int,QProcess::ExitStatus)),this,SLOT(procFinished(int,QProcess::ExitStatus)));
	iRunProc->setProcessChannelMode(QProcess::MergedChannels);

	iMode = ModeNone;
	iReady = false;
	iError = tr("Undefined error");

	QDir logdir(KlogFileDir);
	logdir.mkpath(KlogFileDir);
	QFile logfile(KlogFileName);
	logfile.remove();

	getRepositories();
	iNumAutoUpdates = 0;
}

AptInterface::~AptInterface()
{
	delete iRunProc;
}

bool AptInterface::updateCatalogs()
{
	if( iMode != ModeNone ) return false;
	iMode = ModeAptGetUpdate;

	if( !writeRepositories( getRepositories() ) )
		return false;

	QString runBinary = "/usr/bin/apt-get";
	QStringList runParameters;
	runParameters << "-q" << "update";

	iRunProc->start(runBinary,runParameters);
	return true;
}

bool AptInterface::updatePackageList()
{
	if( iMode != ModeNone ) return false;

	if( !writeRepositories( getRepositories() ) )
		return false;

	toInstall.clear();
	toRemove.clear();
	pkgNamesList.clear();

	QDir logdir(KlogFileDir);
	logdir.mkpath(KlogFileDir);
	QFile logfile(KlogFileName);
	logfile.remove();

	QHashIterator<QString, Package*> i( iPackages );
	while (i.hasNext()) {
		i.next();
		if( i.value()!=0 ) delete i.value();
	}
	iPackages.clear();
	iSelectedCount = 0;

	iMode = ModeReadInfo;
	QString runBinary = "/usr/bin/apt-cache";
	QStringList runParameters;
	runParameters << "dumpavail";

	iRunProc->start(runBinary,runParameters);
	return true;
}

bool AptInterface::isInterfaceReady()
{
	if( iMode==ModeNone && iReady) {
		return true;
	}
	return false;
}

void AptInterface::procError(QProcess::ProcessError error)
{
	QByteArray output = iRunProc->readAllStandardOutput();
	logToFile(output);

	iError.setNum(error);
	iError.prepend(tr("QProcess error "));
	iMainWindow->operationFinished(iMode, false, iError);
	iMode = ModeNone;
}

void AptInterface::procFinished(int exitCode, QProcess::ExitStatus exitStatus)
{
	aptMode lastMode = iMode;
	iMode = ModeNone;
	if( exitCode != 0 ) {
		QByteArray output = iRunProc->readAllStandardOutput();
		logToFile(output);

		if( output.contains("Could not get lock") ) {
			iMainWindow->operationFinished(lastMode, false, "<b>Could not get lock</b><br>" \
										   "The package management system is locked by another process");
		} else if( output.contains("E: Unable to fetch some archives") ) {
			iMainWindow->operationFinished(lastMode, false, "<b>Failed to fetch packages</b><br>" \
										   "Your network connection might be down or your catalogs could be out of date");
		} else if( output.contains("E: Couldn't find package") ) {
			int pos = output.indexOf("E: Couldn't find package");
			QString msg = "<b>Missing package</b><br>";
			if( pos!=-1 )
				msg += output.mid(pos) + "<br>";
			msg += "Your catalogs might be out of date";
			iMainWindow->operationFinished(lastMode, false, msg);
		} else if( output.contains("E: Broken packages") ) {
			iMainWindow->operationFinished(lastMode, false, "<b>Broken packages</b><br>" \
										   "Your system has broken packages, see log for details");
		} else {
			iError.setNum(exitCode);
			iError.prepend(tr("Exit code "));
			QString s;
			s.setNum(exitStatus);
			iError.append(", status " + s );
			iError.append("<br>See the log for details (you can access it from the menu)");
			iMainWindow->operationFinished(lastMode, false, iError);
		}
	} else {

		if( lastMode == ModeAptGetUpdate )
		{
			QByteArray output = iRunProc->readAllStandardOutput();
			logToFile(output);
			iRunProc->close();
			if( output.contains("Could not resolve host") ) {
				iMainWindow->operationFinished(lastMode, false, "<b>Could not resolve host</b><br>" \
											   "Some (or all) of the catalogs might be incomplete. Check your network connection");
			} else {
				QFile lastupdate("/root/.fapman/lastupdate");	// create empty file, access time can be used later
				if( lastupdate.open(QIODevice::WriteOnly) )
					lastupdate.close();
				iMainWindow->operationFinished(lastMode, true, "");
			}
		}

		if( lastMode == ModeReadInfo )
		{
			iNumAutoUpdates = 0;
			if( !parseAptCacheOutput() ) {
				iMainWindow->operationFinished(lastMode, false, iError);
				return;
			}
			iRunProc->close();
			if( !readDpkgDatabase() ) {
				iRunProc->close();
				iMainWindow->operationFinished(lastMode, false, iError);
				return;
			}
			iRunProc->close();
			iReady = true;
			iMainWindow->operationFinished(lastMode, true, "");
		}

		if( lastMode == ModeAptGetSimulateInstall || lastMode == ModeAptGetSimulateRemove )
		{
			if( !parseSimulated() ) {
				iRunProc->close();
				iMainWindow->operationFinished(lastMode, false, iError);
				return;
			}
			iRunProc->close();
			iMainWindow->operationFinished(lastMode, true, "");
		}

		if( lastMode == ModeAptGetInstall || lastMode == ModeAptGetRemove )
		{
			QByteArray output = iRunProc->readAllStandardOutput();
			logToFile(output);

			iRunProc->close();
			iMainWindow->operationFinished(lastMode, true, "");
		}

		if( lastMode == ModeAptGetClean )
		{
			iRunProc->close();
			iMainWindow->operationFinished(lastMode, true, "");
		}
	}
}

bool AptInterface::parseAptCacheOutput()
{
	QList<QByteArray> lines = iRunProc->readAllStandardOutput().split('\n');

	Package* pkg = 0;	
	multiLine multiline=MultiLineNone;
	for( int i=0; i<lines.count(); i++)
	{
		QByteArray line = lines.at(i);
		if( !processPackageDataLine(pkg, line, multiline, false) )
			return false;
	}
	if( pkg!=0 ) pkg->updateStatus(); // for the last package

	//std::cout << "apt-cache: read " << iPackages.count() << " available packages" << std::endl;
	return true;
}

bool AptInterface::readDpkgDatabase()
{
	// available at /var/lib/dpkg/status
	QFile db("/var/lib/dpkg/status");
	if (!db.open(QIODevice::ReadOnly | QIODevice::Text)) {
		iError = tr("Unable to read dpkg database");
		return false;
	}

	Package* pkg = 0;
	multiLine multiline=MultiLineNone;

	while (!db.atEnd()) {
		QByteArray line = db.readLine();
		if( !processPackageDataLine(pkg, line, multiline, true) )
			return false;
	}
	if( pkg!=0 ) pkg->updateStatus(); // for the last package

	db.close();

	//std::cout << "dpkg: now " << iPackages.count() << " packages total" << std::endl;

	return true;
}

bool AptInterface::processPackageDataLine(Package*& pkg, QByteArray& line, multiLine& multiline, bool dpkg)
{
	if( !line.startsWith(" ") )
		line = line.trimmed();

	if( line.startsWith("Package: ") )
	{
		if( pkg!=0 ) pkg->updateStatus(); // for previous package

		QString pkgname = line.mid(9);
		if( pkgname==0 || pkgname == "" ) {
			iError = tr("Empty package name");
			//std::cout << "Encountered empty package name!" << std::endl;
			return false;
		}
		if( dpkg )
			pkg = iPackages.value(pkgname,0);
		if( !dpkg || pkg==0 ) {
			pkg = new Package(pkgname, this);
			iPackages.insert(pkgname,pkg);
			if( dpkg ) {
				pkg->setLocal(true);
				//std::cout << pkgname.toStdString() << std::endl;
			}
		}
		if( dpkg )
			pkg->setInstalled(true);
		multiline=MultiLineNone;
	}

	if( line.startsWith("Status: ") && pkg != 0 && dpkg )
	{
		if( line.mid(8) == "install ok installed" )
			pkg->setInstalled(true);
		else
			pkg->setInstalled(false);
	}
	if( line.startsWith("Section: ") && pkg != 0 && pkg->section()=="" )
	{
		pkg->setSection( line.mid(9) );
	}
	if( line.startsWith("Version: ") && pkg != 0 )
	{
		if( dpkg )
			pkg->setVersionInstalled( line.mid(9) );
		else
			pkg->setVersionAvailable( line.mid(9) );
	}
	if( line.startsWith("Size: ") && pkg != 0 && pkg->size()==0 )
	{
		pkg->setSize( line.mid(6).toInt() );
	}
	if( line.startsWith("Installed-Size: ") && pkg != 0 && pkg->installedSize()==0 )
	{
		pkg->setInstalledSize( line.mid(16).toInt() );
	}
	if( line.startsWith("Maemo-Display-Name: ") && pkg != 0 && pkg->maemoDisplayName()=="" )
	{
		pkg->setMaemoDisplayName( line.mid(20) );
	}

	if( multiline == MultiLineDesc ) {
		if( line.startsWith(" ") && line.trimmed()!="" ) {
			pkg->appendDescLong( line.trimmed() + " " );
		} else {
			multiline = MultiLineNone;
		}
	}
	if( multiline == MultiLineIcon ) {
		if( line.startsWith(" ") && line.trimmed()!="" ) {
			pkg->appendIconData( line.trimmed() );
		} else {
			multiline = MultiLineNone;
		}
	}

	if( line.startsWith("Description: ") && pkg != 0 && pkg->descShort()=="" )
	{
		pkg->setDescShort( line.mid(13) );
		multiline = MultiLineDesc;
	}
	if( line.startsWith("Maemo-Icon-26:") && pkg != 0 && !pkg->hasIconData() )
	{
		if( line.mid(15).trimmed() != "" ) {
			pkg->appendIconData( line.mid(15).trimmed() );
		}
		multiline = MultiLineIcon;
	}

	return true;
}

bool AptInterface::parseSimulated()
{
	QByteArray output = iRunProc->readAllStandardOutput();
	logToFile(output);
	QList<QByteArray> lines = output.split('\n');

	toInstall.clear();
	toRemove.clear();

	for( int i=0; i<lines.count(); i++)
	{
		QString s = lines.at(i);
		if( s.startsWith("Inst ") )
		{
			toInstall <<  s.section(' ',1,1);
		}
		if( s.startsWith("Remv ") )
		{
			toRemove << s.section(' ',1,1);
		}
	}

	/*
	std::cout << "Simulated results:" << std::endl;
	std::cout << "  Install: ";
	for(int x=0; x<toInstall.count(); x++)
		std::cout << toInstall.at(x).toStdString() << " ";
	std::cout << std::endl << "  Remove: ";
	for(int x=0; x<toRemove.count(); x++)
		std::cout << toRemove.at(x).toStdString() << " ";
	std::cout << std::endl;
	*/

	if( toInstall.count()==0 && toRemove.count()==0 )
		return false;

	return true;
}

void AptInterface::simulateInstall()
{
	//pkgNamesList = names_;
	iMainWindow->busyDialog(true, "Operation in progress", "Reading dependencies");

	iMode = ModeAptGetSimulateInstall;

	pkgNamesList.clear();
	QHashIterator<QString, Package*> i( iPackages );
	while (i.hasNext())
	{
		i.next();
		if( i.value()->isMarkedForOperation() )
		{
			QString pkgname = i.value()->name();
			if( i.value()->markedOperation() == Package::PkgOpRemove )
				pkgname.append("-");
			pkgNamesList << pkgname;
		}
	}

	QString runBinary = "/usr/bin/apt-get";
	QStringList runParameters;
	runParameters << "-qsy" << "install";
	runParameters << pkgNamesList;

	iRunProc->start(runBinary,runParameters);
}

/*
void AptInterface::simulateRemove()
{
	iMainWindow->busyDialog(true, "Operation in progress", "Reading dependencies");

	iMode = ModeAptGetSimulateRemove;

	pkgNamesList.clear();
	QHashIterator<QString, Package*> i( iPackages );
	while (i.hasNext())
	{
		i.next();
		if( i.value()->markedOperation() == Package::PkgOpRemove )
			pkgNamesList << i.value()->name();
	}

	QString runBinary = "/usr/bin/apt-get";
	QStringList runParameters;
	runParameters << "-qsy" << "remove";
	runParameters << pkgNamesList;

	iRunProc->start(runBinary,runParameters);
}
*/

void AptInterface::installPackages(QStringList names_, int inst_, int remv_)
{
	QString busytext = "";
	if( inst_ > 0 )
		busytext += QString("Downloading and installing %0 package(s)<br>").arg(inst_);
	if( remv_ > 0 )
		busytext += QString("Removing %0 package(s)<br>").arg(remv_);

	iMainWindow->busyDialog(true, "Operation in progress", busytext);

	iMode = ModeAptGetInstall;

	QString runBinary = "/usr/bin/apt-get";
	QStringList runParameters;
	runParameters << "-qy" << "--force-yes" << "install";
	runParameters << names_;

	iRunProc->start(runBinary,runParameters);
}

/*
void AptInterface::removePackages(QStringList names_)
{
	iMainWindow->busyDialog(true, "Operation in progress", "Removing packages");

	iMode = ModeAptGetRemove;

	QString runBinary = "/usr/bin/apt-get";
	QStringList runParameters;
	runParameters << "-qy" << "--force-yes" << "remove";
	runParameters << names_;

	iRunProc->start(runBinary,runParameters);
}
*/

void AptInterface::aptGetClean()
{
	iMainWindow->busyDialog(true, "Operation in progress", "Cleaning up");
	//std::cout << "apt-get clean" << std::endl;

	iMode = ModeAptGetClean;

	QString runBinary = "/usr/bin/apt-get";
	QStringList runParameters;
	runParameters << "clean";

	iRunProc->start(runBinary,runParameters);
}

void AptInterface::logToFile( QByteArray data )
{
	QFile f(KlogFileName);

	if( f.open( QIODevice::Append | QIODevice::WriteOnly | QIODevice::Text ) )
	{
		QTextStream out(&f);
		out << data;
		f.close();
	}
}

QString AptInterface::readLogFile()
{
	QString log = "";

	QFile own("/root/.fapman/lastlog.txt");
	if( own.open(QIODevice::ReadOnly | QIODevice::Text ) )
	{
		while(!own.atEnd())
		{
			QString line = own.readLine();
			log.append( line );
		}
		own.close();
	}

	if( log=="" )
		log = "The log is empty";

	return log;
}

QStringList AptInterface::getRepositories()
{
	QStringList repos;

	QFile own("/root/.fapman/repos.list");
	if( own.open(QIODevice::ReadOnly | QIODevice::Text ) )
	{
		while(!own.atEnd())
		{
			QString line = own.readLine();
			repos.append(line.trimmed());
		}
		own.close();		
		return repos;
	}

	QFile ham("/etc/apt/sources.list.d/hildon-application-manager.list");
	if( ham.open(QIODevice::ReadOnly | QIODevice::Text ) )
	{
		while(!ham.atEnd())
		{
			QString line = ham.readLine();
			repos.append(line.trimmed());
		}
		ham.close();
	}

	QStringList additional;
	additional
			<< "deb http://repository.maemo.org/extras-testing/ fremantle free non-free"
			<< "deb http://repository.maemo.org/extras-devel/ fremantle free non-free";

	for( int i=0; i<additional.count(); i++ )
	{
		if( !repos.contains(additional.at(i)) && !repos.contains("#"+additional.at(i)))
		{
			repos.append( "#" + additional.at(i) );
		}
	}

	return repos;
}

bool AptInterface::writeRepositories(QStringList repos)
{
	QFile own("/root/.fapman/repos.list");
	if( own.open(QIODevice::WriteOnly | QIODevice::Text ) )
	{
		QTextStream out(&own);
		for( int i=0; i<repos.count(); i++ )
			out << repos.at(i) << "\n";
		own.close();
	}

	QFile ham("/etc/apt/sources.list.d/hildon-application-manager.list");
	if( ham.open(QIODevice::WriteOnly | QIODevice::Text ) )
	{
		QTextStream out(&ham);
		for( int i=0; i<repos.count(); i++ )
			out << repos.at(i) << "\n";

		ham.close();

		return true;
	}

	return false;
}

bool AptInterface::needRepoRefresh()
{
	if( iNumAutoUpdates > 0 )
		return false;

	QFile own("/root/.fapman/repos.list");
	QFile ham("/etc/apt/sources.list.d/hildon-application-manager.list");

	if( own.size() != ham.size() ) {
		iNumAutoUpdates++;
		return true;
	}

	QFileInfo a("/root/.fapman/lastupdate");
	QFileInfo b("/home/user/.hildon-application-manager/last-update");
	QDateTime aDate;
	QDateTime bDate;
	aDate.setTime_t(0);
	bDate.setTime_t(0);
	if( a.exists() )
		aDate = a.lastModified();
	if( b.exists() )
		bDate = b.lastModified();
	aDate = aDate.addSecs(24*60*60); //24h
	bDate = bDate.addSecs(24*60*60); //24h
/*
	std::cout
			<< aDate.toString("dd.MM.yyyy hh:mm").toStdString() << "  "
			<< bDate.toString("dd.MM.yyyy hh:mm").toStdString() << "  "
			<< QDateTime::currentDateTime().toString("dd.MM.yyyy hh:mm").toStdString() << std::endl;
*/
	if( aDate < QDateTime::currentDateTime() && bDate < QDateTime::currentDateTime() ) {
		iNumAutoUpdates++;
		return true;
	}

	return false;
}
