/*
	This file is part of Faster Application Manager.

	Faster Application Manager is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Faster Application Manager is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Faster Application Manager.  If not, see <http://www.gnu.org/licenses/>.

	(C) Heikki Holstila 2010
*/

#include <iostream>
#include "packageview.h"
#include "ui_packageview.h"
#include "package.h"
#include "filterselect.h"
#include "confirmdialog.h"
#include "dimmer.h"
#include "packageselector.h"
#include "help.h"

PackageListWidgetItem::PackageListWidgetItem(Package* p_, QString name_) : QListWidgetItem(name_)
{
	iPackage = p_;
}


void ListItemDelegate::paint(QPainter *painter, const QStyleOptionViewItem &option, const QModelIndex &index) const
{
	QString name = index.data(UserRoleName).toString();
	QString verInst = index.data(UserRoleVerInst).toString();
	QString verAvail = index.data(UserRoleVerAvail).toString();
	QString desc = index.data(UserRoleDescShort).toString();
	Package::operation marked = (Package::operation) index.data(UserRoleMarked).toInt();
	bool installed = index.data(UserRoleInstalled).toBool();
	bool upgradeable = index.data(UserRoleUpgradeable).toBool();
	//int oper = index.data(UserRoleOperation).toInt();

	painter->save();
	QRect r = option.rect;

	QLinearGradient gradientBase(r.topLeft(), r.bottomLeft());
	QColor base = option.palette.color(QPalette::Window);
	QColor base2 = base;
	base.setRgb( base.red()+15, base.green()+15, base.blue()+15 );
	base2.setRgb( base2.red()-15, base2.green()-15, base2.blue()-15 );
	gradientBase.setColorAt(0, base);
	gradientBase.setColorAt(1, base2);

	/*
	QLinearGradient gradientHilight(r.topLeft(), r.bottomLeft());
	QColor hl = option.palette.color(QPalette::Highlight);
	QColor hl2 = hl;
	hl.setRgb( hl.red()+10, hl.green()+10, hl.blue()+10 );
	hl2.setRgb( hl2.red()-30, hl2.green()-30, hl2.blue()-30 );
	gradientHilight.setColorAt(0, hl);
	gradientHilight.setColorAt(1, hl2);

	if( marked != Package::PkgOpNone ) {
		painter->fillRect(r, gradientHilight);
	} else {
		painter->fillRect(r, gradientBase);
	}
	*/
	painter->fillRect(r, gradientBase);

	//painter->drawLine(QPoint(r.left(),r.top()), QPoint(r.right(),r.top()));
	painter->drawLine(QPoint(r.left(),r.bottom()), QPoint(r.right(),r.bottom()));

	QPixmap icon = index.data(Qt::DecorationRole).value<QPixmap>();
	if( icon.isNull() ) {
		// use default icon
		painter->drawPixmap( r.left(), r.top()+4, 48, 48, iDefaultIcon );
	} else {
		painter->drawPixmap( r.left(), r.top()+4, 48, 48, icon );
	}

	QPixmap statusicon;
	if( marked == Package::PkgOpNone )
	{
		if( installed && upgradeable )
			statusicon = iIconPkgNoOpInstalledUpgradeable;
		else if( installed )
			statusicon = iIconPkgNoOpInstalled;
		else if( !installed )
			statusicon = iIconPkgNoOpNotInstalled;
	} else if( marked == Package::PkgOpInstallUpgrade ) {
		if( upgradeable )
			statusicon = iIconPkgUpgrade;
		else
			statusicon = iIconPkgInstall;
	} else if( marked == Package::PkgOpRemove ) {
		statusicon = iIconPkgRemove;
	}

	int ver_w = 0;
	if( QApplication::desktop()->width() > QApplication::desktop()->height() )
	{
		r = option.rect;
		r.setRight( r.right()-statusicon.width()-4 );
		if( installed && !upgradeable ) {
			painter->drawText(r, Qt::AlignTop|Qt::AlignRight, verInst, &r);
			ver_w = r.width();
		} else {
			painter->drawText(r, Qt::AlignTop|Qt::AlignRight, verAvail, &r);
			ver_w = r.width();
		}
	}

	r = option.rect;
	r.setRight( r.right()-statusicon.width()-4-ver_w );
	QFont f = painter->font();
	f.setBold(true);
	painter->setFont(f);
	painter->drawText(r.left()+iDefaultIcon.width()+2, r.top(), r.width(), r.height(), Qt::AlignTop|Qt::AlignLeft, name, &r);
	f.setBold(false);
	painter->setFont(f);

	f.setPointSize( f.pointSize()-4 );
	painter->setFont(f);
	r = option.rect;
	painter->drawText(r.left()+iDefaultIcon.width()+2, r.top(), r.width(), r.height(), Qt::AlignBottom|Qt::AlignLeft, desc, &r);

	r = option.rect;
	painter->drawPixmap(r.right()-statusicon.width()-2, r.top()+4, 24, 24, statusicon);

	painter->restore();
}

void ListItemDelegate::loadIcons()
{
	iDefaultIcon = QPixmap(":/icons/icons/appdefault.png");
	iIconPkgInstall = QPixmap(":/icons/icons/pkg_install.png");
	iIconPkgUpgrade = QPixmap(":/icons/icons/pkg_upgrade.png");
	iIconPkgRemove = QPixmap(":/icons/icons/pkg_remove.png");
	iIconPkgNoOpInstalled = QPixmap(":/icons/icons/pkg_nop_installed.png");
	iIconPkgNoOpNotInstalled = QPixmap(":/icons/icons/pkg_nop_notinstalled.png");
	iIconPkgNoOpInstalledUpgradeable = QPixmap(":/icons/icons/pkg_nop_instupgr.png");
}


QSize ListItemDelegate::sizeHint(const QStyleOptionViewItem&, const QModelIndex&) const
{
	return QSize(400, 60);
}


PackageView::PackageView(QWidget *parent) : QMainWindow(parent), ui(new Ui::PackageView)
{
	iMainWindow = static_cast<MainWindow*>(parent);
	ui->setupUi(this);
#ifdef Q_WS_MAEMO_5
	this->setAttribute(Qt::WA_Maemo5StackedWindow);
	this->setWindowFlags(Qt::Window);
	this->setAttribute(Qt::WA_Maemo5AutoOrientation);
#endif

	connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(orientationChanged()));

	iCatFilterLabels
			<< tr("All marked packages")	// a special case
			<< tr("All user categories")	// special case: don't show "hidden" category
			<< tr("\tDesktop")
			<< tr("\tEducation")
			<< tr("\tGames")
			<< tr("\tGraphics")
			<< tr("\tInternet & Networking")
			<< tr("\tLocation & Navigation")
			<< tr("\tMultimedia")
			<< tr("\tOffice")
			<< tr("\tOther")
			<< tr("\tProgramming")
			<< tr("\tScience")
			<< tr("\tSystem")
			<< tr("\tUtilities")
			<< tr("All packages (ADVANCED)");

	iCatFilterStrings
			<< ""
			<< "user/"
			<< "user/desktop"
			<< "user/education"
			<< "user/games"
			<< "user/graphics"
			<< "user/network"
			<< "user/navigation"
			<< "user/multimedia"
			<< "user/office"
			<< "user/other"
			<< "user/development"
			<< "user/science"
			<< "user/system"
			<< "user/utilities"
			<< "";

	iDefaultCatFilter = CatFilterAllUser;
	iSelectedCatFilter = iDefaultCatFilter;

	iStatFilterLabels
			<< tr("All")
			<< tr("Not installed")
			<< tr("Upgradeable")
			<< tr("Installed");

	iSelectedStatFilter = Package::PkgStatUnknown;

	iDimmer = new dimmer(this);

	iListCoverLabel = new QLabel("No packages",ui->listWidget);
	iListCoverLabel->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
	iListCoverLabel->setAlignment(Qt::AlignCenter);
}

PackageView::~PackageView()
{
    delete ui;
}

void PackageView::orientationChanged()
{
	ui->listWidget->scroll(1,1);	// this causes all items to be repainted
	iListCoverLabel->setGeometry( ui->listWidget->rect() );
}

void PackageView::resizeEvent(QResizeEvent* event)
{
	iListCoverLabel->setGeometry( ui->listWidget->rect() );
	QWidget::resizeEvent(event);
}

void PackageView::openWin(MainWindow::operation op_)
{
	iOperation = op_;
	iListCoverLabel->hide();
	ui->listWidget->clear();
	ui->listWidget->setSortingEnabled(true);

	delete ui->listWidget->itemDelegate();
	ListItemDelegate* delegate = new ListItemDelegate(ui->listWidget);
	delegate->loadIcons();
	ui->listWidget->setItemDelegate( delegate );

	QHashIterator<QString, Package*> i( *iAptInterface->packages() );
	while (i.hasNext())
	{
		i.next();

		QString name = i.value()->name();
		if( i.value()->maemoDisplayName()!="" )
			name = i.value()->maemoDisplayName();
		QString n( name.at(0) );
		n = n.toUpper();
		name.replace(0,1,n);
		bool added = false;

		if( iSelectedStatFilter == Package::PkgStatNotInstalled || iSelectedStatFilter == Package::PkgStatUnknown && !added)
		{
			if( !i.value()->isInstalled() && iSelectedCatFilter!=CatFilterAllMarked && i.value()->section().startsWith(iCatFilterStrings.at(iSelectedCatFilter))
				&& !i.value()->isLocal() )
			{
				if( iSelectedCatFilter==CatFilterAllUser && !i.value()->section().startsWith("user/hidden") ) {
					addListItem(i.value(), name);
					added = true;
				}
				else if( iSelectedCatFilter!=CatFilterAllUser ) {
					addListItem(i.value(), name);
					added = true;
				}
			} else if(iSelectedCatFilter==CatFilterAllMarked && i.value()->isMarkedForOperation() ) {
				addListItem(i.value(), name);
				added = true;
			}
		}
		if( iSelectedStatFilter == Package::PkgStatInstalled || iSelectedStatFilter == Package::PkgStatUnknown && !added)
		{
			if( i.value()->isInstalled() && iSelectedCatFilter!=CatFilterAllMarked && i.value()->section().startsWith(iCatFilterStrings.at(iSelectedCatFilter)) )
			{
				addListItem(i.value(), name);
				added = true;
			} else if(iSelectedCatFilter==CatFilterAllMarked && i.value()->isMarkedForOperation() ) {
				addListItem(i.value(), name);
				added = true;
			}
		}
		if( iSelectedStatFilter == Package::PkgStatUpgradeable || iSelectedStatFilter == Package::PkgStatUnknown && !added)
		{
			if( i.value()->isInstalled() && iSelectedCatFilter!=CatFilterAllMarked && i.value()->versionAvailable()!=i.value()->versionInstalled()
				&& i.value()->versionAvailable()!="" && i.value()->section().startsWith(iCatFilterStrings.at(iSelectedCatFilter))
				&& !i.value()->isLocal() )
			{
				if( iSelectedCatFilter==CatFilterAllUser && !i.value()->section().startsWith("user/hidden") ) {
					addListItem(i.value(), name);
					added = true;
				}
				else if( iSelectedCatFilter!=CatFilterAllUser ) {
					addListItem(i.value(), name);
					added = true;
				}
			} else if(iSelectedCatFilter==CatFilterAllMarked && i.value()->isMarkedForOperation() ) {
				addListItem(i.value(), name);
				added = true;
			}
		}
	}

	updateLabel();
	show();

	if( ui->listWidget->count() == 0 )
	{
		iListCoverLabel->setGeometry( ui->listWidget->rect() );
		iListCoverLabel->show();
	}
}

void PackageView::enableMenu()
{
	ui->menuMenu->setEnabled(true);
}

void PackageView::disableMenu()
{
	ui->menuMenu->setEnabled(false);
}

void PackageView::addListItem(Package* pkg_, QString listname_)
{
	PackageListWidgetItem* p = new PackageListWidgetItem( pkg_, listname_ );

	if( pkg_ != 0 ) {
		QString name = pkg_->name();
		if( pkg_->maemoDisplayName()!="" )
			name = pkg_->maemoDisplayName();
		p->setData(UserRoleName, name);
	} else {
		p->setData(UserRoleName, listname_);
	}

	//p->setData(UserRoleOperation, iOperation);

	if( pkg_ != 0 ) {
		p->setData(UserRoleDescShort, pkg_->descShort());
		p->setData(UserRoleVerAvail, pkg_->versionAvailable());
		p->setData(UserRoleVerInst, pkg_->versionInstalled());
		p->setData(UserRoleMarked, (int)pkg_->markedOperation());
		p->setData(UserRoleInstalled, pkg_->isInstalled());
		p->setData(UserRoleUpgradeable, pkg_->isUpgradeable());

		pkg_->convertIcon();
		p->setData(Qt::DecorationRole, *pkg_->icon());
	}
	ui->listWidget->addItem( p );
}

void PackageView::closeEvent(QCloseEvent *event)
{
	if( iDimmer->busy() ) {
		event->ignore();
		return;
	}

	if( iAptInterface->selectedCount() == 0 )
	{		
		resetWindow();
		iMainWindow->setOperation(MainWindow::OpNone);
		event->accept();
	} else {
		QString c;
		c.setNum( iAptInterface->selectedCount() );
		ConfirmDialog d(true, this);
		d.setText(tr("Returning to main menu"), tr("Clear ") + c + tr(" package selection(s) and lose all the pending changes?"));
		if( d.exec() ) {
			resetWindow();
			iMainWindow->setOperation(MainWindow::OpNone);
			event->accept();
		} else {
			event->ignore();
		}
	}
}

void PackageView::changeEvent(QEvent *e)
{
    QMainWindow::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}

void PackageView::on_btn_Commit_clicked()
{
	/*
	QStringList pkgnames_inst, pkgnames_remv;

	QHashIterator<QString, Package*> i( *iAptInterface->packages() );
	while (i.hasNext())
	{
		i.next();
		if( i.value()->markedOperation() == Package::PkgOpInstallUpgrade )
			pkgnames_inst << i.value()->name();
		if( i.value()->markedOperation() == Package::PkgOpRemove )
			pkgnames_remv << i.value()->name();
	}

	std::cout << "INSTALL/UPGRADE: " << std::endl;
	for( int x=0; x<pkgnames_inst.count(); x++ )
		std::cout << pkgnames_inst.at(x).toStdString() << " ";
	std::cout << std::endl;

	std::cout << "REMOVE: " << std::endl;
	for( int x=0; x<pkgnames_remv.count(); x++ )
		std::cout << pkgnames_remv.at(x).toStdString() << " ";
	std::cout << std::endl;
	*/

	iAptInterface->simulateInstall();
}

void PackageView::on_actionClear_selections_triggered()
{
	QString c;
	c.setNum( iAptInterface->selectedCount() );
	ConfirmDialog d(true, this);
	d.setText(tr("Confirmation"), tr("Clear ") + c + tr(" package selection(s) and lose all the pending changes?"));
	if( d.exec() )
	{
		QHashIterator<QString, Package*> i( *iAptInterface->packages() );
		while (i.hasNext())
		{
			i.next();
			i.value()->setMarkedForOperation(Package::PkgOpNone);
		}

		iAptInterface->resetSelectedCount();

		openWin(iOperation);
	}
}

void PackageView::on_actionCategoryFilter_triggered()
{
	on_btn_CategoryFilter_clicked();
}

void PackageView::on_listWidget_itemClicked(QListWidgetItem* item)
{
	PackageSelector s(static_cast<PackageListWidgetItem*>(item)->package(), this);
	s.exec();

	static_cast<PackageListWidgetItem*>(item)->package()->setMarkedForOperation( s.selectedOperation() );

	if( static_cast<PackageListWidgetItem*>(item)->package()->isMarkedForOperation() ) {
		static_cast<PackageListWidgetItem*>(item)->setData(UserRoleMarked, (int)s.selectedOperation());
	} else {
		static_cast<PackageListWidgetItem*>(item)->setData(UserRoleMarked, (int)s.selectedOperation());
	}

	updateLabel();
}

void PackageView::updateLabel()
{
	QString s;
	s.setNum( iAptInterface->selectedCount() );
	QString s2;
	s2.setNum( ui->listWidget->count() );
	QString statlabel = iStatFilterLabels.at(iSelectedStatFilter);
	if( iSelectedCatFilter == CatFilterAllMarked )
		statlabel = "All";
	ui->label->setText("<font size=\"-2\"><b>" + s + "</b> package(s) marked for operation<br>"
					   + "Showing: <b>" + statlabel + "</b><br>"
					   + "Filter: " + iCatFilterLabels.at(iSelectedCatFilter) + " - " + s2 + " package(s)</font>");

	if( iAptInterface->selectedCount()==0 ) {
		ui->btn_Commit->setEnabled(false);
		ui->actionClear_selections->setVisible(false);
	} else {
		ui->btn_Commit->setEnabled(true);
		ui->actionClear_selections->setVisible(true);
	}
}

void PackageView::on_btn_CategoryFilter_clicked()
{
	FilterSelect f("Category filter", this);
	f.setList( iCatFilterLabels, iSelectedCatFilter );
	if( f.exec() ) {
		iSelectedCatFilter = f.selection();
	}

	if( iSelectedCatFilter == CatFilterAllMarked ) {
		ui->btn_StatusFilter->setEnabled(false);
	} else {
		ui->btn_StatusFilter->setEnabled(true);
	}

	openWin(iOperation);
}

void PackageView::setStatFilter(Package::packageStatus f_)
{
	iSelectedStatFilter = f_;
}

void PackageView::on_actionStatus_filter_triggered()
{
	on_btn_StatusFilter_clicked();
}

void PackageView::on_btn_StatusFilter_clicked()
{
	FilterSelect f("Status filter", this);
	f.setList( iStatFilterLabels, iSelectedStatFilter );
	if( f.exec() ) {
		iSelectedStatFilter = (Package::packageStatus)f.selection();
	}

	openWin(iOperation);
}

void PackageView::resetWindow()
{
	iSelectedCatFilter = iDefaultCatFilter;
	ui->btn_StatusFilter->setEnabled(true);
}

void PackageView::on_actionHelp_triggered()
{
	Help h(this);
	h.exec();
}
