/*-*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4; tab-width: 4 -*- */
/*
 *    Copyright (C) 2009 Luca Vaudano vaudano@gmail.com
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License along
 *    with this program; if not, write to the Free Software Foundation, Inc.,
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.Warning
 */
 /**
  * @file estardict-gui-utils.c
  * @author Luca Vaudano
  */
#include "estardict-gui-utils.h"
#include "estardict-gui.h"
#include "../include/config.h"
#include <stdio.h>


/**
 * @brief Initialization for gettext
 *
 * @return Current locale
 */
char *init_locale (void) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    char *curlocale=setlocale(LC_ALL, "");
    //curlocale = curlocale? curlocale:setlocale(LC_ALL, "C");
    //mylocale = curlocale? strdup(curlocale):NULL;
    bindtextdomain(PACKAGE, LOCALEDIR);
    textdomain(PACKAGE);

    //return mylocale;
    return curlocale;
}


/**
 * @brief Show file selector
 *
 * @param data
 * @param obj
 * @param event_info
 * @return File selector object
 */
void showFileSelector(void *data, Evas_Object *obj, void *event_info) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    Evas_Object *win, *fs, *bg, *vbox, *hbox;

    win = elm_win_add(NULL, "fileselector", ELM_WIN_BASIC);
    elm_win_title_set(win, "File Selector");
    elm_win_autodel_set(win, 1);

    bg = elm_bg_add(win);
    elm_win_resize_object_add(win, bg);
    evas_object_size_hint_weight_set(bg, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
    evas_object_show(bg);

    vbox = elm_box_add(win);
    elm_win_resize_object_add(win, vbox);
    evas_object_size_hint_weight_set(vbox, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
    evas_object_show(vbox);

    fs = elm_fileselector_add(win);
    /* enable the fs file name entry */
    elm_fileselector_is_save_set(fs, EINA_TRUE);
    /* make the file list a tree with dir expandable in place */
    elm_fileselector_expandable_set(fs, EINA_FALSE);
    /* start the fileselector in the home dir */
    elm_fileselector_path_set(fs, getenv("HOME"));
    evas_object_size_hint_weight_set(fs, EVAS_HINT_EXPAND, EVAS_HINT_EXPAND);
    evas_object_size_hint_align_set(fs, EVAS_HINT_FILL, EVAS_HINT_FILL);
    elm_box_pack_end(vbox, fs);
    evas_object_show(fs);

    /* the 'done' cb is called when the user press ok/cancel */
    evas_object_smart_callback_add(fs, "done", data, win);

    hbox = elm_box_add(win);
    elm_box_horizontal_set(hbox, EINA_TRUE);
    elm_box_pack_end(vbox, hbox);
    evas_object_show(hbox);

    evas_object_resize(win, SCREEN_WIDTH, SCREEN_HEIGHT);
    evas_object_show(win);
}


/**
 * @brief Return the name of the file or of the last directory
 *
 * @param path Given path
 * @return Name of the file or of the last directory
 */
gchar* getNameFromPath(const gchar* path) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    gchar* result = NULL;

    if ( path != NULL && strlen(path) != 0 ) {

        // Rid of the slash, if it is the last character
        if (path[strlen(path) - 1] == '/') {
            result = g_strndup(path, strlen(path) - 1);
        } else {
            result = g_strdup(path);
        }

        // Take the name after last slash
        if (g_strrstr(result, "/") != NULL ) {
            result = g_strrstr(result, "/") + 1;
        }
    }

    return result;
}


/**
 * @brief Trim the given string
 *
 * @param str String to trim
 * @return Trimmed string
 */
gchar* trimwhitespace(gchar *str) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    gchar *end;

    // Trim leading space
    while ( g_ascii_isspace(*str) ) {
        str++;
    }

    // Trim trailing space
    end = str + strlen(str) - 1;
    while ( end > str && g_ascii_isspace(*end) ) {
        end--;
    }

  // Write new null terminator
  *(end+1) = 0;

  return str;
}


/**
 * @brief Format the definition
 *
 * @param definition The original definition
 * @return The formatted definition
 */
gchar* formatDefinition(gchar *definition) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);

    if ( definition == NULL)
        return definition;

    // Replace the new line with the <br>
    gchar**  tmp = g_strsplit_set(definition, "\n", -1);
    gchar* result = g_strjoinv("<br>", tmp);
    g_strfreev(tmp);

    return result;
}


/**
 * @brief Hide the dialog window
 *
 * @param data Dialog window
 * @param obj
 * @param event_info
 */
void dialogClicked(void *data, Evas_Object *obj, void *event_info) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    evas_object_hide(data);
}


/**
 * @brief Add spaces at the end of the string for the dialog text
 *
 * @return String with spaces
 */
gchar* addSpaces(gchar* text) {
    return text = g_strconcat(text, "<br><br><br>", NULL);
}


/**
 * @brief Create information dialog
 *
 * @param text String to display
 */
void createDialog(Evas_Object *win , gchar* text) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    g_debug("--> %s", text);
    Evas_Object *inwin, *label;

    inwin = elm_win_inwin_add(win);
    elm_object_style_set(inwin, "minimal");
    evas_object_show(inwin);

    // Internal box object
    Evas_Object *internalBox = elm_box_add(win);

    text = addSpaces(text);

    // Label
    label = elm_label_add(win);
    elm_object_text_set(label, text);
    elm_box_pack_end(internalBox, label);
    evas_object_show(label);

    // OK button
    Evas_Object *okButton = elm_button_add(win);
    elm_object_text_set(okButton, BUTTON_OK);
    evas_object_size_hint_weight_set(okButton, 0.0, 0.0);
    evas_object_size_hint_align_set(okButton, 0.5, 0.5);
    elm_box_pack_end(internalBox, okButton);
    evas_object_show(okButton);

    elm_win_inwin_content_set(inwin, internalBox);
    evas_object_show(internalBox);

    evas_object_smart_callback_add(okButton, "clicked", dialogClicked, inwin);
}


/**
 * @brief Return the word truncate using the WIDTH
 * constant defined in the configure.ac
 *
 * @param word Word to truncate
 * @return Word truncated, if needed
 */
gchar* truncateWord(gchar* word) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);

    if ( word != NULL && strlen(word) > WIDTH ) {
        gchar *result = g_strndup(word, WIDTH - 3);
        word = g_strconcat(result, "...", NULL);
    }
    g_debug("truncateWord %s", word);
    return word;
}


/**
 * @brief Check if a file exists
 *
 * @param filename Name of the file
 * @param True if the file exists, false otherwise
 */
gboolean checkFileExists(const char * filename) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    gboolean exist = FALSE;
    FILE * file = fopen(filename, "r");

    // If file exists
    if (file != NULL) {
        fclose(file);
        exist = TRUE;
    }

    return exist;
}


/**
 * @brief Dummy function for ignoring logging
 */
static void _dummy(const gchar *log_domain,
                     GLogLevelFlags log_level,
                     const gchar *message,
                     gpointer user_data )

{
  /* Dummy does nothing */
  return ;
}


/**
 * @brief Set log level based on the command line parameter
 *
 * * I don't use G_LOG_LEVEL_ERROR because it makes exit from the program. I will use:
 * G_LOG_LEVEL_CRITICAL Error that the UI must show to the user
 * G_LOG_LEVEL_WARNING Warning message
 * G_LOG_LEVEL_MESSAGE Info messages
 * G_LOG_LEVEL_DEBUG Debug messages: enter in every method
 *
 * @param argc
 * @param argv
 */
void setLogLevel(int argc, char **argv) {

    /* Set dummy for all levels */
    g_log_set_handler(G_LOG_DOMAIN, G_LOG_LEVEL_MASK, _dummy, NULL);

    /* Set default handler based on argument for appropriate log level */
    if ( argc > 1) {

         /* If -vv passed print all */
         if(!strncmp("-vv", argv[1], 3)) {
            g_log_set_handler(
                G_LOG_DOMAIN,
                //G_LOG_LEVEL_DEBUG | G_LOG_LEVEL_MESSAGE | G_LOG_LEVEL_WARNING | G_LOG_LEVEL_CRITICAL,
                G_LOG_LEVEL_MASK,
                g_log_default_handler,
                NULL
            );

         /* If -v passed don't print debug level */
         } else if(!strncmp("-v", argv[1], 2)) {
            g_log_set_handler(
                G_LOG_DOMAIN,
                //G_LOG_LEVEL_MESSAGE | G_LOG_LEVEL_WARNING | G_LOG_LEVEL_CRITICAL,
                ~(G_LOG_LEVEL_DEBUG),
                g_log_default_handler,
                NULL
            );

         /* For everything else, set to back to info level */
         } else {
            g_log_set_handler(
                G_LOG_DOMAIN,
                ~(G_LOG_LEVEL_DEBUG),
                g_log_default_handler,
                NULL
            );
         }

    /* If no arguments then set to ONLY warning & critical levels */
    } else {
        g_log_set_handler(
            G_LOG_DOMAIN,
            G_LOG_LEVEL_WARNING | G_LOG_LEVEL_CRITICAL,
            g_log_default_handler,
            NULL
        );
    }

    //g_critical("This is critical");
    //g_warning("This is warning");
    //g_message("This is message");
    //g_debug("This is debug");
}
