/*-*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4; tab-width: 4 -*- */
/*
 *    Copyright (C) 2009 Luca Vaudano vaudano@gmail.com
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License along
 *    with this program; if not, write to the Free Software Foundation, Inc.,
 *    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.Warning
 */
 /**
  * @file estardict-gui-logic.c
  * @author Luca Vaudano
  */
#include "estardict-gui-definitionPage.h"
#include "estardict-gui-dictionariesPage.h"
#include "estardict-gui-settingsPage.h"
#include "estardict-gui-infoPage.h"
#include "estardict-gui-logic.h"
#include "estardict-gui-utils.h"
#include "estardict-config.h"
#include "../engine/estardict-engine.h"
#include <time.h>
#include <stdio.h>
#include <errno.h>
#include <Eet.h>
#include <Ecore.h>

// Dictionaries information
Eina_List *starDictionariesInfo;

/**
 * @brief Destroy
 */
void destroy() {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    freeAllData();

    g_message("Shutdown eet/elm");
    eet_shutdown();
    elm_exit();
}


/**
 * @brief Promote the definition page box
 */
void promoteDefinitionPage() {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);

    // Create this box, if needed
    if ( definitionBox == NULL ) {
        createDefinitionPage();
    }

    // Hide other box pages
    if ( dictionariesBox != NULL ) {
        evas_object_hide(dictionariesBox);
    }
    if ( settingsBox != NULL ) {
        evas_object_hide(settingsBox);
    }
    if ( infoBox != NULL ) {
        evas_object_hide(infoBox);
    }

    // Show the promoted box
    elm_win_resize_object_add(win, definitionBox);
    evas_object_show(definitionBox);
}


/**
 * @brief Promote the dictionaries page box
 */
void promoteDictionariesPage() {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);

    // Create this box, if needed
    if ( dictionariesBox == NULL ) {
        createDictionariesPage();
    }

    // Hide other box pages
    if ( definitionBox != NULL ) {
        evas_object_hide(definitionBox);
    }
    if ( settingsBox != NULL ) {
        evas_object_hide(settingsBox);
    }
    if ( infoBox != NULL ) {
        evas_object_hide(infoBox);
    }

    // Set dictionaries in the list
    Eina_List* dictionaries = getDictionaries();
    addDictionaries(dictionaries);

    // Show the promoted box
    elm_win_resize_object_add(win, dictionariesBox);
    evas_object_show(dictionariesBox);

}


/**
 * @brief Promote the settings page box
 */
void promoteSettingsPage() {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);

    // Create this box, if needed
    if ( settingsBox == NULL ) {
        createSettingsPage();
    }

    // Hide other box pages
    if ( definitionBox != NULL ) {
        evas_object_hide(definitionBox);
    }
    if ( dictionariesBox != NULL ) {
        evas_object_hide(dictionariesBox);
    }
    if ( infoBox != NULL ) {
        evas_object_hide(infoBox);
    }

    const gchar* soundDictionaryPath = getSoundDictionary();
    gchar* soundDictionaryName = getNameFromPath(soundDictionaryPath);
    setSoundDictionaryEntry(truncateWord(soundDictionaryName));

    const gchar* playerPath = getPlayer();
    gchar* playerName = getNameFromPath(playerPath);
    setPlayerEntry(truncateWord(playerName));

    // Show the promoted box
    elm_win_resize_object_add(win, settingsBox);
    evas_object_show(settingsBox);
}


/**
 * @brief Promote the about page box
 */
void promoteAboutPage() {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    Type about = ABOUT;

    // Create this box always because we have
    // an input parameter
    // it's ok no so used page
    createInfoPage(about);

    // Hide other box pages
    if ( definitionBox != NULL ) {
        evas_object_hide(definitionBox);
    }
    if ( dictionariesBox != NULL ) {
        evas_object_hide(dictionariesBox);
    }
    if ( settingsBox != NULL ) {
        evas_object_hide(settingsBox);
    }

    // Show the promoted box
    elm_win_resize_object_add(win, infoBox);
    evas_object_show(infoBox);
}


/**
 * @brief Promote the first usage page box
 */
void promoteFirstUsagePage() {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    Type firstUsage = FIRST_USAGE;

    // Create this box always because we have
    // an input parameter
    // it's ok no so used page
    createInfoPage(firstUsage);

    // Hide other box pages
    if ( definitionBox != NULL ) {
        evas_object_hide(definitionBox);
    }
    if ( dictionariesBox != NULL ) {
        evas_object_hide(dictionariesBox);
    }
    if ( settingsBox != NULL ) {
        evas_object_hide(settingsBox);
    }

    // Show the promoted box
    elm_win_resize_object_add(win, infoBox);
    evas_object_show(infoBox);
}


/**
 * @brief Save sound dictionary button clicked
 *
 * @param data
 * @param obj
 * @param event_info
 */
void saveSoundDictionary(gchar* soundDictionaryPath) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    g_message("Save sound dictionary in the configuration file");
    setSoundDictionary(soundDictionaryPath);
}

/**
 * @brief Method called on file selection for the dictionary
 *
 * @param data Parent window
 * @param obj
 * @param event_info contain the full path of the selected file
 * or NULL if none is selected or cancel is pressed
 */
static void newDictionarySelect(void *data, Evas_Object *obj, void *event_info) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    const gchar* dictionaryPath;
    const char *selected = event_info;

    if (selected) {

        // If it is a directory
        if ( selected[strlen(selected)-1] == '/' ) {
            dictionaryPath = g_strndup(selected, strlen(selected)-1) ;
            addDictionaryToGenlist(dictionaryPath);

             // Delete the parent window
            evas_object_del(data);
        } else {
            g_message("You have to select the dictionary directory.");
            createDialog( (Evas_Object *)data, addSpaces(DIALOG_SELECT_DICTIONARY) );
        }

    // Cancel from file selector
    } else {

        // Delete the parent window
        evas_object_del(data);
    }

}


/**
 * @brief Method called on file selection for the sound dictionary
 *
 * @param data Parent window
 * @param obj
 * @param event_info contain the full path of the selected file
 * or NULL if none is selected or cancel is pressed
 */
static void newSoundDictionarySelect(void *data, Evas_Object *obj, void *event_info) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    gchar* soundDictionaryPath;
    const char *selected = event_info;

    if (selected) {

        // If it is a directory
        if ( selected[strlen(selected)-1] == '/' ) {
            soundDictionaryPath =  g_strdup(selected);
            gchar* soundDictionaryName = getNameFromPath(soundDictionaryPath);
            setSoundDictionaryEntry(truncateWord(soundDictionaryName));

            g_message("soundDictionaryPath %s", soundDictionaryPath);

            saveSoundDictionary(soundDictionaryPath);

            // In any case delete the parent window
            evas_object_del(data);

        } else {
            g_message("You have to select a dictionary. The directory should contain folders a, b, c and so on");
            createDialog( (Evas_Object *)data, addSpaces(DIALOG_SELECT_SOUND_DICTIONARY) );
        }
    } else {

        // In any case delete the parent window
        evas_object_del(data);
    }
}


/**
 * @brief Method called on file selection for the player
 *
 * @param data Parent window
 * @param obj
 * @param event_info contain the full path of the selected file
 * or NULL if none is selected or cancel is pressed
 */
static void newPlayerSelect(void *data, Evas_Object *obj, void *event_info) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    gchar* playerPath;
    const gchar *selected = (gchar *)event_info;

    if (selected) {

        // If it is a directory
        if ( selected[strlen(selected)-1] == '/' ) {
            g_message("You have to select a executable file");
            createDialog( (Evas_Object *)data, addSpaces(DIALOG_SELECT_PLAYER) );

        } else {
            playerPath = g_strdup(selected) ;
            gchar* playerName = getNameFromPath(playerPath);
            setPlayerEntry(truncateWord(playerName));

            // In any case delete the parent window
            evas_object_del(data);
        }

    } else {

        // In any case delete the parent window
        evas_object_del(data);
    }
}


/**
 * @brief New dictionary button clicked
 *
 * @param data
 * @param obj
 * @param event_info
 */
void addDictionary(void *data, Evas_Object *obj, void *event_info) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    showFileSelector(newDictionarySelect, obj, event_info);
}


/**
 * @brief New dictionary button clicked
 *
 * @param data
 * @param obj
 * @param event_info
 */
void addSoundDictionary(void *data, Evas_Object *obj, void *event_info) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    showFileSelector(newSoundDictionarySelect, obj, event_info);
}


/**
 * @brief New player button clicked
 *
 * @param data
 * @param obj
 * @param event_info
 */
void addPlayer(void *data, Evas_Object *obj, void *event_info) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    showFileSelector(newPlayerSelect, obj, event_info);
}


/**
 * @brief Show a list of suggested words
 *
 * @param data
 * @param obj
 * @param event_info
 */
void showSuggestedWords(void *data, Evas_Object *obj, void *event_info) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);

    const Eina_List *l;
    StarDictInfo* starDictInfo = NULL;
       
    if ( getSuggestedWordShow() ) {

        gchar* word = g_strdup(elm_entry_entry_get(data));

        if ( starDictionariesInfo != NULL && eina_list_count( starDictionariesInfo ) != 0 ) {

            // Clear previous suggested words
            clearSuggestedWordList();

            EINA_LIST_FOREACH(starDictionariesInfo, l, starDictInfo) {
       
                if ( starDictInfo != NULL &&
                    word != NULL &&
                    strlen(word) > 0
                ) {
                    GArray* list = NULL;
                    guint i = 0;

                    list = getWordList(starDictInfo, word, MAX_NUMBER_SUGGESTED_WORDS);

                    for ( i = 0; i < list->len; i++) {
                        gchar* word = g_array_index(list, gchar*, i);
                        addSuggestedWord(word);
                        //g_debug("Current item is %s", word);
                        g_free(word);
                    }
                    g_array_free(list, TRUE);

                    showSuggestedWordBox();
                } else {
                    hideSuggestedWordBox();
                }
            }
        }
    }
}


/**
 * @brief Pronounce the inserted word
 *
 * @param data
 * @param obj
 * @param event_info
 */
void speak(void *data, Evas_Object *obj, void *event_info) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    gchar filename[PATH_MAX];
    gchar* word = g_strdup(elm_entry_entry_get(data));

    if ( word != NULL && strlen(word)>0 ) {
        g_message("Speak %s\n", word);

        // Player from configuration
        const gchar* playerPath = getPlayer();
        if ( playerPath == NULL || strlen(playerPath) == 0 ) {
            g_message("You have to select the sound player.");
            createDialog( win, DIALOG_MUST_SELECT_PLAYER);
            return;
        }

        // Sound dictionary from configuration
        const gchar* soundDictionaryPath = getSoundDictionary();
        if ( soundDictionaryPath == NULL || strlen(soundDictionaryPath) == 0 ) {
            g_message("You have to select the sound dictionary directory.");
            createDialog( win, DIALOG_MUST_SELECT_SOUND_DICTIONARY);
            return;
        }

        // Speak
        word = trimwhitespace(word);
        // Use sprinf instead of g_strconcat because word[0] is a char
        sprintf(filename, "%s%c/%s.wav", soundDictionaryPath, word[0], word);
        g_debug("--> File name %s", filename);

        // If file exists
        if ( checkFileExists(filename) == TRUE ) {
            gchar* buf = g_strconcat(playerPath, " ", filename, NULL);
            g_message("--> Command %s", buf);
            int returnValue = system(buf);

            if ( returnValue != 0 ) {
                g_critical("Error during playing the file.");
                createDialog( win, DIALOG_ERROR_PLAYING_FILE );
            }

        } else {
            g_message("Word not found.");
            createDialog( win, DIALOG_WORD_NOT_FOUND );
            return;
        }
    }
}


/**
 * @brief Clear word entry
 *
 * @param data
 * @param obj
 * @param event_info
 */
void clearWord(void *data, Evas_Object *obj, void *event_info) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    elm_entry_entry_set(data, "");
    clearDefinition();
}


/**
 * @brief Decode the error code from the engine, if present
 */
void checkForEngineDefinitionError() {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);

    switch ( estardictEngineError ) {

        case ESTARDICT_ENGINE_ERROR_NO_ERROR: {
            g_debug("Engine no error");
            break;
        }

        case ESTARDICT_ENGINE_ERROR_MEMORY_ERROR_DEFINITION: {
            g_critical("Memory error.");
            createDialog( win, DIALOG_ERROR_MEMORY );
            break;
        }

        case ESTARDICT_ENGINE_ERROR_OPEN_DICT: {
            g_critical("Error open DICT file.");
            createDialog(win, DIALOG_ERROR_OPEN_DICT_FILE );
            break;
        }

        case ESTARDICT_ENGINE_ERROR_SEEKING_DICT: {
            g_critical("Error seeking on DICT file.");
            createDialog( win, DIALOG_ERROR_SEEKING_DICT_FILE);
            break;
        }

        case ESTARDICT_ENGINE_ERROR_READING_DICT: {
            g_critical("Error reading DICT file.");
            createDialog( win, DIALOG_ERROR_READING_DICT_FILE);
            break;
        }

        case ESTARDICT_ENGINE_ERROR_PARSING_DEFINITION: {
            g_critical("Error parsing definition.");
            createDialog( win, DIALOG_ERROR_PARSING_DEFINITION);
            break;
        }

        case ESTARDICT_ENGINE_ERROR_OPEN_IDX: {
            g_critical("Error open IDX file.");
            createDialog( win, DIALOG_ERROR_OPEN_IDX_FILE);
            break;
        }

        case ESTARDICT_ENGINE_ERROR_READING_IDX: {
            g_critical("Error reading IDX file.");
            createDialog( win, DIALOG_ERROR_READING_IDX_FILE);
            break;
        }
        
        case ESTARDICT_ENGINE_ERROR_MEMORY_ERROR_INFO: {
            g_critical("Memory error.");
            createDialog( win, DIALOG_ERROR_MEMORY);
            break;
        }

        default: {
            g_critical("Cannot go here!");
            break;
        }
    }
}


/**
 * @brief Search the given word
 *
 * @param data
 * @param obj
 * @param event_info
 */
void searchWord(void *data, Evas_Object *obj, void *event_info) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    const Eina_List *l;
    StarDictInfo* starDictInfo = NULL;
    gchar* dictionanyName = NULL;
    gboolean wordFound = FALSE;
       
    hideSuggestedWordBox();

    gchar* word = g_strdup(elm_entry_entry_get(data));

    if ( word != NULL && strlen(word)>0 ) {
        clearDefinition();
        g_message("Search %s", word);

        if ( starDictionariesInfo != NULL && eina_list_count( starDictionariesInfo ) != 0 ) {

            EINA_LIST_FOREACH(starDictionariesInfo, l, starDictInfo) {
				
                if ( starDictInfo != NULL ) {
                    gchar* definition = getDefinition(starDictInfo, trimwhitespace(word));

                    if ( definition != NULL ) {
                        wordFound = TRUE;
                        gchar* formattedDefinition = formatDefinition(definition);
                                  
                        // Get dictionary name
                        if ( starDictInfo->bookname != NULL ) {
                            dictionanyName = truncateWord(starDictInfo->bookname);
                        } else {
                            g_message("Dictionary bookname not set");
                        }
                        setDefinition(dictionanyName, formattedDefinition);
                    } else {
                        checkForEngineDefinitionError();
                    }

                } else {
                    g_message("Dictionary not loaded.");
                    createDialog( win, DIALOG_DICTIONARY_NOT_LOADED);
                    return;
                }
            }
            if ( wordFound == FALSE ) {
                g_message("Word not found.");
                createDialog( win, DIALOG_WORD_NOT_FOUND );
            }
                     
        } else {
            g_message("You have to set the dictionary directory.");
            createDialog( win, DIALOG_MUST_SELECT_DICTIONARY);
        }
    }
}


/**
 * @brief Decode the error code from the engine
 *
 * @param starDictInfo Dictionary information
 * @return TRUE in case of error, false otherwise
 */
gboolean checkForEngineInitializationError(StarDictInfo* starDictInfo) {
    gboolean errorPresent = TRUE;

    if ( starDictInfo == NULL ) {

        switch ( estardictEngineError ) {

            case ESTARDICT_ENGINE_ERROR_NO_ERROR: {
                g_message("No error.");
                break;
            }

            case ESTARDICT_ENGINE_ERROR_FIRST_LINE_IFO: {
                g_critical("IFO file has not valid format.");
                createDialog( win, DIALOG_IFO_NOT_VALID);
                break;
            }

            case ESTARDICT_ENGINE_ERROR_SECOND_LINE_IFO: {
                g_critical("IFO file has not valid format 2.");
                createDialog( win, DIALOG_IFO_NOT_VALID);
                break;
            }

            case ESTARDICT_ENGINE_ERROR_OPEN_DIRECTORY: {
                g_critical("Cannot open dictionary directory.");
                createDialog( win, DIALOG_NO_OPEN_DICTIONARY_DIR);
                break;
            }

            case ESTARDICT_ENGINE_ERROR_MANDATORY_FILES_NOT_PRESENT: {
                g_critical("Mandatory files not present.");
                createDialog( win, DIALOG_NO_MANDATORY_FILES);
                break;
            }

            case ESTARDICT_ENGINE_ERROR_OPEN_IDX: {
                g_critical("Error open IDX file.");
                createDialog( win, DIALOG_ERROR_OPEN_IDX_FILE);
                break;
            }

            case ESTARDICT_ENGINE_ERROR_DICTIONARY_FORMAT_NOT_SUPPORTED: {
                g_critical("Dictionary format not supported yet.");
                createDialog( win, DIALOG_DICTIONARY_NOT_SUPPORTED);
                break;
            }

            case ESTARDICT_ENGINE_ERROR_64BIT_DICTIONARY_NOT_SUPPORTED: {
                g_critical("eStarDict doesn't work with 64bit dictionary.");
                createDialog( win, DIALOG_NO_64BIT_DICTIONARY);
                break;
            }

            case ESTARDICT_ENGINE_ERROR_OPEN_DICT: {
                g_critical("Error open DICT file.");
                createDialog(win, DIALOG_ERROR_OPEN_DICT_FILE );
                break;
            }

            default: {
                g_critical("Error loading the dictionary.");
                createDialog( win, DIALOG_ERROR_LOADING_DICTIONARY);
                break;
            }

        }

    // StarDict is not NULL
    } else {
        errorPresent = FALSE;
    }

    return errorPresent;
}


/**
 * @brief Decode the warning code from the engine - Not used
 * 
 * @param starDictInfo Dictionary information
 */
void checkForEngineInitializationWarnings(StarDictInfo* starDictInfo) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);

    if ( starDictInfo != NULL && starDictInfo->warnings != NULL ) {
        GList *li;
        gchar* text = NULL;


        for (li = starDictInfo->warnings; li != NULL; li = li->next) {

            if ( GPOINTER_TO_INT(li->data) == ESTARDICT_ENGINE_WARNINGS_IDXSIZE_NO_MATCH) {
                if ( text != NULL ) {
                    text = g_strconcat(text, "Idx file size doesn't match<br>", NULL);
                } else {
                    text = g_strconcat("Idx file size doesn't match<br>", NULL);
                }

            } else if ( GPOINTER_TO_INT(li->data) == ESTARDICT_ENGINE_WARNINGS_WORD_COUNT_NO_MATCH) {
                if ( text != NULL ) {
                    text = g_strconcat(text, "Word count doesn't match<br>", NULL);
                } else {
                    text = g_strconcat("Word count doesn't match<br>", NULL);
                }
            }

        }

        if (text != NULL) {
            text = g_strconcat("Warnings:<br>", text, "<br><br><br>", NULL);
            createDialog( win, text);
        }
    }
}


/**
 * @brief Load the dictionary from the engine
 * This method is called in a separated thread
 *
 * @param dictPath Dictionary path
 * @return If success, flag for returning to the main thread
 **/
void startLoadDictionary(const gchar *dictPath) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    gchar* path;
    path = (gchar *)dictPath;
    g_message("Dictionary path %s", path);
    
    StarDictInfo* starDictInfo = loadDictionary(path);
    
    if ( starDictInfo != NULL ) {
        starDictionariesInfo = eina_list_append(starDictionariesInfo, starDictInfo);
    }

    // If no error
    //if ( !checkForEngineInitializationError(starDictInfo) ) {
        // No warning dialog
        // checkForEngineInitializationWarnings(starDictInfo);
    //}
}


/**
 * @brief Load  dictionary
 */
void initializeDictionaries() {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    const Eina_List* l;
    gboolean empty = FALSE;
    DictionaryProperties *dictionary;
    Eina_List* dictionaries = getDictionaries();

    if ( dictionaries != NULL && eina_list_count( dictionaries ) != 0 ) {

        EINA_LIST_FOREACH(dictionaries, l, dictionary) {
            g_message("Dictionaries %s", dictionary->dictionaryPath);
            if ( strlen(dictionary->dictionaryPath) != 0 ) {
                startLoadDictionary(dictionary->dictionaryPath);
            } else {
                g_message("Empty dictionary path");
                empty = TRUE;
            }
        }

    } else {
        setTipsOnStartup();
    }

    if ( eina_list_count( dictionaries ) == 1 && empty ) {
        setTipsOnStartup();
    }
}


/**
 * @brief Save dictionary button clicked
 *
 * @param data
 * @param obj
 * @param event_info
 */
void saveDictionaries(void *data, Evas_Object *obj, void *event_info) {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    
    // Clean global variable with all dictionaries
    eina_list_free(starDictionariesInfo);
    starDictionariesInfo = NULL;
    
    // Get dictionaries from GUI
    Eina_List *dictionaries = getDictionaryList();
         
    g_message("Save dictionaries in the configuration file");
    if ( setDictionaries(dictionaries) ) {
        initializeDictionaries();
    }       
       
    promoteDefinitionPage();
}


/**
 * @brief Initialize application
 */
void init() {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    gboolean result = initializeConfiguration();

    if ( result ) {
        initializeDictionaries();
    } else {
        g_critical("Error loading the configuration");
        createDialog( win, DIALOG_ERROR_LOADING_CONFIGURATION);
    }
}


/**
 * @brief Memorize suggestedWordShow value
 *
 * @param data
 * @param obj Suggested word toggle
 * @param event_info
 */
void suggestedWordToggleChanged(void *data, Evas_Object *obj, void *event_info) {
    setSuggestedWordShow( elm_check_state_get(obj) );
}


/**
 * brief Quit without saving from the dictionary list screen
 *
 * @param data
 * @param obj Suggested word toggle
 * @param event_info
 */
void quitDictionaries(void *data, Evas_Object *obj, void *event_info) {

    // Force reload from the configuration file
    clearDictionaryList();
    promoteDefinitionPage();
}


/**
 * @brief Free all data
 */
void freeAllData() {
    g_debug("-> %s %s()", __FILE__, __FUNCTION__);
    const Eina_List *l;
    StarDictInfo* starDictInfo = NULL;
    
    EINA_LIST_FOREACH(starDictionariesInfo, l, starDictInfo) {
        freeData(starDictInfo);
    }

    // Clear dictionary list
    clearDictionaryList();

    // Close configuration data
    if ( !closeConfigFile() ) {
        g_warning("Error saving configuration file.");
    }
}
