#include <unistd.h>
#include <string.h>
#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <hildon-cp-plugin/hildon-cp-plugin-interface.h>
#include <gconf/gconf-client.h>
#include "config.h"

/* TODO: read these from disk */

static const struct voice {
	const char *id;
	const char *name;
} voices[] = {
	{ "",		"Default" },
	{ "en",		"English" },
	{ "en-us",	"English (American)" },
	{ "en-sc",	"English (Scottish)" },
	{ "af",		"Afrikaans" },
	{ "bs",		"Bosnian" },
	{ "ca",		"Catalan" },
	{ "cs",		"Czech" },
	{ "de",		"German" },
	{ "el",		"Greek" },
	{ "eo",		"Esperanto" },
	{ "es",		"Spanish" },
	{ "es-la",	"Spanish (Latin America)" },
	{ "fi",		"Finnish" },
	{ "fr",		"French" },
	{ "hr",		"Croatian" },
	{ "hu",		"Hungarian" },
	{ "it",		"Italian" },
	{ "ku",		"Kurdish" },
	{ "lv",		"Latvian" },
	{ "pl",		"Polish" },
	{ "pt",		"Portuguese (Brazil)" },
	{ "pt-pt",	"Portuguese (European)" },
	{ "ro",		"Romanian" },
	{ "sk",		"Slovak" },
	{ "sr",		"Serbian" },
	{ "sv",		"Swedish" },
	{ "sw",		"Swahihi" },
	{ "ta",		"Tamil" },
	{ "tr",		"Turkish" },
	{ "zh",		"Mandarin Chinese" },
};
static const int num_voices = sizeof(voices) / sizeof(voices[0]);

static const struct effect {
	const char *id;
	const char *name;
} effects[] = {
	{ "",		"(none)" },
	{ "f1",		"Female 1" },
	{ "f2",		"Female 2" },
	{ "f3",		"Female 3" },
	{ "f4",		"Female 4" },
	{ "m1",		"Male 1" },
	{ "m2",		"Male 2" },
	{ "m3",		"Male 3" },
	{ "m4",		"Male 4" },
	{ "m5",		"Male 5" },
	{ "m6",		"Male 6" },
	{ "m7",		"Male 7" },
	{ "croak",	"Croak" },
	{ "fast",	"Fast" },
	{ "klatt",	"Klatt" },
	{ "klatt2",	"Klatt 2" },
	{ "klatt3",	"Klatt 3" },
	{ "whisper",	"Whisper" },
};
static const int num_effects = sizeof(effects) / sizeof(effects[0]);

static const gchar help_message[] =
	"The Speech string can be any string accepted by strftime(3). "
	"The default is \"%H:%M\".  An excerpt from the strftime man page is "
	"provided below:\n"
	"\n\n"
	"Ordinary characters placed in the format string are copied to the output "
	"without conversion. Conversion specifications are introduced by a '%' "
	"character, and terminated by a conversion specifier character, and are "
	"replaced in the output as follows:\n"
	"\n"
	"%a - The abbreviated weekday name according to the current locale. \n"
	"\n"
	"%A - The full weekday name according to the current locale. \n"
	"\n"
	"%b - The abbreviated month name according to the current locale. \n"
	"\n"
	"%B - The full month name according to the current locale. \n"
	"\n"
	"%c - The preferred date and time representation for the current locale. \n"
	"\n"
	"%C - The century number (year/100) as a 2-digit integer. (SU) \n"
	"\n"
	"%d - The day of the month as a decimal number (range 01 to 31). \n"
	"\n"
	"%D - Equivalent to %m/%d/%y. (Yecch -- for Americans only. Americans "
	"should note that in other countries %d/%m/%y is rather common. This "
	"means that in international context this format is ambiguous and should "
	"not be used.) (SU) \n"
	"\n"
	"%e - Like %d, the day of the month as a decimal number, but a leading "
	"zero is replaced by a space. (SU) \n"
	"\n"
	"%E - Modifier: use alternative format, see below. (SU) \n"
	"\n"
	"%F - Equivalent to %Y-%m-%d (the ISO 8601 date format). (C99) \n"
	"\n"
	"%G - The ISO 8601 year with century as a decimal number. The 4-digit "
	"year corresponding to the ISO week number (see %V). This has the same "
	"format and value as %y, except that if the ISO week number belongs to "
	"the previous or next year, that year is used instead. (TZ) \n"
	"\n"
	"%g - Like %G, but without century, i.e., with a 2-digit year (00-99).\n"
	"(TZ) \n"
	"\n"
	"%h - Equivalent to %b. (SU) \n"
	"\n"
	"%H - The hour as a decimal number using a 24-hour clock (range 00 to 23). \n"
	"\n"
	"%I - The hour as a decimal number using a 12-hour clock (range 01 to 12). \n"
	"\n"
	"%j - The day of the year as a decimal number (range 001 to 366). \n"
	"\n"
	"%k - The hour (24-hour clock) as a decimal number (range 0 to 23); "
	"single digits are preceded by a blank. (See also %H.) (TZ) \n"
	"\n"
	"%l - The hour (12-hour clock) as a decimal number (range 1 to 12); "
	"single digits are preceded by a blank. (See also %I.) (TZ) \n"
	"\n"
	"%m - The month as a decimal number (range 01 to 12). \n"
	"\n"
	"%M - The minute as a decimal number (range 00 to 59). \n"
	"\n"
	"%n - A newline character. (SU) \n"
	"\n"
	"%O - Modifier: use alternative format, see below. (SU) \n"
	"\n"
	"%p - Either 'AM' or 'PM' according to the given time value, or the "
	"corresponding strings for the current locale. Noon is treated as 'pm' "
	"and midnight as 'am'. \n"
	"\n"
	"%P - Like %p but in lowercase: 'am' or 'pm' or a corresponding string "
	"for the current locale. (GNU) \n"
	"\n"
	"%r - The time in a.m. or p.m. notation. In the POSIX locale this is "
	"equivalent to '%I:%M:%S %p'. (SU) \n"
	"\n"
	"%R - The time in 24-hour notation (%H:%M). (SU) For a version including "
	"the seconds, see %T below. \n"
	"\n"
	"%s - The number of seconds since the Epoch, i.e., since 1970-01-01 "
	"00:00:00 UTC. (TZ) \n"
	"\n"
	"%S - The second as a decimal number (range 00 to 60). (The range is up "
	"to 60 to allow for occasional leap seconds.) \n"
	"\n"
	"%t - A tab character. (SU) \n"
	"\n"
	"%T - The time in 24-hour notation (%H:%M:%S). (SU) \n"
	"\n"
	"%u - The day of the week as a decimal, range 1 to 7, Monday being 1. See "
	"also %w. (SU) \n"
	"\n"
	"%U - The week number of the current year as a decimal number, range 00 "
	"to 53, starting with the first Sunday as the first day of week 01. See "
	"also %V and %W. \n"
	"\n"
	"%V - The ISO 8601:1988 week number of the current year as a decimal "
	"number, range 01 to 53, where week 1 is the first week that has at least "
	"4 days in the current year, and with Monday as the first day of the "
	"week. See also %U and %W. (SU) \n"
	"\n"
	"%w - The day of the week as a decimal, range 0 to 6, Sunday being 0. See "
	"also %u. \n"
	"\n"
	"%W - The week number of the current year as a decimal number, range 00 "
	"to 53, starting with the first Monday as the first day of week 01. \n"
	"\n"
	"%x - The preferred date representation for the current locale without "
	"the time. \n"
	"\n"
	"%X - The preferred time representation for the current locale without "
	"the date. \n"
	"\n"
	"%y - The year as a decimal number without a century (range 00 to 99). \n"
	"\n"
	"%Y - The year as a decimal number including the century. \n"
	"\n"
	"%z - The time-zone as hour offset from GMT. Required to emit RFC "
	"822-conformant dates (using \"%a, %d %b %Y %H:%M:%S %z\"). (GNU) \n"
	"\n"
	"%Z - The time zone or name or abbreviation. \n"
	"\n"
	"%+ - The date and time in date(1) format. (TZ) (Not supported in "
	"glibc2.) \n"
	"\n"
	"%% - A literal '%' character.\n"
	"\n"
	"\n"
	"Some conversion specifications can be modified by preceding the "
	"conversion specifier character by the E or O modifier to indicate that "
	"an alternative format should be used. If the alternative format or "
	"specification does not exist for the current locale, the behaviour will "
	"be as if the unmodified conversion specification were used. (SU) The "
	"Single Unix Specification mentions %Ec, %EC, %Ex, %EX, %Ey, %EY, %Od, "
	"%Oe, %OH, %OI, %Om, %OM, %OS, %Ou, %OU, %OV, %Ow, %OW, %Oy, where the "
	"effect of the O modifier is to use alternative numeric symbols (say, "
	"roman numerals), and that of the E modifier is to use a locale-dependent "
	"alternative representation.\n";



static void show_help(GtkButton *button, GtkWidget *parent)
{
	GtkWidget *dialog, *label, *panarea;

	dialog = gtk_dialog_new_with_buttons(
		"eSpeakTime Help",
		GTK_WINDOW(parent),
		GTK_DIALOG_MODAL | GTK_DIALOG_NO_SEPARATOR,
		GTK_STOCK_OK, GTK_RESPONSE_OK,
		NULL);

	label = gtk_label_new(help_message);
	gtk_label_set_use_markup(GTK_LABEL(label), TRUE);
	gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);

	panarea = hildon_pannable_area_new();
	gtk_widget_set_size_request(panarea, -1, 800);
	hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA(panarea), label);
	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), panarea);

	g_signal_connect_swapped(dialog, "response", G_CALLBACK(gtk_widget_destroy), dialog);
	gtk_widget_show_all(dialog);
}

static void add_scale(GtkVBox *vbox, GtkSizeGroup *size_group, const char *caption, GtkAdjustment *adjustment)
{
	GtkWidget *scale = gtk_hscale_new(adjustment);
	gtk_scale_set_digits(GTK_SCALE(scale), 0);
	gtk_box_pack_start(GTK_BOX(vbox),
		hildon_caption_new(size_group, caption, scale, NULL, HILDON_CAPTION_MANDATORY),
		FALSE, FALSE, 0);
}

osso_return_t execute(osso_context_t *osso, gpointer data, gboolean user_activated)
{
	struct espeaktime_settings cfg = {
		.voice = "en-us",
		.effect = "",
		.amplitude = 100,
		.pitch = 50,
		.speed = 170,
		.ignore_silent = TRUE,
	};

	GConfClient *client = gconf_client_get_default();
	cfg_read(client, &cfg);

	GtkWidget *dialog;
	dialog = gtk_dialog_new_with_buttons(
		"eSpeakTime Settings",
		GTK_WINDOW(data),
		GTK_DIALOG_MODAL | GTK_DIALOG_NO_SEPARATOR,
		"Test", 1,
		GTK_STOCK_SAVE, GTK_RESPONSE_OK,
		GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
		NULL);

	int k, voice_sel = 0, effect_sel = 0;
	GtkVBox *vbox = GTK_VBOX(gtk_vbox_new(FALSE, 0));
	GtkSizeGroup *title_group = gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL);
	GtkSizeGroup *value_group = gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL);

	/* selectors */
	HildonTouchSelector *voice_selector = HILDON_TOUCH_SELECTOR(hildon_touch_selector_new_text());
	for (k = 0; k < num_voices; k++) {
		hildon_touch_selector_append_text(voice_selector, voices[k].name);
		if (!strcmp(voices[k].id, cfg.voice))
			voice_sel = k;
	}

	HildonTouchSelector *effect_selector = HILDON_TOUCH_SELECTOR(hildon_touch_selector_new_text());
	for (k = 0; k < num_effects; k++) {
		hildon_touch_selector_append_text(effect_selector, effects[k].name);
		if (!strcmp(effects[k].id, cfg.effect))
			effect_sel = k;
	}

	HildonPickerButton *voice = HILDON_PICKER_BUTTON(hildon_picker_button_new(HILDON_SIZE_FINGER_HEIGHT, HILDON_BUTTON_ARRANGEMENT_HORIZONTAL));
	hildon_button_set_title(HILDON_BUTTON(voice), "Voice");
	hildon_picker_button_set_selector(voice, voice_selector);
	hildon_picker_button_set_active(voice, voice_sel);
	hildon_button_add_size_groups(HILDON_BUTTON(voice), title_group, value_group, NULL);
	gtk_button_set_alignment(GTK_BUTTON(voice), 0.0, 0.5);
	gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(voice), FALSE, FALSE, 0);

	HildonPickerButton *effect = HILDON_PICKER_BUTTON(hildon_picker_button_new(HILDON_SIZE_FINGER_HEIGHT, HILDON_BUTTON_ARRANGEMENT_HORIZONTAL));
	hildon_button_set_title(HILDON_BUTTON(effect), "Effect");
	hildon_picker_button_set_selector(effect, effect_selector);
	hildon_picker_button_set_active(effect, effect_sel);
	hildon_button_add_size_groups(HILDON_BUTTON(effect), title_group, value_group, NULL);
	gtk_button_set_alignment(GTK_BUTTON(effect), 0.0, 0.5);
	gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(effect), FALSE, FALSE, 0);

	GtkWidget *hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	HildonEntry *text = HILDON_ENTRY(hildon_entry_new(HILDON_SIZE_AUTO));
	gtk_entry_set_text(GTK_ENTRY(text), "%H:%M");
	gtk_box_pack_start(GTK_BOX(hbox),
		hildon_caption_new(title_group, "Speech string", GTK_WIDGET(text), NULL, HILDON_CAPTION_MANDATORY),
		TRUE, TRUE, 0);

	GtkWidget *help_btn = hildon_button_new_with_text(HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT,
		HILDON_BUTTON_ARRANGEMENT_HORIZONTAL, " ? ", NULL);
	g_signal_connect(help_btn, "clicked", G_CALLBACK(show_help), dialog);
	gtk_box_pack_start(GTK_BOX(hbox), help_btn, FALSE, FALSE, 0);

	GtkAdjustment *amplitude_adj = GTK_ADJUSTMENT(gtk_adjustment_new(cfg.amplitude, 0, 200, 1, 10, 0));
	add_scale(vbox, title_group, "Amplitude", amplitude_adj);

	GtkAdjustment *pitch_adj = GTK_ADJUSTMENT(gtk_adjustment_new(cfg.pitch, 00, 99, 1, 10, 0));
	add_scale(vbox, title_group, "Pitch", pitch_adj);

	GtkAdjustment *speed_adj = GTK_ADJUSTMENT(gtk_adjustment_new(cfg.speed, 80, 390, 1, 10, 0));
	add_scale(vbox, title_group, "Speed (wpm)", speed_adj);

	HildonCheckButton *ignore_silent = HILDON_CHECK_BUTTON(hildon_check_button_new(HILDON_SIZE_FINGER_HEIGHT));
	hildon_check_button_set_active(ignore_silent, cfg.ignore_silent);
	gtk_button_set_label(GTK_BUTTON(ignore_silent), "Ignore silent profile");
	gtk_box_pack_start(GTK_BOX(vbox), GTK_WIDGET(ignore_silent), FALSE, FALSE, 0);

	GtkWidget *panarea = hildon_pannable_area_new();
	gtk_widget_set_size_request(panarea, -1, 800);
	hildon_pannable_area_add_with_viewport(HILDON_PANNABLE_AREA(panarea), GTK_WIDGET(vbox));

	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox), panarea);
	gtk_widget_show_all(dialog);
	while (1) {
		int result = gtk_dialog_run(GTK_DIALOG(dialog));

		g_free(cfg.voice);
		g_free(cfg.effect);
		g_free(cfg.text);
		cfg.voice = g_strdup(voices[hildon_picker_button_get_active(voice)].id);
		cfg.effect = g_strdup(effects[hildon_picker_button_get_active(effect)].id);
		cfg.text = g_strdup(gtk_entry_get_text(GTK_ENTRY(text)));
		cfg.amplitude = gtk_adjustment_get_value(amplitude_adj);
		cfg.pitch = gtk_adjustment_get_value(pitch_adj);
		cfg.speed = gtk_adjustment_get_value(speed_adj);
		cfg.ignore_silent = hildon_check_button_get_active(ignore_silent);

		switch (result) {
		case 1:
			g_print("Test button\n");
			cfg_speak(&cfg, TRUE);
			continue;
		case GTK_RESPONSE_OK:
			g_print("Save\n");
			cfg_write(client, &cfg);
			break;
		}
		break;
	}
	gtk_widget_destroy(GTK_WIDGET(dialog));

	cfg_free(&cfg);
	g_object_unref(G_OBJECT(client));

	return OSSO_OK;
}

osso_return_t save_state(osso_context_t *osso, gpointer data)
{
	g_print("save_state called\n");
	return OSSO_OK;
}

#ifdef TEST
int main(int argc, char *argv[])
{
	gtk_init(&argc, &argv);
	return execute(NULL, NULL, 0);
}
#endif
