/*
  EightyOne - A simple Sudoku solving game
  Copyright (C) 2006  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "GameArea.h"

#include <Lum/Base/L10N.h>
#include <Lum/Base/String.h>

#include <Lum/Model/Boolean.h>

#include <Lum/Array.h>
#include <Lum/Boolean.h>
#include <Lum/Button.h>
#include <Lum/Dialog.h>
#include <Lum/Panel.h>
#include <Lum/Popup.h>
#include <Lum/Text.h>
#include <Lum/Toggle.h>
#include <iostream>
static size_t popupFontSize = 200;

class SetSelector : public Lum::Popup
{
private:
  Lum::Model::ActionRef models[Sudoku::dimension+1]; // Values 0..9, where 0 means empty
  bool                  selected;
  size_t                value;

public:
  SetSelector()
  : selected(false),
    value(Sudoku::dimension+1)
  {
    for (size_t i=0; i<=Sudoku::dimension; i++) {
      models[i]=new Lum::Model::Action();
      Observe(models[i]);
    }
  }

  void PreInit()
  {
    Lum::Array  *array;
    Lum::Panel  *panel;

    panel=new Lum::VPanel();

    array=new Lum::Array();
    array->SetFlex(true,true);
    array->SetHorizontalCount(3);

    for (size_t i=1; i<=Sudoku::dimension; i++) {
      Lum::Text   *text;

      text=new Lum::Text();
      text->SetFlex(true,true);
      text->SetText(Lum::Base::NumberToWString(i),
                    Lum::OS::display->GetFont(popupFontSize));
      text->SetAlignment(Lum::Text::centered);

      array->Add(Lum::Button::Create(text,models[i],true,true));
    }
    panel->Add(array);

    panel->Add(Lum::Button::Create(_(L"GAMEAREA_CLEAR",L"Clear"),models[0],true,false));

    SetMain(panel);

    Popup::PreInit();
  };

  void Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
  {
    for (size_t i=0; i<=Sudoku::dimension; ++i) {
      if (model==models[i] && models[i]->IsFinished()) {
        selected=true;
        value=i;
        Exit();
        break;
      }
    }

    Popup::Resync(model,msg);
  }

  bool HasSelection() const
  {
    return selected;
  }

  size_t GetValue() const
  {
    return value;
  }
};

class MarkSelector : public Lum::Popup
{
private:
  Lum::Model::BooleanRef models[Sudoku::dimension];
  size_t                 value;

public:
  MarkSelector(size_t value)
  : value(value)
  {
    for (size_t i=0; i<=8; i++) {
      models[i]=new Lum::Model::Boolean(value & (1 << i));
      Observe(models[i]);
    }
  }

  void PreInit()
  {
    Lum::Array *array;
    Lum::Panel *panel;

    panel=new Lum::VPanel();

    array=new Lum::Array();
    array->SetFlex(true,true);
    array->SetHorizontalCount(3);

    for (size_t i=0; i<=8; i++) {
      Lum::Toggle *toggle;
      Lum::Text   *text;

      text=new Lum::Text();
      text->SetFlex(true,true);
      text->SetText(Lum::Base::NumberToWString(i+1),
                    Lum::OS::display->GetFont(popupFontSize));
      text->SetAlignment(Lum::Text::centered);

      toggle=new Lum::Toggle();
      toggle->SetFlex(true,true);
      toggle->SetModel(models[i]);
      toggle->SetLabel(text);

      array->Add(toggle);
    }
    panel->Add(array);

    SetMain(panel);

    Popup::PreInit();
  };

  void Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
  {
    for (size_t i=0; i<=8; ++i) {
      if (model==models[i]) {
        if (models[i]->Get()) {
          value|=(1 << i);
        }
        else {
          value=value & ~(1 << i);
        }

        return;
      }
    }

    Popup::Resync(model,msg);
  }

  size_t GetValue() const
  {
    return value;
  }
};

GameArea::GameArea()
 : minFont(Lum::OS::display->GetFont()),
   font(Lum::OS::display->GetFont(Lum::OS::Display::fontScaleCaption1)),
   presetColor(0.5,0.5,0.5,Lum::OS::display->GetColor(Lum::OS::Display::whiteColor)),
   focusBackgroundColor(0.8,0.8,0.8,Lum::OS::display->GetColor(Lum::OS::Display::fillColor)),
   errorColor(1.0,0,0,Lum::OS::display->GetColor(Lum::OS::Display::fillColor)),
   correctColor(0,0.5,0,Lum::OS::display->GetColor(Lum::OS::Display::blackColor)),
   hintColor(1,1,0,Lum::OS::display->GetColor(Lum::OS::Display::blackColor)),
   boxSize(0),
   hintMode(new Lum::Model::Boolean())
{
  if (Lum::OS::display->GetSize()<Lum::OS::Display::sizeNormal) {
    minMarkFont=Lum::OS::display->GetFontByPixel(Lum::OS::Display::fontTypeProportional,7);
  }
  else {
    minMarkFont=Lum::OS::display->GetFont(Lum::OS::Display::fontScaleFootnote);
  }

  SetCanFocus(true);
  RequestFocus();

  hintMode->Set(false);

  Observe(hintMode);
}

int GameArea::GetCellOffset(size_t index) const
{
  return (index+1)*1+(index/3+1)*1+index*boxSize;
}

bool GameArea::SetModel(Lum::Base::Model* model)
{
  this->model=dynamic_cast<GameModel*>(model);

  Control::SetModel(this->model);

  return this->model.Valid();
}

void GameArea::CalcSize()
{
  size_t markBoxSize=0;
  size_t minBoxSize=0;

  // Size of box containing "normal" content text
  for (size_t i=1; i<=Sudoku::dimension; i++) {
    minBoxSize=std::max(minBoxSize,minFont->StringWidth(Lum::Base::NumberToWString(i),
                                                        Lum::OS::Font::bold));
    minBoxSize=std::max(minBoxSize,font->pixelHeight);
  }

  // Size of box if containing markers (3x3 array)
  for (size_t i=1; i<=Sudoku::dimension; i++) {
    markBoxSize=std::max(markBoxSize,
                         minMarkFont->StringWidth(Lum::Base::NumberToWString(i)));
    markBoxSize=std::max(markBoxSize,minMarkFont->pixelHeight);
  }

  markBoxSize*=3;

  // size of area containing of 9x9 boxes with lines in between
  minWidth=(3+1)*1+(9+1)*1+9*std::max(minBoxSize,markBoxSize);
  minHeight=(3+1)*1+(9+1)*1+9*std::max(minBoxSize,markBoxSize);

  // now the current with depending on the current font
  boxSize=0;
  for (size_t i=1; i<=9; i++) {
    boxSize=std::max(boxSize,font->StringWidth(Lum::Base::NumberToWString(i),
                                               Lum::OS::Font::bold));
    boxSize=std::max(boxSize,font->pixelHeight);
  }

  size=(3+1)*1+(9+1)*1+9*std::max(markBoxSize,boxSize);

  width=size;
  height=size;

  Object::CalcSize();
}

/**
  * Find the font that can show dimensions number with in a box of size x size..
  */
Lum::OS::Font* GameArea::FindFont(Lum::OS::Font* minFont, size_t size)
{
  Lum::OS::FontRef font;
  bool             fits;
  size_t           fontSize;

  font=minFont;
  fontSize=font->pixelHeight+1;

  do {
    Lum::OS::FontRef tmpFont;
    size_t           boxSize=0;

    tmpFont=Lum::OS::display->GetFontByPixel(Lum::OS::Display::fontTypeProportional,
                                             fontSize);

    if (tmpFont.Valid()) {
      for (size_t i=1; i<=Sudoku::dimension; i++) {
        boxSize=std::max(boxSize,tmpFont->StringWidth(Lum::Base::NumberToWString(i),
                                                      Lum::OS::Font::bold));
        boxSize=std::max(boxSize,tmpFont->height);
      }

      fits=boxSize<=size && fontSize<=size;
    }
    else {
      fits=fontSize<=size;
    }

    if (fits) {
      font=tmpFont;
      fontSize++;
    }

  } while (fits);

  return font.Get();
}

void GameArea::Layout()
{
  boxSize=std::min((width-(3+1)*1-(9+1)*1)/9,
                   (height-(3+1)*1-(9+1)*1)/9);

  size=(3+1)*1+(9+1)*1+9*boxSize;

  font=FindFont(minFont,boxSize);
  markFont=FindFont(minMarkFont,boxSize/3);

  Object::Layout();
}

void GameArea::Draw(Lum::OS::DrawInfo* draw,
                    int x, int y, size_t w, size_t h)
{
  Object::Draw(draw,x,y,w,h);

  if (!OIntersect(x,y,w,h)) {
    return;
  }

  /* --- */

  size_t offset;

  //
  // Draw game area
  //
  draw->PushForeground(Lum::OS::display->blackColor);
  offset=0;
  for (size_t i=0; i<10; i++) {
    // Vertical
    draw->DrawLine(this->x+offset,this->y,
                   this->x+offset,this->y+size-1);

    // Horizontal
    draw->DrawLine(this->x,this->y+offset,
                   this->x+this->size-1,this->y+offset);

    if (i%3==0) {
      offset++;

      // Vertical
      draw->DrawLine(this->x+offset,this->y,
                     this->x+offset,this->y+this->size-1);

      // Horizontal
      draw->DrawLine(this->x,this->y+offset,
                     this->x+this->size-1,this->y+offset);
    }
    offset=offset+boxSize+1;
  }
  draw->PopForeground();

  if (!model.Valid()) {
    return;
  }

  //
  // Draw values
  //
  draw->PushFont(font,Lum::OS::Font::bold);
  for (size_t i=0; i<Sudoku::dimension*Sudoku::dimension; i++) {

    // Draw background
    if ((HasFocus() || true/*Lum::OS::display->GetTheme()->RequestFingerFriendlyControls()*/) &&
        !model->IsFinished() &&
        i==model->GetFocusedCell()) {
      draw->PushForeground(focusBackgroundColor);
    }
    else {
      draw->PushForeground(Lum::OS::display->whiteColor);
    }
    draw->FillRectangle(this->x+GetCellOffset(i%Sudoku::dimension),
                        this->y+GetCellOffset(i/Sudoku::dimension),
                        boxSize,boxSize);
    draw->PopForeground();

    if (model->GetRiddle(i)!=Sudoku::empty) { // Preset fields
      std::wstring        text(Lum::Base::NumberToWString(model->Get(i)));
      Lum::OS::FontExtent extent;

      font->StringExtent(text,extent,Lum::OS::Font::bold);

      draw->PushForeground(presetColor);
      draw->DrawString(this->x+GetCellOffset(i%Sudoku::dimension)+(boxSize-extent.width+extent.left+extent.right)/2-extent.left,
                       this->y+GetCellOffset(i/Sudoku::dimension)+(boxSize-font->height)/2+font->ascent,
                       text);
      draw->PopForeground();
    }
    else if (model->IsValid() &&
             model->IsFinished()) {
      std::wstring        text(Lum::Base::NumberToWString(model->GetSolution(i)));
      Lum::OS::FontExtent extent;

      font->StringExtent(text,extent,Lum::OS::Font::bold);

      if (model->GetSolution(i)!=model->Get(i)) {
        draw->PushForeground(errorColor);
      }
      else {
        draw->PushForeground(correctColor);
      }

      draw->DrawString(this->x+GetCellOffset(i%Sudoku::dimension)+(boxSize-extent.width+extent.left+extent.right)/2-extent.left,
                       this->y+GetCellOffset(i/Sudoku::dimension)+(boxSize-font->height)/2+font->ascent,
                       text);
      draw->PopForeground();
    }
    else if (model->Get(i)!=Sudoku::empty) { // manually set fields
      std::wstring         text(Lum::Base::NumberToWString(model->Get(i)));
      Lum::OS::FontExtent  extent;
      Sudoku::Bitset       allowed;

      model->GetPotential(i,allowed);

      font->StringExtent(text,extent,Lum::OS::Font::bold);

      if (!allowed.test(model->Get(i))) {
        draw->PushForeground(errorColor);
      }
      else {
        draw->PushForeground(Lum::OS::display->blackColor);
      }

      draw->DrawString(this->x+GetCellOffset(i%Sudoku::dimension)+(boxSize-extent.width+extent.left+extent.right)/2-extent.left,
                       this->y+GetCellOffset(i/Sudoku::dimension)+(boxSize-font->height)/2+font->ascent,
                       text);
      draw->PopForeground();
    }
    else if (hintMode->Get() &&
             i==model->GetFocusedCell()) { // Empty fields with solution hint requested
      std::wstring        text(Lum::Base::NumberToWString(model->GetSolution(i)));
      Lum::OS::FontExtent extent;

      font->StringExtent(text,extent,Lum::OS::Font::bold);

      draw->PushForeground(hintColor);
      draw->DrawString(this->x+GetCellOffset(i%Sudoku::dimension)+(boxSize-extent.width+extent.left+extent.right)/2-extent.left,
                       this->y+GetCellOffset(i/Sudoku::dimension)+(boxSize-font->height)/2+font->ascent,
                       text);
      draw->PopForeground();
    }
    else { // Marked fields
      for (size_t j=0; j<Sudoku::dimension; j++) {
        if (model->GetMarks(i) & (1 << j)) {
          std::wstring         text(Lum::Base::NumberToWString(j+1));
          Lum::OS::FontExtent  extent;

          markFont->StringExtent(text,extent);

          draw->PushFont(markFont);
          draw->PushForeground(Lum::OS::display->blackColor);
          draw->DrawString(this->x+GetCellOffset(i%Sudoku::dimension)+
                           (boxSize/3)*(j%3)+
                           (boxSize/3-extent.width+extent.left+extent.right)/2-extent.left,
                           this->y+GetCellOffset(i/Sudoku::dimension)+
                           (boxSize/3)*(j/3)+
                           (boxSize/3-markFont->height)/2+markFont->ascent,
                           text);
          draw->PopForeground();
          draw->PopFont();
        }
      }
    }
  }
  draw->PopFont();
}

bool GameArea::HandlesClickFocus() const
{
  return CanFocus() &&
         IsVisible() &&
         model.Valid() &&
         !model->IsFinished();
}

bool GameArea::HandlesKeyFocus() const
{
  return CanFocus() &&
         IsVisible() &&
         model.Valid() &&
         !model->IsFinished();
}

bool GameArea::HandleMouseEvent(const Lum::OS::MouseEvent& event)
{
  if (!model.Valid() ||
      model->IsFinished()) {
    return false;
  }

  if ((event.type==Lum::OS::MouseEvent::down && PointIsIn(event) &&
       event.button==Lum::OS::MouseEvent::button1)) {
    size_t tmp=0;

    while (tmp<Sudoku::dimension*Sudoku::dimension) {
      if (event.x>=this->x+GetCellOffset(tmp%Sudoku::dimension) &&
          event.x<this->x+GetCellOffset(tmp%Sudoku::dimension)+(int)boxSize &&
          event.y>=this->y+GetCellOffset(tmp/Sudoku::dimension) &&
          event.y<this->y+GetCellOffset(tmp/Sudoku::dimension)+(int)boxSize) {
        break;
      }

      tmp++;
    }

    if (tmp<Sudoku::size) {
      if (tmp!=model->GetFocusedCell()) {
        model->SetFocusedCell(tmp);
      }

      if (model->Get(model->GetFocusedCell())!=Sudoku::empty) {
        return true;
      }

      if (true/*Lum::OS::display->GetTheme()->RequestFingerFriendlyControls()*/) {
        // No popup dialog, if we need to be finger friendly!
        // We will make use of an external keypad instead in this case.
        return true;
      }

      if (event.x-(this->x+GetCellOffset(tmp%Sudoku::dimension))<=(int)boxSize/2 &&
          event.y-(this->y+GetCellOffset(tmp/Sudoku::dimension))<=(int)boxSize/2) {
        MarkSelector *selector;

        selector=new MarkSelector(model->GetFocusedCellMarks());
        selector->SetParent(GetWindow());
        selector->SetPosition(GetWindow()->GetX()+event.x+5,GetWindow()->GetY()+event.y+5);
        selector->GetWindow()->SetOpacity(0.8);

        if (selector->Open()) {
          selector->EventLoop();
          selector->Close();

          model->SetFocusedCellMarks(selector->GetValue());
        }

        delete selector;
      }
      else {
        SetSelector *selector;

        selector=new SetSelector();
        selector->SetParent(GetWindow());
        selector->SetPosition(GetWindow()->GetX()+event.x+5,GetWindow()->GetY()+event.y+5);
        selector->GetWindow()->SetOpacity(0.8);
        selector->Open();
        selector->EventLoop();
        selector->Close();

        if (selector->HasSelection()) {
          model->SetFocusedCellValue(selector->GetValue());
        }

        delete selector;
      }

    }


    return true;
  }

  return false;
}

bool GameArea::HandleKeyEvent(const Lum::OS::KeyEvent& event)
{
  if (!model.Valid() ||
      model->IsFinished()) {
    return false;
  }

  size_t value;

  if (event.type==Lum::OS::KeyEvent::down) {
    switch (event.key) {
    case Lum::OS::keyLeft:
      if (model->GetFocusedCell()%Sudoku::dimension>0) {
        model->SetFocusedCell(model->GetFocusedCell()-1);
      }
      return true;
    case Lum::OS::keyRight:
      if (model->GetFocusedCell()%Sudoku::dimension<Sudoku::dimension-1) {
        model->SetFocusedCell(model->GetFocusedCell()+1);
      }
      return true;
    case Lum::OS::keyUp:
      if (model->GetFocusedCell()/Sudoku::dimension>0) {
        model->SetFocusedCell(model->GetFocusedCell()-Sudoku::dimension);
      }
      return true;
    case Lum::OS::keyDown:
      if (model->GetFocusedCell()/Sudoku::dimension<Sudoku::dimension-1) {
        model->SetFocusedCell(model->GetFocusedCell()+Sudoku::dimension);
      }
      return true;
    case Lum::OS::keySpace:
    case Lum::OS::keyBackspace:
    case Lum::OS::keyDelete:
      if (model->IsFocusedCellEditable()) {
        model->SetFocusedCellValue(Sudoku::empty);
      }
      return true;
    default:
      if (Lum::Base::WStringToNumber(event.text,value)) {
        if (value>=0 &&
            value<=Sudoku::dimension) {
          if (model->IsFocusedCellEditable()) {
            model->SetFocusedCellValue(value);
          }
        }

        return true;
      }
      break;
    }
  }

  return false;
}

Lum::Model::Boolean* GameArea::GetHintMode() const
{
  return hintMode.Get();
}

void GameArea::Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
{
  if (model==hintMode) {
    Redraw();
  }
  else if (model==this->model) {
    Redraw();
  }

  Object::Resync(model,msg);
}

