/*                             ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
  EightyOne - A simple Sudoku solving game
  Copyright (C) 2006  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "GameModel.h"


GameModel::GameModel()
 : focusedCell(0),
   finished(true)
{
  for (size_t i=0; i<Sudoku::size; i++) {
    marks[i]=0;
  }
}

void GameModel::NotifyValueChanged()
{
  ValueChanged msg;

  Notify(msg);
}

void GameModel::SetFinished(bool finished)
{
  if (this->finished!=finished) {
    StateChanged msg;

    this->finished=finished;

    Notify(msg);
  }
}

/**
  Assign new game
  */
void GameModel::Set(const Sudoku &riddle,
                   const Sudoku &solution)
{
  this->solution=solution;
  this->riddle=riddle;
  this->game=riddle;

  for (size_t i=0; i<Sudoku::size; i++) {
    marks[i]=0;
  }

  focusedCell=0;

  NotifyValueChanged();

  SetFinished(game==solution);
}

/**
  Assign existing game
  */
void GameModel::Set(const Game& game)
{
  riddle.SetAreaAsString(game.riddle);
  solution.SetAreaAsString(game.solution);
  this->game.SetAreaAsString(game.game);

  for (size_t i=0; i<Sudoku::size; i++) {
    marks[i]=0;
  }

  if (game.marks.length()==9*Sudoku::size) {
    for (size_t i=0; i<9*Sudoku::size; i++) {
      if (game.marks[i]!=L' ') {
        marks[i/9]|=(1 << (i%9));
      }
    }
  }

  focusedCell=0;

  NotifyValueChanged();

  SetFinished(this->game==solution);
}

void GameModel::Clear()
{
  game.Clear();
  riddle.Clear();
  solution.Clear();
}

void GameModel::Get(Game& game) const
{
  riddle.GetAreaAsString(game.riddle);
  solution.GetAreaAsString(game.solution);
  this->game.GetAreaAsString(game.game);

  game.marks=L"";

  for (size_t i=0; i<Sudoku::size; i++) {
    for (size_t j=0; j<=8; j++) {
      if (marks[i] & (1 << j)) {
        game.marks.append(L"*");
      }
      else {
        game.marks.append(L" ");
      }
    }
  }
}

void GameModel::Get(Sudoku& game) const
{
  game=this->game;
}

size_t GameModel::Get(size_t pos) const
{
  return game.Get(pos);
}

void GameModel::SetFocusedCell(size_t focusedCell)
{
  assert(focusedCell<Sudoku::size);

  if (this->focusedCell!=focusedCell) {
    this->focusedCell=focusedCell;

    NotifyValueChanged();
  }
}

void GameModel::GetRiddle(Sudoku& game) const
{
  game=this->riddle;
}

void GameModel::GetSolution(Sudoku& game) const
{
  game=this->solution;
}

size_t GameModel::GetRiddle(size_t pos) const
{
  return riddle.Get(pos);
}

size_t GameModel::GetSolution(size_t pos) const
{
  return solution.Get(pos);
}

void GameModel::GetPotential(size_t pos, Sudoku::Bitset& possible) const
{
  game.GetPotential(pos,possible);
}

size_t GameModel::GetMarks(size_t pos) const
{
  return marks[pos];
}

bool GameModel::IsValid() const
{
  return !solution.IsEmpty();
}

bool GameModel::IsFinished() const
{
  return finished;
}

size_t GameModel::GetFocusedCell() const
{
  return focusedCell;
}

bool GameModel::IsFocusedCellEditable() const
{
  return !finished && riddle.Get(focusedCell)==Sudoku::empty;
}

bool GameModel::IsCellInMarkModus() const
{
  return game.Get(focusedCell)==Sudoku::empty;
}

size_t GameModel::GetFocusedCellValue() const
{
  assert(game.Get(focusedCell)!=Sudoku::empty);

  return game.Get(focusedCell);
}

size_t GameModel::GetFocusedCellMarks() const
{
  assert(game.Get(focusedCell)==Sudoku::empty);

  return marks[focusedCell];
}

void GameModel::SetFocusedCellMarks(size_t value)
{
  assert(!finished);
  assert(IsFocusedCellEditable());

  game.Set(focusedCell,Sudoku::empty);
  marks[focusedCell]=value;

  NotifyValueChanged();
}

void GameModel::SetFocusedCellValue(size_t value)
{
  assert(!finished);
  assert(IsFocusedCellEditable());

  game.Set(focusedCell,value);
  marks[focusedCell]=0;

  NotifyValueChanged();

  SetFinished(game==solution);
}

void GameModel::MarkEmpty()
{
  assert(!finished);

  for (size_t cell=0; cell<Sudoku::dimension*Sudoku::dimension; cell++) {
    if (riddle.Get(cell)==Sudoku::empty &&
        game.Get(cell)==Sudoku::empty &&
        marks[cell]==0) {
      marks[cell]=0;
      for (size_t mark=0; mark<Sudoku::dimension; mark++) {
        marks[cell]|=(1 << mark);
      }
    }
  }

  NotifyValueChanged();
}

void GameModel::Solve()
{
  NotifyValueChanged();

  SetFinished(true);
}

