/*
  EightyOne - A simple Sudoku solving game
  Copyright (C) 2006  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <Lum/Base/L10N.h>
#include <Lum/Base/String.h>

#include <Lum/Def/Menu.h>

#include <Lum/Dlg/About.h>
#include <Lum/Dlg/Help.h>
#include <Lum/Dlg/Msg.h>
#include <Lum/Dlg/Progress.h>

#include <Lum/Images/Image.h>
#include <Lum/Images/Loader.h>

#include <Lum/Application.h>
#include <Lum/Array.h>
#include <Lum/AspectRatio.h>
#include <Lum/Button.h>
#include <Lum/Combo.h>
#include <Lum/Dialog.h>
#include <Lum/Image.h>
#include <Lum/Label.h>
#include <Lum/Panel.h>
#include <Lum/Text.h>
#include <Lum/Toggle.h>

#include <Lum/Model/Action.h>
#include <Lum/Model/String.h>
#include <Lum/Model/Table.h>

#include <Lum/Manager/FileSystem.h>

#include "config.h"
#include "Configuration.h"
#include "Dancing.h"
#include "GameArea.h"
#include "GenerateDialog.h"
#include "Statistics.h"

static Lum::Def::AppInfo info;

// TODO
static Sudoku riddle,solution;

class GUIAbortTester : public Sudoku::AbortTester
{
private:
  const Lum::Dlg::ProgressTask& task;

public:
  GUIAbortTester(const Lum::Dlg::ProgressTask& task)
   : task(task)
  {
    // no code
  }

  operator bool() const
  {
    return task.IsAborted();
  };
};

class GenerateTask : public Lum::Dlg::ProgressTask
{
public:
  Sudoku::Weight weight;
  size_t         minimumPoints;

public:
  void Run()
  {
    Sudoku area;

    SetProgress(0);
    SetAction(_(L"PROGRESS_SOLUTION",L"Calculating a new solution..."));
    area.FillRandom();
    riddle=area;
    solution=area;

    if (IsAborted()) {
      return;
    }

    SetProgress(10);
    SetAction(_(L"PROGRESS_RIDDLE",L"Calculating puzzle..."));

    Sudoku::GenerateRiddle(riddle,minimumPoints,GUIAbortTester(*this));

    if (IsAborted()) {
      return;
    }

    SetProgress(90);
    SetAction(_(L"PROGRESS_WEIGHT",L"Calculating weight..."));

    area=riddle;
    area.CalculateWeight(solution,weight);

    if (IsAborted()) {
      return;
    }

    SetProgress(100);
  }
};

class MainDialog : public Lum::Dialog
{
private:
  GameModelRef                  gameModel;
  Lum::Model::ActionRef         clearAction;
  Lum::Model::ActionRef         generateAction;
  Lum::Model::ActionRef         markEmptyAction;
  Lum::Model::ActionRef         solveAction;
  Lum::Model::ActionRef         statAction;
  Lum::Model::ActionRef         helpAction;
  Lum::Model::ActionRef         aboutAction;
  GameArea                      *gameArea;
  GenerateTask                  generateTask;
  Lum::Model::BooleanRef        keypad[Sudoku::dimension];
  Lum::Model::BooleanRef        mark;
  bool                          ignoreFinished;
  bool                          lock;
  Lum::Images::StatefulImageRef hintImage;
  Lum::Images::StatefulImageRef statisticsImage;
  Lum::Images::StatefulImageRef solveImage;
  Lum::Images::StatefulImageRef generateImage;

public:
  MainDialog()
  : gameModel(new GameModel()),
    clearAction(new Lum::Model::Action),
    generateAction(new Lum::Model::Action),
    markEmptyAction(new Lum::Model::Action),
    solveAction(new Lum::Model::Action),
    statAction(new Lum::Model::Action),
    helpAction(new Lum::Model::Action),
    aboutAction(new Lum::Model::Action),
    gameArea(NULL),
    mark(new Lum::Model::Boolean(false)),
    ignoreFinished(false),
    lock(false),
    hintImage(new Lum::Images::StatefulImage()),
    statisticsImage(new Lum::Images::StatefulImage()),
    solveImage(new Lum::Images::StatefulImage()),
    generateImage(new Lum::Images::StatefulImage())
  {
    GetWindow()->SetScreenOrientationHint(Lum::OS::Window::screenOrientationBothSupported);

    Observe(gameModel);

    for (size_t i=0; i<Sudoku::dimension; i++) {
      keypad[i]=new Lum::Model::Boolean(false);
      Observe(keypad[i]);
    }
    Observe(mark);

    generateTask.SetParent(this);
    generateTask.SetCaption(_(L"PROGRESS_TITLE",L"Generating puzzle..."));
    generateTask.minimumPoints=4000;

    Observe(clearAction);
    Observe(generateAction);
    Observe(markEmptyAction);
    Observe(solveAction);
    Observe(statAction);
    Observe(helpAction);
    Observe(aboutAction);
    Observe(GetOpenedAction());
    Observe(GetClosedAction());
    Observe(generateTask.GetAbortedAction());
    Observe(generateTask.GetFinishedAction());

    Lum::Images::loader->SetNormalImagePostfix(L"");
    Lum::Images::loader->LoadApplication(L"Hint.png",hintImage);
    Lum::Images::loader->LoadApplication(L"Statistics.png",statisticsImage);
    Lum::Images::loader->LoadApplication(L"Solve.png",solveImage);
    Lum::Images::loader->LoadApplication(L"Generate.png",generateImage);
  }

  void PreInit()
  {
    gameArea=new GameArea();
    gameArea->SetFlex(true,true);
    gameArea->SetModel(gameModel);
    //gameArea->SetModel(riddle,solution);

    if (Lum::OS::display->IsLandscape()) {
      Lum::Array *array;
      Lum::Panel *horiz,*vert;

      horiz=Lum::HPanel::Create(true,true);
      horiz->Add(gameArea);

      horiz->AddSpace();

      vert=Lum::VPanel::Create(false,true);

      if (true/*Lum::OS::display->GetTheme()->RequestFingerFriendlyControls()*/) {
        array=new Lum::Array();
        array->SetFlex(true,true);
        array->SetHorizontalCount(3);
        array->SetSpace(false,false);

        for (size_t i=0; i<Sudoku::dimension; i++) {
          Lum::Text *text=new Lum::Text(Lum::Base::NumberToWString(i+1),
                                        Lum::OS::Font::normal,
                                        Lum::Text::left,
                                        Lum::OS::display->GetFont(Lum::OS::Display::fontScaleLogo));

          array->Add(Lum::Toggle::Create(text,keypad[i],true,true));
        }

        vert->Add(array);

        array=new Lum::Array();
        array->SetFlex(true,false);
        array->SetHorizontalCount(2);
        array->SetSpace(false,false);

        array->Add(Lum::Button::Create(_(L"MAIN_CLEAR",L"Clear"),clearAction,true,true));
        array->Add(Lum::Toggle::Create(_(L"MAIN_MARK",L"Mark"),mark,true,true));

        vert->Add(array);

        vert->AddSpace();
      }
      else {
        vert->AddSpace(true);
      }

      array=new Lum::Array();
      array->SetFlex(true,false);
      array->SetHorizontalCount(4);
      array->SetSpace(false,false);

      if (hintImage->IsValid()) {
        Lum::Toggle* toggle=new Lum::Toggle();

        toggle->SetFlex(true,false);
        toggle->SetModel(gameArea->GetHintMode());
        toggle->SetLabel(Lum::Image::Create(hintImage));

        array->Add(toggle);
      }
      else {
        array->Add(Lum::Toggle::Create(_(L"GAME_HINT",L"Hint"),gameArea->GetHintMode(),true,false));
      }

      if (statisticsImage->IsValid()) {
        array->Add(Lum::Button::Create(Lum::Image::Create(statisticsImage),statAction,true,false));
      }
      else {
        array->Add(Lum::Button::Create(_(L"GAME_STAT",L"S_tatistics"),statAction,true,false));
      }

      if (solveImage->IsValid()) {
        array->Add(Lum::Button::Create(Lum::Image::Create(solveImage),solveAction,true,false));
      }
      else {
        array->Add(Lum::Button::Create(_(L"SOLVE",L"_Solve"),solveAction,true,false));
      }

      if (generateImage->IsValid()) {
        array->Add(Lum::Button::Create(Lum::Image::Create(generateImage),generateAction,true,false));
      }
      else {
        array->Add(Lum::Button::Create(_(L"GENERATE",L"_New"),generateAction,true,false));
      }

      vert->Add(array);

      horiz->Add(vert);

      SetMain(horiz);
    }
    else {
      Lum::Array *array;
      Lum::Panel *vert;
      Lum::Panel *horiz;
      Lum::Panel *vert2;

      vert=Lum::VPanel::Create(true,true);

      vert->Add(gameArea);

      vert->AddSpace();

      horiz=Lum::HPanel::Create(true,false);

      vert2=Lum::VPanel::Create(false,true);

      if (hintImage->IsValid()) {
        Lum::Toggle* toggle=new Lum::Toggle();

        toggle->SetFlex(false,true);
        toggle->SetModel(gameArea->GetHintMode());
        toggle->SetLabel(Lum::Image::Create(hintImage));

        vert2->Add(toggle);
      }
      else {
        vert2->Add(Lum::Toggle::Create(_(L"GAME_HINT",L"Hint"),gameArea->GetHintMode(),false,true));
      }

      if (statisticsImage->IsValid()) {
        vert2->Add(Lum::Button::Create(Lum::Image::Create(statisticsImage),statAction,false,true));
      }
      else {
        vert2->Add(Lum::Button::Create(_(L"GAME_STAT",L"S_tatistics"),statAction,false,true));
      }

      if (solveImage->IsValid()) {
        vert2->Add(Lum::Button::Create(Lum::Image::Create(solveImage),solveAction,false,true));
      }
      else {
        vert2->Add(Lum::Button::Create(_(L"SOLVE",L"_Solve"),solveAction,false,true));
      }

      if (generateImage->IsValid()) {
        vert2->Add(Lum::Button::Create(Lum::Image::Create(generateImage),generateAction,false,true));
      }
      else {
        vert2->Add(Lum::Button::Create(_(L"GENERATE",L"_New"),generateAction,false,true));
      }

      horiz->Add(vert2);

      horiz->AddSpace();

      vert2=Lum::VPanel::Create(true,true);

      if (true/*Lum::OS::display->GetTheme()->RequestFingerFriendlyControls()*/) {
        array=new Lum::Array();
        array->SetFlex(true,true);
        array->SetHorizontalCount(3);
        array->SetSpace(false,false);

        for (size_t i=0; i<Sudoku::dimension; i++) {
          Lum::Text *text=new Lum::Text(Lum::Base::NumberToWString(i+1),
                                        Lum::OS::Font::normal,
                                        Lum::Text::left,
                                        Lum::OS::display->GetFont(Lum::OS::Display::fontScaleLogo));

          array->Add(Lum::Toggle::Create(text,keypad[i],true,true));
        }

        vert2->Add(array);

        array=new Lum::Array();
        array->SetFlex(true,false);
        array->SetHorizontalCount(2);
        array->SetSpace(false,false);

        array->Add(Lum::Button::Create(_(L"MAIN_CLEAR",L"Clear"),clearAction,true,true));
        array->Add(Lum::Toggle::Create(_(L"MAIN_MARK",L"Mark"),mark,true,true));

        vert2->Add(array);
      }
      else {
        vert2->AddSpace(true);
      }

      horiz->Add(vert2);

      vert->Add(horiz);

      SetMain(vert);
    }

    Lum::Def::Menu *menu=Lum::Def::Menu::Create();

    menu
      ->GroupProject()
        ->ActionQuit(GetClosedAction())
      ->End()
      ->Group(_(L"MENU_GAME",L"_Game"))
        ->Action(Lum::Def::Action(Lum::Def::Desc(_(L"MENU_GAME_NEW",L"_Generate new game"))
                                  .SetShortcut(Lum::OS::qualifierControl,L"g"),
                                  generateAction)
                 .SetOpensDialog())
        ->Action(Lum::Def::Action(Lum::Def::Desc(_(L"MENU_GAME_MARK_EMPTY",L"_Mark empty"))
                                  .SetShortcut(Lum::OS::qualifierControl,L"m"),
                                  markEmptyAction))
        ->Action(Lum::Def::Action(Lum::Def::Desc(_(L"MENU_GAME_SOLVE",L"_Solve game"))
                                  .SetShortcut(Lum::OS::qualifierControl,L"s"),
                                  solveAction))
        ->Separator()
        ->Action(Lum::Def::Action(Lum::Def::Desc(_(L"MENU_GAME_STAT",L"S_tatistics"))
                                  .SetShortcut(Lum::OS::qualifierControl,L"t"),
                                  statAction)
                 .SetOpensDialog())
      ->End()
      ->GroupHelp()
        ->ActionHelp(helpAction)
        ->ActionAbout(aboutAction)
      ->End();

    SetMenu(menu);

    Dialog::PreInit();
  }

  void GenerateRiddle()
  {
    gameModel->Clear();

    generateTask.Start();
  }

  void UpdateKeypad()
  {
    if (gameModel==NULL ||
        !gameModel->IsValid() ||
        gameModel->IsFinished() ||
        !gameModel->IsFocusedCellEditable()) {
      for (size_t i=0; i<Sudoku::dimension; i++) {
        if (keypad[i].Valid()) {
          keypad[i]->Set(false);
          keypad[i]->Disable();
        }
      }

      if (mark.Valid()) {
        mark->Set(false);
        mark->Disable();
      }

      if (clearAction.Valid()) {
        clearAction->Disable();
      }

      return;
    }

    if (gameModel->IsCellInMarkModus()) {
      size_t value=gameModel->GetFocusedCellMarks();
      size_t marks=0;

      for (size_t i=0; i<Sudoku::dimension; i++) {
        keypad[i]->Enable();
        keypad[i]->Set(value & (1 << i));

        if (value & (1 << i)) {
          marks++;
        }
      }

      if (marks==1) {
        mark->Enable();
        mark->Set(gameModel->IsCellInMarkModus());
      }
      else {
        mark->Set(marks>1);
        mark->Disable();
      }

      if (marks>=1) {
        clearAction->Enable();
      }
      else {
        clearAction->Disable();
      }
    }
    else {
      size_t value=gameModel->GetFocusedCellValue();

      for (size_t i=0; i<Sudoku::dimension; i++) {
        keypad[i]->Enable();
        keypad[i]->Set(i+1==value);
      }

      mark->Set(false);
      mark->Enable();

      if (value!=Sudoku::empty) {
        clearAction->Enable();
      }
      else {
        clearAction->Disable();
      }
    }
  }

  void UpdateCellFromKeypad()
  {
    if (gameArea==NULL ||
        gameModel->IsFinished() ||
        !gameModel->IsFocusedCellEditable()) {
      // We ignore possible changes if there is no valid game
      return;
    }

    size_t marks=0;
    for (size_t i=0; i<Sudoku::dimension; i++) {
      if (keypad[i].Valid() && keypad[i]->Get()) {
        marks++;
      }
    }

    if (mark->Get() || marks>1) {
      size_t value=0;

      for (size_t i=0; i<Sudoku::dimension; i++) {
        if (keypad[i].Valid() && keypad[i]->Get()) {
          value|=(1 << i);
        }
      }

      gameModel->SetFocusedCellMarks(value);
    }
    else {
      bool hasValue=false;

      for (size_t i=0; i<Sudoku::dimension; i++) {
        if (keypad[i].Valid() && keypad[i]->Get()) {
          gameModel->SetFocusedCellValue(i+1);
          hasValue=true;
        }
      }

      if (!hasValue) {
        gameModel->SetFocusedCellValue(Sudoku::empty);
      }
    }
  }

  void Resync(Lum::Base::Model *model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==GetOpenedAction() &&
        GetOpenedAction()->IsFinished()) {
      if (LoadConfig()) {
        if (game!=NULL) {
          Sudoku         riddle,solution;
          Sudoku::Weight weight;

          riddle.SetAreaAsString(game->riddle);
          solution.SetAreaAsString(game->solution);

          riddle.CalculateWeight(solution,weight);

          gameModel->Set(*game);

          delete game;
          game=NULL;
        }
      }

      if (gameModel->IsFinished()) {
        markEmptyAction->Disable();
        solveAction->Disable();
        statAction->Disable();
        gameArea->GetHintMode()->Set(false);
        gameArea->GetHintMode()->Disable();
      }
    }
    else if (model==GetClosedAction() &&
             GetClosedAction()->IsFinished()) {
      if (!gameModel->IsFinished() &&
        gameModel->IsValid()) {
        game=new Game();
        gameModel->Get(*game);
      }

      SaveConfig();
    }
    else if (model==generateAction &&
             generateAction->IsFinished()) {
      if (ShowGenerateDialog(this,generateTask.minimumPoints)) {
        targetLevelPoints=generateTask.minimumPoints;
        GenerateRiddle();
      }
    }
    else if (model==generateTask.GetFinishedAction() &&
             generateTask.GetFinishedAction()->IsFinished()) {
      markEmptyAction->Enable();
      solveAction->Enable();
      statAction->Enable();
      gameArea->GetHintMode()->Set(false);
      gameArea->GetHintMode()->Enable();

      gameModel->Set(riddle,solution);
    }
    else if (model==generateTask.GetAbortedAction() &&
             generateTask.GetAbortedAction()->IsFinished()) {
    }
    else if (model==markEmptyAction &&
             markEmptyAction->IsFinished()) {
      gameModel->MarkEmpty();
    }
    else if (model==solveAction &&
             solveAction->IsFinished() &&
             solveAction->IsEnabled()) {
      if (Lum::Dlg::Msg::Ask(this,
                             _(L"SOLVE_TITLE",L"Are you sure?"),
                             _(L"SOLVE_BODY",
                               L"Are you sure that you want to solve the game? \n"
                               L"\n"
                               L"Solving the game cannot be undone!"),
                             _(L"SOLVE_BUTTONS",L"_Solve it!*|_Cancel^"))==0) {
        gameArea->GetHintMode()->Set(false);
        gameArea->GetHintMode()->Disable();
        markEmptyAction->Disable();
        solveAction->Disable();
        statAction->Disable();

        ignoreFinished=true;
        gameModel->Solve();
        ignoreFinished=false;
      }
    }
    else if (model==statAction &&
             statAction->IsFinished()) {
      Sudoku game;
      Sudoku riddle;
      Sudoku solution;

      gameModel->Get(game);
      gameModel->GetRiddle(riddle);
      gameModel->GetSolution(solution);

      StatisticsDialog::Show(GetWindow(),game,riddle,solution);
    }
    else if (model==helpAction &&
             helpAction->IsFinished()) {
      Lum::Dlg::Help::Show(this,
                           _(L"HELP_BODY",
                             L"The interface supports setting numbers and marking\n"
                             L"fields as potentially holding a subset of numbers.\n"
                             L"\n"
                             L"Marks will be shown as small numbers, set values as black,\n"
                             L"predefined number will be drawn grey, and conflicts\n"
                             L"red.\n"
                             L"\n"
                             L"To set/mark a field use the keypad. Explcitely use the mark\n"
                             L"button to tag an entry as marked.\n"
                             L"\n"
                             L"Have fun :-)"));
    }
    else if (model==aboutAction &&
             aboutAction->IsFinished()) {
      Lum::Dlg::About::Show(this,info);
    }
    else if (model==clearAction &&
             clearAction->IsFinished()) {
      if (!lock) {
        lock=true;
        gameModel->SetFocusedCellValue(Sudoku::empty);
        UpdateKeypad();
        lock=false;
      }
    }
    else if (model==gameModel) {
      if (dynamic_cast<const GameModel::ValueChanged*>(&msg)!=NULL) {
        if (!lock) {
          lock=true;
          UpdateKeypad();
          lock=false;
        }
      }
      else if (dynamic_cast<const GameModel::StateChanged*>(&msg)!=NULL) {
        if (gameModel->IsFinished()) {
          if (!ignoreFinished) {
            Lum::Dlg::Msg::ShowOk(this,
                                  _(L"CONGRATS_TITLE",L"Yeeeaarrrr!!\n"),
                                  _(L"CONGRATS_BODY",
                                    L"It looks like you have successfully finished\n"
                                    L"the given puzzle!\n"
                                    L"Congratulations!"));
          }

          gameArea->GetHintMode()->Set(false);
          gameArea->GetHintMode()->Disable();
          markEmptyAction->Disable();
          solveAction->Disable();
          statAction->Disable();
          UpdateKeypad();
        }
      }
    }
    else if (model==mark) {
      if (!lock) {
        lock=true;
        UpdateCellFromKeypad();
        UpdateKeypad();
        lock=false;
      }
    }
    else {
      if (!lock) {
        lock=true;
        for (size_t i=0; i<Sudoku::dimension; i++) {
          if (model==keypad[i]) {
            UpdateCellFromKeypad();
            UpdateKeypad();
          }
        }
        lock=false;
      }
    }

    Dialog::Resync(model,msg);
  }
};

class Main : public Lum::GUIApplication<MainDialog>
{
public:
  bool Initialize()
  {
#if defined(APP_DATADIR)
    Lum::Manager::FileSystem::Instance()->SetApplicationDataDir(Lum::Base::StringToWString(APP_DATADIR));
#endif

    info.SetProgram(Lum::Base::StringToWString(PACKAGE_NAME));
    info.SetVersion(Lum::Base::StringToWString(PACKAGE_VERSION));
    info.SetDescription(_(L"ABOUT_DESC",L"A simple Sudoku implementation..."));
    info.SetAuthor(L"Tim Teulings");
    info.SetContact(L"Tim Teulings <tim@teulings.org>");
    info.SetCopyright(L"(c) 2006, Tim Teulings");
    info.SetLicense(L"GNU Public License");

    std::wstring appDataDir;

    if (Lum::Manager::FileSystem::Instance()->GetEntry(Lum::Manager::FileSystem::appDataDir,
                                                       appDataDir)) {

      Lum::Base::Path path;

      path.SetNativeDir(appDataDir);
      path.AppendDir(L"icons");

      Lum::Images::loader->AddApplicationSearchPath(path);
    }

    Lum::Base::Path path;

    path.SetDir(L"data");
    path.AppendDir(L"icons");
    path.AppendDir(L"16x16");
    Lum::Images::loader->AddApplicationSearchPath(path);

    return Lum::GUIApplication<MainDialog>::Initialize();
  }

  void Cleanup()
  {
    FreeConfig();

    Lum::GUIApplication<MainDialog>::Cleanup();
  }
};

LUM_MAIN(Main,L"EightyOne")
