/*
  EightyOne - A simple Sudoku solving game
  Copyright (C) 2006  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "Sudoku.h"

#include <cassert>
#include <cstdlib>
#include <iostream>
#include <list>
#include <vector>

#include <time.h>

#include <Lum/Base/DateTime.h>

#include "Dancing.h"

static DancingLinksSudoku *dancing=NULL;

Sudoku::Weight::Weight()
: directFills(0),
  crossHatch(0),
  singleCandidate(0),
  singleSquare(0),
  guesses(0),
  rating(0)
{
  // no code
}

Sudoku::Sudoku()
{
  area.resize(size);
  Clear();
}

Sudoku::Sudoku(const Sudoku& sudoku)
{
  area=sudoku.area;
}

Sudoku& Sudoku::operator=(const Sudoku& sudoku)
{
  if (this!=&sudoku) {
    area=sudoku.area;
  }

  return *this;
}

bool Sudoku::operator==(const Sudoku& sudoku)
{
  return area==sudoku.area;
}

size_t Sudoku::GetFieldsFree() const
{
  size_t count=0;

  for (size_t x=0; x<area.size(); x++) {
    if (area[x]==empty) {
      count++;
    }
  }

  return count;
}

size_t Sudoku::GetFieldsSet() const
{
  return area.size()-GetFieldsFree();
}

bool Sudoku::IsValid() const
{
  // horizontal
  for (size_t y=0; y<dimension; y++) {
    Bitset set;

    for (size_t x=0; x<dimension; x++) {
      size_t value;

      value=Get(x,y);
      if (value!=empty) {
        if (set.test(value)) {
          return false;
        }
        set.set(value);
      }
    }
  }

  // vertical
  for (size_t x=0; x<dimension; x++) {
    Bitset set;

    for (size_t y=0; y<dimension; y++) {
      size_t value;

      value=Get(x,y);
      if (value!=empty) {
        if (set.test(value)) {
          return false;
        }
        set.set(value);
      }
    }
  }

  // quadrant 1-9
  for (size_t qx=0; qx<dimension; qx+=quadrantDimension) {
    for (size_t qy=0; qy<dimension; qy+=quadrantDimension) {
      Bitset set;

      for (size_t x=qx; x<qx+quadrantDimension; x++) {
        for (size_t y=qy; y<qy+quadrantDimension; y++) {
          size_t value;

          value=Get(x,y);
          if (value!=empty) {
            if (set.test(value)) {
              return false;
            }
            set.set(value);
          }
        }
      }
    }
  }

  return true;
}

void Sudoku::GetPotential(size_t pos, Bitset& possible, bool incQuad)
{
  GetPotential(pos%dimension,pos/dimension,possible,incQuad);
}

void Sudoku::GetPotential(size_t x, size_t y, Bitset& possible, bool incQuad)
{
  assert(x>=0 && x<dimension && y>=0 && y<dimension);

  // per default everything is possible ;-)
  for (size_t i=1; i<=dimension; i++) {
    possible.set(i);
  }

  // horizontal
  for (size_t i=0; i<dimension; i++) {
    if (i!=x) {
      possible.clear(Get(i,y));
    }
  }

  // vertical
  for (size_t i=0; i<dimension; i++) {
    if (i!=y) {
      possible.clear(Get(x,i));
    }
  }

  if (!incQuad) {
    return;
  }

  // quadrant
  for (size_t i=x/quadrantDimension*quadrantDimension; i<x/quadrantDimension*quadrantDimension+quadrantDimension; i++) {
    for (size_t j=y/quadrantDimension*quadrantDimension; j<y/quadrantDimension*quadrantDimension+quadrantDimension; j++) {
      if (i!=x && j!=y) {
        possible.clear(Get(i,j));
      }
    }
  }
}

void Sudoku::GetPotential(size_t x, size_t y, std::vector<size_t>& list, bool incQuad)
{
  assert(x>=0 && x<dimension && y>=0 && y<dimension && Get(x,y)==empty);

  Bitset set;

  assert(Get(x,y)==empty);

  GetPotential(x,y,set,incQuad);

  if (set.count()>0) {
    list.reserve(dimension);

    for (size_t i=1; i<=dimension; i++) {
      if (set.test(i)) {
        list.push_back(i);
      }
    }
  }
}

void Sudoku::GetAreaAsString(std::wstring& value) const
{
  value.resize(area.size());

  for (size_t i=0; i<area.size(); i++) {
    if (area[i]==empty) {
      value[i]=L' ';
    }
    else {
      value[i]=L'0'+area[i];
    }
  }
}

void Sudoku::SetAreaAsString(const std::wstring& value)
{
  for (size_t i=0; i<area.size(); i++) {
    if (value[i]==L' ') {
    area[i]=empty;
    }
    else {
      area[i]=value[i]-L'0';
    }
  }
}

void Sudoku::Draw()
{
  for (size_t y=0; y<dimension; y++) {
    if (y%quadrantDimension==0) {
      for (size_t x=0; x<dimension; x++) {
        if (x%quadrantDimension==0) {
          std::cout << "|";
        }
        std::cout << "-";
      }
      std::cout << "|" << std::endl;
    }

    for (size_t x=0; x<dimension; x++) {

      if (x%quadrantDimension==0) {
        std::cout << "|";
      }

      if (Get(x,y)==empty) {
        std::cout << ".";
      }
      else {
        std::cout << (int) Get(x,y);
      }
    }
    std::cout << "|" << std::endl;
  }

  for (size_t x=0; x<dimension; x++) {
    if (x%quadrantDimension==0) {
      std::cout << "|";
    }
    std::cout << "-";
  }
  std::cout << "|";

  std::cout << std::endl;
}

void Sudoku::Clear()
{
  for (size_t i=0; i<area.size(); i++) {
    area[i]=empty;
  }
}

void Sudoku::FillRandom()
{
  CalcStep stack[size];
  bool     found=false;
  int      index;

  srand(time(NULL));
  Clear();

  index=0;
  stack[index].x=0;
  stack[index].y=0;
  GetPotential(stack[index].x,stack[index].y,stack[index].potential);

  Bitset set;

  GetPotential(0,0,set);

  while (index>=0 && !found) {
    CalcStep &step=stack[index];

    if (step.potential.size()==0) {
      Set(step.x,step.y,empty);
      --index;
    }
    else {
      size_t pos=(size_t)(double(step.potential.size())*rand()/RAND_MAX);

      Set(step.x,step.y,step.potential[pos]);
      step.potential.erase(step.potential.begin()+pos);

      if (IsValid()) {
        if (index==size-1) {
          found=true;
        }
        else {
          ++index;

          if (step.x==dimension-1) {
            stack[index].x=0;
            stack[index].y=step.y+1;
          }
          else {
            stack[index].x=step.x+1;
            stack[index].y=step.y;
          }
          GetPotential(stack[index].x,stack[index].y,stack[index].potential);
        }
      }
    }
  }

  assert(found);
}

void Sudoku::GenerateRiddle(Sudoku &riddle, size_t minimumPointLimit,
                            const AbortTester& abortTester)
{
  time_t              firstTime=0;
  Sudoku              solution=riddle;
  bool                finished=false;
  std::vector<size_t> filled;

  filled.reserve(Sudoku::dimension*Sudoku::dimension);

  while (!finished) {
    Lum::Base::SystemTime start;
    bool                  foundFree=false;
    bool                  pointsReached=false;
    size_t                fillSize;

    if (abortTester) {
      return;
    }

    filled.clear();

    // Put all filled fields in a list. This is the list of candidates.
    for (size_t i=0; i<Sudoku::dimension*Sudoku::dimension; i++) {
      if (riddle.Get(i)!=Sudoku::empty) {
        filled.push_back(i);
      }
    }

    fillSize=filled.size();

    // While there are still filled fields to check and
    // we have not found a Sudoku with exactly one solution...
    while (!filled.empty() && !foundFree) {
      if (abortTester) {
        return;
      }

      size_t pos=(size_t)(double(filled.size())*rand()/RAND_MAX);

      std::vector<size_t>::iterator iter=filled.begin();

      std::advance(iter,pos);

      size_t idx=*iter;

      // Set one random set field to unset
      riddle.Set(idx,Sudoku::empty);

      // Now check if this Sudoku has exactly one solution. If yes, were are finished.
      if (riddle.HasOneSolution()) {
        foundFree=true;
      }
      else {
        // If this Sudoku has more than one solution reset the field (and check another one)
        riddle.Set(idx,solution.Get(idx));
      }

      // delete this item from the list of candidates
      filled.erase(iter);
    }

    // After a certain complexity is reached, we start calculating the weight
    // of the riddle to check if we have reached the minimum points and thus can stop.
    if (fillSize<=35) {
      Weight weight;
      Sudoku tmp(riddle);

      tmp.CalculateWeight(solution,weight);

      pointsReached=weight.rating>=minimumPointLimit;
    }

    // We cannot further improve our riddle but we have still not reached
    // the minimum points.
    // => Start all over again!
    if (filled.empty() && !pointsReached) {
      riddle=solution;
      finished=false;
    }
    else {
      Lum::Base::SystemTime end;
      end.Sub(start);

      // We are finished either if the minimum points are reached or
      // if iterations start to get very slow.
      finished=pointsReached || (firstTime!=0 && end.GetTime()>5*firstTime);
      if (firstTime==0) {
        firstTime=end.GetTime();
      }
    }
  }
}

bool Sudoku::HasOneSolution()
{
  std::vector<int> solution;

  if (dancing==NULL) {
    dancing=new DancingLinksSudoku();
  }

  dancing->assignRiddle(area);

  if (!dancing->solveit(solution)) {
    return false;
  }

  if (dancing->solveit(solution)) {
    return false;
  }
  else {
    return true;
  }
}

bool Sudoku::CalculateWeightDirectFills(const Sudoku& solution, Weight &weight)
{
  size_t posX;
  size_t posY;
  size_t count;

  // Horizontal

  for (size_t y=0; y<dimension; y++) {
    posX=dimension;
    count=0;
    for (size_t x=0; x<dimension; x++) {
      if (Get(x,y)==empty) {
        posX=x;
        count++;
      }
    }

    if (count==1) {
      //std::cout << posX+1 << "," << y+1 << " (last missing in row)" << std::endl;
      Set(posX,y,solution.Get(posX,y));
      return true;
    }
  }

  // Vertical

  for (size_t x=0; x<dimension; x++) {
    posY=dimension;
    count=0;
    for (size_t y=0; y<dimension; y++) {
      if (Get(x,y)==empty) {
        posY=y;
        count++;
      }
    }

    if (count==1) {
      //std::cout << x+1 << "," << posY+1 << " (last missing in column)" << std::endl;
      Set(x,posY,solution.Get(x,posY));
      return true;
    }
  }

  // Quadrant

  for (size_t qx=0; qx<dimension; qx+=quadrantDimension) {
    for (size_t qy=0; qy<dimension; qy+=quadrantDimension) {

      posX=dimension;
      posY=dimension;
      count=0;

      for (size_t x=qx; x<qx+quadrantDimension; x++) {
        for (size_t y=qy; y<qy+quadrantDimension; y++) {
          if (Get(x,y)==empty) {
            posX=x;
            posY=y;
            count++;
          }
        }
      }

      if (count==1) {
        //std::cout << posX+1 << "," << posY+1 << " (last missing in quadrant)" << std::endl;
        Set(posX,posY,solution.Get(posX,posY));
        return true;
      }
    }
  }

  return false;
}

bool Sudoku::CalculateWeightSingleCandidate(const Sudoku& solution, Weight &weight)
{
  Bitset possibles[size];

  for (size_t y=0; y<dimension; y++) {
    for (size_t x=0; x<dimension; x++) {
      if (Get(x,y)==empty) {
        GetPotential(x,y,possibles[x+y*dimension]);

        if (possibles[x+y*dimension].count()==1) {
          //std::cout << x+1 << "," << y+1 << " (only one value possible for this field)" << std::endl;
          Set(x,y,solution.Get(x,y));
          return true;
        }
      }
    }
  }

  return false;
}

bool Sudoku::CalculateWeightCrossHatch(const Sudoku& solution, Weight &weight)
{
  Bitset possibles[size];

  for (size_t i=0; i<size; i++) {
    for (size_t j=1; j<=dimension; j++) {
      if (Get(i)==empty) {
        possibles[i].set(j);
      }
    }
  }

  for (size_t y=0; y<dimension; y++) {
    for (size_t x=0; x<dimension; x++) {
      size_t value=Get(x,y);

      if (value!=empty) {
        for (size_t i=0; i<dimension; i++) {
          possibles[x+i*dimension].clear(value);
        }
        for (size_t i=0; i<dimension; i++) {
          possibles[i+y*dimension].clear(value);
        }
        for (size_t qx=(x/3)*3; qx<(x/3)*3+quadrantDimension; qx++) {
          for (size_t qy=(y/3)*3; qy<(y/3)*3+quadrantDimension; qy++) {
            possibles[qx+qy*dimension].clear(value);
          }
        }
      }
    }
  }

  for (size_t i=1; i<=dimension; i++) {
    for (size_t qx=0; qx<dimension; qx+=quadrantDimension) {
      for (size_t qy=0; qy<dimension; qy+=quadrantDimension) {

        size_t posX=dimension;
        size_t posY=dimension;
        size_t count=0;

        for (size_t x=qx; x<qx+quadrantDimension; x++) {
          for (size_t y=qy; y<qy+quadrantDimension; y++) {
            if (possibles[x+y*dimension].test(i)) {
              posX=x;
              posY=y;
              count++;
            }
          }
        }

        if (count==1) {
          //std::cout << posX+1 << "," << posY+1 << " (cross hatching)" << std::endl;
          Set(posX,posY,solution.Get(posX,posY));
          return true;
        }
      }
    }
  }

  return false;
}

bool Sudoku::CalculateWeightSingleSquare(const Sudoku& solution, Weight &weight)
{
  Bitset possibles[size];

  for (size_t y=0; y<dimension; y++) {
    for (size_t x=0; x<dimension; x++) {
      if (Get(x,y)==empty) {
        GetPotential(x,y,possibles[x+y*dimension]);
      }
    }
  }

  for (size_t i=1; i<=dimension; i++) {

    // horizontal
    for (size_t y=0; y<dimension; y++) {
      size_t count=0;
      size_t last=dimension;

      for (size_t x=0; x<dimension; x++) {
        if (possibles[x+y*dimension].test(i)) {
          count++;
          last=x;
        }
      }

      if (count==1) {
        Set(last,y,solution.Get(last,y));
        //std::cout << last+1 << "," << y+1 << " (only field for this value in row)" << std::endl;
        return true;
      }
    }

    // vertical
    for (size_t x=0; x<dimension; x++) {
      size_t count=0;
      size_t last=dimension;

      for (size_t y=0; y<dimension; y++) {
        if (possibles[x+y*dimension].test(i)) {
          count++;
          last=y;
        }
      }

      if (count==1) {
        Set(x,last,solution.Get(x,last));
        //std::cout << x+1 << "," << last+1 << " (only field for this value in column)" << std::endl;
        return true;
      }
    }

    // quadrant
    for (size_t qx=0; qx<dimension; qx+=quadrantDimension) {
      for (size_t qy=0; qy<dimension; qy+=quadrantDimension) {

        size_t posX=dimension;
        size_t posY=dimension;
        size_t count=0;

        for (size_t x=qx; x<qx+quadrantDimension; x++) {
          for (size_t y=qy; y<qy+quadrantDimension; y++) {
            if (possibles[x+y*dimension].test(i)) {
              posX=x;
              posY=y;
              count++;
            }
          }
        }

        if (count==1) {
          //std::cout << posX+1 << "," << posY+1 << " (only field for this value in quadrant)" << std::endl;
          Set(posX,posY,solution.Get(posX,posY));
          return true;
        }
      }
    }
  }

  return false;
}

void Sudoku::CalculateWeight(const Sudoku& solution, Weight &weight)
{
  size_t freeFields;

  weight.filledFields=dimension*dimension-GetFieldsFree();
  weight.directFills=0;
  weight.crossHatch=0;
  weight.singleCandidate=0;
  weight.singleSquare=0;
  weight.guesses=0;
  weight.rating=0;

  while ((freeFields=GetFieldsFree())>0) {
    if (CalculateWeightDirectFills(solution,weight)) {
      weight.directFills++;
      weight.rating+=freeFields*1;
      continue;
    }

    if (CalculateWeightCrossHatch(solution,weight)) {
      weight.crossHatch++;
      weight.rating+=freeFields*3;
      continue;
    }

    if (CalculateWeightSingleCandidate(solution,weight)) {
      weight.singleCandidate++;
      weight.rating+=freeFields*10;
      continue;
    }

    if (CalculateWeightSingleSquare(solution,weight)) {
      weight.singleSquare++;
      weight.rating+=freeFields*30;
      continue;
    }


    // if nothing else works, simply guess
    size_t pos=0;
    while (area[pos]!=0) {
      pos++;
    }

    //std::cout << pos%dimension << "," << pos/dimension << " (guessed)" << std::endl;
    area[pos]=solution.area[pos];
    weight.guesses++;
    weight.rating+=freeFields*200;
  }
}


