#include "facebookgraph.h"

#include <QDebug>

const QLatin1String FacebookGraph::OPEN_PAREN("(");
const QLatin1String FacebookGraph::CLOSE_PAREN(")");

FacebookGraph::FacebookGraph(QObject *parent, QString facebookJson) :
    QObject(parent)
{
    this->_facebookJson = facebookJson;
    parseResponse();
}

void FacebookGraph::parseResponse()
{
    qDebug() << "Beginning parseResponse()";
    _parseResult = ERROR;
    if (_facebookJson.isEmpty()){
        qWarning() << "JSON was empty";
        return;
    }

    QScriptEngine script_engine;
    QScriptValue script_value = script_engine.evaluate(OPEN_PAREN + _facebookJson + CLOSE_PAREN);
    if (script_value.isError())
    {
        qCritical() << "Unable to parse JSON: " << _facebookJson;
        return;
    }

    QScriptValue error = script_value.property("error");
    if (error.isValid()) {
        QScriptValue type = error.property("type");
        QString sType;
        if (type.isValid()) sType = type.toString();
        else sType = "unknown";

        QScriptValue message = error.property("message");
        QString sMessage;
        if (message.isValid()) sMessage = message.toString();
        else sMessage = "none";
        qCritical() << "Facebook reporting error. Type:"<<sType<<"  Message:"<<sMessage;
    }

    if (script_value.property("id").isValid()) {
        qDebug() << "Found id node...assuming this is a STATUS_SUCCESSFUL type message";
        _parseResult = STATUS_SUCCESSFUL;
        return;
    }

    if (script_value.property("data").isValid()) {
        qDebug() << "Found data node...assuming this is a MESSAGES type message";
        _parseResult = MESSAGES;
        QScriptValue data = script_value.property("data");
        if (data.isArray()) {
            int arrayCount = data.property("length").toInt32();
            for (int index=0; index<arrayCount; index++)
            {
                //qDebug() << "Processing data index #" << index;
                QScriptValue data_entry = data.property(QString("%1").arg(index));
                _messages.append(parseDataEntry(data_entry));
            }
        }
    }

}

Message* FacebookGraph::parseDataEntry(QScriptValue data_entry)
{
    //Get the field names, and identify the type based on the field names avalable
    //Status has: id, message, updated_time (and possibly comments).  Those are what I requested.  Can have From also
    //Checkin has: id, (from), (tags), place, message, application, created_time
    //Each has comments as well
    //For now, that's all we really care about.  In the future, we may indicate that photos have been uploaded, links,  notes,
    //   to allow the user to see any comments made and respond to them.
    //Link has: id, from, link, name, caption, description, icon, picture, message, created_time

    //Really, the common ones are  id, message, updated_time/created_time, and from
    //Status has updated_time, others have created_time
    //Checkin has place, application
    //Can use one Class with some options and an indicator of type.
    //Still need comment.
    //How about likes and dislikes?
    //If I use posts, I get everything!!
    Message *message = new Message(this);

    if (!data_entry.isValid())
    {
        qWarning()<< "invalid data entry in json";
        message->setText("Invalid message data");
    }
    else
    {

        if (data_entry.property("id").isNumber())
        {
            message->setId(data_entry.property("id").toInteger());
        }

        if (data_entry.property("created_time").isValid())
        {
            message->setTimestamp(parseTime(data_entry.property("created_time")));
        }

        if (data_entry.property("message").isValid()) {
            message->setText(data_entry.property("message").toString());
        }

        if (data_entry.property("type").isValid())
        {
            QString type = data_entry.property("type").toString();
            if (type==QLatin1String("status")) message->setMessageType(Message::STATUS);
            else if (type==QLatin1String("checkin")) message->setMessageType(Message::CHECKIN);
            else if (type==QLatin1String("link")) message->setMessageType(Message::LINK);
            else message->setMessageType(Message::POST);
        }

        if (data_entry.property("likes").isNumber())
        {
            message->setLikeCount(data_entry.property("likes").toInt32());
        }

        QScriptValue comments = data_entry.property("comments");
        if (comments.isValid()) {
            QScriptValue comment_data = comments.property("data");
            if (comment_data.isValid() && comment_data.isArray())
            {
                int commentArrayCount = comment_data.property("length").toInt32();
                for (int commentIndex=0; commentIndex<commentArrayCount; commentIndex++)
                {
                    QScriptValue comment_data_entry = comment_data.property(QString("%1").arg(commentIndex));
                    message->addComment(parseComment(comment_data_entry,message));
                }
            }
        }
    }
    return message;
}

Comment* FacebookGraph::parseComment(QScriptValue comment_data_entry, Message* parent)
{
    Comment *comment = new Comment(parent);
    if (!comment_data_entry.isValid())
    {
        qWarning()<< "invalid comment data entry in json";
        comment->setText(tr("Invalid comment data"));
    }
    else
    {
        QScriptValue text = comment_data_entry.property("message");
        if (text.isValid()) comment->setText(text.toString());

        QScriptValue timestamp = comment_data_entry.property("created_time");
        if (text.isValid()) comment->setTimestamp(parseTime(timestamp));

        QScriptValue from = comment_data_entry.property("from");
        if (from.isValid()) {
            QScriptValue nameNode = from.property("name");
            if (nameNode.isValid()) comment->setAuthorName(nameNode.toString());
            QScriptValue idNode = from.property("id");
            if (idNode.isValid() && idNode.isNumber()) comment->setAuthorId(idNode.toInt32());
        }

    }
    return comment;
}

QDateTime FacebookGraph::parseTime(QScriptValue jsonNode)
{
    QDateTime time;
    if (jsonNode.isDate())
    {
        time = jsonNode.toDateTime();
    }
    else
    {
        //The toDateTime function does not seem to properly parse this date time
        QString timestamp_str = jsonNode.toString();
        time = QDateTime::fromString(timestamp_str,Qt::ISODate);
        //QT doesn't seem to like the time zone data, so we set it explicitly to UTC and then convert to local.
        time.setTimeSpec(Qt::UTC);
        time = time.toLocalTime();
    }
    return time;
}

QString FacebookGraph::getFormattedMessage()
{
    QString output;
    foreach (Message *message,_messages)
    {
         output += (QLatin1String("<p class='message'><b>"));
         if (message->getMessageType()==Message::CHECKIN) output += tr("Checkin");
         else if (message->getMessageType()==Message::STATUS) output += tr("Status");
         else if (message->getMessageType()==Message::LINK) output += tr("Link");
         else output += tr("Post");
         output += QLatin1String(":</b> ");
         output += message->getText();
         output += QLatin1String(" <i>(");
         output += formatTimestamp(message->getTimestamp());
         output += QLatin1String(")</i> ");

         if (message->getLikeCount()>0) {
             QString plural("");
             if (message->getLikeCount()>1) plural = QLatin1String("s");
             output += QString("%1 like%2 ").arg(message->getLikeCount()).arg(plural);
         }
         if (!message->getComments().isEmpty()) {
             output += QLatin1String("<ul class='comments'>");

             foreach (Comment *comment, message->getComments())
             {
                 output += QLatin1String("<li class='comment'>");
                 output += comment->getText();
                 output += QLatin1String(" <i>(");

                 if (comment->getAuthorName().isEmpty()) output += QLatin1String("unknown author");
                 else output += comment->getAuthorName();

                 output += QLatin1String(" - ");
                 output += formatTimestamp(comment->getTimestamp());
                 output += QLatin1String(")</i> ");
             }
             output += QLatin1String("</ul> ");
         }
         output += QLatin1String("</p>");
    }
//    qDebug() << "Output: " << output;
    return output;
}

QString FacebookGraph::formatTimestamp(QDateTime timestamp)
{
    QString timestamp_text;
    if (timestamp.isNull())
        timestamp_text = tr("unknown time");
    else
        timestamp_text = timestamp.toString(Qt::SystemLocaleShortDate);
    return timestamp_text;
}
