/* This file is part of effing
 *
 * Copyright (C) 2010 Richard Merren <richard@merren.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "fbactions.h"
#include "logindialog.h"
#include "constants.h"
#include "messagedto.h"
#include <QByteArray>
#include <QtNetwork/QNetworkAccessManager>
#include <QtNetwork/QNetworkRequest>
#include <QUrl>
#include <QList>
#include <QVariantList>
#include <QVariantMap>
#include <QDebug>
#include <QtScript/QScriptEngine>

FbActions::FbActions(QObject *parent) :
    QObject(parent)
{
    connect(&access_manager,SIGNAL(finished(QNetworkReply*)),this,SLOT(handle_http_response(QNetworkReply*)));
}

void FbActions::request_status()
{
    QString token = get_auth_token();
    if (token.isNull() || token.isEmpty()) {
        emit_simple_message(tr("Not logged in to Facebook"));
    } else {
        emit_simple_message(tr("Requesting status from Facebook..."));
        QUrl url;
        url.setScheme("https");
        url.setHost(AUTH_HOST);
        url.setPath("/me/statuses");
        url.addQueryItem("access_token",token);
        url.addQueryItem("limit","10");
        url.addQueryItem("fields","id,message,updated_time,comments");
        qDebug()<<"about to request url: "<<url.toString();
        QNetworkRequest req(url);
        access_manager.get(req);
    }
}

QString FbActions::get_auth_token()
{
    QString auth_token = settings_manager.get_facebook_auth();
    if (auth_token.isNull() || auth_token.isEmpty()) {
        LoginDialog login_dialog;
        int result = login_dialog.exec();
        if (result==QDialog::Accepted) {
            auth_token = login_dialog.get_auth_token();
            settings_manager.set_facebook_auth(auth_token);
        } else {
            auth_token = QString::null;
        }
    }
    return auth_token;
}

void FbActions::handle_http_response(QNetworkReply *reply)
{
    MessageDTO messageDto;
    messageDto.setCurrentTimestamp();
    if (reply->error()==QNetworkReply::NoError) {
        QByteArray bytes = reply->readAll();
        qDebug()<<"http response: "<<bytes;
        QScriptEngine script_engine;
        QScriptValue script_value = script_engine.evaluate("("+QString(bytes)+")");
        if (script_value.isError())
        {
            messageDto.setText(tr("Unable to parse result."));
        }
        else if (script_value.property("data").isValid())
        {
            //Response from a request for status is a data node
            QScriptValue data = script_value.property("data");
            messageDto.setMessageType(MessageDTO::COLLECTION);
            if (data.isArray())
            {
                int arrayCount = data.property("length").toInt32();
                for (int index=0; index<arrayCount; index++) {
                    MessageDTO statusMessage;
                    QScriptValue data_entry = data.property(QString("%1").arg(index));
                    statusMessage.setMessageType(MessageDTO::STATUS);
                    statusMessage.setText(data_entry.property("message").toString());
                    statusMessage.setTimestamp(parseTime(data_entry.property("updated_time")));
                    if (data_entry.property("comments").isValid()) {
                        QScriptValue comment_data = data_entry.property("comments").property("data");
                        if (comment_data.isValid() && comment_data.isArray())
                        {
                            int commentArrayCount = comment_data.property("length").toInt32();
                            for (int commentIndex=0; commentIndex<commentArrayCount; commentIndex++) {
                                MessageDTO commentMessage;
                                QScriptValue comment_data_entry = comment_data.property(QString("%1").arg(commentIndex));
                                commentMessage.setMessageType(MessageDTO::COMMENT);
                                commentMessage.setText(comment_data_entry.property("message").toString());
                                commentMessage.setTimestamp(parseTime(comment_data_entry.property("created_time")));
                                QScriptValue fromNode = comment_data_entry.property("from");
                                if (fromNode.isValid())
                                {
                                    QScriptValue nameNode = fromNode.property("name");
                                    if (nameNode.isValid()) commentMessage.setAuthor(nameNode.toString());
                                }
                                statusMessage.addChild(commentMessage);
                            }
                        }
                    }
                    messageDto.addChild(statusMessage);
                }
            }
        }
        else if (script_value.property("id").isValid())
        {
            //response from successful submit of post is an id value
            emit status_successfully_set();
            messageDto.setText(tr("Status successfully submitted."));
            request_status();
        }
        else
        {
            messageDto.setText(tr("Unknown response from Facebook."));
        }
    } else {
        messageDto.setText(QString(tr("Network error: %1")).arg(reply->errorString()));
    }
    emit status_available(messageDto.getFormattedMessage());
    reply->deleteLater();
}

void FbActions::unauthorize()
{
    settings_manager.clear_facebook_auth();
    emit_simple_message(tr("Facebook Authorization Removed."));
}

void FbActions::set_status(QString status)
{
    QString token = get_auth_token();
    if (token.isNull() || token.isEmpty()) {
        emit_simple_message(tr("Not logged in to Facebook."));
        return;
    }
    emit_simple_message(tr("Sending status to Facebook..."));
    QUrl url;
    url.setScheme("https");
    url.setHost(AUTH_HOST);
    url.setPath("/me/feed");
    qDebug()<<"about to set status with url: "<<url.toString();
    QNetworkRequest req(url);
    QString encodedStatus(QUrl::toPercentEncoding(status,"",""));
    QString post_data = QString("access_token=%1&message=%2").arg(token,encodedStatus);
    qDebug()<<"using post data: "<<post_data;
    access_manager.post(req,post_data.toAscii());
}

void FbActions::emit_simple_message(QString text)
{
    MessageDTO messageDto;
    messageDto.setText(text);
    messageDto.setCurrentTimestamp();
    emit status_available(messageDto.getFormattedMessage());
}

QDateTime FbActions::parseTime(QScriptValue jsonNode)
{
    QDateTime time;
    if (jsonNode.isDate())
    {
        time = jsonNode.toDateTime();
    }
    else
    {
        //The toDateTime function does not seem to properly parse this date time
        QString timestamp_str = jsonNode.toString();
        time = QDateTime::fromString(timestamp_str,Qt::ISODate);
        //QT doesn't seem to like the time zone data, so we set it explicitly to UTC and then convert to local.
        time.setTimeSpec(Qt::UTC);
        time = time.toLocalTime();
    }
    return time;
}
