/* This file is part of effing
 *
 * Copyright (C) 2010 Richard Merren <richard@merren.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "fbactions.h"
#include "logindialog.h"
#include "constants.h"
#include <QByteArray>
#include <QtNetwork/QNetworkAccessManager>
#include <QtNetwork/QNetworkRequest>
#include <QUrl>
#include <QList>
#include <QVariantList>
#include <QVariantMap>
#include <QDebug>
#include <QtScript/QScriptEngine>
#include <QtScript/QScriptValue>

FbActions::FbActions(QObject *parent) :
    QObject(parent)
{
    connect(&access_manager,SIGNAL(finished(QNetworkReply*)),this,SLOT(handle_http_response(QNetworkReply*)));
}

void FbActions::request_status()
{
    QString message;
    QDateTime timestamp;
    QString token = get_auth_token();
    if (token.isNull() || token.isEmpty()) {
        message = tr("Not logged in to Facebook");
        timestamp = QDateTime::currentDateTime();
        emit status_available(message,timestamp);
    } else {
        emit status_available(tr("Requesting status from Facebook..."),QDateTime::currentDateTime());
        QUrl url;
        url.setScheme("https");
        url.setHost(AUTH_HOST);
        url.setPath("/me/statuses");
        url.addQueryItem("access_token",token);
        url.addQueryItem("limit","1");
        url.addQueryItem("fields","id,message,updated_time,comments");
        qDebug()<<"about to request url: "<<url.toString();
        QNetworkRequest req(url);
        access_manager.get(req);
    }
}

QString FbActions::get_auth_token()
{
    QString auth_token = settings_manager.get_facebook_auth();
    if (auth_token.isNull() || auth_token.isEmpty()) {
        LoginDialog login_dialog;
        int result = login_dialog.exec();
        if (result==QDialog::Accepted) {
            auth_token = login_dialog.get_auth_token();
            settings_manager.set_facebook_auth(auth_token);
        } else {
            auth_token = QString::null;
        }
    }
    return auth_token;
}

void FbActions::handle_http_response(QNetworkReply *reply)
{
    QString message;
    QDateTime timestamp = QDateTime::currentDateTime();
    if (reply->error()==QNetworkReply::NoError) {
        emit status_available(tr("Recieved reply from Facebook..."),timestamp);
        QByteArray bytes = reply->readAll();
        qDebug()<<"http response: "<<bytes;
        QScriptEngine script_engine;
        QScriptValue script_value = script_engine.evaluate("("+QString(bytes)+")");
        if (script_value.isError())
        {
            message = tr("Unable to parse result.");
        }
        else if (script_value.property("data").isValid())
        {
            //Response from a request for status is a data node
            QScriptValue data = script_value.property("data");
            if (data.isArray() && data.property("length").toInt32() > 0)
            {
                QScriptValue data_entry = data.property("0");
                message = data_entry.property("message").toString();
                if (data_entry.property("updated_time").isDate())
                {
                    timestamp = data_entry.property("updated_time").toDateTime();
                }
                else
                {
                    //The toDateTime function does not seem to properly parse this date time
                    QString timestamp_str = data_entry.property("updated_time").toString();
                    timestamp = QDateTime::fromString(timestamp_str,Qt::ISODate);
                    //QT doesn't seem to like the time zone data, so we set it explicitly to UTC and then convert to local.
                    timestamp.setTimeSpec(Qt::UTC);
                    timestamp = timestamp.toLocalTime();

                }
            }
        }
        else if (script_value.property("id").isValid())
        {
            //response from successful submit of post is an id value
            emit status_successfully_set();
            message = tr("Status successfully submitted.");
            request_status();
        }
        else
        {
            message = tr("Unknown response from Facebook.");
        }
    } else {
        message = QString(tr("Network error: %1")).arg(reply->errorString());
    }
    emit status_available(message,timestamp);
    reply->deleteLater();
}

void FbActions::unauthorize()
{
    settings_manager.clear_facebook_auth();
    QString message = tr("Facebook Authorization Removed.");
    QDateTime timestamp = QDateTime::currentDateTime();
    emit status_available(message,timestamp);
}

void FbActions::set_status(QString status)
{
    QString token = get_auth_token();
    if (token.isNull() || token.isEmpty()) {
        emit status_available(tr("Not logged in to Facebook"),QDateTime::currentDateTime());
        return;
    }
    emit status_available(tr("Sending status to Facebook..."),QDateTime::currentDateTime());
    QUrl url;
    url.setScheme("https");
    url.setHost(AUTH_HOST);
    url.setPath("/me/feed");
    qDebug()<<"about to set status with url: "<<url.toString();
    QNetworkRequest req(url);
    QString encodedStatus(QUrl::toPercentEncoding(status,"",""));
    QString post_data = QString("access_token=%1&message=%2").arg(token,encodedStatus);
    qDebug()<<"using post data: "<<post_data;
    access_manager.post(req,post_data.toAscii());
}

