#include <string.h>
#include "vcard-editor.h"
#include "util.h"

#define MAGICC 0xFFFC
#define MAGIC get_magic ()
static const char*
get_magic (void)
{
    static char *magic;
    GString *str;

    if (G_LIKELY (magic))
        return magic;

    str = g_string_new ("");
    g_string_append_unichar (str, MAGICC);
    magic = g_string_free (str, FALSE);

    return magic;
}

enum {
    PROP_NONE,
    PROP_CONTACT,
    PROP_BOOK
};

typedef struct _VCardEditorPrivate VCardEditorPrivate;

struct _VCardEditorPrivate
{
    GtkWidget *text_view;

    EContact *contact;
    EBook *book;

    char *photo_data;
};

#define VCARD_EDITOR_PRIVATE(o) \
(G_TYPE_INSTANCE_GET_PRIVATE ((o), VCARD_EDITOR_TYPE, VCardEditorPrivate))

static void vcard_editor_class_init (VCardEditorClass *klass);
static void vcard_editor_init       (VCardEditor *self);
static void vcard_editor_dispose    (GObject *object);
static void vcard_editor_finalize   (GObject *object);
static void vcard_editor_set_property (GObject *object, guint property_id, const GValue *value, GParamSpec *pspec);
static void vcard_editor_get_property (GObject *object, guint property_id, GValue *value, GParamSpec *pspec);

static void on_save_vcard_btn_clicked (GtkToolButton *button, gpointer user_data);
static void on_contact_changed (GObject *gobject, GParamSpec *pspec, gpointer user_data);

G_DEFINE_TYPE (VCardEditor, vcard_editor, WINDOW_TYPE);

static void
vcard_editor_class_init (VCardEditorClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);

    g_type_class_add_private (klass, sizeof (VCardEditorPrivate));

    object_class->set_property = vcard_editor_set_property;
    object_class->get_property = vcard_editor_get_property;
    object_class->dispose = vcard_editor_dispose;
    object_class->finalize = vcard_editor_finalize;

    g_object_class_install_property
            (object_class, PROP_BOOK,
             g_param_spec_object
                    ("book",
                     "EBook",
                     "The EBook to which the contact will be saved",
                     E_TYPE_BOOK,
                     G_PARAM_READWRITE | G_PARAM_STATIC_NAME |
                     G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
    g_object_class_install_property
            (object_class, PROP_CONTACT,
             g_param_spec_object
                    ("contact",
                     "EContact",
                     "The EContact which is editited",
                     E_TYPE_CONTACT,
                     G_PARAM_READWRITE | G_PARAM_STATIC_NAME |
                     G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
}

static void
vcard_editor_init (VCardEditor *self)
{
    VCardEditorPrivate *priv;
    GtkWidget *vbox;
    GtkWidget *sw;
    GtkWidget *toolbar;
    GtkToolItem *toolitem;

    priv = VCARD_EDITOR_PRIVATE (self);

    g_object_set (G_OBJECT (self), "destroy-with-parent", TRUE, NULL);

    vbox = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (self), vbox);

    sw = gtk_scrolled_window_new (NULL, NULL);
    gtk_box_pack_start (GTK_BOX (vbox), sw, TRUE, TRUE, 0);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
                                    GTK_POLICY_NEVER,
                                    GTK_POLICY_AUTOMATIC);

    priv->text_view = gtk_widget_new (TEXT_VIEW_TYPE, NULL);
    gtk_container_add (GTK_CONTAINER (sw), priv->text_view);

    /* Toolbar */
    toolbar = gtk_toolbar_new ();
    gtk_box_pack_start (GTK_BOX (vbox), toolbar, FALSE, FALSE, 0);

    toolitem = gtk_tool_button_new (NULL, NULL);
    gtk_tool_button_set_icon_name (GTK_TOOL_BUTTON (toolitem), SAVE_ICON);
    gtk_toolbar_insert (GTK_TOOLBAR (toolbar), toolitem, -1);
    g_signal_connect (toolitem, "clicked",
                      G_CALLBACK (on_save_vcard_btn_clicked), self);

    gtk_widget_show_all (vbox);

    g_signal_connect (self, "notify::contact", G_CALLBACK (on_contact_changed), self);
}

static void
vcard_editor_dispose (GObject *object)
{
    VCardEditorPrivate *priv;

    priv = VCARD_EDITOR_PRIVATE (object);

    if (priv->contact) {
        g_object_unref (priv->contact);
        priv->contact = NULL;
    }
    if (priv->book) {
        g_object_unref (priv->book);
        priv->book = NULL;
    }

    G_OBJECT_CLASS (vcard_editor_parent_class)->dispose (object);
}

static void
vcard_editor_finalize (GObject *object)
{
    VCardEditorPrivate *priv;

    priv = VCARD_EDITOR_PRIVATE (object);
    g_free (priv->photo_data);

    G_OBJECT_CLASS (vcard_editor_parent_class)->finalize (object);
}

static void
vcard_editor_set_property (GObject *object, guint property_id, const GValue *value, GParamSpec *pspec)
{
    VCardEditorPrivate *priv;

    priv = VCARD_EDITOR_PRIVATE (object);

    switch (property_id) {
    case PROP_BOOK:
        if (priv->book) {
            g_object_unref (priv->book);
        }
        priv->book = g_value_dup_object (value);
        g_object_notify (object, "book");
        break;
    case PROP_CONTACT:
        if (priv->contact) {
            g_object_unref (priv->contact);
        }
        priv->contact = g_value_dup_object (value);
        g_object_notify (object, "contact");
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
        break;
    }
}

static void
vcard_editor_get_property (GObject *object, guint property_id, GValue *value, GParamSpec *pspec)
{
    VCardEditorPrivate *priv;

    priv = VCARD_EDITOR_PRIVATE (object);

    switch (property_id) {
    case PROP_BOOK:
        g_value_set_object (value, priv->book);
        break;
    case PROP_CONTACT:
        g_value_set_object (value, priv->contact);
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
        break;
    }
}

GtkWidget*
vcard_editor_new (GtkWindow *parent)
{
    return g_object_new (VCARD_EDITOR_TYPE, "transient-for", parent, NULL);
}

static void
on_save_vcard_btn_clicked (GtkToolButton *button, gpointer user_data)
{
    VCardEditorPrivate *priv;
    GtkTextBuffer *buffer;
    GtkTextIter start, end;
    EContact *contact;
    GError *error = NULL;
    char *vcard;
    const char *place_holder;

    priv = VCARD_EDITOR_PRIVATE (user_data);
    g_return_if_fail (priv->book != NULL);

    buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (priv->text_view));
    gtk_text_buffer_get_start_iter (buffer, &start);
    gtk_text_buffer_get_end_iter (buffer, &end);

    vcard = gtk_text_buffer_get_slice (buffer, &start, &end, FALSE);
    place_holder = g_utf8_strchr (vcard, -1, MAGICC);
    if (place_holder && priv->photo_data) {
        GString *str;
        char **tokens;

        tokens = g_strsplit (vcard, MAGIC, 2);
        str = g_string_new (tokens[0]);
        g_string_append (str, priv->photo_data);
        g_string_append (str, tokens[1]);
        g_strfreev (tokens);
        g_free (vcard);
        vcard = g_string_free (str, FALSE);
    }

    contact = e_contact_new_from_vcard (vcard);
    g_free (vcard);
    e_book_commit_contact (priv->book, contact, &error);
    if (error) {
        g_printerr ("Could not save contact: %s\n", error->message);
        g_clear_error (&error);
    }
    g_object_unref (contact);

    gtk_widget_destroy (GTK_WIDGET (user_data));
}

static void
on_contact_changed (GObject *object, GParamSpec *pspec, gpointer user_data)
{
    VCardEditorPrivate *priv;
    GtkTextBuffer *buffer;
    GdkPixbuf *pixbuf = NULL;
    char *vcard;

    priv = VCARD_EDITOR_PRIVATE (object);

    buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (priv->text_view));

    g_free (priv->photo_data);
    priv->photo_data = NULL;
    if (priv->contact) {
        EContactPhoto *photo;

        photo = e_contact_get (priv->contact, E_CONTACT_PHOTO);
        if (photo) {
            if (photo->type == E_CONTACT_PHOTO_TYPE_INLINED) {
                GdkPixbufLoader *loader;
                EVCardAttribute *attr;

                loader = gdk_pixbuf_loader_new ();
                gdk_pixbuf_loader_set_size (loader, 24, 24);
                gdk_pixbuf_loader_write (loader, photo->data.inlined.data,
                                         photo->data.inlined.length,
                                         NULL);
                gdk_pixbuf_loader_close (loader, NULL);
                pixbuf = gdk_pixbuf_loader_get_pixbuf (loader);
                if (pixbuf) {
                    g_object_ref (pixbuf);
                }
                g_object_unref (loader);

                if (pixbuf) {
                    attr = e_vcard_get_attribute (E_VCARD (priv->contact), "PHOTO");
                    priv->photo_data = e_vcard_attribute_get_value (attr);
                    e_vcard_attribute_remove_values (attr);
                    e_vcard_attribute_add_value (attr, MAGIC);
                }
            }
        }
        vcard = e_vcard_to_string (E_VCARD (priv->contact), EVC_FORMAT_VCARD_30);
    } else {
        vcard = NULL;
    }

    if (pixbuf) {
        char **tokens;
        GtkTextIter end;

        tokens = g_strsplit (vcard, MAGIC, 2);
        gtk_text_buffer_set_text (buffer, tokens[0], -1);
        gtk_text_buffer_get_end_iter (buffer, &end);
        gtk_text_buffer_insert_pixbuf (buffer, &end, pixbuf);
        gtk_text_buffer_get_end_iter (buffer, &end);
        gtk_text_buffer_insert (buffer, &end, tokens[1], -1);

        g_strfreev (tokens);
    } else {
        gtk_text_buffer_set_text (buffer, vcard, -1);
    }
}
