/* ebook-client.c - A small general purpose ebook cmdline client
 *
 * Copyright (C) 2008 Nokia Corp.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Joergen Scheibengruber <jorgen.scheibengruber AT nokia.com>
 */

#include "config.h"
#include <string.h>
#include <libebook/e-book.h>
#include "util.h"

typedef struct _Match Match;
struct _Match
{
    EContact *contact;
    GList *matches;

    int ref_count;
};

static Match*
match_new (EContact *contact)
{
    Match *match;

    match = g_slice_new0 (Match);
    match->contact = g_object_ref (contact);

    match->ref_count = 1;

    return match;
}

static Match*
match_add (Match *match, EContact *contact)
{
    match->matches = g_list_prepend (match->matches, g_object_ref (contact));

    return match;
}

static Match*
match_ref (Match *match)
{
    g_atomic_int_inc (&match->ref_count);

    return match;
}

static void
match_unref (Match *match)
{
    if (g_atomic_int_dec_and_test (&match->ref_count)) {

        g_object_unref (match->contact);
        g_list_foreach (match->matches, (GFunc)g_object_unref, NULL);
        g_list_free (match->matches);

        g_slice_free (Match, match);
    }
}

static char*
serialize_string_list (GList *l, gboolean drop_empty, char *separator)
{
    GString *result = NULL;

    while (l) {
        if (!drop_empty || (l->data && ((char*)l->data)[0])) {
            if (!result) {
                result = g_string_new ("");
            } else {
                g_string_append (result, separator);
            }
            g_string_append (result, l->data ? l->data : "");
        }
        l = l->next;
    }
    return result ? g_string_free (result, FALSE) : NULL;
}

static char*
serialize_attrib (EVCardAttribute *attrib)
{
    GString *result;
    GList *values;
    char *serialized_values;

    result = g_string_new (e_vcard_attribute_get_name (attrib));
    g_string_append (result, ":");
    values = e_vcard_attribute_get_values (attrib);
    serialized_values = serialize_string_list (values, FALSE, ";");
    if (serialized_values) {
        g_string_append (result, serialized_values);
        g_free (serialized_values);
    }

    return g_string_free (result, FALSE);
}

static inline gboolean
is_unique_attrib (const char *attrib_name)
{
    const char *unique_attribs[] = {"N", "FN", "UID", "REV", "NICKNAME", "ORG", "TITLE", "PHOTO", "BDAY", "X-EVOLUTION-FILE-AS", NULL};
    const char **itr;

    for (itr = unique_attribs; *itr; itr++) {
        if (0 == strcmp (*itr, attrib_name)) {
            return TRUE;
        }
    }
    return FALSE;
}

static inline gboolean
is_empty (const char *value)
{
    if (value && value[0])
        return FALSE;
    return TRUE;
}

static char*
make_human_readable (EVCardAttribute *attr, EVCardAttribute *existing)
{
    const char *name;
    GList *attr_values;
    GList *existing_values;
    const char *attr_value;
    const char *existing_value;

    name = e_vcard_attribute_get_name (attr);
    attr_values = e_vcard_attribute_get_values (attr);
    attr_value = attr_values ? attr_values->data : NULL;
    existing_values = e_vcard_attribute_get_values (attr);
    existing_value = existing_values ? existing_values->data : NULL;

    if (0 == strcmp (name, "NICKNAME")) {
        if (g_strcmp0 (attr_value, existing_value) && !is_empty (attr_value)) {
            return g_strdup_printf ("Nickname: %s", attr_value);
        }
        return NULL;
    }
    if (0 == strcmp (name, "TITLE")) {
        if (g_strcmp0 (attr_value, existing_value) && !is_empty (attr_value)) {
            return g_strdup_printf ("Title: %s", attr_value);
        }
        return NULL;
    }
    if (0 == strcmp (name, "BDAY")) {
        EContactDate *attr_date, *existing_date;
        char *rv = NULL;

        attr_date = attr_value ? e_contact_date_from_string (attr_value) : NULL;
        existing_date = existing_value ? e_contact_date_from_string (existing_value) : NULL;

        if (!existing_date || (attr_date && !e_contact_date_equal (attr_date, existing_date))) {
            char *date = e_contact_date_to_string (attr_date);
            rv = g_strdup_printf ("Birthday: %s", date);
            g_free (date);
        }
        if (attr_date) {
            e_contact_date_free (attr_date);
        }
        if (existing_date) {
            e_contact_date_free (existing_date);
        }
        return rv;
    }
    if (0 == strcmp (name, "ORG")) {
        char *existing_org;
        char *attr_org;
        char *rv = NULL;

        existing_org = serialize_string_list (existing_values, TRUE, ", ");
        attr_org = serialize_string_list (attr_values, TRUE, ", ");

        if (g_strcmp0 (attr_org, existing_org) && !is_empty (attr_org)) {
            rv = g_strdup_printf ("Company: %s", attr_org);
        }
        g_free (existing_org);
        g_free (attr_org);
        return rv;
    }

    return NULL;
}

static gboolean
merge (EContact *dest, GList *new_attribs)
{
    GList *attribs, *l;
    GHashTable *unique, *multi, *to_add;
    GString *dups = NULL;
    gboolean modified = FALSE;

    unique = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);
    multi = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);

    attribs = e_vcard_get_attributes (E_VCARD (dest));
    for (l = attribs; l; l = l->next) {
        EVCardAttribute *attrib = l->data;
        const char *name;
        GList *values;

        name = e_vcard_attribute_get_name (attrib);
        values = e_vcard_attribute_get_values (attrib);
        if (values) {
            if (is_unique_attrib (name)) {
                g_hash_table_insert (unique, g_strdup (name), attrib);
            } else {
                char *value;

                value = serialize_attrib (attrib);
                g_hash_table_insert (unique, value, attrib);
            }
        }
    }

    for (l = new_attribs; l; l = l->next) {
        EVCardAttribute *attrib = l->data;
        const char *name;

        name = e_vcard_attribute_get_name (attrib);
        if (is_unique_attrib (name)) {
            EVCardAttribute *exists;

            exists = g_hash_table_lookup (unique, name);
            if (exists) {
                char *human_readable;
                human_readable = make_human_readable (attrib, exists);
                if (human_readable) {
                    if (!dups) {
                        dups = g_string_new ("");
                    } else {
                        g_string_append (dups, "\n");
                    }
                    g_string_append (dups, human_readable);
                }
            } else {
                EVCardAttribute *copy;
                copy = e_vcard_attribute_copy (attrib);
                e_vcard_add_attribute (E_VCARD (dest), copy);
                modified = TRUE;
            }
        } else {
            EVCardAttribute *exists;
            char *value;

            value = serialize_attrib (attrib);
            exists = g_hash_table_lookup (multi, value);
            g_free (value);
            if (exists) {
                GList *params;

                params = e_vcard_attribute_get_params (exists);
                if (!params) {
                    GList *l;
                    params = e_vcard_attribute_get_params (attrib);
                    for (l = params; l; l = l->next) {
                        EVCardAttributeParam *copy;

                        copy = e_vcard_attribute_param_copy (l->data);
                        e_vcard_attribute_add_param (exists, copy);
                        modified = TRUE;
                    }
                }
            } else {
                EVCardAttribute *copy;
                copy = e_vcard_attribute_copy (attrib);
                e_vcard_add_attribute (E_VCARD (dest), copy);
                modified = TRUE;
            }
        }
    }
    if (dups) {
        EVCardAttribute *attr;

        attr = e_vcard_attribute_new (NULL, "NOTE");
        e_vcard_attribute_add_value (attr, dups->str);
        e_vcard_add_attribute (E_VCARD (dest), attr);
        modified = TRUE;
        g_warning ("Adding dups to note field: %s", dups->str);
        g_string_free (dups, TRUE);
    }
    return modified;
}

static EContact*
match_merge (Match *match)
{
    EContact *result;
    GList *l;
    GList *new_attribs = NULL;
    gboolean modified;

    if (!match->matches) {
        return NULL;
    }

    result = e_contact_duplicate (match->contact);
    for (l = match->matches; l; l = l->next) {
        GList *attribs;

        attribs = e_vcard_get_attributes (E_VCARD (l->data));
        new_attribs = g_list_concat (new_attribs, g_list_copy (attribs));
    }
    modified = merge (result, new_attribs);
    g_list_free (new_attribs);

    if (modified) {
        return result;
    } else {
        g_object_unref (result);
        return NULL;
    }
}

static gboolean
populate_matches_from_book (EBook *book, GHashTable *matches, GError **error)
{
    EBookQuery *query;
    GList *contacts;
    gboolean rv;

    query = e_book_query_from_string ("(contains \"x-evolution-any-field\" \"\")");
    rv = e_book_get_contacts (book, query, &contacts, error);
    e_book_query_unref (query);
    if (FALSE == rv) {
        return FALSE;
    }

    while (contacts) {
        EContact *contact = contacts->data;
        Match *match;
        const char *name;

        name = _e_contact_get_name (contact);
        if (name && name[0]) {
            match = match_new (contact);
            g_hash_table_insert (matches, g_strdup (name), match);
        }
        g_object_unref (contact);
        contacts = g_list_delete_link (contacts, contacts);
    }

    return TRUE;
}

static gboolean
populate_matches_from_vcf (const char *filename, GHashTable *matches, GList **no_matches, GError **error)
{
    char *contents = NULL;
    char **vcards = NULL;
    int i = -1;
    gboolean rv;

    *no_matches = NULL;
    if (0 == strcmp (filename, "-")) {
        filename = "/dev/stdin";
    }
    rv = g_file_get_contents (filename, &contents, NULL, error);
    if (FALSE == rv) {
        return FALSE;
    }

    vcards = g_strsplit (contents, "END:VCARD\n", -1);
    g_free (contents);
    for (i = 0; vcards[i] && vcards[i][0]; i++) {
        EContact *contact;
        char *vcard;

        vcard = g_strdup_printf ("%sEND:VCARD\n", vcards[i]);
        contact = e_contact_new_from_vcard (vcard);
        if (contact) {
            const char *name;

            name = _e_contact_get_name (contact);
            if (name && name[0]) {
                Match *match;

                match = g_hash_table_lookup (matches, name);
                if (match) {
                    match_add (match, contact);
                    g_hash_table_insert (matches, g_strdup (name), match_ref (match));
                } else {
                    *no_matches = g_list_prepend (*no_matches, g_object_ref (contact));
                }
            }
            g_object_unref (contact);
        }
    }
    g_strfreev (vcards);

    return TRUE;
}

static void
compute_updates (GHashTable *matches, GList **contacts_to_modify)
{
    GHashTableIter iter;
    gpointer key, value;

    *contacts_to_modify = NULL;

    g_hash_table_iter_init (&iter, matches);
    while (g_hash_table_iter_next (&iter, &key, &value)) {
        Match *match = value;
        EContact *result;

        result = match_merge (match);
        if (result) {
            *contacts_to_modify = g_list_prepend (*contacts_to_modify, result);
        }
    }
}

static void
print_contact (EContact *contact)
{
#if 0
    EContact *copy;
    EContactPhoto *photo;
    char *vcard;

    copy = e_contact_duplicate (contact);
    photo = e_contact_get (copy, E_CONTACT_PHOTO);
    if (photo && photo->type == E_CONTACT_PHOTO_TYPE_INLINED) {
        EContactPhoto *photo2 = g_new0 (EContactPhoto, 1);

        photo2->type = E_CONTACT_PHOTO_TYPE_URI;
        photo2->data.uri = g_strdup ("data inlined");
        e_contact_set (copy, E_CONTACT_PHOTO, photo2);
        e_contact_photo_free (photo2);
    }
    e_contact_photo_free (photo);

    vcard = e_vcard_to_string (E_VCARD (copy), EVC_FORMAT_VCARD_30);
    g_object_unref (copy);
    g_print ("%s\n", vcard);
    g_free (vcard);
#endif
    g_debug ("%s", _e_contact_get_name (contact));
}

static void
print_contacts (GList *contacts)
{
    while (contacts) {
        EContact *contact = contacts->data;

        print_contact (contact);
        g_print ("\n");
        contacts = contacts->next;
    }
}

int main (int argc, char *argv [])
{
    GError *error = NULL;
    GOptionContext *context;
    EBook *book;
    GHashTable *matches = NULL;
    GList *contacts_to_add = NULL;
    GList *contacts_to_modify = NULL;
    gboolean rv = TRUE;

    g_type_init ();

    if (argc != 2) {
        g_printerr ("Usage: %s file.vcf\n", argv[0]);
        return 1;
    }

    book = e_book_new_system_addressbook (&error);
    if (error) {
        g_printerr ("Could create system addressbook: %s\n", error->message);
        g_clear_error (&error);
        return 1;
    }

    e_book_open (book, TRUE, &error);
    if (error) {
        g_printerr ("Could not open system addressbook: %s\n", error->message);
        g_clear_error (&error);
        goto out;
    }

    matches = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, (GDestroyNotify)match_unref);
    rv = populate_matches_from_book (book, matches, &error);
    if (error) {
        g_printerr ("Could not get matches from system addressbook: %s\n", error->message);
        g_clear_error (&error);
        goto out;
    }

    rv = populate_matches_from_vcf (argv[1], matches, &contacts_to_add, &error);
    if (error) {
        g_printerr ("Could not load matches from %s: %s\n", argv[1], error->message);
        g_clear_error (&error);
        goto out;
    }

    compute_updates (matches, &contacts_to_modify);
    if (contacts_to_add) {
        //g_debug ("Contacts to add:");
        //print_contacts (contacts_to_add);
        rv = e_book_add_contacts (book, contacts_to_add, &error);
    }
    if (error) {
        g_printerr ("Could not add contacts: %s\n", error->message);
        g_clear_error (&error);
        goto out;
    }

    if (contacts_to_modify) {
        //g_debug ("Contacts to modify:");
        //print_contacts (contacts_to_modify);
        rv = e_book_commit_contacts (book, contacts_to_modify, &error);
    }
    if (error) {
        g_printerr ("Could not modify contacts: %s\n", error->message);
        g_clear_error (&error);
        goto out;
    }

out:
    g_object_unref (book);
    if (matches) {
        g_hash_table_destroy (matches);
    }
    if (contacts_to_add) {
        g_list_foreach (contacts_to_add, (GFunc)g_object_unref, NULL);
        g_list_free (contacts_to_add);
    }
    if (contacts_to_modify) {
        g_list_foreach (contacts_to_modify, (GFunc)g_object_unref, NULL);
        g_list_free (contacts_to_modify);
    }
    return (rv == TRUE);
}
