/*
  DiskUsage - shows the disk usage under a number of operating systems.
  Copyright (C) 2004  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "DiskInfo.h"

#if defined(linux)
  #include <string.h>
  #include <sys/vfs.h>
  #include <mntent.h>
#elif defined(sun)
  #include <stdio.h>
  #include <string.h>
  #include <sys/mnttab.h>
  #include <sys/types.h>
  #include <sys/statvfs.h>
#elif defined(__APPLE__)
  #include <sys/param.h>
  #include <sys/ucred.h>
  #include <sys/mount.h>
#elif defined(__WIN32__) || defined(WIN32)
  #include <windows.h>
  #include <winnt.h>
  #undef GetObject
#endif

#include <iostream>

#include <Lum/Base/String.h>

bool GetDisks(std::list<DiskInfo>& list)
{
#if defined(linux)
  FILE          *file;
  struct mntent *inf;
  struct statfs stats;
  DiskInfo      info;

  file=setmntent("/etc/mtab","r");

  if (file==NULL) {
    return false;
  }

  std::cout << "---" << std::endl;
  memset(&stats,0,sizeof(stats));
  inf=getmntent(file);
  while (inf!=NULL) {
    if (statfs(inf->mnt_dir,&stats)==0 &&
        stats.f_blocks>0 &&
        strcmp(inf->mnt_type,"bind")!=0 &&
        strcmp(inf->mnt_type,"tmpfs")!=0 &&
        strcmp(inf->mnt_type,"none")!=0) {
      info.name=Lum::Base::StringToWString(inf->mnt_fsname);
      info.dir=Lum::Base::StringToWString(inf->mnt_dir);
      info.blocks=stats.f_blocks;
      info.free=stats.f_bfree;
      info.avail=stats.f_bavail;
      info.unit=stats.f_bsize/1024;

      std::cout << inf->mnt_dir << ": " << inf->mnt_type << std::endl;

      list.push_back(info);
    }
    else {
      std::cout << "...skipping " << inf->mnt_dir << " " << inf->mnt_type << std::endl;
    }
    inf=getmntent(file);
  }

  endmntent(file);

  return true;
#elif defined(sun)
  FILE           *file;
  struct mnttab  tab;
  struct statvfs stats;
  DiskInfo       info;

  file=fopen("/etc/mnttab","r");

  if (file==NULL) {
    return false;
  }

  while (getmntent(((Private*)handle)->handle,&tab)==0) {
    memset(&stats,0,sizeof(stats));
    if ((statvfs(tab.mnt_mountp,&stats)==0)) {
      info.name=Lum::Base::StringToWString(tab.mnt_special);
      info.dir=Lum::Base::StringToWString(list[x].f_mntonname);
      info.blocks=stats.f_blocks;
      info.free=stats.f_bfree;
      info.avail=stats.f_bavail;
      info.unit=1; // TODO

      list.push_back(info);
    }
    else {
      break;
    }
  }

  fclose(file);

  return true;
#elif defined(__APPLE__)
  struct statfs* devlist;
  size_t         size;
  DiskInfo       info;

  size=getmntinfo(&devlist,MNT_NOWAIT);

  if (size==0) {
    return false;
  }
  for (size_t x=0; x<size; x++) {
    info.name=Lum::Base::StringToWString(devlist[x].f_fstypename);
    info.dir=Lum::Base::StringToWString(devlist[x].f_mntonname);
    info.blocks=devlist[x].f_blocks;
    info.free=devlist[x].f_bfree;
    info.avail=devlist[x].f_bavail;
    info.unit=1; // TODO

    list.push_back(info);
  }

  return true;
#elif defined(__WIN32__) || defined(WIN32)
  UINT    old;
  wchar_t buffer[10240];
  wchar_t buffer2[MAX_PATH+1];
  size_t  offset=0;
  DWORD   compLength,flags;
  //UINT    type;

  old=SetErrorMode(SEM_FAILCRITICALERRORS);
  GetLogicalDriveStringsW(1024,buffer);

  while (buffer[offset]!='\0') {
    DiskInfo       info;
    ULARGE_INTEGER available,bytes,free;

    size_t x=0;
    while (buffer[offset]!='\0') {
      info.name.append(1,buffer[offset]);
      info.dir.append(1,buffer[offset]);
      ++x;
      ++offset;
    }
    ++offset;

    /*
    type=GetDriveTypeW(info.name.c_str());
    if (type==DRIVE_UNKNOWN || type==DRIVE_NO_ROOT_DIR || type==DRIVE_REMOVABLE) {
      continue;
    }*/

    if (GetVolumeInformationW(info.name.c_str(),buffer2,MAX_PATH+1,NULL,&compLength,&flags,NULL,0)!=0) {
      info.dir=buffer2;

      if (info.dir==L"OS") {
        info.dir=info.name;
      }
    }

    if (GetDiskFreeSpaceExW(info.name.c_str(),&available,&bytes,&free)) {
      info.blocks=bytes.QuadPart/1024;
      info.free=free.QuadPart/1024;
      info.avail=available.QuadPart/1024;
      info.unit=1;
      list.push_back(info);
    }
  }

  SetErrorMode(old);

  return true;
#else
  return false;
#endif
}

