/*
 *  datetoday home widget for the maemo desktop.
 *  Copyright (C) 2010 Nicolai Hess
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <calendar-backend/CalInterface.h>
#include <libhildondesktop/libhildondesktop.h>
#include <clockd/libtime.h>
#include "datetoday-home-widget.h"
#include <math.h>
	
HD_DEFINE_PLUGIN_MODULE(DateTodayPlugin, date_today_plugin, HD_TYPE_HOME_PLUGIN_ITEM)


#define DATE_TODAY_SETTINGS_FILE "/.datetoday_home_widget"
#define HOME_DIR g_get_home_dir()

void 
add_time_out_handler(DateTodayPlugin *desktop_plugin);

guint
date_today_get_current_day()
{
  const time_t current_time = time(NULL);
  struct tm *local_time;
  local_time = localtime(&current_time);
  return local_time->tm_mday;
}

gboolean
date_today_update_day(gpointer data)
{
  g_return_val_if_fail(data, FALSE);
  DateTodayPlugin* desktop_plugin = DATE_TODAY_PLUGIN(data);
  guint today = date_today_get_current_day();
  if(today!=desktop_plugin->today)
    {
      desktop_plugin->today = today;
      gtk_widget_queue_draw_area(desktop_plugin->content,
				 0, 0, desktop_plugin->size, desktop_plugin->size);
    }
  add_time_out_handler(desktop_plugin);
  return FALSE;
}

void 
add_time_out_handler(DateTodayPlugin *desktop_plugin)
{
  const time_t current_time = time(NULL);
  struct tm *local_time;
  local_time = localtime(&current_time);
  int diff = 24 - local_time->tm_hour;
  diff = diff * 60 * 60 - 60*local_time->tm_min - local_time->tm_sec;
  desktop_plugin->time_out_handler = g_timeout_add_seconds(diff, 
							   date_today_update_day, 
							   (gpointer) desktop_plugin);
}

void 
remove_time_out_handler(DateTodayPlugin *desktop_plugin)
{
  g_source_remove(desktop_plugin->time_out_handler);
  desktop_plugin->time_out_handler = 0;  
}

static GtkWidget*
build_ui(gint size)
{
  GtkVBox *contents = GTK_VBOX(gtk_vbox_new(0, FALSE));
  gtk_widget_set_size_request(GTK_WIDGET(contents), size, size);
  gtk_widget_show_all(GTK_WIDGET(contents));
  gtk_widget_add_events(GTK_WIDGET(contents), GDK_BUTTON_PRESS_MASK);
  return GTK_WIDGET(contents);
}

static void
date_today_on_current_desktop(GtkWidget* widget,
			      gpointer data)
{
  DateTodayPlugin *desktop_plugin = DATE_TODAY_PLUGIN(widget);
  gboolean on;
  g_object_get(widget, "is-on-current-desktop", &on, NULL);
  if(on)
    {
      date_today_update_day(desktop_plugin);
    }
  if(on && !desktop_plugin->time_out_handler)
    {
      add_time_out_handler(desktop_plugin);
      return;
    }
  if(!on && desktop_plugin->time_out_handler)
    {
      remove_time_out_handler(desktop_plugin);
    }
}

static DBusHandlerResult
handle_time_change(DBusConnection *bus, DBusMessage *msg, gpointer data)
{
  date_today_update_day(data);
  return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static void
date_today_show_settings_dialog(GtkWidget* widget, gpointer data)
{
  DateTodayPlugin *desktop_plugin = DATE_TODAY_PLUGIN(widget);
  GtkWidget *dialog = gtk_dialog_new_with_buttons("Settings",
						  NULL,
						  0,
						  "Save", 
						  GTK_RESPONSE_ACCEPT, 
						  NULL);
  GtkWidget* content = gtk_dialog_get_content_area(GTK_DIALOG(dialog));
  GtkSizeGroup *group = GTK_SIZE_GROUP(gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL));
  GtkWidget *selector = hildon_touch_selector_new_text();
  GtkWidget *sizeBox = gtk_hbox_new (FALSE, 0);
  GtkWidget* sizeScale = gtk_hscale_new_with_range(50, 200, 10);
  GtkWidget *sizeLabel = gtk_label_new ("");
  GtkWidget *launch_view_button;

  gtk_label_set_markup (GTK_LABEL (sizeLabel), "Size");
  gtk_range_set_value(GTK_RANGE(sizeScale), desktop_plugin->size);
  
  gtk_box_pack_start (GTK_BOX (sizeBox), sizeScale, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (sizeBox), sizeLabel, FALSE, FALSE, 0);


  hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector), "month");
  hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector), "week");
  hildon_touch_selector_append_text (HILDON_TOUCH_SELECTOR (selector), "agenda");

  launch_view_button = hildon_picker_button_new (HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT, 
						 HILDON_BUTTON_ARRANGEMENT_HORIZONTAL);

  hildon_touch_selector_set_active(HILDON_TOUCH_SELECTOR(selector), 0, desktop_plugin->launch_view -1);

  hildon_picker_button_set_selector (HILDON_PICKER_BUTTON (launch_view_button),
                                     HILDON_TOUCH_SELECTOR (selector));
  hildon_button_set_title(HILDON_BUTTON(launch_view_button), "Calendar View");

  gtk_container_add(GTK_CONTAINER(content), launch_view_button);
  gtk_container_add(GTK_CONTAINER(content), sizeBox);

  gtk_widget_show_all(dialog);

  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT) 
  {
    desktop_plugin->launch_view = hildon_touch_selector_get_active(HILDON_TOUCH_SELECTOR(selector), 0) + 1;
    desktop_plugin->size = gtk_range_get_value(GTK_RANGE(sizeScale));
    save_settings(desktop_plugin);
    gtk_widget_set_size_request(GTK_WIDGET(desktop_plugin), desktop_plugin->size, desktop_plugin->size);
    gtk_window_resize (GTK_WINDOW (desktop_plugin),
		       (int)(desktop_plugin->size),
		       (int)(desktop_plugin->size));
    gtk_widget_queue_draw_area(desktop_plugin->content,
			       0, 0, desktop_plugin->size, desktop_plugin->size);
  }
  gtk_widget_destroy (dialog);
}

void 
read_settings(DateTodayPlugin *desktop_plugin)
{
  GKeyFile *keyFile;
  gchar* fileName;
  gboolean fileExists;
  keyFile = g_key_file_new();
  fileName = g_strconcat(HOME_DIR, DATE_TODAY_SETTINGS_FILE, NULL);
  fileExists = g_key_file_load_from_file (keyFile, fileName, G_KEY_FILE_KEEP_COMMENTS, NULL);
  if(fileExists)
  {
    GError *error=NULL;
    desktop_plugin->launch_view = g_key_file_get_integer(keyFile, "config", "launchView", &error);
    if(!error)
    {
      desktop_plugin->size = g_key_file_get_integer(keyFile, "config", "size", &error);
      if(!error)
      {
	g_key_file_free(keyFile);
	g_free(fileName);
	return;
      }
    }
    g_error_free(error);
    error = NULL;
  }
  desktop_plugin->size = 100;
  desktop_plugin->launch_view = 1;
  g_key_file_free(keyFile);
  g_free(fileName);
}

void
save_settings(DateTodayPlugin* desktop_plugin)
{
  GKeyFile *keyFile;
  gchar *fileData;
  FILE *iniFile;
  gsize size;
  gchar *filename;

  keyFile = g_key_file_new();
  g_key_file_set_integer (keyFile, "config", "launchView", desktop_plugin->launch_view);
  g_key_file_set_integer (keyFile, "config", "size", desktop_plugin->size);
  filename = g_strconcat (HOME_DIR, DATE_TODAY_SETTINGS_FILE, NULL);
  fileData = g_key_file_to_data (keyFile, &size, NULL);

  g_file_set_contents(filename, fileData, size, NULL);
  g_key_file_free (keyFile);
  g_free (fileData);
  g_free (filename);
}

void
register_dbus_signal_on_time_change(DateTodayPlugin* desktop_plugin)
{
  desktop_plugin->dbus_conn = NULL;
  desktop_plugin->dbus_proxy = NULL;
  desktop_plugin->dbus_conn = hd_home_plugin_item_get_dbus_g_connection ( &desktop_plugin->hitem, 
									  DBUS_BUS_SYSTEM, 
									  NULL );

  if(desktop_plugin->dbus_conn)
  {
    desktop_plugin->dbus_proxy = dbus_g_proxy_new_for_name(desktop_plugin->dbus_conn,
							   CLOCKD_SERVICE,
							   CLOCKD_PATH,
							   CLOCKD_INTERFACE);
    dbus_g_proxy_add_signal(desktop_plugin->dbus_proxy,
			    CLOCKD_TIME_CHANGED,
			    G_TYPE_INT, G_TYPE_INVALID );
    dbus_g_proxy_connect_signal(desktop_plugin->dbus_proxy,
				CLOCKD_TIME_CHANGED,
				G_CALLBACK(handle_time_change), desktop_plugin, NULL);
  }

}

static void
date_today_plugin_init(DateTodayPlugin *desktop_plugin)
{

  desktop_plugin->today = date_today_get_current_day();
  read_settings(desktop_plugin);
  GtkWidget *contents = build_ui(desktop_plugin->size);
  desktop_plugin->content = contents;
  gtk_container_add(GTK_CONTAINER(desktop_plugin), contents);
  hd_home_plugin_item_set_settings (HD_HOME_PLUGIN_ITEM (desktop_plugin), TRUE);

  g_signal_connect(desktop_plugin, "notify::is-on-current-desktop", G_CALLBACK(date_today_on_current_desktop), NULL);
  g_signal_connect(desktop_plugin, "show-settings", G_CALLBACK(date_today_show_settings_dialog), NULL);

  register_dbus_signal_on_time_change(desktop_plugin);
}

static void
datetoday_realize(GtkWidget* widget)
{
  GdkScreen *screen = gtk_widget_get_screen(widget);
  gtk_widget_set_colormap(widget, gdk_screen_get_rgba_colormap(screen));
  gtk_widget_set_app_paintable(widget, TRUE);
  GTK_WIDGET_CLASS(date_today_plugin_parent_class)->realize(widget);
}

static gboolean
datetoday_expose(GtkWidget* widget, GdkEventExpose *event)
{
  PangoLayout *layout;
  PangoFontDescription *desc;

  cairo_t *cr;
  cr = gdk_cairo_create(GDK_DRAWABLE(widget->window));
  gdk_cairo_region(cr, event->region);
  cairo_clip(cr);

  int width = DATE_TODAY_PLUGIN(widget)->size;
  int height = 0;
  cairo_scale(cr, width/100.0, width/100.0);

  layout = pango_cairo_create_layout(cr);

  char buf[3];
  sprintf(buf, "%d", DATE_TODAY_PLUGIN(widget)->today);
  buf[2] = '\0';
  pango_layout_set_text(layout, buf, -1);

  const char* font = "Sans Bold 30";
  desc = pango_font_description_from_string(font);
  pango_layout_set_font_description(layout, desc);
  pango_font_description_free(desc);

  cairo_set_operator(cr, CAIRO_OPERATOR_SOURCE);
  cairo_set_source_rgba(cr, 0.0, 0.0, 0.0, 0.0);
  cairo_paint(cr);

  cairo_set_source_rgba(cr, 0.8, 0.8, 0.8, 0.7);
  cairo_rectangle(cr, 13, 13, 64, 64);
  cairo_fill(cr);
  cairo_set_source_rgba(cr, 1.0, 1.0, 1.0, 1.0);
  cairo_rectangle(cr, 15, 15, 60, 60);
  cairo_fill(cr);


  pango_layout_get_pixel_size(layout, &width, &height);

  cairo_set_source_rgba(cr, 1.0, 1.0, 1.0, 1.0);
  cairo_set_line_width(cr, 5);

  cairo_set_source_rgba(cr, 1.0, 0.0, 0.0, 1.0);
  cairo_rectangle(cr, 15, 15, 60, 10); 
  cairo_fill(cr); 
  cairo_set_source_rgba(cr, 0.0, 0.0, 0.0, 1.0); 
  cairo_arc(cr, 30, 20, 3, 0, 2*M_PI); 
  cairo_arc(cr, 60, 20, 3, 0, 2*M_PI); 
  cairo_fill(cr); 



  cairo_save(cr); 
  cairo_set_line_width(cr, 3); 
  cairo_set_source_rgba(cr, 0.0, 0.0, 0.0, 1.0); 

  cairo_set_source_rgba(cr, 0.85, 0.85, 0.85, 1.0); 
  cairo_set_line_cap(cr, CAIRO_LINE_CAP_ROUND); 
  cairo_move_to(cr, 30, 13); 
  cairo_line_to(cr, 30, 20); 
  
  cairo_stroke(cr); 
  cairo_move_to(cr, 60, 13); 
  cairo_line_to(cr, 60, 20); 

  cairo_stroke(cr); 
  cairo_restore(cr); 
  cairo_move_to(cr, 45-(width/2.0), 50-(height/2.0)); 
  cairo_set_source_rgba(cr, 0.85, 0.0, 0.0, 1.0); 
  pango_cairo_show_layout(cr, layout);
  
  cairo_set_operator(cr, CAIRO_OPERATOR_OVER);
  cairo_pattern_t* linepattern = cairo_pattern_create_linear(45, 0, 45, 90.0);
  cairo_pattern_add_color_stop_rgba(linepattern, 0.0, 0.8, 0.8, 0.8, 0.0);
  cairo_pattern_add_color_stop_rgba(linepattern, 90.0, 0.0, 0.0, 0.0, 0.42);
  cairo_rectangle(cr, 15, 15, 60, 60);
  cairo_set_source(cr, linepattern);
  cairo_fill(cr);
  cairo_pattern_destroy(linepattern);
  
  cairo_destroy(cr);
  
  return GTK_WIDGET_CLASS(date_today_plugin_parent_class)->expose_event(widget, event);
}

static gint
button_press(GtkWidget* widget, GdkEventButton *event)
{
  DateTodayPlugin *desktop_plugin = DATE_TODAY_PLUGIN(widget);
  DBusGConnection *dbus_conn = hd_home_plugin_item_get_dbus_g_connection ( &desktop_plugin->hitem, 
									   DBUS_BUS_SESSION, 
									   NULL );


  DBusGProxy *cal_proxy = dbus_g_proxy_new_for_name(dbus_conn,
						    CALENDAR_SERVICE,
						    CALENDAR_PATH,
						    CALENDAR_INTERFACE);
  int type = desktop_plugin->launch_view; 
  time_t tt = time(NULL); 
  int msgID = tt; 
  const char* text = "DUMMY";  
  if(cal_proxy)
  {
    dbus_g_proxy_call_no_reply(cal_proxy, 
			       CALENDAR_LAUNCH_METHOD, 
			       G_TYPE_UINT, type, 
			       G_TYPE_INT, msgID, 
			       G_TYPE_STRING, text,
			       DBUS_TYPE_INVALID); 
  }
  g_object_unref(cal_proxy);
  dbus_g_connection_unref(dbus_conn);  
  return FALSE;
}

static void
date_today_plugin_finalize(GObject *object)
{
  DateTodayPlugin *desktop_plugin = DATE_TODAY_PLUGIN(object);
  if(desktop_plugin->time_out_handler)
    {
      remove_time_out_handler(desktop_plugin);
    }

  if (desktop_plugin->dbus_proxy)
    dbus_g_proxy_disconnect_signal (desktop_plugin->dbus_proxy, 
				    CLOCKD_TIME_CHANGED, G_CALLBACK(handle_time_change), desktop_plugin);

  G_OBJECT_CLASS (date_today_plugin_parent_class)->finalize (object);

}

static void
date_today_plugin_class_init(DateTodayPluginClass *klass) 
{
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS(klass);
  widget_class->realize = datetoday_realize;
  widget_class->expose_event = datetoday_expose;
//Define button_release_event instead of button_press_event to avoid calendar application execution during scrolling between desktops
  //widget_class->button_press_event = button_press;
  widget_class->button_release_event = button_press;
//------------------------------------------------------------------------------------------------László Tari <n900nseries@gmail.com>
  G_OBJECT_CLASS(klass)->finalize = date_today_plugin_finalize;
}

static void
date_today_plugin_class_finalize(DateTodayPluginClass *class) 
{
}

