/*
  DateJinni - A simple date conversion tool
  Copyright (C) 2007  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <iomanip>
#include <limits>
#include <sstream>

#include <Lum/Base/DateTime.h>
#include <Lum/Base/L10N.h>
#include <Lum/Base/Size.h>
#include <Lum/Base/String.h>

#include <Lum/Def/MultiView.h>

#include <Lum/Dlg/About.h>
#include <Lum/Dlg/Msg.h>

#include <Lum/Model/Action.h>
#include <Lum/Model/Calendar.h>
#include <Lum/Model/Number.h>
#include <Lum/Model/String.h>
#include <Lum/Model/Time.h>

#include <Lum/Manager/Behaviour.h>
#include <Lum/Manager/FileSystem.h>

#include <Lum/Application.h>
#include <Lum/Button.h>
#include <Lum/ButtonRow.h>
#include <Lum/DateSelect.h>
#include <Lum/Dialog.h>
#include <Lum/Label.h>
#include <Lum/Panel.h>
#include <Lum/String.h>
#include <Lum/Text.h>
#include <Lum/TextValue.h>
#include <Lum/TimeSelect.h>

#include "config.h"

static Lum::Def::AppInfo info;

static double SystemTimeToJulianDate(const Lum::Base::SystemTime& time)
{
  Lum::Base::DateTime dt;

  time.GetUTCDateTime(dt);

  int a=(14-dt.month)/12;
  int y=dt.year+4800-a;
  int m=dt.month+12*a-3;
  int jdn=dt.dayOfMonth+(153*m+2)/5+365*y+y/4-y/100+y/400-32045;
  double jd=jdn+(dt.hour-12)/24.0+dt.minute/1440.0+dt.second/86400.0;

  return jd;
  /*
  std::stringstream buffer;

  buffer.imbue(std::locale(""));

  buffer << std::fixed << std::setprecision(10) << jd;

  return Lum::Base::StringToWString(buffer.str());*/
}

class MainWindow : public Lum::Dialog
{
private:
  Lum::Model::ActionRef    about;

  Lum::Model::ULongRef     unixTime;

  Lum::Model::CalendarRef  utcDate;
  Lum::Model::TimeRef      utcTime;

  Lum::Model::CalendarRef  localDate;
  Lum::Model::TimeRef      localTime;

  Lum::Model::DoubleRef    julianDate;

public:
  MainWindow()
  : about(new Lum::Model::Action()),
    unixTime(new Lum::Model::ULong(Lum::Base::SystemTime().GetTime())),
    utcDate(new Lum::Model::Calendar()),
    utcTime(new Lum::Model::Time()),
    localDate(new Lum::Model::Calendar()),
    localTime(new Lum::Model::Time()),
    julianDate(new Lum::Model::Double())
  {
    GetWindow()->SetScreenOrientationHint(Lum::OS::Window::screenOrientationBothSupported);

    julianDate->Disable();

    ConvertFromUnix();

    Observe(unixTime);

    Observe(utcDate);
    Observe(utcTime);

    Observe(localDate);
    Observe(localTime);

    Observe(julianDate);

    Observe(about);
  }

  void PreInit()
  {
    Lum::Def::MultiView   multiView(Lum::Def::Desc(L"DateJinni"));
    Lum::DateSelect       *dateSel;
    Lum::Label            *label;
    Lum::Panel            *panel;
    Lum::Panel            *panel2;
    Lum::String           *string;
    Lum::Text             *text;
    Lum::TimeSelect       *timeSel;

    Lum::Base::SystemTime now;
    std::wstring          tmp;

    panel=Lum::VPanel::Create(true,true);

    label=Lum::Label::Create(true,false);

    string=new Lum::String();
    string->SetMinWidth(Lum::Base::Size::stdCharWidth,
                        Lum::Base::NumberDigits(std::numeric_limits<time_t>::max()));
    string->SetModel(unixTime);
    string->SetAlignment(Lum::String::right);
    string->SetInputTypeHint(Lum::Object::inputTypeUnsignedDecimalNumber);
    label->AddLabel(_(L"UNIX_TIME",L"Unix:"),string);

    dateSel=new Lum::DateSelect();
    dateSel->SetModel(utcDate);

    timeSel=new Lum::TimeSelect();
    timeSel->SetFormat(Lum::Base::timeFormatHMS);
    timeSel->SetModel(utcTime);

    panel2=Lum::HPanel::Create();
    panel2->Add(dateSel);
    panel2->AddSpace();
    panel2->Add(timeSel);
    label->AddLabel(_(L"UTC_TIME",L"UTC:"),panel2);

    dateSel=new Lum::DateSelect();
    dateSel->SetModel(localDate);

    timeSel=new Lum::TimeSelect();
    timeSel->SetFormat(Lum::Base::timeFormatHMS);
    timeSel->SetModel(localTime);

    panel2=Lum::HPanel::Create();
    panel2->Add(dateSel);
    panel2->AddSpace();
    panel2->Add(timeSel);
    label->AddLabel(_(L"LOCAL_TIME",L"Local:"),panel2);

    string=new Lum::String();
    string->SetMinWidth(Lum::Base::Size::stdCharWidth,20);
    string->SetModel(julianDate);
    string->SetAlignment(Lum::String::right);
    string->SetConverter(new Lum::Model::NumberStringConverter(Lum::Model::NumberStringConverter::floatFixed,10));
    string->SetInputTypeHint(Lum::Object::inputTypeUnsignedFloatNumber);
    label->AddLabel(_(L"JULIAN_DATE",L"Julian date:"),string);

    panel->Add(label);

    panel->AddSpace();

    tmp=_(L"DIFFERENCE",L"Difference to UTC: ");
    tmp.append(Lum::Base::NumberToWString(now.GetDifferenceToUTC()));
    tmp.append(L" ");
    tmp.append(_(L"SECONDS",L" seconds"));
    tmp.append(L"\n\n");
    if (now.IsDSTActive()) {
      tmp.append(_(L"DST_ACTIVE",L"DST is active"));
      tmp.append(L" (");
      tmp.append(Lum::Base::NumberToWString(now.GetDST()));
      tmp.append(L" ");
      tmp.append(_(L"SECONDS",L" seconds"));
      tmp.append(L")");
    }
    else {
      tmp.append(_(L"DST_INACTIVE",L"DST is not active"));
    }

    text=new Lum::Text(tmp);
    text->SetFlex(true,true);

    panel->Add(text);

    Lum::Def::Menu *menu=Lum::Def::Menu::Create()
      ->GroupProject()
        ->ActionQuit(GetClosedAction())
      ->End()
      ->GroupHelp()
        //->Action(Lum::Def::Action(Lum::Def::Desc(_ld(menuHelpHelp)),NULL))
        ->ActionAbout(about)
      ->End();

    SetMenu(menu);
    SetMain(panel);

    Dialog::PreInit();
  }

  void ConvertFromUnix()
  {
    if (!unixTime.Valid() || unixTime->IsNull()) {
      utcDate->SetNull();
      utcTime->SetNull();

      localDate->SetNull();
      localTime->SetNull();

      julianDate->SetNull();

      return;
    }

    Lum::Base::SystemTime sysTime(unixTime->Get());
    Lum::Base::DateTime   dt;

    sysTime.GetUTCDateTime(dt);

    utcDate->Off();
    utcTime->Off();

    bool utcChanged=utcDate->SetDate(dt.dayOfMonth,dt.month,dt.year) ||
                    utcTime->SetTime(dt.hour,dt.minute,dt.second);

    utcDate->On(utcChanged);
    utcTime->On(utcChanged);

    sysTime.GetLocalDateTime(dt);

    localDate->Off();
    localTime->Off();

    bool localChanged=localDate->SetDate(dt.dayOfMonth,dt.month,dt.year) ||
                      localTime->SetTime(dt.hour,dt.minute,dt.second);

    localDate->On(localChanged);
    localTime->On(localChanged);

    julianDate->Set(SystemTimeToJulianDate(sysTime));
  }

  void ConvertFromUtc()
  {
    if (!utcDate.Valid() ||
        !utcTime.Valid() ||
        utcDate->IsNull() ||
        utcTime->IsNull()) {
      unixTime->SetNull();

      localDate->SetNull();
      localTime->SetNull();

      julianDate->SetNull();
    }

    Lum::Base::SystemTime time;

    try {
      time.SetUTCDateTime(utcTime->Get().GetHour(),
                          utcTime->Get().GetMinute(),
                          utcTime->Get().GetSecond(),
                          0,
                          utcDate->Get().GetDayOfMonth(),
                          utcDate->Get().GetMonth(),
                          utcDate->Get().GetYear());
    }
    catch (...) {
      unixTime->SetNull();

      localDate->SetNull();
      localTime->SetNull();

      return;
    }

    Lum::Base::DateTime dt;

    unixTime->Set(time.GetTime());

    time.GetLocalDateTime(dt);

    localDate->Off();
    localTime->Off();

    bool localChanged=localDate->SetDate(dt.dayOfMonth,dt.month,dt.year) ||
                      localTime->SetTime(dt.hour,dt.minute,dt.second);

    localDate->On(localChanged);
    localTime->On(localChanged);

    SystemTimeToJulianDate(time);

    julianDate->Set(SystemTimeToJulianDate(time));
  }

  void ConvertFromLocal()
  {
    if (!localDate.Valid() ||
        !localTime.Valid() ||
        localDate->IsNull() ||
        localTime->IsNull()) {
      unixTime->SetNull();

      utcDate->SetNull();
      utcTime->SetNull();

      julianDate->SetNull();
    }

    Lum::Base::SystemTime time;

    try {
      time.SetLocalDateTime(localTime->Get().GetHour(),
                            localTime->Get().GetMinute(),
                            localTime->Get().GetSecond(),
                            0,
                            localDate->Get().GetDayOfMonth(),
                            localDate->Get().GetMonth(),
                            localDate->Get().GetYear());
    }
    catch (...) {
      unixTime->SetNull();

      utcDate->SetNull();
      utcTime->SetNull();

      return;
    }

    Lum::Base::DateTime dt;

    unixTime->Set(time.GetTime());

    time.GetUTCDateTime(dt);

    utcDate->Off();
    utcTime->Off();

    bool utcChanged=utcDate->SetDate(dt.dayOfMonth,dt.month,dt.year) ||
                    utcTime->SetTime(dt.hour,dt.minute,dt.second);

    utcDate->On(utcChanged);
    utcTime->On(utcChanged);

    julianDate->Set(SystemTimeToJulianDate(time));
  }

  void Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
  {
    if (model==unixTime) {
      ConvertFromUnix();
    }
    else if (model==utcDate || model==utcTime) {
      ConvertFromUtc();
    }
    else if (model==localDate || model==localTime) {
      ConvertFromLocal();
    }
    else if (model==julianDate) {
      // TODO
    }
    else if (model==about && about->IsFinished()) {
      Lum::Dlg::About::Show(this,info);
    }

    Dialog::Resync(model,msg);
  }
};

class Main : public Lum::GUIApplication<MainWindow>
{
protected:
  bool Initialize()
  {
#if defined(APP_DATADIR)
    Lum::Manager::FileSystem::Instance()->SetApplicationDataDir(Lum::Base::StringToWString(APP_DATADIR));
#endif

    info.SetProgram(Lum::Base::StringToWString(PACKAGE_NAME));
    info.SetVersion(Lum::Base::StringToWString(PACKAGE_VERSION));
    info.SetDescription(L"Some date conversion routines...");
    info.SetAuthor(L"Tim Teulings");
    info.SetContact(L"<tim@teulings.org>");
    info.SetCopyright(L"(c) 2007, Tim Teulings");
    info.SetLicense(L"GNU Public License");

    return Lum::GUIApplication<MainWindow>::Initialize();
  }
};

LUM_MAIN(Main,L"DateJinni")
