#include "vimeovideolistmodel.h"
#include "vimeo.h"
#include "utils.h"
#include "playbacklistmodel.h"
#include "transfermanager.h"
#include <QNetworkReply>
#include <QNetworkRequest>

VimeoVideoListModel::VimeoVideoListModel(QObject *parent) :
    AbstractVideoListModel(parent),
    m_loading(true),
    m_moreResults(true),
    m_error(false),
    m_offset(1),
    m_refresh(false)
{
}

void VimeoVideoListModel::clear() {
    AbstractVideoListModel::clear();
    this->setLoading(false);
    this->setOffset(1);
    this->setMoreResults(true);
    this->setError(false);
    this->setRefreshRequired(false);
}

void VimeoVideoListModel::reset() {
    if (!this->loading()) {
        this->clear();
        this->getMoreVideos();
    }
}

void VimeoVideoListModel::setFeed(const QUrl &feed) {
    if (feed != this->feed()) {
        m_feed = feed;

        this->disconnect(Vimeo::instance());

        if (feed == VIMEO_UPLOADS_FEED) {
            this->connect(Vimeo::instance(), SIGNAL(deletedFromUploads(QString)), this, SLOT(onDeletedFromUploads(QString)));
        }
        else if (feed == VIMEO_LIKES_FEED) {
            this->connect(Vimeo::instance(), SIGNAL(likeChanged(QString,bool)), this, SLOT(onLikeChanged(QString,bool)));
        }
        else if (feed == VIMEO_WATCH_LATER_FEED) {
            this->connect(Vimeo::instance(), SIGNAL(addedToWatchLaterAlbum(QString)), this, SLOT(onAddedToWatchLaterAlbum(QString)));
            this->connect(Vimeo::instance(), SIGNAL(deletedFromWatchLaterAlbum(QString)), this, SLOT(onDeletedFromWatchLaterAlbum(QString)));
        }
        else if (feed.queryItemValue("method") == "vimeo.albums.getVideos") {
            this->setAlbumId(feed.queryItemValue("album_id"));
            this->connect(Vimeo::instance(), SIGNAL(addedToAlbum(QString,QString)), this, SLOT(onAddedToAlbum(QString,QString)));
            this->connect(Vimeo::instance(), SIGNAL(deletedFromAlbum(QString,QString)), this, SLOT(onDeletedFromAlbum(QString,QString)));
        }

        emit feedChanged(feed);
    }
}

void VimeoVideoListModel::getVideos(QUrl feed) {
    if (!feed.isEmpty()) {
        this->setFeed(feed);
    }

    this->setLoading(true);
    QNetworkReply *reply = Vimeo::instance()->createReply(this->feed(), this->offset());
    reply->setParent(this);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addVideos()));
}

void VimeoVideoListModel::getMoreVideos() {
    if ((this->moreResults()) && (!this->loading())) {
        if (!this->query().isEmpty()) {
            this->search();
        }
        else {
            this->getVideos();
        }
    }
}

void VimeoVideoListModel::getGroupVideos(const QString &id) {
    QUrl feed(VIMEO_BASE_URL);
    feed.addQueryItem("method", "vimeo.groups.getVideos");
    feed.addQueryItem("group_id", id);
    feed.addQueryItem("per_page", "30");
    feed.addQueryItem("full_response", "true");
    feed.addQueryItem("sort", "newest");
    this->getVideos(feed);
}

void VimeoVideoListModel::getAlbumVideos(const QString &id) {
    QUrl feed(VIMEO_BASE_URL);
    feed.addQueryItem("method", "vimeo.albums.getVideos");
    feed.addQueryItem("album_id", id);
    feed.addQueryItem("per_page", "30");
    feed.addQueryItem("full_response", "true");
    this->getVideos(feed);
}

void VimeoVideoListModel::search(const QString &query, int order) {
    if (!query.isEmpty()) {
        this->setQuery(query);
        this->setOffset(1);
        this->setOrder(order);
    }

    this->setLoading(true);
    QNetworkReply *reply = Vimeo::instance()->createSearchReply(this->query(), this->offset(), this->order());
    reply->setParent(this);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addVideos()));
}

void VimeoVideoListModel::addVideos() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        this->setError(true);
        return;
    }

    QDomDocument doc;
    doc.setContent(reply->readAll());
    QDomElement responseNode = doc.firstChildElement("rsp");

    if (responseNode.attribute("stat") == "ok") {
        QDomNodeList entries = doc.elementsByTagName("video");

        for (int i = 0; i < entries.size(); i++) {
            VideoItem *video = new VideoItem;
            video->loadVimeoVideo(entries.at(i).toElement());
            QMetaObject::invokeMethod(this, "appendVideo", Qt::QueuedConnection, Q_ARG(QSharedPointer<VideoItem>, QSharedPointer<VideoItem>(video)));
        }

        QDomElement statNode = responseNode.firstChildElement("videos");

        this->setMoreResults(statNode.attribute("total").toInt() > this->rowCount());
        this->setOffset(statNode.attribute("page").toInt() + 1);

        if ((!this->albumId().isEmpty()) && (this->moreResults())) {
            this->getVideos();
        }
        else {
            QTimer::singleShot(1000, this, SLOT(stopLoading()));
        }

        emit countChanged(this->rowCount());
    }
    else {
        this->setLoading(false);
        this->setError(true);
    }

    reply->deleteLater();
}

void VimeoVideoListModel::onDeletedFromUploads(const QString &videoId) {
    if (this->removeVideo(videoId)) {
        emit countChanged(this->rowCount());
    }
}

void VimeoVideoListModel::onLikeChanged(const QString &videoId, bool like) {
    if (like) {
        this->setRefreshRequired(true);
    }
    else if (this->removeVideo(videoId)) {
        emit countChanged(this->rowCount());
    }
}

void VimeoVideoListModel::onAddedToWatchLaterAlbum(const QString &videoId) {
    Q_UNUSED(videoId)

    this->setRefreshRequired(true);
}

void VimeoVideoListModel::onDeletedFromWatchLaterAlbum(const QString &videoId) {
    if (this->removeVideo(videoId)) {
        emit countChanged(this->rowCount());
    }
}

void VimeoVideoListModel::onAddedToAlbum(const QString &videoId, const QString &albumId) {
    Q_UNUSED(videoId)

    if (albumId == this->albumId()) {
        this->setRefreshRequired(true);
    }
}

void VimeoVideoListModel::onDeletedFromAlbum(const QString &videoId, const QString &albumId) {
    if ((albumId == this->albumId()) && (this->removeVideo(videoId))) {
        emit countChanged(this->rowCount());
    }
}

void VimeoVideoListModel::deleteSelectedVideosFromUploads() {
    if (this->feed() == VIMEO_UPLOADS_FEED) {
        QStringList videoIds;

        foreach (int row, this->selectedRows()) {
            videoIds << this->data(this->index(row), VideoIdRole).toString();
        }

        Vimeo::instance()->deleteFromUploads(videoIds);
    }

    this->selectNone();
}

void VimeoVideoListModel::addSelectedVideosToLikes() {
    QStringList videoIds;

    foreach (int row, this->selectedRows()) {
        videoIds << this->data(this->index(row), VideoIdRole).toString();
    }

    Vimeo::instance()->addToLikes(videoIds);
    this->selectNone();
}

void VimeoVideoListModel::deleteSelectedVideosFromLikes() {
    if (this->feed() == VIMEO_LIKES_FEED) {
        QStringList videoIds;

        foreach (int row, this->selectedRows()) {
            videoIds << this->data(this->index(row), VideoIdRole).toString();
        }

        Vimeo::instance()->deleteFromLikes(videoIds);
    }

    this->selectNone();
}

void VimeoVideoListModel::addSelectedVideosToAlbum(const QString &albumId) {
    QStringList videoIds;

    foreach (int row, this->selectedRows()) {
        videoIds << this->data(this->index(row), VideoIdRole).toString();
    }

    Vimeo::instance()->addToAlbum(videoIds, albumId);
    this->selectNone();
}

void VimeoVideoListModel::addSelectedVideosToNewAlbum(const QVariantMap &album) {
    QStringList videoIds;

    foreach (int row, this->selectedRows()) {
        videoIds << this->data(this->index(row), VideoIdRole).toString();
    }

    Vimeo::instance()->createAlbum(album, videoIds);
    this->selectNone();
}

void VimeoVideoListModel::deleteSelectedVideosFromAlbum(const QString &albumId) {
    QStringList videoIds;

    foreach (int row, this->selectedRows()) {
        videoIds << this->data(this->index(row), VideoIdRole).toString();
    }

    Vimeo::instance()->deleteFromAlbum(videoIds, albumId);
    this->selectNone();
}

void VimeoVideoListModel::addSelectedVideosToWatchLaterAlbum() {
    QStringList videoIds;

    foreach (int row, this->selectedRows()) {
        videoIds << this->data(this->index(row), VideoIdRole).toString();
    }

    Vimeo::instance()->addToWatchLaterAlbum(videoIds);
    this->selectNone();
}

void VimeoVideoListModel::deleteSelectedVideosFromWatchLaterAlbum() {
    if (this->feed() == VIMEO_WATCH_LATER_FEED) {
        QStringList videoIds;

        foreach (int row, this->selectedRows()) {
            videoIds << this->data(this->index(row), VideoIdRole).toString();
        }

        Vimeo::instance()->deleteFromWatchLaterAlbum(videoIds);
    }

    this->selectNone();
}

void VimeoVideoListModel::downloadSelectedVideos(bool saveAsAudio) {
    TransferManager::instance()->addDownloadTransfers(this->selectedItems(), saveAsAudio);
    this->selectNone();
}

void VimeoVideoListModel::queueSelectedVideos() {
    PlaybackListModel::playbackQueue()->addVideos(this->selectedItems());
    this->selectNone();
}
