#ifndef URLGRABBER_H
#define URLGRABBER_H

#include "enums.h"
#include <QObject>
#include <QUrl>
#include <QList>
#include <QVariant>
#include <QScriptValue>

class QNetworkAccessManager;
class QNetworkReply;

typedef struct {
    Videos::VideoFormat format;
    QVariant value;
    QString displayName;
} VideoFormat;

class UrlGrabber : public QObject
{
    Q_OBJECT

    Q_PROPERTY(bool busy
               READ busy
               NOTIFY busyChanged)

public:
    explicit UrlGrabber(QObject *parent = 0);
    inline QNetworkAccessManager *networkAccessManager() const { return m_nam; }
    inline void setNetworkAccessManager(QNetworkAccessManager *manager) { m_nam = manager; }
    inline QList<VideoFormat> youtubeFormats() const { return m_youtubeFormats; }
    inline QList<VideoFormat> dailymotionFormats() const { return m_dailymotionFormats; }
    inline QList<VideoFormat> vimeoFormats() const { return m_vimeoFormats; }
    inline bool busy() const { return m_busy; }
    inline bool cancelled() const { return m_cancelled; }

    static UrlGrabber* instance();

public slots:
    void getVideoUrl(int service, const QString &id);
    void getYouTubeVideoUrl(const QString &id);
    void getDailymotionVideoUrl(const QString &id);
    void getVimeoVideoUrl(const QString &id);
    void getAvailableVideoFormats(int service, const QString &id);
    void getAvailableYouTubeVideoFormats(const QString &id);
    void getAvailableDailymotionVideoFormats(const QString &id);
    void getAvailableVimeoVideoFormats(const QString &id);
    void getVimeoVideoRedirect(const QUrl &url);
    void setYouTubeFormats(QSet<int> formats);
    void setDailymotionFormats(QSet<int> formats);
    void setVimeoFormats(QSet<int> formats);
    void cancelCurrentOperation();    

private:
    void setBusy(bool isBusy, const QString &message = QString(), int numberOfOperations = 1);
    void setCancelled(bool cancelled) { m_cancelled = cancelled; }
    QScriptValue getYouTubeDecryptionFunction(const QUrl &playerUrl);
    QMap<int, QUrl> getYouTubeVideoUrlMap(QString page, QScriptValue decryptionFunction = QScriptValue());

private slots:
    void checkYouTubePageForEncryption();
    void parseYouTubeVideoPage(QScriptValue decryptionFunction = QScriptValue(), QString page = QString());
    void parseDailymotionVideoPage();
    void parseVimeoVideoPage();
    void checkVimeoVideoRedirect();
    void checkYouTubeVideoFormatsForEncryption();
    void parseYouTubeVideoFormats(QScriptValue decryptionFunction = QScriptValue(), QString page = QString());
    void checkDailymotionVideoFormats();
    void checkVimeoVideoFormats();
    void addYouTubeDecryptionFunctionToCache();

signals:
    void gotVideoUrl(const QUrl &url, Videos::VideoFormat videoFormat);
    void gotVideoFormats(const QVariantList &formats);
    void busyChanged(bool isBusy);
    void busy(const QString &message, int numberOfOperations = 1);
    void busyProgressChanged(int progress);
    void error(const QString &errorString);
    void currentOperationCancelled();
    void youtubeDecryptionFunctionReady(QScriptValue decryptionFunction);

private:
    QNetworkAccessManager *m_nam;
    QList<VideoFormat> m_youtubeFormats;
    QList<VideoFormat> m_dailymotionFormats;
    QList<VideoFormat> m_vimeoFormats;
    Videos::VideoFormat m_format;
    bool m_busy;
    bool m_cancelled;
    QString m_youtubePage;
};

#endif // URLGRABBER_H
