#include "videolistdelegate.h"
#include "../base/utils.h"
#include "../base/abstractvideolistmodel.h"
#include <QApplication>
#include <QPainter>
#include <QMouseEvent>
#include <QMaemo5Style>

VideoListDelegate::VideoListDelegate(QObject *parent) :
    QStyledItemDelegate(parent)
{
}

void VideoListDelegate::paint(QPainter *painter, const QStyleOptionViewItem &option, const QModelIndex &index) const {
    painter->save();

    if ((option.state) & (QStyle::State_Selected) && (!index.data(AbstractVideoListModel::ThumbnailPressedRole).toBool())) {
        painter->drawImage(option.rect, QImage("/etc/hildon/theme/images/TouchListBackgroundPressed.png"));
    }
    else {
        painter->drawImage(option.rect, QImage("/etc/hildon/theme/images/TouchListBackgroundNormal.png"));
    }

    QRect titleRect = option.rect;
    titleRect.setLeft(titleRect.left() + 170);
    titleRect.setRight(titleRect.right() - 5);
    titleRect.setTop(titleRect.top() + 5);
    titleRect.setHeight(60);

    painter->drawText(titleRect, Qt::TextWordWrap, index.data(AbstractVideoListModel::TitleRole).toString());

    QFont smallFont;
    smallFont.setPixelSize(18);
    painter->setFont(smallFont);
    painter->setPen(QMaemo5Style::standardColor("SecondaryTextColor"));

    QRect infoRect = option.rect;
    infoRect.setLeft(titleRect.left());
    infoRect.setRight(titleRect.right());
    infoRect.setBottom(infoRect.bottom() - 5);

    Services::VideoService service = static_cast<Services::VideoService>(index.data(AbstractVideoListModel::ServiceRole).toInt());
    QString ratingText;

    if (service == Services::YouTube) {
        ratingText = QString("%1%  like  %2  views").arg(index.data(AbstractVideoListModel::RatingRole).toInt()).arg(index.data(AbstractVideoListModel::ViewCountRole).toInt());
        painter->drawText(infoRect, Qt::AlignBottom, ratingText);
    }
    else if (service == Services::Vimeo) {
        ratingText = QString("%1  likes  %2  plays").arg(index.data(AbstractVideoListModel::LikesRole).toInt()).arg(index.data(AbstractVideoListModel::ViewCountRole).toInt());
        painter->drawText(infoRect, Qt::AlignBottom, ratingText);
    }
    else if (service == Services::Dailymotion) {
        QRect ratingRect = infoRect;
        ratingRect.setTop(ratingRect.bottom() - 20);
        ratingRect.setWidth(20);

        int rating = index.data(AbstractVideoListModel::RatingRole).toInt();

        for (int i = 1; i <= 5; i++) {
            if (i > rating) {
                painter->setOpacity(0.3);
            }
            else {
                painter->setOpacity(1);
            }

            painter->drawImage(ratingRect, QImage("/usr/share/icons/hicolor/48x48/hildon/imageviewer_favourite.png"));
            ratingRect.moveLeft(ratingRect.right());
        }

        ratingRect.setWidth(infoRect.width() - ratingRect.left());
        ratingText = QString(" (%1)  %2  views").arg(index.data(AbstractVideoListModel::RatingCountRole).toInt()).arg(index.data(AbstractVideoListModel::ViewCountRole).toInt());

        painter->setOpacity(1);
        painter->drawText(ratingRect, Qt::AlignVCenter, ratingText);
    }

    infoRect.setBottom(infoRect.bottom() - 25);

    QString artistText = QString("%1 - %2").arg(index.data(AbstractVideoListModel::ArtistRole).toString(), index.data(AbstractVideoListModel::DateRole).toString());

    painter->drawText(infoRect, Qt::TextWordWrap | Qt::AlignBottom, artistText);

    QRect imageRect = option.rect;
    imageRect.setLeft(imageRect.left() + 5);
    imageRect.setTop(imageRect.top() + 5);
    imageRect.setWidth(160);
    imageRect.setHeight(120);

    if ((option.state) & (QStyle::State_Selected) && (index.data(AbstractVideoListModel::ThumbnailPressedRole).toBool())) {
        painter->setOpacity(0.5);
    }

    QImage image = index.data(AbstractVideoListModel::ThumbnailRole).value<QImage>();

    if (!image.isNull()) {
        painter->drawImage(imageRect, image);
    }
    else {
        painter->drawImage(imageRect.left() + 56, imageRect.top() + 36, QImage("/usr/share/icons/hicolor/48x48/hildon/general_video_file.png"));
    }

    painter->drawRect(imageRect);

    int duration = index.data(AbstractVideoListModel::DurationRole).toInt();

    QRect durationRect = imageRect;
    durationRect.setLeft(duration < 3600 ? durationRect.right() - 50 : durationRect.right() - 70);
    durationRect.setTop(durationRect.bottom() - 20);

    painter->setOpacity(0.5);
    painter->fillRect(durationRect, Qt::black);
    painter->setOpacity(1);
    painter->setPen(Qt::white);
    painter->drawText(durationRect, Qt::AlignCenter, Utils::durationFromSecs(duration));
    painter->restore();
}

bool VideoListDelegate::editorEvent(QEvent *event, QAbstractItemModel *model, const QStyleOptionViewItem &option, const QModelIndex &index) {
    if (event->type() == QEvent::MouseButtonPress) {
        QMouseEvent *mouse = static_cast<QMouseEvent*>(event);

        QRect imageRect = option.rect;
        imageRect.setLeft(imageRect.left() + 5);
        imageRect.setTop(imageRect.top() + 5);
        imageRect.setWidth(160);
        imageRect.setHeight(120);

        if (imageRect.contains(mouse->pos())) {
            return model->setData(index, index.row(), AbstractVideoListModel::ThumbnailPressedRole);
        }

        return true;
    }

    if (event->type() == QEvent::MouseButtonRelease) {
        QMouseEvent *mouse = static_cast<QMouseEvent*>(event);

        if (index.data(AbstractVideoListModel::ThumbnailPressedRole).toBool()) {
            QRect imageRect = option.rect;
            imageRect.setLeft(imageRect.left() + 5);
            imageRect.setTop(imageRect.top() + 5);
            imageRect.setWidth(160);
            imageRect.setHeight(120);

            if (imageRect.contains(mouse->pos())) {
                emit play(index);
            }
        }
        else if (option.rect.contains(mouse->pos())) {
            emit clicked(index);
        }

        return model->setData(index, -1, AbstractVideoListModel::ThumbnailPressedRole);
    }

    return false;
}

QSize VideoListDelegate::sizeHint(const QStyleOptionViewItem &option, const QModelIndex &index) const {
    Q_UNUSED(index)

    return QSize(option.rect.width(), 150);
}
