#include "urlgrabber.h"
#include "extractorplugin.h"
#include "cookiejar.h"
#include "utils.h"
#ifdef GRABBER_PLUGIN
#include "../plugins/pluginmanager.h"
#endif
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QStringList>
#include <QRegExp>
#include <QDebug>

UrlGrabber* grabberInstance = 0;

UrlGrabber::UrlGrabber(QObject *parent) :
    QObject(parent),
    m_nam(0),
    m_extractor(0),
    m_format(Videos::Unknown),
    m_busy(false),
    m_cancelled(false)
{
    if (!grabberInstance) {
        grabberInstance = this;
    }
}

UrlGrabber* UrlGrabber::instance() {
    return grabberInstance;
}

void UrlGrabber::loadExtractorPlugin() {
    if (!m_extractor) {
#ifdef GRABBER_PLUGIN
        m_extractor = PluginManager::extractorPlugin(this);

        if (!m_extractor) {
            m_extractor = new ExtractorPlugin(this);
        }
#else
        m_extractor = new ExtractorPlugin(this);
#endif
    }
}

void UrlGrabber::setYouTubeFormats(QSet<int> formats) {
    m_youtubeFormats.clear();
    QList<int> formatList = formats.toList();
    qSort(formatList.begin(), formatList.end(), qGreater<int>());

    while (!formatList.isEmpty()) {
        int format = formatList.takeFirst();
        VideoFormat videoFormat;
        videoFormat.format = static_cast<Videos::VideoFormat>(format);

        switch (format) {
        case Videos::LQ:
            videoFormat.value = 18;
            videoFormat.displayName = QString("%1 AVC1").arg(tr("Normal (upto 360P)"));
            break;
        case Videos::Normal:
            videoFormat.value = 34;
            videoFormat.displayName = "360P FLV";
            break;
        case Videos::HQ:
            videoFormat.value = 35;
            videoFormat.displayName = "480P FLV";
            break;
        case Videos::HD:
            videoFormat.value = 22;
            videoFormat.displayName = "720P AVC1";
            break;
        case Videos::SuperHD:
            videoFormat.value = 37;
            videoFormat.displayName = "1080P AVC1";
            break;
        default:
            qWarning() << "UrlGrabber::setYouTubeFormats(): Invalid video format";
            break;
        }

        m_youtubeFormats.append(videoFormat);
    }
}

void UrlGrabber::setDailymotionFormats(QSet<int> formats) {
    m_dailymotionFormats.clear();
    QList<int> formatList = formats.toList();
    qSort(formatList.begin(), formatList.end(), qGreater<int>());

    while (!formatList.isEmpty()) {
        int format = formatList.takeFirst();
        VideoFormat videoFormat;
        videoFormat.format = static_cast<Videos::VideoFormat>(format);

        switch (format) {
        case Videos::Normal:
            videoFormat.value = "stream_h264_url";
            videoFormat.displayName = QString("%1 AVC1").arg(tr("Normal"));
            break;
        case Videos::HQ:
            videoFormat.value = "stream_h264_hq_url";
            videoFormat.displayName = "480P AVC1";
            break;
        case Videos::HD:
            videoFormat.value = "stream_h264_hd_url";
            videoFormat.displayName = "720P AVC1";
            break;
        case Videos::SuperHD:
            videoFormat.value = "stream_h264_hd1080_url";
            videoFormat.displayName = "1080P AVC1";
            break;
        default:
            qWarning() << "UrlGrabber::setDailymotionFormats(): Invalid video format";
            break;
        }

        m_dailymotionFormats.append(videoFormat);
    }
}

void UrlGrabber::setVimeoFormats(QSet<int> formats) {
    m_vimeoFormats.clear();
    QList<int> formatList = formats.toList();
    qSort(formatList.begin(), formatList.end(), qGreater<int>());

    while (!formatList.isEmpty()) {
        int format = formatList.takeFirst();
        VideoFormat videoFormat;
        videoFormat.format = static_cast<Videos::VideoFormat>(format);

        switch (format) {
        case Videos::LQ:
            videoFormat.value = "mobile";
            videoFormat.displayName = tr("Mobile");
            break;
        case Videos::Normal:
            videoFormat.value = "sd";
            videoFormat.displayName = "360P AVC1";
            break;
        case Videos::HD:
            videoFormat.value = "hd";
            videoFormat.displayName = "720P AVC1";
            break;
        default:
            qWarning() << "UrlGrabber::setVimeoFormats(): Invalid video format";
            break;
        }

        m_vimeoFormats.append(videoFormat);
    }
}

void UrlGrabber::setBusy(bool isBusy, const QString &message, int numberOfOperations) {
    if (isBusy != this->busy()) {
        m_busy = isBusy;

        if (isBusy) {
            this->setCancelled(false);
            emit busy(message, numberOfOperations);
        }
        else if (!this->cancelled()) {
            emit busyProgressChanged(numberOfOperations);
        }

        emit busyChanged(isBusy);
    }
}

void UrlGrabber::cancelCurrentOperation() {
    this->setCancelled(true);
    this->setBusy(false);
    emit currentOperationCancelled();
}

void UrlGrabber::getVideoUrl(int service, const QString &id) {
    switch (service) {
    case Services::YouTube:
        this->getYouTubeVideoUrl(id);
        break;
    case Services::Dailymotion:
        this->getDailymotionVideoUrl(id);
        break;
    case Services::Vimeo:
        this->getVimeoVideoUrl(id);
        break;
    default:
        qWarning() << "UrlGrabber::getVideoUrl(): No/invalid service specied";
        return;
    }
}

void UrlGrabber::getYouTubeVideoUrl(const QString &id) {
    this->setBusy(true, tr("Loading video"));
    this->loadExtractorPlugin();
    QNetworkRequest request(m_extractor->youtubeRequestUrl(id));
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(parseYouTubeVideoPage()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void UrlGrabber::parseYouTubeVideoPage() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QString response(reply->readAll());
    QVariantList formats = m_extractor->extractYouTubeFormats(response);

    if (formats.isEmpty()) {
        emit error(tr("Unable to retrieve video. Access may be restricted"));
    }
    else {
        QString errorString = formats.first().toMap().value("error").toString();

        if (!errorString.isEmpty()) {
            emit error(errorString);
        }
        else {
            QUrl videoUrl;
            int i = 0;

            while ((videoUrl.isEmpty()) && (i < m_youtubeFormats.size())) {
                int ii = 0;

                while ((videoUrl.isEmpty()) && (ii < formats.size())) {
                    QVariantMap format = formats.at(ii).toMap();

                    if (format.value("value") == m_youtubeFormats.at(i).value) {
                        videoUrl = format.value("url").toUrl();
                        m_format = m_youtubeFormats.at(i).format;
                        emit gotVideoUrl(videoUrl, m_format);
                    }

                    ii++;
                }

                i++;
            }
        }
    }

    reply->deleteLater();
}

void UrlGrabber::getDailymotionVideoUrl(const QString &id) {
    this->setBusy(true, tr("Loading video"));
    this->loadExtractorPlugin();
    QNetworkRequest request(m_extractor->dailymotionRequestUrl(id));
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(parseDailymotionVideoPage()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void UrlGrabber::parseDailymotionVideoPage() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QString response(reply->readAll());
    QVariantList formats = m_extractor->extractDailymotionFormats(response);

    if (formats.isEmpty()) {
        emit error(tr("Unable to retrieve video. Access may be restricted"));
    }
    else {
        QString errorString = formats.first().toMap().value("error").toString();

        if (!errorString.isEmpty()) {
            emit error(errorString);
        }
        else {
            QUrl videoUrl;
            int i = 0;

            while ((videoUrl.isEmpty()) && (i < m_dailymotionFormats.size())) {
                int ii = 0;

                while ((videoUrl.isEmpty()) && (ii < formats.size())) {
                    QVariantMap format = formats.at(ii).toMap();

                    if (format.value("value") == m_dailymotionFormats.at(i).value) {
                        videoUrl = format.value("url").toUrl();
                        m_format = m_dailymotionFormats.at(i).format;
                        emit gotVideoUrl(videoUrl, m_format);
                    }

                    ii++;
                }

                i++;
            }
        }
    }
    
    reply->deleteLater();
}

void UrlGrabber::getVimeoVideoUrl(const QString &id) {
    this->setBusy(true, tr("Loading video"));
    this->loadExtractorPlugin();
    QNetworkRequest request(m_extractor->vimeoRequestUrl(id));
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(parseVimeoVideoPage()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void UrlGrabber::parseVimeoVideoPage() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QString response(reply->readAll());
    QVariantList formats = m_extractor->extractVimeoFormats(response);

    if (formats.isEmpty()) {
        this->setBusy(false);
        emit error(tr("Unable to retrieve video. Access may be restricted"));
    }
    else {
        QString errorString = formats.first().toMap().value("error").toString();

        if (!errorString.isEmpty()) {
            this->setBusy(false);
            emit error(errorString);
        }
        else {
            QUrl videoUrl;
            int i = 0;

            while ((videoUrl.isEmpty()) && (i < m_vimeoFormats.size())) {
                int ii = 0;

                while ((videoUrl.isEmpty()) && (ii < formats.size())) {
                    QVariantMap format = formats.at(ii).toMap();

                    if (format.value("value") == m_vimeoFormats.at(i).value) {
                        videoUrl = format.value("url").toUrl();
                        m_format = m_vimeoFormats.at(i).format;
                        this->getVimeoVideoRedirect(videoUrl);
                    }

                    ii++;
                }

                i++;
            }
        }
    }

    reply->deleteLater();
}

void UrlGrabber::getVimeoVideoRedirect(const QUrl &url) {
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkVimeoVideoRedirect()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void UrlGrabber::checkVimeoVideoRedirect() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QUrl redirect = reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toUrl();

    if (redirect.isEmpty()) {
        emit gotVideoUrl(reply->request().url(), m_format);
    }
    else {
        emit gotVideoUrl(redirect, m_format);
    }

    reply->deleteLater();
}

void UrlGrabber::getAvailableVideoFormats(int service, const QString &id) {
    switch (service) {
    case Services::YouTube:
        this->getAvailableYouTubeVideoFormats(id);
        break;
    case Services::Dailymotion:
        this->getAvailableDailymotionVideoFormats(id);
        break;
    case Services::Vimeo:
        this->getAvailableVimeoVideoFormats(id);
        break;
    default:
        return;
    }
}

void UrlGrabber::getAvailableYouTubeVideoFormats(const QString &id) {
    this->setBusy(true, tr("Retrieving available video formats"));
    this->loadExtractorPlugin();
    QNetworkRequest request(m_extractor->youtubeRequestUrl(id));
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkYouTubeVideoFormats()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void UrlGrabber::checkYouTubeVideoFormats() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QString response(reply->readAll());
    QVariantList formats = m_extractor->extractYouTubeFormats(response);

    if (formats.isEmpty()) {
        emit error(tr("Unable to retrieve video formats. Access may be restricted"));
    }
    else {
        QString errorString = formats.first().toMap().value("error").toString();

        if (!errorString.isEmpty()) {
            emit error(errorString);
        }
        else {
            emit gotVideoFormats(formats);
        }
    }

    reply->deleteLater();
}

void UrlGrabber::getAvailableDailymotionVideoFormats(const QString &id) {
    this->setBusy(true, tr("Retrieving available video formats"));
    this->loadExtractorPlugin();
    QNetworkRequest request(m_extractor->dailymotionRequestUrl(id));
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkDailymotionVideoFormats()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void UrlGrabber::checkDailymotionVideoFormats() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QString response(reply->readAll());
    QVariantList formats = m_extractor->extractDailymotionFormats(response);

    if (formats.isEmpty()) {
        emit error(tr("Unable to retrieve video formats. Access may be restricted"));
    }
    else {
        QString errorString = formats.first().toMap().value("error").toString();

        if (!errorString.isEmpty()) {
            emit error(errorString);
        }
        else {
            emit gotVideoFormats(formats);
        }
    }

    reply->deleteLater();
}

void UrlGrabber::getAvailableVimeoVideoFormats(const QString &id) {
    this->setBusy(true, tr("Retrieving available video formats"));
    this->loadExtractorPlugin();
    QNetworkRequest request(m_extractor->vimeoRequestUrl(id));
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkVimeoVideoFormats()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void UrlGrabber::checkVimeoVideoFormats() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    this->loadExtractorPlugin();
    QString response(reply->readAll());
    QVariantList formats = m_extractor->extractVimeoFormats(response);

    if (formats.isEmpty()) {
        emit error(tr("Unable to retrieve video formats. Access may be restricted"));
    }
    else {
        QString errorString = formats.first().toMap().value("error").toString();

        if (!errorString.isEmpty()) {
            emit error(errorString);
        }
        else {
            emit gotVideoFormats(formats);
        }
    }

    reply->deleteLater();
}
